(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

open HolKernel boolLib Parse bossLib pairLib;
open utilTheory open x86_typesTheory;

open HolDoc;

val _ = new_theory "x86_axiomatic_model";


(******************************************************************************)
(* basic operations on datatypes *)

(* currently NONE case is impossible *)

val loc_def = Define `
  loc e = 
    case e.action of
       Access d l v -> SOME l
    || _ -> NONE`;

(* currently NONE case is impossible *)

val value_of_def = Define `
  value_of e =
    case e.action of
       Access d l v -> SOME v
    || _ -> NONE`;

val proc_def = Define `
  proc e = e.iiid.proc`;

val mem_load_def = Define `
  mem_load e = 
    case e.action of
       Access R (Location_mem a) v -> T
    || _ -> F`;

val mem_store_def = Define `
  mem_store e = 
    case e.action of
       Access W (Location_mem a) v -> T
      || _ -> F`;

val mem_barrier_def = Define `
  mem_barrier e = 
    case e.action of
    (*   Barrier bar -> T || *) 
    _ -> F`;

val iiids_def = Define `
  iiids E = {e.iiid | e IN E.events}`;

val writes_def = Define `
  writes E = {e | e IN E.events /\ ?l v. e.action = Access W l v}`;

val reads_def = Define `
  reads E = {e | e IN E.events /\ ?l v. e.action = Access R l v}`;


val locked_def = Define ` 
  locked E e = (e IN BIGUNION E.atomicity)`;


(******************************************************************************)
(* program order *)

val po_def = Define `
  po E = 
    {(e1, e2) | (e1.iiid.proc = e2.iiid.proc) /\
                e1.iiid.program_order_index <= e2.iiid.program_order_index /\
                e1 IN E.events /\ e2 IN E.events}`;

val po_strict_def = Define `
  po_strict E = 
    {(e1, e2) | (e1.iiid.proc = e2.iiid.proc) /\
                e1.iiid.program_order_index < e2.iiid.program_order_index /\
                e1 IN E.events /\ e2 IN E.events}`;

(*
val po_strict_def = Define `
  po_strict E = strict (po E)
`;
*)

(* FIXME maybe this should be in an earlier x86_base file? *)

val po_iico_def = Define `
  po_iico E = po_strict E UNION E.intra_causality
`;


(******************************************************************************)
(* basic structure: E *)

val well_formed_event_structure_def = Define `
  well_formed_event_structure E =
    (!iiid. FINITE {eiid | ?e :: (E.events). (e.iiid = iiid) /\ (e.eiid = eiid)}) /\
    (FINITE E.procs) /\
    (!e :: (E.events). proc e IN E.procs) /\
    (!e1 e2 :: (E.events). (e1.eiid = e2.eiid) /\ (e1.iiid = e2.iiid) ==> (e1 = e2)) /\
    (DOM E.intra_causality) SUBSET E.events /\
    (RANGE E.intra_causality) SUBSET E.events /\
    acyclic (E.intra_causality) /\
    (!(e1, e2) :: (E.intra_causality). (e1.iiid = e2.iiid)) /\
    (!(e1, e2) :: (E.intra_causality). ~(mem_store e1)) /\
    (!(e1 :: writes E) e2.
      ~(e1 = e2) /\
      (e2 IN writes E \/ e2 IN reads E) /\
      (e1.iiid = e2.iiid) /\ 
      (loc e1 = loc e2) /\
      (?p r. loc e1 = SOME (Location_reg p r))
      ==>
      (e1, e2) IN sTC E.intra_causality \/
      (e2, e1) IN sTC E.intra_causality) /\
    (!es :: (E.atomicity). ?e :: es. mem_load e) /\
    PER E.events E.atomicity /\
    (!es :: (E.atomicity). !e1 e2 :: es. (e1.iiid = e2.iiid)) /\
    (!es :: (E.atomicity). !e1 :: es. !e2 :: (E.events). (e1.iiid = e2.iiid) ==> e2 IN es) /\
    (!e :: (E.events). !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e))`;

val sub_event_structure_def = Define `
  sub_event_structure E' E = 
    (E'.procs = E.procs) /\
    E'.events SUBSET E.events /\ 
    (E'.atomicity = PER_RESTRICT E.atomicity E'.events) /\
    (E'.intra_causality = RRESTRICT E.intra_causality E'.events)`;


val preserved_program_order_def = Define `
  preserved_program_order E = 
       {(e1, e2) | (e1, e2) IN (po_strict E) /\
          (   (?p r. (loc e1 = loc e2) /\ 
                 (loc e1 = SOME (Location_reg p r))) \/
          (mem_load e1 /\ mem_load e2) \/
          (mem_store e1 /\ mem_store e2) \/
          (mem_load e1 /\ mem_store e2) \/
          (mem_store e1 /\ mem_load e2 /\ (loc e1 = loc e2)) \/
          ((mem_load e1 \/ mem_store e1) /\ locked E e2) \/
          (locked E e1 /\ (mem_load e2 \/ mem_store e2)))}`;


(*********************** View Orders ***********************)

val viewed_events_def = Define `
  viewed_events E p =
    {e | e IN E.events /\ ((proc e = p) \/ mem_store e)}`;

val view_orders_well_formed_def = Define `
  view_orders_well_formed E vo =
    (!p :: (E.procs). linear_order (vo p) (viewed_events E p) /\
       !e :: (viewed_events E p). FINITE {e' | (e', e) IN (vo p)}) /\
    (!p. ~(p IN E.procs) ==> (vo p = {}))`; 


(*
val viewed_events_univ_def = Define `
  viewed_events_univ E p =
    UNIV_RELN (viewed_events E p)`;
*)

val get_l_stores_def = Define `
  get_l_stores E l =
    {e | e IN E.events /\ mem_store e /\ (loc e = SOME l)}`;

val write_serialization_candidates_old_def = Define `
  write_serialization_candidates_old E =
    let per_location_store_sets = {es | ?l. es = get_l_stores E l} in
    let per_location_store_set_linearisations = {strict_linearisations es | es IN per_location_store_sets} in
    let choices = all_choices per_location_store_set_linearisations in
      {BIGUNION lin | lin IN choices}`;

val write_serialization_candidates_def = Define `
  write_serialization_candidates E cand =
    (!(e1, e2) :: cand. 
       ?l. e1 IN (get_l_stores E l) /\ e2 IN (get_l_stores E l)) /\
    (!l. strict_linear_order (RRESTRICT cand (get_l_stores E l))
                         (get_l_stores E l))`;

(* |- write_serialization_candidates = write_serialization_candidates_old *)

val lock_serialization_candidates_def = Define `
  lock_serialization_candidates E =
       let lin_ec = strict_linearisations E.atomicity in
          { {(e1, e2) | ?(es1, es2) :: lin. e1 IN es1 /\ e2 IN es2} 
          | lin IN lin_ec}`;

val reads_from_map_candidates_old_def = Define `
  reads_from_map_candidates_old E =
    let reads_and_their_possible_writes =
          {(er, ews) | er IN E.events /\ 
                       ?l v. (er.action = Access R l v) /\ 
                             (ews = {ew | ew IN E.events /\ (ew.action = Access W l v)})} in
      {rfmap | (RANGE rfmap) SUBSET (DOM reads_and_their_possible_writes) /\
               !(ew, er) :: rfmap. ?ews. (er, ews) IN reads_and_their_possible_writes /\ ew IN ews}`;

val reads_from_map_candidates = Define `
  reads_from_map_candidates E rfmap = 
       !(ew, er) :: rfmap. er IN E.events /\ ew IN E.events /\ 
           ?l v. (er.action = Access R l v) /\ 
               (ew.action = Access W l v)`;

(* |- reads_from_map_candidates = reads_from_map_candidates_old *)


val happens_before_def = Define `
  happens_before E X =
       E.intra_causality UNION
       (preserved_program_order E) UNION
       X.write_serialization UNION
       X.lock_serialization UNION
       X.rfmap`;

val check_causality_def = Define `
  check_causality E vo happensbefore =
       !p :: (E.procs). acyclic ((strict (vo p)) UNION happensbefore)`;

      (*
val check_rfmap_def = Define `
  check_rfmap E vo rfmap =
    !p ew er. p IN procs E /\ rfmap ew er /\ (viewed_events_univ E p) ew er ==>
      ~?ew'. ew' IN writes E /\ (vo p) ew ew' /\ (vo p) ew' er /\ (loc ew = loc ew')`;
      *)

val check_rfmap_written_def = Define `
  check_rfmap_written E vo rfmap =
    !p :: (E.procs). 
       !(ew, er) :: (RRESTRICT rfmap (viewed_events E p)).
          !ew' :: (writes E). 
            ~(ew = ew') /\ (ew, ew') IN (vo p) /\ (ew', er) IN (vo p) 
            ==> ~(loc ew = loc ew')`;

       (*
val check_rfmap_initial_def = Define `
  check_rfmap_initial E vo rfmap initial_state =
    !p er. p IN procs E /\ er IN ((reads E) DIFF (RRANGE rfmap)) INTER view_order_events_for_proc E p ==>
      ((?l v. (er.action = Access R l v) /\ (initial_state l = SOME v) /\
       ~?ew'. ew' IN writes E /\ (vo p) ew' er /\ (loc ew' = loc er)))`;
       *)

val check_rfmap_initial_def = Define `
  check_rfmap_initial E vo rfmap initial_state =
    !p :: (E.procs). 
       !er :: (((reads E) DIFF (RANGE rfmap)) 
              INTER viewed_events E p).
         ?l v. (er.action = Access R l v) /\ 
               (initial_state l = SOME v) /\
               !ew' :: writes E. 
                    (ew', er) IN (vo p) ==> ~(loc ew' = loc er)`;

(*
val check_d_def = Define `
  check_d E vo rfmap initial_state final_state write_serialization =
    let state_updates l = 
      case l of
         Location_mem a -> 
           {v | ?ew. ew IN maximal_elements write_serialization /\ (ew.action = Access W l v)}
      || Location_reg p r -> 
           {v | ?ew. ew IN writes E /\ (ew.action = Access W l v) /\ 
                     ~?ew'. ew' IN writes E /\ (vo p) ew ew' /\ (loc ew = loc ew')}
    in
      !l. ((final_state l = NONE) = (initial_state l = NONE) /\ (state_updates l = {})) /\
          !v. (final_state l = SOME v) ==> v IN state_updates l`;
          *)

val state_updates_def = Define `
  state_updates E vo write_serialization l = 
    case l of
       Location_mem a -> 
         {value_of ew | ew IN maximal_elements (get_l_stores E l) write_serialization}
    || Location_reg p r -> 
         {value_of ew | ew IN maximal_elements (get_l_stores E l) (vo p)}`;

val check_final_def = Define `
  check_final E vo initial_state final_state_opt write_serialization =
    if FINITE E.events then
      ?final_state. (final_state_opt = SOME final_state) /\
      !l. if (state_updates E vo write_serialization l) = {} then
            final_state l = initial_state l
          else
            (final_state l) IN (state_updates E vo write_serialization l)
    else
      final_state_opt = NONE`;

val state_updates_mem_def = Define `
  state_updates_mem E write_serialization a = 
    {value_of ew | ew IN maximal_elements (get_l_stores E (Location_mem a)) write_serialization}`;

val check_final_mem_def = Define `
  (check_final_mem E initial_state write_serialization NONE = 
     ~(FINITE E.events)) /\
  (check_final_mem E initial_state write_serialization (SOME final_state) =
     FINITE E.events /\ 
     !a. if (state_updates_mem E write_serialization a) = {} then
           final_state a = initial_state (Location_mem a)
         else
           (final_state a) IN (state_updates_mem E write_serialization a))`;


          (*
val check_atomicity_def = Define `
  check_atomicity E vo =
    !p es_atomic e1 e2. p IN procs E /\ es_atomic IN E.atomicity /\ e1 IN es_atomic /\ e2 IN es_atomic ==>
      (vo p) e1 e2 ==> ~?e. (vo p) e1 e /\ (vo p) e e2 /\ ~(e IN es_atomic)`;
      *)

val check_atomicity_def = Define `
  check_atomicity E vo =
       !p :: (E.procs). !es :: (E.atomicity). 
          !e1 e2:: es. (e1, e2) IN (vo p) ==> 
            !e. (e1, e) IN (vo p) /\ (e, e2) IN (vo p) ==> e IN es`;

val valid_execution_def = Define `
  valid_execution E X =
       view_orders_well_formed E X.vo /\
       X.write_serialization IN write_serialization_candidates E /\
       X.lock_serialization IN lock_serialization_candidates E /\
       X.rfmap IN reads_from_map_candidates E /\
       check_causality E X.vo (happens_before E X) /\
       check_rfmap_written E X.vo X.rfmap /\
       check_rfmap_initial E X.vo X.rfmap X.initial_state /\
       check_atomicity E X.vo`; 

    (* check_d E X.vo X.initial_state X.final_state_opt X.write_serialization /\ *)  (* TODO: Check with Scott *)


val restrict_execution_witness_def =  Define `
  restrict_execution_witness X E =
    <| initial_state := X.initial_state;
       (*final_state_opt := ... *)
       vo := (\p. RRESTRICT (X.vo p) E.events);
       write_serialization := RRESTRICT X.write_serialization E.events;
       lock_serialization := RRESTRICT X.lock_serialization E.events;
       rfmap := RRESTRICT X.rfmap E.events|>`;


val _ = export_theory ();
