(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

structure tactic = struct

open HolKernel boolLib Parse bossLib proofManagerLib computeLib List

fun MY_GEN_ALL th =
   GENL (free_vars_lr (concl th)) th;

(* FIXME THIN_TAC should fail if asm not present, also THIN_TAC seems to take no notice of free vars *)

val CUT = PROVE_HYP;

(* dangerous *)
val CHEAT_TAC = fn (asl,w) => ([], fn [] => mk_thm (asl,w));

(* for matching curried rules *)
val MATCH_MP_TAC' = 
    let
	val r = PROVE [] ``! P Q R. (P ==> Q ==> R) = (P /\ Q ==> R)``
    in
	fn th => MATCH_MP_TAC th ORELSE MATCH_MP_TAC (SIMP_RULE std_ss [r] th)
    end;


(******************************************************************************)
(* tacticals *)

(* version of PAT_ASSUM that doesn't remove asm *)

local
  fun match_with_constants constants pat ob = let
    val (tm_inst, ty_inst) =
        ho_match_term [] empty_tmset pat ob
    val bound_vars = map #redex tm_inst
  in
    null (intersect constants bound_vars)
  end handle HOL_ERR _ => false
in
fun PAT_ASSUM pat thfun (asl, w) =
  case List.filter (can (ho_match_term [] empty_tmset pat)) asl
   of []  => raise ERR "PAT_ASSUM" "No assumptions match the given pattern"
    | [x] => let val (ob, asl') = Lib.pluck (Lib.equal x) asl
             in thfun (ASSUME ob) (asl, w)
             end
    |  _  =>
      let val fvars = free_varsl (w::asl)
          val (ob,asl') = Lib.pluck (match_with_constants fvars pat) asl
      in thfun (ASSUME ob) (asl,w)
      end
end;

val PAT_X_ASSUM = Tactical.PAT_ASSUM;

fun ITAC_OF_THM_TAC tt i = ASSUM_LIST (fn ths => tt (List.nth(List.rev ths,i)));


(******************************************************************************)
(* variables *)

val RENAME_TAC = 
 fn sigma => (* sig is a variable to variable (term to term) substitution, e.g. x->y *)
 fn (asl,w) => 
    let 
	val sub = subst sigma
	val (asl',w') = (List.map sub asl, sub w)
	val sig' = List.map (fn {redex,residue} => {residue=redex,redex=residue}) sigma
    in
	([(asl',w')],fn [th] => INST sig' th)
    end;

(* hack to print the freevars of a goal- useful for checking initial goals *)
val FV_TAC = fn (asl,w) =>
  let val fvs = free_varsl (asl@[w])
      val _ = map (fn fv => (print_term fv; print_type (type_of fv); print("\n"))) fvs
  in ALL_TAC (asl,w) end;

val CLOSED_TAC = fn (asl,w) =>
  let val fvs = free_varsl (asl@[w])
      val _ = map (fn fv => (print_term fv; print_type (type_of fv); print("\n"))) fvs
      val _ = if fvs = [] then () else raise (ERR "CLOSE_TAC" "")
  in ALL_TAC (asl,w) end;


(******************************************************************************)
(* assumption handling *)

(*

peqp'
---------
G |- p=p'
-----------
G |- p==>p'      p'C
-----------     ----------
p,G |- p'        p',G |- C
---------------------------
p,G |- C

*)

val ASM_CONV_RULE =
 fn peqp' =>
 fn p'C =>
    let
	val pimpp' = fst (EQ_IMP_RULE peqp')
	val pp' = UNDISCH pimpp'
        val pC = CUT pp' p'C
    in
	pC
    end handle e as HOL_ERR _ => raise wrap_exn "ASM_CONV_RULE" "" e;

(* int is the assumption to conv *)
val (ASM_CONV_TAC:(thm list->conv)->int->tactic) =
 fn conv => 
 fn i => 
 fn (asl,w) =>
    let
val _ = if length asl <= i then raise Subscript else ()
	val i' = length asl - i - 1
	val (asll,p,aslr) = (take(asl,i'), nth(asl,i'), drop (asl,(i'+1)))
	val peqp' = conv (map ASSUME (asll@aslr)) p
	val p' = snd (dest_eq (concl peqp'))
    in
	([(asll@[p']@aslr,w)], fn [p'C] => ASM_CONV_RULE peqp' p'C)
    end handle e as UNCHANGED => raise wrap_exn "ASM_CONV_TAC" "UNCHANGED" e;

val ASM_EVAL_TAC = ASM_CONV_TAC (fn ths => EVAL);

val ASM_RESTR_EVAL_TAC = fn tms => ASM_CONV_TAC (fn ths => computeLib.RESTR_EVAL_CONV tms);

val assimp = fn ths1 => ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) (ths@ths1)); (* SIMP_CONV can throw unchanged *)
val asimp = fn ths1 => ASM_CONV_TAC (fn ths => SIMP_CONV (std_ss) (ths@ths1));


(******************************************************************************)
(* structural *)

val MY_ASSUME_TAC:thm_tactic = 
 fn bth => 
 fn (asl,w) =>
    ([(asl@[concl bth],w)], (fn [th] => CUT bth th));

val THIN_TAC:thm->tactic = fn th => fn (asl,w) => ([(filter (fn tm => not (aconv tm (concl th))) asl,w)],fn [th] => th);

val TM_THIN_TAC = fn tm =>(PAT_X_ASSUM tm (fn th => ALL_TAC)); 


val REORDER_TAC = fn tm => PAT_X_ASSUM tm (fn th => MY_ASSUME_TAC th);

val REMOVE_VAR_TAC = fn v => REPEAT (FIRST_X_ASSUM (fn th => if free_in v (concl th) then ALL_TAC else raise ERR "tactic" "REMOVE_VAR_TAC"));

val CUT_TAC:term->tactic = fn tm => fn (asl,w) => ([(asl,tm),(asl@[tm],w)], fn [th1,th2] => CUT th1 th2);

(******************************************************************************)
(* standard propositional *)

val INIT_TAC = FIRST_ASSUM ACCEPT_TAC;
val FALSEL_TAC = FIRST_ASSUM CONTR_TAC;
val TRUER_TAC:tactic = fn (asl,w) => if aconv w ``T`` then ([],fn _ => TRUTH) else raise ERR "TRUER_TAC" "";

val CONJL'_TAC : thm_tactic = 
 fn th => 
    let val (th1,th2) = CONJ_PAIR th in
	MY_ASSUME_TAC th1 THEN MY_ASSUME_TAC th2 end;

val CONJL_TAC : tactic = FIRST_X_ASSUM CONJL'_TAC;

val CONJR_TAC = CONJ_TAC;

val DISJL_TAC = FIRST_X_ASSUM DISJ_CASES_TAC;

val DISJR1_TAC = DISJ1_TAC;

val DISJR2_TAC = DISJ2_TAC;

val IMPR_TAC = DISCH_TAC;
			
val IMPL_TAC = 
 fn th => 
 fn (asl,w) => 
		   let
val (a,b) = dest_imp (concl th)
val asl' = filter (fn a => not (aconv a (concl th))) asl 
		   in
			 ([(asl',a),(asl'@[b],w)],fn [th1,th2] => PROVE_HYP (MP th th1) th2) 
		   end

val rec MY_DISJL_TAC = 
 fn tm => 
    if is_disj tm then 
	(PAT_X_ASSUM tm DISJ_CASES_TAC) 
	    THENL [MY_DISJL_TAC(#1(dest_disj tm)),MY_DISJL_TAC(#2(dest_disj tm))]
    else ALL_TAC;


(******************************************************************************)
(* quantifiers *)

val MYSPEC = fn tm => fn th => SPEC tm th handle e => PairRules.PSPEC tm th;

val FORALLL_TAC = fn tm => fn th => MY_ASSUME_TAC (MYSPEC tm th);
val FORALLL_X_TAC = fn tm => fn th => FORALLL_TAC tm th THEN THIN_TAC th;
val FORALLR_TAC = GEN_TAC;

val EXL_TAC = FIRST_X_ASSUM CHOOSE_TAC;
val EXR_TAC = EXISTS_TAC;


(******************************************************************************)
(* equality *)

val EQR_TAC = EQ_TAC THEN IMPR_TAC;
val EQL_TAC = FIRST_X_ASSUM (fn p_eq_q => let val (p_imp_q, q_imp_p) = EQ_IMP_RULE p_eq_q in (ASSUME_TAC p_imp_q) THEN (ASSUME_TAC q_imp_p) end);;
val SYM_TAC = fn tm => PAT_X_ASSUM tm (fn th => ASSUME_TAC (SYM th));


(******************************************************************************)
(* combinations *)

val intros = REPEAT (IMPR_TAC ORELSE FORALLR_TAC);
val elims = REPEAT (EXL_TAC ORELSE CONJL_TAC);

val cintros = REPEAT (IMPR_TAC ORELSE FORALLR_TAC ORELSE CONJR_TAC);

fun defer () = rotate 1;


(******************************************************************************)
(* abbreviations *)

val ii = fn () => e INIT_TAC;
val cu = fn tm => e (CUT_TAC tm);
val fl = fn () => e (FALSEL_TAC);
val ir = fn () => e IMPR_TAC;
val il = fn i => e(ITAC_OF_THM_TAC IMPL_TAC i);
val cr = fn () => e CONJR_TAC;
val dl = fn () => e DISJL_TAC;
val dr1 = fn () => e DISJR1_TAC;
val dr2 = fn () => e DISJR2_TAC;
val al = fn tm => e(FIRST_ASSUM (FORALLL_TAC tm)); (* default not to remove! *)
val xal = fn tm => e(FIRST_X_ASSUM (FORALLL_TAC tm));
val ar = fn () => e(FORALLR_TAC);
val er = fn tm => e(EXISTS_TAC tm);
(* el already bound *)
val eql = fn () => e EQL_TAC;


(******************************************************************************)
(* QCUT_TAC *)

(* alternative versions taking quotes *)
fun normalise_quotation frags =
  case frags of
    [] => []
  | [x] => [x]
  | (QUOTE s1::QUOTE s2::rest) => normalise_quotation (QUOTE (s1^s2) :: rest)
  | x::xs => x :: normalise_quotation xs;

fun contextTerm ctxt q = Parse.parse_in_context ctxt (normalise_quotation q);

fun ptm_with_ctxtty ctxt ty q =
 let val q' = QUOTE "(" :: (q @ [QUOTE "):", ANTIQUOTE(ty_antiq ty), QUOTE ""])
 in Parse.parse_in_context ctxt (normalise_quotation q')
end;

fun QCUT_TAC q (g as (asl,w)) =
let val ctxt = free_varsl (w::asl)
in CUT_TAC (ptm_with_ctxtty ctxt Type.bool q) g
end;

fun Q_DISJ_CASES_TAC q =
 Q.PAT_ASSUM q DISJ_CASES_TAC;

val have = fn x => e(QCUT_TAC x);


(******************************************************************************)
(* simplification and first order proof *)

(* apply tac1 to all asms, until nothing changes, then apply tac2 *)

fun pre_simp tac1 tac2:tactic = fn (asl,w) => 
  let val ns = Lib.upto 0 (length asl  - 1)
      val tacs = map (fn i => TRY (tac1 i)) ns
      val ssimp_once = EVERY tacs
      val tac = 
		    (REPEAT (CHANGED_TAC ssimp_once))
		    THEN tac2
		    THEN (TRY (FALSEL_TAC ORELSE TRUER_TAC))
		    THEN elims
  in
	tac (asl,w)
  end;

val simp:thm list->tactic = fn ths => pre_simp (asimp ths) (ASM_SIMP_TAC (std_ss) ths);

val ssimp = fn ths => pre_simp (assimp ths) (ASM_SIMP_TAC (srw_ss()) ths);

val tac = METIS_TAC;


end;


