(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

open HolKernel boolLib Parse bossLib;
open pairLib listTheory;
open stringTheory;


(*open utilTheory;
open x86_typesTheory;*)
open x86_axiomatic_modelTheory;

open HolDoc;

val _ = new_theory "x86_hb_machine";


val _ = Define `clause_name x = T`;

(* machine with explicit happens-before check.  Supposed to be good
only for "nice" executions, ie those in which each p's mem-reads and
register accesses occur in vo p in po_iico order.  (Susmit hopes he
can prove that all executions are equivalent to a nice one)

TODO: pass in iico data so that it can be used in happens_before - add
a little set of events to the event label.  Then change the type of E
and E' below to make it an event structure instead of a set of events.

UPDATE1:
Changed types of E and E', interface of passing in iico data not clear 
yet. Clauses nowadays universally quantify over a set of events to 
feed into iico!

Given all that, is this good?  Can we do a cheaper-looking check
instead of all of happens-before?

Does it not deadlock?  (We think not, and will try to prove that, indeed).
  *)




(* 

  Defining a HOL version of the x86 machine from before (without locking).

  Choices:

   - define an LTS for the memory-and-register subsystem in isolation,
      not containing the processor state, so that this part doesn't
      depend on all the monad stuff. Plan to join them together with
      some CCS-like synchronisation.  (this factorisation should also
      let us state and prove a correctness result comparing this just
      w.r.t. the axiomatic definition of valid execution, without
      involving the microinstruction semantics - then we should be
      able to glue this result onto facts about that)

   - take the labels that those two pieces synchronise on to be
      exactly the event type that we use in the event structure
      definitions (extended with taus).  That'll simplify the plumbing
      for the correctness proof, as we'll have real events lying
      around in the machine to make view orders etc out of.

   - for now, write the memory-and-register LTS direct in HOL (could
      try lifting to Ott).  My HOL isn't great; it may well be that
      this can be polished up by a licensed HOLer.
   
   - restrict to a particular set of processors (which regrettably has
      to be carried about in lts) instead of just taking an infinite
      system, as otherwise we'll never be able to finish traces.

   - I've kept a tuple representation of the state, after a detour
      into a record representation - the latter gets a bit heavy, with
      gazillions of record updates...(?)

   - there's an unfortunate clash of notation with R for Read events
      and Rg for register banks.

  Questions:

   - how to add locking and iico?  I'm not entirely convinced by
      Thomas's recent x86.machine version - I suspect it may be too
      restrictive, preventing, for example, some initial register-only
      events of two different LOCK'd instructions from being
      interleaved.  Also, I wonder whether instead of explicit lock
      and unlock transitions we should simply annotate each event
      label with a lock-PER identifier?? Also, there's nowhere in the
      semantics that explicitly computes the 2.7 total order of locked
      events - I was expecting something a bit like the G gadgets that
      we're currently using for the 2.6 total orders.

*)




(* utils, stolen from TCP *)

val funupd_def = Define`
(*: update one point of a function:*)
  funupd f x y = \x'. if x'=x then y else f x'
` ;

val funupd2_def = Define`
(*: update one point of a single function in a family:*)
  funupd2 f w x y = funupd f w (funupd (f w) x y)
` ;

val linear_order_extend_def = Define `
  linear_order_extend r y = r UNION {(x,y) | (x,x) IN r } `;



(* unshared machine-specific definitions start here *)
(*
val _ = List.app Hol_datatype [
  `mylabel = Ev of (event # event set) 
           | Tau_mylabel   
  ` ]
*)
(*# ((liiid # proc) option)*)

      (*    | Lock of (liiid # proc) option    
           | Unlock of (liiid # proc) option *)


val _ = type_abbrev ("message", ``:  (event#address#value#(event set) ) ``);
(*#((liiid # proc) option)*)


(*val _ = Define `mk_label e iico  = Ev (e,iico)`; *)
(* val _ = Define `mk_label2 x = Tau`; *)




(************************************************************************)



val _ = type_abbrev ("machine_state", ``:  
          (*E*) event_structure #
          (*M*) (proc -> address -> (value # (event option)) option) #    
          (*F*) (proc -> proc -> message list) #
          (*G*) (address -> event list) #                               
(* TODO: now we're using X, instead of having an erasure property, maybe we should lose the explicit G? *)
          (*Rg*) (proc -> reg -> (value # (event option)) option) #     
          (*X*) execution_witness	
	  ``) handle e => Raise e;


(* 
 

Interface: 

   machine_label

State:

-  C p    a memory for each processor p  (call these big caches, not memory!)
-  F p q  a FIFO of writes for each pair of processors p,q
-  G a    a gadget recording a list of writes for each address a
              (more h/w-guy friendly to just record the last, if that suffices)

-  Rg p   a register state for each processor P


Transitions:

-  enqueue-mem-write  (write label): enqueue ew into F p q  from the writing proc p, for each q

-  deliver-mem-write-1 (no label): delivering sub-writes, case (1),
   where eiid has already been seen by G a (and no subwrite of a
   Ga-older eiid is pending for this processor)

-  deliver-mem-write-2 (no label): delivering sub-writes, case (2),
   where eiid is new for G a (and no subwrite of a Ga-older eiid is
   pending for this processor)

-  reg write

-  reg read

-  ...stuff for locks...

*)



(* The above includes instrumentation: the event option in M and Rg
  records the original event that wrote this value, if any (otherwise
  it came from the initial state; of the events in the message in F
  and the events in G, only the eiid is needed in the uninstrumented
  machine; and the X records the execution built so far. *)
 

val _ = Define `(initial_machine_state: state_constraint->machine_state) initialstate  = 
         ( <| events:={}; intra_causality:={}; atomicity:={} |>, 
           (\p. \a. case initialstate (Location_mem a) of NONE -> NONE || SOME v -> SOME(v,NONE)),
           (\p1. \p2. NIL),
           (\a. NIL),
           (\p. \r. case initialstate (Location_reg p r) of NONE -> NONE || SOME v -> SOME(v,NONE)),
           <| initial_state := initialstate;
              vo := \p . {} ;
              write_serialization := {};
              lock_serialization := {};
              rfmap := {} |>
         )`handle e => Raise e;

val _ = Define `execution_witness_of_machine_state (E,M,F,G,Rg,X)=X`;
val _ = Define `event_set_of_machine_state (E,M,F,G,Rg,X)=E.events`;

val _ = Define `next_eiid_es E e = next_eiid { e.eiid | e IN E.events } `

(************************************************************************)

(* FIXME labels in the following should be lts_label s *)

val (machine_trans_rules, machine_trans_ind, machine_trans_cases) = Hol_reln `
(* defn machine_trans *)

(************************************************************************)

   (* enqueuing sub-writes *)  
   ( ! P M F G Rg e F' a v p E X (X':execution_witness) iico mvl.
   (clause_name "enqueue-mem-write" /\
   (e = mvl.mvl_event) /\
   (iico = mvl.mvl_iico) /\ 
   (e.action = Access W (Location_mem a) v) /\
   (p = e.iiid.proc) /\
   (! q. q IN P ==> ((F' p q) = ([(e,a,(v,iico))] ++ (F p q)))) /\ 
   (! p' q. p' IN P ==> q IN P ==> ~(p=p') ==> (F' p' q = F p' q)) /\
   (X'= X)                  
   ) ==>
   machine_trans P (E,M,F,G,Rg,X) (Vis mvl) (E,M,F',G,Rg,X')
   ) /\

(************************************************************************)

   (* Read own memory *)
   ( ! P M F G Rg e a v p E E' X X' eo iico mvl.
   (clause_name "mem-read" /\
   (e = mvl.mvl_event) /\
   (iico = mvl.mvl_iico) /\ 
   (e.action = Access R (Location_mem a) v) /\
   (p = e.iiid.proc) /\
   (M p a = SOME (v,eo)) /\
   (~ ? e'  v' iico'.  MEM (e',a,v',iico') (F p p)) /\
   (E'= <| events := E.events UNION {e}; 
           intra_causality:=E.intra_causality UNION {(e',e)|e' IN iico}; 
           atomicity :=E.atomicity|>) /\ 
   (X'= X with <| vo := funupd X.vo p (linear_order_extend (X.vo p) e);
                  rfmap := case eo of NONE -> X.rfmap
                                   || SOME ew -> X.rfmap UNION {(ew,e)} |>) /\
(* TODO: would it be cleaner to pull eo from the most recent write in our X.vo p, instead of recording eo in M?*)
   (! e'. (e',e) IN sTC (happens_before E' X') ==>  e' IN viewed_events E' p ==> (e',e') IN X.vo p )
   ) ==>
   machine_trans P (E,M,F,G,Rg,X) (Vis mvl) (E',M,F,G,Rg,X')
   ) /\
(************************************************************************)

  (* delivering sub-writes, case (1), where eiid has already been seen by *)
   (* Ga  (and no subwrite of a Ga-older eiid is pending for this processor)  *)
   ( ! P M F G Rg M' F' p q e a v G0 G1 E X X' iico.
   ((clause_name "deliver-mem-write-1") /\ 
   (e.action = Access W (Location_mem a) v) /\
   (F = funupd2 F' p q ((F' p q) ++ [(e,a,(v,iico))])) /\
   (G a = (G0 ++ [e] ++ G1)) /\
   (! p' e' v'.  p' IN P ==> MEM (e',a,v') (F p' q) ==> ~(MEM e' G1)) /\
   (M' = funupd2 M q a (SOME (v,SOME e))) /\
   (X'= X with <| vo := funupd X.vo q (linear_order_extend (X.vo q) e) |>) /\
   (! e'. (e',e) IN sTC (happens_before E X') ==>  e' IN viewed_events E q ==> (e',e') IN X.vo q )
   ) ==> 
   machine_trans P (E,M,F,G,Rg,X) Tau (E,M',F',G,Rg,X') 
   ) /\

(************************************************************************)

( ! P M F G Rg M' F' G' p q a e v E E' X X' iico.
((clause_name "deliver-mem-write-2") /\ 
(e.action = Access W (Location_mem a) v) /\
(F = (funupd2 F' p q ((F' p q) ++ ([(e,a,(v,iico))])))) /\
( ~ (MEM e (G a))) /\
(G' = funupd G a ([e] ++ G a)) /\
(! p' e' v'.  p' IN P ==> MEM (e',a,v') (F p' q) ==> ~(MEM e' (G a))) /\
(M' = funupd2 M q a (SOME (v,SOME e))) /\
(E'= <| events := E.events UNION {e}; 
        intra_causality:=E.intra_causality UNION {(e',e)|e' IN iico}; 
        atomicity :=E.atomicity|>) /\ 
(X'= X with <| vo := funupd X.vo q (linear_order_extend (X.vo q) e) ;
               write_serialization := X.write_serialization UNION { (e',e) | MEM e' (G a) }|>) /\
   (! e'. (e',e) IN sTC (happens_before E X') ==>  e' IN viewed_events E q ==> (e',e') IN (X.vo q) )
) ==>
machine_trans P (E,M,F,G,Rg,X) Tau (E',M',F',G',Rg,X')
) /\

(************************************************************************)

( ! P M F G Rg e r v p E E' X X' eo iico mvl.
(clause_name "reg-read" /\
(e = mvl.mvl_event) /\
(iico = mvl.mvl_iico) /\ 
(e.action = Access R (Location_reg p r) v) /\
(Rg p r = SOME (v,eo)) /\
(E'= <| events:= E.events UNION {e}; 
        intra_causality := E.intra_causality UNION {(e',e)|e' IN iico}; 
        atomicity:= E.atomicity |>) /\ 
(X'= X with <| vo := funupd X.vo p (linear_order_extend (X.vo p) e);
               rfmap := case eo of NONE -> X.rfmap
                                || SOME ew -> X.rfmap UNION {(ew,e)} |>) /\
(! e'. (e',e) IN sTC (happens_before E' X') ==>  e' IN viewed_events E' p ==> (e',e') IN X.vo p )
) ==>
machine_trans P (E,M,F,G,Rg,X) (Vis mvl) (E',M,F,G,Rg,X')
) /\
(************************************************************************)

( ! P M F G Rg Rg' e r v p E E' X X' iico mvl.
(clause_name "reg-write" /\
(e = mvl.mvl_event) /\
(iico = mvl.mvl_iico) /\ 
(e.action = Access W (Location_reg p r) v) /\
(Rg' = funupd2 Rg p r (SOME (v,SOME e))) /\
(E'= <| events:= E.events UNION {e}; 
        intra_causality := E.intra_causality UNION {(e',e)|e' IN iico}; 
        atomicity:= E.atomicity |>) /\ 
(X'= X with <| vo := funupd X.vo p (linear_order_extend (X.vo p) e) |>)  /\
(! e'. (e',e) IN sTC (happens_before E' X') ==>  e' IN viewed_events E' p ==> (e',e') IN X.vo p )
) ==>
machine_trans P (E,M,F,G,Rg,X) (Vis mvl) (E',M,F,G,Rg',X')
)` handle e => Raise e;






(* (\******************************************************************************\) *)
(* (\* tjr version with records *\) *)

(* val _ = Hol_datatype ` *)
(* machine_state = <| *)
(*   E: event_structure; *)
(*   X: execution_witness; (\* X.write_serialization is G below *\) *)
(*   M: proc -> location -> (value # (event option)) option; *)
(*   G: address -> event list;                  (\* event reln? *\) *)
(*   Q: proc#proc -> event list                 (\* event reln? *\) *)
(* |> *)
(* `; *)


(* (\**************************************\) *)
(* (\* simple ops on events *\) *)

(* val _ = Define ` *)
(* failwith x (s:string) = x *)
(* `; *)

(* val _ = Define ` *)
(* is_mem_w e = case e.action of *)
(*   Access W (Location_mem a) v -> T *)
(*   || _ -> T *)
(* `; *)

(* val _ = Define ` *)
(* dest_mem_w e = case e.action of *)
(*   Access W (Location_mem a) v -> (a,v) *)
(*   || _ -> failwith ARB "dest_mem_w" *)
(* `; *)


(* (\**************************************\) *)
(* (\* E *\) *)

(* val _ = Define ` *)
(* extend_E E eiico E' =  *)
(*   let (e,iico) = eiico in *)
(*   ~ (e.eiid IN {e''.eiid | e'' IN E.events}) /\ *)

(*   (E' = E with <|  *)
(*     events:=(E.events UNION {e}); *)
(*     intra_causality:=(E.intra_causality UNION {(e',e)|e' IN iico}) *)
(*   |>) *)
(* `; *)


(* (\**************************************\) *)
(* (\* X *\) *)

(* (\* actually derived from initial state - seems easiest to reify this *\) *)
(* (\* FIXME *\) *)

(* val _ = Define ` *)
(* extend_vo vo (p,e) = funupd vo p (linear_order_extend (vo p) e) *)
(* `; *)

(* (\* eo is a possible update to the rfmap *\) *)
(* val _ = Define ` *)
(* extend_X X (p,eo,e) X' = ( *)
(*   X' = X with <| *)
(*     vo:=extend_vo X.vo (p,e); *)
(*     rfmap:=case eo of  *)
(*       NONE -> X.rfmap  *)
(*       || SOME ew -> X.rfmap UNION {(ew,e)} *)
(*   |>) *)
(* `; *)

(* val _ = Define ` *)
(* check_happens_before (E',X') (p,e) =  *)
(*   (! e'. (e',e) IN sTC (happens_before E' X') ==>  e' IN viewed_events E' p ==> (e',e') IN X'.vo p ) *)
(* `; *)


(* (\**************************************\) *)
(* (\* Q *\) *)

(* val _ = Define ` *)
(* writes_pending Q (p,q) a =  *)
(*   FILTER (\ e. let (a',v) = dest_mem_w e in a' = a) (Q(p,q)) *)
(* `; *)

(* val _ = Define ` *)
(* enqueue Q f Q' = ! x. Q' x = ((Q x) ++ (f x)) *)
(* `; *)

(* val _ = Define ` *)
(* dequeue Q f Q' = ! x. Q x = ((f x) ++ (Q' x)) *)
(* `; *)


(* (\**************************************\) *)
(* (\* G *\) *)

(* val _ = Define ` *)
(* extend_G G a e G' = enqueue G (\ a'. if a'=a then [e] else []) G' *)
(* `; *)

(* (\* e has already been seen by G and no older writes pending for q *\) *)
(* val _ = Define ` *)
(* earliest_write G Q (q,e) = ? G0 G1. *)
(*   let (a,v) = dest_mem_w e in *)
(*   (G a = (G0 ++ [e] ++ G1)) /\ *)
(*   (! p' (\* :: P *\). (set (Q(p',q))) INTER (set G0) = {}) *)
(* `; *)

(* (\*  *)

(* when a write is dequeued, it needs to be added to G, unless it is *)
(* already on G, in which case it doesn't need to be added; a write to address a *)
(* on proc q should only be dequeued if there are no G a-earlier writes *)
(* that need to be dequeued for q *)

(* *\) *)





(* (\* want to remove wellformedness constraints elsewhere to make *)
(* soundness obvious and to factor spec  *)

(* check_happens_before (E,X,E',X')  *)

(* x.write_serialization is same as G *)

(* *\) *)


(* val _ = Hol_reln ` *)

(* (\******************************************************************************\) *)
(* ( (\* enqueue mem write *\) *)
(*   ! E E' Q Q' a e iico s0 v mvl. *)
(*   (e = mvl.mvl_event) /\ *)
(*   (iico = mvl.mvl_iico) /\  *)
(*   (e.action = Access W (Location_mem a) v) /\ *)
(*   (extend_E E (e,iico) E') /\ *)
(*   (enqueue Q (\ (p,q). if p=e.iiid.proc then [e] else []) Q')  *)
(*   ==> *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E ;Q:=Q  |>)  *)
(*     (Vis mvl) *)
(*     (s0 with <| E:=E';Q:=Q' |>) *)
(* ) /\ *)


(* (\******************************************************************************\) *)
(* ( (\* mem read *\) *)
(*   ! E E' M Q X X' a e eo iico p s0 v mvl. *)
(*   (e = mvl.mvl_event) /\ *)
(*   (iico = mvl.mvl_iico) /\  *)
(*   (e.action = Access R (Location_mem a) v) /\ *)
(*   (p = e.iiid.proc) /\ *)
(*   (M p (Location_mem a) = SOME(v,eo)) /\ *)
(*   (writes_pending Q (p,p) a = []) /\ *)
(*   (extend_E E (e,iico) E') /\  *)
(*   (extend_X X (p,eo,e) X') /\ *)
(*   check_happens_before (E',X') (p,e) *)
(*   ==> *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E ; X:=X ; M:=M; Q:=Q |>)  *)
(*     (Vis mvl) *)
(*     (s0 with <| E:=E'; X:=X'; M:=M; Q:=Q |>) *)
(* ) /\ *)


(* (\******************************************************************************\) *)
(* ( (\* dequeue mem write 1 *\) *)
(*   ! E G M M' Q Q' X X' a e p q s0 v. *)
(*   (e.action = Access W (Location_mem a) v) /\ *)
(*   (dequeue Q (\ (p',q'). if (p',q') = (p,q) then [e] else []) Q') /\ *)
(*   (earliest_write G Q (q,e)) /\ *)
(*   (extend_X X (p,NONE,e) X') /\ *)
(*   (M' = funupd2 M q (Location_mem a) (SOME(v,SOME e))) /\ *)
(*   check_happens_before (E,X') (q,e) *)
(*   ==>  *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E; X:=X ; Q:=Q ; M:=M ; G:=G |>) *)
(*     Tau *)
(*     (s0 with <| E:=E; X:=X'; Q:=Q'; M:=M'; G:=G |>) *)
(* ) /\ *)


(* (\******************************************************************************\) *)
(* ( (\* dequeue mem write 2 *\) *)
(*   (\* case 2, e not seen in G, so update G *\) *)
(*   ! E G G' M M' Q Q' X X' X'' a e p q s0 v.  *)
(*   (e.action = Access W (Location_mem a) v) /\ *)
(*   (dequeue Q (\ (p',q'). if (p',q') = (p,q) then [e] else []) Q') /\ *)
(*   ~(? e. earliest_write G Q (q,e)) /\ *)
(*   (extend_X X (p,NONE,e) X') /\ *)
(*   (X'' = X' with <| write_serialization:=X'.write_serialization UNION {(e',e) | MEM e' (G a)} |>) /\ *)
(*   (M' = funupd2 M q (Location_mem a) (SOME(v,SOME e))) /\ *)
(*   (extend_G G a e G') /\ *)
(*   check_happens_before (E,X') (q,e) *)
(*   ==> *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E; X:=X  ; Q:=Q ; G:=G ; M:=M  |>) *)
(*     Tau *)
(*     (s0 with <| E:=E; X:=X''; Q:=Q'; G:=G'; M:=M' |>) *)
(* ) /\ *)


(* (\******************************************************************************\) *)
(* ( (\* reg read *\) *)
(*   ! E E' M X X' e eo iico p r s0 v mvl. *)
(*   (e = mvl.mvl_event) /\ *)
(*   (iico = mvl.mvl_iico) /\  *)
(*   (e.action = Access R (Location_reg p r) v) /\ *)
(*   (M p (Location_reg p r) = SOME(v,eo)) /\ *)
(*   (extend_E E (e,iico) E') /\  *)
(*   (extend_X X (p,eo,e) X') /\ *)
(*   check_happens_before (E',X') (p,e) *)
(*   ==> *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E ; X:=X ; M:=M |>)  *)
(*     (Vis mvl) *)
(*     (s0 with <| E:=E'; X:=X'; M:=M |>) *)
(* ) /\ *)


(* (\******************************************************************************\) *)
(* ( (\* reg write *\) *)
(*   ! E E' M M' X X' e iico p r s0 v mvl. *)
(*   (e = mvl.mvl_event) /\ *)
(*   (iico = mvl.mvl_iico) /\  *)
(*   (e.action = Access W (Location_reg p r) v) /\ *)
(*   (extend_E E (e,iico) E') /\  *)
(*   (extend_X X (p,NONE,e) X') /\ *)
(*   (M' = funupd2 M p (Location_reg p r) (SOME(v,SOME e))) /\ *)
(*   check_happens_before (E',X') (p,e) *)
(*   ==> *)
(*   machine_trans P  *)
(*     (s0 with <| E:=E ; X:=X ; M:=M  |>)  *)
(*     (Vis mvl) *)
(*     (s0 with <| E:=E'; X:=X'; M:=M' |>) *)
(* )  *)

(* `; *)

(* wf of pth gives uniqueness *)
val _ = Define `
final_state p s = ? n. (p n,p(n+1)) = (SOME s,NONE)
`;


val _ = export_theory ();


