(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * Acute Systematic Tester Database Handler                       *
 *                                                                *
 * Version:  $Id: runtest_db.ml,v 1.510 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)



(* types *)

type test_name   = string
type family_name = string
type expecting   = string
type test_body = string option

type test_type =
  | Exits of expecting option list
  | CompFail of expecting
  | RunFail of expecting
  | Group of test_name list
  | Compile
  | Empty

type test_spec = {
  name : string;
  file : string;
  mutable doc : string;
  mutable options : string list;
  mutable libs : string list;
  mutable expect : test_type;
  mutable families : family_name list;
  mutable body : test_body
}

exception Multiply_defined_test of string
exception End_of_parse
exception Test_not_found of string

(* constructor *)
let new_test_spec test_name file_name = {
  name     = test_name;
  file     = file_name;
  doc      = "";
  options  = [];
  libs     = [];
  expect   = Empty;
  families = [];
  body     = None
}

(* the databases *)

let test_db   = ref ( Hashtbl.create 127 ) (* mapping from a test_name to test_spec *)
let family_db = ref ( Hashtbl.create 127 ) (* mapping from family_name to a reverse list of test_name *)

let add_to_test_map test_spec =
  if Hashtbl.mem !test_db test_spec.name then
    raise (Multiply_defined_test test_spec.name)
  else
    Hashtbl.add !test_db test_spec.name test_spec

let find_test test_name =
  try
    Hashtbl.find !test_db test_name
  with Not_found -> raise (Test_not_found test_name)

let is_grouped t_name =
  let t = find_test t_name in
  match t.expect with
  | Group _ -> true
  | _ -> false

let add_to_family_map family_name test_spec =
  let test_names =
    begin try Hashtbl.find !family_db family_name with Not_found -> [] end in
  Hashtbl.add !family_db family_name (test_spec.name :: test_names)

let find_family family_name =
  List.rev (Hashtbl.find !family_db family_name)

let add_test test_spec =
  add_to_test_map test_spec;
  List.iter (fun family -> add_to_family_map family test_spec) test_spec.families

(******************************************************************************)
(* PARSING TEST FILES Thing                                                   *)
(******************************************************************************)

exception Declared_but_not_matched_keyword

let keywords =
  [ "DOC"; "FAM"; "OPTS"; "LIBS"; "RET"; "GROUP"; "COMPFAIL"; "RUNFAIL";
    "COMPILE" ]

let rec make_kw_re = function
| []   -> ""
| [t]  -> t
| h::t -> h ^ "\\|" ^ (make_kw_re t)

let new_test_hdr_re = Str.regexp "^ *(\\*\\* *\\([^ ]+\\) *$"
let end_test_hdr_re = Str.regexp "^ *\\*)"
let test_kword_re = Str.regexp ("^ *\\*? *\\(" ^ ( make_kw_re keywords ) ^ "\\) *?\\(.*\\)? *$")

let blank_split = Str.split (Str.regexp " +")

(* we have read the header thing, and now want to either strip blank lines for a
 * collection of tests, or collect the implementation for a single test
 *)
let rec parse_file_test_impl chan test_hdr f_name =
  try
    let s = input_line chan in
    let new_test = Str.string_match new_test_hdr_re s 0 in
    match new_test, test_hdr.body with
    | false, None ->
        if s <> "" then test_hdr.body <- Some s;
        parse_file_test_impl chan test_hdr f_name
    | false, Some b  ->
        test_hdr.body <- Some (b^"\n"^s);
        parse_file_test_impl chan test_hdr f_name
    | true, _ ->
        add_test test_hdr;
        Str.matched_group 1 s
  with End_of_file -> begin
    add_test test_hdr;
    raise End_of_parse
  end

(* we already read a new test name, we are reading the headers *)
let rec parse_file_test chan test_hdr f_name =
  let s = input_line chan in
  match Str.string_match end_test_hdr_re s 0 with
  | false ->
      begin
        if(Str.string_match test_kword_re s 0) then
          let cap () = Str.matched_group 2 s in
          match Str.matched_group 1 s with
          | "DOC"   -> test_hdr.doc <- cap ()
          | "FAM"   -> test_hdr.families <- blank_split (cap ())
          | "OPTS"  -> test_hdr.options <- (blank_split (cap ()))
          | "LIBS"  -> test_hdr.libs <- (blank_split (cap ()))
          | "GROUP" -> test_hdr.expect <- Group (blank_split (cap ()))
          | "RET"   ->
              let l = match test_hdr.expect with Exits l -> l | _ -> [] in
              test_hdr.expect <- Exits (Some (cap ())::l)
          | "COMPFAIL"  -> test_hdr.expect <- CompFail    (cap ())
          | "RUNFAIL"  -> test_hdr.expect <- RunFail    (cap ())
          | "COMPILE"  -> test_hdr.expect <- Compile
          | _ -> raise (Failure "Never Happen !!!")
        end;
        parse_file_test chan test_hdr f_name
  | true ->
      let new_name = parse_file_test_impl chan test_hdr f_name in
      parse_file_test chan (new_test_spec new_name f_name) f_name


(* here you want to strip firts line before the first 'two stars' comment *)
let rec parse_file chan f_name =
  try
    let s = input_line chan in
    match Str.string_match new_test_hdr_re s 0 with
    | true ->
        let new_name = Str.matched_group 1 s in
        parse_file_test chan (new_test_spec new_name f_name) f_name;
    | false ->
        parse_file chan f_name
  with
  | End_of_file | End_of_parse -> close_in chan

(******************************************************************************)
(* RECURSE IN DIRECTORIES                                                     *)
(******************************************************************************)

(* the regexp used to detect test files *)
let test_file_re = Str.regexp "^.*.ml$"

(* verify mtimes to know if database is very old or not *)
let is_db_fresh dir_name =
  let db = dir_name ^ "/tests.db" in
  try (
    let db_mtime = (Unix.stat db).Unix.st_mtime in

    let rec aux dir_h =
      let f_name = dir_name ^ "/" ^ (Unix.readdir dir_h) in
      let stat = Unix.stat f_name in
      let mtime = stat.Unix.st_mtime in
      match stat.Unix.st_kind with
      | Unix.S_REG | Unix.S_LNK ->
          if Str.string_match test_file_re f_name 0 && mtime > db_mtime then
            false
          else
            aux dir_h
      | _ -> aux dir_h
    in

    let dir_h = (Unix.opendir dir_name) in
    try
      let res = aux dir_h in
      Unix.closedir dir_h;
      res
    with End_of_file -> Unix.closedir dir_h; true
  ) with e -> false

(* Read database *)
let read_db dir_name =
  let db = dir_name ^ "/tests.db" in
  let f = Unix.openfile db [ Unix.O_RDONLY ] 0117 in
  let chan = Unix.in_channel_of_descr f in
  let tdb, fdb = input_value chan in
  test_db := tdb;
  family_db := fdb;
  close_in chan

(* Save databases *)
let write_db dir_name =
  let db = dir_name ^ "/tests.db" in
  let f = Unix.openfile db [ Unix.O_WRONLY; Unix.O_CREAT; Unix.O_TRUNC ] 0o664 in
  let chan = Unix.out_channel_of_descr f in
  output_value chan (!test_db, !family_db);
  close_out chan

(* parse a file *)
let fill_database_from_file f_name =
  let file_d = Unix.openfile f_name [ Unix.O_RDONLY ] 0o664 in
  let chan = Unix.in_channel_of_descr file_d in
  parse_file chan f_name

(* list a directory *)
let rec fill_database_from_dir dir_name dir_h =
  try
    let f_name = dir_name ^ "/" ^ (Unix.readdir dir_h) in
    match (Unix.stat f_name).Unix.st_kind with
    | Unix.S_REG | Unix.S_LNK ->
        if ( Str.string_match test_file_re f_name 0 ) then
          fill_database_from_file f_name;
        fill_database_from_dir dir_name dir_h
    | _ ->
        fill_database_from_dir dir_name dir_h
  with End_of_file -> Unix.closedir dir_h

(* wrapper for the above one *)
let fill_database dir_name =
  match is_db_fresh dir_name with
  | true ->
      read_db dir_name
  | false ->
      fill_database_from_dir dir_name (Unix.opendir dir_name);
      write_db dir_name

