(*
 *  hashing/normtypes.ml
 *
 *  Representation and manipulation of normalized types.
 *
 *  (c) Copyright 2005, 2006, Mark R. Shinwell, Rok Strnisa.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Longident
open Path
open Typedtree
open Types

external hash256 : 'a -> string = "hash256"

let initial_env =
  Predef.build_initial_env Env.add_type Env.add_exception Env.empty

type ntype_decl = NTDexternal_abstract of string
                | NTDbuiltin_abstract of string
                | NTDlocal_abstract of string
                | NTDvariant of (string * (ntype list)) list
                | NTDrecord of (string * Asttypes.mutable_flag * ntype) list
                | NTDmanifest of ntype

and ntype = NTvar of int
          | NTunivar of int
          | NTarrow of ntype * ntype
          | NTtuple of ntype list
          | NTctor_param of int
          | NTconstructed of string * (Longident.t list) * (ntype list)
          | NTisorecursive_loop of string
          | NTpoly of int * ntype
          | NTother
          | NTloop

type flat_ntype_entry = FNtyvar of int
                      | FNmyname of Env.t * Longident.t
                      | FNstring of string

let code_for_tyvar numeric_id =
  let name = "_tyrep_" ^ (string_of_int numeric_id) in
  let id = Ident.create_typerep name in
  let ty = Predef.type_typerep in
    { exp_desc = Texp_ident (Pident id, { val_type = ty; val_kind = Val_reg },
                             []);
      exp_loc = Location.none;
      exp_type = ty;
      exp_env = Env.empty }

let code_for_myname env modname =
  let lident = Ldot (modname, "myname") in
  let (path, vd) =
    try Env.lookup_value lident env
    with Not_found ->
    begin
      print_endline
        ("myname lookup failed: wanted " ^ (Longident.to_string lident));
      assert false
    end
  in
  let ty = Predef.type_typerep in
    { exp_desc = Texp_ident (path, vd, []);
      exp_loc = Location.none;
      exp_type = ty;
      exp_env = env }

let code_for_string s =
  { exp_desc = Texp_constant (Asttypes.Const_string s);
    exp_env = Env.empty;
    exp_loc = Location.none;
    exp_type = Ctype.instance (Predef.type_string) }

(* Transform a flat_ntype_entry list into a Typedtree.expression list. *)
let rec flat_list_to_code fl =
  match fl with
    [] -> []
  | (FNtyvar id) :: xs ->
      (code_for_tyvar id) :: (flat_list_to_code xs)
  | (FNmyname (env, lid)) :: xs ->
      (code_for_myname env lid) :: (flat_list_to_code xs)
  | (FNstring s) :: xs ->
      (code_for_string s) :: (flat_list_to_code xs)

(* Optimize a flat_ntype_entry list by concatenating adjacent strings. *)
let rec optimize_flat_list_body acc fl =
  match fl with
    [] -> if acc = "" then [] else [FNstring acc]
  | (FNstring s) :: xs -> optimize_flat_list_body (acc ^ s) xs
  | x :: xs ->
      let rest = optimize_flat_list_body "" xs in
        if acc = "" then
          x :: rest
        else
          (FNstring acc) :: x :: rest

let optimize_flat_list fl = optimize_flat_list_body "" fl

let string_of_bool_2 = function true -> "T" | false -> "F"

(* Transform a normalized type tree into a value of type
   flat_ntype_entry list.  Abbreviations emitted:

        E       constructed
        F       function
        T       tuple
        C       constructor parameter (de Bruijned)
        I       isorecursive loop
        Y       polytype
        O       other
        P       loop
        U       univar
*)
let rec transform_ntype env nty =
  match nty with
    NTvar id -> [FNtyvar id]
  | NTunivar dbi -> [FNstring ("U" ^ (string_of_int dbi))]
  | NTarrow (nty1, nty2) ->
      transform_ntype_list env "F" [nty1; nty2]
  | NTtuple ntys ->
      transform_ntype_list env "T" ntys
  | NTctor_param dbi ->
      [FNstring ("C" ^ (string_of_int dbi))]
  | NTpoly (num_univars, body) ->
      (FNstring ("Y" ^ (string_of_int num_univars) ^ "_")) ::
        transform_ntype env body
  | NTconstructed (hash_value, myname_paths, ntys) ->
    begin
      let myname_codes = List.map (fun path -> FNmyname (env, path))
                                 myname_paths
      in
      let components = transform_ntype_list env "" ntys in
        (FNstring "E") ::
        (FNstring hash_value) ::
        myname_codes @
        components
    end
    (*
  | NTconstructed (NTDbuiltin_abstract name, ntys) ->
      transform_ntype_list env ("B" ^ name) ntys
  | NTconstructed (NTDlocal_abstract name, ntys) ->
      transform_ntype_list env ("L" ^ name) ntys
  | NTconstructed (NTDmanifest nty, ntys) ->
      transform_ntype_list env "M" (nty :: ntys)
  | NTconstructed (NTDvariant ctor_types, args) ->
      let args' = transform_ntype_list env "" args in
      let trans acc (name, types) =
        (transform_ntype_list env name types) :: [FNstring "|"] :: acc in
      let name_code = FNstring "V" in
      name_code :: (List.flatten (List.fold_left trans [args'] ctor_types))*)
  | NTisorecursive_loop name ->
      [FNstring ("I" ^ name)]
(*  | NTconstructed (NTDprevious_declaration name, args) ->
      transform_ntype_list env ("U" ^ name) args
  | NTconstructed (NTDrecord field_info, args) ->
      let args' = transform_ntype_list env "" args in
      let trans acc (name, mut, typ) =
        let mut_string = string_of_bool_2 (mut = Asttypes.Mutable) in
          (transform_ntype_list env (name ^ mut_string ^ "|") [typ]) :: acc in
      let name_code = FNstring "R" in
      name_code :: (List.flatten (List.fold_left trans [args'] field_info))*)
  | NTother -> [FNstring "O"]
  | NTloop -> [FNstring "P"]

and transform_ntype_list_helper env prefix ntys =
  match ntys with
    [] -> []
  | nty::ntys ->
      let components = optimize_flat_list (transform_ntype env nty) in
      let o_paren = FNstring "(" in
      let c_paren = FNstring ")" in
      let rest = transform_ntype_list_helper env prefix ntys in
        (* FIXME optimize further here for the parentheses, in a similar
           way to in transform_ntype_list below. *)
        List.append (o_paren :: components) (c_paren :: rest)

and transform_ntype_list env prefix ntys =
  let components = transform_ntype_list_helper env prefix ntys in
    (* FIXME optimize this out by adding the prefix directly onto the
       first string during generation of the components list. *)
    if prefix <> "" then
      (FNstring prefix) :: components
    else
      components

(* Transform a normalized type tree into a Typedtree expression that
   will evaluate to the corresponding runtime type representation block. *)
let code_for_ntype env nty =
  let flat_list = optimize_flat_list (transform_ntype env nty) in
  let components = flat_list_to_code flat_list in
  (* the type of the tuple we are forming might not actually be known
     now, since there might be __tyrep references to other type
     representation blocks whose structure we don't know.  So we just
     mark every component as being of type "typerep" -- hopefully this
     will work ok. *)
  let tys = List.map (fun _ -> Predef.type_typerep) components in
    { exp_desc = Texp_tuple components;
      exp_loc = Location.none;
      exp_env = env;
      exp_type = Ctype.newty (Ttuple tys) }

module TypeSet = Set.Make (struct type t = Types.type_expr
                                  let compare ty1 ty2 =
                                    if ty1 == ty2 then 0 else -1
                           end)

let type_set = ref TypeSet.empty
let initialize_type_set () = type_set := TypeSet.empty
let record_type ty = type_set := TypeSet.add ty !type_set
let erase_type ty = type_set := TypeSet.remove ty !type_set
let seen_type ty = TypeSet.mem ty !type_set

module TypeDeclCache = Map.Make (struct type t = type_declaration
                                    let compare d1 d2 =
                                      if d1 == d2 then 0 else -1 end)

let type_decl_cache = ref TypeDeclCache.empty
let cache_type_decl ty nty =
  type_decl_cache := TypeDeclCache.add ty nty !type_decl_cache
let search_type_decl_cache ty = TypeDeclCache.find ty !type_decl_cache

module TypeDeclStack = Set.Make (struct type t = Path.t
                                        let compare = compare end)

let type_declaration_stack = ref TypeDeclStack.empty
let initialize_type_declaration_stack () =
  type_declaration_stack := TypeDeclStack.empty
let record_type_declaration td =
  type_declaration_stack := TypeDeclStack.add td !type_declaration_stack
let erase_type_declaration td =
  type_declaration_stack := TypeDeclStack.remove td !type_declaration_stack
let already_examining_type_declaration td =
  TypeDeclStack.mem td !type_declaration_stack

let type_to_string ty' =
  let ty = Ctype.repr ty' in
  let buf = Buffer.create 40 in
  let formatter = Format.formatter_of_buffer buf in
  let _ = Printtyp.type_scheme_idents formatter ty in
  let _ = Format.pp_print_flush formatter () in
    Buffer.contents buf

let print_warning ty =
    print_endline "warning: normalizing the following type is not supported:";
    print_endline ("         " ^ (type_to_string ty));
    print_endline ("         all types of this kind will receive the same" ^
                  " type representation.")

let ctor_param_map = ref []
let univar_map = ref []

let rec normalize_type_rec env ty =
  let ty = Ctype.repr ty in
  if seen_type ty then
  begin
    print_endline "warning: may be hitting type normalization bug\n";
    (NTloop, []) (* FIXME probably isn't correct: don't
              we need a de Bruijn number or something
              to identify which loop it is? *)
  end
  else
  begin
  record_type ty;
  let nty_and_deps = match ty.desc with
    Tvar ->
      (* check if this type variable corresponds to a parameter of a
         type declaration that is currently being normalized.  If it does
         then we must emit a de Bruijn parameter rather than a value
         identifier. *)
      begin try
        (NTctor_param (List.assoc ty.id !ctor_param_map), [])
      with Not_found -> (NTvar ty.id, [])
      end
  | Tunivar ->
      begin try
        (NTunivar (List.assoc ty.id !univar_map), [])
      with Not_found -> (NTunivar 0, []) (* assert false (* unbound univar *)*)
      (* FIXME XXX  -- see r5d_record_poly *)
      end
  | Tpoly (body, params) ->
      (* univars are treated up to alpha *)
      let saved_univar_map = !univar_map in
      (* FIXME no idea if the following should have true or false... *)
      let fresh_tyvars, body = Ctype.instance_poly true params body in
      (* allocate de Bruijn indices for the fresh type variables
         and save them in an association list *)
      let next_index = ref 0 in
        List.iter (fun tyvar ->
                     univar_map :=
                       (tyvar.id, !next_index) :: !univar_map;
                     next_index := !next_index + 1) fresh_tyvars;
        let nty, external_deps = normalize_type_rec env body in
          univar_map := saved_univar_map;
          (NTpoly (List.length fresh_tyvars, nty), external_deps)
  | Tarrow (_, ty1, ty2, _) ->
      let nty1, external_deps1 = normalize_type_rec env ty1 in
      let nty2, external_deps2 = normalize_type_rec env ty2 in
        (NTarrow (nty1, nty2), external_deps1 @ external_deps2)
  | Ttuple tys ->
      (* this reverses the order of the tuple, but that doesn't matter. *)
      let ntys, external_deps =
        List.fold_left (fun (nty_acc, deps_acc) -> fun ty ->
                          let nty, deps = normalize_type_rec env ty in
                            (nty :: nty_acc, deps @ deps_acc)) ([], []) tys
      in
        (NTtuple ntys, external_deps)
  | Tconstr (path, args, _) ->
    begin
      (* reverses order of ctor args, but of no consequence *)
(*      print_string ("constructed type: path = " ^ (Path.name path) ^ "\n");*)
      let nargs, external_deps' =
        List.fold_left (fun (nty_acc, deps_acc) -> fun ty ->
                          let nty, deps = normalize_type_rec env ty in
                            (nty::nty_acc, deps @ deps_acc)) ([], []) args in
      begin try
        let type_decl = Env.find_type path env in
        if already_examining_type_declaration path (* or type_decl?? *) then
        begin
        (*  print_string "this is a recursive occurrence\n"; flush stdout;*)
          (NTisorecursive_loop (Path.name path (* or just RHS component? *)),
                                [])
        end
        else
        begin
          try
            let (nty, external_deps) = search_type_decl_cache type_decl in
            let external_deps'' = external_deps @ external_deps' in
           (* print_endline "seen this type decl before\n";*)
            (NTconstructed(nty, external_deps'', nargs), external_deps'')
(*            NTconstructed(NTDprevious_declaration (Path.name path), nargs)*)
(*            NTconstructed (search_type_decl_cache type_decl, nargs)*)
          with Not_found ->
          begin
            record_type_declaration path;
            let nty, external_deps = match type_decl.type_manifest with
              None ->
                begin match type_decl.type_kind with
                  Type_abstract ->
                    (* Whilst we're supposed to be working with any type
                       parameters up to alpha-conversion, we can actually
                       just not bother in this case since the parameters
                       never end up in the normalized tree (the abstract
                       type is just referenced by its path). *)
                    if List.mem path Predef.builtin_paths then
                    begin
                        (NTDbuiltin_abstract (Path.name path), [])
                    end
                    else
                    begin
                      begin match path with
                        Pident id ->
                          (NTDlocal_abstract (Ident.name id), [])
                      | Pdot _ ->
                          let (top_module, path') = Path.split path in
                            begin match path' with
                              None -> assert false
                            | Some path' ->
                              begin
                                (*print_endline ("top_module is " ^
                                  (Longident.to_string top_module));*)
                                (NTDexternal_abstract path',
                                 [top_module])
                              end
                            end
                        (* I don't think the case below can ever arise,
                           since Papply (p1, p2) as an entire path corresponds
                           to a module, not a type. *)
                      | Papply (p1, p2) -> assert false
                      end
                    end
                | _ ->
                  (* If the type declaration is that of a variant or
                     record type, we need to work with the type parameters
                     modulo alpha conversion. *)
                  let saved_ctor_param_map = !ctor_param_map in
                  let nty = begin
                  (* allocate fresh type variables for the type parameters *)
                  let fresh_tyvars =
                    List.map (fun _ -> Ctype.newvar ()) type_decl.type_params
                  in
                  (* allocate de Bruijn indices for the fresh type variables
                     and save them in an association list *)
                  let next_index = ref 0 in
                    List.iter (fun tyvar ->
                                 (*print_endline ("tyvar id " ^
                                   string_of_int tyvar.id);
                                 print_endline ("db index " ^
                                   string_of_int !next_index);*)
                                 ctor_param_map :=
                                   (tyvar.id, !next_index) :: !ctor_param_map;
                                 next_index := !next_index + 1) fresh_tyvars;
                  match type_decl.type_kind with
                    Type_variant (ctor_defs, _) ->
                      (* See Type_record case (next) for explanation of
                         sorting *)
                      (* normalize the constructor declarations *)
                      (* reverses order of ctors and ctor args,
                         but of no consequence *)
                      let ntype_decl, external_deps =
                        List.fold_left (
                          fun (nty_acc, deps_acc) (ctor, ctor_args) ->
                            let ntys, deps =
                              List.fold_left (
                                 (* before normalizing the argument type we
                                    must substitute any type variables in
                                    type_decl.type_params for the new ones
                                    in fresh_tyvars... *)
                                 fun (nty_acc, deps_acc) arg_ty ->
                                   let ty = Ctype.apply env
                                                        type_decl.type_params
                                                        arg_ty fresh_tyvars
                                   in
                                   let nty, deps =
                                     normalize_type_rec env ty
                                   in
                                     (nty::nty_acc, deps @ deps_acc))
                                ([], []) ctor_args
                          in
                            ((ctor, ntys) :: nty_acc, deps @ deps_acc)
                        ) ([], []) ctor_defs in
                      (*
                      let sorted_ntype_decl, sorted_external_deps =
                        let comb = List.combine ntype_decl external_deps in
                        let comb_sorted =
                          let f ((l1,_), _) ((l2,_), _) =
                            String.compare l1 l2 in
                          List.sort f comb in
                          comb in
                        List.split comb_sorted in
                      (NTDvariant sorted_ntype_decl,
                        List.flatten sorted_external_deps)
                      *)
                      (NTDvariant ntype_decl, external_deps)
                  | Type_record (fields, _, _) ->
                      (* reverses order of fields, but of no consequence *)
                      let ntype_decl, external_deps =
                        List.fold_left (
                          fun (nty_acc, deps_acc) (label, mut_flag, field_ty) ->
                            (* substitution here is as per the variant case
                               above *)
                            let ty = Ctype.apply env type_decl.type_params
                              field_ty fresh_tyvars in
                            let nty, deps = normalize_type_rec env ty in
                            ((label, mut_flag, nty) :: nty_acc, deps @ deps_acc)
                            (* Cons deps onto deps_acc instead of append
                             * because we must maintain correspondance with
                             * normalised types. We flatten deps at end. *)
                          ) ([], []) fields in
                      (*
                      let sorted_ntype_decl, sorted_external_deps =
                        (* Both the normalised type declarations AND the
                         * associated external dependencies must be sorted
                         * according to record label value. Therefore we
                         * zip the two lists together prior to sorting. *)
                        let comb = List.combine ntype_decl external_deps in
                        let comb_sorted =
                          let f ((l1,_,_), _) ((l2,_,_), _) =
                            String.compare l1 l2 in
                            List.sort f comb in
                          comb in
                        List.split comb_sorted in
                      (NTDrecord sorted_ntype_decl,
                        List.flatten sorted_external_deps)
                      *)
                      (NTDrecord ntype_decl, external_deps)
                  | Type_abstract -> assert false
                  end
                in
                  ctor_param_map := saved_ctor_param_map;
                  nty
              end
            | Some ty ->
              (* FIXME all this allocation of de Bruijn stuff ought to
                 be shared between this and the previous cases above
                 (just as in Normtypedecl). *)
              let saved_ctor_param_map = !ctor_param_map in
              let nty, external_deps = begin
                (* allocate fresh type variables for the type parameters *)
                let fresh_tyvars =
                  List.map (fun _ -> Ctype.newvar ()) type_decl.type_params
                in
                (* allocate de Bruijn indices for the fresh type variables
                   and save them in an association list *)
                let next_index = ref 0 in
                  List.iter (fun tyvar ->
                                 (*print_endline ("tyvar id " ^
                                   string_of_int tyvar.id);
                                 print_endline ("db index " ^
                                   string_of_int !next_index);*)
                               ctor_param_map :=
                                 (tyvar.id, !next_index) :: !ctor_param_map;
                               next_index := !next_index + 1) fresh_tyvars;
                  let ty = Ctype.apply env type_decl.type_params
                                       ty fresh_tyvars
                  in
                    normalize_type_rec env ty
                end
              in
                ctor_param_map := saved_ctor_param_map;
                (NTDmanifest nty, external_deps)
            in
            let nty_hash = hash256 nty in
              erase_type_declaration path;
              cache_type_decl type_decl (nty_hash, external_deps);
              (NTconstructed (nty_hash, external_deps, nargs),
               external_deps)
          end
        end
      with Not_found ->
        begin
          (* FIXME turn into proper compiler error *)
          print_string ("The following type is unbound: " ^
                        (Path.name path) ^ "\n");
          print_string "Ensure that the compiled interface file defining\n";
          print_string "this type is on the search path (-I option).\n";
          flush stdout;
          assert false
        end
      end
    end
  | Tlink ty -> normalize_type_rec env ty
  | Tsubst ty -> normalize_type_rec env ty
  | _ -> print_warning ty; (NTother, [])
  in
    erase_type ty;
    nty_and_deps
  end
  
module TypeCache = Map.Make (struct type t = Types.type_expr
                                    let compare ty1 ty2 =
                                      if ty1 == ty2 then 0 else -1
                             end)

let type_cache = ref TypeCache.empty
let cache_type ty code = type_cache := TypeCache.add ty code !type_cache
let search_type_cache ty = TypeCache.find ty !type_cache

let rec normalize_type env ty =
  match ty.desc with
    Tvar -> code_for_tyvar ty.id
  | Tlink ty | Tsubst ty -> normalize_type env ty
  | _ ->
    begin
      try search_type_cache ty
      with Not_found ->
      begin
        ctor_param_map := [];
        univar_map := [];
        type_decl_cache := TypeDeclCache.empty;
        initialize_type_declaration_stack ();
        initialize_type_set ();
        let nty, _ = normalize_type_rec env ty in
        assert (List.length !ctor_param_map = 0);
(*        print_endline "about to run code_for_ntype";*)
        let code = code_for_ntype env nty in
        begin
          cache_type ty code;
  (*        print_endline "\n\n";*)
          code
        end
      end
    end

