(*
 *  hashing/normtypedecl.ml
 *
 *  Normalization of type declarations.
 *
 *  (c) Copyright 2006, Mark R. Shinwell.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Path
open Types

(* Normalized type declarations. *)
type ntype_decl = NTDabstract of string
                | NTDvariant of (string * (ntype list)) list
                | NTDrecord of (string * Asttypes.mutable_flag * ntype) list
                | NTDabbreviation of ntype

(* Normalized types. *)
and ntype = NTarrow of ntype * ntype
          | NTtuple of ntype list
          | NTctor_param of int
          | NTconstructed of type_constructor_info * (ntype list)
          | NTunsupported
          | NTvar of int
          | NTunivar of int
          | NTpoly of int * ntype

(* Information about type constructors used within normalized types. *)
and type_constructor_info =
  TCabbreviation of ntype                   (* abbreviation *)
| TCbuiltin of string                       (* built-in abstract tycon *)
| TClocal of string                         (* abstract tycon in same module
                                               or superstructure *)
| TCexternal of string * string             (* external abstract tycon *)
| TCbeing_defined of string                 (* (recursive) reference to
                                               a tycon being defined in
                                               the current type
                                               declaration(s) *)
| TCthrough_functor of int * string         (* something like H.t where
                                               H is an in-scope functor
                                               argument *)

(* Debugging: print a value of type string * ntype_decl. *)
let rec print_ntype_decl (id, ndecl) =
  print_string ("type " ^ id ^ " = ");
  match ndecl with
    NTDabstract s -> print_string (s ^ "  [abstract]\n")
  | NTDvariant ctors ->
    begin
      print_string "\n";
      let first = ref true in
      List.iter (fun (ctor_name, ctor_args) ->
                   (if !first then print_string ("  " ^ ctor_name)
                              else print_string ("| " ^ ctor_name));
                   first := false;
                   (if ctor_args <> [] then print_string " of ");
                   let rec f args =
                     match args with
                       [] -> print_string "\n"
                     | [nty] -> (print_ntype nty; f [])
                     | nty::ntys ->
                         (print_ntype nty; print_string " * "; f ntys)
                   in f ctor_args) ctors
    end
  | NTDrecord fields ->
    begin
      print_string "{";
      List.iter (fun (field_name, mut_flag, field_nty) ->
                   let mut =
                     (if mut_flag = Asttypes.Mutable then "mutable" else "") in
                   print_string ("\n  " ^ mut ^ " " ^ field_name ^ " : ");
                   print_ntype field_nty) fields;
      print_string " }\n";
    end
  | NTDabbreviation nty ->
    begin
      print_ntype nty;
      print_string "  [type abbreviation]\n"
    end

and constructed_prec = 10
and tuple_prec = 9
and arrow_prec = 8
and start_prec = 0

(* Debugging: print a value of type ntype. *)
and print_ntype_body previous_prec assoc_matches_branch nty =
  match nty with
    NTctor_param i -> print_string ("[ctor deBruijn-" ^ (string_of_int i) ^ "]")
  | NTvar i -> print_string ("[tyvar deBruijn-" ^ (string_of_int i) ^ "]")
  | NTunivar i -> print_string ("[tyvar deBruijn-" ^ (string_of_int i) ^ "]")
  | NTarrow (nty1, nty2) ->
    begin
      brackets_helper previous_prec arrow_prec assoc_matches_branch
        (fun () -> print_ntype_body arrow_prec false nty1);
      print_string " -> ";
      brackets_helper previous_prec arrow_prec assoc_matches_branch
        (fun () -> print_ntype_body arrow_prec true nty2);
    end
  | NTtuple ntys ->
    let rec f ntys =
      match ntys with
        [] -> ()
      | [nty] -> print_ntype_body tuple_prec false nty
      | nty::ntys ->
        begin
          brackets_helper previous_prec tuple_prec assoc_matches_branch
            (fun () -> print_ntype_body tuple_prec false nty);
          print_string " * ";
          f ntys
        end
    in f ntys
  | NTconstructed (tci, []) -> print_tycon_info tci
  | NTconstructed (tci, [arg]) ->
    begin
      print_ntype_body constructed_prec false arg;
      print_string " ";
      print_tycon_info tci
    end
  | NTconstructed (tci, args) ->
    begin
      print_string "(";
      let rec f args =
        match args with
          [] -> ()
        | [arg] -> print_ntype_body constructed_prec true arg
        | arg::args ->
          begin
            print_ntype_body constructed_prec true arg;
            print_string ", ";
            f args
          end
      in
      f args;
      print_string ") ";
      print_tycon_info tci
    end
  | NTunsupported -> print_string "[unsupported]"
  | NTpoly _ -> print_string "[poly]"

and brackets_helper previous_prec current_prec assoc_matches_branch s =
  (* helper function to prevent extraneous emission of parentheses *)
  if (not assoc_matches_branch) && (current_prec <= previous_prec) then
  begin
    print_string "(";
    s ();
    print_string ")"
  end
  else
    s ()

and print_ntype nty = print_ntype_body start_prec true nty

(* Debugging: print a value of type type_constructor_info. *)
and print_tycon_info tci =
  match tci with
    TCabbreviation nty ->
    begin
      print_string "abbreviation(";
      print_ntype nty;
      print_string ")";
    end
  | TCbuiltin name ->
      print_string (name ^ "[built-in]")
  | TClocal name ->
      print_string (name ^ "[local]")
  | TCexternal (top_module, rest) ->
      print_string (top_module ^ "." ^ rest ^ "[external]")
  | TCbeing_defined name -> print_string (name ^ "[being defined]")
  | TCthrough_functor (dbi, path) ->
      print_string "<through functor>" (* FIXME *)

(* Map used for de Bruijn indices (see below). *)
let ctor_param_map = ref []
let univar_map = ref []

(* List of module long identifiers whose mynames the current
   normalized type declaration depends on. *)
let myname_dependencies = ref []

(* Map from type variable identifiers to stamps
   (basically de Bruijn indices for type variables).  This map can
   be reset by calling initialize (). *)
let tyvar_map = ref []
let next_tyvar_stamp = ref 0

(* The worker function for normalization of types. *)
let rec process_type_body local_ids defined_ids (functor_args : (Ident.t * int) list) env ty =
  match ty.desc with
  (* type variables *)
    Tvar ->
    begin
      (* first see if it's a constructor parameter that's been assigned
         a de Bruijn index *)
      try NTctor_param (List.assoc ty.id !ctor_param_map)
      with Not_found ->
      begin
        (* next see if it's a type variable that's not a constructor
           parameter but has been seen before *)
        try NTvar (List.assoc ty.id !tyvar_map)
        with Not_found ->
          (* not seen before => allocate it a new index *)
          begin
            let my_stamp = !next_tyvar_stamp in
              next_tyvar_stamp := my_stamp + 1;
              tyvar_map := (ty.id, my_stamp) :: !tyvar_map;
              NTvar my_stamp
          end
      end
    end

  (* univars *)
  | Tunivar ->
    begin
      try NTunivar (List.assoc ty.id !univar_map)
      with Not_found -> NTunivar 0 (* assert false (* unbound univar *)*)
      (* FIXME XXX  -- see r5d_record_poly *)
    end

  (* polytypes *)
  | Tpoly (body, params) ->
    (* univars are treated up to alpha *)
    let saved_univar_map = !univar_map in
    (* FIXME no idea if the following should have true or false... *)
    let fresh_tyvars, body = Ctype.instance_poly true params body in
    (* allocate de Bruijn indices for the fresh type variables
       and save them in an association list *)
    let next_index = ref 0 in
      List.iter (fun tyvar ->
                   univar_map :=
                     (tyvar.id, !next_index) :: !univar_map;
                   next_index := !next_index + 1) fresh_tyvars;
      let nty =
        process_type_body local_ids defined_ids functor_args env body
      in
        univar_map := saved_univar_map;
        NTpoly (List.length fresh_tyvars, nty)

  (* function types *)
  | Tarrow (_, ty1, ty2, _) ->
      NTarrow (process_type_body local_ids defined_ids functor_args env ty1,
               process_type_body local_ids defined_ids functor_args env ty2)

  (* tuple types *)
  | Ttuple tys ->
    NTtuple (List.map (process_type_body local_ids defined_ids functor_args env) tys)

  (* constructed types *)
  | Tconstr (path, args, _) ->
  (*
      print_endline ("trying to find " ^ (Path.name path));
      print_endline "local_ids is:";
      List.iter (fun id -> print_endline ("  " ^ id))
        local_ids;*)
      (* normalize the arguments to the constructed type *)
      let nargs = List.map (process_type_body local_ids defined_ids functor_args env) args in
      (* determine if the type constructor is in fact one that we
         are defining in the type declaration(s) being normalized *)
      begin match path with
          Pident id when List.mem id defined_ids ->
              (* the type constructor is one being defined, so just
                 record that and don't go round in circles *)
            NTconstructed (TCbeing_defined (Ident.name id), nargs)
        | _ when List.mem (Path.name path) local_ids ->
            (* the type constructor is one known to be defined earlier
               in the current module or in a superstructure
               (and thus might not be in the environment).
               This is used by Normtrans.tr_sig_top. *)
            NTconstructed (TClocal (Path.name path), nargs)
        | _ ->
          (* the type constructor is not one that is currently being
             defined.  Unlike in Normtypes, we don't need to pull in the
             declaration of the type constructor, but we do need to
             correctly identify if it is going to require a "myname"
             reference to be emitted.
             
             This part is basically the same logic as used in part of
             Normtypes. *)
        begin try
          (* start by looking up the type declaration *)
          let type_decl = Env.find_type path env in
          (* assign de Bruijn indices for any type parameters, since the
             declaration needs to be treated up to alpha (sigh) *)
          let saved_ctor_param_map = !ctor_param_map in
          let fresh_tyvars =
            List.map (fun _ -> Ctype.newvar ()) type_decl.type_params in
          let next_index = ref 0 in
            List.iter (fun tyvar ->
                         ctor_param_map :=
                           (tyvar.id, !next_index) :: !ctor_param_map;
                         next_index := !next_index + 1) fresh_tyvars;
            let ndecl =
              (* determine if it is an abbreviation; if so, follow it *)
              begin
              match type_decl.type_manifest with
                None ->
                  (* not an abbreviation -- determine whether it is
                     an abstract type defined in another module (that is
                     not a superstructure of the current module). *)
                  begin
                    (* first see if it's a built-in abstract type *)
                    if List.mem path Predef.builtin_paths then
                      TCbuiltin (Path.name path)
                    else
                      (* not a built-in, so examine its path to determine
                         which module it comes from *)
                      begin match path with
                        Pident id ->
                          (* it comes from the current module, or a
                             superstructure.  No myname references need
                             to be emitted since prefix hashing will ensure
                             that the declaration of the type will be
                             included in the overall hash. *)
                          TClocal (Path.name path)
                      | Pdot _ ->
                          (* split the path at the leftmost dot *)
                          let (top_module, path') = Path.split2 path in
                            begin match path' with
                              None -> assert false
                            | Some path' ->
                              begin
                                (* we've now identified that it is an
                                   external abstract type; this will cause
                                   a "myname" reference to be emitted. *)

                                (* FIXME
                                   This also causes myname references to
                                   be emitted for types defined in
                                   substructures of the current module that
                                   are previous to the current declaration
                                   being normalized.  This doesn't match
                                   what happens if such a type is listed
                                   in local_ids... *)
                                myname_dependencies :=
                                  top_module :: !myname_dependencies;
                                TCexternal (
                                  Path.name top_module, path')
                              end
                            end
                          (* I don't think the case below can ever arise,
                             since Papply (p1, p2) as an entire path
                             corresponds to a module, not a type. *)
                      | Papply (p1, p2) -> assert false
                      end
                  end
              | Some ty ->
                  (* abbreviation => expand it, not forgetting to turn
                     type parameters into fresh type variables first so
                     that they correspond to the keys in the de Bruijn
                     index map. *)
                  let ty = Ctype.apply env type_decl.type_params
                                       ty fresh_tyvars
                  in
                    TCabbreviation
                      (process_type_body local_ids defined_ids functor_args env ty)
              end
            in
            begin
              ctor_param_map := saved_ctor_param_map;
              NTconstructed (ndecl, nargs)
            end
          with Not_found ->
          (* check if we have something like H.t where H is the argument
             of a functor that is currently in scope *)
          begin
            match path with
              Pdot _ ->
                let (top_module, path') = Path.split2 path in
                  begin match path' with
                    None -> assert false
                  | Some path' ->
                    begin
                      try
                        let (_, dbi) =
                          List.find (fun (p, _) ->
                          p = Path.head top_module)
                                    functor_args
                        in
                          NTconstructed (
                            TCthrough_functor (dbi, path'), nargs)
                      with Not_found -> 
                      begin
                        print_endline ("warning: cannot find type declaration for " ^
                                      (Path.name path));

            (* FIXME: this assert triggers in addlabels.ml on SMap.t --
               something to do with the use of "include". *)
            (*                          assert false*)
            NTunsupported
                      end
                    end
                  end
            | _ ->
            print_endline ("warning: cannot find type declaration for " ^
                           (Path.name path));
            
            NTunsupported
          end
        end (* begin try ... *)
      end (* match path with ... *)

  (* links to other types *)
  | (Tlink ty | Tsubst ty) -> process_type_body local_ids defined_ids functor_args env ty

  (* other nasties *)
  | _ -> Normtypes.print_warning ty; NTunsupported

(* Normalize a type that has occurred in the type of a record field
   or in a constructor's argument type. *)
let process_type = process_type_body

(* Normalize an abstract type declaration. *)
let process_abstract_type_declaration id env type_decl =
  assert (type_decl.type_kind = Type_abstract);
  assert (type_decl.type_manifest = None );
  (* in this case, the abstract type being defined is not an abbreviation,
     so it is just something like "type t".  Easy. *)
    NTDabstract (Ident.name id)

(* Normalize a concrete type declaration. *)
let process_concrete_type_declaration local_ids defined_ids functor_args
                                      fresh_tyvars env type_decl =
  (* See if the type declaration is that of a variant or that of
     a record type.  The following is basically the same as used in
     Normtypes. *)
  match type_decl.type_kind with
    Type_variant (ctor_defs, _) ->
      (* normalize the constructor declarations *)
      let ntype_decl =
        List.map (
          fun (ctor, ctor_args) ->
                (ctor, List.map (
                   (* before normalizing the argument type we
                      must substitute any type variables in
                      type_decl.type_params for the new ones
                      in fresh_tyvars... *)
                   fun arg_ty ->
                     let ty = Ctype.apply env type_decl.type_params
                                          arg_ty fresh_tyvars
                     in
                       process_type local_ids defined_ids
                                    functor_args env ty) ctor_args)
          ) ctor_defs
      in
        NTDvariant ntype_decl
  | Type_record (fields, _, _) ->
      let ntype_decl =
        List.map (
          fun (label, mut_flag, field_ty) ->
            (* substitution here is as per the variant case
               above *)
            let ty = Ctype.apply env type_decl.type_params
                                 field_ty fresh_tyvars
            in
              (label, mut_flag, process_type local_ids defined_ids
                                             functor_args env ty)
          ) fields
      in
        NTDrecord ntype_decl
  | _ -> assert false

(* Normalize a single type declaration. *)
let process_type_declaration local_ids defined_ids functor_args
                             id env type_decl =
  ctor_param_map := [];
  univar_map := [];
  (* assign de Bruijn indices to any type parameters.  We do this by
     picking as many fresh type variables as there are type parameters and
     associating them to de Bruijn indices, starting from zero.  Then
     before examining any type occurring in the type declaration we
     substitute the actual type parameters for these fresh variables.
     Then, during type normalization of such a type, we will be able to
     identify that such variables are actually type parameters being
     treated up to alpha. *)
  let fresh_tyvars =
    List.map (fun _ -> Ctype.newvar ()) type_decl.type_params in
  let next_index = ref 0 in
    List.iter (fun tyvar ->
                 ctor_param_map := (tyvar.id, !next_index) :: !ctor_param_map;
                 next_index := !next_index + 1) fresh_tyvars;
    (* find out if the type declaration is in fact just the declaration
       of a type abbreviation *)
    match type_decl.type_manifest with
      None ->
      (* not an abbreviation *)
      begin
        match type_decl.type_kind with
          Type_abstract -> process_abstract_type_declaration id env type_decl
        | _ ->
          process_concrete_type_declaration local_ids defined_ids
                                            functor_args fresh_tyvars
                                            env type_decl
      end
    | Some ty ->
      (* before traversing the abbreviation, don't forget to substitute
         out actual type parameters for fresh ones! *)
      let ty = Ctype.apply env type_decl.type_params ty fresh_tyvars in
        NTDabbreviation (process_type local_ids defined_ids functor_args
                                      env ty)

(* Normalize a list of type declarations.
   The argument is a list of (identifier, type declaration) pairs and
   has the same type as the constructor argument of Typedtree.Tstr_type. *)
let normalize_type_declarations functor_args env id_decl_list =
  myname_dependencies := [];
  (* collect the names of the types being defined *)
  let defined_ids = fst (List.split id_decl_list) in
  (* normalize each declaration *)
  (* don't forget: compute this lot before examining !myname_dependencies *)
  let ndecls = List.map (fun (id, decl) ->
                (if !Clflags.dnormtrans then
                print_endline ("normalizing type decl for " ^
                               (Ident.name id)));
                let ndecl = process_type_declaration [] defined_ids functor_args
                                                     id env decl
                in
                begin
                  (if !Clflags.dnormtrans then
                     print_ntype_decl (Ident.name id, ndecl));
                  (Ident.name id, ndecl)
                end
             ) id_decl_list
  in (ndecls, !myname_dependencies)

(* Normalize a type declaration, treating certain type constructors
   as being defined in the current module (or a superstructure of it). *)
let normalize_type_declaration_defined env local_ids functor_args (id, decl) =
  myname_dependencies := [];
  let id_decl_list = [(id, decl)] in
  (* collect the names of the types being defined *)
  let defined_ids = fst (List.split id_decl_list) in
  (* normalize each declaration *)
  (* don't forget: compute this lot before examining !myname_dependencies *)
  let ndecls = List.map (fun (id, decl) ->
                (if !Clflags.dnormtrans then
                print_endline ("normalizing type decl for " ^
                               (Ident.name id)));
                let ndecl = process_type_declaration local_ids defined_ids
                                                     functor_args
                                                     id env decl in
                begin
                  (if !Clflags.dnormtrans then
                     print_ntype_decl (Ident.name id, ndecl));
                  (Ident.name id, ndecl)
                end
             ) id_decl_list
  in (ndecls, !myname_dependencies)

(* Entry points for normalization of types. *)
let extract_and_clear_state () =
  let old_map = !tyvar_map in
    tyvar_map := [];
    old_map

let restore_state new_map = tyvar_map := new_map

let normalize_type local_ids (functor_args : (Ident.t * int) list) env ty =
  myname_dependencies := [];
  let nty = process_type local_ids [] functor_args env ty in
  (nty, !myname_dependencies)
