/*
 *  byterun/polymarshal.c
 *
 *  Runtime support for polymorphic marshalling.
 *
 *  (c) Copyright 2005, 2006, Mark R. Shinwell.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "alloc.h"
#include "fail.h"
#include "intext.h"
#include "memory.h"
#include "mlvalues.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

extern value caml_output_value_to_string(value, value); /* in extern.c */
extern value caml_output_value_to_buffer(value, value, value, value, value);
                                                        /* in extern.c */
extern value caml_input_value_from_string(value, long); /* in intern.c */

/* Concatenate two typerep strings. */
CAMLprim value caml_polymarshal_concatenate(value s1, value s2)
{
    CAMLparam2(s1, s2);
    CAMLlocal1(s);
    int length1;
    int length2;
    int i;

    assert(Tag_val(s1) == String_tag);
    assert(Tag_val(s2) == String_tag);

    length1 = caml_string_length(s1);
    length2 = caml_string_length(s2);

    s = caml_alloc_string(length1 + length2);
    for (i = 0; i < length1; i++) {
        String_val(s)[i] = String_val(s1)[i];
    }
    for (i = 0; i < length2; i++) {
        String_val(s)[i + length1] = String_val(s2)[i];
    }
    String_val(s)[length1 + length2] = '\0';

    CAMLreturn(s);
}

/* Given a typerep and a value, compute a marshalled version of the value
   and combine it with the typerep in such a way that the pair can be
   recovered later. */
CAMLprim value caml_polymarshal_make_package(value v, value flags,
                                             value typerep)
{
    CAMLparam2(v, typerep);
    CAMLlocal1(pair);

    assert(Tag_val(typerep) == String_tag);

    pair = caml_alloc_tuple(2);
    Store_field(pair, 0, v);
    Store_field(pair, 1, typerep);

    CAMLreturn(caml_output_value_to_string(pair, flags));
}

/* Given a typerep and a value, compute a marshalled version of the value
   and combine it with the typerep in such a way that the pair can be
   recovered later.
   
   This version is a wrapper for caml_output_value_to_buffer() rather
   than caml_output_value_to_string().
*/
CAMLprim value caml_polymarshal_make_package_buffer(value buf, value ofs,
                                                    value len, value rest)
{
    CAMLparam4(buf, ofs, len, rest);
    CAMLlocal1(pair);
    CAMLlocal3(v, flags, typerep);

    v = Field(rest, 0);
    flags = Field(rest, 1);
    typerep = Field(rest, 2);

    assert(Tag_val(typerep) == String_tag);

    pair = caml_alloc_tuple(2);
    Store_field(pair, 0, v);
    Store_field(pair, 1, typerep);

    CAMLreturn(caml_output_value_to_buffer(buf, ofs, len, pair, flags));
}

/* Given a marshalled package from caml_polymarshal_make_package, extract
   the value and its typerep.  Then check that the typerep matches what
   is intended; if it does not, raise Marshal_type_mismatch. */
CAMLprim value caml_polymarshal_extract_package(value package, value offset,
                                                value typerep)
{
    CAMLparam2(package, typerep);
    CAMLlocal2(v, packaged_typerep);

    assert(Tag_val(package) == String_tag);
    assert(Tag_val(typerep) == String_tag);

   /* FIXME: implicit declaration of function caml_input_value_from_string */
    v = caml_input_value_from_string(package, 0l);
    packaged_typerep = Field(v, 1);

    if (caml_string_length(packaged_typerep) !=
          caml_string_length(typerep) ||
        memcmp(String_val(packaged_typerep), String_val(typerep),
               caml_string_length(typerep)) != 0) {
        /* FIXME raise the proper exception! */
        caml_invalid_argument("Marshal type mismatch");
    }

    CAMLreturn(Field(v, 0));
}

