(*
 *  polymarshal/polymarshal.ml
 *
 *  Type-passing translation of typed syntax trees for the purposes of
 *  polymorphic marshalling.
 *
 *  (c) Copyright 2005, 2006, Mark R. Shinwell.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Asttypes
open Lambda
open Outcometree
open Path
open Primitive
open Typedtree
open Types

let type_typerep = Predef.type_typerep

let pm_debug s =
  if !Clflags.pmdebug then (print_string s; flush stdout) else ()

let rec find_ident id xs =
  match xs with
    [] -> raise Not_found
  | (id', d)::xs -> if Ident.same id id' then d else find_ident id xs

(* The prefix used when transforming type variable identifiers into
   value identifiers. *)
let vid_prefix = "_tyrep_"

(* Value identifier used when "eta-expanding" marshal expressions. *)
let marshal_eta_vid = "_marshalled"

(* Given a list of identifiers, produce an association list mapping them
   to fresh ones. *)
let rec allocate_fresh_idents idents =
  match idents with
    [] -> []
  | x::xs -> (x, Ident.rename x) :: (allocate_fresh_idents xs)

(* Given an association map from identifiers to identifiers, rename
   the identifiers in a pattern. *)
let rec rename_pat_desc_idents imap pat_desc =
  match pat_desc with
    Tpat_any -> Tpat_any
  | Tpat_var id ->
      begin try
        Tpat_var (find_ident id imap)
      with Not_found -> pat_desc
      end
  | Tpat_alias (pat, id) ->
      Tpat_alias (rename_pat_idents imap pat, List.assoc id imap)
  | Tpat_constant c -> Tpat_constant c
  | Tpat_tuple pats -> Tpat_tuple (List.map (rename_pat_idents imap) pats)
  | Tpat_construct (cd, pats) ->
      Tpat_construct (cd, List.map (rename_pat_idents imap) pats)
  | Tpat_variant (lab, pat_opt, rd) ->
      Tpat_variant (lab, pat_opt, rd) (* FIXME VARIANTS *)
  | Tpat_record entries ->
      Tpat_record (List.map (fun (lab, pat) ->
                               (lab, rename_pat_idents imap pat)) entries)
  | Tpat_array pats -> Tpat_array (List.map (rename_pat_idents imap) pats)
  | Tpat_or (pat1, pat2, path_opt) ->
      Tpat_or (rename_pat_idents imap pat1, rename_pat_idents imap pat2,
               path_opt)

and rename_pat_idents imap pat =
  { pat with pat_desc = rename_pat_desc_idents imap pat.pat_desc }

(* Take a pattern and produce an association list mapping the identifiers
   in the pattern to their types. *)
let rec collect_pat_desc_idents_and_types acc pat_desc pat_ty =
  match pat_desc with
    Tpat_var id -> (id, pat_ty) :: acc
  | Tpat_alias (pat, id) ->
      (id, pat_ty) :: (collect_pat_idents_and_types_body acc pat)
  | Tpat_tuple pats ->
      List.fold_left collect_pat_idents_and_types_body acc pats
  | Tpat_construct (_, pats) ->
      List.fold_left collect_pat_idents_and_types_body acc pats
  | Tpat_record entries ->
      List.fold_left collect_pat_idents_and_types_body acc
        (List.map (fun (_, pat) -> pat) entries)
  | Tpat_array pats ->
      List.fold_left collect_pat_idents_and_types_body acc pats
  | Tpat_or (pat1, pat2, _) ->
      collect_pat_idents_and_types_body
        (collect_pat_idents_and_types_body acc pat1) pat2
  | _ -> acc (* FIXME VARIANTS *)

and collect_pat_idents_and_types_body acc pat =
  collect_pat_desc_idents_and_types acc pat.pat_desc pat.pat_type

let collect_pat_idents_and_types = collect_pat_idents_and_types_body []

(* Create a variable pattern given an identifier and its type, together
   with location and environment information. *)
let make_var_pattern (id, ty) loc env =
  { pat_desc = Tpat_var id;
    pat_loc = loc;
    pat_type = ty;
    pat_env = env }
    
let make_tuple_pattern vids ty loc env =
  { pat_desc =
      Tpat_tuple (List.map (fun vid -> make_var_pattern (vid, ty) loc env) vids);
    pat_loc = loc;
    pat_type = ty;
    pat_env = env }

(* Construct an expression representing a value identifier. *)
let exp_of_vid id ty loc env =
  { exp_desc = Texp_ident (Pident id, { val_type = ty; val_kind = Val_reg },
                           [] (* FIXME? *));
    exp_loc = loc;
    exp_type = ty;
    exp_env = env }

(* Turn a type expression into a type representation string. *)
let typerep_of_type_expr ty = (* DEPRECATED *)
  assert false
(*
  (* FIXME: this is where we will add the code to insert hashes, etc. *)
  let buf = Buffer.create 40 in
  let formatter = Format.formatter_of_buffer buf in
  let _ = Printtyp.type_scheme formatter ty in
  let _ = Format.pp_print_flush formatter () in
    Buffer.contents buf
*)

let type_to_string ty' =
  let ty = Ctype.repr ty' in
  let buf = Buffer.create 40 in
  let formatter = Format.formatter_of_buffer buf in
  let _ = Printtyp.type_scheme_idents formatter ty in
  let _ = Format.pp_print_flush formatter () in
    Buffer.contents buf

(* Generate an association list mapping type variables to value identifiers
   (the alpha |--> t_alpha map) given a list of type variables. *)
let value_idents_of_tyvars =
  List.map (fun tv ->
              (tv,
               Ident.create_typerep (vid_prefix ^ (string_of_int tv.id))))

(* Sort a list of type variables into the canonical order. *)
let sort_tyvar_list = List.sort (fun tv1 -> fun tv2 -> compare tv1.id tv2.id)

(* Helper for typerep construction functions. *)
let typerep_make_exp loc env desc =
  { exp_desc = desc;
    exp_loc = loc;
    exp_type = Ctype.instance type_typerep;
    exp_env = env }

(* Value description for the C primitive for flattening runtime
   typerep blocks to typerep hashes. *)

let type_alpha_typerep =
  Ctype.newty (Tarrow ("", Ctype.newvar (), Predef.type_typerep, Cunknown)) 

let primitive_flatten =
  Texp_ident
    (Pident (Ident.create "%flatten_typerep_block"),
     { val_kind = Val_prim { prim_name = "%flatten_typerep_block";
                             prim_arity = 1;
                             prim_alloc = true;
                             prim_native_name = "%flatten_typerep_block";
                             prim_native_float = false
                           };
       val_type = Ctype.instance type_alpha_typerep },
     [])

let do_flatten loc env tyrep =
  { exp_desc = Texp_apply ({ exp_desc = primitive_flatten;
                             exp_loc = loc;
                             exp_type = Ctype.instance
                                          type_alpha_typerep;
                             exp_env = env },
                           [(Some tyrep, Required)]);
    exp_loc = loc;
    exp_type = Ctype.instance type_typerep;
    exp_env = env }

(* Take a type expression and construct a corresponding Typedtree expression
   which, when evaluated, will return the type representation for that type
   expression.  If the type expression contains free type variables then
   the expression will contain free value identifiers to correspond to them. *)
let type_to_typerep loc env ty = Normtypes.normalize_type env ty

(* As for type_to_typerep, but goes via the flattening and hashing. *)
let type_to_typerep_checked loc env ty =
  do_flatten Location.none env (Normtypes.normalize_type env ty)

(* Addition of a single typerep lambda to an expression. *)
let add_one_typerep_lambda ty tyvar_map exp tyvar =
  (* extract the value identifier for the current type variable *)
  let vid = List.assoc tyvar tyvar_map in
  (* construct the type of the resulting expression (this is the
     type of "exp" with "ty ->" stuck on the front) *)
(*  let new_ty_desc = Tarrow ("", ty, exp.exp_type, Cunknown) in*)
  let new_ty = exp.exp_type (*Ctype.newty new_ty_desc*) in
  (* construct the new expression ("lambda vid -> exp") *)
  (* FIXME unsure about exp.exp_env here *)
  let pat_exp = (make_var_pattern (vid, ty) exp.exp_loc exp.exp_env, exp) in
  let new_exp_desc = Texp_function ([pat_exp], Total) in
    { exp_desc = new_exp_desc;
      exp_loc = exp.exp_loc;
      exp_type = new_ty; (* FIXME this is wrong surely? *)
      exp_env = exp.exp_env }

let add_tupled_typerep_lambdas tyvar_map tyvars exp =
  (* in the tupled case, tuples of typereps are only passed around if there
     is more than one typerep in the tuple.  If there would be only one,
     the extra (singleton) tuple is dropped, so this function should not be
     used in that case.  (Use add_one_typerep_lambda instead.) *)
  assert (List.length tyvars > 1);
  let vids = List.map (fun tyvar -> List.assoc tyvar tyvar_map) tyvars in
  let pat = make_tuple_pattern vids (Ctype.instance Predef.type_typerep)
                               exp.exp_loc exp.exp_env
  in
    { exp_desc = Texp_function ([(pat, exp)], Total);
      exp_loc = exp.exp_loc;
      exp_type = exp.exp_type; (* FIXME wrong as above *)
      exp_env = exp.exp_env }

let do_add_typerep_lambdas tyvar_map tyvars exp =
  if !Clflags.tupled_typereps && (List.length tyvars > 1) then
    add_tupled_typerep_lambdas tyvar_map tyvars exp
  else
    List.fold_left (add_one_typerep_lambda (Ctype.instance Predef.type_typerep)
                                           tyvar_map) exp (List.rev tyvars)

(* Given an expression and its type, add type representation lambdas
   to the front of it to give another expression. *)
let add_typerep_lambdas (exp, ty) loc env =
  let closed = Ctype.closed_schema ty in
  let _ = if closed then pm_debug "closed\n"
                    else pm_debug "open\n" in
  let tyvars_of_ty = Ctype.free_variables ty in
  let gen_tyvars_of_ty = Ctype.free_contravariant_generalized_variables env ty in
  let num_tyvars = List.length tyvars_of_ty in
  let num_gen_tyvars = List.length gen_tyvars_of_ty in
  let _ = pm_debug ("Number of tyvars: " ^ (string_of_int num_tyvars) ^
                        "\n") in
  let _ = pm_debug ("Number of gen tyvars: " ^ (string_of_int num_gen_tyvars) ^
                        "\n") in
    (* if there are no (covariant) generalized type variables, we don't need to
       add any typerep lambdas; ditto if the identifier isn't of function
       type. *)
    if num_gen_tyvars = 0 || (not (Ctype.is_function_type ty)) then
      (*exp_of_vid id ty loc env *) exp
    else
      (* typerep lambdas need to be added.  First, produce a map from
         type variables to their corresponding value identifiers (t_alphas). *)
      let tyvar_map = value_idents_of_tyvars gen_tyvars_of_ty in
      (* produce a sorted list of the generalized type variables
         in a canonical order (this is important to match up typerep
         abstractions and applications). *)
      let gen_tyvars_sorted = sort_tyvar_list gen_tyvars_of_ty in
      (* construct the expression for the body *)
      let body = (*exp_of_vid id ty loc env*) exp in
        do_add_typerep_lambdas tyvar_map gen_tyvars_sorted body

let letrec_memo = ref tyvar_id_memo_empty

(* Rewrite an identifier to use type-passing.
   Note that the argument ty gives the type at which the identifier is used.
   The type value_desc.val_type gives the type scheme of the identifier. *)
let rewrite_identifier orig_exp ids path value_desc loc ty env =
  pm_debug ("rewrite_identifier: " ^ (Path.name path) ^ "\n");
  (* see what kind of value we have.  At the moment we only put type
     applications onto regular values. *)
  match value_desc.val_kind with
    Val_reg ->
      (* see if the identifier has a non-trivial type scheme.  If it does,
         we need to add some typerep applications to it.  Whilst we're in
         the business of collecting the generalized variables to do this,
         we sort them into the canonical order so we get any typerep
         applications in the right order. *)
      pm_debug ("val_type: " ^ (type_to_string value_desc.val_type) ^ "\n");
      pm_debug ("ty: " ^ (type_to_string ty) ^ "\n");
      let val_type = value_desc.val_type in
      let tyvars = Ctype.free_variables val_type in
      let gen_tyvars'' =
        sort_tyvar_list (Ctype.free_contravariant_generalized_variables env value_desc.val_type)
      in
      (* filter out any type variables which have been in-place generalized
         during the type inference process, being careful about let rec-
         bound variables *)
      let (is_rec_id, gen_tyvars) =
      begin
        match path with
          Pident ident ->
          begin
            match tyvar_id_memo_lookup ident !letrec_memo with
              None -> (false,
                       List.filter (fun tv -> List.mem tv.id ids) gen_tyvars'')
            | Some ids -> (true,
                           List.filter (fun tv -> List.mem tv.id ids) gen_tyvars'')
          end
        | _ ->
          (false, List.filter (fun tv -> List.mem tv.id ids) gen_tyvars'')
      end
      in
      (if is_rec_id then pm_debug "is_rec_id\n" else pm_debug "not is_rec_id\n");
  let _ = pm_debug ("Vid " ^ (Path.name path) ^ ". Number of tyvars: " ^ (string_of_int (List.length tyvars)) ^
                        "\n") in
  let _ = pm_debug ("Vid. Number of gen tyvars: " ^ (string_of_int (List.length gen_tyvars)) ^
                        "\n") in
        if List.length gen_tyvars = 0 ||
           (not (Ctype.is_function_type ty)) then
          (* nothing to be added *)
          Texp_ident (path, value_desc, ids)
        else begin
          (* here we determine what substitution has been used to
             instantiate the type scheme.  To do this, we first allocate
             as many fresh type variables as the scheme has parameters. *)
          let fresh_tyvars = List.map (fun _ -> Ctype.newvar ()) gen_tyvars in
          (* apply the type scheme to the fresh type variables *)
          let ty1 =
            begin try
              Ctype.apply env gen_tyvars val_type fresh_tyvars
            with Ctype.Cannot_apply -> assert false
            end
          in
          (* assert that no generalized variables remain in ty1
          let _ = assert (List.length
                            (Ctype.free_contravariant_generalized_variables ty1) = 0) in*)
          (* unify ty and ty1 so that we can read the substitution off
             via the fresh tyvars (they will have been instantiated by
             the unification) *)
          let _ = pm_debug ("ty1 (instance of val_type): " ^ (type_to_string ty1) ^ "\n") in
          let _ = pm_debug ("val_type: " ^ (type_to_string val_type) ^ "\n") in
          let _ = pm_debug ("ty: " ^ (type_to_string ty) ^ "\n") in
          let _ =
            if is_rec_id then () else
            begin try
              Ctype.bodge := true;
              Ctype.unify env ty1 ty;
              Ctype.bodge := false;
            with Ctype.Unify _ -> assert false
            end
          in
          let _ = pm_debug ("+ty1 (instance of val_type): " ^ (type_to_string ty1) ^ "\n") in
          let _ = pm_debug ("+val_type: " ^ (type_to_string val_type) ^ "\n") in
          let _ = pm_debug ("+ty: " ^ (type_to_string ty) ^ "\n") in
          (* for each type in fresh_tyvars, we now (a) convert it to
             a typerep and (b) turn that typerep into an expression which
             we can apply to the identifier. *)
          let app_tyvars = if is_rec_id then gen_tyvars else fresh_tyvars in
          let typereps = List.map (type_to_typerep loc env) app_tyvars in
          (* munge typereps so the list will go into Texp_apply *)
          let _ = pm_debug ("length of typereps="^(string_of_int(List.length typereps))^"\n") in
            if !Clflags.tupled_typereps && (List.length typereps > 1) then
              let ty = Ctype.newty (Ttuple
                (List.map (fun _ -> Ctype.instance Predef.type_typerep)
                          typereps))
              in
              let typerep_tuple =
                { exp_desc = Texp_tuple typereps;
                  exp_loc = orig_exp.exp_loc;
                  exp_type = ty;
                  exp_env = orig_exp.exp_env }
              in
                Texp_apply (orig_exp, [Some typerep_tuple, Required])
            else
              let typereps' = List.map (fun tr -> (Some tr, Required)) typereps
              in
                (* now we construct the application expression *)
                Texp_apply (orig_exp, typereps')
        end
  | _ -> (pm_debug "Skipping this variety of id.\n";
         Texp_ident (path, value_desc, ids))

let rewriting_idents = ref false (* FIXME! *)
let rewriting_idents_map = ref []

let specialize_label ldesc env exp =
  match ldesc.lbl_arg.desc with
  | Tpoly (_, gen_tyvars_of_ty) -> begin
      let (fresh_tyvars, ty1, _) = Ctype.instance_label false ldesc in
  try
    Ctype.bodge := true;
    Ctype.unify env ty1 exp.exp_type;
    Ctype.bodge := false;
    let typereps =
      List.map (type_to_typerep Location.none env) fresh_tyvars in
    (*
            if !Clflags.tupled_typereps && (List.length typereps > 1) then
              let ty = Ctype.newty (Ttuple
                (List.map (fun _ -> Ctype.instance Predef.type_typerep)
                          typereps))
              in
              let typerep_tuple =
                { exp_desc = Texp_tuple typereps;
                  exp_loc = exp.exp_loc;
                  exp_type = ty;
                  exp_env = exp.exp_env }
              in
                
                Texp_apply (exp, [Some typerep_tuple, Required])
            else*)
              let typereps' = List.map (fun tr -> (Some tr, Required)) typereps
              in
                (* now we construct the application expression *)
                { exp_desc = Texp_apply (exp, typereps');
                  exp_loc = exp.exp_loc;
                  exp_type = exp.exp_type;
                  exp_env = exp.exp_env }

  with Ctype.Unify tylist ->
  begin
    List.iter (fun (ty1', ty2') ->
      print_string "couldn't unify:\n";
      print_string (type_to_string ty1');
      print_newline ();
      print_string (type_to_string ty2');
      print_newline ()) tylist;
    assert false
  end


  end
  | _ -> exp

  (* unify the new version of ty1 with ty2 so we can read off the
     substitution *)
(*  let ty1_str = type_to_string ty1 in
  let ty2_str = type_to_string ty2 in*)


 

let add_label_lambdas ldesc exp =
  match ldesc.lbl_arg.desc (*exp.exp_type.desc*) with
  | Tpoly (_, _) -> begin
      let gen_tyvars_of_ty = begin
        match exp.exp_type.desc with
        | Tpoly (_, vars) -> vars
        | _ ->
        Ctype.free_contravariant_generalized_variables exp.exp_env exp.exp_type
      end in
      (* typerep lambdas need to be added.  First, produce a map from
         type variables to their corresponding value identifiers (t_alphas). *)
      let tyvar_map = value_idents_of_tyvars gen_tyvars_of_ty in
      (* produce a sorted list of the generalized type variables
         in a canonical order (this is important to match up typerep
         abstractions and applications). *)
      let gen_tyvars_sorted = sort_tyvar_list gen_tyvars_of_ty in
      (* construct the expression for the body *)
      do_add_typerep_lambdas tyvar_map gen_tyvars_sorted exp
  end
  | _ -> exp

(* Rewrite a single binding of a recursive let expression
   to use type-passing.  A list of bindings is returned.
   
   id = identifier being recursively bound
   ty = type of that identifier (post type inference)
   exp = rhs of binding
   tv_list = list of identifiers of the type variables in ty that are to
             be treated as general.
*)
let rec rewrite_rec_let_binding id ty exp tv_list =
  let exp' = rewrite_expression exp in
  (* determine if the type of the pattern variable contains any
     generalised variables; if it does, we need to add typerep
     lambdas to correspond to them. *)
  let tyvars_of_ty = Ctype.free_variables ty in
  let gen_tyvars_of_ty' =
    Ctype.free_contravariant_generalized_variables exp.exp_env ty in
  (* filter out type variables which were in-place generalized after
     the let rec was typechecked *)
  let gen_tyvars_of_ty =
    List.filter (fun tv -> List.mem tv.id tv_list) gen_tyvars_of_ty' in
  let num_tyvars = List.length tyvars_of_ty in
  let num_gen_tyvars = List.length gen_tyvars_of_ty in
  let _ = pm_debug ("letrec: identifier: " ^ (Ident.unique_name id) ^ "\n") in
  let _ = pm_debug ("letrec: type: " ^ (type_to_string ty) ^ "\n") in
  let _ = pm_debug
            ("letrec: number of tyvars in the type of the identifier: " ^
            (string_of_int num_tyvars) ^ "\n") in
  let _ = pm_debug
            ("letrec: number of those which are generalised: " ^
            (string_of_int num_gen_tyvars) ^ "\n") in
  if num_gen_tyvars > 0 && (Ctype.is_function_type ty) then
    (* need to add lambdas *)
    let _ = pm_debug "Typerep lambdas are to be added.\n" in
    (* first, produce a map from type variables to their
       corresponding value identifiers (t_alphas). *)
    let tyvar_map = value_idents_of_tyvars gen_tyvars_of_ty in
    (* produce a sorted list of the generalized type variables
       in a canonical order (this is important to match up typerep
       abstractions and applications). *)
    let gen_tyvars_sorted = sort_tyvar_list gen_tyvars_of_ty in
      (* add the lambda(s) *)
      do_add_typerep_lambdas tyvar_map gen_tyvars_sorted exp'
  else
    (* no lambdas are to be added *)
    exp'

(* Rewrite the bindings and body of a recursive let expression
   to use type-passing. *)
and rewrite_rec_let_bindings bindings body =
  (* assemble tyvar id memos from all bindings into one memo *)
  let letrec_memo_saved = !letrec_memo in
  let full_memo =
    List.fold_left (fun acc -> fun (_, _, memo) ->
                      tyvar_id_memo_append acc memo)
                   !letrec_memo bindings
  in
  letrec_memo := full_memo;
  let new_bindings =
    (* iterate through the bindings translating each one *)
    List.fold_left (fun acc -> fun (pat, exp, memo) ->
                      let memo = if !rewriting_idents then rewrite_memo memo
                                                      else memo
                      in
                      match pat.pat_desc with
                        Tpat_any ->
                          acc @ [(pat, rewrite_expression exp, memo)]
                      | Tpat_var id ->
                        begin try
                          let ids =
                          begin
                            match tyvar_id_memo_lookup id memo with
                              None -> []
                            | Some ids -> ids
                          end
                          in
                          let new_exp =
                            rewrite_rec_let_binding id pat.pat_type exp ids
                          in
                            acc @ [(pat, new_exp, memo)]
                          with Not_found -> assert false
                        end
                      | _ -> assert false) [] bindings
  in
    letrec_memo := letrec_memo_saved;
    new_bindings, (rewrite_expression body)

(* Rewrite a recursive let expression to use type-passing. *)
and rewrite_rec_let bindings body loc ty env =
  let new_bindings, new_body = rewrite_rec_let_bindings bindings body in
  let new_ty = ty in
    { exp_desc = Texp_let (Recursive, new_bindings, new_body);
      exp_loc = loc;
      exp_type = new_ty;
      exp_env = env }


(*
(* Rewrite a single binding of a non-recursive let expression
   to use type-passing.  A list of bindings is returned. *)
and rewrite_nonrec_let_binding (pat, exp) =
  [(pat, rewrite_expression exp)] (* all the work now done on lambda code *)

(* Rewrite the bindings of a non-recursive let expression
   to use type-passing. *)
and rewrite_nonrec_let_bindings bindings =
  List.fold_left (fun acc -> fun b -> acc @ (rewrite_nonrec_let_binding b))
                 [] bindings
*)

(* The following code mirrors what happens when treating "let rec" bodies
   for non-recursive lets. *)
and rewrite_nonrec_let_bindings bindings body =
  (* first collect identifiers being bound and their types *)
(*  let ids =
    List.fold_left (fun acc -> fun (pat, _, _) ->
                      collect_pat_idents_and_types pat)
                   [] bindings
  in*)
  let new_bindings =
    (* iterate through the bindings translating each one *)
    List.fold_left (fun acc -> fun (pat, exp, memo) ->
                      acc @ [(pat, rewrite_expression exp,
                      if !rewriting_idents then rewrite_memo memo else memo)])
                   [] bindings
  in
  let new_body = rewrite_expression body in
    new_bindings, new_body

(* Rewrite a non-recursive let expression to use type-passing. *)
and rewrite_nonrec_let bindings body loc ty env =
  let new_bindings, new_body = rewrite_nonrec_let_bindings bindings body in
  let new_ty = ty in
    { exp_desc = Texp_let (Nonrecursive, new_bindings, new_body);
      exp_loc = loc;
      exp_type = new_ty;
      exp_env = env }

(* Rewrite an application expression to use type-passing. *)
and rewrite_apply exp args =
  let new_args =
    List.map (fun (exp_opt, opt) ->
                match exp_opt with
                  None -> (None, opt)
                | Some exp -> (Some (rewrite_expression exp), opt)) args
  in
    Texp_apply (rewrite_expression exp, new_args)

(* Rewrite a "typerep" expression. *)
and rewrite_typerep = type_to_typerep_checked

(* Rewrite a "typeof" expression. *)
and rewrite_typeof exp =
  (* first get the type-passing version of the body *)
  let exp' = rewrite_expression exp in
  (* get a fresh identifier ("x" in the comment above) *)
  let x = Ident.create marshal_eta_vid in
  (* construct the typerep package; this will be a function of some
     typerep parameters if the type of exp' contains free type variables. *)
  let body = type_to_typerep_checked exp.exp_loc exp.exp_env exp'.exp_type in
  (* make a pattern corresponding to "x" *)
  let pat = make_var_pattern (x, exp'.exp_type) exp'.exp_loc exp'.exp_env in
    (* return the let binding *)
    { exp_desc = Texp_let (Nonrecursive, [(pat, exp', tyvar_id_memo_empty)], body);
      exp_type = Ctype.instance type_typerep;
      exp_loc = exp.exp_loc;
      exp_env = exp.exp_env }

and rewrite_hashname loc env ty exp =
  let tyrep = rewrite_typerep loc env ty in
  let exp' = rewrite_expression exp in
  let (_, hps) = Hashpackage.add_hash_param tyrep
    Hashpackage.empty_hash_param_set in
  let (_, hps') = Hashpackage.add_hash_param exp' hps in
  let pkg = Hashpackage.make_package () hps' in
  Hashpackage.generate_code pkg

(* DO NOT eta-contract rewrite_memo since the dereference must be
   evaluated upon each call. *)
and rewrite_memo memo = tyvar_id_memo_rename !rewriting_idents_map memo

(* Rewrite an expression to use type-passing. *)
and rewrite_expression e =
  match e.exp_desc with
    Texp_ident (path, value_desc, ids) ->
      if !rewriting_idents &&
         (match path with Pident _ -> true | _ -> false) then
      begin
        try
          let id = (match path with Pident id -> id | _ -> assert false) in
          let new_id = find_ident id !rewriting_idents_map in
          begin
            { e with exp_desc = Texp_ident (Pident new_id, value_desc, ids) }
          end
        with Not_found -> e
      end
      else if !rewriting_idents = false then
        { e with exp_desc =
            rewrite_identifier e ids path value_desc e.exp_loc e.exp_type
                               e.exp_env }
      else e
  | Texp_constant _ -> e
    (* FIXME is the following case for Default correct? *)
  | Texp_let ((Nonrecursive | Default) as rec_type, bindings, body) ->
      if !rewriting_idents then
        { e with exp_desc =
            Texp_let (rec_type,
                      List.map (fun (pat, exp, memo) ->
                                  (pat, rewrite_expression exp,
                                   rewrite_memo memo)) bindings,
                      rewrite_expression body) }
      else
        rewrite_nonrec_let bindings body e.exp_loc e.exp_type e.exp_env
  | Texp_let (Recursive, bindings, body) ->
      if !rewriting_idents then
        { e with exp_desc =
            Texp_let (Recursive,
                      List.map (fun (pat, exp, memo) ->
                                  (pat, rewrite_expression exp,
                                   rewrite_memo memo)) bindings,
                      rewrite_expression body) }
      else
        rewrite_rec_let bindings body e.exp_loc e.exp_type e.exp_env
  | Texp_function (pat_exp_list, partial) ->
      { e with exp_desc =
          Texp_function (List.map
                           (fun (pat, exp) -> (pat, rewrite_expression exp))
                           pat_exp_list,
                         partial) }
  | Texp_apply (exp, args) ->
      { e with exp_desc = rewrite_apply exp args }
  | Texp_match (scrutinee, pat_exp_list, partial) ->
      { e with exp_desc =
          Texp_match (rewrite_expression scrutinee,
                      List.map (fun (pat, exp) ->
                                  (pat, rewrite_expression exp)) pat_exp_list,
                      partial) }
  | Texp_try (exp, pat_exp_list) ->
      { e with exp_desc =
          Texp_try (rewrite_expression exp,
                    List.map (fun (pat, exp) ->
                                (pat, rewrite_expression exp)) pat_exp_list) }
  | Texp_tuple exps ->
      { e with exp_desc = Texp_tuple (List.map rewrite_expression exps) }
  | Texp_construct (desc, exps) ->
      { e with exp_desc =
          Texp_construct (desc, List.map rewrite_expression exps) }
  | Texp_variant (_, None) -> e
  | Texp_variant (lab, Some exp) ->
      { e with exp_desc = Texp_variant (lab, Some (rewrite_expression exp)) }
  | Texp_record (ldesc_exp_list, None) ->
      { e with exp_desc =
          Texp_record (List.map
            (fun (ldesc, exp) ->
              (ldesc, add_label_lambdas ldesc (rewrite_expression exp)))
                                ldesc_exp_list,
                       None) }
  | Texp_record (ldesc_exp_list, Some exp) ->
      { e with exp_desc =
          Texp_record (List.map
            (fun (ldesc, exp) ->
              (ldesc, add_label_lambdas ldesc (rewrite_expression exp)))
                                ldesc_exp_list,
                       Some (rewrite_expression exp)) }
  | Texp_field (exp, ldesc) ->
      let exp' = rewrite_expression exp in
      specialize_label ldesc e.exp_env
        { e with exp_desc = Texp_field (exp', ldesc)}
  | Texp_setfield (exp1, ldesc, exp2) ->
      { e with exp_desc =
          Texp_setfield (specialize_label ldesc e.exp_env (rewrite_expression exp1), ldesc,
                         add_label_lambdas ldesc (rewrite_expression exp2)) }
  | Texp_array exps ->
      { e with exp_desc = Texp_array (List.map rewrite_expression exps) }
  | Texp_ifthenelse (exp1, exp2, None) ->
      { e with exp_desc =
          Texp_ifthenelse (rewrite_expression exp1, rewrite_expression exp2,
                           None) }
  | Texp_ifthenelse (exp1, exp2, Some exp3) ->
      { e with exp_desc =
          Texp_ifthenelse (rewrite_expression exp1, rewrite_expression exp2,
                           Some (rewrite_expression exp3)) }
  | Texp_ifname (exp1, exp2, exp3, None) ->
      { e with exp_desc =
          Texp_ifname (rewrite_expression exp1, rewrite_expression exp2,
                       rewrite_expression exp3, None) }
  | Texp_ifname (exp1, exp2, exp3, Some exp4) ->
      { e with exp_desc =
          Texp_ifname (rewrite_expression exp1, rewrite_expression exp2,
                       rewrite_expression exp3,
                       Some (rewrite_expression exp4)) }
  | Texp_sequence (exp1, exp2) ->
      { e with exp_desc =
          Texp_sequence (rewrite_expression exp1, rewrite_expression exp2) }
  | Texp_while (exp1, exp2) ->
      { e with exp_desc =
          Texp_while (rewrite_expression exp1, rewrite_expression exp2) }
  | Texp_for (id, exp1, exp2, dirn, exp3) ->
      { e with exp_desc =
          Texp_for (id, rewrite_expression exp1, rewrite_expression exp2,
                    dirn, rewrite_expression exp3) }
  | Texp_when (exp1, exp2) ->
      { e with exp_desc =
          Texp_when (rewrite_expression exp1, rewrite_expression exp2) }
  | Texp_send (exp, meth) ->
      { e with exp_desc = Texp_send (rewrite_expression exp, meth) }
  | Texp_new _ -> e (* FIXME OBJECTS *)
  | Texp_instvar _ -> e
  | Texp_setinstvar (path1, path2, exp) ->
      { e with exp_desc =
          Texp_setinstvar (path1, path2, rewrite_expression exp) }
  | Texp_override (path, path_exp_list) ->
      { e with exp_desc =
          Texp_override (path,
                         List.map (fun (path, exp) ->
                                     (path, rewrite_expression exp))
                                  path_exp_list) }
  | Texp_letmodule (id, modexp, exp) -> (* FIXME rewrite modexp? *)
      { e with exp_desc = Texp_letmodule (id, modexp, rewrite_expression exp) }
  | Texp_assert exp ->
      { e with exp_desc = Texp_assert (rewrite_expression exp) }
  | Texp_assertfalse -> e
  | Texp_lazy exp ->
      { e with exp_desc = Texp_lazy (rewrite_expression exp) }
  | Texp_object _ -> e (* FIXME OBJECTS *)
  | Texp_typeof exp -> rewrite_typeof exp
  | Texp_typerep ty -> rewrite_typerep e.exp_loc e.exp_env ty
  | (Texp_fresh | Texp_fieldname _) -> e
  | Texp_namecoercion (path1, path2, exp) ->
      { e with exp_desc =
          Texp_namecoercion (path1, path2, rewrite_expression exp) }
  | Texp_hashname (ty, exp) -> rewrite_hashname e.exp_loc e.exp_env ty exp

(* Rewrite a structure item to use type-passing. *)
let rec rewrite_structure_item si =
  let dummy_expr =
    { exp_desc = Texp_assertfalse;
      exp_type = Ctype.newvar ();
      exp_loc = Location.none;
      exp_env = Env.empty }
  in
  match si with
    Tstr_eval e -> Tstr_eval (rewrite_expression e)
  | Tstr_value ((Nonrecursive | Default) as rec_flag, bindings) ->
      let new_bindings, _ = rewrite_nonrec_let_bindings bindings dummy_expr in
        Tstr_value (rec_flag, new_bindings)
  | Tstr_value (Recursive, bindings) ->
      let new_bindings, _ = (* FIXME bit ugly, same for nonrec case *)
        rewrite_rec_let_bindings bindings dummy_expr in
        Tstr_value (Recursive, new_bindings)
  | Tstr_module (id, modexp) ->
      Tstr_module (id, rewrite_module_expr modexp)
  | Tstr_recmodule (id_and_modexps) ->
      Tstr_recmodule (
        List.map (fun (id, modexp) -> (id, rewrite_module_expr modexp))
                 id_and_modexps)
  | Tstr_modtype _ -> si
  | Tstr_open _ -> si
  | Tstr_class _ -> si (* FIXME OBJECTS *)
  | Tstr_cltype _ -> si
  | Tstr_include (modexp, ids) ->
      Tstr_include (rewrite_module_expr modexp, ids)
  | Tstr_primitive _ -> si
  | Tstr_type _ -> si
  | Tstr_exception _ -> si
  | Tstr_exn_rebind _ -> si

and rewrite_module_expr modexp =
  match modexp.mod_desc with
    Tmod_ident _ -> modexp
  | Tmod_structure (kind, str) ->
      { modexp with mod_desc =
          Tmod_structure (kind, List.map (rewrite_structure_item) str)
      }
  | Tmod_functor (id, mty, modexp) ->
      { modexp with mod_desc =
          Tmod_functor (id, mty, rewrite_module_expr modexp)
      }
  | Tmod_apply (modexp1, modexp2, coercion) ->
      { modexp with mod_desc =
          Tmod_apply(rewrite_module_expr modexp1, 
                     rewrite_module_expr modexp2,
                     coercion)
      }
  | Tmod_constraint (modexp, mty, coercion) ->
      { modexp with mod_desc =
          Tmod_constraint (rewrite_module_expr modexp,
                           mty, coercion)
      }

(* Rewrite a structure to use type-passing, given its signature and
   environment. *)
let rewrite_structure str sg env =
  rewriting_idents := false;
  (List.map rewrite_structure_item str, sg, env)

let rec pick_identifiers ids =
  match ids with
    [] -> []
  | x::xs ->
    let new_name = Ident.create (Ident.name x) in
      (x, new_name) :: (pick_identifiers xs)

let dump_lambda name lam =
  let _ = print_string (name ^ "\n") in
  let _ = flush stdout in 
  let _ = Printlambda.lambda Format.std_formatter lam in
  let _ = Format.pp_print_flush Format.std_formatter in
  let _ = flush stdout in ()

(* Given a binding pat = exp which arises from a let expression,
   together with the lambda code generated by the match compiler for exp,
   insert any necessary typerep lambdas in the bindings for the pattern
   variables of pat.
   
   Note that this is a rewrite on lambda code, not typed syntax trees. *)
let rewrite_let_lambda_code pat exp memo lam =
  pm_debug "rewrite_let_lambda_code starting.\n";
  let match_variables = ref [] in
  let tyreps_in_scope = ref IdentSet.empty in
  (* collect pattern variables and their types from pat.  If there are none,
     then we don't need to add any lambdas. *)
  let pat_idents_and_types = collect_pat_idents_and_types pat in
    if pat_idents_and_types = [] then
      lam
    else
    let pat_idents = List.map (fun (id, ty) -> id) pat_idents_and_types in
    let _ = pm_debug "Examining a let which binds the following:\n" in
    let _ = List.iter (fun id -> pm_debug ((Ident.unique_toplevel_name id) ^ "\n")) pat_idents in
    let at_top = ref true in
    (* the following iteration skeleton is taken from the Lambda module *)
    let rec pm_txfm = function
      Lvar id as l ->
        (* check to see if this identifier is one introduced by the match
           compiler and which we have adjusted (via code below) to have
           extra lambdas on the front.  If we did that, then we need to
           turn this node into an application. *)
        begin try
          let tyrep_args = List.assoc id !match_variables in
          let _ = pm_debug ("Found match_vars entry for: " ^
                                (Ident.unique_name id) ^ "\n") in
            if tyrep_args = [] then l
            else
              (* take care: some of the things in tyrep_args may not be
                 in scope (these correspond to type variables used somewhere
                 in the pattern, but not in the type of "id").  These ones
                 must be replaced by dummy typereps. *)
              let args = List.map (fun id -> Lvar id) tyrep_args in
                if !Clflags.tupled_typereps && (List.length args > 1) then
                  let block = Lprim (Pmakeblock (0, Immutable), args) in
                    Lapply (l, [block])
                else
                  Lapply (l, args)
              
              (*

                something like this should be used, but for the moment
                we fall back on Bytegen which will put <never used> in when
                it reaches an unbound typerep identifier.  Can't get the
                following to work at the moment since rewrite_let_lambda_code
                isn't always called from the toplevel lets (thus
                tyreps_in_scope might not be valid when it is called).
    
                See test/tests_db/match.ml for code which upon compilation
                causes this function to be called inside a nested let when
                there is a typerep identifier on an outer level (that does
                not, incorrectly, occur in tyreps_in_scope).

                This whole area is a disgusting mess and needs to be
                sorted out.

                
                List.map (fun id ->
                pm_debug ("Checking tyreps_in_scope for: " ^
                          (Ident.unique_name id) ^ "\n");
                                     if IdentSet.mem id !tyreps_in_scope then
                                     begin
                                       pm_debug "(found)\n";
                                       Lvar id
                                     end
                                     else
                                     begin
                                       pm_debug "(not found)\n";
                                       Lconst (Const_base (
                                         Const_string "<never used>"))
                                     end)
                                  tyrep_args)

                                  *)
        with Not_found -> l
        end
    | Lconst sc as l -> l
    | Lapply(fn, args) -> Lapply(pm_txfm fn, List.map pm_txfm args)
    | Lfunction(kind, params, body) ->
        (* here we musn't forget to update tyrep-identifiers-in-scope info. *)
        let tyreps =
          List.filter (fun param ->
                         try let s = String.sub (Ident.name param) 0 7 in
                           s = "_tyrep_"
                         with Invalid_argument _ -> false) params
        in
        let old_tyreps_in_scope = !tyreps_in_scope in
        let _ =
          List.iter (fun tyrep ->
                     pm_debug ("adding " ^ (Ident.unique_name tyrep) ^
                               " to tyreps_in_scope.\n");
                       tyreps_in_scope :=
                         IdentSet.add tyrep !tyreps_in_scope) tyreps
        in
        (* transform the body of the function *)
        let rhs = pm_txfm body in
          (* restore tyreps-in-scope info *)
          tyreps_in_scope := old_tyreps_in_scope;
          Lfunction(kind, params, rhs)
    | Llet(str, id, arg, body) ->
        (* check if we are interested in adjusting the binding for
           this identifier *)
        if Ident.equal id (Ident.create marshal_eta_vid) ||
           (Ident.is_coercion_wrapper id) then
        begin
          at_top := false;
          (* ignore bindings introduced by eta-expansion of marshalling
             expressions, together with coercion wrappers *)
          let _ = pm_debug "Found a _marshal identifier or cw; ignoring.\n" in
            Llet(str, id, pm_txfm arg, pm_txfm body)
        end
        else if Ident.equal id (Ident.create "match") && !at_top then
        begin
          at_top := false;
          (* bindings of "match" variables have been introduced by the
             match compiler.  They may need augmenting with typerep lambdas on
             the front so that they don't have unresolvable free variables.
             However this MUST NOT be done in the case where the let
             does NOT generalize, as that could disrupt side effects in the
             "e" part of "let x = e in e'". *)
          let rhs = pm_txfm arg in

          if Typecore.is_nonexpansive exp then
            let _ = pm_debug "Found a `match' binding.\n" in
            (*
            let _ = Printlambda.lambda Format.std_formatter rhs in
            let _ = Format.pp_print_flush Format.std_formatter in
            let _ = pm_debug "\n" in
            *)
            (* extract the free identifiers from the right-hand side *)
            let fvs_set = Lambda.free_variables rhs in
            let fvs = Lambda.IdentSet.elements fvs_set in
            let _ = pm_debug ("Number of free vids: " ^ 
                    (string_of_int (List.length fvs)) ^ "\n") in
            (* filter the free variable list to leave only those identifiers
               which are typereps *)
            let tyrep_fvs =
              List.filter (fun id ->
                let s = Ident.name id in
                  if String.length s < String.length vid_prefix then false
                  else
                    (String.sub s 0 (String.length vid_prefix)) = vid_prefix) fvs
            in
              if List.length tyrep_fvs = 0 then
                let _ = pm_debug "No free tyrep vids.\n" in
                  Llet(str, id, pm_txfm arg, pm_txfm body)
              else
                let _ = pm_debug "There are free tyrep vids.\n" in
                (* cause the lambda-code to become closed by adding lambdas.
                   Also note which ones have been inserted for this identifier,
                   so that we can correctly transform the use points. *)
                let _ =
                  match_variables := ((id, tyrep_fvs) :: !match_variables)
                in
                  if !Clflags.tupled_typereps && List.length tyrep_fvs > 1 then
                    Llet (str, id, Lfunction (Tupled, tyrep_fvs, rhs),
                          pm_txfm body)
                  else
                    let exp =
                      List.fold_left (fun cur_rhs -> fun vid ->
                                        Lfunction (Curried, [vid], cur_rhs))
                                     rhs (List.rev tyrep_fvs)
                    in
                      Llet (str, id, exp, pm_txfm body)
          else
          begin
            at_top := false;
            (* the not non-expansive case: do not add any lambdas (see above) *)
            pm_debug "not non-expansive case of Llet; skipping\n";
            Llet (str, id, rhs, pm_txfm body)
          end
        end
        else if List.exists (Ident.same id) pat_idents then
        begin
          at_top := false;
          (* this identifier is bound by the let-binding, so we may
             be interested in it. *)
          let s = Ident.unique_toplevel_name id in
          let _ = pm_debug ("Found lambda code binding `" ^ s ^ "'.\n") in
          (* extract the type of the pattern variable *)
          let ty = List.assoc id pat_idents_and_types in
          let _ = pm_debug ("Type of the identifier: " ^
                            (type_to_string ty) ^ "\n") in
          (* determine if the type of the pattern variable contains any
             generalised variables; if it does, we need to add typerep
             lambdas to correspond to them. *)
          let tyvars_of_ty = Ctype.free_variables ty in
          let gen_tyvars_of_ty' =
            Ctype.free_contravariant_generalized_variables exp.exp_env ty in
          let _ = pm_debug
            ("Length of gen_tyvars_of_ty': " ^
            (string_of_int (List.length gen_tyvars_of_ty')) ^ "\n") in
          (* filter out type variables that have been in-place generalized
             since the let expression was typechecked *)
          let tyvar_ids =
            begin
              match tyvar_id_memo_lookup id memo with
                None -> [] (* assert false *) (* FIXME wrappers *)
              | Some ids -> ids
            end
          in
          let gen_tyvars_of_ty =
            List.filter (fun tv -> List.mem tv.id tyvar_ids) gen_tyvars_of_ty'
          in
          let num_tyvars = List.length tyvars_of_ty in
          let num_gen_tyvars = List.length gen_tyvars_of_ty in
          let _ = pm_debug
            ("Number of tyvars in the type of that identifier: " ^
            (string_of_int num_tyvars) ^ "\n") in
          let _ = pm_debug
            ("Number of those which are generalised: " ^
            (string_of_int num_gen_tyvars) ^ "\n") in
            if num_gen_tyvars > 0 && (Ctype.is_function_type ty) then
              (* need to add lambdas *)
              let _ = pm_debug "Typerep lambdas are to be added.\n" in
              (* first, produce a map from type variables to their
                 corresponding value identifiers (t_alphas). *)
              let tyvar_map = value_idents_of_tyvars gen_tyvars_of_ty in
              (* produce a sorted list of the generalized type variables
                 in a canonical order (this is important to match up typerep
                 abstractions and applications). *)
              let gen_tyvars_sorted = sort_tyvar_list gen_tyvars_of_ty in
              let tyvar_names =
                List.map (fun tv -> List.assoc tv tyvar_map)
                         gen_tyvars_sorted in
              (* update tyreps-in-scope information *)
              let old_tyreps_in_scope = !tyreps_in_scope in
              let _ =
                List.iter (fun tyvar ->
                             tyreps_in_scope :=
                               IdentSet.add (List.assoc tyvar tyvar_map)
                                            !tyreps_in_scope)
                          gen_tyvars_sorted
              in
              (* get the right-hand side of the let *)
              let rhs = pm_txfm arg in
              (* restore tyreps-in-scope info *)
              let _ = tyreps_in_scope := old_tyreps_in_scope in
              (* stick sufficiently many lambdas on the front with
                 the right argument names (those come from tyvar_map). *)
              let new_rhs =
                if !Clflags.tupled_typereps && List.length tyvar_names > 1 then
                  Lfunction (Tupled, tyvar_names, rhs)
                else
                  List.fold_left (fun cur_rhs -> fun vid ->
                                    Lfunction (Curried, [vid], cur_rhs))
                                 rhs (List.rev tyvar_names)
              in
                Llet(str, id, new_rhs, pm_txfm body)
            else
              (* no lambdas are to be added *)
              let _ = pm_debug "No typerep lambdas are to be added.\n" in
                Llet(str, id, pm_txfm arg, pm_txfm body)
        end
        else
        begin
          at_top := false;
          Llet(str, id, pm_txfm arg, pm_txfm body)
        end
    | Lletrec(ident_lam_list, body) ->
        Lletrec(List.map pm_txfm_decl ident_lam_list, pm_txfm body)
    | Lprim(p, args) -> Lprim(p, List.map pm_txfm args)
    | Lswitch(arg, sw) ->
        Lswitch(pm_txfm arg,
                {sw with sw_consts = List.map pm_txfm_case sw.sw_consts;
                         sw_blocks = List.map pm_txfm_case sw.sw_blocks;
                         sw_failaction =
                           match sw.sw_failaction with
                           | None -> None
                           | Some l -> Some (pm_txfm l)})
    | Lstaticraise (i,args) ->  Lstaticraise (i, List.map pm_txfm args)
    | Lstaticcatch(e1, io, e2) -> Lstaticcatch(pm_txfm e1, io, pm_txfm e2)
    | Ltrywith(e1, exn, e2) -> Ltrywith(pm_txfm e1, exn, pm_txfm e2)
    | Lifthenelse(e1, e2, e3) -> Lifthenelse(pm_txfm e1, pm_txfm e2, pm_txfm e3)
    | Lsequence(e1, e2) -> Lsequence(pm_txfm e1, pm_txfm e2)
    | Lwhile(e1, e2) -> Lwhile(pm_txfm e1, pm_txfm e2)
    | Lfor(v, e1, e2, dir, e3) -> Lfor(v, pm_txfm e1, pm_txfm e2, dir, pm_txfm e3) 
    | Lassign(id, e) -> Lassign(id, pm_txfm e)
    | Lsend (k, met, obj, args) ->
        Lsend (k, pm_txfm met, pm_txfm obj, List.map pm_txfm args)
    | Levent (lam, evt) -> Levent (pm_txfm lam, evt)
    | Lifused (v, e) -> Lifused (v, pm_txfm e)
    and pm_txfm_decl (id, exp) = (id, pm_txfm exp)
    and pm_txfm_case (key, case) = (key, pm_txfm case)
  (* FIXME
     Shadowing can be introduced when typerep args are added onto "match"
     identifiers.  Ought to sort out a way of avoiding that. *)
  in
    let lam' = Lambda.remove_shadowing (pm_txfm lam) in
      pm_debug "rewrite_let_lambda_code finishing.\n";
      lam'

(* Calculate identifiers bound by a structure, ignoring ones bound in
   nested structures (or functors). *)
let bound_idents_of_structure_item acc si =
  match si with
    Tstr_value (_, bindings) ->
      List.fold_left (fun acc -> fun (pat, _, _) ->
                        acc @ (List.map fst
                                        (collect_pat_idents_and_types pat)))
                     acc bindings
  | _ -> acc

let bound_idents_of_structure str =
  List.fold_left bound_idents_of_structure_item [] str

(* Insert at the end of "str" all coercion wrappers given in "wrappers"
   which pertain to identifiers bound in "str". *)
let insert_wrappers str wrappers =
  let wrapper_idents = List.map fst wrappers in
  let bound_idents = bound_idents_of_structure str in
  let rec f ids =
    match ids with
      [] -> []
    | id::ids ->
        (* see if the current wrapper identifier is bound by this
           module *)
        begin try
          if List.mem id bound_idents then
            let wrapper_body = List.assoc id wrappers in
              wrapper_body :: (f ids)
        else
          f ids
        with Not_found -> f ids
        end
  in
    str @ (f wrapper_idents)

let rec rename_structure_item old_ids new_ids wrappers si =
  rewriting_idents := true;
  rewriting_idents_map := List.combine old_ids new_ids;
  match si with
    Tstr_eval e -> Tstr_eval (rewrite_expression e)
  | Tstr_value (rec_flag, bindings) ->
      Tstr_value (rec_flag,
                  List.map (fun (pat, exp, memo) ->
                              (rename_pat_idents !rewriting_idents_map pat,
                               rewrite_expression exp, rewrite_memo memo))
                           bindings)
  | Tstr_module (id, modexp) ->
      Tstr_module (id, rename_module_expr old_ids new_ids wrappers modexp)
  | Tstr_recmodule (id_and_modexps) ->
      Tstr_recmodule (
        List.map (fun (id, modexp) ->
                    (id, rename_module_expr old_ids new_ids wrappers modexp))
                 id_and_modexps)
  | Tstr_modtype _ -> si
  | Tstr_open _ -> si
  | Tstr_class _ -> si (* FIXME OBJECTS *)
  | Tstr_cltype _ -> si
  | Tstr_include (modexp, ids) ->
      Tstr_include (rename_module_expr old_ids new_ids wrappers modexp, ids)
  | Tstr_primitive _ -> si
  | Tstr_type _ -> si
  | Tstr_exception _ -> si
  | Tstr_exn_rebind _ -> si

and rename_module_expr old_ids new_ids wrappers modexp =
  match modexp.mod_desc with
    Tmod_ident _ -> modexp
  | Tmod_structure (kind, str) ->
      { modexp with mod_desc =
          Tmod_structure (kind,
            insert_wrappers
              (List.map (rename_structure_item old_ids new_ids wrappers) str)
              wrappers)
      }
  | Tmod_functor (id, mty, modexp) ->
      { modexp with mod_desc =
          Tmod_functor (id, mty,
                        rename_module_expr old_ids new_ids wrappers modexp)
      }
  | Tmod_apply (modexp1, modexp2, coercion) ->
      { modexp with mod_desc =
          Tmod_apply(rename_module_expr old_ids new_ids wrappers modexp1,
                     rename_module_expr old_ids new_ids wrappers modexp2,
                     coercion)
      }
  | Tmod_constraint (modexp, mty, coercion) ->
      { modexp with mod_desc =
          Tmod_constraint (rename_module_expr old_ids new_ids wrappers modexp,
                           mty, coercion)
      }

let make_coercion_wrapper translator vd1 vd2 env body =
  if (not !Clflags.polymarshal) then body
  else
  if (not (Ctype.is_function_type vd1.val_type)) then body
  else
  let ty1 = vd1.val_type in
  let ty2 = vd2.val_type in
  (*print_endline ("ty1: " ^ (type_to_string ty1));
  print_endline ("ty2: " ^ (type_to_string ty2));*)
  let ty1_fvs = 
    sort_tyvar_list (Ctype.free_generalized_variables env ty1)
  in
  (* calculate generalized type vars in ty2.  The wrapper function
     will have (List.length ty2_fvs) many typerep parameters and their
     names will be related to the ids of ty2_fvs. *)
  let ty2_fvs = 
    sort_tyvar_list (Ctype.free_generalized_variables env ty2)
  in
  let fresh_tyvars = List.map (fun _ -> Ctype.newvar ()) ty1_fvs in
  (* produce a copy of ty1 (==implementation type) with the fresh type
     variables throughout it. *)
  let ty1_copy = Ctype.apply env ty1_fvs ty1 fresh_tyvars in
  (* unify the new version of ty1 with ty2 so we can read off the
     substitution *)
(*  let ty1_str = type_to_string ty1 in
  let ty2_str = type_to_string ty2 in*)
  try
    Ctype.bodge := true;
    Ctype.unify env ty1_copy ty2;
    Ctype.bodge := false;
    let typereps =
      List.map (type_to_typerep Location.none env) fresh_tyvars in
    let typereps = List.map translator typereps in
    let rhs = Lapply (body, typereps) in
    let tyvar_map = value_idents_of_tyvars ty2_fvs in
    let tyvar_names = List.map (fun tv -> List.assoc tv tyvar_map) ty2_fvs in
      if !Clflags.tupled_typereps && List.length tyvar_names > 1 then
        Lfunction (Tupled, tyvar_names, rhs)
      else
        List.fold_left (fun cur_rhs -> fun vid ->
                          Lfunction (Curried, [vid], cur_rhs))
                       rhs (List.rev tyvar_names)
  with Ctype.Unify tylist ->
  begin
    List.iter (fun (ty1', ty2') ->
      print_string "couldn't unify:\n";
      print_string (type_to_string ty1');
      print_newline ();
      print_string (type_to_string ty2');
      print_newline ()) tylist;
    assert false
  end

(* Given a lambda-code body, a type and an environment, return either:
     1. the body unchanged (if there are no free type variables in the type);
     2. the body with as many typerep lambdas added on the front as there are
        free type variables in the type (in the case that there are any).
        
   The resulting "wrapper" has the effect of discarding any typereps
   passed to it.
*)
let make_discard_wrapper ty env rhs =
  let ty_fvs = Ctype.free_generalized_variables env ty in
  if (not !Clflags.polymarshal) || (List.length ty_fvs = 0) then rhs
  else
  let tyvar_map = value_idents_of_tyvars ty_fvs in
  let tyvar_names = List.map (fun tv -> List.assoc tv tyvar_map) ty_fvs in
    if !Clflags.tupled_typereps && List.length tyvar_names > 1 then
      Lfunction (Tupled, tyvar_names, rhs)
    else
      List.fold_left (fun cur_rhs -> fun vid ->
                        Lfunction (Curried, [vid], cur_rhs))
                     rhs (List.rev tyvar_names)

