open Asttypes
open Typedtree
open Types


(* PATTERN ****************************)

let rec string_of_pattern pat =
  Printf.sprintf "desc: %s\nloc: %s\ntype: %s\nenv: %s\n"
  (string_of_pattern_desc pat.pat_desc)
  (string_of_location pat.pat_loc)
  (string_of_type_expr pat.pat_type)
  (string_of_env pat.pat_env)

and string_of_pattern_desc desc =
  match desc with
  | Tpat_any -> "Tpat_any"
  | Tpat_var _ -> "Tpat_var"
  | _ -> "[other]"

and string_of_partial part =
  match part with
  | Partial -> "Partial"
  | Total -> "Total"

and string_of_optional opt =
  match opt with
  | Required -> "Required"
  | Optional -> "Optional"


(* EXPRESSION *************************)

and string_of_expression exp =
  Printf.sprintf "desc: %s\nloc: %s\ntype: %s\nenv: %s\n"
  (string_of_expression_desc exp.exp_desc)
  (string_of_location exp.exp_loc)
  (string_of_type_expr exp.exp_type)
  (string_of_env exp.exp_env)

and string_of_expression_desc desc =
  match desc with
  | Texp_ident _ -> "Texp_ident"
  | Texp_constant _ -> "Texp_constant"
  | Texp_let _ -> "Texp_let"
  | Texp_function _ -> "Texp_function"
  | Texp_apply _ -> "Texp_apply"
  | Texp_match _ -> "Texp_match"
  | _ -> "[other]"


(* STRUCTURE **************************)

and string_of_structure str =
  String.concat "\n" (List.map string_of_structure_item str)

and string_of_structure_item item =
  match item with
  | Tstr_eval exp -> "Tstr_eval " ^ string_of_expression exp
  | Tstr_value (flag, pes) ->
      "Tstr_value " ^ string_of_rec_flag flag ^ "\n"
        ^ string_of_pat_exp_list pes
  | Tstr_primitive _ -> "Tstr_primitive"
  | Tstr_type _ -> "Tstr_type"
  | _ -> "[other]"


(* TYPE EXPRESSION ********************)

and string_of_type_expr type_expr =
  Printf.sprintf "desc: %s\nlevel: %s\nid: %s\n"
  (string_of_type_desc type_expr.desc)
  (string_of_int type_expr.level)
  (string_of_int type_expr.id)

and string_of_type_desc type_desc =
  match type_desc with
  | Tvar -> "Tvar"
  | Tarrow _ -> "Tarrow _"
  | Ttuple _ -> "Ttuple _"
  | Tconstr _ -> "Tconstr _"
  | Tnil -> "Tnil"
  | Tlink x -> "Tlink " ^ string_of_type_expr x
  | Tsubst _ -> "Tsubst _"
  | Tvariant _ -> "Tvariant _"
  | Tunivar -> "Tunivar"
  | Tpoly _ -> "Tpoly _"
  | _ -> "[other]"


(* LOCATION ***************************)

and string_of_location loc =
  "[location]"


(* ENVIRONMENT ************************)

and string_of_env env =
  "[environment]"


(* MISC *******************************)

and string_of_rec_flag flag =
  match flag with
  | Nonrecursive -> "Nonrecursive"
  | Recursive -> "Recursive"
  | Default -> "Default"

and string_of_pat_exp_list pes =
  let rec f (pat, exp, _) =
    string_of_pattern pat ^ " =>\n" ^ string_of_expression exp in
  String.concat " |\n" (List.map f pes)
