(*
 *  hashing/npretty.ml
 *
 *  Pretty-printing of normalized abstract syntax trees.
 *
 *  (c) Copyright 2005, 2006, John N. Billings, Mark R. Shinwell, Rok Strnisa.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

(* Pretty printing of the Normtree AST *)

open List
open Format
open Asttypes
open Normtree

(* FORMATTING *************************)

let _ = pp_set_margin str_formatter 150

let print s = fprintf str_formatter s

let to_string = flush_str_formatter


(* TERMS ******************************)

let p_option (f:'a -> unit) (o:'a option) : unit =
  match o with
    Some x -> print "Some "; f x
  | None -> print "None"

let rec p_ident (ident : ident_t) : unit =
  match ident with
    Id_bruijn n -> print "%d" n
  | Id_string s -> print "%s" s

and p_path (path : npath) =
  match path with
    Bruijn_param b -> p_ident b
  | Hash_param (h, s) ->
      print "(%s, %s)" (Hashpackage.string_of_hash_param h) s;

and p_expr (expr : nexpression) =
  p_expr_desc expr.nexp_desc

and p_expr_desc (expr_desc : nexpression_desc) =
  print "@[<v0>";
  ( match expr_desc with
      Nexp_ident (path, val_desc) ->
        print "Ident ";
        p_path path
    | Nexp_constant c ->
        print "Constant ";
        ( match c with
            Const_int i -> print "%d" i
          | Const_float f -> print "%s" f
          | Const_char c -> print "%c" c 
          | Const_string s -> print "%s" s
          | _ -> print "_"
        )
    | Nexp_let (rec_flag, pat_expr_list, expr) ->
        print "@[<v2>Let@,";
        p_rec_flag rec_flag;
        print "@,";
        p_pat_expr_list pat_expr_list;
        print " in@,";
        p_expr expr;
        print "@]"
    | Nexp_function (pat_expr_list, par) ->
        print "@[<v2>Function@,";
        p_partial par;
        print "@,";
        p_pat_expr_list pat_expr_list;
        print "@]"
    | Nexp_apply (e, eool) ->
        print "@[<v2>Apply@,";
        p_expr e;
        print "@,";
        p_EOOL eool;
        print "@]"
    | Nexp_match (e, pel, par) ->
        print "@[<v2>Match@,";
        p_partial par;
        print "@,";
        p_expr e;
        print "@,";
        p_pat_expr_list pel;
        print "@]"
    | Nexp_try (e, pel) ->
        print "@[<v2>Try@,";
        p_expr e;
        print "@,";
        p_pat_expr_list pel;
        print "@]"
    | Nexp_tuple el ->
        print "@[<v2>Tuple@,";
        p_EL el;
        print "@]"
    | Nexp_construct (cons_desc, el) ->
        print "@[<v2>Construct@,";
        print "Constructor Description...@,";
        p_EL el;
        print "@]"
    | Nexp_variant (lab, eo) ->
        print "@[<v2>Variant@,";
        print "%s@," lab;
        p_option p_expr eo;
        print "@]"
    | Nexp_record (ldel, eo) ->
        print "@[<v2>Record@,";
        print "Label Description...@,";
        p_option p_expr eo;
        print "@]"
    | Nexp_field (e, ld) ->
        print "@[<v2>Field@,";
        p_expr e;
        print "@,";
        print "Label Description...";
        print "@]"
    | Nexp_setfield (e1, ld, e2) ->
        print "@[<v2>Setfield@,";
        p_expr e1;
        print "@,";
        print "Label Description...@,";
        p_expr e2;
        print "@]"
    | Nexp_array el ->
        print "@[<v2>Array@,";
        p_EL el;
        print "@]"
    | Nexp_ifthenelse (e1, e2, eo) ->
        print "@[<v2>Ifthenelse@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@,";
        p_option p_expr eo;
        print "@]"
    | Nexp_ifname (e1, e2, e3, eo) ->
        print "@[<v2>Ifname@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@,";
        p_expr e3;
        print "@,";
        p_option p_expr eo;
        print "@]"
    | Nexp_sequence (e1, e2) ->
        print "@[<v2>Sequence@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@]"
    | Nexp_while (e1, e2) ->
        print "@[<v2>While@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@]"
    | Nexp_for (id, e1, e2, dir_flag, e3) ->
        print "@[<v2>For@,";
        p_ident id;
        print "@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@,";
        print "Dir_flag...@,";
        p_expr e3;
        print "@]"
    | Nexp_when (e1, e2) ->
        print "@[<v2>When@,";
        p_expr e1;
        print "@,";
        p_expr e2;
        print "@]"
    | Nexp_send (e, meth) ->
        print "@[<v2>Send@,";
        p_expr e;
        print "@,";
        print "Method...";
        print "@]"
    | Nexp_new (path, cl_decl) ->
        print "@[<v2>New@,";
        p_path path;
        print "@,";
        print "Class Declaration...";
        print "@]"
    | Nexp_instvar (path1, path2) ->
        print "@[<v2>Instvar@,";
        p_path path1;
        print "@,";
        p_path path2;
        print "@]"
    | Nexp_setinstvar (path1, path2, e) ->
        print "@[<v2>Setinstvar@,";
        p_path path1;
        print "@,";
        p_path path2;
        print "@,";
        p_expr e;
        print "@]"
    | Nexp_override (path, pel) ->
        print "@[<v2>Override@,";
        p_path path;
        print "@,";
        print "(Path * Expression) list...";
        print "@]"
    | Nexp_letmodule (id, me, e) ->
        print "@[<v2>Letmodule@,";
        p_ident id;
        print "@,";
        p_mod_expr me;
        print "@,";
        p_expr e;
        print "@]"
    | Nexp_assert e ->
        print "@[<v2>Assert@,";
        p_expr e;
        print "@]"
    | Nexp_assertfalse ->
        print "@[<v2>Assertfalse@,";
        print "@]"
    | Nexp_lazy e ->
        print "@[<v2>Lazy@,";
        p_expr e;
        print "@]"
    | Nexp_object (cl_str, cl_sig, sl) ->
        print "@[<v2>Object@,";
        print "@]"
    | Nexp_marshal e ->
        print "@[<v2>Marshal@,";
        p_expr e;
        print "@]"
    | Nexp_unmarshal e ->
        print "@[<v2>Unmarshal@,";
        p_expr e;
        print "@]"
    | Nexp_typeof e ->
        print "@[<v2>Typeof@,";
        p_expr e;
        print "@]"
    | Nexp_typerep te ->
        print "@[<v2>Marshal@,";
        print "[type expression]";
        print "@]"
    | Nexp_fresh ->
        print "@[<v2>Fresh@,";
        print "@]"
    | Nexp_fieldname _ ->
        print "@[<v2>Fieldname ...@,";
        print "@]"
    | Nexp_namecoercion _ ->
        print "@[<v2>Name coercion ...@,";
        print "@]"
    | Nexp_hashname _ ->
        print "@[<v2>Hashname ...@,";
        print "@]"
  );
  print "@]"

and p_optional (opt : noptional) =
  match opt with
    NRequired -> print "Required"
  | NOptional -> print "Optional"

and p_partial (part : npartial) =
  match part with
    NPartial -> print "Partial"
  | NTotal -> print "Total"

and p_pattern (pat : npattern) = p_pat_desc pat.npat_desc
and p_top_pattern (top_pat : npattern) = p_top_pat_desc top_pat.npat_desc

and p_pat_desc (pat_desc : npattern_desc) =
  print "PATTERN ";
  match pat_desc with
    Npat_any ->
      print "Any"
  | Npat_var id ->
      print "Var ";
      p_ident id;
  | Npat_alias (pat, id) ->
      print "Alias ";
      p_pattern pat;
      p_ident id;
  | Npat_constant c ->
      print "Constant _"
  | Npat_tuple pat_list ->
      print "Tuple";
      iter p_pattern pat_list
  | Npat_construct (cntr_desc, pat_list) ->
      print "Construct _"
  | Npat_variant (lab, pat_op, row_desc) ->
      print "Variant _"
  | Npat_record ldpl ->
      print "Record _"
  | Npat_array pats ->
      print "Array _"
  | Npat_or (pat1, pat2, path_op) ->
      print "Or _"

and p_top_pat_desc (top_pat_desc : npattern_desc) =
  print "TOP PATTERN ";
  match top_pat_desc with
  | Npat_any ->
      print "Any"
  | Npat_var id ->
      print "Var ";
      p_ident id
  | Npat_alias (pat, id) ->
      print "Alias";
      p_ident id;
      p_top_pattern pat
  | Npat_constant c ->
      print "Constant _"
  | Npat_tuple pat_list ->
      print "Tuple";
      iter p_top_pattern pat_list
  | Npat_construct (cntr_desc, pat_list) ->
      print "Construct _"
  | Npat_variant (lab, pat_op, row_desc) ->
      print "Variant _"
  | Npat_record ldpl ->
      print "Record _"
  | Npat_array pats ->
      print "Array _"
  | Npat_or (pat1, pat2, path_op) ->
      print "Or _"

and p_mod_expr (mod_expr : nmodule_expr) =
  print "@[<v2>[Module Expression]@,";
  p_mod_expr_desc mod_expr.nmod_desc;
  print "@]@,";
  print "@[<v2>[Module Type]@,";
(*  Printtyp.modtype str_formatter mod_expr.nmod_type; FIXME *)
  print "@]"

and p_mod_expr_desc (mod_expr_desc : nmodule_expr_desc) =
  print "@[<v2>";
  ( match mod_expr_desc with
      Nmod_ident path ->
        print "Ident@,";
        p_path path;
    | Nmod_structure (_, str) ->
        p_struct str
    | Nmod_functor (id, mtype, me) ->
        print "Functor@,";
        p_ident id;
        print "@,";
       (*  Printtyp.modtype str_formatter mtype; FIXME *)
        print "@,";
        p_mod_expr me;
    | Nmod_apply (me1, me2, mcoer) ->
        print "APPLY _"
    | Nmod_constraint (me, mtype, mcoer) ->
        print "CONSTRAINT _"
  );
  print "@]"

and p_rec_flag (rec_flag : Asttypes.rec_flag) =
  match rec_flag with
    Asttypes.Nonrecursive ->
      print "Nonrecursive"
  | Asttypes.Recursive ->
      print "Recursive"
  | Asttypes.Default ->
      print "Default"

and p_struct (str : nstructure) =
  print "@[<v2>[Structure]";
  iter p_struct_item str;
  print "@]"

and p_struct_item (struct_item : nstructure_item) =
  print "@,@[<v2>";
  ( match struct_item with
      Nstr_eval e ->
       print "Eval"
    | Nstr_value (rec_flag, top_pat_expr_list) ->
       print "Value@,";
       p_rec_flag rec_flag;
       print "@,";
       p_top_pat_expr_list top_pat_expr_list;
    | Nstr_primitive (id, val_desc) ->
       print "@[<v2>Primitive id=";
       p_ident id;
       print "@, val_desc=...@]";
    | Nstr_type idtdl ->
       print "TYPE _";
       List.iter Normtypedecl.print_ntype_decl idtdl
    | Nstr_exception (id, exc) ->
       print "EXCEPTION _"
    | Nstr_exn_rebind (id, path) ->
       print "EXN_REBIND _"
    | Nstr_module (id, me) ->
       print "@[<v2>Module id='";
       p_ident id;
       print "'@,";
       p_mod_expr me;
       print "@]"
    | Nstr_recmodule idmel ->
       print "RECMODULE _"
    | Nstr_modtype (b, mod_type) ->
        print "@[<v2>[Module Type]@,";
      (* FIXME  Printtyp.modtype str_formatter mod_type; *)
        print "@]"
    | Nstr_open path ->
       print "OPEN _"
    | Nstr_class cl ->
       print "CLASS _"
    | Nstr_cltype clt ->
       print "CLTYPE _"
    | Nstr_include (mexp, idl) ->
       print "INCLUDE _"
  );
  print "@]"


(* TYPES ******************************)

(* TODO *)


(* UTILITY ****************************)

and p_pat_expr_list (pat_expr_list : (npattern * nexpression) list) =
  print "@[<v2>Top Pattern Expression ";
  iter
    (fun (pat, expr) ->
       print "@,";
       p_pattern pat; 
       print " -> ";
       p_expr expr)
    pat_expr_list;
  print "@]"

and p_top_pat_expr_list (top_pat_expr_list : (npattern * nexpression) list) =
  print "@[<v2>Top Pattern Expression ";
  iter
    (fun (top_pat, expr) ->
       print "@,";
       p_top_pattern top_pat; 
       print " -> ";
       p_expr expr)
    top_pat_expr_list;
  print "@]"

and p_EOOL (eool:(nexpression option * noptional) list) =
  print "@[<v2>Expression option * Optional ";
  iter
    (fun (expr_o, opt) ->
       print "@,";
       p_option p_expr expr_o; 
       print " (";
       p_optional opt;
       print ")")
    eool;
  print "@]"

and p_EL (el:nexpression list) =
  print "@[<v2>Expression ";
  iter
    (fun expr ->
       print "@,";
       p_expr expr)
    el;
  print "@]"
