(*
 *  hashing/normtree.ml
 *
 *  Normalized abstract syntax trees.
 *
 *  (c) Copyright 2005, 2006, John N. Billings, Mark R. Shinwell, Rok Strnisa.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

(* Abstract syntax tree after typing and simpifying for hashing *)

open Asttypes

(* Identifiers *)
type ident_t =
  | Id_bruijn of int
  | Id_string of string

let string_of_ident (ident : ident_t) : string =
  match ident with
    Id_bruijn n -> string_of_int n
  | Id_string s -> s

type nvalue_description =
  { nval_type: Normtypedecl.ntype;
    nval_kind: Types.value_kind }

(* Constructor descriptions *)

type nconstructor_description =
  { ncstr_res: Normtypedecl.ntype;                (* Type of the result *)
    ncstr_args: Normtypedecl.ntype list;          (* Type of the arguments *)
    ncstr_arity: int;                    (* Number of arguments *)
    ncstr_tag: Types.constructor_tag;          (* Tag for heap blocks *)
    ncstr_consts: int;                   (* Number of constant constructors *)
    ncstr_nonconsts: int;                (* Number of non-const constructors *)
    ncstr_private: private_flag }        (* Read-only constructor? *)

(* Record label descriptions *)

type nlabel_description =
  { nlbl_res: Normtypedecl.ntype;                 (* Type of the result *)
    nlbl_arg: Normtypedecl.ntype;                 (* Type of the argument *)
    nlbl_mut: mutable_flag;              (* Is this a mutable field? *)
    nlbl_pos: int;                       (* Position in block *)
    nlbl_repres: Types.record_representation;  (* Representation for this record *)
    nlbl_private: private_flag }         (* Read-only field? *)

(* Value expressions for the core language *)

(* Note that we do not work up to alpha-eqivalence for `top-level'
 * patterns occurring within structures *)

type npattern =
  { npat_desc: npattern_desc;
    npat_type: Normtypedecl.ntype }

and npattern_desc =
    Npat_any
  | Npat_var of ident_t
  | Npat_alias of npattern * ident_t
  | Npat_constant of Asttypes.constant
  | Npat_tuple of npattern list
  | Npat_construct of nconstructor_description * npattern list
  | Npat_variant of Asttypes.label * npattern option * Types.row_desc
  | Npat_record of (nlabel_description * npattern) list
  | Npat_array of npattern list
  | Npat_or of npattern * npattern * npath option

and npartial = NPartial | NTotal
and noptional = NRequired | NOptional

and nexpression =
  { nexp_desc: nexpression_desc;
    nexp_type: Normtypedecl.ntype }

and npath =
    Bruijn_param of ident_t
  | Hash_param of Hashpackage.hash_param * string

and nexpression_desc =
    Nexp_ident of npath * nvalue_description
  | Nexp_constant of Asttypes.constant
  | Nexp_let of Asttypes.rec_flag * (npattern * nexpression) list * nexpression
  | Nexp_function of (npattern * nexpression) list * npartial
  | Nexp_apply of nexpression * (nexpression option * noptional) list
  | Nexp_match of nexpression * (npattern * nexpression) list * npartial
  | Nexp_try of nexpression * (npattern * nexpression) list
  | Nexp_tuple of nexpression list
  | Nexp_construct of nconstructor_description * nexpression list
  | Nexp_variant of Asttypes.label * nexpression option
  | Nexp_record of (nlabel_description * nexpression) list * 
                   nexpression option
  | Nexp_field of nexpression * nlabel_description
  | Nexp_setfield of nexpression * nlabel_description * nexpression
  | Nexp_array of nexpression list
  | Nexp_ifthenelse of nexpression * nexpression * nexpression option
  | Nexp_ifname of nexpression * nexpression * nexpression * nexpression option
  | Nexp_sequence of nexpression * nexpression
  | Nexp_while of nexpression * nexpression
  | Nexp_for of ident_t * nexpression * nexpression *
                Asttypes.direction_flag * nexpression
  | Nexp_when of nexpression * nexpression
  | Nexp_send of nexpression * nmeth
  | Nexp_new of npath * Types.class_declaration
  | Nexp_instvar of npath * npath
  | Nexp_setinstvar of npath * npath * nexpression
  | Nexp_override of npath * (npath * nexpression) list
  | Nexp_letmodule of ident_t * nmodule_expr * nexpression
  | Nexp_assert of nexpression
  | Nexp_assertfalse
  | Nexp_lazy of nexpression
  | Nexp_object of nclass_structure * Types.class_signature * string list
  | Nexp_marshal of nexpression
  | Nexp_unmarshal of nexpression
  | Nexp_typeof of nexpression
  | Nexp_typerep of Normtypedecl.ntype
  | Nexp_fresh
  | Nexp_fieldname of npath
  | Nexp_namecoercion of npath * npath * nexpression
  | Nexp_hashname of Normtypedecl.ntype * nexpression

and nmeth =
    Nmeth_name of string
  | Nmeth_val of ident_t

(* Value expressions for the class language *)

and nclass_expr =
  { ncl_desc: nclass_expr_desc;
    ncl_type: Types.class_type }

and nclass_expr_desc =
    Nclass_ident of npath
  | Nclass_structure of nclass_structure
  | Nclass_fun of npattern * (ident_t * nexpression) list * nclass_expr *
                  npartial     
  | Nclass_apply of nclass_expr * (nexpression option * noptional) list
  | Nclass_let of Asttypes.rec_flag *  (npattern * nexpression) list *
                  (ident_t * nexpression) list * nclass_expr
  | Nclass_constraint of nclass_expr * string list * string list *
                         Types.Concr.t

and nclass_structure =
  { ncl_field: nclass_field list;
    ncl_meths: ident_t Types.Meths.t }

and nclass_field =
    NCf_inher of nclass_expr * (string * ident_t) list *
                 (string * ident_t) list
  | NCf_val of string * ident_t * nexpression
  | NCf_meth of string * nexpression
  | NCf_let of Asttypes.rec_flag * (npattern * nexpression) list *
               (ident_t * nexpression) list
  | NCf_init of nexpression

(* Value expressions for the module language *)

and nmodule_expr =
  { nmod_desc: nmodule_expr_desc;
    nmod_type: nmodule_type }

and nmodule_expr_desc =
    Nmod_ident of npath
  | Nmod_structure of Typedtree.myname_kind * nstructure
  | Nmod_functor of ident_t * nmodule_type * nmodule_expr
  | Nmod_apply of nmodule_expr * nmodule_expr * nmodule_coercion
  | Nmod_constraint of nmodule_expr * nmodule_type * nmodule_coercion

and nstructure = nstructure_item list

and nstructure_item =
    Nstr_eval of nexpression
  | Nstr_value of Asttypes.rec_flag * (npattern * nexpression) list
  | Nstr_primitive of ident_t * nvalue_description
  | Nstr_type of (string * Normtypedecl.ntype_decl) list
  | Nstr_exception of ident_t * (Normtypedecl.ntype list)
  | Nstr_exn_rebind of ident_t * npath
  | Nstr_module of ident_t * nmodule_expr
  | Nstr_recmodule of (ident_t * nmodule_expr) list
  | Nstr_modtype of ident_t * nmodule_type
  | Nstr_open of npath
  | Nstr_class of (ident_t * int * string list * nclass_expr) list
  | Nstr_cltype of (ident_t * Types.cltype_declaration) list
  | Nstr_include of nmodule_expr * ident_t list

and nmodule_coercion =
    Ncoerce_none
  | Ncoerce_structure of (int * nmodule_coercion) list
  | Ncoerce_functor of nmodule_coercion * nmodule_coercion
  | Ncoerce_primitive of Primitive.description * Normtypedecl.ntype

(* Module types *)

and nmodule_type =
    Nmty_ident of npath
  | Nmty_signature of nsignature
  | Nmty_functor of ident_t * nmodule_type * nmodule_type

and nsignature = nsignature_item list

and nsignature_item =
    Nsig_value of ident_t * nvalue_description
  | Nsig_type of ident_t * Normtypedecl.ntype_decl * Types.rec_status
  | Nsig_exception of ident_t * (Normtypedecl.ntype list)
  | Nsig_module of ident_t * nmodule_type * Types.rec_status
  | Nsig_modtype of ident_t * nmodtype_declaration
  | Nsig_class of ident_t * Types.class_declaration * Types.rec_status
  | Nsig_cltype of ident_t * Types.cltype_declaration * Types.rec_status

and nmodtype_declaration =
    Nmodtype_abstract
  | Nmodtype_manifest of nmodule_type

