(*
 *  hashing/hashpackage.ml
 *
 *  Representation of parameterized hashes.
 *
 *  (c) Copyright 2005, Mark R. Shinwell, Rok Strnisa.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Path
open Primitive
open Typedtree
open Types

external hash256 : 'a -> string = "hash256"

type hash_param = int

let string_of_hash_param (hp : hash_param) : string =
  string_of_int hp

let string_of_path path =
  Longident.to_string (fst (Path.split path))

let myname_compare e1 e2 =
  match e1.exp_desc with
    Texp_ident (path1, _, _) ->
      begin match e2.exp_desc with
        Texp_ident (path2, _, _) ->
          String.compare (string_of_path path1) (string_of_path path2)
      | _ -> compare e1 e2
      end
  | _ -> compare e1 e2


module ExpMap =
  Map.Make (struct type t = expression
                   let compare = myname_compare end)

type hash_param_set = { next_stamp : int;
                        map : hash_param ExpMap.t }

type 'a package = 'a * hash_param_set

let empty_hash_param_set = { next_stamp = 0;
                             map = ExpMap.empty }

let add_hash_param data pset =
  try (ExpMap.find data pset.map, pset)
  with Not_found ->
    let stamp = pset.next_stamp in
    let new_map = ExpMap.add data stamp pset.map in
    (stamp, { next_stamp = stamp + 1; map = new_map })

let make_package constant_part hash_param_set = (constant_part, hash_param_set)

(* Turn a string into a piece of Typedtree code representing it. *)
(* There is another one of these in Normtypes, and they should be
   shared. FIXME *)
let string_to_typedtree s =
  { exp_desc = Texp_constant (Asttypes.Const_string s);
    exp_env = Env.empty;
    exp_loc = Location.none;
    exp_type = Ctype.instance (Predef.type_string) }

(* Type and primitive descriptor for the hash function. *)
let type_alpha_string =
  Ctype.newty (Tarrow ("", Ctype.newvar (),
                           Ctype.instance (Predef.type_string), Cunknown))

let primitive_hash256 =
  Texp_ident
    (Pident (Ident.create "hash256_checked"),
     { val_kind = Val_prim { prim_name = "hash256_checked";
                             prim_arity = 1;
                             prim_alloc = true;
                             prim_native_name = "hash256_checked";
                             prim_native_float = false
                           };
       val_type = Ctype.instance type_alpha_string }, [])

(* Means of calling the hash function via a piece of Typedtree code. *)
let do_hash256 exp =
  { exp_desc = Texp_apply ({ exp_desc = primitive_hash256;
                             exp_loc = Location.none;
                             exp_type = Ctype.instance type_alpha_string;
                             exp_env = Env.empty },
                           [(Some exp, Required)]);
    exp_loc = Location.none;
    exp_type = Ctype.instance Predef.type_string;
    exp_env = Env.empty }

(* Turn a parameterized hash package into a piece of Typedtree code. *)
let generate_code (constant_part, hash_param_set) =
  (* hash the constant part *)
  let constant_hash = hash256 constant_part in
  (* turn the string into a Typedtree representation *)
  let constant_hash_exp = string_to_typedtree constant_hash in
    if hash_param_set.map = ExpMap.empty then
      constant_hash_exp
    else
      let constant_hash_ty = constant_hash_exp.exp_type in
      (* assemble the user-specified pieces of Typedtree code into a piece of
         Typedtree code tupling them together with the constant part *)
      let user_parts =
        List.rev (ExpMap.fold (fun code -> fun _ -> fun acc -> code::acc)
                                    hash_param_set.map [])
      in
      let user_parts_tys = List.map (fun code -> code.exp_type) user_parts in
      let user_parts_exp =
        { exp_desc = Texp_tuple (constant_hash_exp :: user_parts);
          exp_env = Env.empty;
          exp_loc = Location.none;
          exp_type = Ctype.newty (Ttuple (constant_hash_ty :: user_parts_tys)) }
      in
        (* wrap the tupling code in a call to the hashing function *)
        do_hash256 user_parts_exp

let get_param_set (constant_part, hash_param_set) = hash_param_set

let combine_param_sets hps1 hps2 =
  ExpMap.fold (fun code -> fun param -> fun hps ->
    let (_, new_hps) = add_hash_param code hps in new_hps)
    hps2.map hps1

(* Combine the hash packages hps1 and hps2.
   Hash parameters in hps1 will be renamed before they are inserted into
   (a copy of) hps2.  Note that this renaming does not affect the parameters
   in the tree parts of the hash packages.
*)
let combine_packages f (c1, hps1) (c2, hps2) =
  (f c1 c2, combine_param_sets hps1 hps2)

let map_constant_part f (cp, hps) = (f cp, hps)

let listify pkgs =
  let pkg = List.fold_left (combine_packages (fun acc -> fun pkg -> pkg :: acc))
                           (make_package [] empty_hash_param_set) pkgs
  in
    map_constant_part List.rev pkg

let add_hash_params_to_package (constant_part, hps1) hps2 =
  (constant_part, combine_param_sets hps1 hps2)
