(*
 *  dcl/test.ml
 *
 *  Test for distributed, type-safe channels
 *
 *  (c) Copyright 2006, John N. Billings <jnb26@cam.ac.uk>
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * Example usage of distributed channel library. We bounce a lambda
 * (actually, just a code pointer and environment) between two hosts.
 *
 * One host is initially listening, whilst the other is initially sending.
 *
 * Example usage, with host A listening and host B sending:
 *
 *   Host A: ./test <addr_of_A> <port_of_A>
 *   Host B: ./test <addr_of_B> <port_of_B> <addr_of_A> <port_of_A>
 *
 * Note that the listener should be started before the sender.
 *
 * Resolvable host names are OK.
 *
 * The compiled code for this test must be identical on both machines,
 * otherwise an unmarshal failure will occur.
 *)  

open Unix

(* Parse command line arguments *)
let (send, src_addr, src_port, dst_addr, dst_port) =
  let usage () =
    Printf.printf
      "Send:    %s <src_addr> <src_port> <dst_addr> <dst_port>\n\
       Receive: %s <src_addr> <src_port>\n" Sys.argv.(0) Sys.argv.(0);
    exit 0 in
  let len = Array.length Sys.argv in
  if len <> 5 && len <> 3 then usage ();
  (* gethostbyname on addresses *)
  let get_addr s =
    let h =
      try
        Unix.gethostbyname s
      with Not_found ->
        Printf.printf "Cannot resolve address %s\n" s;
        exit 0 in
    assert (Array.length h.h_addr_list <> 0);
    h.h_addr_list.(0) in
  try
    let src_addr = get_addr Sys.argv.(1) in
    let src_port = int_of_string (Sys.argv.(2)) in
    let (send, dst_addr, dst_port) =
      if len = 5 then
        (true, Some (get_addr Sys.argv.(3)), Some(int_of_string (Sys.argv.(4))))
      else
        (false, None, None) in
    (send, src_addr, src_port, dst_addr, dst_port)
  with _ -> usage ()

(* Initialise server thread *)
let (tid, com_h) = DChan.init ~inet_addr:src_addr ~port:src_port ()

(* Generate shared channel name *)
let chan : (unit -> unit) name = hashname (unit -> unit, "foo")

(* Function to marshal *)
let rec f n src_addr src_port dst_addr dst_port () =
  Printf.printf
    "n %d, src %s:%d, dst %s:%d\n" n (Unix.string_of_inet_addr src_addr)
    src_port (Unix.string_of_inet_addr dst_addr) dst_port;
  flush Pervasives.stdout;
  let send () =
    Unix.sleep 1;
    DChan.send dst_addr dst_port chan
      (f (n - 1) dst_addr dst_port src_addr src_port) in
  match n with
  | 0 -> (* We're done *)
      exit 0
  | 1 -> (* Send last message and exit *)
      send ();
      exit 0
  | n when n > 1 -> send ()
  | _ -> failwith "n must be non-negative"

(* Receiver *)
let r f = f ()

let _ =
  let val_of_option x =
    match x with
    | Some y -> y
    | None -> assert false in
  (* Register receiver, with replication, no timeout *)
  ignore (DChan.register_recv com_h chan r true None);
  (* Start ball rolling iff sender *)
  if send then f 9 src_addr src_port (val_of_option dst_addr)
    (val_of_option dst_port) ()
  else print_endline "Waiting for message";
  Thread.join tid
