(*
 *  dchan/chan.ml
 *
 *  Local, type-safe channels
 *
 *  (c) Copyright 2006, John N. Billings <jnb26@cam.ac.uk>
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Exist

let debug = Debug.debug "Chan"
let error = Debug.error "Chan"
let warn = Debug.warn "Chan"

(* Hashtable keys *)
module Name = Exist.Make(struct type 'a t = 'a name end)
open Name

type t
type recv_handle_t = Name.t * t name

module HandlerTable = Hashtbl.Make(struct
  type t = Name.t
  (* We cannot use (=) because our existentials are functional values.
   * Furthermore, we only want to check equality of the packed keys and
   * not the function pointers. *)
  let equal = fun x y -> 
    let g = {
      f = fun k1 -> { f = fun k2 ->
        ifname k1 = k2 then true else false }
    } in
    use (use g x) y
  (* Only want to hash over the packed keys, not the function pointers *)
  let hash = fun x ->
    let g = { f = fun x -> Hashtbl.hash x } in
    Name.use g x
end)

type 'a handler = {
  h_chan : 'a name;
  h_dat : 'a list ref;
  h_recvs : (t name * (bool * ('a -> unit))) list ref;
}

(* Hashtable values are of type Handler.t *)
module Handler = Exist.Make(struct type 'a t = 'a handler end)
open Handler

type com_handle_t = {
  table : Handler.t HandlerTable.t;
  lock : Mutex.t;
}

let do_with_lock f lock =
  let res = begin
    try
      Mutex.lock lock;
      f ()
    with e ->
      Mutex.unlock lock;
      raise e
  end in
  Mutex.unlock lock;
  res

let init () = {
  table = HandlerTable.create 16;
  lock = Mutex.create ();
}

(* Remove hashtable entry iff it is `empty' *)
let reap table key handler =
  match (!(handler.h_dat), !(handler.h_recvs)) with
  | ([], []) ->
      debug "Reaping hashtable entry";
      HandlerTable.remove table key
  | _ -> ()

let send ~com_h ~chan ~dat =
  debug "Send invoked";
  let key = Name.pack chan in
  let f () =
    try
      let dat' = HandlerTable.find com_h.table key in
      let g = {
        f = fun handler ->
          ifname handler.h_chan = chan then begin
            debug "Adding data to existing handler";
            match !(handler.h_recvs) with
            | [] ->
                debug "No receptor to pair up with";
                handler.h_dat := dat :: !(handler.h_dat)
            | r :: rs ->
                match r with (id, (replicate, recv)) ->
                  debug "Pairing up with receptor";
                  if replicate then begin
                    debug "Replicating";
                    (* Put to back for fairness *)
                    handler.h_recvs := rs @ [r]
                  end else
                    handler.h_recvs := rs;
                  ignore (Thread.create recv dat);
                  reap com_h.table key handler
          end else assert false
       } in
       Handler.use g dat';
    with Not_found ->
      debug "Adding new handler to table";
      let handler = {
        h_chan = chan;
        h_dat = ref [dat];
        h_recvs = ref [];
      } in
      HandlerTable.add com_h.table key (Handler.pack handler) in
  do_with_lock f com_h.lock

let unregister_recv ~com_h ~recv_h =
  match recv_h with (key, id) -> 
    debug "Unregister_recv invoked";
    let f () =
      let dat = HandlerTable.find com_h.table key in
      let g = {
        f = fun handler ->
          handler.h_recvs := List.remove_assoc id !(handler.h_recvs);
          reap com_h.table key handler
      } in Handler.use g dat in
    do_with_lock f com_h.lock

let register_recv ~com_h ~chan ~recv ~replicate ~timeout =
  debug "Recv invoked";
  let key = Name.pack chan in
  let f () =
    let id = fresh in begin
      match timeout with
      | Some t ->
          let f () =
            Thread.delay t;
            debug "Timeout hit";
            unregister_recv com_h (key, id) in
          ignore (Thread.create f ())
      | _ -> ()
    end;
    let r = (id, (replicate, recv)) in
    try
      let dat = HandlerTable.find com_h.table key in
      let g = { f = fun handler ->
        ifname handler.h_chan = chan then begin
          debug "Adding receptor to existing handler";
          match !(handler.h_dat) with
          | [] ->
              debug "No data to pair up with";
              handler.h_recvs := r :: !(handler.h_recvs)
          | (d::ds) ->
              debug "Pairing up with receptor";
              handler.h_dat := ds;
              if replicate then begin
                debug "Replicating";
                (* Put to back for fairness *)
                handler.h_recvs := !(handler.h_recvs) @ [r]
              end;
              ignore (Thread.create recv d);
              reap com_h.table key handler
        end else assert false
      } in
      Handler.use g dat;
      (Name.pack chan, id)
    with Not_found ->
      debug "Adding new handler to table";
      let handler = {
        h_chan = chan;
        h_dat = ref [];
        h_recvs = ref [r];
      } in
      HandlerTable.add com_h.table key (Handler.pack handler);
      (key, id) in
  do_with_lock f com_h.lock
