(*
 * dcl/debug.ml
 *
 * Debugging output
 *
 * Copyright (c) 2005, David Scott, Fraser Research Inc.
 *   <djs@fraserresearch.org>
 * Copyright (c) 2005, 2006, John Billings <jnb26@cam.ac.uk>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* Debugging output *)

(* Setting to false disables debug messages, not warnings or errors *)
let enabled = true

(* If true causes the output stream to be automatically flushed per-write *)
let autoflush = true

(* Prevent interleaving of output streams *)
let out_lock = Mutex.create ()

(* Writes a message to a particular output stream *)
let output msg =
  try
    Mutex.lock out_lock;
    output_string stderr msg;
    if autoflush then flush stderr;
    Mutex.unlock out_lock
  with e ->
    Mutex.unlock out_lock

(* Defines the appearance of all the messages *)
let format name m tid message =
  Printf.sprintf "%s [%s(%d)]: %s\n" name m tid message

(* Returns integer thread ID of current thread *)
let get_tid () =
  Thread.id (Thread.self ())

(* Call this to output a debug message from file/module m *)
let debug m message = 
    if enabled then output (format "Debug" m (get_tid ()) message)

(* Call this to output a warning message from file/module m *)
let warn m message = output (format "Warning" m (get_tid ()) message)

(* Call this to output an error message from file/module m *)
let error m message = output (format "Error" m (get_tid ()) message)
