
(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  All rights reserved.  This file is distributed    *)
(*  under the terms of the GNU Library General Public License, with    *)
(*  the special exception on linking described in file LICENSE-OCAML.  *)
(*                                                                     *)
(***********************************************************************)

(* $Id: apervasives.mli,v 1.4 2004/12/22 15:27:09 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)



(* The initially opened module.

   This module provides the basic operations over the built-in types
   (numbers, booleans, strings, exceptions, references, lists, arrays,
   input-output channels, ...)

   This module is automatically opened at the beginning of each compilation.
   All components of this module can therefore be referred by their short
   name, without prefixing them by [Pervasives].
*)



(*  {6 Exceptions} *)

(* external raise : exn -> 'a = "%raise" *)
(*  Raise the given exception value *)

(* val invalid_arg : string -> 'a *)
(*  Raise exception [Invalid_argument] with the given string. *)

(* val failwith : string -> 'a *)
(*  Raise exception [Failure] with the given string. *)

(* exception Exit *)
(*  The [Exit] exception is not raised by any library function.  It is
    provided for use in your programs.*)


(*  {6 Comparisons} *)


(* external ( = ) : 'a -> 'a -> bool = "%equal" *)
(*  [e1 = e2] tests for structural equality of [e1] and [e2].
   Mutable structures (e.g. references and arrays) are equal
   if and only if their current contents are structurally equal,
   even if the two mutable objects are not the same physical object.
   Equality between functional values may raise [Invalid_argument].
   Equality between cyclic data structures may not terminate. *)

(* external ( <> ) : 'a -> 'a -> bool = "%notequal" *)
(*  Negation of {!Pervasives.(=)}. *)

(* external ( < ) : 'a -> 'a -> bool = "%lessthan" *)
(*  See {!Pervasives.(>=)}. *)

(* external ( > ) : 'a -> 'a -> bool = "%greaterthan" *)
(*  See {!Pervasives.(>=)}. *)

(* external ( <= ) : 'a -> 'a -> bool = "%lessequal" *)
(*  See {!Pervasives.(>=)}. *)

(* external ( >= ) : 'a -> 'a -> bool = "%greaterequal" *)
(*  Structural ordering functions. These functions coincide with
   the usual orderings over integers, characters, strings
   and floating-point numbers, and extend them to a
   total ordering over all types.
   The ordering is compatible with [(=)]. As in the case
   of [(=)], mutable structures are compared by contents.
   Comparison between functional values may raise [Invalid_argument].
   Comparison between cyclic structures may not terminate. *)

(* external compare : 'a -> 'a -> int = "compare" *)
(*  [compare x y] returns [0] if [x=y], a negative integer if
   [x<y], and a positive integer if [x>y]. The same restrictions
   as for [=] apply. [compare] can be used as the comparison function
    required by the {!Set.Make} and {!Map.Make} functors. *)


(* val min : 'a -> 'a -> 'a  *)
val min : int -> int -> int
(*  Return the smaller of the two arguments. *)

(* val max : 'a -> 'a -> 'a *)
val max : int -> int -> int
(*  Return the greater of the two arguments. *)

(* external ( == ) : 'a -> 'a -> bool = "%eq" *)
(*  [e1 == e2] tests for physical equality of [e1] and [e2].
   On integers and characters, physical equality is identical to structural
   equality. On mutable structures, [e1 == e2] is true if and only if
   physical modification of [e1] also affects [e2].
   On non-mutable structures, the behavior of [(==)] is
   implementation-dependent; however, it is guaranteed that
   [e1 == e2] implies [e1 = e2]. *)

(* external ( != ) : 'a -> 'a -> bool = "%noteq" *)
(*  Negation of {!Pervasives.(==)}. *)


(*  {6 Boolean operations} *)

val not : bool -> bool
(*  The boolean negation. *)

(* external ( && ) : bool -> bool -> bool = "%sequand" *)
(*  The boolean ``and''. Evaluation is sequential, left-to-right:
   in [e1 && e2], [e1] is evaluated first, and if it returns [false],
   [e2] is not evaluated at all. *)

(* external ( & ) : bool -> bool -> bool = "%sequand" *)
(*  @deprecated {!Pervasives.(&&)} should be used instead. *)

(* external ( || ) : bool -> bool -> bool = "%sequor" *)
(*  The boolean ``or''. Evaluation is sequential, left-to-right:
   in [e1 || e2], [e1] is evaluated first, and if it returns [true],
   [e2] is not evaluated at all. *)

(*  external ( or ) : bool -> bool -> bool = "%sequor" *)
(*  @deprecated {!Pervasives.(||)} should be used instead.*)

(*  {6 Integer arithmetic} *)

(*  Integers are 31 bits wide (or 63 bits on 64-bit processors).
   All operations are taken modulo 2{^31} (or 2{^63}).
   They do not fail on overflow. *)

(* external ( ~- ) : int -> int = "%negint" *)
(*  Unary negation. You can also write [-e] instead of [~-e]. *)

(* external succ : int -> int = "%succint" *)
(*  [succ x] is [x+1]. *)

(* external pred : int -> int = "%predint" *)
(*  [pred x] is [x-1]. *)

(* external ( + ) : int -> int -> int = "%addint" *)
(*  Integer addition. *)

(* external ( - ) : int -> int -> int = "%subint" *)
(*  Integer subtraction. *)

(* external ( * ) : int -> int -> int = "%mulint" *)
(*  Integer multiplication. *)

(* external ( / ) : int -> int -> int = "%divint" *)
(*  Integer division.
   Raise [Division_by_zero] if the second argument is 0.
   Integer division rounds the real quotient of its arguments towards zero.
   More precisely, if [x >= 0] and [y > 0], [x / y] is the greatest integer
   less than or equal to the real quotient of [x] by [y].  Moreover,
   [(-x) / y = x / (-y) = -(x / y)].  *)

(* external ( mod ) : int -> int -> int = "%modint" *)
(*  Integer remainder.  If [y] is not zero, the result
   of [x mod y] satisfies the following properties:
   [x = (x / y) * y + x mod y] and
   [abs(x mod y) < abs(y)].
   If [y = 0], [x mod y] raises [Division_by_zero].
   Notice that [x mod y] is negative if and only if [x < 0]. *)

val abs : int -> int
(*  Return the absolute value of the argument. *)


(* these would need to be thunked *)
(* val max_int : int *)
(*  The greatest representable integer. *)
(* val min_int : int *)
(*  The smallest representable integer. *)



(*  {7 Bitwise operations} *)


(* external ( land ) : int -> int -> int = "%andint" *)
(*  Bitwise logical and. *)

(* external ( lor ) : int -> int -> int = "%orint" *)
(*  Bitwise logical or. *)

(* external ( lxor ) : int -> int -> int = "%xorint" *)
(*  Bitwise logical exclusive or. *)


val lnot : int -> int
(*  Bitwise logical negation. *)

(* external ( lsl ) : int -> int -> int = "%lslint" *)
(*  [n lsl m] shifts [n] to the left by [m] bits.
   The result is unspecified if [m < 0] or [m >= bitsize],
   where [bitsize] is [32] on a 32-bit platform and
   [64] on a 64-bit platform. *)

(* external ( lsr ) : int -> int -> int = "%lsrint" *)
(*  [n lsr m] shifts [n] to the right by [m] bits.
   This is a logical shift: zeroes are inserted regardless of
   the sign of [n].
   The result is unspecified if [m < 0] or [m >= bitsize]. *)

(* external ( asr ) : int -> int -> int = "%asrint" *)
(*  [n asr m] shifts [n] to the right by [m] bits.
   This is an arithmetic shift: the sign bit of [n] is replicated.
   The result is unspecified if [m < 0] or [m >= bitsize]. *)

(* Note: floating point stuff omitted completely. *)


(*  {6 String operations}

   More string operations are provided in module {!String}.
*)

(* val ( ^ ) : string -> string -> string *)
(*  String concatenation. *)


(*  {6 Character operations}

   More character operations are provided in module {!Char}.
*)

val int_of_char : char -> int
(*  Return the ASCII code of the argument. *)

val char_of_int : int -> char
(*  Return the character with the given ASCII code.
   Raise [Invalid_argument "char_of_int"] if the argument is
   outside the range 0--255. *)


(*  {6 Unit operations} *)

(* external ignore : 'a -> unit = "%ignore" *)
(*  Discard the value of its argument and return [()].
   For instance, [ignore(f x)] discards the result of
   the side-effecting function [f].  It is equivalent to
   [f x; ()], except that the latter may generate a
   compiler warning; writing [ignore(f x)] instead
   avoids the warning. *)


(*  {6 String conversion functions} *)

val string_of_bool : bool -> string
(*  Return the string representation of a boolean. *)

val bool_of_string : string -> bool
(*  Convert the given string to a boolean.
   Raise [Invalid_argument "bool_of_string"] if the string is not
   ["true"] or ["false"]. *)


val string_of_int : int -> string
(*  Return the string representation of an integer, in decimal. *)


val int_of_string : string -> int



(*  Convert the given string to an integer.
   The string is read in decimal (by default) or in hexadecimal,
   octal or binary if the string begins with [0x], [0o] or [0b]
   respectively.
   Raise [Failure "int_of_string"] if the given string is not
   a valid representation of an integer. *)

(* val string_of_float : float -> string *)
(*  Return the string representation of a floating-point number. *)

(* external float_of_string : string -> float = "float_of_string" *)
(*  Convert the given string to a float.  Raise [Failure "float_of_string"]
   if the given string is not a valid representation of a float. *)

(*  {6 List operations}

   More list operations are provided in module {!List}.
*)

(* val ( @ ) : 'a list -> 'a list -> 'a list *)
(*  List concatenation. *)


(*  {6 Input/output} *)

(* type in_channel *)
(*  The type of input channel. *)

(* type out_channel *)
(*  The type of output channel. *)

(* val stdin : in_channel *)
(*  The standard input for the process. *)

(* val stdout : out_channel *)
(*  The standard output for the process. *)

(* val stderr : out_channel *)
(*  The standard error ouput for the process. *)


(*  {7 Output functions on standard output} *)




val print_char : char -> unit
(*  Print a character on standard output. *)

val print_string : string -> unit
(*  Print a string on standard output. *)

val print_int : int -> unit
(*  Print an integer, in decimal, on standard output. *)

(* val print_float : float -> unit *)
(*  Print a floating-point number, in decimal, on standard output. *)

val print_endline : string -> unit
(*  Print a string, followed by a newline character, on
   standard output. *)

val print_newline : unit -> unit
(*  Print a newline character on standard output, and flush
   standard output. This can be used to simulate line
   buffering of standard output. *)



(*  {7 Output functions on standard error} *)

val prerr_char : char -> unit
(*  Print a character on standard error. *)

val prerr_string : string -> unit
(*  Print a string on standard error. *)

val prerr_int : int -> unit
(*  Print an integer, in decimal, on standard error. *)

(* val prerr_float : float -> unit *)
(*  Print a floating-point number, in decimal, on standard error. *)

val prerr_endline : string -> unit
(*  Print a string, followed by a newline character on standard error
   and flush standard error. *)

val prerr_newline : unit -> unit
(*  Print a newline character on standard error, and flush
   standard error. *)


(*  {7 Input functions on standard input} *)

val read_line : unit -> string
(*  Flush standard output, then read characters from standard input
   until a newline character is encountered. Return the string of
   all characters read, without the newline character at the end. *)

val read_int : unit -> int
(*  Flush standard output, then read one line from standard input
   and convert it to an integer. Raise [Failure "int_of_string"]
   if the line read is not a valid representation of an integer. *)

(* val read_float : unit -> float *)
(*  Flush standard output, then read one line from standard input
   and convert it to a floating-point number.
   The result is unspecified if the line read is not a valid
   representation of a floating-point number. *)

(* {7 General output functions} *)

(* Completely ommitted in minicaml (since we don't have types for channels). *)

