(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * util.ml                                                        *
 *                                                                *
 * Version: $Id: util.ml,v 1.517 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* Miscellaneous helper functions, including some needed *before* Pretty *)


(* exceptions *)

exception Unimplemented of string  (* something is unimplemented *)
exception Never_happen of string   (* can't happen *)

(* == MISCELLANEOUS HELPERS == *)

(* the identity function *)
let id x = x

(* the conditional application function *)
let if_ b f x = if b then f x else x

let partial_map : ('a -> 'b option) -> 'a list -> 'b list
  = fun f xs
 -> let rec go xs = match xs with
                      [] -> []
                    | (x::xs) -> match f x with
                                   None    -> go xs
                                 | Some(y) -> y::go xs
    in
    go xs

let concat_map : ('a -> 'b list) -> 'a list -> 'b list
  = fun f xs
 -> let rec go xs = match xs with
                      [] -> []
                    | (x::xs') -> f x @ go xs'
    in
    go xs

let rec findfirst : ('a -> 'b option) -> 'a list -> 'b option
  = fun f xs
  -> match xs with
      []      -> None
    | (x::xs) -> match f x with
                   None         -> findfirst f xs
                 | Some(_) as y -> y

let rec findfirst_ext : ('a -> 'b option) -> 'a list -> ('b * 'a list) option
  = fun f xs
  -> match xs with
      []      -> None
    | (x::xs) -> match f x with
                   None    -> findfirst_ext f xs
                 | Some(y) -> Some(y,xs)

let is_Some : 'a option -> bool
  = fun x
 -> match x with Some(_) -> true | None -> false

let is_None : 'a option -> bool
  = fun x
 -> match x with Some(_) -> false | None -> true

let the : 'a option -> 'a
  = fun x
 -> match x with Some(y) -> y | None -> raise (Invalid_argument "the of None")

let the' : string -> 'a option -> 'a
  = fun s x
 -> match x with Some(y) -> y | None -> raise (Invalid_argument ("the' of None: "^s))

let option_lift : ('a -> 'b) -> 'a option -> 'b option =
  fun f ->
    function None -> None
      | Some x -> Some (f x)

let stringmap : (char -> 'a) -> string -> 'a list
  = fun f s
 -> let dopos i = f (String.get s i) in
    let rec loop i n = if i = n then [] else dopos i :: loop (i+1) n
    in
    loop 0 (String.length s)


(* function composition *)
let (<.>) f g x = f (g x)

(* function application *)
(* CAREFUL!  Unlike Haskell, the RHS doesn't necessarily scope as far
   as possible; you may need to use begin..end to delimit it.  For
   example, x <$> y >>- z parses as (x <$> y) >>- z, which is probably
   not what was intended. *)
let (<$>) f x = f x

(* sum type *)
type ('a,'b) sum = Inl of 'a | Inr of 'b
let summap : ('a -> 'c) * ('b -> 'c) -> ('a,'b) sum -> 'c
  = fun (f,g) x
 -> match x with
      Inl(a) -> f a
    | Inr(b) -> g b

(* (take,drop) *)
let splitAt : int -> 'a list -> 'a list * 'a list
  = fun n ys
 -> let rec go n xs ys =
    match ys with
      []       -> (List.rev xs, [])
    | (y::ys)  -> if n<=0 then
                    (List.rev xs, y::ys)
                  else
                    go (n-1) (y::xs) ys
    in
    go n [] ys

let take : int -> 'a list -> 'a list
  = fun n ys -> fst (splitAt n ys)
let drop : int -> 'a list -> 'a list
  = fun n ys -> snd (splitAt n ys)

(* versions of List.mem and List.assoc that take an explicit equality argument *)
let mem_by : ('a -> 'a -> bool) -> 'a -> 'a list -> bool
  = fun eq x ys
 -> let rec go ys =
    match ys with
      []      -> false
    | (y::ys) -> eq x y || go ys
    in
    go ys

let assoc_by : ('a -> 'a -> bool) -> 'a -> ('a * 'b) list -> 'b
  = fun eq x xys
 -> let rec go xys =
    match xys with
      []            -> raise Not_found
    | ((x',y)::xys) -> if eq x x' then y else go xys
    in
    go xys

let rec firstdup : ('a -> 'a -> bool) -> 'a list -> 'a option
  = fun eq xs
 -> match xs with
      []      -> None
    | [x]     -> None
    | (x::xs) -> if mem_by eq x xs
                 then Some x
                 else firstdup eq xs

(* remove adjacent duplicates *)
let uniq : ('a -> 'a -> bool) -> 'a list -> 'a list
  = fun eq
 -> let rec go y = function
      [] -> []
    | (x::xs) -> if eq x y then go y xs else (x :: go x xs)
    in
    function
      [] -> []
    | (x::xs) -> x :: go x xs

(* for building complex compare functions, in lexicographic order *)
let (|=|) : int -> int -> int
  = fun c1 c2
 -> if c1 <> 0 then c1 else c2

(* short-circuit version of the above *)
let (|==|) : int -> (unit -> int) -> int
  = fun c1 c2
 -> if c1 <> 0 then c1 else c2 ()

let compare_list : int list -> int
  = List.fold_left (|=|) 0

(* turn Not_found into None *)
let maybe : ('a -> 'b) -> 'a -> 'b option
  = fun f x
 -> try Some(f x) with Not_found -> None


let remove_duplicates cmp l =
  let sorted_l = List.fast_sort cmp l in
  let rec filter l =
    match l with [] -> []
    | [e] -> [e]
    | (e1::e2::es) ->
        if e1 = e2 then filter (e1::es)
        else e1::(filter (e2::es))
  in filter sorted_l


(* functional O(1) queue data structure *)

module AQueue:
sig
  type 'a t
  exception Empty
  val create : unit -> 'a t
  val add : 'a -> 'a t -> 'a t
  val untake : ('a * 'a t) -> 'a t
  val take : 'a t -> 'a * 'a t
  val peek : 'a t -> 'a
  val length : 'a t -> int
  val iter : ('a -> unit) -> 'a t -> unit
  val fold_front : ('a -> 'b -> 'a) -> 'a -> 'b t -> 'a
  val fold_rear : ('b -> 'a -> 'a) -> 'b t -> 'a -> 'a
  val toList : 'a t -> 'a list
  val fromList : 'a list -> 'a t
  val removeFirst : ('a -> bool) -> 'a t -> 'a t
  val empty : 'a t -> bool
end =

struct
  type 'a t = AQueue of 'a list * 'a list

  exception Empty

  let create () = AQueue ([],[])

  let norm hd tl = match hd with
      [] -> AQueue (List.rev tl, [])
    | _  -> AQueue (hd, tl)

  let add x (AQueue(hd,tl)) = norm hd (x::tl)

  let untake (x, (AQueue(hd,tl))) = AQueue(x::hd,tl)

  let take (AQueue(hd,tl)) = match hd with
      []    -> raise Empty
    | x::xs -> x, norm xs tl

  let peek (AQueue(hd, tl)) = match hd with
      []    -> raise Empty
    | x::xs -> x

  let length (AQueue (hd, tl)) = List.length hd + List.length tl

  let iter f (AQueue (hd, tl)) =
      List.iter f hd ;
      List.iter f (List.rev tl)

  let fold_front f a (AQueue (hd, tl)) =
    List.fold_left f a (List.rev_append tl hd)

  let fold_rear f (AQueue (hd, tl)) a =
    List.fold_right f (List.rev_append tl hd) a

  let toList (AQueue (hd, tl)) =
      List.rev_append tl hd

  let fromList l = AQueue (l, [])


  let removeFirst f (AQueue (hd, tl)) =
    let rec go = fun hd tl ->
          match hd, tl with
            [],    [] -> raise Empty
          | [],    _  -> go (List.rev tl) []
          | x::xs, _  -> if f x then xs, tl
                                else let xs', tl' = go xs tl in x::xs',tl'
    in
    let hd', tl' =  go hd tl in norm hd' tl'

  let empty q = try let _ = peek q in false with Empty -> true
end

let qmap f queue =
  let rec q2q f queue1 queue2 =
    try
      let (v, q1') = AQueue.take queue1 in
      q2q f q1' (AQueue.add (f v) queue2)
    with AQueue.Empty -> queue2
  in
  let queue2 = AQueue.create () in
  q2q f queue queue2

