(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * Type checker                                                   *
 *                                                                *
 * Version: $Id: typecheck.ml,v 1.600 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* VARIABLE NAMING CONVENTIONS

   Variable names in this file are named according to a variant of
   Hungarian notation (Simonyi and Heller, BYTE, 16(8), 131--138,
   August 1991.)  For each type, there is a canonical variable name.
   Additional names are obtained by adding primes or indices.

   s      string
   n      integer
   b      boolean
   x      'a
   y      'a
   -s     list
   xys    list of pairs of an x and a y
   -o     option

   tci    tcistate
   tco    tcostate
   m      monadic computation
   f      function (yielding a monadic computation)
   eq     equality function

   rn     rule name
   tcc    tcctxt
   tcci   tcctxt_item
   sign   signature
   si     signature item
   str    structure
   sti    structure item
   ps     pretty-printing state
   v      version
   av     atomic version number expression
   vc     version constraint
   avc    atomic version constraint
   tvc    tail version constraint
   vl     version literal
   h      hash (option)
   env    type environment or name and type environment
   nenv   name environment
   ty     type
   k      kind
   id     (internal) identifier
   iid    internal identifier
   eid    external identifier
   tn     (internal) type name
   itn    internal type name
   t      external type name
   etn    external type name
   mn     module name
   eq     equation
   eqs    equations
   hmn    hash,module name pair
   e      expression
   m      match
   p      pattern
   s      store
   store  store
   def    definition
   d      definition
   defs   definitions
   ds     definitions
   ls     likespec
   c0     con0
   c1     con1
   op     op
   rs     resolvespec
   rsi    resolvespec item
   pe     <<pat>>expr pair
   l      location
   rcs    "recursive": type annotation if recursive, None if not
   mb     body of Marshalled
   idme   <<id>>(mtch,e) triple
   mh     body of module hash
   ih     body of import hash
   senv   store environment

   From this you should be able to gather the philosophy of naming,
   and interpret any names I've missed.

*)

let fresh_printer_state dumplevel skipdefs dumptex =
  Pretty.initial_printer_state (Some Econst.string_of_ident) (Pretty.pm_user dumplevel skipdefs dumptex)


(* override map for testing purpose *)

let list_map s f l =
  let len = List.length l in
  (* Debug.print (function () -> ("Calling map from " ^ s ^ " length is "));
  Debug.print (function () -> (string_of_int(len))); *)
  let r =
    if len > 10000 then
      List.rev (List.rev_map f l)
    else
      List.map f l in
  (* Debug.print (function () -> ("mapped OK; onwards and upwards!\n")); *)
  r


open Ast       (* abstract syntax tree *)
open Basecon   (* basic constructors *)
open Util      (* common and helpers *)
open Pretty    (* pretty printer *)
open Format    (* pretty fprintf and friends *)
(* open Library *)   (* library *)
open Econst    (* the initial environment *)

open Tysupp (* typechecking support *)

(* more freshening help *)
let freshen_internal_ident   (id,x) = swap (fst (ident_int   id)) and (fresh) in (id,x)
let freshen_internal_typname (tn,x) = swap (fst (typname_int tn)) and (fresh) in (tn,x)

(* type abbrev for user-specified equality *)
type 'a eq = 'a -> 'a -> bool

(* list-based inefficient subseteq *)
(* TODO: horrendously inefficient *)
let subseteq : 'a eq -> 'a list -> 'a list -> bool
  = fun eq xs ys
 -> let rec in_by x ys = match ys with
      []      -> false
    | (y::ys) -> eq x y || in_by x ys
    in
    let rec go xs = match xs with
                      []      -> true
                    | (x::xs) -> in_by x ys && go xs
    in
    go xs


(* == FINITE MAPS == *)

(* destructive finite map implementation *)

module HashedITName =
  struct
    type t = internal_typname
    let equal (x,_) (y,_) = (x = y)
    let hash (x,_) = Hashtbl.hash x
  end

module ITNameHash = Hashtbl.Make(HashedITName)

type 'b itnmap = 'b ITNameHash.t

let map_empty : unit -> 'b itnmap
  = fun ()
 -> ITNameHash.create 10

let map_in : 'b itnmap -> internal_typname -> bool
  = fun xys x0
 -> ITNameHash.mem xys x0

let map_assoc : 'b itnmap -> internal_typname -> 'b option
  = fun xys x0
 -> try
      Some (ITNameHash.find xys x0)
    with
      Not_found -> None

let map_remove : 'b itnmap -> internal_typname -> unit
  = fun xys x0
 -> ITNameHash.remove xys x0

let map_plus : 'b itnmap -> internal_typname -> 'b -> unit
  = fun xys x0 y0
 -> ITNameHash.remove xys x0;
    ITNameHash.add xys x0 y0

let map_from_list : (internal_typname * 'b) list -> 'b itnmap
  = fun xys
 -> let xys' = map_empty () in
    ignore (list_map "map_from_list" (fun (x0,y0) -> map_plus xys' x0 y0) xys);
    xys'

let map_dom : 'b itnmap -> internal_typname list
  = fun xys
 -> let r = ref [] in
    ITNameHash.iter (fun k _ -> r := k::!r) xys;
    !r

let map_map : (internal_typname -> 'b -> 'c) -> 'b itnmap -> 'c itnmap
  = fun f xys
 -> let xys' = map_empty () in
    ITNameHash.iter (fun x0 y0 -> map_plus xys' x0 (f x0 y0)) xys;
    xys'

let map_iter : (internal_typname -> 'b -> unit) -> 'b itnmap -> unit
  = fun f xys
 -> ITNameHash.iter f xys

(* extracting a list from a map *)
let map_domain_to_list : 'b itnmap -> internal_typname list
  = fun xs
 -> let r = ref [] in
    ITNameHash.iter (fun k _ -> r := k::!r) xs;
    !r

let map_to_assoc : 'b itnmap -> (internal_typname * 'b) list
  = fun xs
 -> let r = ref [] in
    ITNameHash.iter (fun k v -> r := (k,v)::!r) xs;
    !r

(* == TYPECHECKER CONTEXT == *)

(* (for printing errors) *)

type tcctxt_item
  = TCCRule of string
  | TCCHide of tcctxt_item  (* just here for side-effect on printer_state; don't print *)
  | TCCBox of tcctxt_item   (* just here for print; avoid side-effect on printer_state *)
  | TCCSignature of signature
  | TCCSignatureItem of signature_item
  | TCCStructure of structure
  | TCCAtomicVersion of atomic_version
  | TCCVersion of version
  | TCCVersionCon of version_constraint
  | TCCRSI of resolvespec_item
  | TCCHash of hash
  | TCCHashOption of hash option
  | TCCEnv of nametypeenv
  | TCCNEnv of nameenv
  | TCCTEnv of typeenv
  | TCCTyp of typ
  | TCCKind of kind
  | TCCIIdent of internal_ident
  | TCCEIdent of external_ident
  | TCCITypname of internal_typname
  | TCCETypname of external_typname
  | TCCModname of modname
  | TCCEqModname of modname option
  | TCCEq of eqn
  | TCCEqs of eqs
  | TCCX of hash_or_modname
  | TCCExpr of expr
  | TCCExprOption of expr option
  | TCCMtch of mtch
  | TCCPat of pat
  | TCCStore of store
  | TCCSDefinition of source_definition
  | TCCSDefinitions of source_definitions
  | TCCCDefinition of definition
  | TCCCDefinitions of definitions
  | TCCLikespec of likespec
  | TCCMBody of marshalled_body
  | TCCLoc of Location.t
  | TCCNameValue of name_value

type tcctxt = tcctxt_item list list

(* cunningly preserves/updates the printer state as it goes; when
   additional info arrives, it uses it instead of or as well as the
   information it already has. *)
let render_tcctxt_item : printer_state -> Format.formatter -> tcctxt_item -> printer_state
  = fun ps ppf tcci
 -> let rec go ppf tcci =
    match tcci with
      TCCRule(rn)            -> fprintf ppf "rule %s" rn; ps
    | TCCHide(tcci')         -> go null_formatter tcci'
    | TCCBox(tcci')          -> let _ = go ppf tcci' in ps
    | TCCSignature(sign)     -> fprintf ppf "signature %a" (pp_print_signature ps) sign; ps
    | TCCSignatureItem(si)   -> fprintf ppf "signature item %a" (pp_print_signature_item ps) si; ps
    | TCCStructure(str)      -> fprintf ppf "structure %a" (pp_print_structure ps) str; ps
    | TCCAtomicVersion(av)   -> fprintf ppf "atomic version number expression %a" (pp_print_atomic_version ps) av; ps
    | TCCVersion(v)          -> fprintf ppf "version number expression %a" (pp_print_version ps) v; ps
    | TCCVersionCon(vc)      -> fprintf ppf "version constraint %a" (pp_print_version_constraint ps) vc; ps
    | TCCRSI(rsi)            -> fprintf ppf "resolvespec item %a" (pp_print_resolvespec_item ps) rsi; ps
    | TCCHashOption(None)    -> fprintf ppf "absent hash"; ps
    | TCCHashOption(Some(h)) | TCCHash(h)
                             -> fprintf ppf "hash %a" (pp_print_hash ps) h; ps
    | TCCEnv(env)            -> if !(Opts.showtcenv) then
                                  (fprintf ppf "name and type environment ";
                                   pp_print_nametypeenv_ps ps ppf env)
                                else
                                  pp_print_nametypeenv_ps ps null_formatter env
    | TCCNEnv(env)           -> if !(Opts.showtcenv) then
                                  (fprintf ppf "name environment ";
                                   pp_print_nameenv ps ppf env; ps)
                                else
                                  (pp_print_nameenv ps null_formatter env; ps)
    | TCCTEnv(env)           -> if !(Opts.showtcenv) then
                                  (fprintf ppf "type environment ";
                                   pp_print_typeenv_ps ps ppf env)
                                else
                                  pp_print_typeenv_ps ps null_formatter env
    | TCCTyp(ty)             -> fprintf ppf "type %a" (pp_print_typ ps) ty; ps
    | TCCKind(k)             -> fprintf ppf "kind %a" (pp_print_kind ps) k; ps
    | TCCIIdent(id)          -> fprintf ppf "internal identifier %a" (pp_print_ident_use ps) id; ps
    | TCCEIdent(eid)         -> fprintf ppf "external identifier %a" (pp_print_ident_ext ps) eid; ps
    | TCCITypname(tn)        -> fprintf ppf "internal type name %a" (pp_print_typname_use ps) tn; ps
    | TCCETypname(etn)       -> fprintf ppf "external type name %a" (pp_print_typname_ext ps) etn; ps
    | TCCModname(mn)         -> fprintf ppf "module name %a" (pp_print_modname ps) mn; ps
    | TCCEqModname(None)     -> fprintf ppf "unloaded module"; ps
    | TCCEqModname(Some(mn)) -> fprintf ppf "loaded module %a" (pp_print_modname ps) mn; ps
    | TCCEq(eq)              -> fprintf ppf "equation %a" (pp_print_eq ps) eq; ps
    | TCCEqs(eqs)            -> fprintf ppf "equations %a" (pp_print_eqs ps) eqs; ps
    | TCCX(hmn)              -> ((match hmn with
                                    Hash(h)     -> fprintf ppf "hash %a" (pp_print_hash ps) h
                                  | Modname(mn) -> fprintf ppf "module name %a" (pp_print_modname ps) mn); ps)
    | TCCExprOption(None)    -> fprintf ppf "absent expression"; ps
    | TCCExprOption(Some(e)) | TCCExpr(e)
                             -> fprintf ppf "expression %a" (pp_print_expr ps) e; ps
    | TCCMtch(m)             -> fprintf ppf "match %a" (pp_print_mtch ps) m; ps
    | TCCPat(p)              -> (fprintf ppf "pattern ";
                                 pp_print_pat_ps ps ppf p)
    | TCCStore(s)            -> fprintf ppf "store %a" (pp_print_store ps) s; ps
    | TCCSDefinition(def)     -> (fprintf ppf "source definition ";
                                 pp_print_source_definition_ps ps ppf def)
    | TCCSDefinitions(defs)   -> (fprintf ppf "source definitions ";
                                 pp_print_source_definitions_ps ps ppf defs)
    | TCCCDefinition(def)     -> (fprintf ppf "compiled definition ";
                                  pp_print_definition_ps ps ppf def)
    | TCCCDefinitions(defs)   -> (fprintf ppf "compiled definitions ";
                                  pp_print_definitions_ps ps ppf defs)
    | TCCLikespec(ls)        -> fprintf ppf "likespec %a" (pp_print_likespec ps) ls; ps
    | TCCMBody(mb)           -> fprintf ppf "marshalled value %a" (pp_print_marshalled_value ps) (Marshalled_value mb); ps
    | TCCLoc(l)              ->  fprintf ppf "at location: %a" (pp_print_source_loc ps) l; ps
    | TCCNameValue(nv)       -> fprintf ppf "name value %a" (pp_print_name_value ps) nv; ps
    in
    go ppf tcci

let rec firstnth
  = fun n xs ->
    if n<=0 then [] else match xs with []->[] | x::xs0 -> x::(firstnth (n-1) xs0)

let render_tcctxt : printer_state -> tcctxt -> string
  = fun ps0 tcc
 -> let render_block ppf tccis
      = fprintf ppf "@[<v 3>In ";
        ignore (pp_concat_ps ps0 ppf fprintf "@ " render_tcctxt_item tccis);
        fprintf ppf "@]"
    in
    Pretty.wrap (fun ps ppf tccis ->
      fprintf ppf "@[<v>";
      pp_concat ppf fprintf "@ " render_block tccis;
      fprintf ppf "@]") ps0 (firstnth !(Opts.tcdepth) tcc)
(* P added firstnth hack to get output managable*)


(* == TYPE INFERENCE == *)

(* Type inference is just an option on inference; the only change is
   what happens when an equality is asserted.
*)


type flavour =
  | Flexi of typ option ref      (* ordinary unifiable metavar *)
  | Skolem of internal_typname   (* skolem constant; atomic constant during unification,
                                    but maps to tn in final substitution *)

type infstate = (eqs * string * flavour) itnmap
(* Status of all unification meta(type)variables.

   Note that a type variable is in this map IFF it is a
   metavariable; type variables not in the map are not
   unifiable.  Thus all unknowns must start out by
   being mapped to None.

   We require that each tyvar is used only at a
   single colour; the eqs is present to check this
   (it is not used otherwise).

   The string is info on where this tyvar came from, for debugging
   only.

   The typ option ref is present for flexible tyvars and absent for
   skolem constants.

   Note that the type stored is not necessarily as
   ground as possible; to determine the type a
   variable currently has, it may be necessary to
   indirect multiple times through the map, possibly
   beneath other constructors.
*)

type hashset = (Digest.t, unit) Hashtbl.t
let hashset_empty : unit -> hashset = fun () -> Hashtbl.create 20
let hashset_add : hashset -> Digest.t -> unit
  = fun h x ->
    Hashtbl.add h x ()
let hashset_mem : hashset -> Digest.t -> bool
  = fun h x ->
    try (Hashtbl.find h x; true) with Not_found -> false

(* hash OK-ness depends on the hash alone, and is true for all time;
   so make it global. *)
let ok_hash_set = hashset_empty ()


(* == TYPECHECKER MONAD == *)

(* Monadic (plumbing) stuff:

   Every typechecking function takes a tci as argument (first? last?),
   and returns a tco as result.  The normal result is obtained as
   r.tco_val.

   This scheme ensures that plumbing extra stuff around the
   typechecker is really easy.
*)

type tcistate = { tci_ctxt : tcctxt;  (* current context *)
                  tci_inf  : infstate option;  (* inferring?  If so, inference state *)
                  tci_pctr : int ref;  (* progress counter *)
                  tci_lang : language;  (* which syntax? *)
                  tci_hok  : hashset;  (* hashes known to be OK *)
                  tci_iidmap : internal_ident IIdentMap.t;  (* freshening map: old to new iid *)
                  tci_itnmap : internal_typname ITypnameMap.t;  (* freshening map: old to new itn *)
                     (* NB: dom itnmap and rng itnmap are always distinct from dom (the tci_inf).
                            That is, metatyvars are never mapped or targetted by itnmap.  *)
                }

type 'a tcostate = { tco_val : 'a;  (* return value *)
                     (* imperative, so no need to pass it out: tco_inf : infstate option; *)
                   }

type 'a tcm = tcistate -> 'a tcostate

(* helper to construct a tco from a return value (and the relevant tci). *)
let return : 'a -> 'a tcm
  = fun x tci
 -> { tco_val = x }

(* run *)
let runTC : language -> 'a tcm -> 'a
  = fun lang m
 -> let tco = m { tci_ctxt = []; tci_inf = None;
                  tci_pctr = ref 0; tci_lang = lang;
                  tci_hok = ok_hash_set;
                  tci_iidmap = IIdentMap.empty;
                  tci_itnmap = ITypnameMap.empty;
                }
    in
    tco.tco_val
(* see also runTC_echo_errs below *)

let runTCinf0 : language -> (internal_typname * eqs) list -> 'a tcm -> 'a
(* do an inference, with type names listed being metavariables (colour specified) *)
(* see also runTCinf below *)
  = fun lang tns m
 -> let f (tn,eqs) = (tn,(eqs,"runTCinf0",Flexi (ref None))) in
    let inf = map_from_list (list_map "runTCinf0" f tns) in
    let tco = m { tci_ctxt = []; tci_inf = Some inf;
                  tci_pctr = ref 0; tci_lang = lang;
                  tci_hok = ok_hash_set;
                  tci_iidmap = IIdentMap.empty;
                  tci_itnmap = ITypnameMap.empty;
                }
    in
    tco.tco_val

(* bind *)
let (>>=) :    'a tcm
            -> ('a -> 'b tcm)
            -> 'b tcm
  = fun m f tci
 -> let tco = m tci in
    f tco.tco_val tci
    (* no longer need to thread infstate: { tci with tci_inf = tco.tco_inf }*)

let (>>-) :    unit tcm
            -> 'b tcm
            -> 'b tcm
  = fun m1 m2 tci
 -> let tco = m1 tci in
    m2 tci  (* see comment above *)
    (* written in full rather than in terms of >>= for speed *)

(* monadic operators *)

let inCtxt : tcctxt_item list -> 'a tcm -> 'a tcm
  = fun tcc m tci
 -> m { tci with tci_ctxt = tcc :: tci.tci_ctxt }

(* Debug.print_string_really ((match tcc with
                             (TCCRule s::_) -> s
                           | _              -> "...")^", ");
*)
(* tr ',' '^Q^J' < foo.out | tr -d ' ' | sort | uniq -c *)

let getCtxt : tcctxt tcm
  = fun tci
 -> return tci.tci_ctxt tci

let debugprint : string -> 'a tcm -> 'a tcm
  = fun s m -> m (* fun tci -> Debug.print_string_really (s^" start\n"); let x=m tci in Debug.print_string_really (s^" end\n"); x
*)

let dprint : string -> unit tcm
  = fun s tci
 -> Debug.print_string_really (s ^ "\n");
    return () tci

let getinf : tcistate -> infstate
(* INTERNAL USE ONLY: extract the inference state from the monad state *)
  = fun tci
 -> match tci.tci_inf with
      None      -> raise (Never_happen "getinf: inference-only operation called in non-inferring context")
    | Some(inf) -> inf

let inferringTC : bool tcm
(* are we inferring?  (as opposed to checking) *)
  = fun tci
 -> return (is_Some tci.tci_inf) tci

let withLangTC : language -> 'a tcm -> 'a tcm
  = fun lang m tci
 -> m { tci with tci_lang = lang }

let getLangTC : language tcm
(* what language form is this? *)
  = fun tci
 -> return tci.tci_lang tci

let addHashTC : Digest.t -> unit tcm
  = fun x tci
 -> hashset_add tci.tci_hok x;
    return () tci

let hashMemTC : Digest.t -> bool tcm
  = fun x tci
 -> let b = hashset_mem tci.tci_hok x in
    return b tci

let isMetaTC : internal_typname -> bool tcm
(* is this variable a metavariable? *)
  = fun tn tci
 -> match tci.tci_inf with
      None      -> return false tci
    | Some(inf) -> return (is_Some (map_assoc inf tn)) tci

let isFlexiTC : internal_typname -> bool tcm
(* is this variable a flexi metavariable? *)
  = fun tn tci
 -> match tci.tci_inf with
      None      -> return false tci
    | Some(inf) ->
        match map_assoc inf tn with
          None                -> return false tci  (* not metavar *)
        | Some (_,_,Skolem _) -> return false tci  (* skolem constant *)
        | Some (_,_,Flexi _ ) -> return true  tci  (* flexi metavar *)

let withFreshIId : internal_ident -> (internal_ident -> 'a tcm) -> 'a tcm
  = fun iid f tci
 -> let iid' = fresh_prettyname iid in
    f iid' { tci with tci_iidmap = IIdentMap.add iid iid' tci.tci_iidmap }

let withFreshIIds : internal_ident list -> (internal_ident list -> 'a tcm) -> 'a tcm
  = fun iids f tci
 -> let iids' = List.map fresh_prettyname iids in
    f iids' { tci with tci_iidmap = List.fold_right2 IIdentMap.add iids iids' tci.tci_iidmap }

let withFreshITn : internal_typname -> (internal_typname -> 'a tcm) -> 'a tcm
  = fun itn f tci
 -> let b = (isMetaTC itn tci).tco_val in   (* HACK XXX TODO tidy *)
    if b then
      f itn tci
    else begin
      let itn' = fresh_prettyname itn in
      f itn' { tci with tci_itnmap = ITypnameMap.add itn itn' tci.tci_itnmap }
    end

let freshtypeTC : eqs -> typ tcm
(* get a fresh, unconstrained but unifiable, type (of colour eqs) *)
  = fun eqs tci
 -> let tn = fresh_internal_typname "_uu" in
    map_plus (getinf tci) tn (eqs, "freshtypeTC", Flexi (ref None));
    { tco_val = TVar(tn);
    }

let freshenTC : eqs -> internal_typname -> internal_typname tcm
(* get a fresh, unconstrained but unifiable, metavariable (of colour eqs) *)
  = fun eqs tn0 tci
 -> let tn = fresh_prettyname tn0  in
    map_plus (getinf tci) tn (eqs, "freshenTC", Flexi (ref None));
    { tco_val = tn;
    }

let freshtyvarTC : eqs -> internal_typname tcm
(* get a fresh, unconstrained but unifiable, metavariable (of colour eqs) *)
  = fun eqs tci
 -> let tn = fresh_internal_typname "_uv"  in
    map_plus (getinf tci) tn (eqs, "freshenTC", Flexi (ref None));
    { tco_val = tn;
    }

let freshskolemTC : eqs -> internal_typname -> internal_typname tcm
(* get a fresh skolem constant; *not* unifiable *)
  = fun eqs tn0 tci
 -> let tn = fresh_internal_typname ("_s_"^string_of_prettyname tn0)  in
    map_plus (getinf tci) tn (eqs, "freshskolemTC", Skolem tn0);
    { tco_val = tn;
    }

let psTC : printer_mode -> printer_state tcm
  = fun mode tci
 -> let ps0 = initial_printer_state None mode in
    match tci.tci_inf with
      None      -> return ps0 tci
    | Some(inf) -> return (add_typname_binders ps0 (map_dom inf)) tci

(* do some side-effecting operation within the monad *)
let doTC : (unit -> unit) -> unit tcm
  = fun f tci
 -> f (); return () tci


(* progress reporting *)

let progressStep : int -> unit tcm
    = let divisor = 256 in
      fun d tci ->
      (if !(Opts.showprogress) then
        let n = !(tci.tci_pctr) + d in
        tci.tci_pctr := n mod divisor;
        if n >= divisor then
          Debug.print_string_really ".");
      return () tci


(* == EXCEPTIONS == *)

exception TCFail of string         (* doesn't pass type check *)

(* Errors *)

let tcerr_ps : (printer_state -> string * string * string) -> string tcm
  = fun f
 -> getCtxt >>= function tcctxt ->
    psTC (pm_debug ()) >>= function ps0 ->
    let (rule,err,code) = f (notrail_HACK ps0) in  (* TODO: ugh! fix tcfail invocations to use ppf *)
    inferringTC >>= function b ->
    return ("\nType "^(if b then "inference" else "checking")^" error in "^rule^": "^err^"\n"
            ^(if code = "" then "" else (code^"\n"))
            ^render_tcctxt ps0 tcctxt)

let tcerr : string * string * string -> string tcm
  = fun stuff
 -> tcerr_ps (function _ -> stuff)

let tcfail_ps : (printer_state -> string * string * string) -> 'a tcm
  = fun f
 -> tcerr_ps f >>= function s ->
    raise (TCFail(s))

let tcfail : string * string * string -> 'a tcm
  = fun stuff
 -> tcfail_ps (function _ -> stuff)

let tctrace_ps : Opts.debugclass -> (printer_state -> string * string * string) -> unit tcm
  = fun cls f
 -> if !Opts.debug && List.mem cls !Opts.debugs then
      tcerr_ps f >>= function s ->
      doTC (fun () -> Debug.print' cls (fun () -> s))
    else
      return ()

let tctrace : Opts.debugclass -> string * string * string -> unit tcm
  = fun cls stuff
 -> tctrace_ps cls (function _ -> stuff)


(* == MONADIC HELPERS == *)

(* monadic map *)
let rec mapM :    ('a -> 'b tcm)
               -> 'a list
               -> 'b list tcm
  = fun f xs
 -> match xs with
      [] -> return []
    | (x::xs) -> f x       >>= function y ->
                 mapM f xs >>= function ys ->
                 return (y::ys)

let rec mapMz :    ('a -> unit tcm)
                -> 'a list
                -> unit tcm
  = fun f xs
 -> mapM f xs >>= function (_:unit list) ->
    return ()

let rec zipMz : ('a -> 'b -> unit tcm)
             -> 'a list
             -> 'b list
             -> unit tcm
  = fun f xs ys
 -> match (xs,ys) with
      ([]   ,[]   ) -> return ()
    | (x::xs,y::ys) -> f x y >>-
                       zipMz f xs ys
    | _             -> tcfail ("zipMz","IMPOSSIBLE: zipMz applied to unequal-length lists","")

(* monadic one-armed conditional *)
let unless : bool -> unit tcm -> unit tcm
  = fun b m_fail
 -> if b then return () else m_fail


(* more monadic operators *)

let lookupIId : internal_ident -> internal_ident tcm
  = fun iid tci
 -> match maybe (IIdentMap.find iid) tci.tci_iidmap with
      Some iid' -> return iid' tci
    | None -> tcfail_ps (fun ps -> "lookupIId", "Unbound identifier (NEVER HAPPENS)",
                                   print_ident_use ps iid) tci

let lookupITn : internal_typname -> internal_typname tcm
  = fun itn tci
 -> match maybe (ITypnameMap.find itn) tci.tci_itnmap with
      Some itn' -> return itn' tci
    | None -> return itn tci
              (* XXX UGLY HACK TODO: for now, we try both in the map and not in
                      the map; really we should do one or the other, but TK0 uses
                      this, and at some callsites we should and some we shouldn't.
                      No time to untangle right now, so this fixes it I hope.  *)
        (*
         * tcfail_ps (fun ps -> "lookupITn", "Unbound type name (NEVER HAPPENS)",
         *                          print_typname_use ps itn) tci
         *)


(* == FREE VARIABLE MONAD == *)

(* Free-type-variable function type:

   Given an (environment,free-names-so-far) pair, return a
   (free-names-now,result) pair.

   Monadically, the environment is treated as an environment; scope
   changes are implemented by modify_envF, which embeds a monad within
   the new scope.
*)

type 'a ftv =    eqs                        (* current colour *)
                 * eqs itnmap * eqs itnmap  (* in-scope and free variables *)
              -> eqs itnmap * 'a            (* free variables and result *)

(* monadic return *)
let returnF : 'a -> 'a ftv
  = fun x (eqs,env,names)
 -> (names,x)

(* bind *)
let ( *>>=) :    'a ftv
              -> ('a -> 'b ftv)
              -> 'b ftv
  = fun m f (eqs,env,names)
 -> let (names' ,x) = m (eqs,env,names) in
    f x (eqs,env,names')

let ( *>>-) :    unit ftv
              -> 'b ftv
              -> 'b ftv
  = fun m1 m2
 -> m1 *>>= (function () -> m2)

(* run, extracting names only *)
let runFnames : eqs -> unit ftv -> eqs itnmap
  = fun eqs m
 -> match m (eqs,map_empty (),map_empty ()) with
      (names',()) -> names'

(* is typname free? *)
let free_in : internal_typname -> unit ftv -> bool
  = fun tn m
 -> map_in (runFnames [](*dummy*) m) tn

(* monadic map *)
let rec mapMzF : ('a -> unit ftv) -> 'a list -> unit ftv
  = fun f xs
 -> match xs with
      []      -> returnF ()
    | (x::xs) -> f x *>>- mapMzF f xs

(* modify free names *)
let modify_namesF : (eqs * eqs itnmap * eqs itnmap -> eqs itnmap) -> unit ftv
  = fun f (eqs,env,names)
 -> (f (eqs,env,names), ())

(* modify environment within scope *)
let modify_envF : (eqs * eqs itnmap -> eqs itnmap) -> 'a ftv -> 'a ftv
  = fun f m (eqs,env,names)
 -> m (eqs, f (eqs,env), names)

(* add a free name *)
let add_freeF : internal_typname -> unit ftv
  = fun tn
 -> modify_namesF (fun (eqs,env,names) -> if not (map_in env tn) then (map_plus names tn eqs; names) else names)

(* add a binder *)
let add_binderF : internal_typname -> 'a ftv -> 'a ftv
  = fun tn
 -> modify_envF (function (eqs,env) -> (map_plus env tn eqs; env))

(* change colour *)
let atColourF : eqs -> 'a ftv -> 'a ftv
  = fun eqs m (_,env,names)
 -> m (eqs,env,names)


(* == FREE TYPE NAMES == *)

(* freetyvars of typ *)
let rec freetyvars_typ : typ -> unit ftv
  = fun ty
 -> match ty with
    | TTyCon0 _        -> returnF ()
    | TTup(tys)
    | TSum(tys)        -> mapMzF freetyvars_typ tys
    | TTyCon1(_,ty)    -> freetyvars_typ ty
    | TFunc(ty1,ty2)   -> mapMzF freetyvars_typ [ty1;ty2]
    | TTyName(n)       -> returnF ()
    | TVar(tn)         -> add_freeF tn
    | TXDot(x,etn)     -> returnF ()  (* NO FREE TYPNAME HERE *)
    | TForall(tn,ty)
    | TExists(tn,ty)   -> add_binderF tn (freetyvars_typ ty)

(* freetyvars of kind *)
let freetyvars_kind : kind -> unit ftv
  = fun k
 -> match k with
      KType    -> returnF ()
    | KEq(ty)  -> freetyvars_typ ty

(* freetyvars of con0 *)
let freetyvars_con0 : con0 -> unit ftv
  = fun c0
 -> match c0 with
    | Nil(ty)
    | NONE(ty) -> freetyvars_typ ty
    | B0 _     -> returnF ()

(* freetyvars of con1 *)
let freetyvars_con1 : con1 -> unit ftv
  = fun c1
 -> match c1 with
      SOME
    | TieCon
    | Node       -> returnF ()
    | Inj(i,ty)  -> freetyvars_typ ty
    | B1 _       -> returnF ()


(* freetyvars of pattern *)
let rec freetyvars_pat : pat -> unit ftv
  = fun p
 -> match p with
      PWild(ty)    -> freetyvars_typ ty
    | PVar(id,ty)  -> freetyvars_typ ty
    | PC0(c0)      -> freetyvars_con0 c0
    | PC1(c1,p)    -> freetyvars_con1 c1 *>>-
                      freetyvars_pat p
    | PCons(p1,p2) -> mapMzF freetyvars_pat [p1;p2]
    | PTup(ps)     -> mapMzF freetyvars_pat ps
    | PTyped(p,ty) -> freetyvars_pat p *>>-
                      freetyvars_typ ty

let freetyvars_op : op -> unit ftv
  = fun op
    -> match op with
         Ref(ty) | Deref(ty) | Assign(ty) | Assign'(ty) | Equal(ty) | ListConcat(ty)
           -> freetyvars_typ ty
       | Less
       | LessEq
       | Greater
       | GreaterEq
       | Mod
       | Land
       | Lor
       | Lxor
       | Lsl
       | Lsr
       | Asr
       | UMinus
       | Plus
       | Minus
       | Times
       | Divide
       | StrConcat
           -> returnF ()
       | CompareName(ty)
       | NameToString(ty)
       | CreateThread(ty)
       | Exit(ty)
           -> freetyvars_typ ty
       | Self
       | Kill
       | CreateMutex
       | Lock
       | TryLock
       | Unlock
       | CreateCVar
       | Wait
       | Waiting
       | Signal
       | Broadcast
       | Thunkify
       | Unthunkify
           -> returnF ()


(* freetyvars of eqn *)
let freetyvars_eq : eqn -> unit ftv
  = function
      EHash(h,t,ty) -> atColourF [] (freetyvars_typ ty)
    | EMod(mn,t,ty) -> atColourF [] (freetyvars_typ ty)

(* freetyvars of eqs *)
let freetyvars_eqs : eqs -> unit ftv
  = fun eqs
 -> mapMzF freetyvars_eq eqs

(* freetyvars of resolvespecitem *)
let freetyvars_rsitem : resolvespec_item -> unit ftv
  = fun rsi
 -> match rsi with
      Resolve_static_link
    | Resolve_here_already
    | Resolve_url(_)
                      -> returnF ()

(* freetyvars of resolvespec *)
let freetyvars_resolvespec : resolvespec -> unit ftv
  = fun rs
 -> mapMzF freetyvars_rsitem rs

let freetyvars_op_or_econst : op_or_econst -> unit ftv
  = function
      OEOp op -> freetyvars_op op
    | OEEconst iid -> returnF ()

(* freetyvars of <<pat>>expr *)
let rec freetyvars_pate : (pat*expr) -> unit ftv
  = fun (p,e)
 -> freetyvars_pat p *>>-
    freetyvars_expr e

(* freetyvars of mtch *)
and freetyvars_mtch : mtch -> unit ftv
  = fun mtch
 -> mapMzF freetyvars_pate mtch

(* freetyvars of expr *)
and freetyvars_expr : expr -> unit ftv
  = fun e
    -> match e.desc with
      LocC0(c0)      -> freetyvars_con0 c0
    | LocC1(c1,e)    -> freetyvars_con1 c1 *>>-
                        freetyvars_expr e
    | LocCons(e1,e2) -> mapMzF freetyvars_expr [e1;e2]
    | LocTup(es)     -> mapMzF freetyvars_expr es
    | LocOp(oe,es)   -> freetyvars_op_or_econst oe *>>-
                        mapMzF freetyvars_expr es
    | LocLazyOp(lo,es) -> mapMzF freetyvars_expr es
    | LocLoc(l)      -> returnF ()
    | LocFn(mtch)    -> freetyvars_mtch mtch
    | LocFun(ps,e)   -> mapMzF freetyvars_pat ps *>>-
                        freetyvars_expr e
    | LocTAbs(tn,e)  -> add_binderF tn (freetyvars_expr e)
    | LocInEnv _     -> raise (Never_happen "freetyvars_expr of InEnv")
    | LocClos _      -> raise (Never_happen "freetyvars_expr of Clos")
    | LocTClos _     -> raise (Never_happen "freetyvars_expr of TClos")

    | LocId(_)
    | LocDot(_,_)
    | LocHashDot(_,_)
                        -> returnF ()

    | LocIf(e1,e2,e3)   -> mapMzF freetyvars_expr [e1;e2;e3]

    | LocWhile(e1,e2)
    | LocSeq(e1,e2)
    | LocApp(e1,e2)
                        -> mapMzF freetyvars_expr [e1;e2]
    | LocTApp(e,ty)     -> freetyvars_expr e *>>-
                           freetyvars_typ ty

    | LocPack(ty1,e,ty2)-> freetyvars_typ ty1 *>>-
                           freetyvars_expr e *>>-
                           freetyvars_typ ty2

    | LocNamecase(e1,tn,iid1,iid2,e,e2,e3)
                        -> freetyvars_expr e1 *>>-
                           freetyvars_expr e *>>-
                           add_binderF tn (freetyvars_expr e2) *>>-
                           freetyvars_expr e3

    | LocUnpack(tn,iid,e1,e2)
                        -> freetyvars_expr e1 *>>-
                           add_binderF tn (freetyvars_expr e2)

    | LocMatch(e,m)     -> freetyvars_expr e *>>-
                           freetyvars_mtch m

    | LocLet(e1,(p,e2))
                        -> freetyvars_expr e1 *>>-
                           (freetyvars_pat p *>>-
                            freetyvars_expr e2)
    | LocLetMulti(ty,(ps,e),(iid,e2))
                        -> freetyvars_typ ty *>>-
                           mapMzF freetyvars_pat ps *>>-
                           freetyvars_expr e *>>-
                           freetyvars_expr e2
    | LocLetrec(ty,(id,(mtch,e)))
                        -> freetyvars_typ ty *>>-
                           freetyvars_mtch mtch *>>-
                           freetyvars_expr e
    | LocLetrecMulti(ty,(ps,(iid,(e1,e2))))
                         -> freetyvars_typ ty *>>-
                            mapMzF freetyvars_pat ps *>>-
                            freetyvars_expr e1 *>>-
                            freetyvars_expr e2
    | LocRaise(e)       -> freetyvars_expr e
    | LocTry(e,mtch)    -> freetyvars_expr e *>>-
                           freetyvars_mtch mtch

    | LocMarshal(e1,e2,ty)
                        -> freetyvars_expr e1 *>>-
                           freetyvars_expr e2 *>>-
                           freetyvars_typ ty
    | LocUnmarshal(e,ty)
                        -> freetyvars_expr e *>>-
                           freetyvars_typ ty

    | LocMarshalz(mk,e2,ty)
                        -> atColourF [] (freetyvars_expr e2) *>>-
                           freetyvars_typ ty

    | LocRET(ty)
    | LocSLOWRET(ty)
                        -> freetyvars_typ ty

    | LocCol(e,eqs',ty) -> freetyvars_typ ty *>>-
                           freetyvars_eqs eqs' *>>-
                           atColourF eqs' (freetyvars_expr e)

    | LocOP(n,oe,es)    -> atColourF [] (
                             freetyvars_op_or_econst oe *>>-
                             mapMzF freetyvars_expr es
                           )

    | LocResolve(e,mn,rs)
    | LocResolve_blocked(e,mn,rs)
                        -> freetyvars_expr e *>>-
                           freetyvars_resolvespec rs

    | LocPar(e1,e2)     -> mapMzF freetyvars_expr [e1;e2]
    | LocFresh(ty)
    | LocCFresh(ty)     -> freetyvars_typ ty
    | LocHashMvf(x,eid,ty)
                        -> freetyvars_typ ty
    | LocHashTs(ty1,e,ty)
                        -> mapMzF freetyvars_typ [ty1;ty] *>>-
                           freetyvars_expr e
    | LocHashHts(ty1,e2,e1,ty)
                        -> mapMzF freetyvars_expr [e1;e2] *>>-
                           mapMzF freetyvars_typ [ty1;ty]
    | LocSwap(e1,e2,e3) -> mapMzF freetyvars_expr [e1;e2;e3]
    | LocFreshfor(e1,e2)-> mapMzF freetyvars_expr [e1;e2]
    | LocSupport(ty,e)  -> freetyvars_typ ty *>>-
                           freetyvars_expr e
    | LocTie(mn,eid)    -> returnF ()
    | LocNameOfTie(e)
    | LocValOfTie(e)    -> freetyvars_expr e
    | LocNameValue(_)   -> returnF ()  (* NB: type inside is always ground *)


(* freetyvars of signature_item [BINDING] *)
let freetyvars_sigitem : signature_item -> 'a ftv -> 'a ftv
  = fun sigitem m
 -> match sigitem.desc with
      SVal(id,ty) -> freetyvars_typ ty *>>-
                     m
    | STyp(tn,k)  -> freetyvars_kind k *>>-
                     add_binderF (typname_int tn) (
                       m
                     )

(* freetyvars of signature *)
let freetyvars_signature : signature -> unit ftv
  = fun sign
 -> List.fold_right freetyvars_sigitem sign.desc (returnF ())

(* freetyvars of structure_item [BINDING] *)
let rec freetyvars_stritem : structure_item -> 'a ftv -> 'a ftv
  = fun sti m
 -> match sti.desc with
      StrVal(id,e)  -> freetyvars_expr e *>>-
                       m
    | StrValMulti(id,ps,e)
                    -> mapMzF freetyvars_pat ps *>>-
                       freetyvars_expr e *>>-
                       m
    | StrTyp(tn,ty) -> freetyvars_typ ty *>>-
                       add_binderF (typname_int tn) (
                         m
                       )

(* freetyvars of structure *)
let freetyvars_structure : structure -> unit ftv
  = fun str
 -> List.fold_right freetyvars_stritem str.desc (returnF ())

(* freetyvars of withspec *)
let freetyvars_withspec : withspec -> unit ftv
  = fun ws
 -> mapMzF (fun (mn,etn,ty) -> atColourF [] (freetyvars_typ ty)) ws

(* freetyvars of likespec *)
let freetyvars_likespec : likespec -> unit ftv
  = function
      LikeNone    -> returnF ()
    | LikeMod _   -> returnF ()
    | LikeStr str -> freetyvars_structure str

(* freetyvars of source_definition *)
let freetyvars_source_definition : source_definition -> unit ftv
  = function
      Mod_user(mn,md,mu) -> let weqs = list_map "freetyvars_source_definition" eMod mu.mu_withspec in
                            atColourF weqs (
                              freetyvars_signature mu.mu_sign *>>-
                              freetyvars_structure mu.mu_str
                            ) *>>-
                            freetyvars_withspec mu.mu_withspec
    | Mod_alias(mn,ma)   -> freetyvars_signature ma.ma_sign
    | Import_user(mn,md,iu)
                         -> freetyvars_signature iu.iu_sign *>>-
                            freetyvars_likespec iu.iu_likespec
    | Mark_user(mk)      -> returnF ()


(* == TYPECHECKING HELPERS == *)

(* check type has no free flexi metatypevariables *)
let assert_typ_is_ground : string -> typ -> unit tcm
  = fun rn ty
 -> inferringTC >>= function b ->
    if b then
      mapM isFlexiTC (map_domain_to_list (runFnames [](*dummy*) (freetyvars_typ ty))) >>= function ms ->
      if List.exists id ms then
        tcfail_ps (fun ps -> rn, "Colour-crossing type not fully specified", print_typ ps ty)
      else
        return ()
    else
      return ()


(* == TYPE SUBSTITUTION == *)

type tysubst = (typ,internal_typname) sum itnmap  (* internal_typname for skolem vars *)

let unfold_tysubst : (typ,internal_typname) sum -> typ
    = function
        Inl(ty) -> ty
      | Inr(tn) -> TVar(tn)

(* Perform typ-for-internal_typname substitution on program.
   NB: ASSUMES NO SHADOWING OF DOMAIN OF SUBSTITUTION.
   Substitution is simultaneous, and performed once. *)

let tysub_typname : tysubst -> internal_typname -> typ
  = fun sub tn
 -> match map_assoc sub tn with
      None           -> TVar(tn)
    | Some(x)        -> unfold_tysubst x

let tysub_typ : tysubst -> typ -> typ
  = fun sub ty ->
    let binder tn =
      match map_assoc sub tn with
        None           -> tn
      | Some(Inl(_))   -> raise (Never_happen ("tysub_typ: shadowing: "^debug_string_of_prettyname tn))
      | Some(Inr(tn')) -> tn'
    in
    let rec go ty =
    match ty with
      TVar(tn)        -> tysub_typname sub tn
    | TTyCon0 tc0     -> ty
    | TTyCon1(tc1,ty')-> TTyCon1(tc1,go ty')
    | TTup(tys)       -> TTup (list_map "tysub_typ" go tys)
    | TSum(tys)       -> TSum (list_map "tysub_typ" go tys)
    | TFunc(tya,tyb)  -> TFunc(go tya, go tyb)
    | TXDot(x,etn)    -> ty
    | TTyName(n)      -> ty
    | TForall(tn,ty') -> TForall(binder tn,go ty')  (* ASSUMING NO SHADOWING *)
    | TExists(tn,ty') -> TExists(binder tn,go ty')  (* ASSUMING NO SHADOWING *)
    in
    go ty

let tysub_kind : tysubst -> kind -> kind
  = fun sub k
 -> match k with
      KType   -> KType
    | KEq(ty) -> KEq(tysub_typ sub ty)

let tysub_eqn : tysubst -> eqn -> eqn
  = fun sub
 -> function
     EHash(h,etn,ty) -> EHash(h,etn,tysub_typ sub ty)
   | EMod(mn,etn,ty) -> EMod(mn,etn,tysub_typ sub ty)

let tysub_eqs : tysubst -> eqs -> eqs
  = fun sub eqs
 -> list_map "tysub_eqs" (tysub_eqn sub) eqs

let tysub_con0 : tysubst -> con0 -> con0
  = fun sub c0
 -> match c0 with
    | Nil(ty)  -> Nil(tysub_typ sub ty)
    | NONE(ty) -> NONE(tysub_typ sub ty)
    | B0 _     -> c0

let tysub_con1 : tysubst -> con1 -> con1
  = fun sub c1
 -> match c1 with
      SOME
    | TieCon
    | Node       -> c1
    | Inj(i,ty)  -> Inj(i, tysub_typ sub ty)
    | B1 _       -> c1

let tysub_op : tysubst -> op -> op
  = fun sub op
 -> match op with
      Ref(ty)    -> Ref(tysub_typ sub ty)
    | Deref(ty)  -> Deref(tysub_typ sub ty)
    | Assign(ty) -> Assign(tysub_typ sub ty)
    | Assign'(ty) -> Assign'(tysub_typ sub ty)
    | Equal(ty)  -> Equal(tysub_typ sub ty)
    | Less | LessEq | Greater | GreaterEq
    | Mod | Land | Lor | Lxor | Lsl | Lsr | Asr
    | UMinus | Plus | Minus | Times | Divide
                 -> op
    | ListConcat(ty) -> ListConcat(tysub_typ sub ty)
    | StrConcat  -> op
    | CompareName(ty) -> CompareName(tysub_typ sub ty)
    | NameToString(ty) -> NameToString(tysub_typ sub ty)
    | CreateThread(ty) -> CreateThread(tysub_typ sub ty)
    | Self | Kill | CreateMutex | Lock | TryLock | Unlock | CreateCVar
    | Wait | Waiting | Signal | Broadcast | Thunkify | Unthunkify
        -> op
    | Exit(ty) -> Exit(tysub_typ sub ty)

let tysub_resolvespec_item : tysubst -> resolvespec_item -> resolvespec_item
  = fun sub rsi
 -> match rsi with
      Resolve_static_link | Resolve_here_already | Resolve_url(_) -> rsi

let tysub_resolvespec : tysubst -> resolvespec -> resolvespec
  = fun sub rs
 -> list_map "tysub_resolvespec" (tysub_resolvespec_item sub) rs

let tysub_op_or_econst : tysubst -> op_or_econst -> op_or_econst
  = fun sub -> function
      OEOp op          -> OEOp(tysub_op sub op)
    | OEEconst _ as oe -> oe

let rec tysub_expr : tysubst -> expr -> expr
  = fun sub e
 -> let rec go e =
    match e.desc with
      LocC0(c0)      -> {e with desc = LocC0(tysub_con0 sub c0)}
    | LocC1(c1,e)    -> {e with desc = LocC1(tysub_con1 sub c1, go e)}
    | LocCons(e1,e2) -> {e with desc = LocCons(go e1, go e2)}
    | LocTup(es)     -> {e with desc = LocTup(list_map "tysub_expr" go es)}
    | LocOp(oe,es)   -> {e with desc = LocOp(tysub_op_or_econst sub oe, list_map "tysub_expr:2" go es)}
    | LocLazyOp(lo,es) -> {e with desc = LocLazyOp(lo, list_map "tysub_expr:3" go es)}

    | LocLoc(l)      -> e
    | LocFn(mtch)    -> {e with desc = LocFn(tysub_mtch sub mtch)}
    | LocFun(ps,e)   -> {e with desc = LocFun (list_map "tysub_expr:fun" (tysub_pat sub) ps, go e)}
    | LocTAbs(tn,e)  -> {e with desc = LocTAbs(tn, go e)}  (* ASSUME NO SHADOWING *)
    | LocInEnv _     -> raise (Never_happen "tysub at InEnv")
    | LocClos _      -> raise (Never_happen "tysub at Clos")
    | LocTClos _     -> raise (Never_happen "tysub at TClos")
    | LocId(_) | LocDot(_,_) | LocHashDot(_,_) -> e
    | LocIf(e1,e2,e3) -> {e with desc = LocIf(go e1, go e2, go e3)}
    | LocWhile(e1,e2) -> {e with desc = LocWhile(go e1, go e2)}
    | LocSeq(e1,e2)   -> {e with desc = LocSeq(go e1, go e2)}
    | LocApp(e1,e2)   -> {e with desc = LocApp(go e1, go e2)}
    | LocTApp(e,ty)   -> {e with desc = LocTApp(go e, tysub_typ sub ty)}
    | LocPack(ty1,e,ty2)
      -> {e with desc = LocPack(tysub_typ sub ty1, go e, tysub_typ sub ty2)}
    | LocNamecase(e1,tn,iid1,iid2,e,e2,e3)  (* ASSUME NO SHADOWING *)
      -> {e with desc = LocNamecase(go e1,tn,iid1,iid2,go e,go e2,go e3)}
    | LocUnpack(tn,iid,e1,e2)  (* ASSUME NO SHADOWING *)
      -> {e with desc = LocUnpack(tn,iid,go e1,go e2)}
    | LocMatch(e,m)   -> {e with desc = LocMatch(go e, tysub_mtch sub m)}
    | LocLet(e1,pe)   -> {e with desc = LocLet(go e1, tysub_pe sub pe)}
    | LocLetMulti(ty,(ps,e),(iid,e2))
                   -> {e with desc = LocLetMulti(tysub_typ sub ty,
                               (list_map "tysub_expr:letmulti" (tysub_pat sub) ps,
                                go e),
                                (iid,(go e2)))}
    | LocLetrec(ty,(id,(mtch,e)))
                   -> {e with desc = LocLetrec(tysub_typ sub ty,
                                               (id,(tysub_mtch sub mtch, go e)))}
    | LocLetrecMulti(ty,(ps,(iid,(e1,e2))))
                   -> {e with desc = LocLetrecMulti(tysub_typ sub ty,
                                  (list_map "tysub_expr:letrecmulti" (tysub_pat sub) ps,
                                   (iid,(go e1, go e2))))}
    | LocRaise(e)       -> {e with desc = LocRaise(go e)}
    | LocTry(e,mtch)    -> {e with desc = LocTry(go e,tysub_mtch sub mtch)}
    | LocMarshal(e1,e2,ty)
                     -> {e with desc = LocMarshal(go e1, go e2, tysub_typ sub ty)}
    | LocMarshalz(mk,e2,ty)
                     -> {e with desc = LocMarshalz(mk, go e2, tysub_typ sub ty)}
    | LocUnmarshal(e,ty)
                     -> {e with desc = LocUnmarshal(go e, tysub_typ sub ty)}
    | LocRET(ty)        -> {e with desc = LocRET(tysub_typ sub ty)}
    | LocSLOWRET(ty)    -> {e with desc = LocSLOWRET(tysub_typ sub ty)}
    | LocCol(e,eqs,ty)  -> {e with desc = LocCol(go e, tysub_eqs sub eqs, tysub_typ sub ty)}
    | LocOP(n,oe,es)    -> {e with desc = LocOP(n,tysub_op_or_econst sub oe,list_map "op" go es)}
    | LocResolve(e,mn,rs)
                     -> {e with desc = LocResolve(go e,mn,tysub_resolvespec sub rs)}
    | LocResolve_blocked(e,mn,rs)
                     -> {e with desc = LocResolve_blocked(go e,mn,tysub_resolvespec sub rs)}
    | LocPar(e1,e2) -> {e with desc = LocPar(go e1, go e2)}
    | LocFresh(ty)  -> {e with desc = LocFresh(tysub_typ sub ty)}
    | LocCFresh(ty) -> {e with desc = LocCFresh(tysub_typ sub ty)}
    | LocHashMvf(x,eid,ty) -> {e with desc = LocHashMvf(x,eid,tysub_typ sub ty)}
    | LocHashTs(ty1,e,ty) -> {e with desc = LocHashTs(tysub_typ sub ty1,go e,tysub_typ sub ty)}
    | LocHashHts(ty1,e2,e1,ty) -> {e with desc = LocHashHts(tysub_typ sub ty1,go e2,go e1,tysub_typ sub ty)}
    | LocSwap(e1,e2,e3) -> {e with desc = LocSwap(go e1,go e2,go e3)}
    | LocFreshfor(e1,e2)-> {e with desc = LocFreshfor(go e1,go e2)}
    | LocSupport(ty,e)  -> {e with desc = LocSupport(tysub_typ sub ty,go e)}
    | LocTie(mn,eid)    -> e
    | LocNameOfTie(e)   -> {e with desc = LocNameOfTie(go e)}
    | LocValOfTie(e)    -> {e with desc = LocValOfTie(go e)}
    | LocNameValue(nv)  -> e  (* NB: type inside is always ground *)
    in
    go e

and tysub_mtch : tysubst -> mtch -> mtch
  = fun sub mtch
 -> list_map "tysub_mtch" (tysub_pe sub) mtch

and tysub_pe : tysubst -> (pat * expr) -> (pat * expr)
  = fun sub (p,e)
 -> (tysub_pat sub p, tysub_expr sub e)

and tysub_pat : tysubst -> pat -> pat
  = fun sub p
 -> let rec go p =
    match p with
      PWild(ty)    -> PWild(tysub_typ sub ty)
    | PVar(iid,ty) -> PVar(iid,tysub_typ sub ty)
    | PC0(c0)      -> PC0(tysub_con0 sub c0)
    | PC1(c1,p)    -> PC1(tysub_con1 sub c1, go p)
    | PCons(p1,p2) -> PCons(go p1, go p2)
    | PTup(ps)     -> PTup(list_map "tysub_pat" go ps)
    | PTyped(p,ty) -> PTyped(go p, tysub_typ sub ty)
    in
    go p

let tysub_signature_item : tysubst -> signature_item -> signature_item
  = fun sub si
 -> match si.desc with
      SVal(id,ty) -> {si with desc =SVal(id, tysub_typ sub ty)}
    | STyp(tn,k)  -> {si with desc =STyp(tn, tysub_kind sub k)}

let tysub_signature : tysubst -> signature -> signature
  = fun sub sign
 -> {sign with desc=list_map "tysub_signature" (tysub_signature_item sub) sign.desc}

let tysub_structure_item : tysubst -> structure_item -> structure_item
  = fun sub sti
 -> match sti.desc with
      StrVal(id,e)  -> {sti with desc=StrVal(id, tysub_expr sub e)}
    | StrValMulti(id,ps,e) -> {sti with desc=StrValMulti(id, list_map "tysub_structure_item" (tysub_pat sub) ps, tysub_expr sub e)}
    | StrTyp(tn,ty) -> {sti with desc=StrTyp(tn, tysub_typ sub ty)}

let tysub_structure : tysubst -> structure -> structure
  = fun sub str
 -> {str with desc=list_map "tysub_structure" (tysub_structure_item sub) str.desc}

let tysub_typeenv_entry : tysubst -> typeenv_entry -> typeenv_entry
  = fun sub
 -> function
     Env_val(id,ty)   -> Env_val(id, tysub_typ sub ty)
   | Env_loc(l,ty)    -> Env_loc(l, tysub_typ sub ty)
   | Env_mod(mn,sign) -> Env_mod(mn,tysub_signature sub sign)
   | Env_typ(tn,k)    -> Env_typ(tn,tysub_kind sub k)

let tysub_typeenv : tysubst -> typeenv -> typeenv
  = fun sub env
 -> mktypeenv (list_map "tysub_typeenv" (tysub_typeenv_entry sub) (typeenv_list_of_typeenv env))

let tysub_withspec : tysubst -> withspec -> withspec
  = fun sub ws
 -> list_map "tysub_withspec" (function (mn,etn,ty) -> (mn, etn, tysub_typ sub ty)) ws

let tysub_likespec : tysubst -> likespec -> likespec
  = fun sub ls
 -> match ls with
      LikeNone | LikeMod(_) -> ls
    | LikeStr(str)          -> LikeStr(tysub_structure sub str)

let tysub_source_definition : tysubst -> source_definition -> source_definition
  = fun sub d
 -> (if !(Opts.showprogress) then
       Debug.print_string_really "o"
    );
    match d with
      Mod_user(mn,md,mu) ->
        Mod_user(mn,md,{
                      mu_sign     = tysub_signature sub mu.mu_sign;
                      mu_vne      = mu.mu_vne;
                      mu_str      = tysub_structure sub mu.mu_str;
                      mu_withspec = tysub_withspec sub mu.mu_withspec; })
    | Mod_alias(mn,ma) ->
        Mod_alias(mn,{ ma_sign    = tysub_signature sub ma.ma_sign;
                       ma_modname = ma.ma_modname; })
    | Import_user(mn,md,iu) ->
        Import_user(mn,md,{
                         iu_sign        = tysub_signature sub iu.iu_sign;
                         iu_vce         = iu.iu_vce;
                         iu_likespec    = tysub_likespec sub iu.iu_likespec;
                         iu_resolvespec = tysub_resolvespec sub iu.iu_resolvespec;
                         iu_mo          = iu.iu_mo; })
    | Mark_user(mk) ->
        Mark_user(mk)


(* == UNIFICATION == *)

(* includes some extra monadic operators for the moment; will fix once understood - K *)

let rec chase : eqs -> typ -> typ tcm
(* If the given type is a variable, find the type to which it is
   ultimately mapped (don't descend inside type though); otherwise,
   return the type unchanged.  *)
  = fun eqs ty tci
 -> match ty with
      TVar(tn) ->
        (match map_assoc (getinf tci) tn with
           None    -> return ty tci  (* not a metavariable *)
         | Some(eqs',org,r) ->
             match r with
               Skolem _ -> return ty tci  (* skolem constant *)
             | Flexi(r) ->
                 if (not (eqs_eq eqs' eqs)) then  (* DEBUGGING ONLY *)
                   tcfail_ps (fun ps ->
                     "chase", "Incompatible colours (internal error)",
                     "Metavariable "^print_typname_use ps tn^" with origin "^org
                     ^"\nCreated at "^print_eqs ps eqs'
                     ^"\nAttempted chase at "^print_eqs ps eqs
                             ) tci
                 else
                   match !r with
                   | None            -> return ty tci  (* unconstrained *)
                   | Some(ty')       -> chase eqs ty' tci)  (* chase on down *)


    | _ -> return ty tci  (* not a (meta/)variable *)

let rec update : bool -> nametypeenv -> eqs -> internal_typname -> typ -> unit tcm
(* update a type name, so that it maps to the given type *)
  = fun isnotinskolem ((nenv,tenv) as env) eqs tn ty tci
 -> let is_not_meta_tn tn = is_None (map_assoc (getinf tci) tn) in
    let is_meta_tn tn = is_Some (map_assoc (getinf tci) tn) in
    match map_assoc (getinf tci) tn with
      None -> raise (Never_happen "update: attempt to map a non-metavariable")
    | Some(eqs',org,r) ->
        if (not (eqs_eq eqs' eqs)) then  (* DEBUGGING ONLY *)
          tcfail_ps (fun ps ->
                     "update", "Incompatible colours (internal error)",
                     "Metavariable "^print_typname_use ps tn^" with origin "^org
                     ^"\nCreated at "^print_eqs ps eqs'
                     ^"\nAttempted update at "^print_eqs ps eqs
                     ) tci
        else
          match r with
            Skolem _ ->
            begin match lookup_typeenv_itypname tenv tn with
              Some(KEq(ty1)) -> unify_typ env eqs ty1 ty tci
            | _              -> match ty with
                 TVar(tn') when isnotinskolem && is_meta_tn tn' ->
                      update false env eqs tn' (TVar(tn)) tci
               | _ -> tcfail_ps (fun ps -> "update",
                     "Attempt to map a skolem constant (possibly: existential type escapes unpack?)",
                     "Skolem variable "^print_typname_use ps tn^"\nwith type"^print_typ ps ty) tci
            end
          | Flexi(r) ->
              match !r with
                Some(_) -> raise (Never_happen "update: metavar not fully dereferenced")
              | None ->
                  (let ftvs = runFnames eqs (freetyvars_typ ty) in
                  if map_in ftvs tn then
                    tcfail_ps (fun ps ->
                            "update", "Unification would result in infinite type and we don't want any of *that* sort in *here*, now, do we?",
                            "Metavariable "^print_typname_use ps tn
                            ^" assigned type "^print_typ ps ty) tci
                  else (* XXX HACK TODO for now, turn off this safety check too, to allow metatyvars inside an unpack to be unified with the binder.  Fix later, because we really want this check I think! *)
                  (*         if List.exists is_not_meta_tn (map_dom ftvs) then
                   *   tcfail_ps (fun ps ->
                   *           "update", "Attempt to unify with possibly alpha-varying tyvar; probable typechecker confusion (INTERNAL ERROR)",
                   *           "Metavariable "^print_typname_use ps tn
                   *           ^" assigned type "^print_typ ps ty) tci
                   *
                   * else *)
                    (r := Some(ty);
                     return () tci))

and is_meta : (internal_typname -> bool) tcm
  = fun tci
 -> return (function tn -> is_Some (map_assoc (getinf tci) tn)) tci

and unify_typ : nametypeenv -> eqs -> typ -> typ -> unit tcm
(* attempt to unify two types at a given colour; colour used only for checking *)
(* NB: do not use; see below for tc_unify_typ that uses environment and eqs fully
   (defined in terms of this one) *)
  = fun env eqs ty1 ty2
 -> chase eqs ty1 >>= function ty1' ->
    chase eqs ty2 >>= function ty2' ->
    is_meta >>= function is_meta_tn ->
    match (ty1',ty2') with
      (TVar(tn1)    ,_            ) when is_meta_tn tn1 ->
                                    (match ty2' with
                                       TVar(tn2) when tn_eq tn1 tn2
                                                        -> return ()
                                     | _                -> update true env eqs tn1 ty2')
    | (_            ,TVar(tn2)    ) when is_meta_tn tn2 -> update true env eqs tn2 ty1'

    | (TTyCon0 c1      ,TTyCon0 c2 )      when c1 = c2 -> return ()
    | (TTyCon1(c1,ty1'),TTyCon1(c2,ty2')) when c1 = c2 -> unify_typ env eqs ty1' ty2'
    | (TTup(tys1)   ,TTup(tys2)   ) when List.length tys1 = List.length tys2
                                    -> zipMz (unify_typ env eqs) tys1 tys2
    | (TSum(tys1)   ,TSum(tys2)   ) when List.length tys1 = List.length tys2
                                    -> zipMz (unify_typ env eqs) tys1 tys2
    | (TFunc(ty1a,ty1b),TFunc(ty2a,ty2b))
                                    -> zipMz (unify_typ env eqs) [ty1a;ty1b] [ty2a;ty2b]
    | (TXDot(x1,etn1),TXDot(x2,etn2)) when x_eq x1 x2 && etn1 = etn2
                                    -> return ()
    | (TVar(tn1)    ,TVar(tn2)    ) when tn_eq tn1 tn2  (* concrete, not meta- *)
                                    -> return ()
    | (TForall(itn1,ty1b), TForall(itn2,ty2b))
    | (TExists(itn1,ty1b), TExists(itn2,ty2b))
        -> (isFlexiTC itn1 >>= function b1 ->
           isFlexiTC itn2 >>= function b2 ->
           if b1 || b2 then
             tcfail_ps (fun ps ->
                        "unify_typ","Cannot unify type binding construct having metavar binder (INTERNAL ERROR?)",
                        print_typ ps ty1' ^"\n and \n"^print_typ ps ty2')
           else
             (* this is safe because we check in update that only metavars appear
                free in an update RHS *)
             let ty2b' = swap_prettyname itn1 itn2 ty2b in
             unify_typ env eqs ty1b ty2b')
    | (_            ,_            ) -> tcfail_ps (fun ps ->
                                                  "unify_typ","Types cannot be unified",
                                                  print_typ ps ty1' ^"\n and \n"^print_typ ps ty2')

(* ------------------------------------------------------------------ *

Thoughts about unifying with binding constructs
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

I think we can get away with this:

- all types in the tci_inf map (the global unifying substitution) have
  the property that any free tyvars within them are metavars, either
  ordinary (flexible) metavars or skolem (constant) metavars.
- the point is that skolem metavars can't be unified, so they don't
  change (which would lose polymorphism); but they are metavars, so
  they are known not to be subject to binders elsewhere that might
  cause them to alpha-vary.
- when unifying two binding type expressions, we first construct a
  skolem metavar, and use that (by swapping) to name the binder in
  each.  We then unify the bodies.

==> actually, it doesn't work.

Here are the two use patterns we need to support:

1. unify((exists t1. t1*int), (exists t2. t2*int))
   -> should say "yes"
2. extract_exists_body((exists t1. t1*int), t0)
   -> should say "SOME(t0*int)"

1. is fine, and is supported by swapping t1/t2 or t2/t1 then unifying
the bodies, with the restriction that no metavar RHS may mention a
non-meta tyvar.

2. is a problem.  Why?  What is the answer to

     extract_exists_body((__u), t0)
     -> ?

It clearly needs to unify __u with something, but what?  If it maps
__u to exists t3. __u1, then we're stuck later, because the body of
the existential cannot now mention its binder, t3, due to the
restriction!

--KW

 * ------------------------------------------------------------------ *)

let rec explode_forall : eqs -> internal_typname -> typ -> typ tcm
(* attempt to extract the body of typ, assuming it is a forall.
   Substitutes the given tyvar for the binder. *)
(* NB: do not use; see below for tc_explode_forall that uses environment and eqs fully
   (defined in terms of this one) *)
  = fun eqs itn0 ty
 -> chase eqs ty >>= function ty' ->
    is_meta >>= function is_meta_tn ->
    match ty' with
    | TForall(itn1,ty1) ->
        let ty1' = swap_prettyname itn1 itn0 ty1 in
        return ty1'
    | _ ->
        tcfail_ps (fun ps ->
                   "explode_forall","Type is not a forall type",print_typ ps ty')

let rec explode_exists : eqs -> internal_typname -> typ -> typ tcm
(* attempt to extract the body of typ, assuming it is an exists.
   Substitutes the given tyvar for the binder. *)
(* NB: do not use; see below for tc_explode_exists that uses environment and eqs fully
   (defined in terms of this one) *)
  = fun eqs itn0 ty
 -> chase eqs ty >>= function ty' ->
    is_meta >>= function is_meta_tn ->
    match ty' with
    | TExists(itn1,ty1) ->
        let ty1' = swap_prettyname itn1 itn0 ty1 in
        return ty1'
    | _ ->
        tcfail_ps (fun ps ->
                   "explode_exists","Type is not an existential type",print_typ ps ty')


let rec get_substTC_gen : bool -> bool -> typ option
  -> ((tysubst -> 'b tcm),           (* either a continuation to receive a substitution *)
      typ * (typ -> 'b tcm)) sum     (* or a type and a continuation to receive the resulting type of that type *)
  -> 'b tcm
(* Fully expand each variable, with underspecified types being defaulted
   to the given type if present; return the resulting substitution. *)
(* WARNING: assumes binders never shadow metavariables *)
(* followSkolem: substitute the final var for skolem constants *)
(* doDefault:    default unconstrained vars if possible *)
  = fun followSkolem doDefault deftyo todo tci
 -> let inf = getinf tci in
    let dom = map_dom inf in
    let sub = map_map (fun tn _ -> ref None) inf in
    let rec go ty : typ =
    match ty with
      TVar(tn)        ->
        (match map_assoc sub tn with
           None      -> ty   (* not a metavar *)
         | Some(r)   ->      (* is a metavar; look up *)
         match !r with
         | Some(x)   -> unfold_tysubst x  (* already computed *)
         | None      ->      (* not yet computed; compute it *)
             let x   = match the (map_assoc inf tn) with
                         (_,org,r') ->
                       match r' with
                       | Skolem tn0 -> if followSkolem then Inr(tn0) else Inl(ty) (* skolem constant *)
                       | Flexi(r') ->
                       match !r' with
                         None -> if not doDefault then Inl(ty) else
                                 (match deftyo with
                                    None -> let _ =
                                            tcfail_ps (fun ps ->
                                      "chase_inside","Type underspecified and defaulting disabled",
                                      "Metavariable "^print_typname_use ps tn
                                      ^" with origin "^org) tci
                                            in Inl(TTyCon0 TUnit)  (* dummy, not reached due to exception *)
                                  | Some(defty) -> Inl(defty))  (* defaulting *)
                       | Some(ty0') -> Inl(go ty0') in
             r := Some(x);  (* store computed (fully-expanded) type in substitution, *)
             unfold_tysubst x  (* and return it *)
        )
    | TTyCon0 _       -> ty
    | TTyCon1(c1,ty') -> TTyCon1(c1,go ty')
    | TTup(tys)       -> TTup (list_map "get_substTC:TTup" go tys)
    | TSum(tys)       -> TSum (list_map "get_substTC:TSum" go tys)
    | TFunc(tya,tyb)  -> TFunc(go tya, go tyb)
    | TXDot(x,etn)    -> ty
    | TTyName(n)      -> ty
    | TForall(tn,ty') -> TForall(tn, go ty')  (* ASSUMES NO SHADOWING *)
    | TExists(tn,ty') -> TExists(tn, go ty')  (* ASSUMES NO SHADOWING *)
    in
    (match todo with
      Inl(k) ->  (* return the whole substitution *)
        map_iter (fun tn _ -> let (_:typ) = go (TVar tn) in ()) inf;
        k (map_map (fun _ r -> the !r) sub) tci
    | Inr(ty0,k) ->  (* just return the resulting type of this type *)
        let ty = go ty0 in
        k ty tci)



let rec get_substTC : typ option -> tysubst tcm
(* Fully expand each variable, with underspecified types being defaulted
   to the given type if present; return the resulting substitution. *)
(* WARNING: assumes binders never shadow metavariables *)
  = fun deftyo tci
 -> get_substTC_gen true true deftyo (Inl(function subst -> return subst)) tci


let rec get_one_partial_substTC : typ -> typ tcm
(* Fully expand one type as much as possible so far, without
   doing defaulting or following skolem vars *)
(* WARNING: assumes binders never shadow metavariables *)
  = fun ty tci
 -> get_substTC_gen false false None (Inr(ty, function ty' -> return ty')) tci


(* == THE MAIN TYPECHECKING JUDGEMENTS == *)

(* sf: signature flat *)
let tc_sf : signature -> unit tcm
  = fun sign0
 -> inCtxt [TCCRule "sf"; TCCSignature sign0; TCCLoc sign0.loc] <$>
    let rec go sign =
    match sign with
      [] -> return ()
    | (s::sign') -> (match s.desc with
    | (SVal(x,ty))     -> go sign'
    | (STyp(tn,KEq ty))
      -> unless (not (free_in (typname_int tn) (freetyvars_signature {sign0 with desc=sign'})))
                      (* TODO: this is inefficient (quadratic); should
                         check the other way, accumulating env and
                         checking against it as I go, rather than
                         scanning what's ahead *)
           (tcfail_ps (fun ps ->
                       "sf","Eq-kinded type field must not be used internally within flat sig",
                       "Type field "^print_typname_bind ps tn)) >>-
         go sign'
    | (STyp(tn,KType)) -> go sign')
    in
    go sign0.desc

(* Sf: not relevant, as the AST doesn't distinguish sig and Sig *)

(* stf: structure flat *)
let tc_stf : structure -> unit tcm
  = fun str0
 -> inCtxt [TCCRule "stf"; TCCStructure str0; TCCLoc str0.loc] <$>
    let rec go str =
    match str with
      [] -> return ()
    | s::str' -> (match s.desc with
    | (StrVal(x,e)) -> go str'
    | (StrValMulti(x,ps,e))
      -> (getLangTC >>= function lang ->
          unless (lang = SugaredSourceInternalForm)
            (tcfail ("stf","Syntactic sugar not allowed here","")) >>-
          go str')
    | (StrTyp(tn,ty))
      -> unless (not (free_in (typname_int tn) (freetyvars_structure {str0 with desc=str'})))
           (tcfail_ps (fun ps ->
                       "stf","Type field must not be used internally within flat structure",
                       "Type field "^print_typname_bind ps tn)) >>-
         go str')
    in
    go str0.desc


(* this judgement is used in tc_v*ok, but is defined later.  Keith
   guarantees that it is used safely, and loops will not result.
   However, we'd like to know that the topological sort is correct
   with this edge removed, so we resort to this trickery. *)
let rec_tc_hok = ref (fun _ -> raise (Never_happen "rec_tc_hok not filled in!"))

(* test h ok in version grammar; here and only here HashLits are OK *)
let tc_vhok : string -> nameenv -> hash -> unit tcm
  = fun rulename nenv h
 -> getLangTC >>= function lang ->
    match h with
      HashLit(_) -> return ()  (* is always fine; there's no checking we can do *)
    | HashM(_)
    | HashI(_)
    | HashName(_) -> unless (lang = CompiledForm) (tcfail (rulename,"Structured hash permitted only in compiled code","")) >>-
                     !rec_tc_hok nenv h

let tc_avnxok : string -> bool -> nameenv -> atomic_version -> unit tcm
  = fun rulename eok nenv av
 -> inCtxt [TCCRule rulename; TCCAtomicVersion av] <$>
    match av with
    | VLiteral(VNumber(n))
                  -> unless (n>=0)
                       (tcfail_ps (fun ps ->
                                   rulename,"Version number must be positive",
                                   "Version fragment "^print_atomic_version ps av))
    | VLiteral(VHash(h)) -> tc_vhok rulename nenv h
    | VMyname when not eok
                  -> tcfail (rulename,"myname not permitted in version number *value*","")
    | VMyname -> return ()

let rec tc_vnxok : string -> bool -> nameenv -> version -> unit tcm
  = fun rulename eok nenv v
 -> inCtxt [TCCRule rulename; TCCVersion v] <$>
    match v with
      VAtomic(av) -> tc_avnxok rulename eok nenv av
    | VCons(av,v) -> tc_avnxok rulename eok nenv av >>-
                     tc_vnxok rulename eok nenv v

let tc_vneok : nameenv -> version -> unit tcm
  = fun nenv v
 -> tc_vnxok "vneok" true nenv v

let tc_vnok : nameenv -> version -> unit tcm
  = fun nenv v
 -> tc_vnxok "vnok" false nenv v

let tc_ahvcxok : string -> nameenv -> typeenv option -> atomic_hash_version_constraint -> unit tcm
  = fun rulename nenv envo ahvc
 -> let eok = is_Some envo in
    let tcfail' s = tcfail (rulename,s,"") in
    match ahvc with
    | AVCHHash(h)     -> tc_vhok rulename nenv h
    | AVCHModname(_) when not eok
                      -> tcfail' "Module name not allowed in vc version constraint"
    | AVCHModname(mn) -> unless (modname_mem_typeenv mn (the envo))
                          (tcfail' "Module name in version constraint not in scope")

let tc_avcxok : string -> nameenv -> typeenv option -> atomic_version_constraint -> unit tcm
  = fun rulename nenv envo avc
 -> let eok = is_Some envo in
    let tcfail' s = tcfail (rulename,s,"") in
    match avc with
      AVCHashVersion ahvc -> tc_ahvcxok rulename nenv envo ahvc
    | AVCNumber n
        -> unless (n>=0) (tcfail' "Version number must be positive")

let tc_vcxok : string -> nameenv -> typeenv option -> version_constraint -> unit tcm
  = fun rulename nenv envo vc
 -> let eok = is_Some envo in
    inCtxt [TCCRule rulename; TCCVersionCon vc] <$>
    let tcfail' s = tcfail (rulename,s,"") in
    let goTVC tvc =
    match tvc with
      TVCAtomic(avc)     -> tc_avcxok rulename nenv envo avc
    | TVCBetween(n1,n2)  -> unless (0<=n1 && n1<=n2)
                              (tcfail' "Version interval must be sane")
    | TVCBefore(n2)      -> unless (n2>=0)
                              (tcfail' "Version interval must be sane")
    | TVCAfter(n1)       -> unless (n1>=0)
                              (tcfail' "Version interval must be sane")
    | TVCAnything        -> return ()
    in
    match vc with
      VCNameEqual(ahvc)
        -> (getLangTC >>= function lang ->
            match ahvc with
              AVCHModname(_) -> tc_ahvcxok rulename nenv envo ahvc
            | _ when (lang <> CompiledForm) -> tcfail' "Exact hash constraint in user source must be to module name only"
            | _ -> tc_ahvcxok rulename nenv envo ahvc)
    | VCDotted(avcs,tvc)
        -> mapMz (tc_avcxok rulename nenv envo) avcs >>-
           goTVC tvc

let tc_vceok : nametypeenv -> version_constraint -> unit tcm
  = fun (nenv,env) vc
 -> tc_vcxok "vceok" nenv (Some env) vc

let tc_vcok : nameenv -> version_constraint -> unit tcm
  = fun nenv vc
 -> tc_vcxok "vcok" nenv None vc


let tc_urlok : string -> unit tcm
  = fun url
 -> try
      ignore (Uri.parse url);
      return ()
    with
      Uri.BadURI(s) -> tcfail ("urlok",s,"URI: "^url)

let tc_rsiok : resolvespec_item -> unit tcm
  = fun rsi
 -> inCtxt [TCCRule "rsiok"; TCCRSI rsi] <$>
    begin
    match rsi with
      Resolve_static_link  -> return ()
    | Resolve_here_already -> return ()
    | Resolve_url(url)     -> tc_urlok url
    end

let tc_rsok : resolvespec -> unit tcm
  = fun rs
 -> mapMz tc_rsiok rs


(* these two judgements are used in hok, but are defined later.  Keith
   guarantees that they are used safely, and loops will not result.
   However, we'd like to know that the topological sort is correct
   with these edges removed, so we resort to this trickery. *)
let rec_tc_sok = ref (fun _ -> raise (Never_happen "rec_tc_sok not filled in!"))
let rec_tc_sts = ref (fun _ -> raise (Never_happen "rec_tc_sts not filled in!"))

let tc_mhok : nameenv -> mod_hash_body -> unit tcm
  (* NB: assumes Eok econst_env *)
  = fun nenv mh
 ->
   Debug.print' Opts.DBC_namevalues (fun () -> "tc_mhok" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
   !rec_tc_sts (nenv,econst_env) mh.mh_eqs mh.mh_str mh.mh_sign0 >>-
    tc_stf mh.mh_str >>-
    tc_sf mh.mh_sign0 >>-
    tc_vneok nenv mh.mh_vne

let tc_ihok : nameenv -> import_hash_body -> unit tcm
  (* NB: assumes Eok econst_env *)
  = fun nenv ih
 -> !rec_tc_sts (nenv,emptytypeenv) [] ih.ih_likestr (limitdom ih.ih_sign0) >>-
    !rec_tc_sok (nenv,emptytypeenv) ih.ih_sign0 >>-
    tc_stf ih.ih_likestr >>-
    tc_sf ih.ih_sign0 >>-
    tc_vcok nenv ih.ih_vce

let tc_nhok : nameenv -> abstract_name -> unit tcm
  = fun nenv n
 -> match lookup_nameenv n nenv with
    | Some (NEnv_nmod(_,mn,mh)) -> tc_mhok nenv mh
    | Some (NEnv_nimp(_,mn,ih)) -> tc_ihok nenv ih
    | Some (NEnv_type(_)) ->
        tcfail ("nhok","Type name cannot be used as hash","")
    | Some (NEnv_tname(_,ty)) ->
        tcfail ("nhok","Term name cannot be used as hash","")
        (* what do we call it?  `term name' doesn't sound right -K *)
    | None ->
        tcfail ("nhok","Unknown name used as hash","")


let tc_hok : nameenv -> hash -> unit tcm
(* XXX unsafe because nameenv might change.  In fact probably safe, since
 * En ought to be monotonic, but we're not checking... ouch!  P/K agree it's
 * OK to do it this way.
 *)
  = fun nenv h
 -> hashMemTC (hash_of_hash h) >>= function b ->
    if b then
      return ()
    else
    (inCtxt [TCCRule "hok"; TCCHash h] <$>
    let tcfail' s = tcfail_ps (fun ps -> "hok",s,"Hash "^print_hash ps h) in
    match h with
    | HashM(lithash,m,mh)
      -> tc_mhok nenv mh >>-
         addHashTC lithash
    | HashI(lithash,m,ih)
      -> tc_ihok nenv ih >>-
         addHashTC lithash
    | HashName(n)
      -> tc_nhok nenv n >>-
         addHashTC n
    | HashLit(_)
      -> tcfail' "Cannot typecheck a numeric hash")

(* fill in recursive reference *)
let () = rec_tc_hok := tc_hok


let tc_MS00 : typeenv -> modname -> (signature * typeenv) tcm
  (* NB: caller guarantees tc_Eok env *)
  (* if typeenv is env2 @ mn:sign @ env1, then result is Some(sign,env1) *)
  (* (recall environment is stored right-to-left, so we have env1 |- sign ok) *)
  = fun env mn
 -> inCtxt [TCCRule "MS00"; TCCTEnv env; TCCModname mn] <$>
    match lookup_typeenv_modname_ext env mn with
      None       -> tcfail_ps (fun ps -> "MS00","Module name "^print_modname ps mn ^" not in environment\n "^ fst(print_typeenv_ps ps env )^ "\n\n","")
    | Some(sign,env') -> return (sign, env')


let tc_MS0 : nametypeenv -> modname -> signature tcm
  (* NB: the typeenv is used, but the nameenv is unused;
     it's only here for symmetry with hS0, which *does* use it
      essentially. *)
  (* NB: caller guarantees tc_Eok env *)
  = fun (nenv,env) mn
 -> inCtxt [TCCRule "MS0"; TCCEnv (nenv,env); TCCModname mn] <$>
    match lookup_typeenv_modname env mn with
      None       -> tcfail_ps (fun ps -> "MS0","Module name "^print_modname ps mn ^" not in environment\n "^ fst(print_typeenv_ps ps env )^ "\n\n","")
    | Some(sign) -> return sign


(* get sign0 of a hash *)
let tc_hS00 : string -> nameenv -> hash -> signature tcm
  = fun rulename nenv h
 -> (match h with
       HashM(_,_,mh) -> return mh.mh_sign0
     | HashI(_,_,ih) -> return ih.ih_sign0
     | HashName(n) ->
         (match lookup_nameenv n nenv with
          | Some (NEnv_nmod(_,mn,mh)) -> return mh.mh_sign0
          | Some (NEnv_nimp(_,mn,ih)) -> return ih.ih_sign0
          | Some (NEnv_type(_))       -> tcfail (rulename,"Type name cannot be used as hash","")
          | Some (NEnv_tname(_,ty))   -> tcfail (rulename,"Term name cannot be used as hash","")
              (* what do we call it?  `term name' doesn't sound right -K *)
          | None                      -> tcfail (rulename,"Unknown name used as hash",""))
     | HashLit _   -> tcfail (rulename, "Cannot typecheck numeric hash", ""))


let tc_hS0 : nametypeenv -> hash -> signature tcm
  (* NB: the nameenv is used, but the typeenv is unused;
     it's only here for symmetry with MS0, which *does* use it
      essentially. *)
  (* NB: caller guarantees tc_Eok env *)
  = fun (nenv,env) h
 -> inCtxt [TCCRule "hS0"; TCCEnv (nenv,env); TCCHash h] <$>
    (tc_hok nenv h >>-
     tc_hS00 "hS0" nenv h)


(* look up X.t *)
(* None => no module; Some None => no field; Some (Some k) => success *)
let flat_xetn : nametypeenv -> (hash_or_modname * external_typname) -> kind option option tcm
  (* NB: caller guarantees tc_Eok env, and x ok *)
  = fun (nenv,env) (x,etn)
 -> match x with
      Hash(h)     -> (tc_hS0 (nenv,env) h >>= function sign ->
                     (* hash signature is always flattened already *)
                     return (Some (lookup_signature_etn sign etn)))  (* XXX could be made faster by putting a map in the global hashMemTC *)
    | Modname(mn) -> return (lookup_typeenv_modname_typ env mn etn)


(* abstract_in: tt abstractin h *)
let abstract_in : string -> nameenv -> external_typname -> hash -> unit tcm
  = fun rulename nenv etn h
 -> (tc_hS00 rulename nenv h >>= function sign ->
    match lookup_signature_etn sign etn with
      None
        -> tcfail_ps (fun ps ->
                      rulename,"Type field not present in hash",
                      print_typ ps (TXDot(Hash(h),etn)))
    | Some(KEq(_) as k)
        -> tcfail_ps (fun ps ->
                      rulename,"Type field not abstract in hash",
                      print_typ ps (TXDot(Hash(h),etn)) ^ " has kind " ^ print_kind ps k)
    | Some(KType)
        -> return ())


(* type has kind Type (this property is independent of colour) *)
let tc_TK0 : nametypeenv -> typ -> unit tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun (nenv,tenv) ty
 -> inCtxt [TCCRule "TK0"; TCCEnv (nenv,tenv); TCCTyp ty] <$>
    begin
    let rec go tenv ty =
    match ty with
    | TTyCon0 _   -> return ()

    | TTup(tys)
    | TSum(tys)   -> (unless (List.length tys >= 2)
                        (tcfail ("TK0","Tuple or sum with less than two components","")) >>-
                      mapMz (go tenv) tys)

    | TTyCon1(_,ty)
                  -> go tenv ty

    | TFunc(ty1,ty2) -> mapMz (go tenv) [ty1;ty2]
    | TVar(tn)    -> (isMetaTC tn >>= function ismeta ->
                      if ismeta then return () else
                      (* NB: dom itnmap is disjoint from the set of metatyvars,
                         so this order is OK. *)
                      lookupITn tn >>= function tn' ->
                      unless (itypname_mem_typeenv tn' tenv)
                        (tcfail_ps (fun ps ->
                                    "TK0","Type variable not in scope",
                                    print_typname_use ps tn)))

    | TXDot(Modname(mn),etn)
                  -> (flat_xetn (nenv,tenv) (Modname mn, etn) >>= function koo ->
                      match koo with
                        (Some (Some k))
                        -> return ()
                      | (Some None)
                        -> tcfail_ps (fun ps ->
                                       "TK0","Type field not present in named module",
                                       print_typ ps ty)
                      | None
                        -> tcfail_ps (fun ps ->
                                       "TK0","Named module not found",
                                       print_typ ps ty))

    | TXDot(Hash(h),etn)
                  -> (tc_hok nenv h >>-
                      flat_xetn (nenv,tenv) (Hash h, etn) >>= function koo ->
                      match koo with
                        (Some (Some KType))
                        -> return ()
                      | (Some (Some (KEq _)))
                        -> tcfail_ps (fun ps ->
                                      "TK0","Type field not abstract in hash",
                                      print_typ ps ty)
                      | (Some None)
                        -> tcfail_ps (fun ps ->
                                      "TK0","Type field not present in hash",
                                      print_typ ps ty)

                      | None
                        -> tcfail_ps (fun ps ->
                                       "TK0","Hash not found",
                                       print_typ ps ty))

    | TTyName(n)  -> (match lookup_nameenv n nenv with
                        Some(NEnv_type _) ->
                          return ()
                      | Some(_) ->
                          tcfail_ps (fun ps ->
                                     "TK0","Name is not a type name",
                                     print_name_use ps n)
                      | None ->
                          tcfail_ps (fun ps ->
                                     "TK0","Type name not in scope",
                                     print_name_use ps n))

    | TForall(tn,ty)
    | TExists(tn,ty)
                  -> (withFreshITn tn (function tn' ->
                      let tenv' = addtypeenv (Env_typ(tn',KType)) tenv in
                      go tenv' ty))
    in
    go tenv ty
    end

let tc_Kok : nametypeenv -> kind -> unit tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun env k
 -> inCtxt [TCCRule "Kok"; TCCEnv env; TCCKind k] <$>
    match k with
      KType   -> return ()
    | KEq(ty) -> tc_TK0 env ty


let tc_ExTok : nametypeenv -> internal_ident -> typ -> unit tcm
  (* assuming E ok, show E,x:T ok *)
  = fun ((nenv,tenv) as env) id ty
 -> inCtxt [TCCRule "ExTok"; TCCEnv env; TCCIIdent id; TCCTyp ty] <$>
    begin
    progressStep 1 >>-
    unless (not (iident_mem_typeenv id tenv))
      (tcfail ("ExTok","Identifier name duplicated in enviroment","")) >>-
    tc_TK0 env ty
    end


let tc_EtKok : nametypeenv -> internal_typname -> kind -> unit tcm
  (* assuming E ok, show E,t:K ok *)
  = fun ((nenv,tenv) as env) tn k
 -> inCtxt [TCCRule "EtKok"; TCCEnv env; TCCITypname tn; TCCKind k] <$>
    begin
    isFlexiTC tn >>= function isflexi ->  (* NB: skolem constants allowed *)
    unless (not isflexi)
      (tcfail ("EtKok","Type name in environment is flexi metavar (INTERNAL ERROR?)","")) >>-
    unless (not (itypname_mem_typeenv tn tenv))
      (tcfail ("EtKok","Type name duplicated in enviroment","")) >>-
    tc_Kok env k
    end


let tc_sok : nametypeenv -> signature -> unit tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun (nenv,tenv) sign0
 -> inCtxt [TCCRule "sok"; TCCEnv (nenv,tenv); TCCSignature sign0; TCCLoc sign0.loc] <$>
    let rec go tenv sign =
    match sign with
      [] -> return ()
    | s::sign' -> (match s.desc with
    | (SVal(id,ty))
      -> withFreshIId (ident_int id) (function iid' ->
           tc_ExTok (nenv,tenv) iid' ty >>-
           unless (not (List.mem (ident_ext id) (list_map "tc_sok" ident_ext (sign_valfieldnames (at_loc sign0.loc sign')))))  (* TODO: inefficient *)
             (tcfail_ps (fun ps ->
                         "sok","Value field name duplicated in signature",
                         "Field "^print_ident_bind ps id^" in signature tail "^print_signature ps (at_loc sign0.loc sign))) >>-
           go (addtypeenv (Env_val(iid',ty)) tenv) sign'
         )
    | (STyp(tn,k))
      -> let (tn,sign') = freshen_internal_typname (tn,sign') in
         tc_EtKok (nenv,tenv) (typname_int tn) k >>-
         unless (not (List.mem (typname_ext tn) (list_map "tc_sok[2]" typname_ext (sign_typfieldnames (at_loc sign0.loc sign')))))  (* TODO: inefficient *)
           (tcfail_ps (fun ps ->
                       "sok","Type field name duplicated in signature",
                       "Field "^print_typname_bind ps tn^" in signature tail "^print_signature ps (at_loc sign0.loc sign))) >>-
         go (addtypeenv (Env_typ(typname_int tn,k)) tenv) sign')
    in
    go tenv sign0.desc

(* fill in recursive reference *)
let () = rec_tc_sok := tc_sok


let tc_Enok0 : nameenv -> nameenv -> nameenv tcm
  = fun nenv nenv0
 ->
   Debug.print' Opts.DBC_namevalues (fun () -> "Enok0" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
   inCtxt [TCCRule "Enok0"; TCCNEnv nenv] <$>
    let rec go nenv =
    match nenv with
      [] ->
        return nenv0
    | (ei::nenv') ->
        go nenv' >>= function nenv'' ->
        let n = name_of_nameenv_entry ei in
        (unless (not (name_mem_nameenv n nenv''))
           (tcfail_ps (fun ps ->
             "Enok0","Name duplicated in name environment",
                "Name "^print_name_use ps n)) >>-
         match ei with
          NEnv_nmod(n,mn,mh) -> tc_mhok nenv'' mh
        | NEnv_nimp(n,mn,ih) -> tc_ihok nenv'' ih
        | NEnv_type(n)       -> return ()
        | NEnv_tname(n,ty)   -> tc_TK0 (nenv'',emptytypeenv) ty
        ) >>-
        return (addnameenv ei nenv'')
    in
    go (nameenv_list_of_nameenv nenv)

let tc_Etok0 : typeenv -> nametypeenv -> nametypeenv tcm
(* E0,E ok (i.e., env @ env0 ok) assuming E0 ok *)
(* returns E0,E (i.e., env @ env0), for convenience *)
(* NB: caller guarantees tc_Eok E0 *)
  = fun tenv (nenv0,tenv0)
 -> inCtxt [TCCRule "Eok0"; TCCTEnv tenv] <$>
    let rec go tenv =
    match tenv with
      []
      -> return tenv0
    | (ei::tenv')
      -> go tenv' >>= function tenv'' ->
         (match ei with
            (Env_val(id,ty))
            -> tc_ExTok (nenv0,tenv'') id ty
          | (Env_loc(l,ty))
            -> unless (not (location_mem_typeenv l tenv''))
               (tcfail_ps (fun ps ->
                           "Eok","Location name duplicated in enviroment",
                           "Location "^print_location ps l)) >>-
               tc_TK0 (nenv0,tenv'') ty
          | (Env_mod(mn,sign))
            -> unless (not (modname_mem_typeenv mn tenv''))
                 (tcfail_ps (fun ps ->
                             "Eok","Module name duplicated in enviroment",
                             "Module name "^print_modname ps mn)) >>-
               tc_sok (nenv0,tenv'') sign
          | (Env_typ(tn,k))
            -> tc_EtKok (nenv0,tenv'') tn k
         ) >>-
         return (addtypeenv ei tenv'')
    in
    go (typeenv_list_of_typeenv tenv) >>= function tenv' ->
    return (nenv0,tenv')


let tc_Eok : nametypeenv -> unit tcm
  = fun (nenv,tenv)
 ->
   Debug.print' Opts.DBC_namevalues (fun () -> "Eok" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
   inCtxt [TCCRule "Eok"; TCCEnv (nenv,tenv)] <$>
    tc_Enok0 nenv emptynameenv >>= function _ ->
    tc_Etok0 tenv (nenv,emptytypeenv) >>= function _ ->
    return ()


let tc_eqok : nametypeenv -> eqn -> unit tcm
  = fun (nenv,env) eq
 -> inCtxt [TCCRule "eqok"; TCCEq eq] <$>
    begin
    match eq with
      EHash(h,etn,ty) ->
       (tc_hok nenv h >>-
        (match h with
             HashM(_,emn,mh) ->
               return (mh.mh_sign0,"structure",mh.mh_str)
           | HashI(_,emn,ih) ->
               return (ih.ih_sign0,"likespec",ih.ih_likestr)
           | HashName(n) ->
               (match lookup_nameenv n nenv with
                | Some (NEnv_nmod(_,mn,mh)) -> return (mh.mh_sign0,"structure",mh.mh_str)
                | Some (NEnv_nimp(_,mn,ih)) -> return (ih.ih_sign0,"likespec",ih.ih_likestr)
                | Some (NEnv_type(_))       -> tcfail ("eqok","Type name cannot be used as hash in equation","")
                | Some (NEnv_tname(_,ty))   -> tcfail ("eqok","Term name cannot be used as hash in equation","")
                    (* what do we call it?  `term name' doesn't sound right -K *)
                | None                      -> tcfail ("eqok","Unknown name used as hash in equation",""))
           | HashLit _ ->
               tcfail ("eqok","Cannot typecheck numeric hash","")
        ) >>= function (sign,strdesc,str) ->

        match lookup_signature_etn sign etn with
          None           -> tcfail_ps (fun ps ->
                                       "eqok","Type field name in equation not in hash signature",
                                       "Equation "^print_eq ps eq
                                       ^" in hash with signature "^print_signature ps sign)
        | Some(KEq(ty')) -> tcfail_ps (fun ps ->
                                       "eqok","Equation on type field with Eq kind",
                                       "Equation "^print_eq ps eq
                                       ^" in hash with signature "^print_signature ps sign)
        | Some(KType)    ->

        match lookup_structure_etn str etn with
          None -> tcfail_ps (fun ps ->
                "eqok","Type field name in equation present in hash signature but not in "^strdesc^" (weird!)",
                "Equation "^print_eq ps eq
                ^" in hash with signature "^print_signature ps sign
                ^" and structure "^print_structure ps str)
        | Some(ty') ->

        assert_typ_is_ground "eqok" ty >>-
        unless (typ_eq ty ty' (* NB: syntactic equality *) )
          (tcfail_ps (fun ps ->
                   "eqok","Type of equated type field is not syntactically equal to the given type",
                   "Equation "^print_eq ps eq
                   ^" where type field has type "^print_typ ps ty')))

    | EMod(mn,etn,ty) ->
       (tc_MS00 env mn >>= function (sign,env1) ->
        match lookup_signature_etn sign etn with
          None           -> tcfail_ps (fun ps ->
                                       "eqok","Type field name in equation not in module signature",
                                       "Equation "^print_eq ps eq
                                       ^" in module with signature "^print_signature ps sign)
        | Some(KEq(ty')) -> tcfail_ps (fun ps ->
                                       "eqok","Equation on type field with Eq kind",
                                       "Equation "^print_eq ps eq
                                       ^" in module with signature "^print_signature ps sign)
        | Some(KType)    ->

        assert_typ_is_ground "eqok" ty >>-
        tc_TK0 (nenv,env1) ty)  (* NB: the condition on this type is much weaker than for hash equalities *)
                         (* NB: env1 not env: ty must be well-formed in the env prefix up to mn *)

   end


let tc_eqsok : nametypeenv -> eqs -> unit tcm
  = fun env eqs
    -> if !Opts.disable_eqsok_typecheck then
        return ()
      else
      inCtxt [TCCRule "eqsok"; TCCEqs eqs] <$>
      let mnetns = partial_map (function EHash _ -> None | EMod (mn,etn,_) -> Some (mn,etn)) eqs in
      (match firstdup mnetn_eq mnetns with
         None -> return ()
       | Some (mn,etn) -> tcfail_ps (fun ps ->
           "eqsok","Equation set contains duplicate LHS",
           print_typ ps (TXDot(Modname(mn),etn)))) >>-
      mapMz (tc_eqok env) eqs

(* the various maps from t, MM_M.t, h.t *)
let itypnamemap : nametypeenv -> internal_typname -> typ option tcm
(* nenv for consistency with other maps only; not actually used *)
  = fun (nenv,tenv) tn
 -> match lookup_typeenv_itypname tenv tn with
      None          -> return None
 (* V: HACK comment out the following code which failed on nested forall's.
       Now it works! I can't understand.
         tcfail_ps (fun ps ->
        "itypnamemap", "IMPOSSIBLE: itypnamemap: typname not bound by environment",
        "Typname "^print_typname_use ps tn) *)
    | Some(KType  ) -> return None
    | Some(KEq(ty)) -> return (Some(ty))


let xdotmap : nametypeenv -> eqs -> (hash_or_modname * external_typname) -> typ option tcm
  (* NB: caller guarantees tc_Eok env, and h ok if x is an h *)
  = fun env eqs (x,etn)
 -> flat_xetn env (x,etn) >>= function koo ->
    match koo with
      None -> tcfail_ps (function ps ->
        ("xdotmap", "IMPOSSIBLE: module not in scope ",
         print_typ ps (txdot x etn)))
    | Some None -> tcfail_ps (function ps ->
	("xdotmap", "IMPOSSIBLE: etn not bound in signature ",
         print_typ ps (txdot x etn)))
    | Some(Some (KType  )) ->
        return (findfirst
                  (function eq ->
                    let (x',etn',ty) = match eq with
                                         EMod(mn',etn',ty) -> (Modname(mn'),etn',ty)
                                       | EHash(h',etn',ty) -> (Hash(h'),etn',ty) in
                    if xetn_eq (x,etn) (x',etn') then
                      Some(ty)
                    else
                      None) eqs)
    | Some(Some(KEq(ty))) ->
        return (if typ_eq ty (txdot x etn) then None else Some ty)
    (* NB: since we have eqs ok, and that implies that x.etn can only
       have an equation if x.etn is abstract in its natural signature,
       we know it can't both have an equation and be KEq().  So these two
       cases do not overlap, and we remain deterministic. *)

let dotmap : nametypeenv -> eqs -> (modname * external_typname) -> typ option tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun env eqs (mn,etn)
 -> xdotmap env eqs (Modname mn, etn)

let hashdotmap : nametypeenv -> eqs -> (hash * external_typname) -> typ option tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun env eqs (h,etn)
 -> xdotmap env eqs (Hash h, etn)

(* NB: all three of the above maps are directed: if we keep
   following them as far as possible, we will eventually reach a
   canonical form.  This is essential to the below algorithm.

   The property holds because: no recursive hashes or modules
   (i.e., expansion of h.t or MM_M.t can't contain h or MM_M), and
   typname maps only arise from equality kinds in environments,
   and those kinds must depend only on earlier typnames/modules.

   Furthermore, the maps are deterministic: there's only ever one
   thing that a given t/h.t/MM_M.t can map to.

   Note that this method works for Teq, only because the same
   equalities hold for ty1 as for ty2. *)


let tc_Tdireq : nametypeenv -> eqs -> typ -> typ tcm
(* directed equality: canonicalise type in E |- eqs *)
(* NB: caller guarantees Eok and eqsok *)
  = fun ((nenv,tenv) as env) eqs ty
 -> inCtxt [TCCRule "Tdireq"; TCCEnv env; TCCEqs eqs; TCCTyp ty] <$>
    let rec go env ty0
      (* transitively and congruently apply the maps *)
      (* easy and deterministic because the maps are from atomic things *)
      = match ty0 with
        | TTyCon0 _      -> return ty0
        | TTup(tys)      -> mapM (go env) tys >>= (return <.> tTup)
        | TSum(tys)      -> mapM (go env) tys >>= (return <.> tSum)
        | TTyCon1(c1,ty) -> go env ty       >>= (function ty' -> return (TTyCon1(c1,ty')))
        | TFunc(ty1,ty2) -> (go env ty1     >>= function ty1' ->
                             go env ty2     >>= function ty2' ->
                             return (TFunc(ty1',ty2')))
        | TVar(tn)       -> (isMetaTC tn >>= function ismeta ->
                             if ismeta then return ty0 else
                             (* NB: dom itnmap is disjoint from the set of metatyvars,
                                so this order is OK. *)
                             lookupITn tn >>= function tn' ->
                             itypnamemap env tn'        >>= transopt env (TVar(tn')))
        | TTyName(n)     -> return ty0
        | TXDot(Modname(mn),etn) -> dotmap      env eqs (mn,etn) >>= transopt env ty0
        | TXDot(Hash(h),etn)     -> abstract_in "Tdireq" nenv etn h >>-
                                    hashdotmap  env eqs (h,etn) >>= transopt env ty0
        | TForall(tn,ty) -> withFreshITn tn (function tn' ->
                            tc_EtKok env tn' KType >>-
                            let env' = (nenv,addtypeenv (Env_typ(tn',KType)) tenv) in
                            go env' ty >>= function ty' ->
                            return (TForall(tn',ty')))
        | TExists(tn,ty) -> withFreshITn tn (function tn' ->
                            tc_EtKok env tn' KType >>-
                            let env' = (nenv,addtypeenv (Env_typ(tn',KType)) tenv) in
                            go env' ty >>= function ty' ->
                            return (TExists(tn',ty')))
    and transopt env ty0 tyopt
      = match tyopt with
          None      -> return ty0
        | Some(ty') -> go env ty'
    in
    go env ty


let tc_unify_typ : nametypeenv -> eqs -> typ -> typ -> unit tcm
(* attempt to unify two types at a given colour in a given environment.
   NB: Assumes Eok and eqsok.
   Types assigned to metavars are always as concrete as possible. *)
  = fun env eqs ty1 ty2
 -> inCtxt [TCCRule "unify_typ"; TCCEnv env; TCCEqs eqs; TCCTyp ty1; TCCTyp ty2] <$>
    begin
    tc_Tdireq env eqs ty1 >>= function ty1' ->
    tc_Tdireq env eqs ty2 >>= function ty2' ->
    unify_typ env eqs ty1' ty2'
    end

let tc_explode_forall : nametypeenv -> eqs -> internal_typname -> typ -> typ tcm
(* attempt to extract the body of typ, assuming it is a forall.
   Substitutes the given tyvar for the binder.
   NB: Assumes Eok and eqsok. *)
  = fun env eqs itn ty
 -> inCtxt [TCCRule "explode_forall"; TCCEnv env; TCCEqs eqs; TCCTyp ty] <$>
    begin
    tc_Tdireq env eqs ty >>= function ty' ->
    explode_forall eqs itn ty'
    end


let tc_explode_exists : nametypeenv -> eqs -> internal_typname -> typ -> typ tcm
(* attempt to extract the body of typ, assuming it is an exists.
   Substitutes the given tyvar for the binder.
   NB: Assumes Eok and eqsok. *)
  = fun env eqs itn ty
 -> inCtxt [TCCRule "explode_exists"; TCCEnv env; TCCEqs eqs; TCCTyp ty] <$>
    begin
    tc_Tdireq env eqs ty >>= function ty' ->
    explode_exists eqs itn ty'
    end


let tc_Teq0 : nametypeenv -> eqs -> typ -> typ -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = fun env eqs ty1 ty2
 -> chase eqs ty1 >>= function ty1' ->  (* canonicalise *)  (* XXX temp *)
    chase eqs ty2 >>= function ty2' ->  (* canonicalise *)  (* XXX temp *)
    inCtxt [TCCRule "Teq0"; TCCEnv env; TCCEqs eqs; TCCTyp ty1'; TCCTyp ty2'] <$>
    begin
    inferringTC >>= function inferring ->
    if inferring then
      tc_unify_typ env eqs ty1 ty2
    else
      tc_Tdireq env eqs ty1 >>= function ty1' ->  (* canonicalise *)
      tc_Tdireq env eqs ty2 >>= function ty2' ->  (* canonicalise *)
      unless (typ_eq ty1' ty2')  (* now we can use syntactic equality *)
        (tcfail_ps (fun ps ->
                 "Teq0","Types not equal",
                 "Types "^print_typ ps ty1^"\n and "^print_typ ps ty2
                 ^"\nare canonically "^print_typ ps ty1'^"\n and "^print_typ ps ty2')) >>-
      return ()
    end


let tc_Teq : nametypeenv -> eqs -> typ -> typ -> unit tcm
  (* same as tc_Teq0 above, but does Eok and eqsok for you *)
  = fun env eqs ty1 ty2
 -> inCtxt [TCCRule "Teq"; TCCEnv env; TCCEqs eqs; TCCTyp ty1; TCCTyp ty2] <$>
    begin
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_Teq0 env eqs ty1 ty2
    end


let tc_Teqp : nametypeenv -> eqs -> typ -> typ -> bool tcm
  (* same as tc_Teq above, but returns bool *)
  = fun env eqs ty1 ty2
 -> inCtxt [TCCRule "Teqp"; TCCEnv env; TCCEqs eqs; TCCTyp ty1; TCCTyp ty2] <$>
    begin
    inferringTC >>= function inferring ->
    unless (not inferring) (tcfail ("tc_Teqp","Invoked during inference","")) >>-
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_Tdireq env eqs ty1 >>= function ty1' ->  (* canonicalise *)
    tc_Tdireq env eqs ty2 >>= function ty2' ->  (* canonicalise *)
    return (typ_eq ty1' ty2')  (* now we can use syntactic equality *)
    end


let tc_Keq0 : nametypeenv -> eqs -> kind -> kind -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = fun env eqs k1 k2
 -> inCtxt [TCCRule "Keq0"; TCCEnv env; TCCEqs eqs; TCCKind k1; TCCKind k2] <$>
    match (k1,k2) with
      (KType   ,KType   ) -> return ()
    | (KEq(ty1),KEq(ty2)) -> tc_Teq0 env eqs ty1 ty2
    | (_       ,_       ) -> tcfail_ps (fun ps ->
                                     "Keq0","Kinds not equal",
                                     "Kinds "^print_kind ps k1^" and "^print_kind ps k2)


let tc_Keq : nametypeenv -> eqs -> kind -> kind -> unit tcm
  (* same as tc_Keq0 above, but does Eok and eqsok for you *)
  = fun env eqs k1 k2
 -> inCtxt [TCCRule "Keq"; TCCEnv env; TCCEqs eqs; TCCKind k1; TCCKind k2] <$>
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_Keq0 env eqs k1 k2

let rec tc_s_kindrel_0 : string                                        (* rule name to use *)
                      -> (nametypeenv -> eqs -> kind -> kind -> unit tcm)  (* kind relation to use *)
                      -> nametypeenv -> eqs -> signature -> signature -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = fun rulename kindrel (nenv,tenv) eqs sign1 sign2
 -> inCtxt [TCCEnv (nenv,tenv); TCCEqs eqs; TCCSignature sign1; TCCSignature sign2; TCCRule rulename] <$>
    match (sign1.desc,sign2.desc) with
      ([]                   ,[]                   )
             -> return ()
    |  ([],_) | (_, [])
	  -> tcfail_ps (fun ps ->
            rulename,"Type and value fields misaligned in signatures",
            "In signatures "^print_signature ps sign1^" and "^print_signature ps sign2)
    | (s1::sign1', s2::sign2') -> (match (s1.desc, s2.desc) with

    | (SVal(id1,ty1),SVal(id2,ty2))
        -> (tc_Teq0 (nenv,tenv) eqs ty1 ty2 >>-
            unless (ident_ext id1 = ident_ext id2)
              (tcfail_ps (fun ps ->
                       rulename,"External identifiers in signatures don't match",
                       "Identifiers "^print_ident_bind ps id1^" and "^print_ident_bind ps id2
                       ^" in signatures "^print_signature ps sign1^" and "^print_signature ps sign2)) >>-
            unless (not (List.mem (ident_ext id1) (list_map "tc_s_kindrel_0" ident_ext (sign_valfieldnames {sign1 with desc=sign1'}))))
              (tcfail_ps (fun ps ->
                       rulename,"Value field name duplicated in signature",
                       "Field "^print_ident_bind ps id1^" in signature tail "^print_signature ps sign1)) >>-
            let iid = fresh in (* xyzzy *)
            let (id1,sign1') = swap (fst (ident_int id1)) and iid in (id1,sign1') in
            let (id2,sign2') = swap (fst (ident_int id2)) and iid in (id2,sign2') in
            tc_ExTok (nenv,tenv) (ident_int id1) ty1 >>-
            tc_s_kindrel_0 rulename kindrel (nenv,addtypeenv (Env_val (ident_int id1,ty1)) tenv) eqs {sign1 with desc=sign1'} {sign1 with desc=sign2'})
    | (STyp(tn1,k1),STyp(tn2,k2))
        -> (kindrel (nenv,tenv) eqs k1 k2 >>-
            unless (typname_ext tn1 = typname_ext tn2)
              (tcfail_ps (fun ps ->
                       rulename,"External type names in signatures don't match",
                       "Type names "^print_typname_bind ps tn1^" and "^print_typname_bind ps tn2
                       ^" in signatures "^print_signature ps sign1^" and "^print_signature ps sign2)) >>-
            unless (not (List.mem (typname_ext tn1) (list_map "tc_s_kindrel_0[2]" typname_ext (sign_typfieldnames {sign1 with desc=sign1'}))))
              (tcfail_ps (fun ps ->
                       rulename,"Type field name duplicated in signature",
                       "Field "^print_typname_bind ps tn1^" in signature tail "^print_signature ps sign1)) >>-
            let itn = fresh in (* xyzzy *)
            let (tn1,sign1') = swap (fst (typname_int tn1)) and itn in (tn1,sign1') in
            let (tn1,sign2') = swap (fst (typname_int tn2)) and itn in (tn2,sign2') in
            tc_EtKok (nenv,tenv) (typname_int tn1) k1 >>-
            tc_s_kindrel_0 rulename kindrel (nenv,addtypeenv (Env_typ (typname_int tn1,k1)) tenv) eqs {sign1 with desc=sign1'} {sign1 with desc=sign2'})
    | (_                   ,_                   )
        -> tcfail_ps (fun ps ->
                   rulename,"Type and value fields misaligned in signatures",
                   "In signatures "^print_signature ps sign1^" and "^print_signature ps sign2))

let tc_seq0 : nametypeenv -> eqs -> signature -> signature -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = tc_s_kindrel_0 "seq0" tc_Keq0

let tc_seq : nametypeenv -> eqs -> signature -> signature -> unit tcm
  (* NB: same as tc_seq0 but checked Eok and eqsok as well *)
  = fun env eqs sign1 sign2
 -> inCtxt [TCCRule "seq"; TCCEnv env; TCCEqs eqs; TCCSignature sign1; TCCSignature sign2; TCCLoc sign1.loc; TCCLoc sign2.loc] <$>
    begin
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_seq0 env eqs sign1 sign2
    end


let tc_Ksub0 : nametypeenv -> eqs -> kind -> kind -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = fun env eqs k1 k2
 -> inCtxt [TCCRule "Ksub0"; TCCEnv env; TCCEqs eqs; TCCKind k1; TCCKind k2] <$>
    match (k1,k2) with
      (KEq(ty1),KType) -> tc_TK0 env ty1
    | (_       ,_    ) -> tc_Keq0 env eqs k1 k2


let tc_Ksub : nametypeenv -> eqs -> kind -> kind -> unit tcm
  (* NB: same as tc_Ksub0 but checks Eok and eqsok as well *)
  = fun env eqs k1 k2
 -> inCtxt [TCCRule "Ksub"; TCCEnv env; TCCEqs eqs; TCCKind k1; TCCKind k2] <$>
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_Ksub0 env eqs k1 k2


let tc_ssub0 : nametypeenv -> eqs -> signature -> signature -> unit tcm
  (* NB: caller guarantees tc_Eok env && tc_eqsok eqs *)
  = tc_s_kindrel_0 "ssub0" tc_Ksub0


let tc_ssub : nametypeenv -> eqs -> signature -> signature -> unit tcm
  (* NB: same as tc_ssub0 but checked Eok and eqsok as well *)
  = fun env eqs sign1 sign2
 -> inCtxt [TCCRule "ssub"; TCCEnv env; TCCEqs eqs; TCCSignature sign1; TCCSignature sign2; TCCLoc sign1.loc; TCCLoc sign2.loc] <$>
    begin
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_ssub0 env eqs sign1 sign2
    end


let tc_eqMS : nametypeenv -> modname option -> signature -> unit tcm
  = fun (nenv,tenv) mno sign'
 -> inCtxt [TCCRule "eqMS"; TCCEnv (nenv,tenv); TCCEqModname mno; TCCSignature sign'; TCCLoc sign'.loc] <$>
    begin
    tc_Eok (nenv,tenv) >>-
    match mno with
      None     -> tc_sok (nenv,tenv) sign'
    | Some(mn) -> match lookup_typeenv_modname tenv mn with
                    None       -> tcfail ("eqMS","Module name not in environment","")
                  | Some(sign) -> tc_ssub0 (nenv,tenv) [] sign sign'
    end


let tc_XsxT : nametypeenv
          -> eqs
          -> hash_or_modname  (* X: either h or MM_M *)
          -> signature
          -> external_ident
          -> typ tcm              (* *canonical* type for X.x *)
  (* returns the canonical type for X.x; this is selfified and has concrete types substituted in *)
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  (* Caller also guarantees that the hash or modname, in the given environment,
     has the signature that is passed as the initial sign1. *)
  = fun (nenv,tenv) eqs x sign1 eid
 -> inCtxt [TCCRule "XsxT"; TCCEnv (nenv,tenv); TCCEqs eqs; TCCX x;
            TCCSignature sign1; TCCEIdent eid; TCCLoc sign1.loc] <$>
    let rec go tenv sign1 =
    match sign1 with
       | []
        -> tcfail ("XsxT","Field not found in signature","")
       | (s::sign1') -> (match s.desc with
   (SVal(id1,ty1))
        -> if ident_ext id1 = eid then
             tc_Tdireq (nenv,tenv) eqs ty1  (* safe even in inference, because we are
                                               really just doing selfification and substitution,
                                               not testing equality *)
           else
             go (addtypeenv (Env_val(ident_int id1,ty1)) tenv) sign1'
    | (STyp(tn1,k1))
        -> let self = txdot x (typname_ext tn1) in
           go (addtypeenv (Env_typ(typname_int tn1,KEq(self))) tenv) sign1')
       in
    go tenv sign1.desc


(* like tc_XS, but relating to one, type, signature_item only *)
let tc_XS1 : nametypeenv -> eqs -> hash_or_modname -> (external_typname * kind) -> unit tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun env eqs x (etn,k2)
 -> inCtxt [TCCRule "XS1"; TCCEnv env; TCCEqs eqs; TCCX x; TCCETypname etn; TCCKind k2] <$>
    begin
    flat_xetn env (x,etn) >>= function koo ->
    match koo with
      None           -> tcfail ("XS1", "Module name not in environment","")
    | Some None      -> tcfail ("XS1", "Field not found in signature","")
    | Some (Some k1) ->
        match k2 with
          KType    -> return ()
        | KEq(ty2) -> tc_Teq0 env eqs (txdot x etn) ty2
              (* NB: we don't care about k1, because self has kind k1 in env,
                 and Teq0 will use that in its determination anyway. *)
    end


(* the full TK, but without checking env and eqs *)
let tc_TK00 : nametypeenv -> eqs -> typ -> kind -> unit tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun ((_,tenv) as env) eqs ty k
 -> inCtxt [TCCRule "TK00"; TCCEnv env; TCCEqs eqs; TCCTyp ty; TCCKind k] <$>
    begin
    match ty with
    | TTyCon0 _
    | TTyCon1 _
    | TTup(_)
    | TSum(_)
    | TFunc(_,_)
    | TForall(_,_)
    | TExists(_,_)
    | TTyName(_)
      -> (tc_TK0 env ty >>-  (* all these things have kind Type *)
          match k with
            KType    -> return ()
          | KEq(ty') -> tc_Teq0 env eqs ty ty')

    | TVar(tn)
      -> (lookupITn tn >>= function tn' ->
          isMetaTC tn' >>= function ismeta ->
          let mk = if ismeta then Some(KType) else lookup_typeenv_itypname tenv tn' in
          match mk with
            Some(_) -> (match k with
                           KType    -> return ()
                         | KEq(ty') -> tc_Teq0 env eqs ty ty')
                         (* NB: we don't care what kind tn has in the environment, as
                            Teq0 will use that in its determination anyway. *)
          | None -> tcfail ("TK","Type name not in scope",""))

    | TXDot(x,etn)
      -> tc_XS1 env eqs x (etn,k)
    end

let tc_TK : nametypeenv -> eqs -> typ -> kind -> unit tcm
  = fun env eqs ty k
 -> inCtxt [TCCRule "TK"] <$>
    begin
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_TK00 env eqs ty k
    end


let tc_Esub : nametypeenv -> typeenv -> typeenv -> unit tcm
  = fun (nenv,tenv) env1 env2
 -> inCtxt [TCCNEnv nenv; TCCBox (TCCTEnv env1); TCCBox (TCCTEnv env2); TCCHide (TCCTEnv env1); TCCRule "Esub"] <$>
    let rec go env env1 env2 =
      match (env1,env2) with
        (Env_val(iid1,ty1)::env1,
         Env_val(iid2,ty2)::env2) ->
           tc_Teq0 (nenv,env) [] ty1 ty2 >>-
           go (addtypeenv (Env_val(iid1,ty1)) env) env1 (swap_prettyname iid1 iid2 env2)
      | (Env_loc(l1,ty1)::env1,
         Env_loc(l2,ty2)::env2) ->
           unless (l_eq l1 l2)
             (tcfail ("Esub","Locations in environments do not match","")) >>-
           tc_Teq0 (nenv,env) [] ty1 ty2 >>-
           go (addtypeenv (Env_loc(l1,ty1)) env) env1 env2
      | (Env_mod((emn1,imn1),sign1)::env1,
         Env_mod((emn2,imn2),sign2)::env2) ->
           unless (mn_ext_eq emn1 emn2)
             (tcfail ("Esub","External module names in environments do not match","")) >>-
           tc_ssub0 (nenv,env) [] sign1 sign2 >>-
           go (addtypeenv (Env_mod((emn1,imn1),sign1)) env) env1 (swap_prettyname imn1 imn2 env2)
      | (Env_typ(itn1,k1)::env1,
         Env_typ(itn2,k2)::env2) ->
           tc_Ksub0 (nenv,env) [] k1 k2 >>-
           go (addtypeenv (Env_typ(itn1,k1)) env) env1 (swap_prettyname itn1 itn2 env2)
      | ([],[]) -> return ()
      | _ -> tcfail ("Esub","Environment lengths do not match","")
    in
    tc_Eok (nenv,tenv) >>-
    tc_Etok0 env1 (nenv,tenv) >>= function _ ->
    go tenv (List.rev (typeenv_list_of_typeenv env1)) (List.rev (typeenv_list_of_typeenv env2))


(* this judgement is used in store, but is defined later.  Keith
   guarantees that it is used safely, and loops will not result.
   However, we'd like to know that the topological sort is correct
   with these edges removed, so we resort to this trickery. *)
let rec_tc_eT = ref (fun _ -> raise (Never_happen "rec_tc_eT not filled in!"))

let tc_store : nametypeenv -> typeenv -> store -> unit tcm
  = fun (nenv,tenv) senv s
 -> inCtxt [TCCRule "store"; TCCEnv (nenv,tenv); TCCTEnv senv; TCCStore s] <$>
    begin
    getLangTC >>= function lang ->
    unless (lang = CompiledForm)
            (tcfail ("store","Language should be compiled [INTERNAL ERROR]","")) >>-
    let senv_list = typeenv_list_of_typeenv senv in
    unless (List.for_all (function tee ->
                            match tee with
                              Env_val(_,_) -> false
                            | Env_loc(_,_) -> true
                            | Env_mod(_,_) -> false
                            | Env_typ(_,_) -> false) senv_list)
      (tcfail ("store","Store environment Es contains some non-loc bindings","")) >>-
    let envlocs    = partial_map (function tee ->
                                    match tee with
                                        Env_val(_,_) -> None
                                      | Env_loc(l,_) -> Some(l)
                                      | Env_mod(_,_) -> None
                                      | Env_typ(_,_) -> None) senv_list in
    let storelocs  = list_map "tc_store" fst s in
    unless (List.length envlocs = List.length storelocs &&  (* no duplicates in storelocs *)
            subseteq l_eq envlocs storelocs &&
            subseteq l_eq storelocs envlocs)
      (tcfail ("store","Store environment domain and store domain do not agree","")) >>-
    let env_all = (nenv,appendtypeenv senv tenv) in
    tc_Eok env_all >>-
    mapMz (function tee ->
             match tee with
               Env_loc(l,ty) -> (match lookup_store_loc s l with
                                   None -> tcfail_ps (fun ps ->
                                                      "store","Environment mentions location not in store",
                                                      "Location "^print_location ps l)
                                 | Some(e)
                                     -> !rec_tc_eT env_all [] (primtoexpr e) >>= function ty' ->
                                        tc_Teq0 env_all [] ty (TTyCon1(TRef,ty')))
             | _ -> return ()) senv_list
    end


let rec go_eT : nametypeenv -> eqs -> expr -> typ tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun ((nenv,tenv) as env) eqs e
 -> inCtxt [TCCRule "eT:e"; TCCHide (TCCEnv env); TCCExpr e; TCCLoc e.loc] <$>
    begin
    progressStep 64 >>-
    match e.desc with
      LocC0(c0)          -> (let ty = typeof_C0 c0 in
                          tc_TK0 env ty >>-
                          return ty)
    | LocC1(SOME,e)      -> (go_eT env eqs e >>= function ty1 ->
                          return (TTyCon1(TOption,ty1)))
    | LocC1(TieCon,e)    -> (go_eT env eqs e >>= function ty1 ->
                          return (TTyCon1(TTie,ty1)))
    | LocC1(Node,e1)     -> ( match e1.desc with
                              | LocTup(e::l::[]) ->
                                  (go_eT env eqs e >>= function ty1 ->
                                   go_eT env eqs l >>= function ty2 ->
                                   tc_Teq0 env eqs ty2 (TTyCon1(TList,(TTyCon1(TTree,ty1)))) >>-
                                   return (TTyCon1(TTree,ty1)))
                              | _ -> tcfail ("eT", "Node constructor applied to the wrong arguments", "") )
    | LocC1(Inj(i,TSum(tys)),e) when 1 <= i && i <= List.length tys
                      -> (let (ty1,ty2) = (TSum(tys), List.nth tys (i-1)) in
                          go_eT env eqs e >>= function ty ->
                          tc_Teq0 env eqs ty2 ty >>-
                          tc_TK0 env ty1 >>-
                          return ty1)
    | LocC1(Inj _, _)    -> tcfail ("eT", "Inj with non-sum type annotation or index out of range", "")
    | LocC1(c1,e)        -> (let (ty1,ty2) = typeof_C1 c1 in
                          go_eT env eqs e >>= function ty ->
                          tc_Teq0 env eqs ty1 ty >>-
                          tc_TK0 env ty2 >>-
                          return ty2)
    | LocCons(e1,e2)     -> (go_eT env eqs e1 >>= function ty1 ->
                          go_eT env eqs e2 >>= function ty2 ->
                          tc_Teq0 env eqs (TTyCon1(TList,ty1)) ty2 >>-
                          return ty2)
    | LocTup(es)         -> (unless (List.length es >= 2)
                            (tcfail ("eT","Tuple must have at least 2 elements","")) >>-
                          mapM (go_eT env eqs) es >>= function tys ->
                          return (TTup(tys)))
    | LocOp(oe,es)    -> (let (etys,ty0) = info_of_op_or_econst oe in
                          let arity = List.length etys in
                          let nargs = List.length es in
                          getLangTC >>= function lang ->
                          (if lang <> SugaredSourceInternalForm then
                            unless (nargs = arity)
                              (tcfail ("eT","Op must be saturated in core syntax",
                                       "Expected " ^ string_of_int arity ^ " arguments"
                                       ^ " but received " ^ string_of_int nargs))
                          else  (* source *)
                            unless (nargs = 0)
                              (tcfail ("eT","Op must have no arguments in source syntax "^
                                       "[INTERNAL PARSER ERROR]: " ^ string_of_int nargs ^ " is not 0!",""))) >>-
                          unless (nargs <= arity)  (* defensively double-check *)
                              (tcfail ("eT","Op over-saturated",
                                       "Expected " ^ string_of_int arity ^ " arguments"
                                       ^ " but received " ^ string_of_int nargs)) >>-
                          mapM (go_eT env eqs) es >>= function tys ->
                          let (etys1,etys2) = splitAt nargs etys in
                          zipMz (tc_Teq0 env eqs) etys1 tys >>-
                          let ty = List.fold_right tFunc etys2 ty0 in
                          tc_TK0 env ty >>-
                          return ty)

           (* FZ LocLazyOp is tailored for LoAnd and LoOr *)
    | LocLazyOp(lo,es)   -> (let (etys,ty0) = ([TTyCon0 TBool;TTyCon0 TBool],TTyCon0 TBool) in
                          let nargs = List.length es in
                          getLangTC >>= function lang ->
                          (if lang <> SugaredSourceInternalForm then
                            unless (nargs = 2)
                              (tcfail ("eT","LazyOp must be saturated in core syntax",
                                       "Expected 2 arguments"
                                       ^ " but received " ^ string_of_int nargs))
                          else  (* source *)
                            unless (nargs = 0)
                              (tcfail ("eT","LazyOp must have no arguments in source syntax "^
                                       "[INTERNAL PARSER ERROR]",""))) >>-
                          unless (nargs <= 2)  (* defensively double-check *)
                              (tcfail ("eT","LazyOp over-saturated",
                                       "Expected 2 arguments"
                                       ^ " but received " ^ string_of_int nargs)) >>-
                          mapM (go_eT env eqs) es >>= function tys ->
                          let (etys1,etys2) = splitAt nargs etys in
                          zipMz (tc_Teq0 env eqs) etys1 tys >>-
                          let ty = List.fold_right tFunc etys2 ty0 in
                          tc_TK0 env ty >>-
                          return ty)
    | LocLoc(loc)        -> (match lookup_typeenv_location tenv loc with
                            None     -> tcfail ("eT","Location not in scope","")
                          | Some(ty) -> return ty)
    | LocFn(mtch)        -> (getLangTC >>= function lang ->
                          let isprimitive = match mtch with
                                              [(PVar(_,_),_)] -> true
                                            | _               -> false
                          in
                          unless (isprimitive || lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          go_mTT env eqs mtch >>= function (ty1,ty2) ->
                          return (TFunc(ty1,ty2)))
    | LocFun(ps,e')      -> (let vs = binding_vars_of_pats ps in
                          withFreshIIds vs (function vs ->
                          getLangTC >>= function lang ->
                          unless (lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          mapM (go_pTE env eqs) ps >>= function ty0envs ->
                          let (ty0s,tenvs) = List.split ty0envs in
                          tc_Etok0 (concattypeenv (List.rev tenvs)) env >>= function env' ->
                          go_eT env' eqs e' >>= function ty' ->
                          let ty0 = List.fold_right tFunc ty0s ty' in
                          return ty0))
    | LocTAbs(itn,e1) -> (freshskolemTC eqs itn >>= function itn1 ->  (* occs in term will be substituted back to itn *)
                          let tenv' = mktypeenv [Env_typ (itn1, KType)] in
                          tc_Etok0 tenv' env >>= function env' ->
                          let e1' = swap_prettyname itn itn1 e1 in  (* XXX inefficient *)
                          go_eT env' eqs e1' >>= function ty ->
                          get_one_partial_substTC ty >>= function ty' ->
                          tctrace_ps Opts.DBC_tcquant (fun ps -> "eT:TAbs","foo",print_typ ps (TForall(itn1,ty'))) >>-
                          return (TForall(itn1,ty')))
    | LocInEnv _         -> raise (Never_happen "eT:e InEnv")
    | LocClos _          -> raise (Never_happen "eT:e Clos")
    | LocTClos _         -> raise (Never_happen "eT:e TClos")
    | LocId(iid)         -> (lookupIId iid >>= function iid' ->
                          match lookup_typeenv_iident tenv iid' with
                            None     -> tcfail ("eT","Identifier not in scope","")
                          | Some(ty) -> return ty)
    | LocDot(mn,eid)     -> (tc_MS0 env mn >>= function sign ->
                          tc_XsxT env eqs (Modname(mn)) sign eid >>= function ty ->
                          tc_TK0 env ty >>-  (* sanity check *)
                          return ty)
    | LocHashDot(h,eid)  -> (tc_hS0 env h >>= function sign ->
                          tc_XsxT env eqs (Hash(h)) sign eid >>= function ty ->
                          tc_TK0 env ty >>-  (* sanity check *)
                          return ty)
    | LocIf(e1,e2,e3)    -> (go_eT env eqs e1 >>= function ty1 ->
                          go_eT env eqs e2 >>= function ty2 ->
                          go_eT env eqs e3 >>= function ty3 ->
                          tc_Teq0 env eqs ty1 (TTyCon0 TBool) >>-
                          tc_Teq0 env eqs ty2 ty3 >>-
                          return ty2)
    | LocWhile(e1,e2)    -> (go_eT env eqs e1 >>= function ty1 ->
                          go_eT env eqs e2 >>= function ty2 ->
                          tc_Teq0 env eqs ty1 (TTyCon0 TBool) >>-
                          tc_Teq0 env eqs ty2 (TTyCon0 TUnit) >>-
                          return (TTyCon0 TUnit))
    | LocSeq(e1,e2)      -> (go_eT env eqs e1 >>= function ty1 ->
                          go_eT env eqs e2 >>= function ty2 ->
                          tc_Teq0 env eqs ty1 (TTyCon0 TUnit) >>-
                          return ty2)
    | LocApp(e1,e2)      -> (go_eT env eqs e1 >>= function ty1 ->
                          go_eT env eqs e2 >>= function ty2 ->
                          inferringTC >>= function inferring ->
                          (if inferring then
                             freshtypeTC eqs >>= function ty1a ->
                             freshtypeTC eqs >>= function ty1b ->
                             tc_unify_typ env eqs ty1 (TFunc(ty1a,ty1b)) >>-
                             return (ty1a,ty1b)
                           else
                             tc_Tdireq env eqs ty1 >>= function ty1' ->
                             match ty1' with
                               TFunc(ty1'a,ty1'b)
                                 -> return (ty1'a,ty1'b)
                             | _ -> tcfail_ps (fun ps ->
                                            "eT","Application of non-function",
                                            "\"Function\" has canonical type "^print_typ ps ty1')
                          ) >>= function (ty1a',ty1b') ->
                          tc_Teq0 env eqs ty1a' ty2 >>-
                          return ty1b')
    | LocTApp(e1,ty2) -> (go_eT env eqs e1 >>= function ty0 ->
                          freshtyvarTC eqs >>= function itn0 ->
                          tc_explode_forall env eqs itn0 ty0 >>= function ty0b ->
                          tctrace_ps Opts.DBC_tcquant (fun ps -> "eT:TApp","foo",print_typ ps ty0 ^"\n"^print_typname_use ps itn0 ^ "\n" ^ print_typ ps ty0b) >>-
                          tc_Tdireq env eqs ty2 >>= function ty2' ->  (* rewrite any bound tynames *)
                          tc_unify_typ env eqs (TVar itn0) ty2' >>-
                          return ty0b)
    | LocPack(ty1,e1,ty2) -> (tc_Tdireq env eqs ty2 >>= function ty2' ->  (* rewrite any bound tynames *)
                              tc_TK0 env ty2' >>-
                              go_eT env eqs e1 >>= function ty0 ->
                              freshtyvarTC eqs >>= function itn2 ->
                              tc_explode_exists env eqs itn2 ty2' >>= function ty2b ->
                              tc_Tdireq env eqs ty1 >>= function ty1' ->  (* rewrite any bound tynames *)
                              tc_unify_typ env eqs (TVar itn2) ty1' >>-
                              tc_Teq0 env eqs ty0 ty2b >>-
                              return ty2')
    | LocUnpack(itn,iid,e1,e2) -> (go_eT env eqs e1 >>= function ty1 ->
                              freshskolemTC eqs itn >>= function itn1 ->  (* occs in term will be substituted back to itn *)
                              tc_explode_exists env eqs itn1 ty1 >>= function ty ->
                              withFreshIId iid (function iid ->
                                let tenv' = mktypeenv [Env_val (iid, ty); Env_typ (itn1, KType)] in
                                tc_Etok0 tenv' env >>= function env' ->
                                let e2' = swap_prettyname itn itn1 e2 in  (* XXX inefficient *)
                                go_eT env' eqs e2'  >>= function ty2 ->
                                tc_TK0 env ty2 >>-  (* check itn1 doesn't escape! *)
                                return ty2))
    | LocNamecase(e1,itn,iid1,iid2,e0,e2,e3)
                      -> (go_eT env eqs e1 >>= function ty1 ->
                          freshskolemTC eqs itn >>= function itn1 ->  (* occs in term will be substituted back to itn *)
                          tc_explode_exists env eqs itn1 ty1 >>= function ty1e ->
                          freshtypeTC eqs >>= function ty1a ->
                          freshtypeTC eqs >>= function ty1b ->
                          tc_unify_typ env eqs ty1e (TTup [ty1a; ty1b]) >>-
                          tc_unify_typ env eqs ty1a (TTyCon1(TName,TVar itn1)) >>-
                          go_eT env eqs e0 >>= function ty0 ->
                          freshtypeTC eqs >>= function ty0n ->
                          tc_unify_typ env eqs ty0 (TTyCon1(TName, ty0n)) >>-
                          withFreshIId iid1 (function iid1 ->
                          withFreshIId iid2 (function iid2 ->
                          let tenv' = mktypeenv [Env_val (iid2, ty1b); Env_val (iid1, ty0);
                                                 Env_typ (itn1, KEq(ty0n))] in
                          tc_Etok0 tenv' env >>= function env' ->
                          let e2' = swap_prettyname itn itn1 e2 in  (* XXX inefficient *)
                          go_eT env' eqs e2')) >>= function ty2 ->
                          tc_TK0 env ty2 >>-  (* check itn1 doesn't escape *)
                          go_eT env eqs e3 >>= function ty3 ->
                          tc_Teq0 env eqs ty2 ty3 >>-
                          return ty3)
    | LocMatch(e,m)   -> (go_eT env eqs e >>= function ty ->
                          go_mTT env eqs m >>= function (ty1,ty2) ->
                          tc_Teq0 env eqs ty ty1 >>-
                          return ty2)
    | LocLet(e1,(p,e2)) -> (
                          let vs = binding_vars_of_pat p in
                          withFreshIIds vs (function vs' ->
                          getLangTC >>= function lang ->
                          unless (lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          go_pTE env eqs p >>= function (ty0,tenv') ->
                          go_eT env eqs e1 >>= function ty1 ->
                          tc_Teq0 env eqs ty0 ty1 >>-
                          go_eT (nenv,appendtypeenv tenv' tenv) eqs e2))  (* @ well-formed by freshness *)
    | LocLetMulti (ty,(ps,e'),(iid,e'')) -> (
                          withFreshIId iid (function iid ->
                          let vs = binding_vars_of_pats ps in
                          withFreshIIds vs (function vs ->
                          getLangTC >>= function lang ->
                          unless (lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          mapM (go_pTE env eqs) ps >>= function ty0envs ->
                          let (ty0s,tenvs) = List.split ty0envs in
                          tc_Etok0 (concattypeenv (List.rev tenvs)) env >>= function env' ->
                          go_eT env' eqs e' >>= function ty' ->
                          let ty0 = List.fold_right tFunc ty0s ty' in
                          tc_Teq0 env eqs ty ty0 >>-
                          tc_ExTok env iid ty >>-
                          go_eT (nenv,addtypeenv (Env_val(iid,ty)) tenv) eqs e'')))
    | LocLetrec(ty,(iid,(mtch,e))) ->
                          (withFreshIId iid (function iid ->
                          getLangTC >>= function lang ->
                          let isprimitive = match mtch with
                                              [(PVar(_,_),_)] -> true
                                            | _               -> false
                          in
                          unless (isprimitive || lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          tc_ExTok env iid ty >>-  (* eqs ok because monotonic in env *)
                          let env' = (nenv,addtypeenv (Env_val(iid,ty)) tenv) in
                          go_mTT env' eqs mtch >>= function (tya,tyb) ->
                          let ty' = TFunc(tya,tyb) in
                          tc_Teq0 env eqs ty ty' >>-
                          go_eT env' eqs e))
    | LocLetrecMulti(ty,(ps,(iid,(e',e'')))) -> (
                          withFreshIId iid (function iid ->
                          let vs = binding_vars_of_pats ps in
                          withFreshIIds vs (function vs ->
                          getLangTC >>= function lang ->
                          unless (lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          mapM (go_pTE env eqs) ps >>= function ty0envs ->
                          let (ty0s,envs) = List.split ty0envs in
                          tc_ExTok env iid ty >>-
                          let env' = (nenv, addtypeenv (Env_val(iid,ty)) tenv) in
                          tc_Etok0 (concattypeenv (List.rev envs)) env' >>= function env'' ->
                          go_eT env'' eqs e' >>= function ty' ->
                          let ty0 = List.fold_right tFunc ty0s ty' in
                          tc_Teq0 env eqs ty ty0 >>-
                          go_eT env' eqs e'')))
    | LocRaise(e)        -> (go_eT env eqs e >>= function ty' ->
                          tc_Teq0 env eqs ty' (TTyCon0 TExn) >>-
                          inferringTC >>= function inferring ->
                          (if inferring then
                             freshtypeTC eqs
                          else
                            tcfail ("eT","Can't guess type of raise when typechecking; insufficient type annotation","")) >>= function ty ->
                          return ty)
    | LocTry(e,mtch)     -> (go_eT env eqs e >>= function ty0 ->
                          go_mTT env eqs mtch >>= function (tya,tyb) ->
                          tc_Teq0 env eqs tya (TTyCon0 TExn) >>-
                          tc_Teq0 env eqs tyb ty0 >>-
                          return ty0)
    | LocMarshal(e1,e2,ty)-> (go_eT env eqs e1 >>= function ty1 ->
                          tc_Teq0 env eqs ty1 (TTyCon0 TString) >>-
                          go_eT env eqs e2 >>= function ty2 ->
                          tc_Teq0 env eqs ty2 ty >>-
                          return (TTyCon0 TString))
    | LocMarshalz(mk,e2,ty)-> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","marshalz not allowed in user syntax","")) >>-
                          go_eT env [] e2 >>= function ty2 ->
                          tc_Teq0 env eqs ty2 ty >>-
                          return (TTyCon0 TString))
    | LocUnmarshal(e,ty) -> (tc_TK0 env ty >>-
                          go_eT env eqs e >>= function ty0 ->
                          tc_Teq0 env eqs ty0 (TTyCon0 TString) >>-
                          return ty)
    | LocRET(ty)         -> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","RET not allowed in user syntax","")) >>-
                          tc_TK0 env ty >>-
                          return ty)
    | LocSLOWRET(ty)     -> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","SLOWRET not allowed in user syntax","")) >>-
                          tc_TK0 env ty >>-
                          return ty)
    | LocCol(e,eqs',ty)  -> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","Brackets not allowed in user syntax","")) >>-
                          tc_eqsok env eqs' >>-
                          tc_TK0 env ty >>-
                          assert_typ_is_ground "eT:Col" ty >>-
                          go_eT env eqs' e >>= function ty0 ->
                          tc_Teq0 env eqs' ty0 ty >>-
                          return ty)
    | LocOP(n,oe,es)  -> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","OP not allowed in user syntax","")) >>-
                          let ty0 = typeof_op_or_econst oe in
                          tc_TK0 env ty0 >>-
                          assert_typ_is_ground "eT:OP" ty0 >>-
                          let arity = arity_of_op_or_econst oe in
                          let nargs = List.length es in
                          unless (n = arity)
                            (tcfail_ps (fun ps ->
                                        "eT","OP node operator arity differs from annotation",
                                        "Operator arity is "^string_of_int arity
                                        ^" but OP node arity is "^string_of_int n)) >>-
                          unless (nargs = arity)
                            (tcfail_ps (fun ps ->
                                        "eT","OP node has incorrect number of arguments",
                                        "Operator and node arity is "^string_of_int arity
                                        ^" but node has "^string_of_int nargs^" arguments")) >>-
                          mapM (go_eT env eqs) es >>= function tys ->
                          (* XXX would be better if there was an op_info-like thing for all op_or_econsts *)
                          match collect_argtys arity ty0 with
                            None -> tcfail_ps (fun ps ->
                                               "eT","Op or Econst in OP node: arity and type disagree [INTERNAL ERROR]",
                                               "Arity is "^string_of_int arity^" but type is "^print_typ ps ty0)
                          | Some(etys,res_ty) ->
                          zipMz (tc_Teq0 env eqs) etys tys >>-
                          return res_ty)
    | LocResolve(e,mn,rs)
    | LocResolve_blocked(e,mn,rs)
                      -> (getLangTC >>= function lang ->
                          unless (lang = CompiledForm)
                            (tcfail ("eT","Resolve/Resolve_blocked not allowed in user syntax","")) >>-
                          unless (is_Some (lookup_typeenv_modname tenv mn))
                            (tcfail ("eT","Module name not in environment","")) >>-
                          match e.desc with
                            LocDot(mn,eid) -> (go_eT env eqs e >>= function ty ->
                                            tc_rsok rs >>-
                                            return ty)
                          | _ -> tcfail ("eT","Resolve expression must refer to a module projection",""))
    | LocPar(e1,e2)   -> (getLangTC >>= function lang ->
                          unless (lang = SugaredSourceInternalForm)
                            (tcfail ("eT","Syntactic sugar not allowed in core syntax","")) >>-
                          go_eT env eqs e1 >>= function ty1 ->
                          tc_Teq0 env eqs ty1 (TTyCon0 TUnit) >>-
                          go_eT env eqs e2 >>= function ty2 ->
                          return ty2)
    | LocFresh (ty)   -> (tc_TK0 env ty >>-
                          return (TTyCon1 (TName, ty)))
    | LocCFresh (ty)  -> (getLangTC >>= function lang ->
                          unless (lang <> CompiledForm)
                            (tcfail ("eT","cfresh not allowed in compiled syntax","")) >>-
                          unless (lang <> SourceInternalFreshForm)
                            (tcfail ("eT","cfresh not allowed in compiled-fresh syntax","")) >>-
                          tc_TK0 env ty >>-
                          return (TTyCon1 (TName, ty)))
    | LocHashMvf(Modname(mn),x,ty) -> (go_eT env eqs ({loc=e.loc; desc=LocDot(mn,x); }) >>= function ty1 ->
                                       tc_Teq0 env eqs ty ty1 >>-
                                       return (TTyCon1 (TName, ty)))
    | LocHashMvf(Hash(h),x,ty)     -> (go_eT env eqs ({loc=e.loc; desc=LocHashDot(h,x);}) >>= function ty1 ->
                                       tc_Teq0 env eqs ty ty1 >>-
                                       return (TTyCon1 (TName, ty)))
    | LocHashTs(ty1,e1,ty2)        -> (tc_TK0 env ty1 >>-
                                       tc_TK0 env ty2 >>-
                                       tc_Teq0 env eqs ty1 ty2 >>-
                                       go_eT env eqs e1 >>= function ty3 ->
                                       tc_Teq0 env eqs ty3 (TTyCon0 TString) >>-
                                       return (TTyCon1 (TName, ty2)))
    | LocHashHts(ty1,e1,e2,ty2)    -> (tc_TK0 env ty1 >>-
                                       tc_TK0 env ty2 >>-
                                       tc_Teq0 env eqs ty1 ty2 >>-
                                       go_eT env eqs e1 >>= function ty3 ->
                                       tc_Teq0 env eqs ty3 (TTyCon0 TString) >>-
                                       go_eT env eqs e2 >>= function ty4 ->
                                       freshtypeTC eqs >>= function ty0 ->
                                       tc_unify_typ env eqs ty4 (TTyCon1(TName, ty0)) >>-
                                       return (TTyCon1 (TName, ty2)))
    | LocSwap (e1,e2,e3)   -> (go_eT env eqs e1 >>= function ty1 ->
                               freshtypeTC eqs >>= function ty0 ->
                               tc_unify_typ env eqs ty1 (TTyCon1 (TName,ty0)) >>-
                               go_eT env eqs e2 >>= function ty2 ->
                               tc_Teq0 env eqs ty2 (TTyCon1 (TName,ty0)) >>-
                               go_eT env eqs e3 >>= function ty3 ->
                               return ty3)
    | LocFreshfor (e1,e2)  -> (go_eT env eqs e1 >>= function ty1 ->
                               freshtypeTC eqs >>= function ty0 ->
                               tc_unify_typ env eqs ty1 (TTyCon1 (TName,ty0)) >>-
                               go_eT env eqs e2 >>= function ty2 ->
                               return (TTyCon0(TBool)))
    | LocSupport (ty,e1)  ->   (tc_TK0 env ty >>-
                               go_eT env eqs e1 >>= function ty1 ->
                               return (TTyCon1(TList, TTyCon1(TName,ty))))
    | LocTie (mn,x) ->         (go_eT env eqs ({loc=e.loc; desc=LocDot(mn,x)}) >>= function ty ->
                                return (TTyCon1(TTie, ty)))
    | LocNameOfTie (e1) ->    (go_eT env eqs e1 >>= function ty ->
                               freshtypeTC eqs >>= function ty0 ->
                               tc_unify_typ env eqs ty (TTyCon1(TTie, ty0)) >>-
                               return (TTyCon1(TName,ty0)))
    | LocValOfTie (e2) ->     (go_eT env eqs e2 >>= function ty ->
                               freshtypeTC eqs >>= function ty0 ->
                               tc_unify_typ env eqs ty (TTyCon1(TTie, ty0)) >>-
                               return ty0)
    | LocNameValue(nv) ->     (tc_nvT0 nenv nv >>= function ty ->
                               return (TTyCon1(TName,ty)))
    end

and go_mTT : nametypeenv -> eqs -> mtch -> (typ * typ) tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun ((nenv,tenv) as env) eqs mtch
 -> inCtxt [TCCRule "eT:m"; TCCHide (TCCEnv env); TCCMtch mtch] <$>
    begin
    match mtch with
      [] -> tcfail ("mTT","Empty match not allowed","")
    | [(p,e)]      -> (let vs = binding_vars_of_pat p in
                    withFreshIIds vs (function vs' ->
                    go_pTE env eqs p >>= function (ty,tenv') ->
                    go_eT (nenv,appendtypeenv tenv' tenv) eqs e >>= function ty' ->  (* @ w-f by freshness *)
                    return (ty,ty')))  (* and eqs ok in env'@env because eqsok is monotonic *)
    | (pe::pes) -> go_mTT env eqs [pe] >>= function (ty0,ty0') ->
                   go_mTT env eqs pes  >>= function (ty1,ty1') ->
                   tc_Teq0 env eqs ty0  ty1  >>-
                   tc_Teq0 env eqs ty0' ty1' >>-
                   return (ty0,ty0')
    end


and go_pTE : nametypeenv -> eqs -> pat -> (typ * typeenv) tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs;
     this function guarantees then that env' @ env is tc_Eok;
     since tc_eqsok is monotonic in env that's still OK too *)
  = fun env eqs pat
 -> inCtxt [TCCRule "eT:p"; TCCHide (TCCEnv env); TCCPat pat] <$>
    begin
    match pat with
      PWild(ty)     -> ((* this is too strong: tc_Tdireq env eqs ty >>= function ty -> *) (* look up any bound tyvars *)
                        tc_TK0 env ty >>-
                        return (ty,emptytypeenv))
    | PVar(iid,ty)  -> (lookupIId iid >>= function iid' ->
                        (* this is too strong: tc_Tdireq env eqs ty >>= function ty -> *) (* look up any bound tyvars *)
                        tc_ExTok env iid' ty >>-
                        return (ty,addtypeenv (Env_val(iid',ty)) emptytypeenv))
    | PC0(c0)       -> (let ty = typeof_C0 c0 in
                        tc_TK0 env ty >>-
                        return (ty,emptytypeenv))
    | PC1(SOME,p)   -> (go_pTE env eqs p >>= function (ty0, env0) ->
                        return((TTyCon1(TOption,ty0), env0)))

    | PC1(Node,p)   -> (match p with
                        | PTup [p1;p2] ->
                            ( go_pTE env eqs p1 >>= function (ty1, env1) ->
                              go_pTE env eqs p2 >>= function (ty2, env2) ->
                              tc_Teq0 env eqs (TTyCon1(TList,(TTyCon1(TTree,ty1)))) ty2 >>-
                              let env' = appendtypeenv env2 env1 in
                              tc_Etok0 env' env >>= function _ ->  (* see NOTE 1 below *)
                              return((TTyCon1(TTree,ty1)), env') )
                        | _ -> tcfail("eT:p", "malformed tree pattern", "") )

    | PC1(c1,p)     -> (let (ty1,ty2) = typeof_C1 c1 in
                        go_pTE env eqs p >>= function (ty0,tenv'0) ->
                        tc_Teq0 env eqs ty1 ty0 >>-
                        tc_TK0 env ty2 >>-
                        return (ty2,tenv'0))
    | PCons(p1,p2)  -> (go_pTE env eqs p1 >>= function (ty1,tenv'1) ->
                        go_pTE env eqs p2 >>= function (ty2,tenv'2) ->
                        tc_Teq0 env eqs (TTyCon1(TList,ty1)) ty2 >>-
                        let tenv' = appendtypeenv tenv'2 tenv'1 in
                        tc_Etok0 tenv' env >>= function _ ->  (* see NOTE 1 below *)
                        return (ty2,tenv'))
    | PTup(ps)      -> (mapM (go_pTE env eqs) ps >>= function tes ->
                        let (tys,tenv's) = List.split tes in
                        let tenv' = concattypeenv tenv's in
                        tc_Etok0 tenv' env >>= function _ ->  (* see NOTE 1 below *)
                        return (TTup(tys), tenv'))
    | PTyped(p,ty)  -> (go_pTE env eqs p >>= function (ty0,tenv') ->
                        (* this is too strong: tc_Tdireq env eqs ty >>= function ty -> *) (* look up any bound tyvars *)
                        tc_Teq0 env eqs ty0 ty >>-
                        return (ty,tenv'))
    end
(* NOTE 1: The main environment (env) has already been checked, so
   don't check it again.  However, TODO: we still duplicate some
   effort, since env'1 and env'2 have already been checked
   individually.  Can't see how to easily fix, and the effort is
   likely small (unlike the effort in checking env!!) *)


and tc_nvT0 : nameenv -> name_value -> typ tcm
  = fun nenv nv
 -> inCtxt [TCCRule "tc_nvT0"; TCCHide (TCCNEnv nenv); TCCNameValue nv] <$>
    begin
    match nv with
    | VHashMvf(n,h,xx,ty1) ->
        (* I think the type should make sense without the environment and eqs *)
        (go_eT emptynametypeenv [] (no_loc (LocHashDot(h,xx))) >>= function ty1' ->
        assert_typ_is_ground "nvT0:name_value" ty1 >>-
        tc_Teq0 emptynametypeenv [] ty1 ty1' >>-
        let n' = match mkAbstrNameMvf(h,xx,ty1) with
                   (VHashMvf(n',_,_,_)) -> n'
                 | _ -> raise (Never_happen "tc_nvT0:Mvf") in
        unless (n = n')
          (tcfail ("nvT0","Name value hash incorrect","")) >>-
        return ty1)
    | VHashTs(n,ty1,ss) ->
        (assert_typ_is_ground "nvT0:name_value" ty1 >>-
        (* I think the type should make sense without the environment and eqs *)
        tc_TK0 emptynametypeenv ty1 >>-
        let n' = match mkAbstrNameTs(ty1,ss) with
                   (VHashTs(n',_,_)) -> n'
                 | _ -> raise (Never_happen "tc_nvT0:Ts") in
        unless (n = n')
          (tcfail ("nvT0","Name value hash incorrect","")) >>-
        return ty1)
    | VHashHts(n,ty1,ss,nv') ->
        (assert_typ_is_ground "nvT0:name_value" ty1 >>-
        (* I think the type should make sense without the environment and eqs *)
        tc_TK0 emptynametypeenv ty1 >>-
        tc_nvT0 nenv nv' >>= function _ ->
        let n' = match mkAbstrNameHts(ty1,ss,nv') with
                   (VHashHts(n',_,_,_)) -> n'
                 | _ -> raise (Never_happen "tc_nvT0:Hts") in
        unless (n = n')
          (tcfail ("nvT0","Name value hash incorrect","")) >>-
        return ty1)
    | VHashName(n,ty1) ->
        (assert_typ_is_ground "nvT0:name_value" ty1 >>-
        (* I think the type should make sense without the environment and eqs *)
        tc_TK0 emptynametypeenv ty1 >>-
        match lookup_nameenv n nenv with
        | Some (NEnv_tname(_,ty2)) ->
            assert_typ_is_ground "nvT0:name_value" ty2 >>-
            unless (typ_eq ty1 ty2)
              (tcfail ("nvT0","Name value type does not syntactically match type in environment","")) >>-
            return ty1
        | Some _ -> tcfail ("nvT0","Name not of proper sort","")
        | None   -> tcfail ("nvT0","Name not in scope",""))
    end



(* entry point *)
let tc_eT0 : nametypeenv -> eqs -> expr -> typ tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = go_eT

(* same as above, but checks env and eqs for you *)
let tc_eT : nametypeenv -> eqs -> expr -> typ tcm
  = fun env eqs e
 -> inCtxt [TCCRule "eT"; TCCEnv env; TCCEqs eqs; TCCLoc e.loc] <$>
    begin
    tc_Eok env >>-
    tc_eqsok env eqs >>-
    tc_eT0 env eqs e
    end

(* another entry point *)
let tc_pTE0 : nametypeenv -> eqs -> pat -> (typ * typeenv) tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = go_pTE

(* fill in recursive reference *)
let () = rec_tc_eT := function tenv -> function eqs -> function ex -> tc_eT tenv eqs ex


(* K has convinced himself that the old tc_sts (which checked a
   structure against a signature) was equivalent to picking the
   obvious signature for the structure and then applying ssub to the
   result.  He has therefore replaced it with the new version, below.
   This is good, because now the new tc_sts can be used in
   withspec-checking at boundary only in tc_dE *)
let tc_sts0 : nametypeenv -> eqs -> structure -> signature tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun (nenv,tenv) eqs str0
 -> inCtxt [TCCRule "sts0"; TCCEnv (nenv,tenv); TCCEqs eqs; TCCStructure str0; TCCLoc str0.loc] <$>
    let rec go tenv sign' str =
    match str with
      [] -> return (no_loc (List.rev sign'))
    | (s::str') -> (match s.desc with
    | (StrVal(id1,e))
        ->
  Debug.print' Opts.DBC_namevalues (fun () -> "sts0:strval" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
	  (withFreshIId (ident_int id1) (function iid1 ->
      (* TODO: is removing this check like this OK? *)
	  unless (true (* isvalue false eqs (exprtoprim e) *))  (* don't require closures here; treat Fn as value (!) *)
              (tcfail_ps (fun ps ->
                       "sts0","Structure binds value field to non-value expression",
                       "Structure field name "^print_ident_bind ps id1)) >>-
            tc_eT0 (nenv,tenv) eqs e >>= function ty ->
            unless (not (List.mem (ident_ext id1) (list_map "tc_sts0" ident_ext (str_valfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "sts0","Value field name duplicated in structure",
                       "Field name "^print_ident_bind ps id1)) >>-
            go (addtypeenv (Env_val(iid1,ty)) tenv) (no_loc (SVal((ident_ext id1,iid1),ty))::sign') str'))
    | (StrValMulti(id1,pats,e))
        -> (withFreshIId (ident_int id1) (function iid1 ->
            getLangTC >>= function lang ->
            unless (lang = SugaredSourceInternalForm)
              (tcfail ("sts0","Syntactic sugar not allowed in core syntax","")) >>-
            unless (List.length pats >= 1)
              (tcfail ("sts0","Pattern list must be nonempty in sugared structure value","")) >>-
            (* is always a value, since it's an abstraction! *)
            let vs = binding_vars_of_pats pats in
            withFreshIIds vs (function vs ->
            mapM (tc_pTE0 (nenv,tenv) eqs) pats >>= function ty0envs ->
            let (ty0s,tenvs) = List.split ty0envs in
            tc_Etok0 (concattypeenv (List.rev tenvs)) (nenv,tenv) >>= function env' ->
            tc_eT env' eqs e >>= function ty' ->
            let ty = List.fold_right tFunc ty0s ty' in
            unless (not (List.mem (ident_ext id1) (list_map "tc_sts0[2]" ident_ext (str_valfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "sts0","Value field name duplicated in structure",
                       "Field name "^print_ident_bind ps id1)) >>-
            go (addtypeenv (Env_val(iid1,ty)) tenv) (no_loc (SVal((ident_ext id1,iid1),ty))::sign') str')))
    | (StrTyp(tn1,ty))
        -> (tc_TK0 (nenv,tenv) ty >>-
            unless (not (List.mem (typname_ext tn1) (list_map "tc_sts0[3]" typname_ext (str_typfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "sts","Type field name duplicated in structure",
                       "Field name "^print_typname_bind ps tn1)) >>-
            go (addtypeenv (Env_typ(typname_int tn1,KEq(ty))) tenv) (no_loc (STyp(tn1,KEq(ty)))::sign') str')
	    )
    in
    go tenv [] str0.desc

let tc_sts : nametypeenv -> eqs -> structure -> signature -> unit tcm
  (* NB: caller guarantees tc_Eok env and tc_eqsok eqs *)
  = fun env eqs str sign
 -> inCtxt [TCCRule "sts"; TCCEnv env; TCCEqs eqs; TCCStructure str; TCCSignature sign; TCCLoc sign.loc] <$>
    begin
      let (nenv, tenv) = env in
      Debug.print' Opts.DBC_namevalues (fun () -> "sts:" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
    tc_sts0 env eqs str >>= function sign0 ->
    tc_ssub0 env eqs sign0 sign
    end

(* fill in recursive reference *)
let () = rec_tc_sts := tc_sts


(* same as tc_sts, but doesn't need a signature.  Would be superfluous
   if we could return a principal signature *)
let tc_stok : nametypeenv -> eqs -> structure -> unit tcm
  (* NB: caller guarantees tc_Eok env *)
  = fun (nenv,tenv) eqs str0
 -> inCtxt [TCCRule "stok"; TCCEnv (nenv,tenv); TCCEqs eqs; TCCStructure str0; TCCLoc str0.loc] <$>
    let rec go tenv str =
    match str with
      []
        -> return ()
    | (s::str') -> (match s.desc with
    | StrVal(id1,e)
        -> (unless (isvalue (false, false) eqs (exprtoprim e))  (* don't require closures here; treat Fn as value (!) *)
              (tcfail_ps (fun ps ->
                       "stok","Structure binds value field to non-value expression",
                       "Structure field name "^print_ident_bind ps id1)) >>-
            tc_eT0 (nenv,tenv) eqs e >>= function ty0 ->
            unless (not (List.mem (ident_ext id1) (list_map "tc_stok" ident_ext (str_valfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "stok","Value field name duplicated in signature",
                       "Field name "^print_ident_bind ps id1)) >>-
            withFreshIId (ident_int id1) (function iid1 ->
            tc_ExTok (nenv,tenv) iid1 ty0 >>-
            go (addtypeenv (Env_val(iid1,ty0)) tenv) str'))
    | (StrValMulti(id1,pats,e))
        -> (getLangTC >>= function lang ->
            unless (lang = SugaredSourceInternalForm)
              (tcfail ("stok","Syntactic sugar not allowed in core syntax","")) >>-
            unless (List.length pats >= 1)
              (tcfail ("stok","Pattern list must be nonempty in sugared structure value","")) >>-
            (* is always a value, since it's an abstraction! *)
            let vs = binding_vars_of_pats pats in
            withFreshIIds vs (function vs ->
            mapM (tc_pTE0 (nenv,tenv) eqs) pats >>= function ty0envs ->
            let (ty0s,tenvs) = List.split ty0envs in
            tc_Etok0 (concattypeenv (List.rev tenvs)) (nenv,tenv) >>= function env' ->
            tc_eT0 env' eqs e >>= function ty' ->
            let ty0 = List.fold_right tFunc ty0s ty' in
            unless (not (List.mem (ident_ext id1) (list_map "tc_stok[2]" ident_ext (str_valfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "stok","Value field name duplicated in structure",
                       "Field name "^print_ident_bind ps id1)) >>-
            withFreshIId (ident_int id1) (function iid1 ->
            tc_ExTok (nenv,tenv) iid1 ty0 >>-
            go (addtypeenv (Env_val(iid1,ty0)) tenv) str')))
    | StrTyp(tn1,ty)
        -> (tc_TK0 (nenv,tenv) ty >>-
            unless (not (List.mem (typname_ext tn1) (list_map "tc_stok[3]" typname_ext (str_typfieldnames {str0 with desc=str'}))))  (* TODO: inefficient *)
              (tcfail_ps (fun ps ->
                       "stok","Type field name duplicated in signature",
                       "Field name "^print_typname_bind ps tn1)) >>-
            let (tn1,str') = freshen_internal_typname (tn1,str') in
            tc_EtKok (nenv,tenv) (typname_int tn1) (KEq(ty)) >>-
            go (addtypeenv (Env_typ(typname_int tn1,KEq(ty))) tenv) str')
    )
    in
    go tenv str0.desc


let eqs_of_sign_str : hash -> signature -> structure -> eqs
(* WARNING: only makes sense if str is flat *)
  = fun h sign0 str
 -> partial_map (function s -> match s.desc with
                   (SVal(id,ty)) -> None
                 | (STyp(tn,KEq(_))) -> None
                 | (STyp(tn,KType)) ->
                     let etn = typname_ext tn in
                     match lookup_structure_etn str etn with
                       None -> raise (Never_happen ("eqs_of_sign_str: abstract type field in signature but not in structure"))
                     | Some(ty') ->
                         Some(EHash(h,etn,ty'))
                ) sign0.desc

let rec typ_has_tdot : typ -> bool
  = fun ty
 -> match ty with
    | TTyCon0 _
    | TVar(_)
                       -> false
    | TTup(tys)
    | TSum(tys)        -> List.exists typ_has_tdot tys
    | TTyCon1(_,ty)    -> typ_has_tdot ty
    | TFunc(ty1,ty2)   -> typ_has_tdot ty1 || typ_has_tdot ty2
    | TXDot(Modname(_),_) -> true
    | TXDot(Hash(_),_)    -> false  (* don't recurse inside hashes *)
    | TTyName(_)       -> false
    | TForall(_,ty)
    | TExists(_,ty)    -> typ_has_tdot ty

let eq_has_tdot : eqn -> bool
  = function
      EHash(h,etn,ty) -> typ_has_tdot ty
    | EMod(mn,etn,ty) -> true


let eqs_has_tdot : eqs -> bool
  = fun eqs
 -> List.exists eq_has_tdot eqs


let sign_has_tdot : signature -> bool
  = fun sign
 -> List.exists (function si ->
                 match si.desc with
                   SVal(_,ty)      -> typ_has_tdot ty
                 | STyp(_,KType)   -> false
                 | STyp(_,KEq(ty)) -> typ_has_tdot ty) sign.desc

let c0_has_tdot : con0 -> bool
  = fun c0
 -> typ_has_tdot (typeof_C0 c0)

let c1_has_tdot : con1 -> bool
  = fun c1
 -> match c1 with
    | Inj(i,ty) -> typ_has_tdot ty
    | SOME      -> false
    | TieCon    -> false
    | Node      -> false
    | B1 _      -> typ_has_tdot (snd (typeof_C1 c1))

let op_has_tdot : op -> bool
  = fun op
 -> typ_has_tdot (typeof_Op op)

let op_or_econst_has_tdot : op_or_econst -> bool
  = function
      OEOp op -> op_has_tdot op
    | OEEconst iid -> false

let rec expr_has_tdot : expr -> bool
  = fun e
 -> match e.desc with
     LocC0(c0)          -> c0_has_tdot c0
    | LocC1(c1,e1)       -> expr_has_tdot e1
    | LocCons(e1,e2)     -> expr_has_tdot e1 || expr_has_tdot e2
    | LocTup(es)         -> List.exists expr_has_tdot es
    | LocOp(oe,es)       -> op_or_econst_has_tdot oe || List.exists expr_has_tdot es
    | LocLazyOp(lo,es)   -> List.exists expr_has_tdot es
    | LocLoc(loc)        -> false
    | LocFn(mtch)        -> mtch_has_tdot mtch
    | LocFun(ps,e1)      -> List.exists pat_has_tdot ps || expr_has_tdot e1
    | LocTAbs(_,e)       -> expr_has_tdot e
    | LocInEnv _         -> raise (Never_happen "expr_has_tdot at InEnv")
    | LocClos _          -> raise (Never_happen "expr_has_tdot at Clos")
    | LocTClos _         -> raise (Never_happen "expr_has_tdot at TClos")
    | LocId(iid)         -> false
    | LocDot(mn,eid)     -> false
    | LocHashDot(h,eid)  -> false
    | LocIf(e1,e2,e3)    -> expr_has_tdot e1 || expr_has_tdot e2 || expr_has_tdot e3
    | LocSeq(e1,e2) | LocApp(e1,e2)
    | LocWhile(e1,e2)    -> expr_has_tdot e1 || expr_has_tdot e2
    | LocTApp(e,ty)      -> expr_has_tdot e || typ_has_tdot ty
    | LocPack(ty1,e,ty2) -> typ_has_tdot ty1 || expr_has_tdot e || typ_has_tdot ty2
    | LocNamecase(e1,tn,iid1,iid2,e,e2,e3)
                         -> List.exists expr_has_tdot [e1;e;e2;e3]
    | LocUnpack(tn,iid,e1,e2)
                        -> expr_has_tdot e1 || expr_has_tdot e2
    | LocMatch(e,m)      -> expr_has_tdot e || mtch_has_tdot m
    | LocLet(e1,pe2)     -> expr_has_tdot e1 || mtch_has_tdot [pe2]
    | LocLetMulti(ty,(ps,e1),(iid,e))
                      -> typ_has_tdot ty
                      || List.exists pat_has_tdot ps
                      || expr_has_tdot e1
                      || expr_has_tdot e
    | LocLetrec(ty,(iid,(mtch,e))) -> typ_has_tdot ty || mtch_has_tdot mtch || expr_has_tdot e
    | LocLetrecMulti(ty,(ps,(iid,(e1,e2))))
                   -> typ_has_tdot ty
                   || List.exists pat_has_tdot ps
                   || expr_has_tdot e1
                   || expr_has_tdot e2
    | LocRaise(e)        -> expr_has_tdot e
    | LocTry(e,mtch)     -> expr_has_tdot e || mtch_has_tdot mtch
    | LocMarshal(e1,e2,ty)
                         -> expr_has_tdot e1 || expr_has_tdot e2 || typ_has_tdot ty
    | LocMarshalz(mk,e2,ty)
                         -> expr_has_tdot e2 || typ_has_tdot ty
    | LocUnmarshal(e,ty)
                      -> expr_has_tdot e || typ_has_tdot ty
    | LocCol(e,eqs,ty)   -> expr_has_tdot e || eqs_has_tdot eqs || typ_has_tdot ty

    | LocRET(ty) | LocSLOWRET(ty)
                      -> typ_has_tdot ty
    | LocOP(n,oe,es)     -> op_or_econst_has_tdot oe || List.exists expr_has_tdot es
    | LocResolve(e,mn,rs)
    | LocResolve_blocked(e,mn,rs)
                      -> expr_has_tdot e
    | LocPar(e1,e2)     -> expr_has_tdot e1 || expr_has_tdot e2
    | LocFresh(ty)
    | LocCFresh(ty)
    | LocHashMvf(_,_,ty)
                        -> typ_has_tdot ty
    | LocHashTs(ty1,e,ty)
                        -> typ_has_tdot ty1 || expr_has_tdot e || typ_has_tdot ty
    | LocHashHts(ty1,e2,e1,ty)
                        -> List.exists expr_has_tdot [e1;e2] || List.exists typ_has_tdot [ty1;ty]
    | LocSwap(e1,e2,e3) -> List.exists expr_has_tdot [e1;e2;e3]
    | LocFreshfor(e1,e2)-> expr_has_tdot e1 || expr_has_tdot e2
    | LocSupport(ty,e)  -> typ_has_tdot ty || expr_has_tdot e
    | LocTie(mn,eid)    -> false
    | LocNameOfTie(e)
    | LocValOfTie(e)    -> expr_has_tdot e
    | LocNameValue(nv)  -> typ_has_tdot (type_of_name_value nv)

and mtch_has_tdot : mtch -> bool
  = fun mtch
 -> List.exists (function (p,e) ->
                 pat_has_tdot p || expr_has_tdot e) mtch

and pat_has_tdot : pat -> bool
  = fun pat
 -> match pat with
      PWild(ty) | PVar(_,ty)
        -> typ_has_tdot ty
    | PC0(c0)       -> c0_has_tdot c0
    | PC1(SOME,p)   -> pat_has_tdot p
    | PC1(c1,p)     -> c1_has_tdot c1 || pat_has_tdot p
    | PCons(p1,p2)  -> pat_has_tdot p1 || pat_has_tdot p2
    | PTup(ps)      -> List.exists pat_has_tdot ps
    | PTyped(p,ty)  -> pat_has_tdot p || typ_has_tdot ty


let str_has_tdot : structure -> bool
  = fun str
 -> List.exists (function sti ->
                 match sti.desc with
                   StrVal(_,e)  -> expr_has_tdot e
                 | StrValMulti(_,pats,e) -> List.exists pat_has_tdot pats || expr_has_tdot e
                 | StrTyp(_,ty) -> typ_has_tdot ty) str.desc


let tc_likeok : nametypeenv -> likespec -> unit tcm
  = fun ((_,tenv) as env) ls
 -> inCtxt [TCCRule "likeok"; TCCEnv env; TCCLikespec ls] <$>
    begin
    tc_Eok env >>-
    match ls with
      LikeNone    -> return ()
    | LikeMod(mn) -> unless (is_Some (lookup_typeenv_modname tenv mn))
                       (tcfail("likeok","Module name not in scope",""))
    | LikeStr(str) -> tc_stok env [] str
    end


(* check if a mod_user_body is OK, ignoring mode and delta issues *)
let tc_mubE : nametypeenv -> modname -> mod_user_body -> typeenv tcm
(* NB: assumes Eok env *)
  = fun ((nenv,tenv) as env) mn mu
 -> getLangTC >>= function lang ->
    unless (lang <> CompiledForm)
      (tcfail ("dE","Attempt to typecheck source module in compiled mode [NEVER HAPPEN]","")) >>-
    let weqs = list_map "tc_mu" eMod mu.mu_withspec in
    tc_eqsok env weqs >>-  (* only module equations, by construction *)
    (if !Opts.internal_weqs then
      tc_sts env weqs mu.mu_str mu.mu_sign
    else
      tc_sts0 env [] mu.mu_str >>= function sign0 ->
      tc_seq0 env weqs sign0 mu.mu_sign
    ) >>-
    tc_vneok nenv mu.mu_vne >>-
    return (addtypeenv (Env_mod(mn,mu.mu_sign)) emptytypeenv)


(* check if an import_user_body is OK, ignoring mode and delta issues *)
let tc_iubE : nametypeenv -> modname -> import_user_body -> typeenv tcm
(* NB: assumes Eok env *)
  = fun ((nenv,tenv) as env) mn iu
 -> getLangTC >>= function lang ->
    unless (lang <> CompiledForm)
      (tcfail ("dE","Attempt to typecheck source module in compiled mode [NEVER HAPPEN]","")) >>-
    tc_vceok env iu.iu_vce >>-
    tc_likeok env iu.iu_likespec >>-
    tc_eqMS env (if !Opts.disable_import_typecheck then None else iu.iu_mo) iu.iu_sign >>-
    tc_rsok iu.iu_resolvespec >>-
    return (addtypeenv (Env_mod(mn,iu.iu_sign)) emptytypeenv)


let tc_dE : nametypeenv -> source_definition -> typeenv tcm
  = fun ((nenv,tenv) as env) def
 -> inCtxt [TCCRule "dE"; TCCEnv env; TCCSDefinition def] <$>
    begin
    getLangTC >>= function lang ->
    unless (lang <> CompiledForm)
        (tcfail ("dE","Attempt to typecheck source module in compiled mode [NEVER HAPPEN]","")) >>-
    match def with
      Mod_user(mn,md,mu)
        -> (tc_Eok env >>-  (* TODO: inefficient *)
            tc_mubE env mn mu >>= function tenv' ->
            (* now do any delta or mode stuff required *)
            return tenv')
    | Mod_alias(mn,ma)
        -> (tc_Eok env >>-  (* TODO: inefficient *)
            (match lookup_typeenv_modname tenv ma.ma_modname with
               None -> tcfail ("dE","Module alias to module name not in scope","")
             | Some(sign) -> return sign) >>= function sign ->
            unless (s_eq ma.ma_sign (selfify_signature (Modname(ma.ma_modname)) sign))
              (tcfail ("dE","Module alias signature does not agree with selfified module signature from environment","")) >>-
            return (addtypeenv (Env_mod(mn,ma.ma_sign)) emptytypeenv))
    | Import_user(mn,md,iu)
      ->  (tc_Eok env >>-  (* TODO: inefficient *)
           tc_iubE env mn iu >>= function tenv' ->
           (* now do any delta or mode stuff required *)
           return tenv')
    | Mark_user(mk)
        -> (tc_Eok env >>-
            return emptytypeenv)
    end


let tc_cdE0 : nametypeenv -> definition -> typeenv tcm
(* NB: assumes Eok env *)
  = fun ((nenv,_) as env) def
 -> inCtxt [TCCRule "cdE"; TCCEnv env; TCCCDefinition def] <$>
    begin
    getLangTC >>= function lang ->
    unless (lang = CompiledForm)
        (tcfail ("cdE","Attempt to typecheck compiled module in non-compiled mode [NEVER HAPPEN]","")) >>-
    match def with
      Mod_compile(mn,mc)
        -> (tc_hok nenv mc.mc_hash >>-
            tc_eqsok (nenv,emptytypeenv) mc.mc_eqs >>-
            tc_sok (nenv,emptytypeenv) mc.mc_sign0 >>-
            tc_stf mc.mc_str >>-
            let eqs0 = eqs_of_sign_str mc.mc_hash mc.mc_sign0 mc.mc_str in
            let sign1 = type_flatten_sign (selfify_signature (Hash mc.mc_hash) mc.mc_sign0) in
            (* NB: sign1 is flat and concrete by construction; no need to test *)
            unless (s_eq mc.mc_sign1 sign1)
              (tcfail ("dE","Compiled signature does not match flattened selfified semicompiled signature","")) >>-
            tc_eqsok env (mc.mc_eqs @ eqs0) >>-  (* XXX inefficient *)
            tc_sts env (mc.mc_eqs @ eqs0) mc.mc_str mc.mc_sign1 >>-
            tc_vnok nenv mc.mc_vn >>-
            unless (not (eqs_has_tdot mc.mc_eqs))
              (tcfail ("dE","Compiled equation set has remaining module projections","")) >>-
            unless (not (sign_has_tdot mc.mc_sign0))
              (tcfail ("dE","Semicompiled signature has remaining module projections","")) >>-
            unless (not (sign_has_tdot mc.mc_sign1))
              (tcfail ("dE","Compiled signature has remaining module projections","")) >>-
            unless (not (str_has_tdot mc.mc_str))
              (tcfail ("dE","Structure has remaining module projections","")) >>-
            return (addtypeenv (Env_mod(mn,mc.mc_sign1)) emptytypeenv))
    | Mod_imod(mn,mi)
        -> (tc_hok nenv mi.mi_hash >>-
            tc_eqsok (nenv,emptytypeenv) mi.mi_eqs >>-
            tc_sok (nenv,emptytypeenv) mi.mi_sign0 >>-
            let sign1 = type_flatten_sign (selfify_signature (Hash mi.mi_hash) mi.mi_sign0) in
            (* NB: sign1 is flat and concrete by construction; no need to test *)
            unless (s_eq mi.mi_sign1 sign1)
              (tcfail ("dE","Compiled signature does not match flattened selfified semicompiled signature","")) >>-
            let str_combined = at_loc mi.mi_str_loc (List.rev_append mi.mi_str_done mi.mi_str_todo) in  (* TODO XXX a little inefficient *)
            tc_stf str_combined >>-
            let eqs0 = eqs_of_sign_str mi.mi_hash mi.mi_sign0 str_combined in
            tc_eqsok env (mi.mi_eqs @ eqs0) >>-  (* XXX inefficient *)
            tc_sts env (mi.mi_eqs @ eqs0) str_combined mi.mi_sign1 >>-
            tc_vnok nenv mi.mi_vn >>-
            unless (not (eqs_has_tdot mi.mi_eqs))
              (tcfail ("dE","Compiled equation set has remaining module projections","")) >>-
            unless (not (sign_has_tdot mi.mi_sign0))
              (tcfail ("dE","Semicompiled signature has remaining module projections","")) >>-
            unless (not (sign_has_tdot mi.mi_sign1))
              (tcfail ("dE","Compiled signature has remaining module projections","")) >>-
            unless (not (str_has_tdot str_combined))
              (tcfail ("dE","Structure has remaining module projections","")) >>-
            let go si =
              match si.desc with
                StrVal(x,e) -> unless (isvalue (false,false) mi.mi_eqs (exprtoprim e))
                                 (tcfail ("dE","Evaluated portion of Imod contains non-value:",external_ident_to_string (ident_ext x)))
              | StrTyp _ -> return ()
              | StrValMulti(x,_,_) -> tcfail ("dE","StrValMulti in Imod:",external_ident_to_string (ident_ext x))
            in
            mapMz go mi.mi_str_done >>-
            return (addtypeenv (Env_mod(mn,mi.mi_sign1)) emptytypeenv))
    | Import_compile(mn,ic)
        -> (tc_hok nenv ic.ic_hash >>-
            let sign1 = type_flatten_sign (selfify_signature (Hash ic.ic_hash) ic.ic_sign0) in
            (* NB: sign1 is flat and concrete by construction; no need to test *)
            unless (s_eq ic.ic_sign1 sign1)
              (tcfail ("dE","Compiled signature does not match flattened selfified semicompiled signature","")) >>-
            tc_sok (nenv,emptytypeenv) ic.ic_sign0 >>-
            tc_sts (nenv,emptytypeenv) [] ic.ic_likestr (limitdom ic.ic_sign0) >>-
            tc_stf ic.ic_likestr >>-
            tc_rsok ic.ic_resolvespec >>-
            tc_vcok nenv ic.ic_vc >>-
            unless (not (sign_has_tdot ic.ic_sign0))
              (tcfail ("dE","Semicompiled signature has remaining module projections","")) >>-
            unless (not (sign_has_tdot ic.ic_sign1))
              (tcfail ("dE","Compiled signature has remaining module projections","")) >>-
            unless (not (str_has_tdot ic.ic_likestr))
              (tcfail ("dE","Structure has remaining module projections","")) >>-
            tc_eqMS env (if (!Opts.disable_import_typecheck) then None else ic.ic_mo) ic.ic_sign0 >>-
            return (addtypeenv (Env_mod(mn,ic.ic_sign1)) emptytypeenv))
    | Mod_fresh(mn,mu)
        -> (withLangTC SourceInternalFreshForm (
              tc_mubE env mn mu))
    | Import_fresh(mn,iu)
        -> (withLangTC SourceInternalFreshForm (
              tc_iubE env mn iu))
    | Mark_compile(mk)
        -> (return emptytypeenv)
    end

let tc_cdsE0 : nametypeenv -> definitions -> typeenv tcm
  = fun (nenv,tenv) ds
(* NB: assumes Eok (nenv,tenv) *)
 -> inCtxt [TCCRule "cdsE0"; TCCEnv (nenv,tenv); TCCCDefinitions ds] <$>
    let rec go tenv ds =
    match ds with
      []      -> return emptytypeenv
    | (d::ds) -> tc_cdE0 (nenv,tenv) d >>= function tenv' ->
                 go (appendtypeenv tenv' tenv) ds >>= function tenv'' ->
                 return (appendtypeenv tenv'' tenv')
    in
    go tenv ds >>= function tenv' ->
    match firstdup mn_eq (partial_map modname_of_typeenv_entry (typeenv_list_of_typeenv tenv')) with
      Some mn -> tcfail_ps (fun ps ->
                            "cdsE","Duplicate module name in environment",print_modname ps mn)
    | None    ->
    return tenv'

let tc_cdsE : nametypeenv -> definitions -> typeenv tcm
  = fun env ds
 -> inCtxt [TCCRule "cdsE"; TCCEnv env; TCCCDefinitions ds] <$>
    tc_Eok env >>-
    tc_cdsE0 env ds >>= function tenv' ->
    tc_Etok0 tenv' env >>= function _ ->
    return tenv'


let tc_nameok : string -> string -> tycon0 -> nametypeenv -> name_value -> unit tcm
  = fun rulename objectname objecttype (nenv,_) ti_name
 -> inCtxt [TCCRule rulename; TCCHide (TCCNEnv nenv); TCCNameValue ti_name] <$>
    begin
    tc_nvT0 nenv ti_name >>= function ty ->
    unless (typ_eq ty (TTyCon0 objecttype))
        (tcfail (rulename,objectname ^ " name has wrong type",""))
    end

let tc_threadnameok : nametypeenv -> name_value -> unit tcm
  = fun env nv
 -> tc_nameok "threadnameok" "Thread" TThread env nv

let tc_namesok : string -> string -> tycon0 -> nametypeenv -> name_value -> name_value -> unit tcm
  = fun rulename objectname objecttype (nenv,_) nv1 nv2
 -> inCtxt [TCCRule rulename; TCCHide (TCCNEnv nenv); TCCNameValue nv1] <$>
    begin
    tc_nvT0 nenv nv1 >>= function ty ->
    unless (typ_eq ty (TTyCon0 objecttype))
        (tcfail (rulename,objectname ^ " name has wrong type","")) >>-
    (if nv1 == nv2 then  (* pointer comparison - efficiency hack *)
      return ()
    else begin
      tc_nvT0 nenv nv1 >>= function ty ->
      unless (typ_eq ty (TTyCon0 objecttype))
        (tcfail (rulename,objectname ^ " name has wrong type","")) >>-
      unless (name_value_eq nv1 nv2)
        (tcfail (rulename,objectname ^ " name differs from map index",""))
    end)
    end

let tc_threadok : nametypeenv -> (name_value * thread_info) -> unit tcm
  = fun ((nenv,_) as env) (nv, ti)
 -> inCtxt [TCCRule "threadok"; TCCHide (TCCNEnv nenv); TCCNameValue nv] <$>
    begin
    tc_namesok "threadok" "Thread" TThread env nv ti.ti_name >>-
    (* TODO: check ti.ti_state (perhaps elsewhere) *)
    tc_cdsE0 env ti.ti_defs >>= function tenv' ->
    tc_Etok0 tenv' env >>= function env' ->
    tc_eT env' [] (primtoexpr ti.ti_expr) >>= function ty -> (* XXX ugh! prim conversion *)
    (if !Opts.nonunitthread then
      return ()  (* TODO would be nice to check if it has the same type as last time, if this thread had a last time - but unclear how to do that. *)
    else
      tc_Teq0 env' [] ty (TTyCon0 TUnit))
    end

let tc_mutexok : nametypeenv -> (name_value * mutex_info) -> unit tcm
  = fun ((nenv,_) as env) (nv, mi)
 -> inCtxt [TCCRule "mutexok"; TCCHide (TCCNEnv nenv); TCCNameValue nv] <$>
    begin
    tc_namesok "mutexok" "Mutex" TMutex env nv mi.mtxi_name >>-
    (match mi.mtxi_state with
      None ->
        return ()
    | Some lmi ->
        tc_threadnameok env lmi.lmtxi_owner >>-  (* TODO: check exists (elsewhere) *)
        mapMz (tc_threadnameok env) (AQueue.toList lmi.lmtxi_waiting)
    )
    end


let tc_cvarok : nametypeenv -> (name_value * cvar_info) -> unit tcm
  = fun ((nenv,_) as env) (nv, ci)
 -> inCtxt [TCCRule "cvarok"; TCCHide (TCCNEnv nenv); TCCNameValue nv] <$>
    begin
    tc_namesok "cvarok" "CVar" TCVar env nv ci.cvi_name >>-
    mapMz (tc_threadnameok env) (AQueue.toList ci.cvi_waiting)
    end


(* a little bit of result caching.  Invariant: it is the case that:
   tc_Eok !last_econst_env
   tc_cdsE !last_econst_env !last_ds >>= function env ->
   last_env' = env @ econst_env
*)
let last_econst_env = ref emptytypeenv  (* obviously emptytypeenv is Eok *)
let last_defs = ref []                   (* also, [] [] is cdsok... *)
let last_tenv_ds = ref emptytypeenv        (* ... and yields an empty E *)

let tc_configT : configuration -> unit tcm
    = fun cfg ->
      inCtxt [TCCRule "configT"] <$> begin
      getLangTC >>= function lang ->
      unless (lang = CompiledForm)
        (tcfail ("configT","Attempt to typecheck configuration in non-compiled mode [NEVER HAPPEN]","")) >>-
      let nenv = cfg.cfg_nenv in
      let senv = mktypeenv (list_map "tc_configT" (function (l,ty) -> Env_loc (l,ty)) cfg.cfg_senv) in
      (* now typecheck defs, if we need to *)
      (if econst_env == !last_econst_env && cfg.cfg_defs == !last_defs then begin
        (* NB: optimisation: this is physical (pointer) equality! *)
        (* NB: don't bother checking nenv or senv, since they are only ever extended, never retracted *)
        Debug.print' Opts.DBC_tcopt (fun () -> "tc_configT: winning on duplicate env");
        return !last_tenv_ds
      end
      else begin
        let tenv = appendtypeenv senv econst_env in
        (* tc_Eok (nenv,tenv) >>-
           Do NOT check this here; senv may depend on tenv_ds.
           In fact, tc_Eok senv is checked inside tc_store. *)
        tc_cdsE0 (nenv,tenv) cfg.cfg_defs >>= function tenv_ds ->
          doTC (fun () ->
            last_econst_env := econst_env;
            last_defs := cfg.cfg_defs;
            last_tenv_ds := tenv_ds) >>-
          return tenv_ds
      end) >>= function tenv_ds ->
      let tenv' = appendtypeenv tenv_ds econst_env in
      tc_Etok0 tenv' (nenv, emptytypeenv) >>= function _ ->  (* check E_const and E_ds *)
      tc_store (nenv, tenv') senv cfg.cfg_store >>-  (* check E_s and store *)
      let env'' = (nenv, appendtypeenv senv tenv') in
      (* now check the various runtime components *)
      mapMz (tc_threadok env'') (nameValueMap_assocs cfg.cfg_threads) >>-
      mapMz (tc_mutexok  env'') (nameValueMap_assocs cfg.cfg_mutexes) >>-
      mapMz (tc_cvarok   env'') (nameValueMap_assocs cfg.cfg_cvars  ) >>-
      (* now do sanity checks *)
      (* TODO: each thread on cfg_threads appears exactly once (no more no
         less) in the bag comprised of runnable bunion dom(slowcall)
         bunion things-waiting-on-mutexes bunion
         things-waiting-on-cvars *)
      (* TODO: owners of locked mutexes exist *)
      (* TODO: things waiting on slowcall are SLOWRETs and in state TsSlowCall *)
      (* TODO: things waiting on mutexes are LOCKs and in state TsMutexBlocked *)
      (* TODO: things waiting on cvars are WAITINGs and in state TsCVarWaiting *)
      (* TODO: others are none of the above and in state TsRunnable *)
      (* TODO: anything else? *)
      return ()
      end


let tc_mbT : marshalled_body -> unit tcm
  = fun mb
 -> inCtxt [TCCRule "mbT"; TCCMBody mb] <$>
    begin
    getLangTC >>= function lang ->
    unless (lang = CompiledForm)
        (tcfail ("mbT","Attempt to typecheck marshalled body in non-compiled mode [NEVER HAPPEN]","")) >>-
    (match mb.mb_ne with
      None ->
        tcfail ("mbT","Attempt to typecheck marshalled body lacking name environment","")
    | Some nenv ->
        return nenv) >>= function nenv ->
    let tenv = econst_env in
    let env = (nenv,tenv) in
    tc_Eok env >>-
    let senv = mktypeenv (list_map "tc_mbT" (function (l,ty) -> Env_loc(l,ty)) mb.mb_storeenv) in
    tc_Etok0 senv env >>= function (nenv1,tenv1) ->
    tc_cdsE (nenv1,tenv1) mb.mb_defs >>= function tenv' ->
    let env'' = (nenv1,appendtypeenv tenv' tenv1) in
    tc_eT0 env'' [] mb.mb_expr >>= function ty0 ->
    (* TODO: unsure if we should allow env'' to prove the equality partly too *)
    (* K says: yes, you must, because |- e : T includes type equality, but tc_eT omits it on output *)
    tc_Teq0 env'' [] ty0 mb.mb_typ >>-
    let env''' = (nenv,appendtypeenv tenv' tenv) in
    tc_store env''' senv mb.mb_store >>-
    return ()
    end


(* == INTERFACE HELPERS == *)

(* generate substitution rather than infstate *)
let runTCinf : typ option -> language -> (internal_typname * eqs) list -> 'a tcm -> 'a * tysubst
  = fun deftyo lang tns m
 -> runTCinf0 lang tns
      (m >>= function result ->
       get_substTC deftyo >>= function sub ->
       return (result,sub))

(* run, but print error messages on the way up, since uncaught
   exceptions get their bodies truncated before the runtime prints them *)
let echo_errs : ('a -> 'b) -> ('a -> 'b)
  = fun f x
 -> try f x with TCFail(s) -> (Debug.print_string_really s; raise (TCFail "type error"))

let runTC_echo_errs : language -> 'a tcm -> 'a
  = fun lang m
 -> echo_errs (runTC lang) m

let runTCinf_echo_errs : typ option -> language -> (internal_typname * eqs) list -> 'a tcm -> 'a * tysubst
  = fun deftyo lang tns m
 -> echo_errs (runTCinf deftyo lang tns) m


(* == PUBLIC INTERFACE == *)

(* calculating free type variables (with colours) *)
(* the colour is the colour at which the tyvar appears *)
let freetyvars_expr : eqs -> expr -> (internal_typname * eqs) list
  = fun eqs ex -> (map_to_assoc <.> runFnames eqs <.> freetyvars_expr) ex
let freetyvars_source_definition : source_definition -> (internal_typname * eqs) list
  = map_to_assoc <.> runFnames [] (* defs always black *) <.> freetyvars_source_definition

(* inference *)
(* deftyo: default type to use for underspecified type variables, if desired;
   metas:  list of unifiable tyvars;
   env:    type environment of expression;
   eqs:    colour of expression;
   e:      expression to infer;
   expected_ty: the type we expect the result to have, if any
   -> expr:  type-inferred expression (has no free unifiable tyvars)
      typ:   the inferred type of the expression (ditto)
*)
let tyinf_expr : typ option -> (internal_typname * eqs) list -> nametypeenv -> eqs -> expr -> typ option -> expr * typ
    = fun deftyo metas env eqs e expected_ty ->
      let (ty,sub) =
        runTCinf_echo_errs deftyo SugaredSourceInternalForm metas
          (tc_eT env eqs e >>= function ty ->
           match expected_ty with
             None -> return ty
           | Some ety -> (inCtxt [TCCRule "tyinf_expr"; TCCTyp ety; TCCLoc e.loc] <$>
                         tc_Teq0 env eqs ty ety) >>-
                         return ty
          ) in
      (tysub_expr sub e,
       tysub_typ sub ty)

(* as above, but for a source_definition yielding an environment *)
let tyinf_source_definition : typ option -> (internal_typname * eqs) list -> nametypeenv -> source_definition -> source_definition * typeenv
    = fun deftyo metas env d ->
      let (env',sub) =
        runTCinf_echo_errs deftyo SugaredSourceInternalForm metas
          (tc_dE env d) in
      (tysub_source_definition sub d,
       tysub_typeenv sub env')

(* checking *)

(* Checking has changed.  It now no longer does true checking; instead
   it does a full type inference.  This is because it was deemed
   desirable to have a non-fully-annotated language - specifically,
   Raise no longer bears a type annotation as the implementors thought
   it not worth the bother of keeping it correct as it floated through
   contexts.  Without it, we have to do inference every time.  Ugh.  KW.

   Note that for this checking, we must always allow defaulting, since
   expressions like ((raise v1) (raise v2)) have an ambiguous
   intermediate type even if the result type is known and all types in
   the source program were unambiguous.  KW.
*)

(* iscore: the expression is a compiled expression;
   env:    type environment of expression;
   eqs:    colour of expression;
   e:      expression to check;
   expected_ty:  if specified, the type the expression should have;
   -> typ:   the type of the expression
*)
let runTCinf_check_echo_errs : language -> 'a tcm -> 'a * tysubst
    = fun lang m ->
      runTCinf_echo_errs
        (Some (TTyCon0 TUnit))  (* always allow defaulting; intermediate states may be ambiguous *)
        lang
        []  (* no metas *)
        m

let tcheck_expr : language -> nametypeenv -> eqs -> expr -> typ option -> typ
    = fun lang env eqs e expected_ty ->
      let (ty,sub) =
        runTCinf_check_echo_errs lang
          (tc_eT env eqs e >>= function ty ->
            (match expected_ty with
              None -> return ()
            | Some ety -> inCtxt [TCCRule "tcheck_expr"; TCCTyp ety; TCCLoc e.loc] <$>
                          (tc_Teq0 env eqs ty ety)) >>-
            return ty)
      in
      tysub_typ sub ty

(* as above, but for a compiled definition yielding an environment *)
let tcheck_definition : nametypeenv -> definition -> typeenv option -> typeenv
    = fun ((nenv,tenv) as env) d expected_tenv ->
      let (tenv',sub) =
      runTCinf_check_echo_errs CompiledForm
        (tc_Eok env >>-
         tc_cdE0 env d >>= function tenv' ->
          (match expected_tenv with
            None -> return ()
          | Some etenv -> inCtxt [TCCRule "tcheck_definition"; TCCTEnv etenv] <$>
                         (tc_Esub (nenv,tenv) tenv' etenv)) >>-
          return tenv')
          in
      tysub_typeenv sub tenv'

(* as above, but for a compiled_unit yielding an env and possibly a type *)
let tcheck_cdefs_eo : nametypeenv -> (definitions * expr option) -> (typeenv * typ option) option -> typeenv * typ option
    = fun ((nenv,tenv) as env) (ds,eo) expected_tenvtyoo ->
      Debug.print' Opts.DBC_namevalues (fun () -> "tcheck_defs" ^ Pretty.print_nameenv (fresh_printer_state 3 0 false) nenv);
      let ((tenv',tyo),sub) =
      runTCinf_check_echo_errs CompiledForm
        (tc_cdsE env ds >>= function tenv' ->
         (match expected_tenvtyoo with
           Some(etenv,_) -> inCtxt [TCCRule "tcheck_cdefs_eo"; TCCTEnv etenv] <$>
                            (tc_Esub (nenv,tenv) tenv' etenv)
         | None -> return ()) >>-
         (match eo with
           None   -> return None
         | Some e -> tc_eT (nenv,appendtypeenv tenv' tenv) [] e >>= function ty ->
                     return (Some ty)) >>= function tyo ->
         (match (tyo, expected_tenvtyoo) with
           (None,Some(_,Some(_))) -> tcfail ("tcheck_cdefs_eo","Can't have a return type without an expression","")
         | (Some ty,Some(_,Some(ety))) ->
             inCtxt [TCCRule "tcheck_cdefs_eo"; TCCTyp ety] <$>
             (tc_Teq0 (nenv,appendtypeenv tenv' tenv) [] ty ety)
         | _ -> return ()) >>-
         return (tenv',tyo))
      in
      (tysub_typeenv sub tenv',
       match tyo with None -> None | Some ty -> Some (tysub_typ sub ty))

let tcheck_configuration config =
  let ((),sub) =
    runTCinf_check_echo_errs CompiledForm
      (tc_configT config)
  in
  ()

let tcheck_mb marshalled_body =
  let ((),sub) =
    runTCinf_check_echo_errs CompiledForm
      (tc_mbT marshalled_body)
  in
  ()

(* this is noisy, find a quiet version? *)
let tcheck_ssub_inf env eqs sign1 sign2 expected_ty =
  let (ty, sub) =
    runTCinf_check_echo_errs CompiledForm
      (tc_ssub env eqs sign1 sign2 >>= function ty ->
	( match expected_ty with
	  None -> return ()
	| Some ety -> raise (Unimplemented "tcheck_ssub with supplied type") 	      (* KW: TODO? *)
	      ))
  in (* tysub_typ sub ty *) ()




(* this is quiet (whereas the above are noisy) as it's used by eval.ml within resolve methods *)

  let tcheck_ssub env eqs sign1 sign2 = (* runTC CompiledForm (tc_ssub env eqs sign1 sign2) *)
    tcheck_ssub_inf env eqs sign1 sign2 None
