(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * parsertypes.ml                                                 *
 *                                                                *
 * Version: $Id: parsertypes.ml,v 1.506 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Ast

type compiled_or_not = Compiled | Source

type econsts_allowed_or_not = Econsts_allowed | Econsts_not_allowed

type runtime_opt =
  | Lithash

type runtime_opts = runtime_opt list

type stdlib_or_not = InStdLib | InUserCode

type parser_base_mode =
  | Program_mode of     (* programs *)
      compiled_or_not
    * econsts_allowed_or_not
    * stdlib_or_not
  | Caml_mode           (* Caml mli files *)
  | Marshalled_mode     (* code inside marshalled: implies econsts allowed *)
  | Hash_mode           (* code inside hashes: implies econsts allowed *)


type parser_mode = parser_base_mode * runtime_opts

(* (* TODO: think how this should work *) | Anything_mode       (* no mode restrictions:
                           probably useful for testing since we'll
                           want to be able to parse arbitrary
                           intermediate states for testing the type
                           checker *)
*)

let compiled_or_marshalled_mode = function (pbm, opts) -> match pbm with
  | Program_mode (Compiled, _, _) | Marshalled_mode -> true | _ -> false

let source_mode = function (pbm, opts) -> match pbm with
  | Program_mode (Source, _, _) -> true | _ -> false

let econsts_allowed_mode = function (pbm, opts) -> match pbm with
  | Program_mode (_, Econsts_allowed, _)
  | Marshalled_mode
  | Hash_mode     -> true
  | _ -> false

let caml_mode = function (pbm, opts) -> match pbm with
  | Caml_mode -> true | _ -> false

let marshalled_mode = function (pbm, opts) -> match pbm with
  | Marshalled_mode -> true | _ -> false

let hash_mode = function (pbm, opts) -> match pbm with
  | Hash_mode -> true | _ -> false

let stdlib_mode = function (pbm, opts) -> match pbm with
  | Program_mode (_, _, InStdLib) -> true | _ -> false

(* maps from user specified strings to internal names. *)

module StringMap =
  Map.Make (struct type t = string let compare = compare end)
module StringStringMap =
  Map.Make (struct type t = string * string let compare = compare end)

type parser_maps = { ei_map : internal_ident StringMap.t;
                     et_map : internal_typname StringMap.t;
                     em_map : modname StringMap.t;
                     eim_map : modname StringStringMap.t;
                   }

type parser_state = { pm : parser_mode;
                      parser_maps: parser_maps;
                      econst_string_of_ident : internal_ident -> string;
                      econst_ident_of_string : string -> internal_ident;
                    }

(* when we process a new module definition it might or might not come
with a user-specified internal name.  In general we can define a
module with a user-specified internal name but refer to it later by
just its external name, though not vice versa.  To handle this easily,
we store the current map of external names to real internal names in
em_map for all module definitions; we store the current map from
external and user-specified-internal names to real internal names in
eim_map. *)

(* the argument to new_parser_state is an option:
   Some f , where f is usually Econst.ident_of_string for the parsing of external constants
   None   , when such a function is non needed (e.g. when in Caml_mode)
*)

let new_parser_state pm econsts_opt =
  let (econst_ident_of_string,econst_string_of_ident) =
    match econsts_opt with
    | None -> ((fun _ -> raise (Failure "No econsts")), (fun _ -> raise (Failure "No econsts")))
    | Some (f,g) -> (f,g)
  in
  { pm = pm;
    parser_maps = { ei_map = StringMap.empty;
                    et_map = StringMap.empty;
                    em_map = StringMap.empty;
                    eim_map = StringStringMap.empty;
                  };
    econst_ident_of_string = econst_ident_of_string;
    econst_string_of_ident = econst_string_of_ident;
  }


let replace_mode ps pbm' = let (pbm, opts) =
  ps.pm in {ps with pm = (pbm', opts);}

let replace_opts ps opts' = let (pbm, opts) =
  ps.pm in {ps with pm = (pbm, opts');}

let replace_mode_opts ps pm' =
  {ps with pm = pm';}

let lithash_allowed ps = let (pbm, opts) = ps.pm in
(List.mem Lithash opts)


type ('a,'b) parser_entry = Myparsing.envp -> (Mylexing.lexbuf  -> 'b) -> Mylexing.lexbuf -> 'a

