(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * hashifysupp.ml                                                 *
 *                                                                *
 * Version: $Id: hashifysupp.ml,v 1.13 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Ast
open Util
open Tysupp

(* == EXCEPTIONS == *)
exception Bad_sourcedef_valuability of string
exception Withspec_equation_from_import of string
exception Withspec_wrt_bad_type_field of string
exception Withspec_types_not_equal of string
exception Linkok_not of string



exception Hashify of Exceptions.hashify_exn

let fatal e =
  match e with
    Withspec_equation_from_import s->
      raise (Hashify (Exceptions.Withspec_equation_from_import s))
  | Withspec_wrt_bad_type_field s->
      raise (Hashify (Exceptions.Withspec_wrt_bad_type_field s))
  | Withspec_types_not_equal s->
      raise (Hashify (Exceptions.Withspec_types_not_equal s))
  | Linkok_not s ->
      raise (Hashify (Exceptions.Linkok_not s))
  | Bad_sourcedef_valuability s ->
      raise (Hashify (Exceptions.Bad_sourcedef_valuability s))
  | e -> raise (Never_happen ("Unknown exception in hashify: " ^ Printexc.to_string e))


(* == COMPILATION MAP == *)

type compilation_map = definition ModnameMap.t

let (empty_cm : compilation_map) = ModnameMap.empty

let add_compiled_module cm d =
  match d with
  | Mark_compile _ -> cm
  | _ ->
      let m = modname_of_definition d in
      assert (not (ModnameMap.mem m cm)); ModnameMap.add m d cm

let add_compiled_modules cm ds =
  List.fold_left (fun d -> add_compiled_module d) cm ds;;

(* == CFRESH == *)

let no_cfresh opt_res e =
  let result =
    ( match opt_res with
        | None -> ref true
        | Some res -> res )
  in let (_:prim_expr) =
      early_emap_prim_expr false
        ( fun j e1 -> match e1 with
            | CFresh ty -> result := false ; Some (CFresh ty)
            | _ -> None )
        None e
  in !result


let no_guarded_cfresh e = (* TODO: is this sufficient for ordinary expressions? *)
  let result = ref true in
  let (_:expr) =
    early_emap_expr false
      (fun _ e ->
	match e.desc with
	  LocFn(_) -> result := !result && (no_cfresh None (exprtoprim e)); None
	| LocFun(_) -> result := !result && (no_cfresh None (exprtoprim e)); None
	| LocTApp(_,_) -> result := !result && (no_cfresh None (exprtoprim e)); None
	| _ -> None)
      None e
  in !result

let no_hashMvf opt_res e =
  let result =
    ( match opt_res with
        | None -> ref true
        | Some res -> res )
  in let (_:prim_expr) =
      early_emap_prim_expr false
        ( fun j e1 -> match e1 with
            | HashMvf _ -> result := false ; Some e1
            | _ -> None )
        None e
  in !result

let no_tie opt_res e =
  let result =
    ( match opt_res with
        | None -> ref true
        | Some res -> res )
  in let (_:prim_expr) =
      early_emap_prim_expr false
        ( fun j e1 -> match e1 with
            | Tie _ -> result := false ; Some e1
            | _ -> None )
        None e
  in !result


(* == VALUABILITIES == *)

let (find_valuabilities : compilation_map -> Ast.modname -> Ast.valuabilities)
    =
  fun cm -> fun modname ->
    Debug.print' Opts.DBC_hashify (function () -> "fv: looking for: " ^ Pretty.print_modname (Dump.fresh_printer_state_all ()) modname);

    let non_valuabilities =
      { term_valuability = Nonvaluable;
        type_valuability = Nonvaluable;
      } in
    let vubs' = begin match
      try
        ModnameMap.find modname cm
      with Not_found -> raise (Never_happen("find vubs: modname not found in cm"))
    with
      | Mod_compile ((ext, int), mc_body) -> Debug.print(function () ->
	  "vubs for " ^ ext ^
	  "Term: " ^ (function Valuable -> "Valuable" | CValuable -> "CValuable" | Nonvaluable -> "Nonvaluable") mc_body.mc_vubs.term_valuability
	  ^ "Type: " ^ (function Valuable -> "Valuable" | CValuable -> "CValuable" | Nonvaluable -> "Nonvaluable") mc_body.mc_vubs.type_valuability
	    );
	  mc_body.mc_vubs
      | Import_compile (_, ic_body) -> ic_body.ic_vubs
      | Mod_fresh _ -> non_valuabilities
      | Import_fresh _ -> non_valuabilities
      | Mark_compile _ -> raise (Never_happen "find_valuabilities of Mark_compile")
      | Mod_imod _ -> raise (Never_happen "find_valuabilities of Mod_imod")
      end
    in vubs'
  ;;


let check_valuability_expr : compilation_map -> Ast.expr -> Ast.valuability -> bool
  =
  fun cm -> fun e -> fun vub ->
    (* Debug.print' Opts.DBC_hashify (function () -> "cve: " ^ Pretty.print_expr (Dump.fresh_printer_state_all ()) e); *)

    let aux_expr vubs e =
      let result = ref true
      in let (_:expr) =
          early_emap_expr false
            ( fun j e1 -> match e1.desc with
                | LocDot((ext, int) as mn',_) | LocTie((ext, int) as mn',_) ->
                    result := !result && (
		      let r = List.mem ((find_valuabilities cm mn').term_valuability) vubs
		      in Debug.print(function () -> "emap modname " ^ ext ^ if r then "true" else "false"); r)
			;
                    None
		| _ -> None )
             None e
      in !result
    in
    let aux_type vubs result t =
      early_tmap_typ
        ( fun t1 -> match t1 with
            | TXDot(Modname ((ext, int) as mn'),_) ->
                result := !result && (
		  let r = List.mem ((find_valuabilities cm mn').type_valuability) vubs
		  in Debug.print(function () -> "tmap modname " ^ ext ^ (if r then "true" else "false")); r
		      ) ;
                None
            | _ -> None )
        t
    in
    let aux vubs e =
      (aux_expr vubs e)
       &&
     (
      Debug.print (function () -> "tmap");
      let result = ref true
       in let (_:expr) = (* Debug.print (function () -> "tmap[2]: " ^ Pretty.print_expr (Dump.fresh_printer_state 1 0) e); *)
	 tmap_expr (aux_type vubs result) e
       in !result)
    in

    let rec evg = fun e ->
            (match e with
               | Dot _ | Tie _ | Id _ -> Some true
	       (* need to include these as the impl introduces an extra reduction rule to reduce them to values *)
	       | HashMvf(h, i, t) -> Some true
	       | HashTs(t1, e, t2) -> Some (is_value_like evg (false, false) [] e)
	       | HashHts(t1, e1, e2, t2) -> Some( is_value_like evg (false, false) [] e1 &&
		    is_value_like evg (false, false) [] e2)
               | _ -> None )

    in
    let rec evg_cfresh = fun e ->
      (match e with
      | Dot _ | Tie _ | Id _ | CFresh _ -> Some true (* check that cfreshes are guarded comes later *)
	       (* need to include these as the impl introduces an extra reduction rule to reduce them to values *)
      | HashMvf(h, i, t) -> Some true
      | HashTs(t1, e, t2) -> Some (is_value_like evg (false, false) [] e)
      | HashHts(t1, e1, e2, t2) -> Some( is_value_like evg (false, false) [] e1 &&
					is_value_like evg (false, false) [] e2)
      | _ -> None )

    in
    let ep = exprtoprim(e) in
    match vub with
      | Valuable ->
          (Debug.print (function () -> "isvaluelike"); is_value_like evg (false, false) [] ep)
          &&
           (Debug.print (function () -> "valuable"); aux [Valuable] e)
          &&
          ( Debug.print (function () -> "cfresh"); no_cfresh None ep)

      | CValuable ->
          (Debug.print' Opts.DBC_hashify (fun () -> "isvaluelike"); (is_value_like evg_cfresh (false, false) [] ep))
            && (Debug.print' Opts.DBC_hashify (fun () -> "cfresh"); no_guarded_cfresh e)
	    &&
          (Debug.print' Opts.DBC_hashify (fun () -> "aux"); (aux [Valuable;CValuable] e))

      | Nonvaluable ->
     	  no_cfresh None ep
;;


let (derive_valuabilities : compilation_map -> Ast.source_definition -> Ast.valuabilities)
   =
  fun cm ->  fun sourcedef ->
    Debug.print' Opts.DBC_hashify (function () -> "dv: " ^ Pretty.print_source_definition (Dump.fresh_printer_state_all ()) sourcedef);

    let check_deps vubs e =
      match e.desc with
	StrTyp(_) -> true
      |	StrVal(i,e) | StrValMulti(i,_,e) ->

	  let result = ref true in
	  let (_:expr) = early_emap_expr false
	      (fun _ e ->
		match e.desc with LocTie(m, x) | LocDot (m, x) ->
		  result := !result && (List.mem ((find_valuabilities cm m).term_valuability) vubs);
		  None
	      | _ -> None) None e
	  in !result
    in


    let check_val vub str_item =
      ( match str_item.desc with
          | StrVal (i,e) -> check_valuability_expr cm e vub
          | StrValMulti (i,pl,e) ->
	      true
          | StrTyp (t,k) -> true )
    in
    let basic_test_type vubs result t =
      early_tmap_typ
        ( fun t1 -> match t1 with
            | TXDot (Modname mn',t') ->
                result := !result && (List.mem ((find_valuabilities cm mn').type_valuability) vubs) ;
                None
            | _ -> None )
        t
    in
    let check_type_str vubs s =
      let result = ref true
      in let (_:structure) = tmap_structure (basic_test_type vubs result) s
      in !result
    in
    let check_type_sig vubs s =
      let result = ref true
      in let (_:signature) = tmap_signature (basic_test_type vubs result) s
      in !result
    in
    let check_type_withspec vubs ws =
      let aux (mn,etn,t) =
           (List.mem ((find_valuabilities cm mn).type_valuability) vubs)
        && ( let result = ref true
             in let junk = (basic_test_type vubs result t)
             in !result )
      in List.for_all aux ws
    in

    (* TODO: this isn't right for anything except hash according to the spec *)
    let check_type_likespec vubs ls =
      ( match ls with
        | LikeNone -> true
	| LikeMod mn ->
	    let vubs' = find_valuabilities cm mn in
	    if vubs'.type_valuability != Valuable then
	     false
	    else true
        | LikeStr s -> check_type_str vubs s )
    in
    let check_type_mod vubs mb =
         (check_type_sig vubs mb.mu_sign)
      && (check_type_str vubs mb.mu_str)
      && (check_type_withspec vubs mb.mu_withspec)
    in
     let check_type_imp vubs ib =
         (check_type_sig vubs ib.iu_sign)
      && (check_type_likespec vubs ib.iu_likespec)
     in


      match sourcedef with
        | Mod_user ((ext, int) as mn, mode, mn_body) ->
            ( match mode with
                | MHash ->
                    if (List.for_all (check_val Valuable) mn_body.mu_str.desc)
                      && (check_type_mod [Valuable] mn_body)
                    then
		      { term_valuability = Valuable; type_valuability = Valuable; }
                    else
		      (let vubss =
			List.map
			  (function e -> match e.desc with
			    (StrVal((ext, (iid, pn)), _) as e') -> pn ^ ": " ^ if (check_val Valuable e) then "valuable" else "nonvaluable"
			  |  (StrTyp((ext, (iid, pn)), _) as e') -> pn ^ ": " ^ if (check_val Valuable e) then "valuable" else "nonvaluable"
			  | StrValMulti _ -> raise (Never_happen "strvalmulti should have been desugared away")

			      ) mn_body.mu_str.desc

		      in
		      let vubstring = List.fold_right
			  (function a -> function b -> a ^ "," ^ b) vubss "" in

		      fatal (Bad_sourcedef_valuability (ext ^ ": " ^ vubstring)))

		| MBangHash ->
		    if (List.for_all (check_deps [Valuable]) mn_body.mu_str.desc
			  && (check_type_mod [Valuable]) mn_body)
		    then
		      { term_valuability = Valuable; type_valuability = Valuable; }
		    else    fatal (Bad_sourcedef_valuability (ext))

                | MCfresh ->
		    if (List.for_all (check_val CValuable) mn_body.mu_str.desc)
                     then if
		       (check_type_mod [Valuable;CValuable] mn_body)
                     then { term_valuability = CValuable; type_valuability = CValuable; }
                     else fatal (Bad_sourcedef_valuability (ext ^ ": mod not cfresh (check_type_mod)"))
		    else
		      (let vubss =
			List.map
			  (function e -> match e.desc with
			    (StrVal((ext, (iid, pn)), _) as e') -> pn ^ ": " ^ if (check_val Valuable e) then "valuable" else if (check_val CValuable e) then "cvaluable" else "nonvaluable"
			  | (StrTyp((ext, (iid, pn)), _) as e') -> pn ^ ": " ^ if (check_val Valuable e) then "valuable" else if (check_val CValuable e) then "cvaluable" else "nonvaluable"
			  | StrValMulti _ -> raise (Never_happen "strvalmulti should have been desugared away")
				) mn_body.mu_str.desc
		      in
		      let vubstring = List.fold_right
			  (function a -> function b -> a ^ "," ^ b) vubss "" in
		      fatal (Bad_sourcedef_valuability (ext ^ ": " ^ vubstring)))

		| MBangCfresh ->
		    if List.for_all (check_deps [Valuable; CValuable]) mn_body.mu_str.desc
			&&
		      List.for_all
			(fun s -> match s.desc with StrTyp _ -> true
			| StrVal(_, e) | StrValMulti (_, _, e) ->
			    no_guarded_cfresh  e )
			mn_body.mu_str.desc
			&&
		      (check_type_mod [Valuable; CValuable]) mn_body
		    then { term_valuability = CValuable; type_valuability = CValuable; }
		        else    fatal (Bad_sourcedef_valuability (ext))



                | MFresh ->
                    if (List.for_all (check_val Nonvaluable) mn_body.mu_str.desc)
                    then { term_valuability = Nonvaluable; type_valuability = Nonvaluable; }
                    else fatal (Bad_sourcedef_valuability (ext ^ ": mod not fresh")))

           | Import_user ((ext, int) as mn, mode, mn_body) ->
            ( match mode with
                | MHash | MBangHash->
                    if (check_type_likespec [Valuable] mn_body.iu_likespec)
                    then { term_valuability = Valuable; type_valuability = Valuable; }
                    else fatal (Bad_sourcedef_valuability (ext ^ ": (hash)check_type failed on import"))

                | MCfresh | MBangCfresh ->
                    if (check_type_likespec [Valuable;CValuable] mn_body.iu_likespec)
                    then { term_valuability = CValuable; type_valuability = CValuable; }
                    else fatal (Bad_sourcedef_valuability (ext ^ ": (cfresh) check_type failed on import"))

                | MFresh -> { term_valuability = Nonvaluable; type_valuability = Nonvaluable; } )

	   | Mod_alias ((ext, int) as mn, mn_body) ->
               ( match (try ModnameMap.find mn cm with Not_found -> assert false) with
               | Import_compile _ -> raise (Bad_sourcedef_valuability (ext ^ ": alias to import"))
               | _ ->
                   let v = find_valuabilities cm mn
                   in if not(v.term_valuability = Nonvaluable) && not(v.type_valuability=Nonvaluable)
                   then v
                   else fatal (Bad_sourcedef_valuability (ext ^ ": alias not nonvaluable")))

           | Mark_user _ -> raise (Never_happen "derive_valuabilities of Mark_user")
