(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * The Evaluator                                                  *
 *                                                                *
 * Version: $Id: eval.ml,v 1.731 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)



open Ast
open Misc
open Linkok
open Util
(* open Library *)
open Pretty
open Tysupp
open Evalsupp
open Basecon


(* TODO: sort out user internal names everywhere *)

(* -=-- ---------- ------------------------------------ ---------- --=- *)

(* argh, hack, but threading it through would be pretty horrible *)
let skipdefs_ref = ref 0

let printer_state = ref None
let printer_state_err = ref None
let printer_state_pretty = ref None

let ps = function () -> match !printer_state with
  None -> printer_state := Some (Dump.fresh_printer_state !Opts.dumptrace !skipdefs_ref !Opts.dumptex); Util.the(!printer_state)
| Some ps -> ps

let pserr = function () -> match !printer_state_err with
  None -> printer_state_err := Some (Dump.fresh_printer_state !Opts.dumptypefail !skipdefs_ref !Opts.dumptex); Util.the(!printer_state_err)
| Some ps -> ps

let pspretty = function () -> match !printer_state with
  None -> printer_state := Some (Dump.fresh_printer_state 1 !skipdefs_ref !Opts.dumptex); Util.the(!printer_state_pretty)
| Some ps -> ps

let print_string_really = Debug.print_string_really


(* Shouldn't happen if the type checker is doing its job. *)
exception Stuck of string
(* models exceptions; exceptions are modelled by exceptions for efficiency *)
(* not in smallstep any more *)
(*exception Threw_exception of configuration  (* with expr being the content of the exception *)*)


exception Terminating

exception Typecheck_of_configuration of configuration

exception Typecheck_on_unmarshal of  Ast.marshalled_body
exception Typecheck_on_marshal
exception Typecheck_on_get_URI of  string

exception Runtime_mismatch of string



exception Refocus_down_value
exception Refocus_not_implemented


exception Not_a_redex of string
exception Not_a_simple_redex

exception NotAPseudoRedex
exception Refocus_up_no_redex
(* exception Refocus_no_redex : Grah, we use Refocus_down_value even when we /mean/ no redex and *don't* have a value (raise case)... *)
exception Pull_one_context_CtxTop

exception Blocked   (* Needed for thunkify *)
exception ExnResult of prim_expr  (* used in prim_equal, thunkify and unthunkify *)

(* Thrown by reduce_expr when it can't do a reduction *)
exception Value
exception Acute_exception

exception Deadlock of configuration

let ps_empty_debug () = initial_printer_state (Some Econst.string_of_ident) (pm_debug ())


let ps_empty_marshal () = initial_printer_state (Some Econst.string_of_ident) (pm_marshal ())


(* -=-- ---------- ------------------------------------ ---------- --=- *)

exception Mark_not_found of string
(* cut_defs_by_mark mk defs splits defs into (defs1, defs2) where,
defs2 does not contain Mark mk; raise Mark_not_found mk if Mark mk
doesn't appear in defs. *)

(* cut_defs_by_mark' assumes that the definitions come in reversed
order; it produces its output in exactly the order needed by
cut_defs_by_mark *)

let rec cut_defs_by_mark' mk defs_after_mark = function
  | ((Mod_compile _) as def) :: defs
  | ((Import_compile _) as def) :: defs  ->
      cut_defs_by_mark' mk (def :: defs_after_mark) defs
  | ((Mark_compile mk'') as def) :: defs ->
             if mk'' = mk then (List.rev defs, defs_after_mark)
             else cut_defs_by_mark' mk (def :: defs_after_mark) defs
  | ((Mod_fresh _) as def) :: defs -> raise (Never_happen "mod_fresh in cut_defs_by_mark")
  | ((Import_fresh _) as def) :: defs -> raise (Never_happen "import_fresh in cut_defs_by_mark")

  | (Mod_imod _)::defs -> assert false
  | [] -> raise (Mark_not_found "")

let cut_defs_by_mark mk defs =
  cut_defs_by_mark' mk [] (List.rev defs)

(* -=-- ---------- ------------------------------------ ---------- --=- *)

let list_minus eq xs ys = List.filter (fun x -> not (mem_by eq x ys)) xs;;

let list_union eq xs ys = xs @ (list_minus eq ys xs);;

(* if the modname list argument contains no duplicates than the result
will also contain no duplicates *)

let rec prune : definitions -> modname list -> modname list * definitions
  = fun defs modnames ->
  match defs with
  | [] -> (modnames, [])
  | (Mod_compile (modname, mc_body)) as def :: defs' ->
      let (modnames'', defs'') = prune defs' modnames in
      if mem_by mn_eq modname modnames'' then begin
        let str = mc_body.mc_str in
        let str' = {loc = str.loc;
                     desc =
                     List.map (
                     function e -> match e.desc with
                       (StrVal(x, body)) -> {e with desc =
                                              StrVal(x, primtoexpr (flatten_all_closures [] (exprtoprim body)))}
                     | _ -> e
                           ) str.desc} in
        let modnames''' = list_union mn_eq
            (fmv_str str')
            (list_minus mn_eq modnames'' [modname]) in
        (modnames''', def :: defs'')
      end else (modnames'', defs'')
  | (Import_compile (modname, ic_body)) as def :: defs' ->
      let (modnames'', defs'') = prune defs' modnames in
      if mem_by mn_eq modname modnames'' then begin
        let modnames''' = list_union mn_eq (fmv_str ic_body.ic_likestr)
            (list_union mn_eq
               (fmv_mo ic_body.ic_mo)
               (list_minus mn_eq modnames'' [modname])) in
        (modnames''', def :: defs'')
      end else (modnames'', defs'')
  | (Mark_compile mk) as def :: defs' ->
      let (modnames'', defs'') = prune defs' modnames in
      (modnames'', def :: defs'')
  | (Mod_fresh _) :: defs' -> raise (Never_happen "mod_fresh in prune")
  | (Import_fresh _) :: defs' -> raise (Never_happen "import_fresh in prune")
  | (Mod_imod _) :: defs' -> assert false
;;

(* for sanity we assume the definitions are in reversed order in
make_imports_rev; note that we never mess with the internal names so
the fact that a reversed list doesn't follow the usual binding order
makes no difference. *)

let rec make_imports_rev defs modnames =
  match defs with
  | [] -> []
  | (Mod_compile (modname, mc_body)) as def :: defs' -> (
      try
        if mem_by mn_eq modname modnames then begin
          let h = mc_body.mc_hash in
          let sign0 = mc_body.mc_sign0 in
          let vubs = mc_body.mc_vubs in
          let sign1 = mc_body.mc_sign1 in
          let str = mc_body.mc_str in
          let sign_ats = limitdom sign0 in
          let likestr = filter_str_sign str sign_ats in
          let ic_body' = { ic_hash = h;
                           ic_sign0 = sign0;
                           ic_vubs = vubs;
                           ic_sign1 = sign1;
                           ic_vc = VCNameEqual (AVCHHash h);
                           ic_likestr = likestr;
                           ic_resolvespec = [Resolve_here_already];
                           ic_mo = None; } in
          let modnames' = list_minus mn_eq modnames [modname] in
          Import_compile (modname, ic_body') :: make_imports_rev defs' modnames'
        end
        else make_imports_rev defs' modnames
      with
        Filter_sign_missing_fields as e -> raise (Never_happen "Filter_sign_missing_fields in make_imports_rev") (* WAS e *)
            )
  | (Import_compile (modname, ic_body)) as def :: defs' ->
      if mem_by mn_eq modname modnames then begin
        let ic_body' = { ic_body with ic_mo = None; } in
        let modnames' = list_minus mn_eq modnames [modname] in
        Import_compile (modname, ic_body') :: make_imports_rev defs' modnames'
      end else make_imports_rev defs' modnames
  | (Mark_compile mk) as def :: defs' ->
      (* def :: *) make_imports_rev defs' modnames
  | (Mod_fresh _) :: defs' -> raise (Never_happen "mod_fresh in make_imports'")
  | (Import_fresh _) :: defs' -> raise (Never_happen "import_fresh in make_imports'")
  | (Mod_imod _) :: defs -> assert false


let make_imports defs modnames = List.rev (make_imports_rev (List.rev defs) modnames)

let cut_locs locs xenv =
  List.filter (fun (l,x) -> mem_by l_eq l locs) xenv

let rec print_list locs =
  match locs with [] -> ""
    | (e::es) -> string_of_int e ^ ", " ^ (print_list es)

let reachable_names
   : nameenv -> loctyplist -> store -> definitions -> modname list -> prim_value list -> typ -> nameenv
   = fun en storeenv store defs reachable_defs vs ty ->

     let rec reachable_names =
       fun (e0 : prim_expr) ->
       let r = ref [] in
       let f _ e = match e with
         NameValue(VHashHts(n1, t, s, nv)) ->
           r := ((n1::!r) @ (reachable_names (NameValue nv))); None
       | NameValue(n) -> r := ((abstract_name_of_name_value n)::(!r)); None
       | _ -> None
       in
       let e' = early_emap_prim_expr false
           f None e0 in
       !r
     in
     let reachable_names = List.flatten (List.map reachable_names vs) in
     let reachable_names' = reachable_names @ (names_of_modnames en (List.map (fun (ext,int) -> ext) reachable_defs)) in
     let en' = limitnameenv en reachable_names' in
     en'

(* NB: assumes type is properly closed, i.e., has no tyvars bound by the tenv *)
let marshal
  : Ast.smallstep_outer -> string -> Ast.prim_expr -> Ast.typ -> Ast.prim_expr
  = fun config mk expr ty ->
    try
      Debug.print' Opts.DBC_marshal (function () -> "Marshalling: " ^ print_prim_expr (pserr()) expr ^ " with respect to mark " ^ mk);
      let expr = flatten_all_closures [] expr in
      let store' = List.map (fun (l, e) -> (l, flatten_all_closures [] e)) config.scfg_store in

      (* defs1, defs2 *)
      let (defs_before_mark, defs_after_mark) = cut_defs_by_mark mk config.scfg_defs in
      let (reachable_defs, reachable_locs) = reachable_modules_locs config.scfg_defs store' (defs_after_mark, expr) in

      Debug.print' Opts.DBC_marshal (function () ->  ("\n *** defs_before_mark = " ^ string_of_int (List.length defs_before_mark)));
      Debug.print' Opts.DBC_marshal (function () ->  ("\n *** defs_after_mark = " ^ string_of_int (List.length defs_after_mark)));

      Debug.print' Opts.DBC_marshal (function () -> "Reachable defs:");
      List.iter (fun d -> (Debug.print' Opts.DBC_marshal (function () -> Pretty.print_modname (pserr()) d))) reachable_defs;

      let store' = cut_locs reachable_locs store' in
      let storeenv' = cut_locs reachable_locs config.scfg_senv in

      Debug.print' Opts.DBC_marshal (function () -> "Reachable store:" ^(print_list reachable_locs));

      let defs1 = List.filter (function def' -> match def' with
	                           Mod_compile(mn,_) | Import_compile(mn, _) -> List.mem mn reachable_defs
	                         | _ -> false)
                    defs_before_mark in

      let defs1' = List.map (function Mod_compile(mn,_) | Import_compile(mn, _) -> mn | _ -> assert false) defs1 in
      let imports_before_mark = make_imports defs_before_mark defs1' in

      Debug.print' Opts.DBC_marshal (function () ->  ("\n imports = " ^ string_of_int (List.length imports_before_mark )));

      let defs2 =  List.filter (function def' -> match def' with
	                            Mod_compile(mn,_) | Import_compile(mn, _) -> List.mem mn reachable_defs
	                          | _ -> false)
                     defs_after_mark in

      let defs' = imports_before_mark @ defs2 in
      Debug.print' Opts.DBC_marshal (function () ->  ("\n *** length defs' = " ^ string_of_int (List.length defs')));

      let en' = if !Opts.mttc
      then begin
	let en' = reachable_names config.scfg_nenv storeenv' store' defs' reachable_defs (expr::(List.map (fun (a, b) -> b) store')) ty
	in Debug.print' Opts.DBC_marshal (function () -> "nameenv " ^ print_nameenv (pserr()) en'); Some en' end

      else None in       (* FZ check the option *)
      let store'' = List.map (function (l,e) -> (l, flatten_all_closures [] e)) store' in
      let defs'' = List.map def_flatten_all_closures defs' in

      let marshalled_body = { mb_ne = en';
                              mb_defs = defs'';
                              mb_store = store'';
                              mb_storeenv = storeenv';
                              mb_expr = (primtoexpr expr);
                              mb_typ = ty;
                            }
      in
      let s = print_marshalled_value (ps_empty_marshal ()) (Marshalled_value (marshalled_body)) in
      Debug.print' Opts.DBC_marshal (fun () -> ("marshalling to:" ^ s));

      if (!Opts.mttc) then begin
        try
          Typecheck.tcheck_mb marshalled_body
        with e ->
          if (!Opts.terminate_on_rttc) then
            raise (Typecheck_on_marshal);
      end;

      C0 (B0(String s))
    with Mark_not_found _ -> (Raise (C0 (B0 MarshalFail)))


let mutex_removeBlocked = fun mutexes n n' -> (* mutex_info NameValueMap, mutex name, thread name *)
  let mutex = try NameValueMap.find n mutexes with Not_found -> raise(Never_happen ("mutex not found")) in
  (match mutex.mtxi_state with
     Some lmtx -> let lwait = AQueue.removeFirst (name_value_eq n') lmtx.lmtxi_waiting in
                  let lmtx' = { lmtx with lmtxi_waiting = lwait } in
                  let mtx' = { mutex with mtxi_state = Some lmtx' } in
                  NameValueMap.add n mtx' mutexes
   | None      -> raise (Never_happen ("thread blocked on empty mutex")))

let cvar_removeWaiting = fun cvars n n' ->
  let cvar = try NameValueMap.find n cvars with Not_found -> raise(Never_happen ("cvar not found")) in
  let lwait = AQueue.removeFirst (name_value_eq n') cvar.cvi_waiting in
  let cvar' = { cvar with cvi_waiting = lwait } in
  NameValueMap.add n cvar' cvars


(* unmarshalling*)

type unmarshal_failure = string
(*
  | Couldnt_parse_string of exn
  | Value_doesnt_type of exn
  | Expected_and_actual_mismatch of typ * typ
*)

exception Unmarshal_failure of unmarshal_failure

(* TODO: be selective and catch exceptions more carefully *)
(* NB: assumes type is properly closed, i.e., has no tyvars bound by the tenv *)
let unmarshal :
     Ast.smallstep_outer -> string -> typ -> Ast.smallstep_outer * Ast.prim_expr
  = fun config marshalled_string ty
  ->
    try
       Debug.print' Opts.DBC_marshal (fun () -> ("unmarshal of " ^ marshalled_string ));
      (* prerr_endline("unmarshal of " ^ marshalled_string); *)
      let initial_parser_state = Parsertypes.new_parser_state
          (Parsertypes.Marshalled_mode, if !Opts.lithash then [Parsertypes.Lithash] else [])
          (Some (Econst.ident_of_string,Econst.string_of_ident)) in
      Debug.print' Opts.DBC_marshal (fun () -> "unmarshal2");
      let Marshalled_value mb =
        try Parsetools.parse_marshalled_value_from_string marshalled_string initial_parser_state
        with
          (Myparsing.Runtime_mismatch e') -> raise (Myparsing.Runtime_mismatch ("unmarshal:" ^ e'))
        | e -> raise (Unmarshal_failure ("Couldn't parse string: " ^ Printexc.to_string e))
          (* Parsetools.Wrapped_parse_error(s) -> raise (Unmarshal_failure s) *)
          (* if we check for this we should check for Unknown_ident errors, and for lexer errors;
             all should never happen *)
      in
      Debug.print (fun () -> "unmarshal3: " ^ print_marshalled_value (ps()) (Marshalled_value mb));
      begin
        if !Opts.mttc then  (* FZ controlla opzioni *)
          begin
            if not (typ_eq mb.mb_typ ty) then
              raise (Unmarshal_failure ("Expected_and_actual_type_mismatch: " ^ print_typ (pserr()) mb.mb_typ ^ " and " ^ print_typ (pserr()) ty));
            try
              Typecheck.tcheck_mb mb
            with e -> (
              if (!Opts.terminate_on_rttc) then
                raise (Typecheck_on_unmarshal mb)
              else () )
          (* FZ TODO check that the En are consistent *)
          end
        else
          if not (typ_eq mb.mb_typ ty) then raise (Unmarshal_failure ("Expected_and_actual_type_mismatch"))
          else Debug.print' Opts.DBC_marshal (fun () -> "Types equal: " ^ print_typ (pserr()) mb.mb_typ ^ " and " ^ print_typ (pserr()) ty)
      end;

      Debug.print' Opts.DBC_reachable
	(fun () -> let store = mb.mb_store in let e = List.assoc 3 store in
	Ast.print_hack_all e);


      Debug.print' Opts.DBC_marshal (fun () -> "unmarshal[45]");

      (* FZ says: now we use fresh_location to construct sigma, so we do not *)
      (*          need locs_to_avoid anymore                                 *)
(*      let locs_to_avoid = List.map fst config.scfg_store in *)
      Debug.print' Opts.DBC_marshal (fun () -> "unmarshal6");
(*    let mb' = freshen_mb mb locs_to_avoid in *)
      let mb' = freshen_mb mb in
      Debug.print' Opts.DBC_marshal (fun () -> "unmarshal7");

        let nenv =
          if (!Opts.mttc) then
            ( match mb.mb_ne with
                | None -> raise (Unmarshal_failure "no name-env informations in marshalled value")
                | Some mne -> merge_nameenv (config.scfg_nenv,mne)) (* TODO: catch error and raise Typecheck_on_unmarshal *)
          else config.scfg_nenv
        in
          ({config with
              scfg_nenv  = nenv;
              scfg_senv  = config.scfg_senv @  mb'.mb_storeenv;
              scfg_store = config.scfg_store @ mb'.mb_store;
              scfg_defs  = config.scfg_defs @ mb'.mb_defs; }, exprtoprim mb'.mb_expr)

    with Unmarshal_failure s ->
   (* raise (TerminateOnUnmarshalFailure (s, (st,Unmarshal(C0(String marshalled_string),ty)))) *)
      Debug.print' Opts.DBC_marshal (fun () -> ("Unmarshal failed: " ^ s));
      (config, Raise (C1 (B1 UnmarshalFail,C0 (B0(String s)))))


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*                           matching                                   *)

let con0_equals (c1,c2) =
  match (c1,c2) with
    (Nil _, Nil _)
  | (NONE _, NONE _) -> true
  | (B0 b1, B0 b2) -> b1 = b2
  | (_,_) -> false

let con1_equals (c1, c2) =
  match (c1, c2) with
    (Inj(i1, _), Inj(i2, _)) -> i1 = i2
  | (SOME, SOME) -> true
  | (TieCon, TieCon) -> true
  | (Node,Node) -> true
  | (B1 b1, B1 b2)   -> b1 = b2
  | (_,_) -> false

(* Try to match the pattern p and the expression e, return association list
   of variable bindings.  The parser enforces that each variable only occurs
   once in the pattern, so we don't need to worry about that here.
   *)
exception Doesnt_match

(* matching as in paper, with eqs taken into account *)
let rec do_match_eqs eqs p e acc =
  match p with
    PWild(_) -> acc
  | PVar(i,t)    -> if !Opts.really_hack_optimise
        then IIdentMap.add i e acc
        else
        IIdentMap.add i (Col(e, eqs, t)) acc  (* env bindings have outer Col *)
  | PC0(c1)      ->
      (match e with
        (C0 c2) -> if con0_equals(c1,c2) then acc else raise Doesnt_match
      | _       -> raise Doesnt_match)
  | PC1(c,p1)    ->
      (match e with
        C1(c1,e1) -> if con1_equals(c, c1) then do_match_eqs eqs p1 e1 acc else raise Doesnt_match
      | _         -> raise Doesnt_match)
  | PCons(p1,p2) -> (match e with
                       Cons(e1,e2) -> do_match_eqs eqs p1 e1 (do_match_eqs eqs p2 e2 acc)
                     | _ -> raise Doesnt_match)
  | PTup(ps)     -> (match e with
                       Tup(es) -> do_matches_eqs eqs ps es acc
                     | _ -> raise Doesnt_match)
  | PTyped(p,_)  -> do_match_eqs eqs p e acc

and do_matches_eqs eqs ps es acc =
  match (ps,es) with
    (p::ps, e::es) -> do_match_eqs eqs p e (do_matches_eqs eqs ps es acc)
  | ([],[])        -> acc
  | _ -> raise Doesnt_match

(* cases is a (<<p_i>>e_i)list.
   "matching exp cases" finds the least i such that p_i matches exp, and
   returns the right substitution together with e_i .
   If nothing maches, Raise(MatchFail) is returned. *)

let rec matching_eqs eqs exp cases =
  match cases with
    (p,e)::cs -> (try ((do_match_eqs eqs p exp IIdentMap.empty), e)
                  with Doesnt_match -> matching_eqs eqs exp cs)
  | [] -> raise Doesnt_match
let single_matching_eqs eqs v pat =
  (* print_string ("single_matching: " ^ print_pat ps_empty pat ^ " against " ^ print_prim_expr ps_empty v^"\n"); *)
  do_match_eqs eqs pat v IIdentMap.empty


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*                          resolution                                  *)

(* Assumes all these definitions are freshened up already.  *)

let rec find_sign : definitions -> modname -> signature
    = fun dfs0 mn ->
      match dfs0 with
        [] -> raise (Never_happen "find_sign: empty defintions")
      | df::dfs0a -> match df with
        | Mod_compile(mn',mcbody) ->
            if not(mn_eq mn mn') then
              find_sign dfs0a mn
            else
              mcbody.mc_sign1
        | Import_compile(mn',icbody) ->
            if not(mn_eq mn mn') then
              find_sign dfs0a mn
          else
              icbody.ic_sign1
        | Mark_compile(mk) -> find_sign dfs0a mn

	| _ -> raise (Never_happen "uncompiled module in defs")
(*  FZ

resolve_definition_pri implements ``module field instantiation -
module case, via import sequence''.


resolve_definition_pri recurses through the reversed list of
definitions, dfs_rev, trying to construct the chain of definitions
definition_1,...,definition_n.

dfs_rev contains the list definitions, reversed.

mn corresponds to MMmm.

eid corresponds to xx.

n is used to record how many matching cimport have been found (remark,
what is called definition_n in the specification is caractherised by
n=0).  When going through the firts cimport (n=0), the set of equation
eqst is updated with eqs_of_sign_str(h_n,Sig_0n,Str_n).

mn_chain records the name of the module currently searched.
Initially, it coincides with MMmm, but after going through a cimport
that defines the name mn_chain, then mn_chain is updated with the name
specified in the resolvespec.

mn_n records the name of the module defined by the first cimport (that
is, the cimport defined by definition_n), if any.  This is used to
construct the substitution to be applied to v'.

eqs0 is the initial set of equations, used by the optimisation maybe_col

eqst records the set of equations that must be added to eqs0.  It is
updated after foing through the first matching cimport.

*)

let rec resolve_definition_pri : eqs -> eqs -> definitions -> smallstep_outer -> modname -> external_ident -> modname -> modname -> int -> typ -> smallstep_outer * prim_expr
    = fun eqs0 eqst dfs_rev conf mn eid mn_n mn_chain n t ->
      match dfs_rev with
        [] -> raise (Never_happen "resolve_definition_pri: empty definitions")
      | df::dfs_rev_t -> match df with

        | Mod_compile(mn',mcbody) ->
            Debug.print(function () -> "mod compile");
            if not(mn_eq mn_chain mn') then
              resolve_definition_pri
                eqs0 eqst dfs_rev_t conf mn eid mn_n mn_chain n t
            else
              begin
                 Debug.print(function () -> "matched");
              let v =
                match lookup_structure_eid mcbody.mc_str eid with
                  Some(v)->v | None -> raise (Never_happen "resolve_definition_pri : Mod_compile : lookup_structure_eid returned None") in

              let nmn = if n=0 then mn else mn_n in
              let rec mk_subst = function str ->
                match str with
                | [] -> IIdentMap.empty
                | (s::str') -> ( match s.desc with
                  StrVal((eid,iid),_)-> IIdentMap.add iid (Dot(nmn,eid)) (mk_subst str')
                | _ ->  mk_subst str') in
              let subst = mk_subst mcbody.mc_str.desc in
              let eqs1 = List.append eqst (List.append mcbody.mc_eqs
                  (Typecheck.eqs_of_sign_str
                     mcbody.mc_hash mcbody.mc_sign0 mcbody.mc_str)) in
(*              let v' = esub_expr (subst, ITypnameMap.empty) eqs1 (exprtoprim v) in  (* TODO: KW: Fix type environment *)
                 *)

   let v'= exprtoprim v in
(*
                 Debug.print
                 (function () ->
                 let s_list = (iidmap_to_list subst) in
                 "subst of length : " ^ string_of_int (List.length s_list) ^
                 "\nsubst is " ^
                 (let strings =
                 List.map (function ((i',pname), s) ->
                 pname
                 ^ ":" ^
                 Pretty.print_prim_expr (ps()) s
                 ) s_list
                 in
                 List.fold_right (function s1 -> function s2 -> s1 ^ "," ^ s2) strings "" )
                 ^
                 "\nv before subst: " ^
                 Pretty.print_prim_expr (ps() ) v
                 ^ "\nv after subst: " ^
                 Pretty.print_prim_expr (ps() ) v'
                 );
              *)

              let typ =
                if (n=0) then
                  match lookup_signature_eid mcbody.mc_sign1 eid with
                    Some(typ)->typ | None -> raise (Never_happen "resolve_definition_pri : Mod_compile : lookup_signature_eid returned None")
                else t in
             let v'' = maybe_col eqs0 (v',eqs1,typ) in
             (conf,v'')
              end

        | Mark_compile _ -> resolve_definition_pri
              eqs0 eqst dfs_rev_t conf mn eid mn_n mn_chain n t

        | Import_compile(mn',icbody) ->
            if not(mn_eq mn_chain mn') then
              resolve_definition_pri
                eqs0 eqst dfs_rev_t conf mn eid mn_n mn_chain n t
            else
              begin
              Debug.print' Opts.DBC_linkok (function () -> "import:matched");
              match icbody.ic_mo with
                None ->
                  Debug.print' Opts.DBC_linkok (function () -> "import:matched:resolve");
                  (conf,Resolve(Dot(mn,eid), mn', icbody.ic_resolvespec))
              | Some(mn'') ->
                  if (n=0)
                  then
                    let typ =
                      match lookup_signature_eid icbody.ic_sign1 eid with
                        Some(typ)->typ
                      | None -> raise (Never_happen "resolve_definition_pri : Import_compile : lookup_signature_eid returned None")
                    in
                    let eqs' = List.append eqst
                        (Typecheck.eqs_of_sign_str
                           icbody.ic_hash icbody.ic_sign0 icbody.ic_likestr) in
                    resolve_definition_pri
                      eqs0 eqs' dfs_rev_t conf mn eid mn' mn'' (n+1) typ
                  else
                    resolve_definition_pri
                      eqs0 eqst dfs_rev_t conf mn eid mn_n mn'' (n+1) t
              end
	| _ -> raise (Never_happen ("Resolve: uncompiled def in defs"))

let resolve_definition : eqs -> smallstep_outer -> modname -> external_ident -> smallstep_outer * prim_expr
    = fun eqs conf mn eid ->
      resolve_definition_pri eqs [] (List.rev conf.scfg_defs) conf mn eid mn mn 0 (TTyCon0 TVoid)


(* split dfs0 into rev(dfs1_rev);;import mn...;;dfs2  *)

let rec find_import : definitions -> definitions -> modname -> (definitions * (modname*import_compile_body) * definitions)
    = fun dfs0 dfs1_rev mn ->
      match dfs0 with
        [] -> raise (Never_happen "find_import: empty definitions")
      | df::dfs0a -> match df with
        | Mod_compile(mn',mcbody) ->
            if not(mn_eq mn mn') then
              find_import dfs0a (df::dfs1_rev) mn
          else
              raise (Never_happen "find_import : Mod_compile : mn_eq mn mn'") (* until concurrency*)
        | Import_compile(mn',icbody) ->
            if not(mn_eq mn mn') then
              find_import dfs0a (df::dfs1_rev) mn
          else
              (dfs1_rev,(mn',icbody),dfs0a)
        | Mark_compile(mk) -> find_import dfs0a (df::dfs1_rev) mn
	| _ -> raise (Never_happen ("Find import: uncompiled module in defs"))


(* OLD COMMENT:
For resolve_URI, P going to punt on actually doing http right
now. Browsing the hump, there's a "big brother" tool by Francois
Poitier which probably has handy code, or one could use wget and
indirect via the filesystem. Also going to punt on doing GetURI
accesses in a separate thread - unnecessary in our current
nonconcurrent language. *)


exception GetURIfailure;;

(* getURI gets a file, lexs, parses, and typechecks it, returning the
definitions contained therein.  It raises getURIfailure if it doesn't
work for any reason.*)

(* TODO: this does not allow Econsts, whereas the semantics does. which? *)

(* P assuming the parser returns a suitably fresh defs... *)
(* F CHECK hum, unsure about the "suitably fresh" *)
let getURI : string -> (nameenv * definitions) option
    = fun s ->
      try
	let (ne,defs,eo),ps =  (* FZ this parser state is odd *)
          try
            let uri = Uri.parse s in
            let initial_parser_state = Parsertypes.new_parser_state
                (Parsertypes.Program_mode
                   (Parsertypes.Compiled, Parsertypes.Econsts_allowed,
                    Parsertypes.InUserCode),
                 if !Opts.lithash then [Parsertypes.Lithash] else [])
                (Some (Econst.ident_of_string,Econst.string_of_ident))
            in Parsetools.parse_compiled_unit_from_uri uri initial_parser_state
        (* with Sys_error(_) -> raise GetURIfailure in *)
          with
            Myparsing.Runtime_mismatch e -> raise (Myparsing.Runtime_mismatch ("resolve: " ^ e))
          | e -> (print_endline("failed: " ^ Printexc.to_string e)); raise GetURIfailure in
	match eo with
          Some _ -> raise (Never_happen "getURI: found expression")
	| None -> if (!Opts.mttc) then
           (try
             let _ = Typecheck.tcheck_cdefs_eo emptynametypeenv (defs,None) None in ()
           with e ->
             if (!Opts.terminate_on_rttc) then
               raise (Typecheck_on_get_URI s););
            Some (ne, defs)
      with GetURIfailure -> None


let resolve_here_already : smallstep_outer -> prim_expr ->
  modname -> resolvespec -> smallstep_outer * prim_expr
      = fun conf e mn rs ->
        let (ext, int) = mn in
        let (dfs1_rev, (mn_import,body_import), dfs2) =
             find_import conf.scfg_defs [] mn in
        match findfirst   (* FZ ugly : rewrite *)  (* TODO: why is this code duplicated in resolve_URI? *)
            (fun x ->
              if
                try linkok conf.scfg_nenv x (Import_compile(mn_import,body_import)); true
                with Linkok.Linkok_not s -> (Debug.print' Opts.DBC_linkok (fun () -> "Linkok failed: " ^ s); false)
              then
                match x with Import_compile(mn',_) -> Some(mn')
                | Mod_compile(mn', _) -> Some(mn')
                | _ -> Debug.print (fun () -> "x is neither cimport nor cmodule");
                    raise (Never_happen "resolve_here_already: x is neither cimport nor cmodule")
              else None)
            dfs1_rev with
          None -> (conf,Resolve(e,mn,rs))
        | Some(mn') ->
            let df_import' = Import_compile(mn_import,{body_import with ic_mo=Some(mn') }) in
            (
            {conf with scfg_defs =(List.rev_append dfs1_rev (df_import'::dfs2))}
              ,e)


(* could optimize above and in the semantics instead of leaving e to
   be resolved again, but would mean some code duplication *)

let resolve_URI : smallstep_outer -> prim_expr ->
  modname -> resolvespec -> (nameenv * definitions) -> smallstep_outer * prim_expr
    = fun conf e mn rs (ne', dfs_uri) ->
      try (
        let dfs_uri_rev = List.rev dfs_uri in
        let (dfs1_rev,((mn_import,body_import) as df_import),dfs2) =
          find_import conf.scfg_defs [] mn  in (
        match body_import.ic_mo with
          Some(_) -> raise (Never_happen "resolve_URI : body_import.ic_mo exists") (* until concurrency*)
        | None -> (
            match findfirst    (* FZ ugly : rewrite *)  (* TODO: why is this code duplicated in resolve_here_already? *)
                (fun x ->
                  if try linkok conf.scfg_nenv x (Import_compile(mn_import,body_import)); true
                  with Linkok.Linkok_not s -> (Debug.print' Opts.DBC_linkok (fun () -> "Linkok failed: " ^ s); false )
                  then
                    match x with
                      Import_compile(mn',_) -> Some(mn')
                    | Mod_compile(mn', _) -> Some(mn')
                    | _ -> Debug.print (fun () -> "x is neither cimport nor cmodule"); raise (Never_happen "resolve_URI : x is neither cimport nor cmodule")
                  else None)
                dfs_uri_rev with
              None -> raise GetURIfailure
            | Some(mn') ->
                let df_import' = Import_compile(mn_import,{body_import with ic_mo=Some(mn') }) in
                 let ne =
                  try  (* TODO : spec says this is superfluous without rttc, add flag? *)
                    merge_nameenv (conf.scfg_nenv, ne')
                  with
                    e -> raise e (* TODO: sensible error (flag?) *)
				   in
                ( {conf with scfg_nenv = ne; scfg_defs=(List.rev_append dfs_uri_rev (List.rev_append dfs1_rev (df_import'::dfs2)))},e)

          )))
      with
        GetURIfailure -> (conf, Resolve(e,mn,rs))

(* -=-- ---------- ------------------------------------ ---------- --=- *)

let rec update_store l new_val s =
  match s with
    (l1,v1)::ss -> if l=l1 then (l,new_val)::ss else (l1,v1)::(update_store l new_val ss)
  | []          -> raise (Stuck "update_store")

let rec prim_append : prim_expr -> prim_expr -> prim_expr
  = fun xs ys
 -> match xs with
      Cons(x,xs)  -> Cons(x,prim_append xs ys)
    | C0(Nil(ty)) -> ys
    | _           -> raise (Stuck "prim_append")

(* expects *values* as arguments *)
let prim_equal : tysubst -> (bool * bool) -> prim_expr -> prim_expr -> prim_expr
  = fun tenv closures v1 v2
 -> if not (isvalue closures [] v1 && isvalue closures [] v2) then
      raise (Never_happen "prim_equal: nonvalue")  (* not checked in real impl *)
    else begin
    let go_c0 c01 c02 =
      match (c01,c02) with
        (Nil(ty1) ,Nil(ty2) )
      | (NONE(ty1),NONE(ty2)) ->
          if not (typ_eq_with tenv ty1 ty2) then
            raise (Never_happen "prim_equal: type mismatch")  (* not checked in real impl *)
          else
            true
      | (_,_) -> con0_equals(c01,c02)
    in
    let go_c1 c11 c12 =
      match (c11,c12) with
        (Inj(i1,ty1),Inj(i2,ty2)) ->
          if i1 = i2 then
            if not (typ_eq_with tenv ty1 ty2) then
              raise (Never_happen "prim_equal: type mismatch")  (* not checked in real impl *)
            else
              true
          else
            false
      | (_,_) -> con1_equals(c11,c12)
    in
    let rec go_nv nv1 nv2 =
      if name_value_eq nv1 nv2 then
        if match (nv1, nv2) with  (* not checked in real impl *)
           | (VHashMvf(abs1, h1, eid1, t1), VHashMvf(abs2, h2, eid2, t2)) ->
               h_eq h1 h2 && eid_eq eid1 eid2 && typ_eq t1 t2
           | (VHashTs(abs1, t1, s1), VHashTs(abs2, t2, s2)) ->
               typ_eq t1 t2 && String.compare s1 s2 = 0
           | (VHashHts(abs1, t1, s1, nv1'), VHashHts(abs2, t2, s2, nv2')) ->
               typ_eq t1 t2 && String.compare s1 s2 = 0 && go_nv nv1' nv2'
           | (VHashName(abs1, t1), VHashName(abs2, t2)) ->
               typ_eq t1 t2
           | _ -> false then
          true
        else
          raise (Never_happen "prim_equal: name_value lit same but value different")
      else false
    in
    let rec go v1 v2 =
      match (v1, v2) with
      | (Col(v1',_,_),_) -> go v1' v2  (* ignore brackets; invisible at runtime *)
      | (_,Col(v2',_,_)) -> go v1 v2'  (* ignore brackets; invisible at runtime *)
      | (C0(c01)      ,C0(c02)      ) -> go_c0 c01 c02
      | (C1(c11,e1)   ,C1(c12,e2)   ) -> go_c1 c11 c12 && go e1 e2
      | (Cons(e11,e21),Cons(e12,e22)) -> go e11 e12 && go e21 e22
      | (Tup(es1)     ,Tup(es2)     ) -> (try
                                            List.for_all2 go es1 es2
                                          with
                                            Invalid_argument _ -> false)  (* lengths differ *)
      | (Loc(l1)      ,Loc(l2)      ) -> l_eq l1 l2
      | (Clos _       ,Clos _       ) -> raise (ExnResult (C1 (B1 InvalidArgument,C0 (B0 (String "equal: functional value")))))
      | (Fn _         ,Fn _         ) -> raise (ExnResult (C1 (B1 InvalidArgument,C0 (B0 (String "equal: functional value")))))
      | (TAbs _       ,TAbs _       ) -> raise (ExnResult (C1 (B1 InvalidArgument,C0 (B0 (String "equal: type-functional value")))))
      | (TClos _      ,TClos _      ) -> raise (ExnResult (C1 (B1 InvalidArgument,C0 (B0 (String "equal: type-functional value")))))
      | (App(v11,v21) ,App(v12,v22) ) -> go v11 v12 && go v21 v22
      | (Id(iid1)     ,Id(iid2)     ) -> id_eq iid1 iid2
      | (Pack(t11, v1, t21), Pack(t12, v2, t22))
                                      -> raise (ExnResult (C1 (B1 InvalidArgument,C0 (B0 (String "equal: existential package")))))
      | (NameValue(nv1), NameValue(nv2)) -> go_nv nv1 nv2

      | (_            ,_            ) -> if v1 = v2 then  (* should have caught all same-constructor cases above *)
                                           raise (Never_happen "prim_equal: case missed")
                                         else
                                           false
    in
    try
      C0 (B0 (Bool(go v1 v2)))
    with
      ExnResult e -> Raise(e)
    end


(* P is no longer suspicious of the Ref and Assign cases below, eg for
   let (x:(unit->int) ref) = ref %[unit->int] (function ()->3) in
   let (f:int->unit) = function (y:int) -> x:= %[unit->int] (function ()->y) in
   (!f 4) ()

   NB: this is _not_ lambda-r or lambda-d we have here - values _are_
   all (expr var) closed; wherever necessary there is a _closure_ grabbing this
   environment...

   NB2: K says: ah, but they're not necessarily *type*-closed, so we
   have to be careful again.
*)

(*---- some mutex/cvar helpers ----*)

let reduce_op_unlock : name_value -> threaded_smallstep_inner -> smallstep_outer -> smallstep_outer * prim_expr option
    = fun n' tinf conf ->
      try
        let mutex = NameValueMap.find n' conf.scfg_mutexes in
        match mutex.mtxi_state with
        | None -> (conf, Some(C0 (B0 Unit)))
        | Some lmtxi when AQueue.empty lmtxi.lmtxi_waiting ->
            let mutex' = { mutex with mtxi_state = None } in
            let conf' = { conf with scfg_mutexes = NameValueMap.add n' mutex' conf.scfg_mutexes } in
            (conf', Some(C0 (B0 Unit)))
        | Some lmtxi ->
            let (t_name,lmtxiw) = AQueue.take lmtxi.lmtxi_waiting in
            let mutex' = { mutex with mtxi_state = Some { lmtxi with lmtxi_waiting=lmtxiw } } in
            let thread = { (try NameValueMap.find t_name conf.scfg_threads with Not_found -> raise (Never_happen "reduce_op_unlock: thread not found") ) with
                           tsc_state = TsRunnable;
                           tsc_expr = C0 (B0 Unit) } in
            let conf' = { conf with
                          scfg_threads = NameValueMap.add t_name thread conf.scfg_threads;
                          scfg_mutexes = NameValueMap.add n' mutex' conf.scfg_mutexes;
                          scfg_runnable = AQueue.add t_name conf.scfg_runnable
                        } in
           (conf', Some(C0 (B0 Unit)))
        with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentMutex))))

let restart_one t_name conf =
  let thread = (try NameValueMap.find t_name conf.scfg_threads with Not_found -> raise(Never_happen "restart_one: thread not found")) in
  match thread.tsc_expr with
  | Op(OEOp(Waiting), [_;mn]) ->
      let thread' = { thread with tsc_expr = Op(OEOp(Lock), [mn]);
                                  tsc_state = TsRunnable } in
      { conf with scfg_runnable = AQueue.untake (t_name, conf.scfg_runnable);
                  scfg_threads = NameValueMap.add t_name thread' conf.scfg_threads }
  | _ -> raise (Never_happen "thread redex was not Waiting cn cm !!")

(* TODO: Should we just take/return a collection type here? *)
(* TODO: Signal whether we're returning Value or Exception? *)
(* --- currently no point with smallstep as it is. Think about optimising smallstep... ? *)

let is_name_fresh n' conf =
  not (NameValueMap.mem n' conf.scfg_mutexes) &&
  not (NameValueMap.mem n' conf.scfg_cvars)   &&
  not (NameValueMap.mem n' conf.scfg_threads)

(* given a possibly-bracketed name value expression, return the name
   value and its apparent type *)
(* this is necessary to correctly handle the situation where a client
   creates a name at an abstract type, and hands it to the library
   which knows it is implemented concretely as a mutex name or
   similar. *)
let get_name_value : string -> prim_expr -> (name_value * typ)
    = fun str v ->
      let (nv, ty) = (match v with
      | NameValue(nv) -> (nv, TTyCon1 (TName,type_of_name_value nv))
      | Col(v,_,ty) ->
          let rec go = function
            | NameValue(nv) -> nv
            | Col(v,_,_) -> go v
            | _ -> raise (Stuck (str ^ ": expected a name value argument"))
          in
          (go v, ty)
      | _ -> raise (Stuck (str ^ ": expected a name value argument")))
      in
      Debug.print' Opts.DBC_namevalues (function () ->
        let ps = pserr () in
        "get_name_value "^str^": "^print_prim_expr ps v^"\n => "^print_typ ps ty);
      (nv,ty)


(* ---- thunkify ------------- *)

(* This is tail-recursive.
   thname  current thread name (needed to raise ThunkifySelf)
   config0 initial configuration
   config  current configuration (= config0 - {thunkified objects})
   ps      accumulate the patterns (in reverse order)
   es      accumulate thks (in reverse order)
   tks     linked list of objects to thunkify (as an AST prim_expr value)

   Note: the expression returned contains syntactic sugar
 *)

let rec thunkify
  : name_value -> smallstep_outer -> smallstep_outer -> pat list -> prim_expr list -> prim_expr ->
    smallstep_outer * prim_expr
  = fun thname config0 config ps es tks ->
  match tks with
    C0 (Nil _) ->
      (* essentially just "function ps -> unthunkify es", but (1) ps/es need
         reversing and conversion into acute lists from ocaml lists, and
         (2) result must be desugared *)
      let es' = List.fold_left (fun e x -> Cons (x, e)) (C0 (Nil (TTyCon0 TThunklet))) es in
      let ps' = List.fold_left (fun e x ->PCons (x, e)) (PC0(Nil (TTyCon0 TThunkkey))) ps in
      let x = fresh_internal_ident "tks" in
      let t = TTyCon1 (TList, TTyCon0 TThunkkey) in
      config, Fn (no_loc [PVar(x, t), Match (Id x, no_loc [ps', Op(OEOp Unthunkify, [es'])])])
  | Cons (tk, tks0) ->
    (match tk with
      C1(B1 Mutex, v) ->
        let (n,tn) = get_name_value "thunkify mutex" v in
        if not (typ_eq tn (TTyCon1 (TName, TTyCon0 TMutex))) then  (* not tested in a real implementation *)
          raise (Never_happen "thunkify Mutex: ill-typed name");
      (try
         let mtxi  = NameValueMap.find   n config.scfg_mutexes in
         let mtxs' = NameValueMap.remove n config.scfg_mutexes in
         let nn = fresh_internal_ident "n" in
         thunkify thname config0 {config with scfg_mutexes = mtxs'}
         (* Mutex n, ThunkedMutex (n, state)  *)
           (PC1(B1 Mutex, PVar(nn, TTyCon1 (TName, TTyCon0 TMutex)))::ps)
           (C1(B1 ThunkedMutex, Tup [Id nn; C0 (B0 (Bool (mtxi.mtxi_state != None)))])::es) tks0
       with
         Not_found -> raise (ExnResult(Raise(C0(B0 NonExistentMutex)))))
    | C1(B1 CVar, v) ->
        let (n,tn) = get_name_value "thunkify cvar" v in
        if not (typ_eq tn (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
          raise (Never_happen "thunkify CVar: ill-typed name");
      (try
         let cvi  = NameValueMap.find   n config.scfg_cvars in
         let cvs' = NameValueMap.remove n config.scfg_cvars in
         let nn = fresh_internal_ident "n" in
         thunkify thname config0 {config with scfg_cvars = cvs' }
         (* CVar n, ThunkedCVar (n, state)  *)
         ((PC1 (B1 CVar, PVar(nn, TTyCon1 (TName, TTyCon0 TCVar))))::ps )
         (C1(B1 ThunkedCVar, Tup [Id nn; C0 (B0 (Bool (not (AQueue.empty cvi.cvi_waiting))))])::es) tks0
       with
         Not_found -> raise (ExnResult(Raise(C0(B0 NonExistentCVar)))))
    | C1(B1 Thread, Tup[v; C0 (B0 thunkifymode)]) ->
        let (n,tn) = get_name_value "thunkify thread" v in
        if not (typ_eq tn (TTyCon1(TName,TTyCon0 TThread))) then  (* not tested in a real implementation *)
          raise (Never_happen "thunkify Thread: ill-typed name");
      (try
         if name_value_eq n thname then raise (ExnResult (Raise (C0 (B0 ThunkifySelf))));
         let tsc   = NameValueMap.find   n config.scfg_threads in
         let tscs' = NameValueMap.remove n config.scfg_threads in
         let nn  = fresh_internal_ident "n" in
         let nth = fresh_internal_ident "thunkifymode" in
         let unitpat = PVar(fresh_internal_ident "u", TTyCon0(TUnit)) in
         match tsc.tsc_def, tsc.tsc_state with
           None, TsRunnable  ->  (* NB: we can't be here if we're in a fast call, since we
                                    can only reach this point when the Acute scheduler is
                                    between reductions.   So no need to consider that case. *)
           let (_, esub,eqs,ti) = pull_context true tsc in
           let config' =  {config with
               scfg_runnable = AQueue.removeFirst (name_value_eq tsc.tsc_name) config.scfg_runnable;
               scfg_threads = tscs' }
           in
           let e' = flatten_all_closures eqs ti.ti_expr in
           thunkify thname config0 config'
              (* Thread (n, thunkifymode), ThunkedThread (n, fn () -> ti.ti_exp) *)
              ((PC1(B1 Thread, PTup[PVar(nn, TTyCon1 (TName, TTyCon0 TThread)); PVar(nth, TTyCon0 TThunkifymode)]))::ps)
              ((C1(B1 ThunkedThread, Tup [Id nn; Fn (no_loc [unitpat,e'])]))::es) tks0
         | None, TsSlowcall when thunkifymode = Interrupting ->
           let (_, esub,eqs,ti) = pull_context true { tsc with tsc_expr = Raise (C0 (B0 ThunkifyEINTR)) } in
           thunkify thname config0 {config with
               scfg_slowcall = NameValueMap.remove n config.scfg_slowcall;
               scfg_threads = tscs' }
              (* Thread (n, thunkifymode), ThunkedThread (n, fn () -> ti.ti_exp) *)
              ((PC1(B1 Thread, PTup[PVar(nn, TTyCon1 (TName, TTyCon0 TThread)); PVar(nth, TTyCon0 TThunkifymode)]))::ps)
              ((C1(B1 ThunkedThread, Tup [Id nn; Fn (no_loc [unitpat, flatten_all_closures eqs ti.ti_expr])]))::es) tks0
         | None, TsMutexBlocked n' when thunkifymode = Interrupting ->
           let (_, esub,eqs,ti) = pull_context true { tsc with tsc_expr = Raise (C0 (B0 ThunkifyEINTR)) } in
           thunkify thname config0 {config with
               scfg_mutexes = mutex_removeBlocked config.scfg_mutexes n' n;
               scfg_threads = tscs' }
              (* Thread (n, thunkifymode), ThunkedThread (n, fn () -> ti.ti_exp) *)
              ((PC1(B1 Thread, PTup[PVar(nn, TTyCon1 (TName, TTyCon0 TThread)); PVar(nth, TTyCon0 TThunkifymode)]))::ps)
              ((C1(B1 ThunkedThread, Tup [Id nn; Fn (no_loc [unitpat, flatten_all_closures eqs ti.ti_expr])]))::es) tks0
         | None, TsCVarWaiting n' when thunkifymode = Interrupting ->
           let (_, esub,eqs,ti) = pull_context true { tsc with tsc_expr = Raise (C0 (B0 ThunkifyEINTR)) } in
           thunkify thname config0 {config with
               scfg_cvars = cvar_removeWaiting config.scfg_cvars n' n;
               scfg_threads = tscs' }
              (* Thread (n, thunkifymode), ThunkedThread (n, fn () -> ti.ti_exp) *)
              ((PC1(B1 Thread, PTup[PVar(nn, TTyCon1 (TName, TTyCon0 TThread)); PVar(nth, TTyCon0 TThunkifymode)]))::ps)
              ((C1(B1 ThunkedThread, Tup [Id nn; Fn (no_loc [unitpat, flatten_all_closures eqs ti.ti_expr])]))::es) tks0
         | None, _    -> raise Blocked
         | Some _, _  -> raise (ExnResult (Raise (C0 (B0 ThunkifyThreadInDefinition))))
       with Not_found -> raise (ExnResult (Raise (C0 (B0 NonExistentThread)))))
    | _ -> raise (Never_happen ("thunkify: tk is not a value of the proper type: " ^ print_prim_expr (pserr()) tk)))
  | _ -> raise (Never_happen "thunkify: tks is not a list value")

and name_exists name config =
    NameValueMap.mem name config.scfg_mutexes ||
    NameValueMap.mem name config.scfg_cvars   ||
    NameValueMap.mem name config.scfg_threads

(* tail-recursive implementation *)
and unthunkify: smallstep_outer -> prim_expr -> smallstep_outer
  = fun config thks ->
  match thks with
    C0 (Nil _) -> config
  | Cons (thk, thks0) ->
    let nenv = config.scfg_nenv in
    (match thk with
      C1(B1 ThunkedMutex, Tup[v; b]) ->
        let (n,tn) = get_name_value "unthunkify mutex" v in
        if not (typ_eq tn (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
          raise (Never_happen "unthunkify Mutex: ill-typed name");
       if name_exists n config then
            raise (ExnResult (Raise (C0 (B0 ExistentName))))
       else let mtxi = {mtxi_name=n;
                        mtxi_state=None} in (* TODO: fix mtxi_state *)
            unthunkify { config with
              scfg_mutexes = NameValueMap.add n mtxi config.scfg_mutexes; } thks0
    | C1(B1 ThunkedCVar, Tup[v0; v]) ->
        let (n,tn) = get_name_value "unthunkify cvar" v0 in
        if not (typ_eq tn (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
          raise (Never_happen "unthunkify CVar: ill-typed name");
       if name_exists n config then
            raise (ExnResult (Raise (C0 (B0 ExistentName))))
       else let cvi = {cvi_name=n;
                       cvi_waiting=AQueue.create()} in (* TODO: fix cvi_waiting *)
            unthunkify { config with
              scfg_cvars =  NameValueMap.add n cvi config.scfg_cvars; } thks0
    | C1(B1 ThunkedThread, Tup[v; f]) ->
        let (n,tn) = get_name_value "unthunkify thread" v in
        if not (typ_eq tn (TTyCon1(TName,TTyCon0 TThread))) then  (* not tested in a real implementation *)
          raise (Never_happen "unthunkify Thread: ill-typed name");
       if name_exists n config then
            raise (ExnResult (Raise (C0 (B0 ExistentName))))
       else let tsc = {tsc_name=n;
                       tsc_state= TsRunnable;
                       tsc_defs = [];
                       tsc_ec   = CtxTop;
                       tsc_env  = empty_etysubst;  (* TODO: really? *)
                       tsc_eqs  = [];              (* TODO: really? *)
                       tsc_def  = None;
                       tsc_expr = App (f, C0 (B0 Unit));
                       tsc_next_expr = None } in
            unthunkify { config with
              scfg_runnable = AQueue.add n config.scfg_runnable;
              scfg_threads = NameValueMap.add n tsc config.scfg_threads; } thks0
    | _ -> raise (Never_happen "unthunkify: thk is not a value of the proper type"))
  | _ -> raise (Never_happen "unthunkify: thks is not a list value")


(* name comparison *)
and compare_name : string -> prim_expr -> prim_expr -> int
    = fun source v1 v2 ->
      match (v1,v2) with
      | (Col(v1',_,_),_) -> compare_name source v1' v2
      | (_,Col(v2',_,_)) -> compare_name source v1 v2'
      | (NameValue(nv1),NameValue(nv2)) ->
          let r = name_value_compare nv1 nv2 in
          if r < 0 then -1 else if r > 0 then 1 else 0  (* ugh, spec says we must normalise this *)
      | _ -> raise (Never_happen (source ^ ": compare_name: arguments not name values"))

(* reduce op.  Args already reduced to black values *)
and reduce_op : (bool * bool) -> op ->  prim_expr list -> threaded_smallstep_inner -> smallstep_outer -> smallstep_outer * prim_expr option
    = fun  closures op args tinf conf ->
      let (_,tenv) as env = tinf.tsc_env in
      let eqs = tinf.tsc_eqs in
      match (op,(List.map (get_val env eqs) args)) with
          (Ref(ty),   [v])         ->
            let l = fresh_location() in
            let ty' = get_typ env eqs ty in  (* remove tenv dependency before placing ty in senv *)
            ({conf with scfg_senv=(l,(TTyCon1 (TRef,ty')))::conf.scfg_senv; scfg_store=(l,v)::conf.scfg_store},Some(Loc(l)))
        | (Deref(ty), _)  -> raise (Never_happen "reduce_op: Deref")  (* handled elsewhere *)
        | (Assign(ty), _) -> raise (Never_happen "reduce_op: Assign")  (* handled elsewhere *)

        | (Equal(ty), [v1;v2])     -> (conf, Some(prim_equal tenv closures v1 v2))

        | (Less      ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Bool (n1 <    n2)))))
        | (LessEq    ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Bool (n1 <=   n2)))))
        | (Greater   ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Bool (n1 >    n2)))))
        | (GreaterEq ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Bool (n1 >=   n2)))))
        | (Mod       ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 mod  n2)))))
        | (Land      ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 land n2)))))
        | (Lor       ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 lor  n2)))))
        | (Lxor      ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 lxor n2)))))
        | (Lsl       ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 lsl  n2)))))
        | (Lsr       ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 lsr  n2)))))
        | (Asr       ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 asr  n2)))))
        | (UMinus    ,[C0(B0(Int(n1)))])                 -> (conf, Some(C0 (B0(Int  (-n1)))))
        | (Plus      ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 +    n2)))))
        | (Minus     ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 -    n2)))))
        | (Times     ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) -> (conf, Some(C0 (B0(Int  (n1 *    n2)))))
        | (Divide    ,[C0(B0(Int(n1)));C0(B0(Int(n2)))]) ->
            (try
              (conf, Some(C0 (B0(Int  (n1 /    n2)))))
            with
              Division_by_zero -> (conf,Some(Raise(C0 (B0 DivisionByZero)))))

        | (ListConcat(ty), [xs;ys])                  -> (conf, Some(prim_append xs ys))
        | (StrConcat, [C0(B0(String(s)));C0(B0(String(t)))]) -> (conf, Some(C0(B0(String(s^t)))))

        (* mutex stuff *)
        | (CreateMutex, [v']) ->
            let (n',tn') = get_name_value "CreateMutex" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op CreateMutex: ill-typed name");
            if is_name_fresh n' conf then
              let cfgm = NameValueMap.add n' { mtxi_name = n'; mtxi_state = None } conf.scfg_mutexes in
              ({ conf with scfg_mutexes = cfgm }, Some(C0 (B0 Unit)))
            else
              (conf, Some(Raise(C0 (B0 ExistentName))))
        | (Lock, [v']) -> (
            let (n',tn') = get_name_value "Lock" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Lock: ill-typed name");
            try
              let mutex =  NameValueMap.find n' conf.scfg_mutexes in
              match mutex.mtxi_state with
              | Some lmtxi ->
                  let lmtxi' = { lmtxi with lmtxi_waiting = AQueue.add tinf.tsc_name lmtxi.lmtxi_waiting } in
                  let tinf' = { tinf with tsc_state = TsMutexBlocked n' } in
                  (
                    { conf with
                      scfg_mutexes = NameValueMap.add n' { mutex with mtxi_state = Some lmtxi' } conf.scfg_mutexes;
                      scfg_threads = NameValueMap.add tinf.tsc_name tinf' conf.scfg_threads
                    },
                    None
                  )
              | None ->
                  let lmtxi = { lmtxi_owner = tinf.tsc_name ; lmtxi_waiting = AQueue.create () } in
                  ({ conf with scfg_mutexes = NameValueMap.add n' { mutex with mtxi_state = Some lmtxi } conf.scfg_mutexes}, Some(C0 (B0 Unit)))
            with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentMutex))))
          )
        | (TryLock, [v']) -> (
            let (n',tn') = get_name_value "TryLock" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op TryLock: ill-typed name");
            try
              let mutex = NameValueMap.find n' conf.scfg_mutexes in
              match mutex.mtxi_state with
              | Some lmtxi -> (conf, Some(C0(B0(Bool false))))
              | None ->
                  let lmtxi = { lmtxi_owner = tinf.tsc_name ; lmtxi_waiting = AQueue.create () } in
                  ({ conf with scfg_mutexes = NameValueMap.add n' { mutex with mtxi_state = Some lmtxi } conf.scfg_mutexes}, Some(C0 (B0( Bool true))))
            with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentMutex))))
          )
        | (Unlock, [v']) ->
            let (n',tn') = get_name_value "Unlock" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Unlock: ill-typed name");
            reduce_op_unlock n' tinf conf

        (* cvar stuff *)
        | (CreateCVar, [v']) ->
            let (n',tn') = get_name_value "CreateCVar" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op CreateCVar: ill-typed name");
            if is_name_fresh n' conf then
              let cfgcv = NameValueMap.add n' { cvi_name = n'; cvi_waiting  = AQueue.create () } conf.scfg_cvars in
              ({ conf with scfg_cvars = cfgcv }, Some(C0 (B0 Unit)))
            else
              (conf, Some(Raise(C0 (B0 ExistentName))))
        | (Wait, [cn ; mn]) -> (
            let (n',tn') = get_name_value "Wait(cn)" cn in
            let (n'',tn'') = get_name_value "Wait(mn)" mn in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Wait: ill-typed cvar name");
            if not (typ_eq tn'' (TTyCon1(TName,TTyCon0 TMutex))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Wait: ill-typed mutex name");
            try
              let mutex = NameValueMap.find n'' conf.scfg_mutexes in (
                try (
                  let cvar  = NameValueMap.find n' conf.scfg_cvars in
                  match mutex.mtxi_state with
                  | None ->
                      (conf, Some(Raise(C0 (B0 MutexEPERM))))
                  | Some _ -> (* this hack is because we have to perform the same rewrite at redex position
                               * for threads waiting on the mutex we will unlock.
                               * > We don't care about the configuration returned by the helper function,
                               * > unless it's a raise (which means the mutex didn't exist)
                               * -PH Thu, 01 Jul 2004 13:39:07 +0200
                               *)
                      match reduce_op_unlock n'' tinf conf with
                      | (conf' , Some(Raise _)) as result -> result
                      | (conf' , _ ) ->
                          let cvar' = { cvar with cvi_waiting = AQueue.add tinf.tsc_name cvar.cvi_waiting } in
                          let thread = { tinf with tsc_expr = Op(OEOp(Waiting), [cn;mn]);
                                                   tsc_state = TsCVarWaiting n'} in
                          ( { conf' with
                              scfg_threads = NameValueMap.add tinf.tsc_name thread conf'.scfg_threads;
                              scfg_cvars = NameValueMap.add n' cvar' conf'.scfg_cvars
                            } ,
                            None
                          )
                )
                with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentCVar))))
              )
            with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentMutex))))
          )
        | (Waiting, [_; _]) ->
            raise (Never_happen "Waiting cannot be reduced directly, only rewrite")
        | (Signal, [v']) -> (
            let (n',tn') = get_name_value "Signal" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Signal: ill-typed name");
            try
              let cvar  = NameValueMap.find n' conf.scfg_cvars in
              let (t_name,cviw) = AQueue.take cvar.cvi_waiting in
              let cvar' = { cvar with cvi_waiting = cviw } in

              let conf' = restart_one t_name conf in
              ( { conf' with scfg_cvars = NameValueMap.add n' cvar' conf.scfg_cvars }, Some(C0 (B0 Unit)) )
            with
            | AQueue.Empty -> ( conf, Some(C0 (B0 Unit)) )
            | Not_found -> (conf, Some(Raise(C0 (B0 NonExistentCVar))))
          )
        | (Broadcast, [v']) -> (
            let (n',tn') = get_name_value "Broadcast" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TCVar))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Broadcast: ill-typed name");
            try
              let cvar  = NameValueMap.find n' conf.scfg_cvars in
              let cvar' = { cvar with cvi_waiting = AQueue.create() } in

              let conf' = AQueue.fold_rear restart_one cvar.cvi_waiting conf in
              ( { conf' with scfg_cvars = NameValueMap.add n' cvar' conf.scfg_cvars }, Some(C0 (B0 Unit)) )
            with
            | Not_found -> (conf, Some(Raise(C0 (B0 NonExistentCVar))))
         )

        (* thread stuff *)
        | (CreateThread _, [v'; v1; v2]) ->
            let (n',tn') = get_name_value "CreateThread" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TThread))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op CreateThread: ill-typed name");
            if is_name_fresh n' conf then
              let tsc_new = { tsc_name = n';
                              tsc_state= TsRunnable;
                              tsc_defs = [];
                              tsc_ec   = CtxTop;
                              tsc_env  = empty_etysubst;
                              tsc_eqs  = [];  (* values are blackened *)
                              tsc_def  = None;
                              tsc_expr = App (get_val env [] v1, get_val env [] v2);
                              tsc_next_expr = None } in
              let cfgth  = NameValueMap.add n' tsc_new conf.scfg_threads in
              let cfgrun = AQueue.add n' conf.scfg_runnable in
              ({ conf with scfg_threads = cfgth; scfg_runnable = cfgrun }, Some(C0 (B0 Unit)))
            else
              (conf, Some(Raise(C0 (B0 ExistentName))))
        | (Self, [C0 (B0 Unit)]) -> (conf, Some (NameValue tinf.tsc_name))
        | (Kill, [v']) ->
            let (n',tn') = get_name_value "Kill" v' in
            if not (typ_eq tn' (TTyCon1(TName,TTyCon0 TThread))) then  (* not tested in a real implementation *)
              raise (Never_happen "reduce_op Kill: ill-typed name");
           (try
              let thread = NameValueMap.find n' conf.scfg_threads in
              let conf' = { conf with scfg_threads = NameValueMap.remove n' conf.scfg_threads } in
              if name_value_eq n' tinf.tsc_name then
                (conf', None)
              else
              let conf'' = match thread.tsc_state with
              | TsRunnable  ->
                  let cfgrun = AQueue.removeFirst (name_value_eq n') conf.scfg_runnable in
                  { conf' with scfg_runnable = cfgrun }
              | TsSlowcall | TsResolveBlocked ->
                  let cfgslo = NameValueMap.remove n' conf.scfg_slowcall in
                  { conf' with scfg_slowcall = cfgslo }
              | TsMutexBlocked n'' ->
                  let mutex = try NameValueMap.find n'' conf.scfg_mutexes with Not_found -> raise (Never_happen "reduce-op: mutex not found")in
                  (match mutex.mtxi_state with (* update mutex waiting list *)
                    Some lmtx -> let lwait = AQueue.removeFirst (name_value_eq n') lmtx.lmtxi_waiting in
                                 let lmtx' = { lmtx with lmtxi_waiting = lwait } in
                                 let mtx' = { mutex with mtxi_state = Some lmtx' } in
                                 let cfgmt = NameValueMap.add n'' mtx' conf.scfg_mutexes in
                                 { conf' with scfg_mutexes = cfgmt }
                  | None      -> raise (Never_happen ("reduce_op(kill): thread blocked on empty mutex")))
              | TsCVarWaiting n'' ->
                  let cvar =try  NameValueMap.find n'' conf.scfg_cvars with Not_found -> raise (Never_happen "reduce-op: cvar not found")in (* update cvar waiting list *)
                  let lwait = AQueue.removeFirst (name_value_eq n') cvar.cvi_waiting in
                  let cvar' = { cvar with cvi_waiting = lwait } in
                  let cfgcv = NameValueMap.add n'' cvar' conf.scfg_cvars in
                  { conf' with scfg_cvars = cfgcv }
              in
              !Debug.thread_exit_hook None;
              (
                if !Opts.printcleandeath then (
                  let s = "Thread killed: \n"  (* Get thread name? Get time? *)
                  in
                  let ch = Dump.safe_open_out true in
                  output ch s 0 (String.length s);
                  flush ch;
                  Dump.safe_close_out ch
                ) else ()
              );
              (conf'', Some(C0 (B0 Unit)))
            with Not_found -> (conf, Some(Raise(C0 (B0 NonExistentThread)))))

        (* thunkify stuff *)
        | (Thunkify, [tks]) ->
           (try
              let conf', e = thunkify tinf.tsc_name conf conf [] [] (get_val env [] tks) in
              conf', Some e
           with
              Blocked ->
              (* Pervasives.prerr_endline "thunkify blocked"; *)
              conf, Some (OP(1, OEOp(Thunkify), [tks])) (* a spin lock *)
              (* NB: this is (1) inefficient and (2) depends on some scheduling fairness *)
            | ExnResult e -> conf, Some e)
        | (Unthunkify, [thks]) ->
           (try
              let conf' = unthunkify conf (get_val env [] thks) in
              conf', Some (C0 (B0 Unit))
           with
              ExnResult e -> conf, Some e)

        | (CompareName t, [v1; v2]) ->
            let r = compare_name "CompareName" v1 v2 in
            (conf, Some (C0 (B0(Int r))))
        | (NameToString t, [v]) ->
            let (nv, t') = get_name_value "NameToString" v in
            (* TODO check t == t' *)
            let s = string_of_lithash (abstract_name_of_name_value nv) in
            (conf, Some (C0 (B0(String s))))
        | ((Exit t), [C0 (B0(Int i))]) ->
            let conf' = {conf with
                              scfg_runnable = AQueue.create ();
                              scfg_slowcall = Ast.NameValueMap.empty;
                              scfg_threads =  Ast.NameValueMap.empty;}
            in (conf', None)
        | _ ->
            let rec print_list = function [] -> ""
              | (v::vs) -> (print_prim_expr (ps()) v) ^ ", " ^ (print_list vs)
            in

            let s = "reduce_op 2: " ^ print_prim_expr (ps()) (Op (OEOp op,args)) in
            raise (Stuck s)



(* M has /deleted/ the big step evaluator, small step only these days *)

(*let reduce_fast _ = raise (Util.Unimplemented "bigstep evaluator")*)

(*  -=-- Small-step evaluator code -------------------- ---------- --=- *)

(* Hm, how is this going to work with processes? *)
(* Refocus should still just operate on expressions, I think *)
(* --- no change to refocus, feed the new configs into the smallstep evaluator. *)
(* get something that works onthe oldcode,modulo typechecker? *)


(* Think about module initialisation---do we need evalmode back after all? *)
(* Can we spawn threads during module initialisation? *)

(*

Overview
~~~~~~~~

This is the small-step evaluator.  It implements pretty much the
semantics in the document, except for using
environments and closures rather than substitution.

Here is a quick guide to how it works (thanks Mair, 2004-04-30).

Refocus to shuffle (context, expression) pair (context, redex) pair.
Reduce to perform reduction, resulting in new (context, expression)
pair.

Begin with refocus_down: peel layers of context off the expression and
add to context. If expression turns out to be a value, start sticking
contexts back on again---refocus_up---until either we can go down again by a
different branch, or we find a redex, or the context is empty
(CtxTop). If the context is empty and the expression is a value or a
raise v, then we're done. If not, something's gone wrong.

The context representation is that of Didier's zipper things:
basically you want the contexts to be represented as something like a
LIFO list. A context is an expression with a hole in. When you add
a new context to an existing context, you do it by stuffing the old
context into the hole, so that the newest one is outermost.

Raises are slightly tricky---well, not really, but a bit different,
because the reduction axiom involves the context. refocus returns (C,
raise v), reduce pattern matches on the expression as normal, and all
is well.

refocus_down uses the exception Value of int to tell its caller when
its got a value. The int is the arity of the value, so that App()
knows without having to chase down again whether App(v,v) is a value
or not. We also need to store this arity in the CtxAppR evaluation
context, as otherwise the arity of the v in App(v, e) will be lost as
we evaluate e.

*)


(* deletes the innermost element of the evaluation context (used for Raise) *)
and delete_innermost_context: threaded_smallstep_inner -> threaded_smallstep_inner
    = fun eee ->
      match eee.tsc_ec with
      CtxTop -> raise (Never_happen "trying to delete from CtxTop")
    | CtxC1(c, ec')                     -> { eee with tsc_ec = ec' }
    | CtxConsL (ec', e')                -> { eee with tsc_ec = ec' }
    | CtxConsR (e', ec')                -> { eee with tsc_ec = ec' }
    | CtxTup (es, ec', vs)              -> { eee with tsc_ec = ec' }
    | CtxInEnv (env', ec')              -> { eee with tsc_ec = ec'; tsc_env = env' }
    | CtxIf (ec', e1, e2)               -> { eee with tsc_ec = ec' }
    | CtxSeq (ec', e')                  -> { eee with tsc_ec = ec' }
    | CtxAppL (ec', e')                 -> { eee with tsc_ec = ec' }
    | CtxAppR (e', ec')              -> { eee with tsc_ec = ec' }
    | CtxTApp (ec', t)                  -> { eee with tsc_ec = ec' } (* NEW *)
    | CtxUnpack (it, iid, ec', e)       -> { eee with tsc_ec = ec' } (* NEW *)
    | CtxNamecaseL (ec', it, iid1, iid2, e1, e2, e3) -> { eee with tsc_ec = ec' } (* NEW *)
    | CtxNamecaseR (v, it, iid1, iid2, ec', e1, e2) -> { eee with tsc_ec = ec' } (* NEW *)
    | CtxMatch (ec', m')                -> { eee with tsc_ec = ec' }
    | CtxRaise (ec')                    -> { eee with tsc_ec = ec' }
    | CtxTry (ec', m)                   -> { eee with tsc_ec = ec' }
    | CtxMarshalL (ec', e2, t)          -> { eee with tsc_ec = ec' }
    | CtxMarshalR (v1, ec', t)          -> { eee with tsc_ec = ec' }
    | CtxMarshalz (eqs', mk, ec', t)    -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }
    | CtxUnmarshal(ec', t)              -> { eee with tsc_ec = ec' }
    | CtxCol (ec', eqs', t)             -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }
    | CtxOp (op, es, ec', vs)           -> { eee with tsc_ec = ec' }
    | CtxLazyOp (lo, ec', e')           -> { eee with tsc_ec = ec' }
    | CtxOP (eqs', n, oe, es, ec', vs)  -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }
    | CtxHashTs(eqs', t0, ec', t1)       -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }(* NEW FROM HERE DOWN*)
    | CtxHashHtsL(eqs', t0, ec', e, t1)  -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }
    | CtxHashHtsR(eqs', t0, v, ec', t1) -> { eee with tsc_ec = ec'; tsc_eqs = eqs' }
    | CtxSwapL(ec', e1, e2)             -> { eee with tsc_ec = ec' }
    | CtxSwapM(e1, ec', e2)             -> { eee with tsc_ec = ec' }
    | CtxSwapR(e1, e2, ec')             -> { eee with tsc_ec = ec' }
    | CtxFreshforL(ec', e)              -> { eee with tsc_ec = ec' }
    | CtxFreshforR(e, ec')              -> { eee with tsc_ec = ec' }
    | CtxSupport(t, ec')                -> { eee with tsc_ec = ec' }
    | CtxNameOfTie(ec')                 -> { eee with tsc_ec = ec' }
    | CtxValOfTie(ec')                  -> { eee with tsc_ec = ec' }
    | CtxPack(t1,ec',t2)                -> { eee with tsc_ec = ec' }
    (* TODO: What should we do here? *)
    | CtxStrVal(_) -> raise (Never_happen "delete innermost context in SC")
    | CtxImodule(_) -> raise (Never_happen "delete innermost context in TC")





and subst_imods defs =
  let cmod_of_imod = function
      Mod_imod(mn, mcbody) ->
	Mod_compile(mn,
		    {mc_hash=mcbody.mi_hash;
                      mc_vubs=mcbody.mi_vubs;
                      mc_eqs=mcbody.mi_eqs;
                      mc_sign0=mcbody.mi_sign0;
                      mc_sign1=mcbody.mi_sign1;
                      mc_vn=mcbody.mi_vn;
                      mc_str=
                      {loc=mcbody.mi_str_loc;
			desc=(List.rev mcbody.mi_str_done) @ mcbody.mi_str_todo}})
    | _ -> assert false
  in
  match defs with
    ((Mod_imod _) as def) ::defs' -> (cmod_of_imod def)::defs'
  | defs -> defs


(* composes an evaluation context with an expression, yielding an expression
   and a new environment and equation set and some evaluation context (the outermost?  is this included in the expr??) *)
(* not sure what scs flag does *)

(*
see the strval case.

there are two cases of pull_context.

If scs is false, we have an ec and expr, and we wish to stick the whole of ec
onto the expr to to get a CtxTop or CtxStrVal and an expr.

If scs is true, we have an ec and an expr, where the ec includes a
CtxImodule. In this case we only want to pull out the expr
corresponding to the current strval, leaving the imod in the context.

We use either (makes no odds) for collecting values from normal
expressions, the first type for constructing configurations, we use
the second type for collecting values from initialised module fields.
*)

and pull_context : bool -> threaded_smallstep_inner -> eval_ctx * etysubst * eqs * thread_info (* (env, eqs, thread_info) *)
    = fun scs ({ tsc_env = env; tsc_eqs = eqs; tsc_ec = ec; tsc_expr = e; tsc_name = n; tsc_defs = defs;tsc_def=def } as eee) ->
      match ec with
        CtxTop -> (CtxTop, env, eqs, {ti_name=eee.tsc_name;ti_state=eee.tsc_state;ti_defs= subst_imods eee.tsc_defs;ti_expr=eee.tsc_expr})
      | CtxC1(c, ec')                    -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = C1(c, e) }
      | CtxConsL (ec', e')               -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Cons(e, e')  }
      | CtxConsR (e', ec')               -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Cons(e', e) }
      | CtxTup (es, ec', vs)             -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Tup((List.rev es) @ (e::vs)) }
      | CtxInEnv (env', ec')             -> pull_context scs { eee with tsc_env = env'; tsc_ec = ec'; tsc_expr = InEnv(env, e) }
      | CtxIf (ec', e1, e2)              -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = If(e, e1, e2) }
      | CtxSeq (ec', e')                 -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Seq(e, e') }
      | CtxAppL (ec', e')                -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = App(e, e') }
      | CtxAppR (e', ec')             -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = App(e', e) }
      | CtxTApp (ec', t)                  -> pull_context scs { eee with tsc_ec = ec'; tsc_expr = TApp(e,t) } (* NEW *)
      | CtxUnpack (ity, iid, ec', e')       -> pull_context scs { eee with tsc_ec = ec'; tsc_expr = Unpack(ity, iid, e, e') } (* NEW *)
      | CtxNamecaseL (ec', ity, iid1, iid2, e1, e2, e3) -> pull_context scs { eee with tsc_ec = ec'; tsc_expr = Namecase(e, ity, iid1, iid2, e1, e2, e3) } (* NEW *)
      | CtxNamecaseR (v, ity, iid1, iid2, ec', e1, e2) -> pull_context scs { eee with tsc_ec = ec'; tsc_expr = Namecase(v, ity, iid1, iid2, e, e1, e2) } (* NEW *)
      | CtxMatch (ec', m')               -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Match(e, m') }
      | CtxRaise (ec')                   -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Raise(e) }
      | CtxTry (ec', m)                  -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Try(e, m) }
      | CtxMarshalL (ec', e2, t)         -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Marshal(e, e2, t) }
      | CtxMarshalR (v1, ec', t)         -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Marshal(v1, e, t) }
      | CtxMarshalz (eqs', mk, ec', t)   -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec'; tsc_expr = Marshalz(mk, e, t) }
      | CtxUnmarshal(ec', t)             -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Unmarshal(e, t) }
      | CtxCol (ec', eqs', t)            -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec'; tsc_expr = Col(e, eqs, t) }
      | CtxOp (op, vs, ec', es)          -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Op(op, vs @ (e::es)) }
      | CtxLazyOp (lo, ec' ,e')          -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = LazyOp(lo, [e;e']) }
      | CtxOP (eqs', n, oe, es, ec', vs) -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec'; tsc_expr = OP(n, oe, ((List.rev es) @ (e::vs))) }
      | CtxHashTs(eqs', t0, ec', t1)      -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec'; tsc_expr = HashTs(t0, e, t1)}
      | CtxHashHtsL(eqs', t0, ec', e', t1)   -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec';tsc_expr = HashHts(t0, e, e',t1)}
      | CtxHashHtsR(eqs', t0, v, ec', t1)  -> pull_context scs { eee with tsc_eqs = eqs'; tsc_ec = ec'; tsc_expr = HashHts(t0, v, e, t1)}
      | CtxSwapL(ec', e1, e2)              -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Swap(e,e1,e2)}
      | CtxSwapM(e1, ec', e2)              -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Swap(e1, e, e2) }
      | CtxSwapR(e1, e2, ec')              -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Swap(e1, e2, e) }
      | CtxFreshforL(ec', e')               -> pull_context scs { eee with                tsc_ec = ec'; tsc_expr = Freshfor(e, e') }
      | CtxFreshforR(e', ec')               -> pull_context scs { eee with                tsc_ec = ec'; tsc_expr = Freshfor(e', e) }
      | CtxSupport(t, ec')                 -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Support(t, e) }
      | CtxNameOfTie(ec')                  -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = NameOfTie(e)}
      | CtxValOfTie(ec')                   -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = ValOfTie(e) }
      | CtxPack(t1,ec',t2)                 -> pull_context scs { eee with                 tsc_ec = ec'; tsc_expr = Pack(t1,e,t2) }
      | CtxStrVal(i, ec', loc) ->
          if not scs then
            (ec, env, eqs, {ti_name=eee.tsc_name;ti_state=eee.tsc_state;ti_defs=eee.tsc_defs;ti_expr=eee.tsc_expr})
          else
            (match ec' with
            | CtxImodule(eqs1, mn, mcbody, ec'') ->
                (* construct a ... cmodule? *)
                (let def = Mod_compile(mn,
                                    {mc_hash=mcbody.ctx_mc_hash;
                                      mc_vubs=mcbody.ctx_mc_vubs;
                                      mc_eqs=mcbody.ctx_mc_eqs;
                                      mc_sign0=mcbody.ctx_mc_sign0;
                                      mc_sign1=mcbody.ctx_mc_sign1;
                                      mc_vn=mcbody.ctx_mc_vn;
                                      mc_str=
                                      {loc=mcbody.ctx_mc_str_loc;
                                        desc=List.rev_append mcbody.ctx_mc_str_done (at_loc loc (StrVal(i,primtoexpr e)) :: mcbody.ctx_mc_str_todo)
                                      }})

                in
                match eee.tsc_next_expr with
                | None ->
                    pull_context scs {eee with tsc_defs = def::eee.tsc_defs;
                                      tsc_ec = ec'';
                                      tsc_next_expr=None;
                                      tsc_eqs = eqs1;
                                    }
                |  Some nexpr ->
                    pull_context scs {eee with tsc_defs = def::eee.tsc_defs;
                                      tsc_ec = ec'';
                                      tsc_expr=nexpr;
                                      tsc_next_expr=None;
                                      tsc_eqs = eqs1;
                                    })
            | _ -> raise (Never_happen "SC not in TC")) (* TODO, or can we have nested SCs? *)
      | CtxImodule(_)  -> raise (Never_happen "TC not containing SC, should have been converted to cmod.")



let rec pull_contexts : smallstep_outer -> configuration =
  fun ss_outer ->
  let threads' =  NameValueMap.map
      (function ssi -> let (_, env, eqs, t) = pull_context true ssi in
      (if not (env = empty_etysubst && eqs = []) then raise
	  (Never_happen ("pulled contexts: env is " ^ print_etysubst (pserr()) env
			 ^ "\n eqs are " ^ print_eqs (pserr()) eqs));
       t))
      ss_outer.scfg_threads
  in
  { cfg_nenv=ss_outer.scfg_nenv;
    cfg_defs=ss_outer.scfg_defs;
    cfg_senv=ss_outer.scfg_senv;
    cfg_store=ss_outer.scfg_store;
    cfg_runnable=ss_outer.scfg_runnable;
    cfg_slowcall=ss_outer.scfg_slowcall;
    cfg_threads=threads';
    cfg_mutexes=ss_outer.scfg_mutexes;
    cfg_cvars=ss_outer.scfg_cvars;}

let souter_of_conf : configuration -> smallstep_outer =
  fun conf ->
    let threads' =  NameValueMap.map
        (function ti ->
          { tsc_ec = CtxTop;
            tsc_state = ti.ti_state;
            tsc_env = empty_etysubst;
            tsc_eqs = [];
            tsc_expr = ti.ti_expr;
            tsc_def = None;
            tsc_defs = ti.ti_defs;
            tsc_name=ti.ti_name;
            tsc_next_expr = None
          }) conf.cfg_threads
    in
    {
    scfg_nenv = conf.cfg_nenv;
    scfg_defs = conf.cfg_defs;
    scfg_senv = conf.cfg_senv;
    scfg_store = conf.cfg_store;
    scfg_runnable = conf.cfg_runnable;
    scfg_slowcall = conf.cfg_slowcall;
    scfg_threads = threads';
    scfg_mutexes = conf.cfg_mutexes;
    scfg_cvars = conf.cfg_cvars
  }


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*               runtime type checking                                  *)


(* note that runtime type checking won't happen exactly at every
reduction step as specified in the semantics, as (eg) we propagate
exceptions more quickly *)



(* hackery to abbreviate print of the same thing multiple times in
succession.  Ought to do this by a config_eq check, but easier to
check the string.  Won't be necessary when we're small-step, in any
case *)

let last_config_string : string option ref = ref None

(* step count; incremented by the individual evaluators, not by this
   function (since it is not always called) *)
let step : int ref = ref 0

let reset () =
  last_config_string := None;
  step := 0


(* M NOTES:
   this used to take env and eqs. i think the use of env has been correctly replaced in the calls to pull_context, except that we can't use
   the alternate pretty printing style used in -printclos. The eqs didn't seem to be used at all and have gone entirely.
   Both these are expression specific so had to go when we had a queue of expressions. Afacit we're always at top level here so eqs should be empty
   - K writes: seems fine to me.
*)

let eval_trace_or_typecheck
    = fun sconfig ->
      (

      if (!Opts.dumptrace > 0 && !step > !Opts.dumpfrom && !step mod !Opts.dumpstepinterval = 0) then
        begin
            let this_config_string =
	      print_smallstep_outer (ps()) sconfig in
            if not(!Opts.dumpall) &&  !last_config_string=None || !last_config_string = Some this_config_string then (
              last_config_string := Some this_config_string;
              print_string_really ("\n["^string_of_int !step^":]-->...") )
            else  (
              last_config_string := Some this_config_string;
              print_string_really ("\n["^string_of_int !step^":]-->\n\n" ^ this_config_string ^ "\n");
              )
        end;



           if (!Opts.rttc)   then
          begin
	    let config = pull_contexts sconfig in
	     let config' =
            {config with
            cfg_store = List.map (function (l,expr) -> (l, flatten_all_closures [] expr))
              config.cfg_store;
            cfg_threads = NameValueMap.map (function ti -> {ti with ti_expr = flatten_all_closures [] ti.ti_expr; ti_defs = List.map def_flatten_all_closures ti.ti_defs})
              config.cfg_threads;
            cfg_defs = List.map def_flatten_all_closures config.cfg_defs
          }
      in

            try
              if (!Opts.dumptrace > 0) then
                print_string_really ("    runtime typecheck :     "); (* do here to ensure flushed *)
              Typecheck.tcheck_configuration config';  (* we used to check that the type remained the same, but now there's a type for each thread... *)
              if (!Opts.dumptrace > 0) then
                print_string_really "ok.\n";
            with
              Typecheck.TCFail(s) ->
                begin
                  if (!Opts.dumptrace >= !Opts.dumptypefail) then
                    print_string_really ("\nruntime typechecking failed with error: "^s^"\n")
                  else
                    print_string_really
                      ("\nruntime typechecking failed for configuration\n"
                       ^ print_configuration (Dump.fresh_printer_state !Opts.dumptypefail !skipdefs_ref false) config'
                       ^ "\nwith error: "^s^"\n");
                  if (!Opts.terminate_on_rttc) then raise (Typecheck_of_configuration config')
                end
          end;
      step := !step + 1;
      (match !Opts.printstepinterval with
	None -> ()
      | Some(interval) ->
          if !step mod interval = 0 then (
            print_string ("Step "^string_of_int !step (* ^ ":" ^ print_hack (inner'.tsc_expr) *) ^  "\r"); flush stdout;
            );
          )
      )





(* ****************************************************************** *)

(* P notes        Mon Feb 16 15:53:53 GMT 2004


If reduce_once takes a triple (ec,eqs,e) then we may need to look
either down e or down ec in order to find the redex, eg (in casual
syntax):

    (ec,                                    eqs, C1.C2.C3.(2+2)  )  representing  ec.C1.C2.C3.(2+2)
or
    ((_,1).(_,2).(_,3).let p=_ in e.CtxTop, eqs, 7               )  representing  let p=(((7,1),2),3) in e

Moreover, may have to look arbitrarily far in either case.
(In Didier's notes the second case doesn't appear as he doesn't have fancy values).
(In Andy's Frame-stack semantics notes you see reduction steps for both directions).

Question: do we want the refocus function (that does this) to return
with the redex entirely in the term part or split (where possible)
between the context and the term?  Let's have a quick look through the
reduction axioms (though fudging applications for now):

            CtxC1 (c, ec')
            CtxConsL (ec', e1)
            CtxConsR (v1, ec')
            CtxTup (es, ec', vs)
            CtxIf  (ec', e1, e2)              C0(Bool(b))
            CtxAnd (ec', e1)                  C0(Bool(b))
            CtxOr (ec', e1)                   C0(Bool(b))
            CtxSeq (ec', e1)                  C0(Unit)
            CtxAppL(ec', e1)                   ...
            CtxAppR(v1, ec')                   ...
            CtxLet (ec', binding)             v
            CtxRaise(ec')
            CtxTry(ec',m)                     Raise(e)
            CtxTry(ec',m)                     v
            CtxMarshal (mk, ec', ty)          v
            CtxMarshalz (mk, ec', ty)         v
            CtxUnmarshal (ec', ty)            v
            CtxCol (ec', eqs', ty)            Raise(e)
            CtxCol (ec', eqs', ty)            C0(c0)
            CtxCol (ec', eqs', ty)            C1(c1,v)
            CtxCol (ec', eqs', ty)            Cons(v1,v2)
            CtxCol (ec', eqs', ty)            Tup(vs)
            CtxCol (ec', eqs', ty)            ... app...
            CtxCol (ec', eqs', ty)            Loc(l)
            CtxCol (ec', eqs', ty)            Fn(m)
            CtxCol (ec', eqs', ty)            v        if ...
            CtxCol (ec', eqs', ty)            Col(v,eqs'',ty')
            CtxOP (n, e0, es1, ec', vs2)
            CtxTop
            c(C')                             Raise(m)    if ...
             *                                While(e1,e2)
             *                                Letrec(ty, ...)
             *                                ...resolve...
             *                                Dot(...)


Hmm, not sure. Perhaps easiest to understand if the redex is entirely
in the term part.  Is there an efficiency loss? Can't see.

Note that it's true that in either up or down case, when you stop you
know something about what redex might be applicable, some (or all?) of
the reduction rules might be found while looking in both directions.

Question (thinking about looking up): for which C is there a v such
that C.v might be a value?

            CtxC1 (c, ec')                y
            CtxConsL (ec', e1)
            CtxConsR (v1, ec')            y
            CtxTup (es, ec', vs)          y if es=[], maybe if es=/=[]
            CtxIf  (ec', e1, e2)
            CtxAnd (ec', e1)
            CtxOr (ec', e1)
            CtxSeq (ec', e1)
            CtxAppL(ec', e1)
            CtxAppR(v1, ec')              maybe
            CtxLet (ec', binding)
            CtxRaise(ec')                 never a value, but not reducible
            CtxTry(ec',m)
            CtxTry(ec',m)
            CtxMarshal (mk, ec', ty)
            CtxMarshalz (mk, ec', ty)
            CtxUnmarshal (ec', ty)
            CtxCol (ec', eqs', ty)        y if ...
            CtxOP (n, e0, es1, ec', vs2)  y if ...
            CtxTop



*)


(* Experiment: write a pure "refocus" function, making no attempt to
   actually do any reduction.  Make not much attempt to avoid isvalue
   checks. Have it return with the redex entirely in the term part,
   for clarity when comparing with the semantics.  Ignore built-in
   operators and special constants for now.  Ignore closures for now.
   Use exceptions to pass back the value case.  What about Id? ignore
   for now, as ignoring closures, but will have to decide whether to
   see a reduction or not.  Think probably should see one (and the
   closure semantics should make clear).*)





(* search downwards; raise Refocus_down_value if the expression is a
   value; otherwise, return the redex *)
let rec refocus_down : threaded_smallstep_inner -> threaded_smallstep_inner
    = fun (eee) ->
      (* Debug.dbgassert (fun () -> (!Opts.evaluator = Opts.smallstep)); *)
          if (!Opts.dumptrace > 1 && !Opts.showfocussing) then
          begin
            let this_config_string = print_threaded_smallstep_inner
                (ps()) (eee) in
            print_string_really ("\n--refocusing_down:\n "  ^ this_config_string ^ "\n");
          end;
      (* auxiliaries for wrapping constructors *)

      let wrap_cons_unary eee' =
        (try refocus_down eee'
        with
         Refocus_down_value -> raise Refocus_down_value) in

      let wrap_cons_binary eee' eee'' =
        (try refocus_down eee'
        with
         Refocus_down_value -> (
            try refocus_down eee''
            with
             Refocus_down_value -> raise Refocus_down_value )) in

      let rec wrap_cons_list eees =
        match eees with
          [] -> raise Refocus_down_value
        | eee'::eees' -> (
            try refocus_down eee'
            with
             Refocus_down_value -> wrap_cons_list eees') in

      (* auxiliary for wrapping destructors *)

      let wrap_dest_unary eee' = (
        try refocus_down eee'
        with
         Refocus_down_value -> eee) in

      let rec collect_n_args n (ctx, eqs, acc) =
        if (n=0) then (ctx, eqs, acc) else
        match ctx with
          (* here, we should be reducing e' to a value before collecting up *)
          (* CtxAppL(ec', e') -> collect_n_args (n-1) (ec', eqs, e'::acc)  *)
        | CtxAppR(e', ec') -> collect_n_args (n-1) (ec', eqs, e'::acc)
        | other ->
            raise Refocus_down_value
      in

      let e = eee.tsc_expr in
      let ec = eee.tsc_ec in
      let eqs = eee.tsc_eqs in
      let env = eee.tsc_env in
      match e with
        C0(_)        -> raise Refocus_down_value
      | C1(c,e1)     -> wrap_cons_unary { eee with tsc_ec = CtxC1(c, ec); tsc_expr = e1 }
      | Cons(e1,e2)  -> wrap_cons_binary { eee with tsc_ec = CtxConsL(ec, e2); tsc_expr = e1 }
                                         { eee with tsc_ec = CtxConsR(e1, ec); tsc_expr = e2 }
      | Tup(es)      -> refocus_down_list eee es
                          (fun (es1,vs2) -> CtxTup(es1,ec,vs2))
                          (fun () -> raise Refocus_down_value 0)
      | Op(OEOp(Deref t), [e']) -> (
          try refocus_down {eee with tsc_ec=CtxOp(OEOp(Deref t), [], ec,[]);
                             tsc_expr=e'}
          with Refocus_down_value -> eee
              )
      | Op(OEOp(Assign' t), [l;e']) -> (
          try refocus_down {eee with tsc_ec=CtxOp(OEOp(Assign' t), [l], ec,[]);
                             tsc_expr=e'}
          with Refocus_down_value -> eee
              )
      | Op(oe,es)    -> (* refocus_down_list eee es
                          (fun (es1,vs2) -> CtxOp(oe,es1,ec,vs2))
                          (fun () -> eee)  (* if es are all values, this is the redex *) *)
          (* K: TODO: tidy up: use refocus_down_list *)
          let rec go es vs =
            match es with
              [] -> eee
            | (e'::es') ->
                try
                  refocus_down { eee with tsc_ec = CtxOp(oe, List.rev vs, ec, es'); tsc_expr = e' }
                with
                  Refocus_down_value ->
                    go es' (e'::vs)
          in
          go es []

      | LazyOp(lo,es) ->
          ( match es with
          | [e1;e2] -> wrap_dest_unary { eee with tsc_ec = CtxLazyOp (lo, ec, e2) ; tsc_expr = e1 }
          | _ -> raise (Util.Never_happen "refocus_down LazyOp") )

      | Loc(_) -> raise Refocus_down_value
      | Fn(_)-> eee
      | Fun (_) -> raise (Never_happen "refocus_down Fun")
      | TAbs(it, e') -> (* TCLOS: *) eee
            (* raise Refocus_down_value *)
      | InEnv(env',e') ->
          (try
            refocus_down { eee with tsc_ec = CtxInEnv(env,ec); tsc_env = env'; tsc_expr = e' }
          with
            Refocus_down_value ->
              eee)
      | Clos _ -> raise Refocus_down_value
      | TClos _ -> raise Refocus_down_value
      | Id(i) -> eee
      | Dot(_,_) -> eee
      | HashDot(_,_) -> raise (Never_happen "refocus_down HashDot")
      | If(e1,e2,e3) -> wrap_dest_unary { eee with tsc_ec = CtxIf(ec, e2, e3); tsc_expr = e1 }
      | While(e1,e2) -> eee
(*
 *       | And(e1,e2)   -> wrap_dest_unary { eee with tsc_ec = CtxAnd(ec, e2); tsc_expr = e1 }
 *       | Or(e1,e2)    -> wrap_dest_unary { eee with tsc_ec = CtxOr(ec, e2) ; tsc_expr = e1 }
 *)
      | Seq(e1,e2)   -> wrap_dest_unary { eee with tsc_ec = CtxSeq(ec, e2); tsc_expr = e1 }
      | App(e1,e2)   -> (
          try refocus_down { eee with tsc_ec = CtxAppL(ec, e2); tsc_expr = e1 }
          with
           Refocus_down_value -> (
              try refocus_down { eee with tsc_ec = CtxAppR(e1, ec); tsc_expr = e2 }
              with
                Refocus_down_value  ->
                  match e1 with
                  (* | Fn(m) -> eee *)
                  | Clos(_) -> eee
                  | _ ->  raise (Never_happen ("[App]App not of fn: " ^  print_prim_expr (pserr()) e1 ^ " (from "
                                                  ^  print_prim_expr (pserr()) e ^ ")"))
             ))
      | TApp(e', t) ->(
           try
            refocus_down {eee with tsc_ec = CtxTApp(ec,t);
                           tsc_expr=e'}
          with Refocus_down_value -> eee)
      | Match(e,m) -> wrap_dest_unary { eee with tsc_ec = CtxMatch(ec, m); tsc_expr = e }
      | Let _ -> raise (Never_happen "refocus_down Let")
      | Letrec(ty,binding) -> eee
      | Raise(e1) -> (
          try refocus_down { eee with tsc_ec = CtxRaise(ec); tsc_expr = e1 }
          with
           Refocus_down_value ->
             match ec with CtxTop -> raise Refocus_down_value
             | _ -> eee)

      | Try(e1,m)    -> (
          try refocus_down { eee with tsc_ec = CtxTry(ec, m); tsc_expr = e1 }
          with
           Refocus_down_value ->   eee
               )
      | Marshal(e1,e2,ty1) -> (
          try refocus_down {eee with tsc_ec = CtxMarshalL(ec, e2, ty1); tsc_expr = e1}
          with Refocus_down_value ->
            try
            refocus_down {eee with tsc_ec = CtxMarshalR(e1, ec, ty1); tsc_expr = e2}
                with Refocus_down_value -> eee
              )
      | Marshalz(mk,e2,ty1) -> (
            try
            refocus_down {eee with tsc_ec = CtxMarshalz(eee.tsc_eqs, mk, ec, ty1); tsc_expr = e2; tsc_eqs=[]}
            with Refocus_down_value -> eee
                )
      | Unmarshal(e1,ty1) -> wrap_dest_unary { eee with tsc_ec = CtxUnmarshal(ec, ty1); tsc_expr = e1 }
      | RET _ -> raise (Unimplemented "refocus_down RET")
      | Col(e1,eqs',ty) ->  (
          try refocus_down { eee with tsc_ec = CtxCol(ec, eqs, ty); tsc_eqs = eqs'; tsc_expr = e1 }  (* NB: the *outer* colour goes in the context *)
          with
            Refocus_down_value ->  (
              match get_typ env eqs ty with  (* can't assume types on brackets are fully expanded: consider /\ a.  [  ] ^ a *)
                TXDot(Hash h, etn) when (not(xetn_mem_eqs (Hash h,etn) eqs)) && (xetn_mem_eqs (Hash h,etn) eqs') ->
                  raise Refocus_down_value
              | (TTyCon1 (TRef,t)) -> raise Refocus_down_value
              | (TTyCon1 (TName,t)) -> raise Refocus_down_value
              | _ -> eee)
                )
 (* Just checking... *)
      | OP(_, OEOp(Assign t), _) -> raise (Never_happen "Assign in an OP")
      | OP(_, OEOp(Assign' t), _) -> raise (Never_happen "Assign' in an OP")
      | OP(_, OEOp(Deref t), _) -> raise (Never_happen "Deref in an OP")
      | OP(n,e0,es)   -> refocus_down_list eee es
                           (fun (es1,vs2) -> CtxOP(eqs,n,e0,es1,ec,vs2))
                           (fun () -> eee)
      | Resolve(_)
      | Resolve_blocked(_) -> eee
      | LetrecMulti(_) | LetMulti(_) -> raise (Never_happen "refocus_down {Let,Letrec}Multi")
      | ValOfTie(e') -> wrap_dest_unary {eee with tsc_ec = CtxValOfTie(ec);
                                          tsc_expr=e'}
      | NameOfTie(e')-> wrap_dest_unary {eee with tsc_ec = CtxNameOfTie(ec);
                                          tsc_expr=e'}
      | Tie(mn, iid) -> eee
      | Support(t, e') -> wrap_dest_unary {eee with tsc_ec = CtxSupport(t,ec);
                                          tsc_expr=e'}
      | Freshfor(e1, e2) -> (
          try
            refocus_down {eee with tsc_ec=CtxFreshforL(ec, e2); tsc_expr=e1}
          with Refocus_down_value ->
            (try refocus_down {eee with tsc_ec=CtxFreshforR(e1, ec); tsc_expr=e2}
                with Refocus_down_value -> eee))
      | Swap(e1, e2, e3) -> (
          try
            refocus_down {eee with tsc_ec=CtxSwapL(ec, e2, e3); tsc_expr=e1}
          with Refocus_down_value ->
            (try refocus_down {eee with tsc_ec=CtxSwapM(e1, ec, e3); tsc_expr=e2}
                with Refocus_down_value ->
                            (try refocus_down {eee with tsc_ec=CtxSwapR(e1, e2, ec); tsc_expr=e3}
                with Refocus_down_value -> eee)))
      | NameValue(nv) -> raise Refocus_down_value
      | CFresh(t) -> eee
      | Fresh(t) -> eee
      | Par(e1, e2) -> eee
      | SLOWRET t -> eee
      | Namecase(e', ity, iid1, iid2, e1, e2, e3) -> (
          try refocus_down {eee with tsc_ec = CtxNamecaseL(ec, ity, iid1, iid2, e1, e2, e3);
                             tsc_expr = e'}
          with Refocus_down_value -> (
            try refocus_down {eee with tsc_ec = CtxNamecaseR(e', ity, iid1, iid1, ec, e2, e3);
                               tsc_expr=e1}
                with Refocus_down_value -> eee))
      | HashHts(t1, e1, e2, t2) ->(
          try
            refocus_down {eee with tsc_ec=CtxHashHtsL(eqs, t1, ec, e2, t2); tsc_expr=e1;tsc_eqs=[]}
          with Refocus_down_value -> (try
             refocus_down {eee with tsc_ec=CtxHashHtsR(eqs, t1, e1, ec, t2); tsc_expr=e2;tsc_eqs=[]}
              with Refocus_down_value -> eee)
              )
      | HashTs(t1, e', t2) -> (try
                    refocus_down {eee with tsc_ec=CtxHashTs(eqs, t1, ec, t2); tsc_expr=e';tsc_eqs=[]}
             with Refocus_down_value -> eee)
      | HashMvf(hmn, eid, t) -> eee
      | Pack(t1, e', t2) -> refocus_down {eee with tsc_ec=CtxPack(t1,ec,t2); tsc_expr=e'}
      | Unpack(ity, iid, e1, e2) -> (
           try
            refocus_down {eee with tsc_ec = CtxUnpack(ity, iid, ec, e2);
                           tsc_expr=e1}
          with Refocus_down_value -> eee)




(* helper: try each of a list of expressions *)
and refocus_down_list     : threaded_smallstep_inner  (* this eval ctx *)
                         -> prim_expr list    (* the expressions to check for value-ness *)
                         -> (prim_expr list * prim_value list -> eval_ctx)  (* build a context *)
                         -> (unit -> 'a)        (* success (all es are values) continuation *)
                         -> 'a
    = fun eee es mkCtx success ->
      let rec go es1 vs2 =
        match es1 with
          [] -> success ()
        | (e1::es1') ->
            try
              refocus_down { eee with tsc_ec = mkCtx (es1',vs2); tsc_expr = e1 }
            with
              Refocus_down_value ->
                go es1' (e1::vs2)
      in
      go (List.rev es) []  (* evaluate right-to-left *)



(* refocus_up (ec,eqs,v) looks up through ec for a redex.
   Call only where v is an eqs-value.

   returns (ec',eqs',e') with the redex at the top of e' if it finds one;
   raises ... if ...

   Note need to call refocus_down every so often in the middle of this

   NB FEW OF THE INTERESTING CASES ARE IMPLEMENTED
*)

let rec refocus_up : threaded_smallstep_inner -> threaded_smallstep_inner
    = fun eev ->
      (* Debug.dbgassert (fun () -> (!Opts.evaluator = Opts.smallstep)); *)
      if (!Opts.dumptrace > 1 && !Opts.showfocussing) then
        begin
          let this_config_string = print_threaded_smallstep_inner
              (ps()) eev in
          print_string_really ("\n--refocusing_up -- \n" ^this_config_string ^ "\n" );
        end;

      let ec = eev.tsc_ec in
      let env = eev.tsc_env in
      let eqs = eev.tsc_eqs in
      let v = eev.tsc_expr in
      match ec with
        CtxTop                  -> raise Refocus_up_no_redex
      | CtxC1(c, ec')           -> refocus_up { eev with tsc_ec = ec'; tsc_expr = C1(c, v) }
      | CtxConsL (ec', e')      ->  (
          let eee' = { eev with tsc_ec = CtxConsR (v,ec'); tsc_expr = e' } in
          try refocus_down eee' with
            Refocus_down_value -> refocus_up { eev with tsc_ec = ec'; tsc_expr = Cons(v,e') }
          )
      | CtxConsR (v', ec')      -> refocus_up { eev with tsc_ec = ec'; tsc_expr = Cons(v', v) }
      | CtxTup (es, ec', vs)    ->
          refocus_up_list eev es (v::vs)
            (fun (es1',vs2) -> CtxTup (es1', ec', vs2))
            (fun vs2 -> refocus_up { eev with tsc_ec = ec'; tsc_expr = Tup(vs2) })
      | CtxInEnv (env', ec')    -> { eev with tsc_ec = ec'; tsc_env = env'; tsc_expr = InEnv(env, v) }
      | CtxIf (ec', e1, e2)     -> { eev with tsc_ec = ec'; tsc_expr = If(v, e1, e2) }
      | CtxLazyOp (lo, ec', e') -> { eev with tsc_ec = ec'; tsc_expr = LazyOp(lo, [v;e']) }
      | CtxSeq (ec', e')        -> { eev with tsc_ec = ec'; tsc_expr = Seq(v, e') }
      | CtxAppL (ec', e')       ->  (
          let eee' = { eev with tsc_ec = CtxAppR(v, ec'); tsc_expr = e' }  in
          try refocus_down eee'
          with
            Refocus_down_value -> (
               (* App(v, e') is a redex if v is a Clos, or if it is a saturated application of an econst_id or op *)
              match v with
                Clos(_) ->  { eev with tsc_ec = ec'; tsc_expr = App(v, e') }
              | _ -> raise (Never_happen ("[CtxAppL] App not of fn: " ^  print_prim_expr (pserr()) v ^ " (from "
                                                ^  print_prim_expr (pserr()) (App(v, e')) ^ ")"))
            ))
      | CtxAppR (v', ec')       ->   (
          match v' with
            Clos(_) -> { eev with tsc_ec = ec'; tsc_expr = App(v', v) }
          | _ -> raise (Never_happen ("[CtxAppR] App not of fn: " ^  print_prim_expr (pserr()) v' ^ " (from "
                                            ^  print_prim_expr (pserr()) (App(v',v)) ^ ")"))
                )
      | CtxMatch (ec', m)       -> { eev with tsc_ec = ec'; tsc_expr = Match(v, m) }

      | CtxRaise (ec')          -> { eev with tsc_ec = ec'; tsc_expr = Raise(v) }

      | CtxTry (ec', m)         -> { eev with tsc_ec = ec'; tsc_expr = Try(v, m) }
      | CtxMarshalL (ec', e1,  t)  -> (
          let eee' =      { eev with tsc_ec = CtxMarshalR(v, ec', t); tsc_expr=e1 } in
          try refocus_down eee' with
            Refocus_down_value ->  { eev with tsc_ec = ec'; tsc_expr = Marshal(v, e1, t) }
                )
      | CtxMarshalR (m, ec', t)  -> { eev with tsc_ec = ec'; tsc_expr = Marshal(m, v, t) }

      | CtxMarshalz (eqs, m, ec', t)  -> { eev with tsc_ec = ec'; tsc_expr = Marshalz(m, v, t); tsc_eqs = eqs }

      | CtxUnmarshal(ec', t)    -> { eev with tsc_ec = ec'; tsc_expr = Unmarshal( v, t) }
      | CtxCol (ec', eqs', t)   -> (  (* NB: eqs' is the *outer* colour, and eqs is the colour on the bracket *)
          (* This /may/ be a value *)
          let eee' = { eev with tsc_ec = ec'; tsc_eqs = eqs'; tsc_expr = Col(v, eqs, t) } in
          match get_typ env eqs t with
            TXDot(Hash h, etn) when (not(xetn_mem_eqs (Hash h,etn) eqs')) && (xetn_mem_eqs (Hash h,etn) eqs)
              -> refocus_up eee'  (* Col(v, eqs', t) is a value *)
          | (TTyCon1 (TRef,t)) -> refocus_up eee'
          | (TTyCon1 (TName,t)) -> refocus_up eee'
          | _ ->  eee' )
      | CtxOp (oe, vs, ec', es) ->
          (* refocus_up_list eev es (v::vs)
            (fun (es1',vs2) -> CtxOp(oe, es1', ec', vs2))
            (fun vs2 -> { eev with tsc_ec = ec'; tsc_expr = Op(oe, vs2) }) *)
          (* K: TODO: tidy up: use refocus_up_list *)
          let rec go es vs =
            match es with
              [] -> {eev with tsc_ec=ec'; tsc_expr = Op(oe, List.rev vs)}
            | (e'::es') ->
                let eee' = {eev with tsc_ec = CtxOp(oe, List.rev vs, ec', es'); tsc_expr = e'} in
                try refocus_down eee'
                with
                  Refocus_down_value ->
                    go es' (e'::vs)
          in
          go es (v::(List.rev vs))

      | CtxOP (eqs1, n, e0, es, ec', vs) ->
          refocus_up_list eev es (v::vs)
            (fun (es1',vs2) -> CtxOP(eqs1, n, e0, es1', ec', vs2))
            (fun vs2 -> { eev with tsc_ec = ec'; tsc_eqs = eqs1; tsc_expr = OP(n, e0, vs2) })

      | CtxHashTs(eqs1, t1, ec', t2) ->
           {eev with tsc_expr=HashTs(t1, v, t2);
            tsc_eqs=eqs1; tsc_ec=ec'}
      | CtxHashHtsL(eqs1, t1, ec', e', t2) -> (
          try refocus_down
              {eev with tsc_ec=CtxHashHtsR(eqs,t1, v, ec',t2);
                tsc_expr=e'}
          with Refocus_down_value ->
            {eev with tsc_eqs=eqs1; tsc_expr=HashHts(t1, v, e', t2);
              tsc_ec=ec'})
      | CtxHashHtsR(eqs1, t1, v1, ec', t2) ->
            {eev with tsc_eqs=eqs1; tsc_expr=HashHts(t1, v1, v, t2);
              tsc_ec=ec'}
      | CtxSwapL(ec', e1, e2) -> (
          try refocus_down
              {eev with tsc_expr=e1; tsc_ec=CtxSwapM(v, ec', e2)}
          with Refocus_down_value ->
            (try refocus_down
              {eev with tsc_expr=e2; tsc_ec=CtxSwapR(v, e1, ec')}
            with Refocus_down_value ->
              {eev with tsc_ec = ec';
                tsc_expr = Swap(v, e1, e2)}))
      | CtxSwapM (v1, ec', e2) -> (
          try refocus_down
              {eev with tsc_expr=e2; tsc_ec=CtxSwapR(v1, v, ec')}
          with Refocus_down_value ->
            {eev with tsc_ec = ec';
                tsc_expr = Swap(v1, v, e2)})
      | CtxSwapR(v1, v2, ec') ->
          {eev with tsc_ec = ec'; tsc_expr=Swap(v1, v2, v)}
      | CtxFreshforL(ec', e1) -> (
          try refocus_down
              {eev with tsc_expr=e1; tsc_ec=CtxFreshforR(v, ec')}
          with Refocus_down_value ->
            {eev with tsc_ec=ec';
              tsc_expr=Freshfor(v, e1)})
     |  CtxFreshforR(e1, ec') -> (
            {eev with tsc_ec=ec';
              tsc_expr=Freshfor(e1, v)})
     |  CtxSupport(t, ec') ->
          {eev with tsc_ec=ec';
            tsc_expr=Support(t, v)}
      | CtxNameOfTie(ec') ->{eev with tsc_ec=ec';
            tsc_expr=NameOfTie(v)}
      | CtxValOfTie(ec') ->{eev with tsc_ec=ec';
            tsc_expr=ValOfTie(v)}
      | CtxNamecaseL(ec', ity,iid1, iid2, e1, e2, e3) ->
        (try refocus_down
              {eev with tsc_expr=e1;
                tsc_ec=CtxNamecaseR(v, ity, iid1, iid2, ec', e2, e3)}
          with Refocus_down_value ->
            {eev with tsc_ec=ec';
              tsc_expr=Namecase(v, ity, iid1, iid2, e1, e2, e3)})
     | CtxNamecaseR(v', ity, iid1, iid2, ec', e2, e3) ->
            {eev with tsc_ec=ec';
              tsc_expr=Namecase(v', ity, iid1, iid2, v, e2, e3)}
     | CtxTApp(ec',t) ->
         {eev with tsc_ec=ec'; tsc_expr=TApp(v, t)}
     | CtxPack(t1, ec', t2) ->
         refocus_up {eev with tsc_ec=ec'; tsc_expr=Pack(t1, v, t2)}
     | CtxUnpack(ity, iid, ec', e') ->
         {eev with tsc_ec=ec'; tsc_expr=Unpack(ity, iid, v,e')}
     | CtxStrVal(id, CtxImodule(eqs1, mn, mbody, ec'), loc) -> raise Refocus_up_no_redex
     | CtxStrVal(_) | CtxImodule(_) -> raise (Never_happen("strval or imodule in refocus_up"))






and refocus_up_list : threaded_smallstep_inner  (* this eval ctx *)
                   -> prim_expr list -> prim_value list  (* unchecked expretscons in reverse order, known values in usual order *)
                   -> (prim_expr list * prim_value list -> eval_ctx)  (* build a context *)
                   -> (prim_value list -> 'a)  (* success (all es are values) continuation *)
                   -> 'a
    = fun eev es vs mkCtx success ->
      let rec go es1 vs2 =
        match es1 with
          [] -> success vs2
        | (e1::es1') ->
            let eee' = { eev with tsc_ec = mkCtx (es1', vs2); tsc_expr = e1 } in
            try
              refocus_down eee'
            with
              Refocus_down_value ->
                go es1' (e1::vs2)
      in
      go es vs


let refocus : threaded_smallstep_inner -> threaded_smallstep_inner
    = fun eee ->
      (* Debug.dbgassert (fun () -> (!Opts.evaluator = Opts.smallstep)); *)
      try refocus_down eee with
        Refocus_down_value -> (
          try
            refocus_up eee
          with
            Refocus_up_no_redex ->
           let (_, env, eqs, tv) = pull_context false eee in (* don't put strvals back together *)
           match tv.ti_expr with
              | Raise (_) -> raise Acute_exception
              | _ -> raise Value (* perform check for C0 (B0 Unit) only at thread death point, as module expressions may be anything. *))

let reduce_redex :
  (Ast.smallstep_outer * Ast.threaded_smallstep_inner) ->
    (Ast.smallstep_outer * Ast.threaded_smallstep_inner option)
    = fun (config, inner) ->
      let (defs, senv, s, ec, eqs, ((menv, tenv) as env), e) = (config.scfg_defs, config.scfg_senv, config.scfg_store, inner.tsc_ec, inner.tsc_eqs, inner.tsc_env, inner.tsc_expr) in
      let retn e = (config, Some {inner with tsc_expr = e}) in
      match e with
        Id(i) ->
          (try
            match IIdentMap.find i (fst env) with
              Col(v,eqs',ty') -> retn (maybe_col eqs (v,eqs',ty'))
                | v ->
                    if (!Opts.really_hack_optimise) then retn v else
                    raise (Never_happen "reduce_redex: non-Col binding in environment")
              with
                Not_found ->
                  raise (Never_happen ("reduce_redex: identifier out of scope: "^ print_prim_expr (pserr()) e
                                      ^"\n in inner " ^  print_threaded_smallstep_inner (pserr()) (inner))))
          | If(C0(B0(Bool(true ))),e2,e3)  -> retn e2
          | If(C0(B0(Bool(false))),e2,e3) -> retn e3
          | LazyOp(lop, vs) ->    (* FZ need brackets ? *)
              ( match vs with
              | [C0(B0(Bool(b))) as v1; e2] -> retn (if b = (match lop with
                                                           LoAnd -> false
                                                         | LoOr  -> true) then v1 else e2)
              | _ -> raise (Never_happen "reduce_redex: lazy_op") )
          (* KW: technically, this rule should simply introduce a black bracket around v1, and
             change to a LazyOP.  Once that's reduced, it should do this test.  But I see
             If doesn't do this, and neither does Seq.  I think we are relying on some
             assumed result that brackets don't affect ground types.  Someone care to
             write it down? *)
          | Seq(C0(B0(Unit)),e2)          -> retn e2
          | While(e1,e2) -> retn (If(e1, Seq(e2, While(e1,e2)), C0 (B0 Unit)))
          | Fn(m) -> (match m.desc with ([(PVar(x,ty),e)]) ->  (* build non-rec closure *)
              retn (Clos(Lazy.lazy_from_val env,x,ty,[],e,None))
            | _ ->  raise (Never_happen "reduce_redex: sugared Fn") )
          | TAbs(itn,e) -> (* TCLOS *)

              retn (TClos(env,itn,e))  (* build type closure *)
          | App(Clos(env',x2,ty2,bs2,e1,x1o),v) ->
              assert (isvalue (true, true) eqs v);
              let v = get_val env eqs v in  (* remove tenv dependencies *)
              let (menv',tenv') = Lazy.force env' in
              if (!Opts.really_hack_optimise) then
                retn (InEnv((IIdentMap.add x2 (apply_bs bs2 v) menv', tenv'), e1))
              else
                retn (InEnv((IIdentMap.add x2 (apply_bs bs2 (Col(v,eqs,ty2))) menv', tenv'), e1))  (* Col required so domain of env always has Col *)
          | TApp(TClos((menv',tenv'),itn,e),ty) ->
              let ty = get_typ env eqs ty in  (* remove tenv dependencies *)
              retn (InEnv((menv',ITypnameMap.add itn ty tenv'), e))  (* XXX should we check for colour traversal here as in the unifier in Typecheck? *) (* TCLOS *)
          | Match(v, m) ->
              (match (try Some (matching_eqs eqs v m.desc) with Doesnt_match -> None) with
                Some(env',e2) -> let (menv,tenv) = env in
                retn (InEnv((append_iidmap env' menv,tenv), e2))
              | None -> retn (Raise(C1 (B1 MatchFail, expr_of_loc m.loc)))
                    )
          | Let(v, binding) -> raise (Never_happen "reduce_redex: sugared form (Let)")
          | Letrec(ty1, (x1,(m,e2))) ->  (* build rec closure *)
              let (x2,ty2,e1) = match m.desc with
                [(PVar(x2,ty2),e1)] -> (x2,ty2,e1)
              | _ -> raise (Never_happen "reduce_redex: sugared Letrec") in
              let (menv,tenv) = env in
              (* create a recursive loop (cycle in heap!) *)
              if not (!Opts.really_hack_optimise) then
                let rec v1 = Col(Clos(lazy (IIdentMap.add x1 v1 menv,tenv),x2,ty2,[],e1,Some(x1,ty1,[])),eqs,ty1) in
                let env' = match v1 with Col(Clos(env',_,_,_,_,_),_,_) -> Lazy.force env'
                | _ -> raise (Never_happen "closure not closure") in  (* ensure we pay the cost now *)
                retn (InEnv(env', e2))
              else
                let rec v1 = (Clos(lazy (IIdentMap.add x1 v1 menv,tenv),x2,ty2,[],e1,Some(x1,ty1,[]))) in
                let env' = match v1 with (Clos(env',_,_,_,_,_)) -> Lazy.force env'
                | _ -> raise (Never_happen "closure not closure") in  (* ensure we pay the cost now *)
                retn (InEnv(env', e2))

          | Try(v, m) -> retn v
          | Marshal(C0(B0(String mk)), e, t) -> let t' = get_typ env eqs t in retn (Marshalz(mk, maybe_col [] (e, eqs, t'), t'))
          | Marshalz(mk, e, t) ->let t' = get_typ env eqs t in
            let (menv, tenv) = env in
            retn (marshal config mk (get_val env [] e) (get_typ env eqs t'))
          | Unpack(itn,x1,Pack(ty1,e1,exty2),e2) ->
              let exty2 = get_typ env eqs exty2 in
              let (itn',ty2) =
                match exty2 with  (* not tested in a real implementation *)
                | TExists(itn',ty2) -> (itn',ty2)
                | _ -> raise (Never_happen "reduce_redex: unpack of non-existential-typed value")
              in
              if !Opts.abstract_existentials then begin
                raise (Unimplemented "unpack, abstractly")
              end else begin
              (* first, the non pack-abstraction-preserving version *)
                let (menv,tenv) = env in
                let ty2' = swap_prettyname itn itn' ty2 in  (* XXX inefficent *)
                let menv' =
                if !Opts.really_hack_optimise then
                  IIdentMap.add x1 e1 menv
                else
                  IIdentMap.add x1 (Col(e1,eqs,ty2')) menv
                in
                let tenv' = ITypnameMap.add itn ty1 tenv in
                retn (InEnv((menv',tenv'), e2))
              end
          | Namecase(Pack(ty1,Tup([nv1;v2]),exty2),itn,x1,x2,nv2,e2,e3) ->
              let exty2 = get_typ env eqs exty2 in
              let (itn',ty2a,ty2b) =
                match exty2 with  (* not tested in a real implementation *)
                | TExists(itn',TTup[ty2a;ty2b]) -> (itn',ty2a,ty2b)
                | _ -> raise (Never_happen "reduce_redex: namecase of non-existential-typed value")
              in
              let equal = (compare_name "Namecase" nv1 nv2 = 0) in
              Debug.print' Opts.DBC_namecase (fun () -> "namecase: " ^ print_prim_expr (ps()) nv1 ^ "=?" ^ print_prim_expr (ps()) nv2 ^ "==> " ^ string_of_bool equal);
              if equal then begin  (* names match *)
                if !Opts.abstract_existentials then begin
                  raise (Unimplemented "namecase, abstractly")
                end else begin
                  Debug.print' Opts.DBC_namecase (fun () -> "namecase: matches OK");
                  (* first, the non pack-abstraction-preserving version *)
                  let (menv,tenv) = env in
                  let (ty2a',ty2b') = swap_prettyname itn itn' (ty2a,ty2b) in
                  let menv' =
                  if !Opts.really_hack_optimise then
                     IIdentMap.add x2 v2 (IIdentMap.add x1 nv1 menv)
                  else
                  IIdentMap.add x2 (Col(v2,eqs,ty2b')) (IIdentMap.add x1 (Col(nv1,eqs,ty2a')) menv)
                  in
                  let tenv' = ITypnameMap.add itn ty1 tenv in
                  retn (InEnv((menv',tenv'), e2))
                end
              end else begin  (* names don't match *)
                Debug.print' Opts.DBC_namecase (fun () -> "namecase: doesn't match");
                retn e3
              end
          | Col(e', eqs', t) ->  (* BRACKET PUSHING: *)
             (
              Debug.print (fun () -> "reduce_redex Col: "^(print_prim_expr (ps()) e));
              assert ((isvalue (true, true) eqs' e'));
              let t0 = get_typ env eqs t in
              if (!Opts.really_hack_optimise) then retn e' else

              match (t0, e') with
                (TXDot(Hash h, etn), _) ->
                  if (* h.t \notelementof dom (eqs') *)
                    not(xetn_mem_eqs (Hash h, etn) eqs')
                  then (
                    (* bracket elimination *)
                    match e' with Col(e'', eqs'', TXDot(Hash h'', etn'')) when
                      hetn_eq (h, etn)
                        (h'', etn'') ->
                          Debug.print(fun () -> ("Bracket elimination\n"));
                          retn (maybe_col eqs (e'', eqs'', TXDot(Hash h'', etn'')))
                    | _ ->
                        raise (Stuck ("reduce_step Col Hashdot [1]: expr of "
                                      ^ print_prim_expr (pserr()) e
                                      ^ " eqs of "
                                      ^ print_eqs (pserr()) eqs))
                          )
                  else (* h.t \elementof dom (eqs') *)
                    (
                    try
                      let t' = the (lookup_eqs_xetn eqs (Hash h, etn)) in
                      Debug.print(fun () -> ("Bracket revelation\n"));
                      retn (maybe_col eqs (e', eqs', t'))
                    with _ -> raise (Stuck ("reduce_step Col Hashdot [2]: expr of "
                                            ^ print_prim_expr (pserr()) e
                                            ^ " eqs of "
                                            ^ print_eqs (pserr()) eqs)))
              | ((TTyCon1 (TOption,t)), C0(NONE t')) -> retn (C0(NONE t))
              | ((TTyCon1 (TOption,t)), C1((SOME),v)) -> retn (C1((SOME),maybe_col eqs (v, eqs', t)))
              | ((TTyCon1 (TTree, t)), C1(Node,v)) -> retn (C1(Node,maybe_col eqs (v, eqs', TTup [t; TTyCon1(TList, TTyCon1(TTree, t))] )))
              | ((TTyCon1 (TList,t)), C0(Nil t')) -> retn (C0(Nil t))
              | ((TTyCon1 (TList,t)),Cons(v1, vs)) ->
                  retn (Cons(maybe_col eqs (v1, eqs', t), maybe_col eqs (vs, eqs', TTyCon1 (TList,t))))
              | (t, C0(c)) -> (
                  if typ_eq (get_typ env eqs t) (typeof_C0 c) then retn (C0 c)
                  else raise (Stuck ("reduce_step Col C0: " ^ print_typ (pserr()) (get_typ env eqs t) ^ " is not " ^ print_typ (pserr()) (typeof_C0 c)
                                     ^ " in expr: " ^ print_prim_expr (pserr()) e
                                     ^ " with env " ^ print_etysubst (pserr()) env
                                )                                                                    ))
              | (t, C1(c1, e')) ->
                  let (ty1,ty2)=typeof_C1 c1 in
                  if typ_eq t ty2 then retn (C1(c1, maybe_col eqs (e', eqs', ty1)))
                  else  raise (Stuck "reduce_step Col C1")
              | (TTup(ts),Tup(es)) ->
                     retn (Tup(List.map2 (function t -> function v -> maybe_col eqs (v, eqs', t)) ts es))
              | (TFunc(ty2',ty3'),Clos(env'',x2,ty2,bs2,e1,x1o)) ->
                  let x1o' = match x1o with
                    None -> None
                  | Some(x1,ty1,bs1) -> Some(x1,TFunc(ty2',ty3'),maybe_cons_bs eqs (eqs',ty1) bs1) in
                  retn (Clos(env'',x2,ty2',maybe_cons_bs eqs (eqs',ty2) bs2,maybe_col eqs (e1,eqs',ty3'),x1o'))
              | (TForall(itn',ty3'),TClos(env'',itn,e1)) -> (* TCLOS *)
                  let ty3'' =  swap_prettyname itn' itn ty3' in
                  retn (TClos(env'',itn,maybe_col eqs (e1,eqs',ty3'')))

              | (TExists(itn, ty2), Pack(ty0, v, ty1)) ->
                  retn (Pack(ty0, maybe_col eqs (v,eqs', tsub_typ [itn, ty0] ty2), TExists(itn, ty2)))

                    (* (env, Pack(ty0, v, ty1)) *)
              | (tother,other) ->
                  raise (Stuck ("reduce_step Col: expression is " ^ print_prim_expr (pserr()) e ))
                    )
          | Raise(v) -> (
              match ec with
                CtxTry(ec', m) -> (
                  try
                    let (subst, ex) = matching_eqs eqs v m.desc in
                    let (menv,tenv) = env in
                    (config, Some {inner with tsc_ec=ec'; tsc_expr=InEnv((append_iidmap subst menv,tenv), ex)})
                      with
                    Doesnt_match -> (config, Some {inner with tsc_ec=ec'; tsc_expr=Raise(v)}))
              | CtxStrVal(_, CtxImodule(eqs1, _, _, ec'), _) ->
                  (config, Some {inner with tsc_ec=ec'; tsc_expr=Raise(v); tsc_eqs=eqs1}) (* No real need to keep the eqs as the thread will go anyway, but...? *)
              | CtxInEnv _ ->
                  let tsc = delete_innermost_context (inner) in
                  (config, Some {inner with tsc_ec=tsc.tsc_ec; tsc_eqs=tsc.tsc_eqs; tsc_env=tsc.tsc_env; tsc_expr=Raise(get_val env eqs v)})  (* remove tenv deps *)
              | _ ->
                  let tsc = delete_innermost_context (inner) in
                  (config, Some {inner with tsc_ec=tsc.tsc_ec; tsc_eqs=tsc.tsc_eqs; tsc_env=tsc.tsc_env; tsc_expr=Raise(v)}) (* TODO: should this be get_val too? *)
                        )
          | Op(OEOp(Deref t), [v]) -> (
              match v with
              | Col(v', eqs', t'0) ->
                  (match get_typ env eqs t'0 with
                  | TTyCon1 (TRef, t') ->  (* not tested in a real implementation *)
                      retn (maybe_col eqs (Op(OEOp(Deref t'), [v']), eqs', t'))
                  | _ -> raise (Never_happen "Deref of value with incorrectly-typed bracket"))
              | Loc(l) -> (
                  try
                    retn( List.assoc l s)
                  with Not_found -> raise (Stuck "tid reduce_op"))
              | _ -> raise (Never_happen "Deref not of (coloured) location")
                    )

          | Op(OEOp(Assign t), [l;v']) -> (
              match l with Col(l', eqs', t'0) ->
                (match get_typ env eqs t'0 with
                | TTyCon1 (TRef,t') ->  (* not tested in a real implementation *)
                    retn( maybe_col eqs (Op(OEOp(Assign t'), [l'; maybe_col eqs' (v', eqs, t')]), eqs', (TTyCon0 TUnit)))
                | _ -> raise (Never_happen "Assign to location with incorrectly-typed bracket"))
              |  _ ->
                  let e' = maybe_col [] (v', eqs, t) in
                  retn( Op(OEOp(Assign' t), [l;e'])))
          | Op(OEOp(Assign' t), [Loc(l);v']) ->
              let s' = update_store l (get_val env eqs v') s in
              ({config with scfg_store=s'}, Some {inner with tsc_expr=C0 (B0 Unit)})
          | Op(oe, vs) ->
              let tys = fst (info_of_op_or_econst oe) in
              let arity = List.length tys in
              let es = List.map2 (fun v ty -> maybe_col [] (v,eqs,ty)) vs tys in
              retn (OP(arity, oe, es))

          | App(_, _) -> raise (Never_happen("App of Op"))

          | HashDot(h, eid) -> raise (Unimplemented "step reduce: HashDot")
          | RET t -> raise (Never_happen "step reduce: RET")
          | SLOWRET t -> raise (Never_happen "step reduce: SLOWRET in runnable")
          | Unmarshal(C0(B0(String s)), t) -> let (config', e') = unmarshal config s (get_typ env eqs t)
              in
            (config', Some ({inner with tsc_expr=e'}))
                  (* Paranoid checking... *)
          | OP(n, OEOp(Deref t), [l]) -> raise (Never_happen "OP containing deref")
          | OP(n, OEOp(Assign t), [l;e']) -> raise (Never_happen "OP containing assign")
          | OP(n, OEOp(Assign' t), [l;e']) -> raise (Never_happen "OP containing assign' ")
          | OP(n, OEOp(op), es) ->
              let (conf', expr) = reduce_op (true, true) op es inner config in (
                match expr with
                |Some e ->
                    (conf', Some({inner with tsc_expr = e}))
                | None ->
                    (conf' , None)
              )
          | OP(n, OEEconst(i), es) ->(
              let r = Econst.apply i inner.tsc_name (List.map (get_val env eqs) es) in
              (match r with
                SLOWRET(t) ->
                  let config' = {config with scfg_slowcall = (NameValueMap.add inner.tsc_name () config.scfg_slowcall);
                                  scfg_threads = NameValueMap.add inner.tsc_name {inner with tsc_state = TsSlowcall} config.scfg_threads}
                  in
                  (config', None)
              | _ -> (config, Some({inner with tsc_expr=r} ))))
          | HashMvf(Hash h, eid, t) ->
              let nv = Pretty.mkAbstrNameMvf (h,eid,get_typ env eqs t) in
              (config, Some {inner with tsc_expr=NameValue(nv)})
          | HashMvf(Modname m, eid, t) ->
              let (ext, int) = m in
              raise (Never_happen ("modname in compiled expr: " ^ print_prim_expr (pserr()) e))
          | HashTs(t1, C0 (B0(String s)), t2) ->
              if not (typ_eq_with tenv t1 t2) then
                raise (Never_happen "types not equal in HashTs - perhaps canonicalisation is needed after all?") else  (* I thought it wasn't needed here, probably *)
              let nv = Pretty.mkAbstrNameTs (get_typ env eqs t1,s) in
               (config, Some {inner with tsc_expr=NameValue(nv)})
          | HashHts(t1, C0 (B0(String s)), NameValue n, t2) ->
              if not (typ_eq_with tenv t1 t2) then
                raise (Never_happen "types not equal in HashHts - perhaps canonicalisation is needed after all?") else  (* I thought it wasn't needed here, probably *)
              let nv = Pretty.mkAbstrNameHts (get_typ env eqs t1,s,n) in
               (config, Some {inner with tsc_expr=NameValue(nv)})

          | Fresh(t) ->
              let (ne', n) = new_name_value config.scfg_nenv (get_typ env eqs t) in
              ({config with scfg_nenv=ne'},
               Some {inner with tsc_expr = NameValue n})
          | CFresh _ -> raise (Never_happen "CFresh in running code")
          | Swap(v1, v2, v3) ->  (* TODO: XXX very slow; better to use real swap if possible; needs locations to be names *)
              let (n,tn)  = get_name_value "Swap(1)" v1 in
              let (n',tn') = get_name_value "Swap(2)" v2 in
              let locs = reachable_locs v3 config.scfg_store in
              let locs' = List.map (function l -> (l, fresh_location())) locs in
              let swapnames n n' eqs e =
                match e with
                | NameValue a ->
                    if (name_value_compare a n) = 0 then Some(NameValue n') else
                    if (name_value_compare a n') = 0 then Some(NameValue n) else
                    None
                | Loc(l) -> Some (Loc(List.assoc l locs'))
                |_ -> None
              in
              let store' = config.scfg_store @
                List.map
                  (function (l, l') ->
                    (l',
                     (early_emap_prim_expr false (swapnames n n') (Some [])
                        (flatten_all_closures [] (List.assoc l (config.scfg_store))) )))
                  locs'  (* TODO: is black right for the eqs? *)  (* K thinks yes *)
              in
              let senv' = config.scfg_senv @
                List.map
                  (function (l, l') ->
                    (l', (List.assoc l (config.scfg_senv))))
                  locs'
              in
              let v3' = early_emap_prim_expr false (swapnames n n')  (Some eqs) (flatten_all_closures [] v3) in
              ({config with scfg_store=store'; scfg_senv=senv'}, Some {inner with tsc_expr=v3'})

          | Freshfor(v0, v) ->
              let (n,tn) = get_name_value "Freshfor" v0 in
              let locs = reachable_locs v config.scfg_store in
              let ranlocs =
                List.map (function l -> List.assoc l config.scfg_store) locs
              in
              let fn = (free_names v) @
                List.flatten (List.map (function e -> free_names v) ranlocs)
              in
              (config, Some {inner with tsc_expr = C0 (B0 (Bool (List.mem n fn)))})
          | Support(t, v) -> (* TODO: should we be doing typeenv substitution on v? *)
              let locs = reachable_locs v config.scfg_store in
              let nset =
                List.map (function l -> List.assoc l config.scfg_store) locs
              in
              let fn = (free_names (flatten_all_closures eqs v)) @
                List.flatten (List.map (function e -> free_names (flatten_all_closures [] e)) nset) (* TODO: eqs here, or black? *)
              in
              let nfset = List.filter (function n ->
                typ_eq (type_of_name_value n ) (get_typ env eqs t)) fn    (* TODO: how is this implemented in a non-type-passing impl?? *)
              in
              let rec build_filtered_list l =
                match l with
                | [] -> C0 (Nil (TTyCon1 (TName, t)))
                | [x] -> Cons(NameValue x, C0 (Nil (TTyCon1 (TName, t))))
                | (x::y::ys) when x = y -> build_filtered_list (x::ys)
                | (x::xs) -> Cons(NameValue x, (build_filtered_list xs))
              in
              let filter_dups = build_filtered_list (List.sort (compare) nfset)
              in
              (config, Some {inner with tsc_expr=filter_dups})
          | Dot(mn, eid) ->
              let (config', e') =
             (* print_endline("About to resolve definition"); *)
                resolve_definition eqs config mn eid in
             (*print_endline(fst mn ^ "." ^ print_ident_ext (pserr()) eid ^ "Resolved to " ^ print_prim_expr (pserr()) e'); *)
             (config', Some {inner with tsc_expr=e'} )
          |  Resolve(e,mn,rs)-> Debug.print' Opts.DBC_linkok(fun () -> "resolving from rs");( match rs with
              [] -> (config, Some {inner with tsc_expr=Raise(C0 (B0 ResolveFail))})
            |  Resolve_static_link::rs' ->
                (config, Some {inner with tsc_expr=Raise(C0 (B0 ResolveFail))})
            |  Resolve_here_already::rs' ->
                let (config', e') = resolve_here_already config e mn rs' in
		(config', Some {inner with tsc_expr=e'})
	    |  (Resolve_url(s))::rs'->
		Threadpool.add (Inr(inner.tsc_name, s));
		({config with
		   scfg_slowcall = NameValueMap.add inner.tsc_name () config.scfg_slowcall;
		   scfg_threads = NameValueMap.add inner.tsc_name
		     {inner with
		       tsc_state = TsResolveBlocked;
		       tsc_expr =  Resolve_blocked(e, mn, rs')} config.scfg_threads},
		 None)
		  )
          | Resolve_blocked(e,mn,rs)  -> raise (Never_happen "reduce_redex : resolve_blocked") (* shouldn't be runnable *)
          | NameOfTie(C1 (TieCon, Tup[v1; v2])) -> (config, Some {inner with tsc_expr=v1})
          | ValOfTie(C1 (TieCon, Tup[v1; v2])) -> (config, Some {inner with tsc_expr=v2})
          | Tie(mn, eid) ->
              raise (Unimplemented ("reduce_redex: tie"))

          | InEnv((menv',tenv'), v) ->
              if not (isvalue (true, true) eqs v) then
                raise (Never_happen ("expression inenv not a value: "
                                     ^ Pretty.print_prim_expr (ps()) v ^ " ---in--- "
                                     ^ Pretty.print_threaded_smallstep_inner (ps()) inner))
              else
                (config, Some {inner with tsc_expr=get_val (menv',tenv') eqs v;})
                (* TODO ugh - I don't like this, but it's
                necessary: there could still be free tyvars even in a
                value, that need to be substituted out, and this is an
                easy and correct way of doing that *)

          | _ -> raise(Unimplemented ("unknown redex in reduce_redex : " ^
                                      Pretty.print_prim_expr (pserr()) inner.tsc_expr))

let reduce_expr = fun (config, inner) ->
  let inner' =
    try
      let r = refocus (inner) in r
    with Stack_overflow -> prerr_endline("Stack overflowed[rf]"); raise Stack_overflow
  in
  if !Opts.showfocussing then (
    begin
      let this_config_string =
        print_smallstep_outer (ps()) {config with scfg_runnable = AQueue.add inner.tsc_name config.scfg_runnable} in
      print_string_really (this_config_string ^ "\n--refocused to-- \n");
      let this_config_string =
        print_smallstep_outer (ps()) {config with scfg_runnable = AQueue.add inner'.tsc_name config.scfg_runnable}in
      print_string_really (this_config_string ^ "\n\n-->\n")
    end;
  );
      eval_trace_or_typecheck config;

      reduce_redex (config, inner')



(* Hm, care with moving expr from current to waiting & back *)

let reduce_defs_expr (conf, sc) =
    (* insert various run/debug output here *)

  match sc.tsc_next_expr with Some e -> (
    (* need to evaluate thing on tsc_expr before moving on *)
    try
      reduce_expr (conf, sc)
    with Value -> (* don't forget, refocus will tell you things are values without the value needing to have been at redex.
                     only option is to trust it and pull before testing anything.
                     However, our pulling constructs a typecheck-able module, and we just want the value inside the strval.
                  *)
      (
   let (ec, env, eqs, v) = pull_context false sc in
      match ec with
        CtxStrVal((eid,iid), ec', loc) -> (
          match ec' with CtxImodule(eqs1, mn, mbody, ec'') -> (
            (* subst *)
            let str_item = primtoexpr (flatten_all_closures [] v.ti_expr)
            in
            let strs = mbody.ctx_mc_str_todo in


            let strs' = List.map (function s ->
                match s.desc with StrVal((eid1, iid1) as id1, str_item1) ->
                  let str_item1' =
   (* print_endline("GOING TO MAP");*)
                   early_emap_expr false
                   (fun eqs e ->
                   match e.desc with LocId(i) when i=iid -> Some {str_item with loc=e.loc}
                   | _ -> None) None str_item1


                  in
   (*print_endline("DONE: " ^ print_ident_ext (pserr()) eid1 ^ ": " ^ print_expr (pserr()) str_item1');*)
                  {s with desc=StrVal(id1, str_item1')}
                | s' -> s) strs
            in
            (* shunt *)
            let def' =
              Mod_imod(mn,
                       {
                       mi_hash = mbody.ctx_mc_hash;
                       mi_vubs = mbody.ctx_mc_vubs;
                       mi_eqs = mbody.ctx_mc_eqs;
                       mi_sign0 = mbody.ctx_mc_sign0;
                       mi_sign1 = mbody.ctx_mc_sign1;
                       mi_vn = mbody.ctx_mc_vn;
                       mi_str_done=({loc=loc; desc=StrVal((eid, iid), str_item)})::mbody.ctx_mc_str_done;
                       mi_str_todo=strs';
                       mi_str_loc = loc})
            in
            match sc.tsc_next_expr with
              None ->
                (conf, Some {sc with tsc_defs=def'::sc.tsc_defs})
            |   Some e ->
                (conf, Some {sc with tsc_defs=def'::sc.tsc_defs; tsc_expr=e;tsc_next_expr=None; tsc_ec=CtxTop; tsc_env=env;tsc_eqs=[];}) (* TODO: must be black? *)
                  )
          |     _ -> raise (Never_happen ("Not in imodule when we should be: " ^ Pretty.print_threaded_smallstep_inner (pserr()) sc))
                )

      | _ -> raise (Never_happen ("Not in strvalwhen we should be:"  ^ Pretty.print_threaded_smallstep_inner (pserr()) sc))
            )
      | Acute_exception ->
          let (ec, env, eqs, v) = pull_context false sc in
          !Debug.thread_exit_hook (Some v.ti_expr);
          (if (!Opts.printerrordeath) then
            let s = "2- Thread "^(Pretty.print_name_value (ps_empty_debug()) sc.tsc_name)^" raised exception: " ^
              Pretty.print_prim_expr (ps_empty_debug()) v.ti_expr
              ^ "\n" (* Get thread name? Get time? Add lots of stars and dire warnings?*)
            in
            let ch = Dump.safe_open_out true in
            output ch s 0 (String.length s);
            flush ch;
            Dump.safe_close_out ch
          );
          ({conf with scfg_threads = NameValueMap.remove sc.tsc_name conf.scfg_threads}, None) (* Thread deleted *)
      )
      | _ ->



  match sc.tsc_defs with [] ->
    (
    raise (Never_happen "Called reduce_defs_expr with empty defs")
      )

  | ((Mod_fresh(_) as def)::defs)
  | ((Import_fresh(_) as def)::defs) ->
      let m = ModnameMap.empty in
      let mdefs = List.fold_right  (* XXX inefficient *)
          (function a -> function m ->
            match a with
              Mod_compile(mn, _)
            | Import_compile(mn, _) ->
                ModnameMap.add mn a m
            | Mod_fresh _ | Import_fresh _ | Mod_imod _ ->
                raise (Never_happen "fresh modules in evaluated defs list")
            | Mark_compile _ -> m
                )
          conf.scfg_defs m in
      let (ne', def') =
          Hashify.hashify_fresh_def conf.scfg_nenv mdefs def { term_valuability = Nonvaluable ; type_valuability = Nonvaluable ; }
      in
      let def1 =
        match def' with Mod_compile((eid,iid) as mn, mbody) ->
          (* prerr_endline("constructing imod " ^ eid); *)
          Mod_imod(mn,
                   {mi_hash=mbody.mc_hash;
                     mi_eqs=mbody.mc_eqs;
                     mi_vubs=mbody.mc_vubs;
                     mi_sign0=mbody.mc_sign0;
                     mi_sign1=mbody.mc_sign1;
                     mi_vn=mbody.mc_vn;
                     mi_str_loc=mbody.mc_str.loc;
                     mi_str_done=[]; (*mbody.mc_str.desc; (* []; *) (* M swapped these two for DEBUGGING ONLY. SWAP BACK *)*)
                     mi_str_todo=mbody.mc_str.desc;
                   })

 (* make imodule *)
        | _ -> def'
      in
      ({conf with scfg_nenv=ne'}, Some ({sc with tsc_defs=(def1::defs)}))

  | (Mod_imod(mn, mcbody) as def)::defs -> (
      (* mcbody contains two lists of structure items *)

      let donelist = mcbody.mi_str_done in
      let todolist = mcbody.mi_str_todo in
      let loc = mcbody.mi_str_loc in

      match todolist with
        [] ->
          let def' = Mod_compile(mn,
                                 {mc_hash=mcbody.mi_hash;
                                   mc_vubs=mcbody.mi_vubs;
                                   mc_eqs=mcbody.mi_eqs;
                                   mc_sign0=mcbody.mi_sign0;
                                   mc_sign1=mcbody.mi_sign1;
                                   mc_vn=mcbody.mi_vn;
                                   mc_str=
                                   {loc=mcbody.mi_str_loc;
                                     desc=(List.rev mcbody.mi_str_done)}})
          in
          let e' =
          match sc.tsc_next_expr with Some e' -> e' | None -> sc.tsc_expr
          in

          ({conf with scfg_defs=
             conf.scfg_defs@[def']},
           Some {sc with tsc_defs=defs;
                  tsc_expr=e';
                  tsc_next_expr=None;
                } )
      | (s::strs) ->
         (* if !str_ref = None then () else
          (let (Some s') = !str_ref in assert (s = s'); str_ref := None); *)
          (match s.desc with
        | StrVal(((eid, iid) as id), str_item) -> (
            (* prerr_endline("Doing " ^ print_ident_ext (pserr()) eid); *)
           let ctx = CtxImodule(sc.tsc_eqs, mn,
                                 {
                                 ctx_mc_hash = mcbody.mi_hash;
                                 ctx_mc_vubs = mcbody.mi_vubs;
                                 ctx_mc_eqs = mcbody.mi_eqs;
                                 ctx_mc_sign0 = mcbody.mi_sign0;
                                 ctx_mc_sign1 = mcbody.mi_sign1;
                                 ctx_mc_vn = mcbody.mi_vn;
                                 ctx_mc_str_done=donelist;
                                 ctx_mc_str_todo=strs;
                                 ctx_mc_str_loc = loc}, sc.tsc_ec)
                in
            try
                (* ATTEMPT 1: one-step, as it should be... *)
              reduce_expr
                  (conf,
                   {sc with tsc_ec = CtxStrVal(id, ctx, s.loc);
                     tsc_eqs = mcbody.mi_eqs; (* TODO: Check *)
                     tsc_expr = exprtoprim (str_item);
                     tsc_next_expr = Some(sc.tsc_expr);
                     tsc_defs = (defs)
                   })

            with Value ->
           (* substitute *)
                              let strs' =
              List.map (function s ->
                match s.desc with StrVal((eid1, iid1) as id1, str_item1) ->
                  let str_item1' = primtoexpr (
                    esub_expr
                      (
                    (IIdentMap.add iid (exprtoprim str_item) IIdentMap.empty), ITypnameMap.empty)
                      [] (exprtoprim str_item1)) (* TODO: use the right eqs.*)
                  in
                  {s with desc=StrVal(id1, str_item1')}
                | s' -> s) strs
              in
           (* shunt *)

              let def' = Mod_imod(mn, {mcbody with
                                        mi_str_todo=strs';
                                        mi_str_done=s::(mcbody.mi_str_done)})
              in
              (
              match sc.tsc_next_expr with
                None ->
              (conf, Some {sc with tsc_defs=def'::defs})
              | Some e ->
                  (conf, Some {sc with tsc_defs=def'::defs; tsc_expr=e;tsc_next_expr=None; tsc_eqs=[] }) (* TODO: check eqs *)
                  )
            |   Acute_exception ->
                !Debug.thread_exit_hook (Some (exprtoprim str_item));
                (if (!Opts.printerrordeath) then
                  let s = "3- Thread "^(Pretty.print_name_value (ps_empty_debug()) sc.tsc_name)^" raised exception: " ^
                    Pretty.print_prim_expr (ps_empty_debug()) (exprtoprim str_item)
                    ^ "\n" (* Get thread name? Get time? Add lots of stars and dire warnings?*)
                  in
                  let ch = Dump.safe_open_out true in
                  output ch s 0 (String.length s);
                  flush ch;
                  Dump.safe_close_out ch
                  );
                ({conf with scfg_threads = NameValueMap.remove sc.tsc_name conf.scfg_threads}, None) (* Thread deleted *)
                  )
        | StrTyp(_) -> (* TODO:  does this count as a reduction step? *)
            let def' = Mod_imod(mn, {mcbody with
                                      mi_str_todo=strs;
                                      mi_str_done=s::(mcbody.mi_str_done)})
            in
            (conf, Some {sc with tsc_defs=def'::defs})
        | StrValMulti(_) -> raise (Never_happen "strvalmulti in evaluator: should have been desugared")
              )
            )
        (* TODO: evaluate e? *)


   | Mod_compile((eid,iid) as mn, mbody)::defs' ->
       let def1 =
          (* prerr_endline("constructing imod " ^ eid); *)
          Mod_imod(mn,
                   {mi_hash=mbody.mc_hash;
                     mi_eqs=mbody.mc_eqs;
                     mi_vubs=mbody.mc_vubs;
                     mi_sign0=mbody.mc_sign0;
                     mi_sign1=mbody.mc_sign1;
                     mi_vn=mbody.mc_vn;
                     mi_str_loc=mbody.mc_str.loc;
                     mi_str_done=[];
                     mi_str_todo=mbody.mc_str.desc;
                   })

 (* make imodule *)
      in
      (conf, Some ({sc with tsc_defs=(def1::defs')}))



  | d::defs' -> ({conf with scfg_defs=conf.scfg_defs @ [d]},
                            Some {sc with tsc_defs=defs'})






let reduce_process_once conf0  =

(* insert various run/debug output here *)


   let p = conf0.scfg_runnable in
   try (
     let (n, p') = AQueue.take p in
     let conf = {conf0 with scfg_runnable=p'} in (* TODO: OK to leave it in threads? *)
     let hd =
       try
       NameValueMap.find n conf.scfg_threads
       with Not_found -> raise (Never_happen ("thread " ^ print_name_value (ps()) n ^ " not in scfg_threads[1]: SCFG is " ^ print_smallstep_outer (Dump.fresh_printer_state 1 0 false) conf0))
     in
     let (conf', pnew) =
       match (hd.tsc_defs, hd.tsc_next_expr) with
         ((d::defs),_) -> reduce_defs_expr (conf,hd)
       | (_, Some _) -> reduce_defs_expr (conf,hd)
       | _ ->
         (
         try
           reduce_expr (conf, hd)
         with
           Value -> (* put back together and make assertions on the thread_info *)
             let (_, env, eqs, tv) = pull_context true hd in
             !Debug.thread_exit_hook (Some tv.ti_expr);

             (if not (!Opts.nonunitthread) then
               if not ( env = empty_etysubst && eqs = [] && tv.ti_expr = C0 (B0 Unit) )
               then raise (Never_happen ("Didn't end with a unit, but nonunitthread not specified"))

                   ); (* [Distinguished threads] Or if it's not unit, don't delete it? Or? *)
             (* We just delete the thread (check it's unit, perhaps?)*)
             (* Given relevant opt, print that thread exited cleanly to stdout file *)
             if (!Opts.printcleandeath || (!Opts.nonunitthread)) then (
               let s = "Thread exited cleanly: \n"  (* Get thread name? Get time? *)
                 ^ Pretty.print_thread_info (Dump.fresh_printer_state (if !Opts.dumpfinal=0 then 1 else !Opts.dumpfinal) !skipdefs_ref false) tv
                 ^ "\n"
               in
               let ch = Dump.safe_open_out true in
               output ch s 0 (String.length s);
               flush ch;
               Dump.safe_close_out ch
                 );
             let conf' = {conf with scfg_threads = NameValueMap.remove hd.tsc_name conf.scfg_threads} in
             (conf', None)
         | Acute_exception ->
             let (_, env, eqs, tv) = pull_context true hd in
             if not ( env = empty_etysubst && eqs = [] )
             then () (* raise (Never_happen("pulled context not empty: eqs are " ^ Pretty.print_eqs (pserr()) eqs
                                       ^ " and etysubst is " ^ Pretty.print_etysubst (pserr()) env
                                       ^ "for expression " ^ Pretty.print_prim_expr (pserr()) tv.ti_expr
                                       ^ " orginal eqs was " ^ Pretty.print_eqs (pserr()) hd.tsc_eqs
                                       ^ " original env was " ^ Pretty.print_etysubst (pserr()) hd.tsc_env
                                       ^ " original expr was " ^ Pretty.print_prim_expr (pserr()) hd.tsc_expr
                                       ^ " original ti was " ^ Pretty.print_threaded_smallstep_inner (pserr()) hd
                                                                  )) *)
             ;
             !Debug.thread_exit_hook (Some tv.ti_expr);

             if (!Opts.printerrordeath) then (
               match tv.ti_expr with
               | Raise(v) ->
                   let s = "Thread "^(Pretty.print_name_value (ps_empty_debug()) hd.tsc_name)^" raised exception: "
                     ^ Pretty.print_prim_expr (ps_empty_debug()) (v)
                     ^ "\n" (* Get thread name? Get time? Add lots of stars and dire warnings?*)
                   in
                   let ch = Dump.safe_open_out true in
                   output ch s 0 (String.length s);
                   flush ch;
                   Dump.safe_close_out ch
               | _ -> raise (Never_happen("Acute exception not a raise"))
             );
             (* We just delete the thread (check it's unit, perhaps?)*)
             (* Given relevant opt, print that thread exited on exception to std{out,err?} file *)
	     let conf' = {conf with scfg_threads = NameValueMap.remove hd.tsc_name conf.scfg_threads} in
	     (conf', None)
       )
     in
     match pnew with
       None -> Some conf'
     | Some expr ->
         let queue' = AQueue.add n (conf'.scfg_runnable) in
         let threads' = NameValueMap.add n expr conf'.scfg_threads in
         Some ({conf' with scfg_runnable=queue'; scfg_threads=threads'})
           )
   with AQueue.Empty ->
       None (* either everything is blocked, or we're done. reduce_lots deals with it *)


let reduce_process_lots conf =
    (* insert run/debug output here *)
    (* how do we use the queue things here?
       Just put & take?
    *)
try
  let rec reducing =
    fun sconf ->

      (* Do this check each time *)
      let sconf' =
        (match (Threadpool.fetch false (* don't block *)) with
        Some (Inl(tid, expr)) ->
          let te = try NameValueMap.find tid sconf.scfg_threads
          with Not_found -> raise (Never_happen("Slowcall entry not in scfg_threads"))
          in
          let te' = {te with tsc_expr = expr; tsc_state = TsRunnable} in

          let sconf1 =  {sconf with
                          scfg_runnable = AQueue.add tid sconf.scfg_runnable;
                          scfg_threads = NameValueMap.add tid te' sconf.scfg_threads;
                          scfg_slowcall= (NameValueMap.remove tid sconf.scfg_slowcall)
                        }
          in sconf1
	| Some (Inr(tid, ropt)) -> (
	    match ropt with None ->
	      let te = try NameValueMap.find tid sconf.scfg_threads
	      with Not_found -> raise (Never_happen("Resolveblocked entry not in scfg_threads"))
	      in
	      let expr' =
		match te.tsc_expr with Resolve_blocked(e, mn, rs) -> Resolve(e, mn, rs)
		| _ -> raise (Never_happen("resolveblocked entry doesn't contain resolveblocked"))
	      in
	      let te' = {te with tsc_expr = expr'; tsc_state = TsRunnable} in
	      let sconf1 =  {sconf with
                              scfg_runnable = AQueue.add tid sconf.scfg_runnable;
                              scfg_threads = NameValueMap.add tid te' sconf.scfg_threads;
                              scfg_slowcall= (NameValueMap.remove tid sconf.scfg_slowcall)
                            }
              in sconf1
	    | Some (ne', defs') ->
		let te = try NameValueMap.find tid sconf.scfg_threads
		with Not_found -> raise (Never_happen("Resolveblocked entry not in scfg_threads"))
		in match te.tsc_expr with
		  Resolve_blocked(e, mn, rs) ->
		    let sconf', expr' = resolve_URI sconf e mn rs (ne', defs') in
		    let te' = {te with tsc_expr = expr'; tsc_state=TsRunnable} in
		    let sconf1 =
		      {sconf' with
                        scfg_runnable = AQueue.add tid sconf.scfg_runnable;
                        scfg_threads = NameValueMap.add tid te' sconf.scfg_threads;
                        scfg_slowcall= (NameValueMap.remove tid sconf.scfg_slowcall)
                      }
		    in sconf1
		| _ -> raise (Never_happen "Resolveblocked entry doesn't contain resolveblocked")
		      )
        | None -> sconf)
      in

      (* Do whatever reduction *)
      match reduce_process_once sconf'
      with None -> (
        if sconf'.scfg_threads = NameValueMap.empty
        then sconf'
        else
          if
            sconf'.scfg_slowcall = NameValueMap.empty
          then raise (Deadlock (pull_contexts sconf'))
          else
            begin
              match  (Threadpool.fetch true (* block until found something *)) with
                Some (Inl(tid, expr)) ->
                  let te = try
                    NameValueMap.find tid sconf'.scfg_threads
                  with Not_found -> raise (Never_happen ("slowcall entry not in scfg_threads"))
                 in
                  let te' =  {te with tsc_state = TsRunnable; tsc_expr = expr} in
                  reducing({sconf' with
                             scfg_threads  = NameValueMap.add tid te' sconf'.scfg_threads;
                             scfg_runnable = AQueue.add tid sconf'.scfg_runnable;
                             scfg_slowcall = NameValueMap.remove tid sconf'.scfg_slowcall})
	      | Some (Inr(tid, ropt)) -> (
		  match ropt with None ->
		    let te = try NameValueMap.find tid sconf.scfg_threads
		    with Not_found -> raise (Never_happen("Resolveblocked entry not in scfg_threads"))
		    in
		    let expr' =
		      match te.tsc_expr with Resolve_blocked(e, mn, rs) -> Resolve(e, mn, rs)
		      | _ -> raise (Never_happen("resolveblocked entry doesn't contain resolveblocked"))
		    in
		    let te' = {te with tsc_expr = expr'; tsc_state = TsRunnable} in
		    let sconf1 =  {sconf with
				    scfg_runnable = AQueue.add tid sconf.scfg_runnable;
				    scfg_threads = NameValueMap.add tid te' sconf.scfg_threads;
				    scfg_slowcall= (NameValueMap.remove tid sconf.scfg_slowcall)
                            }
		    in reducing sconf1
		  | Some (ne', defs') ->
		      let te = try NameValueMap.find tid sconf.scfg_threads
		      with Not_found -> raise (Never_happen("Resolveblocked entry not in scfg_threads"))
		      in match te.tsc_expr with
			Resolve_blocked(e, mn, rs) ->
			  let sconf', expr' = resolve_URI sconf e mn rs (ne', defs') in
			  let te' = {te with tsc_expr = expr'; tsc_state=TsRunnable} in
			  let sconf1 =
			    {sconf' with
                              scfg_runnable = AQueue.add tid sconf.scfg_runnable;
                              scfg_threads = NameValueMap.add tid te' sconf.scfg_threads;
                              scfg_slowcall= (NameValueMap.remove tid sconf.scfg_slowcall)
			    }
			  in reducing sconf1
		      | _ -> raise (Never_happen "Resolveblocked entry doesn't contain resolveblocked")
			    )
	      | None -> raise (Never_happen ("blocking fetch returned without fetching"))
            end
                    )
      | Some nsconf ->
          reducing nsconf

  in
  let sconf = souter_of_conf conf in
  let sconf_ret = reducing sconf in

  let conf_ret = pull_contexts sconf_ret in
  (* check yada yada *)
  assert (
  conf_ret.cfg_slowcall = NameValueMap.empty
    (* && Could be mutexes & cvars lying around,
       check that they have empty queues? *)
   );
  conf_ret
with Not_found -> raise (Failure "odd: something in reduce_process_once raised Not_found")

let create_tid (ne, (defs, expr)) =
  let (ne', n) = new_name_value ne (TTyCon0 TThread)  in
  (ne', {
   ti_name  = n;
   ti_state = TsRunnable;
   ti_defs  = defs;
   ti_expr  = expr;
 } )


let execute_program ((ne : nameenv),  ((defs, expr) : program))
    = (* let (defs,e') = lose_the_it_variable prog in *)

(* nasty hack *)
  Threadpool.get_URI := getURI;
  let (ne', td) = create_tid (ne, (defs, expr)) in
  let t' = NameValueMap.add td.ti_name td NameValueMap.empty in
  let r' = AQueue.add td.ti_name (AQueue.create ()) in
   let conf = {
     cfg_nenv = ne';
     cfg_defs = [];
     cfg_store = initial_store;
     cfg_senv = initial_store_env;
     cfg_runnable = r';
     cfg_slowcall = Ast.NameValueMap.empty;
     cfg_threads = t';
     cfg_mutexes = Ast.NameValueMap.empty;
     cfg_cvars = NameValueMap.empty
   } in
   try
     let conf' = reduce_process_lots conf
     in Exceptions.Success conf'
   with Deadlock config -> Exceptions.Deadlock config

