(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * (Pretty?) printer for expression trees.                        *
 *                                                                *
 * Version: $Id: pretty.ml,v 1.626 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* local modules *)
open Format
open Ast
open Util
open Basecon

(* == string set, and code to find the next safe string == *)

module OrderedString = struct
  type t = string
  let compare = Pervasives.compare
end

module StringSet = Set.Make(OrderedString)

let find_safe_string : bool -> StringSet.t -> string -> string
  = fun canon set prefix
 -> let make i =
      if not (!Opts.dumptex || !Opts.marshaltex) then
        prefix ^ "_" ^ string_of_int (i-1)
      else
        prefix ^ "\\ttsub{" ^ string_of_int (i-1) ^ "}"
    in
    (* this more efficient implementation does a binary search up then down *)
    let rec attempt_up i
      = if StringSet.mem (make i) set then
          attempt_up (i*2)
        else
          attempt_down (i/2) i
    and attempt_down i j  (* know i is in set, and j is not *)
      = if i+1 = j then
          make j
        else
          let ij = (i+j)/2 in
          if StringSet.mem (make ij) set then
            attempt_down ij j
        else
            attempt_down i ij
    in
    if StringSet.mem prefix set
    || (not canon && !Opts.suffixall)
    (* start from x_0 rather than x if requested and not in canonical mode *)
    then
      attempt_up 1
    else
      prefix

(* matches safe strings returned by find_safe_string, other than those
   cases where find_safe_string is merely the identity *)
let safe_string_regexp = Str.regexp ".*_\\(0\\|[1-9][0-9]*\\)$"


(* == pretty-printer state == *)

type printer_mode = {
  pm_full : bool;  (* true if printing the whole thing,
                      false for an abbreviated form with hash(...) instead of hashes *)
  pm_canon : bool; (* true if using canonical variable names,
                      false if using user's variable names *)
  pm_wide : bool;  (* true if using infinite width (single-line output),
                      false for normal linebreaks
                      [linebreaks and formatting choices must not affect hashes] *)
  pm_trail : bool; (* true if hashes should be extracted and listed in trailer *)
  pm_lithash : bool;  (* true if hashes and names should be printed as literals *)
  pm_dumplevel : int;  (* as described in opts.ml *)
  pm_enbrace_defns : bool; (* true if definitions are surrounded by { } in a config *)
  pm_ignore_defns : int option; (* either Some n, to ignore the first n definitions,
                                       or None, to ignore all definitions.
                        This is reset in pp_print_configuration based on the dumplevel *)
  pm_closed : bool;  (* true if free names (idents, tyvars) should be a fatal error *)
  pm_hackvars : bool; (* true if Var, TVar shouldn't be printed with {? around them *)
  pm_tex : bool;      (* true if extracted hashes should be printed in \textit{ } *)
  pm_human : bool;    (* true if type annotations are not to be printed *)
}


(* the pp_print_program and pp_print_configuration pay attention to
the pm_dumplevel.  pp_print_definition pays attention to the
pm_ignore_defns, decrementing it as appropriate with None infinite. *)

let pm_user dumplevel definitions_lib_size dumptex =
  {
   pm_full           = true;
   pm_canon          = false;
   pm_wide           = false;
   pm_trail          = true;
   pm_lithash        = false;
   pm_dumplevel      = dumplevel;
   pm_enbrace_defns  = false;
   pm_ignore_defns   = (match dumplevel with
                          4 -> Some 0
                        | 3 -> Some definitions_lib_size
                        | _ -> None);
   pm_closed         = false;
   pm_hackvars       = false;
   pm_tex            = dumptex;
   pm_human          = !Opts.dumphuman;
 }
let pm_hash =
  { pm_full          = true;
    pm_canon         = true;
    pm_wide          = true;
    pm_trail         = false;
    pm_lithash       = true;
    pm_dumplevel     = 4;
    pm_enbrace_defns = true;
    pm_ignore_defns  = Some 0;
    pm_closed        = true;
    pm_hackvars      = false;
    pm_tex           = false;
    pm_human         = false;
  }
let pm_lib_hack =
  { pm_full          = true;
    pm_canon         = true;
    pm_wide          = true;
    pm_trail         = false;
    pm_lithash       = true;
    pm_dumplevel     = 4;
    pm_enbrace_defns = true;
    pm_ignore_defns  = Some 0;
    pm_closed        = false;
    pm_hackvars      = true;
    pm_tex           = false;
    pm_human         = false;
  }
let pm_marshal () =
  { pm_full          = true;
    pm_canon         = true;
    pm_wide          = false;
    pm_trail         = !Opts.marshaltex (* false *) ;
    pm_lithash       = !Opts.lithash;
    pm_dumplevel     = 4;
    pm_enbrace_defns = true;
    pm_ignore_defns  = Some 0;
    pm_closed        = false;
    pm_hackvars      = false;
    pm_tex           = !Opts.marshaltex;
    pm_human         = false;
  }
let pm_debug () =
  { pm_full          = true;
    pm_canon         = false;
    pm_wide          = false;
    pm_trail         = true;
    pm_lithash       = !Opts.lithash;
    pm_dumplevel     = 4;
    pm_enbrace_defns = false;
    pm_ignore_defns  = Some 0;
    pm_closed        = false;
    pm_hackvars      = false;
    pm_tex           = false;
    pm_human         = false;
  }
let pm_objectcode definitions_lib_size =
  { pm_full          = true;
    pm_canon         = false;
    pm_wide          = false;
    pm_trail         = false;
    pm_lithash       = !Opts.lithash;
    pm_dumplevel     = 4;
    pm_enbrace_defns = false (* irrelevant *);
    pm_ignore_defns  = Some definitions_lib_size;
    pm_closed        = false;
    pm_hackvars      = false;
    pm_tex           = false;
    pm_human         = false;
  }


(* map from hash-of-hashes to names, with generator for new names *)
module HashedHash =
  struct
    type t = Digest.t
    let equal h1 h2 = h1 = h2
    let hash h = Hashtbl.hash h
  end

module HashHash = Hashtbl.Make(HashedHash)

type hashmap =
    { hm_genname : hash -> string;
      hm_map : string HashHash.t;  (* NB: mutable *)
    }

(* generate a fresh map *)
let newhashmap () =
  let r = ref 0 in
  let genname h =

    let s = match external_modname_of_hash h with
              Some emn -> "h"^string_of_int !r^"_"^external_modname_to_string emn
            | None     -> "h"^string_of_int !r
    in
    r := !r + 1; s in
  let map = HashHash.create 300 in
  { hm_genname = genname;
    hm_map = map;
  };;

(* map from hash-of-hashes to names, with generator for new names *)
module AbstrName =
  struct
    type t = Digest.t
    let equal h1 h2 = h1 = h2
    let hash h = Hashtbl.hash h
  end

module HashAbstrName = Hashtbl.Make(AbstrName)

type abstrnamemap =
    { an_genname : abstract_name -> string;
      an_map : string HashAbstrName.t;  (* NB: mutable *)
    }

(* generate a fresh map *)
let newabstrnamemap () =
  let r = ref 0 in
  let genname h =
    let s = "n"^string_of_int !r
    in r := !r + 1; s
  in
  let map = HashAbstrName.create 300 in
  { an_genname = genname;
    an_map = map;
  };;

type printer_state =
    { ie_map : string IIdentMap.t;
      ie_rmap : StringSet.t;
      it_map : string ITypnameMap.t;
      it_rmap : StringSet.t;
      im_map : string ModnameMap.t;
      im_index_map : int EModnameMap.t;
                     (* the index to use when next generating a new internal name string *)
      econst_string_of_ident : Ast.internal_ident -> string;
      ps_mode : printer_mode;
      hashmap : hashmap;
      abstrnamemap : abstrnamemap;
      ppftrail : formatter;  (* a formatter for output that will appear at the end *)
      ppftrailbuf : Buffer.t
    }


let set_ignore_defns ps v = { ps with ps_mode = { ps.ps_mode with pm_ignore_defns = v; }; }

let notrail_HACK ps = { ps with ps_mode = { ps.ps_mode with pm_trail = false }}

(* == buffers == *)

let margin_max = 999999998  (* DO NOT ALTER THIS CONSTANT! *)

let normal_margin = 130 (* 150 *)  (* 72 *)  (* 150 is ok for the paper examples, but this doesn't seem to have any effect thereon *)

(* create a buffer according to a given printer_state *)
let mkppf wide
  = let buf = Buffer.create 80 in
    let ppf = formatter_of_buffer buf in
    (if wide then begin
      (* to get single-line output: set margin to 1 so we get lots of
         newlines; set newline formatter to just print a space; set
         indent formatter to just print a space *)
      pp_set_margin ppf 1;
      let (out,flush,newline,indent) = pp_get_all_formatter_output_functions ppf () in
      pp_set_all_formatter_output_functions ppf out flush (fun () -> out " " 0 1) (fun (_:int) -> out " " 0 1)
    end else begin
      pp_set_margin ppf normal_margin;
    end);
    pp_open_vbox ppf 0;
    (ppf,buf)

(* close a buffer, extracting the contents *)
let endppf (ppf,buf)
  = pp_close_box ppf ();
    pp_print_flush ppf ();
    let s = Buffer.contents buf in
    Buffer.reset buf;
    s

(* create a new trailer formatter *)
let opentrail mode =
  let (ppftrail,ppftrailbuf) = mkppf mode.pm_wide in
  pp_open_vbox ppftrail 0;  (* is closed by flushtrail *)
  (ppftrail,ppftrailbuf);;

(* flush the trailer formatter and put its content onto the main formatter *)
let flushtrail ps ppf =
  pp_close_box ps.ppftrail ();
  let s = endppf (ps.ppftrail, ps.ppftrailbuf) in
  if !Opts.showtrailer && String.length s > 0 then
    (if not ps.ps_mode.pm_tex then
      fprintf ppf "@\n@[<v>where@,%s@]@\n" s
    else
      fprintf ppf "@\n@[<v>\\ttrm{where }@,%s@]@\n" s)
  else
    ()

(* For each call, create a new buffer, use it, and reset it.  This makes the calls reentrant. *)
let wrap f ps x =
  let (ppf,buf) = mkppf ps.ps_mode.pm_wide in
  try
    f ps ppf x;
    flushtrail ps ppf;
    endppf (ppf,buf)
  with
    Failure s ->
      "(Failed: "^s^" -- possibly string over 16MB, buy a new OCaml development team)"
let wrap' f ps x =
  let (ppf,buf) = mkppf ps.ps_mode.pm_wide in
  try
    let ps' = f ps ppf x in
    flushtrail ps ppf;
    (endppf (ppf,buf), ps')
  with
    Failure s ->
      ("(Failed: "^s^" -- possibly string over 16MB, buy a new OCaml development team; ps' invalid)",ps)


(* == more printer state stuff == *)

(* Global hash-to-short-representation map, for printing; not always used *)
let global_hashmap = newhashmap ();;

let global_abstrnamemap = newabstrnamemap ();;

let initial_printer_state econst_string_of_ident_opt mode =
(* TODO: WARNING: the strings returned by econst_string_of_ident must
   never overlap with names generated by find_safe_string.  This is
   because such strings are never inserted into ie_rmap, and thus
   never avoided.  The names to be avoided are ones that end in
   "_[0-9]+".
*)
  let econst_string_of_ident =
    match econst_string_of_ident_opt with
    | None   -> fun (iid, pn) -> raise (Not_found)
    | Some f -> fun id -> try let s = f id in
                          assert (not (Str.string_match safe_string_regexp s 0));
                          s
	with Not_found -> raise Not_found
  in
  let (ppftrail,ppftrailbuf) = opentrail mode in
    { ie_map = IIdentMap.empty;
      ie_rmap = StringSet.empty;
      it_map = ITypnameMap.empty;
      it_rmap = StringSet.empty;
      im_map = ModnameMap.empty;
      im_index_map = EModnameMap.empty;
      econst_string_of_ident = econst_string_of_ident;
      ps_mode = mode;
      hashmap = if !Opts.globalhashmap then global_hashmap else newhashmap ();
      abstrnamemap = if !Opts.globalhashmap then global_abstrnamemap else newabstrnamemap ();
      ppftrail = ppftrail;
      ppftrailbuf = ppftrailbuf;
    }

(* reset all the variable naming maps (i.e., be in empty environment) *)
let empty_scope : printer_state -> printer_state
  = fun ps
 -> { ps with
      ie_map = IIdentMap.empty;
      ie_rmap = StringSet.empty;
      it_map = ITypnameMap.empty;
      it_rmap = StringSet.empty;
      im_map = ModnameMap.empty;
      im_index_map = EModnameMap.empty;
    }

let empty_ident_scope : printer_state -> printer_state
  = fun ps
 -> { ps with
      ie_map = IIdentMap.empty;
      ie_rmap = StringSet.empty;
    }

let empty_typname_scope : printer_state -> printer_state
  = fun ps
 -> { ps with
      it_map = ITypnameMap.empty;
      it_rmap = StringSet.empty;
    }

let paranoia = false

(* check that we're not shadowing the domain or range *)
let safe_to_add_ident key data map =
  if paranoia then
  begin try
    let s = IIdentMap.find key map in Debug.print (fun () -> "safe_to_add: false"); false
  with Not_found -> (Debug.print (fun () -> ("safe_to_add: true, -> "^data^" ok")); true) end
  && not (IIdentMap.fold (fun _ d' b -> b || d' = data) map false)
  else
    true  (* really not worth the cost of walking the entire map! *)
  ;;

let safe_to_add_typname key data map =
  if paranoia then
  begin try
    let s = ITypnameMap.find key map in Debug.print (fun () -> "safe_to_add: false"); false
  with Not_found -> (Debug.print (fun () -> ("safe_to_add: true, -> "^data^" ok")); true) end
  && not (ITypnameMap.fold (fun _ d' b -> b || d' = data) map false)
  else
    true  (* really not worth the cost of walking the entire map! *)
  ;;

(* check that we're not shadowing the domain; range irrelevant for modnames because
   we use the pair (external,internal) for uniqueness, not just internal *)
let safe_to_add_modname key keystr data map =
  if paranoia then
  begin try
    let s = ModnameMap.find key map in Debug.print (fun () -> ("safe_to_add: false: "^keystr^" maps to "^s^" already; "^data^" not possible")); false
  with Not_found -> (Debug.print (fun () -> ("safe_to_add: true, "^keystr^" -> "^data^" ok")); true) end
  else
    true  (* really not worth the cost of walking the entire map! *)
  ;;

let add_modname_binder ps m =
  let ext = modname_ext m in
  let (_, (_,instr)) = m in (* internal modname string
                                 = "M", for user modules
                                 = "Lib_" ^ ext, for hacked library modules *)
  let extstring = external_modname_to_string ext in
  let intindex = try EModnameMap.find ext ps.im_index_map with Not_found -> 0 in
  let intstring =
  if instr = "Lib_" ^ extstring then (* hacked library module *)
    instr
  else if not (ps.ps_mode.pm_tex) then
    "M" ^ string_of_int intindex
  else
    "M" ^ "\\ttsub{" ^ string_of_int (intindex) ^ "}" in
  Debug.print (fun ()->("pretty: add_modname_binder: assert"));
  (if (not (safe_to_add_modname m extstring intstring ps.im_map)) then
    print_endline ("Weird: can't add map for "^extstring));
  assert (safe_to_add_modname m extstring intstring ps.im_map);
  { ps with im_map = ModnameMap.add m intstring ps.im_map;
  im_index_map = EModnameMap.add ext (intindex + 1) ps.im_index_map; };;

let add_ident_binder ps ((_,prefix) as ident) =
  try
    let _ = ps.econst_string_of_ident ident in ps
  with Not_found ->
    let prefix' = if ps.ps_mode.pm_canon then "x" else prefix in
    let s = find_safe_string ps.ps_mode.pm_canon ps.ie_rmap prefix' in
    assert (safe_to_add_ident ident s ps.ie_map );
    {ps with ie_map = IIdentMap.add ident s ps.ie_map;
             ie_rmap = StringSet.add s ps.ie_rmap; };;

let add_ident_binders ps idents =
  List.fold_left add_ident_binder ps idents

let add_typname_binder ps ((_,prefix) as typname) =
  let prefix' = if ps.ps_mode.pm_canon then "t" else prefix in
  let s = find_safe_string ps.ps_mode.pm_canon ps.it_rmap prefix' in
  assert (safe_to_add_typname typname s ps.it_map );
  {ps with it_map = ITypnameMap.add typname s ps.it_map;
           it_rmap = StringSet.add s ps.it_rmap; };;

let add_typname_binders ps typnames =
  List.fold_left add_typname_binder ps typnames

let add_esubst ps re =  (* disposes of current ident scope first *)
  let ps0 = empty_ident_scope ps in
  add_ident_binders ps0 (List.map fst (iidmap_to_list re))

let add_tysubst ps re =  (* disposes of current typname scope first *)
  let ps0 = empty_typname_scope ps in
  add_typname_binders ps0 (List.map fst (itnmap_to_list re))

let add_etysubst ps (menv,tenv) =  (* disposes of current ident and typname scopes first *)
  add_esubst (add_tysubst ps tenv) menv

exception Cannot_happen
exception Pretty_fatal of string
(* standard modules *)
open String

(* let utf8 = ref false*)
(* let polyprint = ref false *)

let indent_level = ref 1

let pp_ibreak ppf () = pp_print_break ppf 1 (!indent_level)
let pp_icut   ppf () = pp_print_break ppf 0 (!indent_level)

(* guide to reading (see standard library Format documentation for more info):

   @[<options> ... @]  pretty-printing box
                         <hv>: horizontal-vertical box
                         <v>: vertical box
                         <hov>: horizontal or vertical box
   @   (at-space)      possible break with no indent
   %a                  insert result of external format here
                         - takes two arguments, f and v, and
                           invokes f ppf v at this point
                           where ppf is the current formatter.

   Note that fprintf here is Format.fprintf, not Printf.fprintf!

*)

let pp_print_esc_string ppf s = pp_print_string ppf ("\"" ^  String.escaped s ^ "\"")

(* concatenate the results of applying f to ppf and each element of
   xs, interpolating the results of applying fsep to ppf and xsep
   between each adjacent pair of elements.  Like String.concat. *)
let pp_concat ppf fsep xsep f xs =
  let rec go xs = match xs with
    [] -> ()
  | [x] -> f ppf x
  | (x::xs) -> f ppf x; fsep ppf xsep; go xs
  in
  go xs

(* concatenate as above, but threading ps through *)
let pp_concat_ps ps ppf fsep xsep f xs =
  let rec go ps xs = match xs with
    [] -> ps
  | [x] -> f ps ppf x
  | (x::xs) -> let ps' = f ps ppf x in
               fsep ppf xsep;
               go ps' xs
  in
  go ps xs

(* concatenate as above, but threading ps through and expecting f to return an additional boolean specifying whether a separator is needed after this and threading through a polymorphic y*)
let pp_concat_ps_b ps ppf fsep xsep f xs y =
  let rec go ps xs y = match xs with
    [] -> (y,ps)
  | [x] -> let (y',b,ps') = f ps ppf x y in (y',ps')
  | (x::xs) ->
      let (y',b,ps') = f ps ppf x y in
      if b then fsep ppf xsep else ();
      go ps' xs y'
  in
  go ps xs y

(* a formatter that simply throws the output away *)
let null_formatter = make_formatter (fun _ _ _ -> ()) (fun _ -> ())

(* An operator needs brackets around it if it would otherwise be parsed as an infix operator.
   In lexer.mll, this is the case if it starts with one of the following characters followed
   by a string of symbolchar. It makes a few random keywords infix ops for no good reason.
    Adding the brackets never hurts, so this is a conservative estimate. *)
let op_needs_brackets op =
  let first = String.get op 0 in
       List.mem first ['='; '<'; '>'; '|'; '&'; '$'; '@'; '^'; '+'; '-'; '*'; '/'; '%']
    || List.mem op ["mod"; "land"; "lor"; "lxor"; "lsr"; "asr"]

let opt_parens br ppf f x =
  if br then
    fprintf ppf "@[(@[<hov>%a@])@]" f x
  else
    f ppf x ;;

let annot : 'a prettyname -> string
  = fun i
 -> if !Opts.shownames then
      "(*" ^ string_of_name (name_of_prettyname i) ^ "*)"
    else
      ""

(* print identifiers and module names. *)
let pp_print_ident_use ps ppf ident =
  Debug.print (fun () -> ("pretty: pp_print_ident_use: ident = " ^ string_of_name (name_of_prettyname ident)));
  pp_print_string ppf
    begin try IIdentMap.find ident ps.ie_map ^ annot ident with
      Not_found ->
        begin try "%\"" ^ ps.econst_string_of_ident ident ^ "\"" with
          Not_found ->
	    (let s =
            if ps.ps_mode.pm_hackvars then string_of_prettyname ident
		else
	      ("{:"^string_of_prettyname ident^"?:}" ^ annot ident) in
            if ps.ps_mode.pm_closed then
              raise (Util.Never_happen ("Closed entity (mode pm_hash) contains free identifier "^s))
            else
              s)
        end
    end

let pp_print_typname_use ps ppf typname =
  Debug.print (fun () -> ("pretty: pp_print_typname_use: typname = " ^ string_of_int (Obj.magic typname : int)));
  pp_print_string ppf
    (try ITypnameMap.find typname ps.it_map ^ annot typname
    with Not_found ->
      (let s =
	if ps.ps_mode.pm_hackvars then string_of_prettyname typname
	    else
	  ("{:"^string_of_prettyname typname^"?:}" ^ annot typname) in
      if ps.ps_mode.pm_closed then
        raise (Util.Never_happen ("Closed entity (mode pm_hash) contains free tyvar "^s))
      else
        s));;

let pp_print_modname ps ppf modname =
  Debug.print (fun () -> ("pretty: pp_print_modname: (" ^ external_modname_to_string (modname_ext modname) ^ ", " ^ string_of_int (Obj.magic (modname_int modname) : int) ^ ")"));
  let ext = external_modname_to_string (modname_ext modname) in
  let inter = try ModnameMap.find modname ps.im_map ^ annot (modname_int modname) with
                Not_found ->
                  (let s =
		    if ps.ps_mode.pm_hackvars then string_of_prettyname (modname_int modname)
		    else
		      ("{:"^string_of_prettyname (modname_int modname)^"?:}" ^ annot (modname_int modname)) in
                  if ps.ps_mode.pm_closed then
                    raise (Util.Never_happen ("Closed entity (mode pm_hash) contains free module name "^s))
                  else
                    s) in
  let inter_unsafe = string_of_int (Obj.magic (modname_ext modname) : int) in
  Debug.print (fun () -> ("pretty: pp_print_modname: found(" ^ ext ^ ", " ^ inter ^ ")"));
  pp_print_string ppf (ext ^ "[" ^ inter ^ (* "," ^ inter_unsafe ^ *) "]")

let pp_print_ident_ext ps ppf identext = pp_print_string ppf (external_ident_to_string identext)
let pp_print_typname_ext ps ppf typnameext = pp_print_string ppf (external_typname_to_string typnameext)

let pp_print_ident_bind ps ppf identbind =
  fprintf ppf "%a[%a]"
    (pp_print_ident_ext ps) (ident_ext identbind)
    (pp_print_ident_use ps) (ident_int identbind)

let pp_print_typname_bind ps ppf typnamebind =
  fprintf ppf "%a[%a]"
    (pp_print_typname_ext ps) (typname_ext typnamebind)
    (pp_print_typname_use ps) (typname_int typnamebind)

(* store locations *)
let pp_print_location ps ppf l = pp_print_string ppf ("{<" ^ string_of_int l ^ ">}")

(* source locations *)
let pp_print_source_loc ps ppf l = pp_print_string ppf
    (Location.print () l)


let canonicalise_eqs = ref None  (* will be filled in by Tysupp; avoid recursive dependency *)

(* x1; x2; x3; ...; xn *)
let pp_print_queue f ppf (q: 'a AQueue.t) =
  let l = AQueue.toList q in
  pp_open_hovbox ppf 0;
  pp_concat ppf fprintf ";@ " f l;
  pp_close_box ppf ()

(* k1: v1; k2: v2; ...; kn: vn *)
let pp_print_namevaluemap ps f ppf m =
  let myf (k: Ast.name_value) v xs = (k,v)::xs in
  let l = NameValueMap.fold myf m [] in
  pp_open_hovbox ppf 0;
  pp_concat ppf fprintf ";@ " (f ps) l;
  pp_close_box ppf ()

(* print constructors with arity 0 *)
let rec pp_print_con0 ps ppf c =
  match c with
  | Nil t          -> fprintf ppf "([] %a)" (pp_print_typ_argument ps) t
  | NONE t         -> fprintf ppf "(None %a)" (pp_print_typ_argument ps) t
  | B0 b0          -> pp_print_basecon0 ppf b0

(* print constructors with arity 1 *)
and pp_print_con1 ps ppf c =
  match c with
  | Inj(i,t)        -> fprintf ppf "inj%a%a%a%a"
                         pp_ibreak () pp_print_int i
                         pp_ibreak () (pp_print_typ_argument ps) t
  | SOME            -> pp_print_string ppf "Some"
  | TieCon          -> pp_print_string ppf "TieCon"
  | Node            -> pp_print_string ppf "Node"
  | B1 b1           -> pp_print_basecon1 ppf b1

(* print exactly one match *)
and pp_print_prim_mtch_one precedence ps ppf (p,e) =
  pp_open_hvbox ppf 0;
  let ps' = pp_print_pat_pri ps ppf p in
  pp_print_string ppf " ->"; pp_ibreak ppf ();
  pp_print_prim_exb precedence ps' ppf e;
  pp_close_box ppf ();

(* print matches *)
and pp_print_prim_mtch' precedence ps ppf m =
  let rec go xs = match xs with
    [] -> ()
  | [x] -> pp_print_prim_mtch_one precedence ps ppf x
  | (x::xs) -> pp_print_prim_mtch_one 70 ps ppf x; fprintf ppf "@ | "; go xs
  in
  pp_open_hvbox ppf 0;
  go m;
  pp_close_box ppf ()

and pp_print_prim_mtch precedence ps ppf m =
  pp_print_prim_mtch' precedence ps ppf m.desc (* don't try and print location *)

and pp_print_mtch ps ppf m =
  pp_print_prim_mtch 80 ps ppf (mtchtoprim Location.none m)

(* e1 e2 ... en *)
and pp_print_elist ps ppf t =
  pp_open_hovbox ppf 0;
  pp_concat ppf fprintf " @ " (pp_print_prim_ex 0 ps) t;
  pp_close_box ppf ()

(* e1, e2, ..., en *)
and pp_print_etuple ps ppf t =
  pp_open_hovbox ppf 0;
  pp_concat ppf fprintf ",@ " (pp_print_prim_exb 40 ps) t;
  pp_close_box ppf ()



(* print marshalled values *)
and pp_print_marshalled_value ps ppf (Marshalled_value mb) =
  let ps' = empty_scope ps in
  fprintf ppf "@[<hv>marshalled (%a%a,%a%a)@]"
    pp_icut ()
    (pp_print_marshalled_body ps')
      (mb.mb_ne, mb.mb_defs, mb.mb_storeenv, mb.mb_store, exprtoprim mb.mb_expr)
    pp_ibreak ()
    (pp_print_typ ps') mb.mb_typ

and pp_print_marshalled_body ps ppf (ne, defs, senv, store, expr) =
  if not(ps.ps_mode.pm_tex ) then (
    fprintf ppf "@[<hv>";
    ( match ne with
    | Some ne' ->
        fprintf ppf "%a,@ {"
          (pp_print_nameenv ps) ne';
    | None -> fprintf ppf "_,{" );
    let ps' = pp_print_definitions_pri ps ppf defs in
    fprintf ppf "@ }, {%a},@ {%a},@,"
      (pp_print_loctyplist ps') senv
      (pp_print_store ps') store;
    fprintf ppf  "%a@]" (pp_print_prim_expb ps') expr )
  else (
    fprintf ppf "@[<hv>";
    ( match ne with
    | Some ne' ->
        fprintf ppf "%a,@ \\{"           (* P is boggled by the lack of \ in the output *)
          (pp_print_nameenv ps) ne';
    | None -> fprintf ppf "_,\\{" );
    let ps' = pp_print_definitions_pri ps ppf defs in
    fprintf ppf "@ \\}, \\{%a\\},@ \\{%a\\},@ @,"
      (pp_print_loctyplist ps') senv
      (pp_print_store ps') store;
    fprintf ppf  "%a@]" (pp_print_prim_expb ps') expr )



(* print bracket sequence *)
and pp_print_bs ps ppf bs =
  let go ppf (eqs,ty) =
    fprintf ppf "@[<hv>[]_{ %a@;<1 2>}^{ %a@ }@]"
      (pp_print_eqs_naked ps) eqs
      (pp_print_typ ps) ty
  in
  pp_concat ppf fprintf ".@," go (List.rev bs)

(* convert binops to strings *)
and pp_print_op_ty ps ppf (s,t) =
  fprintf ppf "%a %a"
    pp_print_string s
    (pp_print_typ_argument ps) t

and pp_print_op ps ppf = function
  | Assign t  -> pp_print_op_ty ps ppf (":=",t)
  | Assign' t -> pp_print_op_ty ps ppf (":='",t)
  | Equal t   -> pp_print_op_ty ps ppf ("=",t)
  | Less      -> pp_print_string ppf "<"
  | LessEq    -> pp_print_string ppf "<="
  | Greater   -> pp_print_string ppf ">"
  | GreaterEq -> pp_print_string ppf ">="
  | Mod       -> pp_print_string ppf "mod"
  | Land      -> pp_print_string ppf "land"
  | Lor       -> pp_print_string ppf "lor"
  | Lxor      -> pp_print_string ppf "lxor"
  | Lsl       -> pp_print_string ppf "lsl"
  | Lsr       -> pp_print_string ppf "lsr"
  | Asr       -> pp_print_string ppf "asr"
  | Plus      -> pp_print_string ppf "+"
  | Minus     -> pp_print_string ppf "-"
  | Times     -> pp_print_string ppf " * "
 (* FZ: spaces prevent that the star is considered as a comment start or end *)
  | Divide    -> pp_print_string ppf "/"
  | ListConcat t -> pp_print_op_ty ps ppf ("@",t)
  | StrConcat -> pp_print_string ppf "^"
  | Ref t     -> pp_print_op_ty ps ppf ("ref",t)
  | Deref t   -> pp_print_op_ty ps ppf ("!",t)
  | UMinus    -> pp_print_string ppf "-"
  | CompareName t -> pp_print_op_ty ps ppf ("compare_name",t)
  | NameToString t -> pp_print_op_ty ps ppf ("name_to_string",t)
  | CreateThread t -> pp_print_op_ty ps ppf ("create_thread",t)
  | Self      -> pp_print_string ppf "self"
  | Kill      -> pp_print_string ppf "kill"
  | CreateMutex -> pp_print_string ppf "create_mutex"
  | Lock      -> pp_print_string ppf "lock"
  | TryLock   -> pp_print_string ppf "try_lock"
  | Unlock    -> pp_print_string ppf "unlock"
  | CreateCVar -> pp_print_string ppf "create_cvar"
  | Wait      -> pp_print_string ppf "wait"
  | Waiting   -> pp_print_string ppf "waiting"
  | Signal    -> pp_print_string ppf "signal"
  | Broadcast -> pp_print_string ppf "broadcast"
  | Thunkify  -> pp_print_string ppf "thunkify"
  | Unthunkify -> pp_print_string ppf "unthunkify"
  | Exit t    -> pp_print_op_ty ps ppf ("exit",t)

and pp_print_lazyop ps ppf =
  function
    | LoAnd -> pp_print_string ppf "&&"
    | LoOr -> pp_print_string ppf "||"

and pp_print_binop precedence ps ppf myop e1 e2 =
  match op_getprecedence myop with
    Some (my_pr, pr_left, pr_right) -> (* infix operator *)
      opt_parens (precedence < my_pr) ppf (fun ppf () ->
        fprintf ppf "%a%a%a%a%a"
          (pp_print_prim_exb pr_left ps) e1
          pp_ibreak () (pp_print_op ps) myop
          pp_ibreak () (pp_print_prim_exb pr_right ps) e2) ()
  | None -> (* not a infix operator *)
      opt_parens (precedence < 10) ppf (fun ppf () ->
        fprintf ppf "%a%a%a%a%a"
          (pp_print_op ps) myop
          pp_ibreak () (pp_print_prim_exb 0 ps) e1
          pp_ibreak () (pp_print_prim_exb 0 ps) e2) ()

and pp_print_app_op precedence ps ppf myop es =
   match es with
     [e1; e2] when arity_of_op myop = 2 -> pp_print_binop precedence ps ppf myop e1 e2
   | _ when arity_of_op myop > List.length es -> (* not fully applied operators *)
      opt_parens (precedence < 10) ppf (fun ppf () ->
        fprintf ppf "@[(%a)@ %a@]"
          (pp_print_op ps) myop
          (pp_print_elist ps) es) ()
   | _ -> (* fully applied operators *)
      opt_parens (precedence < 10) ppf (fun ppf () ->
        fprintf ppf "@[%a@ %a@]"
          (pp_print_op ps) myop
          (pp_print_elist ps) es) ()

and pp_print_app_op_or_econst precedence ps ppf (oe, es) =
    match oe with
      OEOp myop   -> pp_print_app_op precedence ps ppf myop es
    | OEEconst id -> opt_parens (precedence < 10) ppf (fun ppf () ->
                       fprintf ppf "@[%a@ %a@]"
                         (pp_print_ident_use ps) id
                         (pp_print_elist ps) es) ()


and op_getprecedence myop = match myop with
(* INFIXOP4 *)
  | Lsl | Lsr | Asr       -> Some (21, 20, 21)
(* INFIXOP3 *)
  | Mod | Land | Lor | Lxor | Times | Divide -> Some (23, 23, 22)
(* INFIXOP2 *)
  | Plus | Minus -> Some (25, 25, 24)
(* AT *)
  | ListConcat _ -> Some (29, 28, 29)
(* INFIXOP1 *)
  | StrConcat -> Some (31, 30, 31)
(* INFIXOP0 *)
  | Equal _ | Less | LessEq | Greater | GreaterEq -> Some (33, 33, 32)
(* COLONEQUAL *)
  | Assign _ | Assign' _ -> Some (39, 38, 39)
(* Non-infix operators *)
  | Ref _ | Deref _ | UMinus
  | CompareName _ | NameToString _ | CreateThread _ | Self | Kill
  | CreateMutex | Lock | TryLock | Unlock
  | CreateCVar | Wait | Waiting | Signal | Broadcast
  | Thunkify | Unthunkify | Exit _ -> None

and lazyop_getprecedence = function LoAnd -> 35 | LoOr -> 37


(* Pretty print expressions:
   The precedence argument takes values
     0: simple_expr   (C0, C1, Tup, Loc, Id, Dot, HashDot,
                       Fresh, CFresh, HashMVf, HashTx, HashHts,
                       NameValue, Swap, Support, NameOfTie, ValOfTie)
    10: simple_expr_or_app_ty_list
                      (App, TApp)
    20-40: Binary infix operators       associativity
      21:  INFIXOP4, FRESHFOR                (R)
      23:  INFIXOP3, STAR                    (L)
      25:  INFIXOP2, PLUS, MINUS             (L)
      27:  COLONCOLON                        (R)
      29:  AT                                (R)
      31:  INFIXOP1                          (R)
      33:  INFIXOP0, EQUAL, LESS, GREATER    (L)
      35:  AMPERAMPER                        (R)
      37:  BARBAR                            (R)
      39:  COLONEQUAL                        (R)

    50: if-then-else, while-do-done

    60-80: SEMI, LET, IN, WITH

      65:  let ... in [75]     (Let, LetMulti, Letrec, LetrecMulti, Unpack, Fun)
      70:  64 ; [75]           (Seq, Par)
      75:  match ... with [70] (Fn, Match, Namecase)
        where [nn] = if "expr in brackets" then nn else precedence

   100: ending with type  (Marshal, Unmarshal, Pack)
 *)

and pp_print_prim_exb precedence (ps : printer_state) (ppf : formatter) ex =
   fprintf ppf "@[<hov>%a@]" (pp_print_prim_ex precedence ps) ex

(* code in this routine assumes is in context @[<hov> __ @] *)
and pp_print_prim_ex precedence (ps : printer_state) (ppf : formatter) ex =
  match ex with
    (* simple expressions first *)
    C0(c)   -> pp_print_con0 ps ppf c
  | C1(c,e) -> fprintf ppf "%a%a%a"
                 (pp_print_con1 ps) c
                 pp_ibreak ()
                 (pp_print_prim_exb 0 ps) e
  | Tup(es) -> fprintf ppf "(@[%a@])" (pp_print_etuple ps) es
  | Loc(l)  -> pp_print_location ps ppf l
  | Id i   -> pp_print_ident_use ps ppf i
  | Dot(m,x) -> fprintf ppf "%a.%a"
                  (pp_print_modname ps) m
                  (pp_print_ident_ext ps) x
  | HashDot (h, x) -> fprintf ppf "%a.%a"
                        (pp_print_hash ps) h
                        (pp_print_ident_ext ps) x

  | Raise(e)  -> fprintf ppf "@[raise%a%a@]" pp_ibreak () (pp_print_prim_exb 0 ps) e

  | Fresh(t) ->  fprintf ppf "@[fresh%a%a@]" pp_ibreak () (pp_print_typ_argument ps) t
  | CFresh(t) -> fprintf ppf "@[cfresh%a%a@]" pp_ibreak () (pp_print_typ_argument ps) t

  | HashMvf (hm,x,t) ->
        fprintf ppf "@[hash(%a.%a)%a%a@]"
          (pp_print_hash_or_modname ps) hm
	  (pp_print_ident_ext ps) x
          pp_ibreak ()
          (pp_print_typ_argument ps) t

 | HashTs (t1,e,t2) ->
        fprintf ppf "@[<hv>hash(%a,@ %a)%a%a@]"
          (pp_print_typ ps) t1
          (pp_print_prim_exb 40 ps) e
          pp_ibreak ()
          (pp_print_typ_argument ps) t2

 | HashHts (t1,e1,e2,t2) ->
        fprintf ppf "@[<hv>hash(%a,@ %a,@ %a)%a%a@]"
          (pp_print_typ ps) t1
          (pp_print_prim_exb 40 ps) e1
          (pp_print_prim_exb 40 ps) e2
          pp_ibreak ()
          (pp_print_typ_argument ps) t2

 | NameValue (n) -> pp_print_name_value ps ppf n

 | Swap (e1,e2,e3) ->
        fprintf ppf "@[<hv>swap@ %a@ and@ %a in@ %a@]"
          (pp_print_prim_exb 0 ps) e1
          (pp_print_prim_exb 0 ps) e2
          (pp_print_prim_exb 0 ps) e3

 | Support (t,e) ->
        fprintf ppf "@[<hv>support@ %a@ %a@]"
          (pp_print_typ_argument ps) t
          (pp_print_prim_exb 0 ps) e

 | Tie (m,x) ->
        fprintf ppf "@[<hv>%a@@%a@]"
          (pp_print_modname ps) m
          (pp_print_ident_ext ps) x

 | NameOfTie (e) ->
        fprintf ppf "name_of_tie%a%a" pp_ibreak () (pp_print_prim_ex 0 ps) e

 | ValOfTie (e) ->
        fprintf ppf "val_of_tie%a%a" pp_ibreak () (pp_print_prim_ex 0 ps) e

 (* operators/constants and applications *)
 (* NB: This section is quite complicated/ugly, because we tread many
        special cases of non-eta-expanded operator/econst applications *)

  | App(App(LazyOp(lo,[]), e1), e2) when not !Opts.printclos || ps.ps_mode.pm_canon ->
        let my_pr = lazyop_getprecedence lo in
        opt_parens (precedence < my_pr) ppf (fun ppf () ->
          fprintf ppf "@[%a@ %a %a@]"
            (pp_print_prim_exb (my_pr-1) ps) e1
            (pp_print_lazyop ps) lo
            (pp_print_prim_exb my_pr ps) e2) ()
  | App(LazyOp(lo,[]), e) when not !Opts.printclos || ps.ps_mode.pm_canon ->
        opt_parens (precedence < 10) ppf (fun ppf () ->
          fprintf ppf "@[(%a)@ %a@]"
            (pp_print_lazyop ps) lo
            (pp_print_prim_exb 9 ps) e) ()
  | LazyOp (lo,el) when not ps.ps_mode.pm_canon && !Opts.printclos ->
          fprintf ppf "@[{:LazyOp %a%a%a :}@]"
            (pp_print_lazyop ps) lo
            pp_ibreak ()
            (pp_print_elist ps) el

  | LazyOp (lo,el) -> (match el with
    | [] -> fprintf ppf "(%a)" (pp_print_lazyop ps) lo
    | [e] ->
        opt_parens (precedence < 10) ppf (fun ppf () ->
          fprintf ppf "@[(%a)@ %a@]"
            (pp_print_lazyop ps) lo
            (pp_print_prim_exb 9 ps) e) ()
    | [e1;e2] ->
        let my_pr = lazyop_getprecedence lo in
        opt_parens (precedence < my_pr) ppf (fun ppf () ->
          fprintf ppf "@[%a@ %a %a@]"
            (pp_print_prim_exb (my_pr-1) ps) e1
            (pp_print_lazyop ps) lo
            (pp_print_prim_exb my_pr ps) e2) ()
    | _ -> raise (Util.Never_happen "pp_print_expr:LazyOp"))

  | App(App(App(Op(oe,[]), e1), e2), e3) when (not !Opts.printclos || ps.ps_mode.pm_canon)
                                              && arity_of_op_or_econst oe = 3 ->
        pp_print_app_op_or_econst precedence ps ppf (oe, [e1;e2;e3])

  | App(App(Op(oe,[]), e1), e2) when (not !Opts.printclos || ps.ps_mode.pm_canon)
                                     && arity_of_op_or_econst oe = 2 ->
        pp_print_app_op_or_econst precedence ps ppf (oe, [e1;e2])

  | App(Op(oe,[]), e1) when not !Opts.printclos || ps.ps_mode.pm_canon ->
        pp_print_app_op_or_econst precedence ps ppf (oe, [e1])

  | Op (oe,el) when !Opts.printclos && not ps.ps_mode.pm_canon ->
        fprintf ppf "@[{:Op %a%a%a :}@]"
          (pp_print_op_or_econst ps) oe
          pp_ibreak ()
          (pp_print_elist ps) el

  | Op (oe,el) -> pp_print_app_op_or_econst precedence ps ppf (oe, el)

 (* applications (left-associative) *)
  | App(e1,e2) ->
      opt_parens (precedence < 10) ppf (fun ppf () ->
        fprintf ppf "%a%a%a"
          (pp_print_prim_ex 10 ps) e1
          pp_ibreak () (pp_print_prim_ex  9 ps) e2) ()

  | TApp(e,t) ->
      opt_parens (precedence < 10) ppf (fun ppf () ->
        fprintf ppf "%a%a%a"
          (pp_print_prim_ex 10 ps) e
          pp_ibreak () (pp_print_typ_argument ps) t) ()

 (* other binary operators/constructors *)
  | Freshfor (e1,e2) ->
     opt_parens (precedence < 21) ppf (fun ppf () ->
        fprintf ppf "@[%a@ freshfor@ %a@]"
          (pp_print_prim_exb 20 ps) e1
          (pp_print_prim_exb 21 ps) e2) ()

  | Cons(e1,e2) ->
     opt_parens (precedence < 27) ppf (fun ppf () ->
        fprintf ppf "%a%a:: %a"
          (pp_print_prim_exb 26 ps) e1
           pp_ibreak ()
          (pp_print_prim_exb 27 ps) e2) ()

 (* If/While *)
  | If(e1,e2,e3) -> opt_parens (precedence < 50) ppf (fun ppf () ->
                      fprintf ppf "@[<hv>@[<hv>if%a%a@ then@]%a%a@ else%a%a@]"
                        pp_ibreak ()  (pp_print_prim_exb 40 ps) e1
                        pp_ibreak ()  (pp_print_prim_exb 50 ps) e2
                        pp_ibreak ()  (pp_print_prim_exb 50 ps) e3
                                      ) ()
  | While(e1,e2) -> opt_parens (precedence < 50) ppf (fun ppf () ->
                      fprintf ppf "@[<hv>@[<hv>while%a%a@ do@]%a%a@ done@]"
                        pp_ibreak ()  (pp_print_prim_exb 40 ps) e1
                        pp_ibreak ()  (pp_print_prim_exb 80 ps) e2) ()

(* Sequences, Lets, Matches *)
  | Seq(e1,e2) -> let br  = precedence < 70 in
                  let p75 = if br then 75 else precedence in
                  opt_parens br ppf (fun ppf () ->
                    fprintf ppf "@[<hv>%a; @ %a@]"
                      (pp_print_prim_exb 64 ps) e1
                      (pp_print_prim_exb p75 ps) e2) ()

  | Par(e1,e2) -> let br  = precedence < 70 in
                  let p75 = if br then 75 else precedence in
                  opt_parens br ppf (fun ppf () ->
                    fprintf ppf "@[<hv>%a ||| @ %a@]"
                      (pp_print_prim_exb 64 ps) e1
                      (pp_print_prim_exb p75 ps) e2) ()

  | Let(e1,(p,e2)) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>@[<hv>let%a@[<hv>"
           pp_ibreak ();
       let ps' = pp_print_pat_pri ps ppf p in
       fprintf ppf " =%a%a@]@ in@]@ %a@]"
         pp_ibreak ()
         (pp_print_prim_exb 100 ps) e1
         (pp_print_prim_exb p75 ps') e2) ()

  | LetMulti(t,(pl,e1),(x,e2)) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       opt_parens br ppf (fun ppf () ->
         let ps' = add_ident_binder ps x in
         fprintf ppf "@[<hv>@[<hv>let%a@[<hv>(%a : %a)"
           pp_ibreak ()
           (pp_print_ident_use ps') x
           (pp_print_typ ps) t;
         let ps'' = pp_print_pat_list_ps ps' ppf pl in
         fprintf ppf " =%a%a@]@ in@]@ %a@]"
           pp_ibreak ()
           (pp_print_prim_exb 75 ps') e1
           (pp_print_prim_exb p75 ps'') e2) ()

  | Letrec(t,(i,(m,e))) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       opt_parens br ppf (fun ppf () ->
         let ps' = add_ident_binder ps i in
         fprintf ppf "@[<hv>@[<hv>let rec%a@[<hv>%a : %a =%a@[<hv>function%a%a@]@]@ in@]@ %a@]"
           pp_ibreak ()
           (pp_print_ident_use ps') i
           (pp_print_typ ps) t
           pp_ibreak ()
           pp_ibreak ()
           (pp_print_prim_mtch 80 ps') m
           (pp_print_prim_exb p75 ps') e) ()

  | LetrecMulti(t,(pl,(x,(e1,e2)))) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       opt_parens br ppf (fun ppf () ->
         let ps' = add_ident_binder ps x in
         fprintf ppf "@[<hv>@[<hv>let rec%a@[<hv>(%a : %a)"
           pp_ibreak()
           (pp_print_ident_use ps') x
           (pp_print_typ ps) t;
         let ps'' = pp_print_pat_list_ps ps' ppf pl in
         fprintf ppf " =%a%a@]@ in@]@ %a@]"
           pp_ibreak ()
           (pp_print_prim_exb 75 ps'') e1
           (pp_print_prim_exb p75 ps') e2) ()

  | Unpack(t,x,e1,e2) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       let ps' = add_typname_binder ps t in
       let ps'' = add_ident_binder ps' x in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[let%a@[<hv>{%a@ ,@ %a}@ =%a%a@] in@ %a@]"
           pp_ibreak ()
           (pp_print_typname_use ps') t
           (pp_print_ident_use ps'') x
           pp_ibreak()
           (pp_print_prim_exb 100 ps) e1
           (pp_print_prim_exb p75 ps'') e2 ) ()

  | Fun(pl,e1) ->
       let br  = precedence < 65 in
       let p75 = if br then 75 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>fun ";
         let ps' = pp_print_pat_list_ps ps ppf pl in
         fprintf ppf " ->%a%a@]"
           pp_ibreak () (pp_print_prim_exb p75 ps') e1) ()

  | TAbs(t,e) ->
      let br  = precedence < 65 in
      let p75 = if br then 75 else precedence in
      opt_parens br ppf (fun ppf () ->
        let ps' = add_typname_binder ps t in
        fprintf ppf "Function %a ->@ %a"
          (pp_print_typname_use ps') t
          (pp_print_prim_ex p75 ps') e) ()



  | Fn({desc=[p,e]}) -> (* special case only one match, print less \n *)
       let br = precedence < 75 in
       let precedence' = if br then 80 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "function ";
         let ps' = pp_print_pat_pri ps ppf p in
         pp_print_string ppf " ->"; pp_ibreak ppf ();
         pp_print_prim_exb precedence' ps' ppf e;
(*         pp_close_box ppf ()*)) ()

  | Fn(m) ->
       let br  = precedence < 75 in
       let precedence' = if br then 80 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>function%a%a@]"
           (fun ppf () -> pp_print_break ppf 1 2 (* length of "| " *) ) ()
           (pp_print_prim_mtch precedence' ps) m) ()

  | Match(e1,{desc=[p,e]}) -> (* special case only one match, print less \n *)
       let br  = precedence < 75 in
       let precedence' = if br then 80 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>match %a with "
           (pp_print_prim_exb 80 ps) e1;
         let ps' = pp_print_pat_pri ps ppf p in
         pp_print_string ppf " ->"; pp_ibreak ppf ();
         pp_print_prim_exb precedence' ps' ppf e;
         pp_close_box ppf ()) ()

  | Match(e,m) ->
       let br  = precedence < 75 in
       let precedence' = if br then 80 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>match %a with%a%a@]"
           (pp_print_prim_exb 80 ps) e
           (fun ppf () -> pp_print_break ppf 1 2 (* length of "| " *) ) ()
           (pp_print_prim_mtch precedence' ps) m) ()

  | Try(e,m)  ->
       let br  = precedence < 75 in
       let p70 = if br then 70 else precedence in
       opt_parens br ppf (fun ppf () ->
         fprintf ppf "@[<hv>try%a%a@ with%a%a@]"
           pp_ibreak ()
           (pp_print_prim_exb 80 ps) e
           pp_ibreak ()
           (pp_print_prim_mtch precedence ps) m) ()


  | Namecase(e1,t,x1,x2,e,e2,e3) ->
      let br  = precedence < 75 in
      let p70 = if br then 70 else precedence in
      let ps' = add_typname_binder ps t in
      let ps'' = add_ident_binder ps' x1 in
      let ps''' = add_ident_binder ps'' x2 in
      opt_parens br ppf (fun ppf () ->              (* FZ TODO *)
        fprintf ppf "@[namecase %a with@;  @[<hv>{%a,(%a,%a)} when %a = %a -> %a@]  @[<hv>otherwise -> %a@]@]"
          (pp_print_prim_exb 80 ps) e1
          (pp_print_typname_use ps') t
          (pp_print_ident_use ps'') x1
          (pp_print_ident_use ps''') x2
          (pp_print_ident_use ps'') x1
          (pp_print_prim_exb 32 ps) e  (* less than EQUAL *)
          (pp_print_prim_exb p70 ps''') e2
          (pp_print_prim_exb p70 ps) e3) ()


  (* Expressions ending with type *)
  | Pack(t1,e,t2) -> opt_parens (precedence < 100) ppf (fun ppf () ->
                       fprintf ppf "@[{%a@ ,@ %a}@ as@ %a@]"
                         (pp_print_typ ps) t1
                         (pp_print_prim_exb 80 ps) e
                         (pp_print_typ ps) t2) ()

  | Marshal(e1, e2, t) -> opt_parens (precedence < 100) ppf (fun ppf () ->
                            fprintf ppf "@[<hv>marshal %a%a%a@ : %a@]"
                             (pp_print_prim_exb 0 ps) e1
                             pp_ibreak ()
                             (pp_print_prim_exb 0 ps) e2
                             (pp_print_typ ps) t) ()

  | Marshalz(mk, e2, t) -> opt_parens (precedence < 100) ppf (fun ppf () ->
                           fprintf ppf "@[<hv>marshalz %a%a%a@ : %a@]"
                             (pp_print_esc_string) mk
                             pp_ibreak ()
                             (pp_print_prim_exb 0 ps) e2
                             (pp_print_typ ps) t) ()

  | Unmarshal(e, t) -> opt_parens (precedence < 100) ppf (fun ppf () ->
                         fprintf ppf "@[<hv>unmarshal%a%a@ as %a@]"
                           pp_ibreak ()
                           (pp_print_prim_exb 0 ps) e
                           (pp_print_typ ps) t) ()

 (* Other (atomic) constructs cannot appear in user source programs;
    only the Col case can appear inside marshalled values *)
  | InEnv(env',e) ->
        fprintf ppf "@[{:INENV: ";
        let ps' = add_etysubst ps env' in
        pp_print_etysubst_short ps' ppf env';
        fprintf ppf ",@ %a:}@]"
          (pp_print_prim_exb 40 ps') e

  | Clos(env',x2,ty2,bs2,e1,x1o) ->
        let ((menv',tenv') as env') = Lazy.force env' in
        let env_del = ((match x1o with
          Some(x1,_,_) -> IIdentMap.remove x1 menv'  (* don't recurse forever! *)
        | None -> menv'),tenv') in
        fprintf ppf "@[{:CLOSURE: ";
        let ps' = add_etysubst ps env' in
        pp_print_etysubst_short ps' ppf env_del;
        let ps'' = add_ident_binder ps' x2 in
        fprintf ppf ",@ %a:%a,@ %a,@ %a,@ %a:}@]"
          (pp_print_ident_use ps'') x2
          (pp_print_typ ps'') ty2
          (pp_print_bs ps'') bs2
          (pp_print_prim_exb 40 ps'') e1
          (fun ppf () -> match x1o with
            None -> pp_print_string ppf "nonrec"
          | Some(x1,ty1,bs1) -> fprintf ppf "rec(%a:%a,@ %a)"
                                  (pp_print_ident_use ps'') x1
                                  (pp_print_typ ps'') ty1
                                  (pp_print_bs ps'') bs1) ()

  | TClos(env',itn,e) ->
        fprintf ppf "@[{:TCLOSURE: ";
        let ps' = add_etysubst ps env' in
        pp_print_etysubst_short ps' ppf env';
        let ps'' = add_typname_binder ps' itn in
        fprintf ppf ",@ %a,@ %a:}@]"
          (pp_print_typname_use ps'') itn
          (pp_print_prim_exb 40 ps'') e

  | RET t     -> fprintf ppf "@[{:RET@ %a@ :}@]" (pp_print_typ ps) t
  | SLOWRET t -> fprintf ppf "@[{:SLOWRET@ %a@ :}@]" (pp_print_typ ps) t
  | Resolve_blocked (expr, mn, resolvespec) ->
      fprintf ppf "@[{:RESOLVE_BLOCKED {%a%a@,} {%a%a@,} {%a%a@,} :}@]"
        pp_ibreak ()
        (pp_print_prim_exb 40 ps) expr
        pp_ibreak ()
        (pp_print_modname ps) mn
        pp_ibreak ()
        (pp_print_resolvespec ps) resolvespec
  | Resolve (expr, mn, resolvespec) ->
      fprintf ppf "@[{:RESOLVE {%a%a@,} {%a%a@,} {%a%a@,} :}@]"
        pp_ibreak ()
        (pp_print_prim_exb 40 ps) expr
        pp_ibreak ()
        (pp_print_modname ps) mn
        pp_ibreak ()
        (pp_print_resolvespec ps) resolvespec
  | OP (n, oe, exprs) ->
      fprintf ppf "@[<hv>{:OP {%a}^%d %a{"
        (pp_print_op_or_econst ps) oe
        n
        pp_ibreak ();
      pp_concat ppf
        (fun ppf () -> pp_print_string ppf "}"; pp_ibreak ppf (); pp_print_string ppf "{") ()
        (pp_print_prim_exb 40 ps) exprs;
      fprintf ppf "} :}@]"
  | Col(e,eqs,t) ->
      if ps.ps_mode.pm_human then
        fprintf ppf "@[<hv>[%a%a@ ]_{}^{}@]"
          pp_ibreak () (pp_print_prim_exb 40 ps) e
      else if not (ps.ps_mode.pm_tex) then
        fprintf ppf "@[<hv>[%a%a@ @[<hv>]_{ %a@ }^{ %a@ }@]@]"  (* seem to be enough brackets, maybe don't need more *)
          pp_ibreak () (pp_print_prim_exb 40 ps) e
          (pp_print_eqs_naked ps) eqs
          (pp_print_typ ps) t
      else
        fprintf ppf "@[<hv>\\ttcol{%a@;<1 2>@[<hv>}{\\{%a@;<0 2>\\}}{%a}@]@]"  (* seem to be enough brackets, maybe don't need more *)
          (pp_print_prim_exb 40 ps) e
          (pp_print_eqs_naked ps) eqs
          (pp_print_typ ps) t


and pp_print_abstract_name_really ps ppf n =
  let an = string_of_lithash n in
    an.[1] <- '$';
    pp_print_string ppf (an)

and pp_print_abstract_name ps ppf an =
  if ps.ps_mode.pm_trail then begin
    let hstr =
      try
        HashAbstrName.find ps.abstrnamemap.an_map an
      with
        Not_found ->
          let hstr = ps.abstrnamemap.an_genname an in
          HashAbstrName.add ps.abstrnamemap.an_map an hstr;
          let (ppfsubtrail,ppfsubtrailbuf) = opentrail ps.ps_mode in (* FZ CHECK *)
          let ps' = { ps with ppftrail = ppfsubtrail; ppftrailbuf = ppfsubtrailbuf } in
          if ps.ps_mode.pm_tex then
            fprintf ps.ppftrail "    @[@<0>%s%a@<0>%s = %a@]@ " "\\textit{"
              pp_print_string hstr "}"
              (pp_print_abstract_name_really ps') an
          else
            fprintf ps.ppftrail "    @[%a = %a@]@ "
              pp_print_string hstr
              (pp_print_abstract_name_really ps') an;
          flushtrail ps' ps.ppftrail;
          hstr
    in
    if ps.ps_mode.pm_tex then fprintf ppf "@<0>%s%s@<0>%s" "\\textit{" hstr "}"
                         else pp_print_string ppf hstr
  end else
    pp_print_abstract_name_really ps ppf an

and pp_print_hash_or_modname ps ppf hm =
  match hm with
    Hash h -> pp_print_hash ps ppf h
  | Modname m -> pp_print_modname ps ppf m

and pp_print_op_or_econst ps ppf oe =
  match oe with
    OEOp op     -> pp_print_op ps ppf op
  | OEEconst id -> pp_print_ident_use ps ppf id

and pp_print_prim_expr ps ppf ex =
  fprintf ppf "@[<hov>%a@]" (pp_print_prim_ex 80 ps) ex  (* without brackets, unless unavoidable *)

and pp_print_prim_expb ps ppf ex = pp_print_prim_ex 0 ps ppf ex  (* with brackets, unless avoidable *)

and pp_print_expr ps ppf ex = pp_print_prim_expr ps ppf (exprtoprim ex)  (* without brackets, unless unavoidable *)


and pp_print_expr_loc ps ppf ex =
  (* tag with location *)
  pp_print_string ppf " {";
  (pp_print_source_loc ps ppf ex.loc);
  pp_print_string ppf ": ";
  (pp_print_prim_exb 80 ps ppf (exprtoprim ex));
  pp_print_string ppf "} "



(* TODO: pretty-print evaluation contexts ....
*)

and pp_print_hole (ps : printer_state) (ppf : formatter) () =
      fprintf ppf "_"

and pp_print_one_eval_ctx_pri (ps,env,eqs) (ppf : formatter) ec =
(* given internal ps, env, and eqs, return external ps, env, and eqs, and delayed printing function *)
  let br = true in
  match ec with
    (* simple expressions first *)
      CtxTop ->
        let f () = fprintf ppf "Top"
        in (ps,env,eqs,f) (* shouldn't happen *)
    | CtxC1(c, _) ->
        let f () = fprintf ppf "%a %a" (pp_print_con1 ps) c (pp_print_hole ps) ()
        in (ps,env,eqs,f)
    | CtxTup (es, _, vs) ->
        let f () = fprintf ppf "(@[%a%a%a%a%a@])"
            (pp_print_etuple ps) es
            (pp_print_string) (match es with [] ->"" | _ -> ",")
            (pp_print_hole ps) ()
            (pp_print_string) (match vs with [] ->"" | _ -> ",")
            (pp_print_etuple ps) vs
        in (ps,env,eqs,f)

    | CtxRaise _  ->
        let f () = fprintf ppf "raise %a" (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxHashTs (eqs', t1, _, t2) ->
        let f () =
          fprintf ppf "@[<hv>hash(%a,@ %a)%a%a@]"
            (pp_print_typ ps) t1
            (pp_print_hole ps) ()
            pp_ibreak ()
            (pp_print_typ_argument ps) t2
        in (ps,env,eqs',f)

    | CtxHashHtsL (eqs', t1, _, e2, t2) ->
        let f () = opt_parens br ppf (fun ppf () ->
          fprintf ppf "@[<hv>hash(%a,@ %a,@ %a)%a%a@]"
            (pp_print_typ ps) t1
            (pp_print_hole ps) ()
            (pp_print_prim_exb 40 ps) e2
            pp_ibreak ()
            (pp_print_typ_argument ps) t2) ()
        in (ps,env,eqs',f)

    | CtxHashHtsR (eqs', t1, e1, _,  t2) ->
        let f () =
          fprintf ppf "@[<hv>hash(%a,@ %a,@ %a)%a%a@]"
            (pp_print_typ ps) t1
            (pp_print_prim_exb 40 ps) e1
            (pp_print_hole ps) ()
            pp_ibreak ()
            (pp_print_typ_argument ps) t2
        in (ps,env,eqs',f)

    | CtxSwapL (_,  e2, e3) ->
        let f () =
          fprintf ppf "@[<hv>swap@ %a@ and@ %a in@ %a@]"
            (pp_print_hole ps) ()
            (pp_print_prim_exb 0 ps) e2
            (pp_print_prim_exb 0 ps) e3
        in (ps,env,eqs,f)

    | CtxSwapM (e1, _,  e3) ->
        let f () =
          fprintf ppf "@[<hv>swap@ %a@ and@ %a in@ %a@]"
            (pp_print_prim_exb 0 ps) e1
            (pp_print_hole ps) ()
            (pp_print_prim_exb 0 ps) e3
        in (ps,env,eqs,f)

    | CtxSwapR (e1, e2, _ ) ->
        let f () =
          fprintf ppf "@[<hv>swap@ %a@ and@ %a in@ %a@]"
            (pp_print_prim_exb 0 ps) e1
            (pp_print_prim_exb 0 ps) e2
            (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxSupport   (t, _) ->
        let f () =
          fprintf ppf "@[<hv>support@ %a %a@]"
            (pp_print_typ_argument ps) t
            (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxNameOfTie _ ->
        let f () = fprintf ppf "name_of_tie %a" (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxValOfTie _ ->
        let f () = fprintf ppf "val_of_tie %a" (pp_print_hole ps) ()
        in (ps,env,eqs,f)

   (* operators/constants and applications *)
    | CtxLazyOp (lo, _, e2) ->
        let f () =
          let my_pr = lazyop_getprecedence lo in
          fprintf ppf "@[%a@ %a %a@]"
            (pp_print_hole ps) ()
            (pp_print_lazyop ps) lo
            (pp_print_prim_exb my_pr ps) e2
        in (ps,env,eqs,f)

    | CtxOp (oe, vs, _, es) ->
        let f () = fprintf ppf (if !Opts.printclos && not ps.ps_mode.pm_canon then
                                     format_of_string "@[<hov>{:Op %a :}@]"
                                else format_of_string "@[<hov>%a@]")
            (fun ppf () ->
              pp_concat ppf pp_ibreak ()
              (fun ppf f -> f ps ppf)
              (  [(fun ps ppf -> pp_print_op_or_econst ps ppf oe)]
               @ (List.map (fun v -> (fun ps ppf -> pp_print_prim_exb 0 ps ppf v)) vs)
               @ [(fun ps ppf -> pp_print_hole ps ppf ())]
               @ (List.map (fun v -> (fun ps ppf -> pp_print_prim_exb 0 ps ppf v)) es))) ()
        in (ps,env,eqs,f)

    | CtxOP (eqs', n, oe, es, _, vs) ->
         (* assert ((eqs = []));*) (* TODO: We should be able to leave this in, but we had a bug with
                                            this assertion failing. Revisit. *)
        let f () = fprintf ppf "@[{:OP {%a}^%d {@[%a}}@[<hov>%a@] :}@]"
            (pp_print_op_or_econst ps) oe
            n
            (pp_print_eqs_naked ps) eqs
            (fun ppf () ->
              pp_concat ppf pp_ibreak ()
                (fun ppf f -> f ps ppf)
                (  [(fun ps ppf -> ())]
                 @ (List.map (fun v -> (fun ps ppf -> pp_print_prim_exb 0 ps ppf v)) es)
                 @ [(fun ps ppf -> pp_print_hole ps ppf ())]
                 @ (List.map (fun v -> (fun ps ppf -> pp_print_prim_exb 0 ps ppf v)) vs))) ()
        in (ps,env,eqs',f)

   (* applications *)
    | CtxAppL (_, e2) ->
        let f () =
          fprintf ppf "%a %a"
            (pp_print_hole ps) ()
            (pp_print_prim_ex 9 ps) e2
        in (ps,env,eqs,f)

    | CtxAppR (e1, _) ->
        let f () =
          fprintf ppf "%a %a"
            (pp_print_prim_exb 10 ps) e1
            (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxTApp (_, t) ->
      let f () =
        fprintf ppf "%a %a"
          (pp_print_hole ps) ()
          (pp_print_typ_argument ps) t
      in (ps,env,eqs,f)

   (* other binary operators/constructors *)
    | CtxFreshforL (_,  e2) ->
        let f () =
          fprintf ppf "@[%a@ freshfor@ %a@]"
            (pp_print_hole ps) ()
            (pp_print_prim_exb 21 ps) e2
        in (ps,env,eqs,f)

    | CtxFreshforR (e1, _ ) ->
        let f () =
         fprintf ppf "@[%a@ freshfor@ %a@]"
            (pp_print_prim_exb 20 ps) e1
            (pp_print_hole ps) ()
        in (ps,env,eqs,f)

    | CtxConsL (_, e2) ->
        let f () = fprintf ppf "%a%a:: %a" (pp_print_hole ps) () pp_ibreak () (pp_print_prim_exb 27 ps) e2
        in (ps,env,eqs,f)

    | CtxConsR (e1, _) ->
        let f () = fprintf ppf "%a%a:: %a" (pp_print_prim_exb 26 ps) e1 pp_ibreak () (pp_print_hole ps) ()
        in (ps,env,eqs,f)

   (* If *)
    | CtxIf (_, e2, e3) ->
        let f () =
          fprintf ppf "@[<hv>@[<hv>if%a%a@ then@]%a%a@ else%a%a@]"
            pp_ibreak ()  (pp_print_hole ps) ()
            pp_ibreak ()  (pp_print_prim_exb 50 ps) e2
            pp_ibreak ()  (pp_print_prim_exb 50 ps) e3
        in (ps,env,eqs,f)

   (* Sequences *)
    | CtxSeq (_, e2) ->
        let f () =
          fprintf ppf "@[<hv>%a;@ %a@]"
            (pp_print_hole ps) ()
            (pp_print_prim_exb 75 ps) e2
        in (ps,env,eqs,f)

    | CtxUnpack (t, x, _, e2) ->
      let ps' = add_typname_binder ps t in
      let ps'' = add_ident_binder ps' x in
      let f () =
        fprintf ppf "@[let%a@[<hv>{%a@ ,@ %a}@ =%a%a@] in@ %a@]"
          pp_ibreak ()
          (pp_print_typname_use ps') t
          (pp_print_ident_use ps'') x
          pp_ibreak ()
          (pp_print_hole ps) ()
          (pp_print_prim_exb 75 ps'') e2
      in (ps,env,eqs,f)

    | CtxMatch (_, m) ->
        let f () =
          fprintf ppf "@[<hv>match %a with%a%a@]"
            (pp_print_hole ps) ()
            (fun ppf () -> pp_print_break ppf 1 2 (* length of "| " *) ) ()
            (pp_print_prim_mtch 80 ps) m
        in (ps,env,eqs,f)

    | CtxTry (_, m) ->
        let f () =
          fprintf ppf "@[<hv>try%a%a@ with%a%a@]"
            pp_ibreak ()
            (pp_print_hole ps) ()
            pp_ibreak ()
            (pp_print_prim_mtch 80 ps) m
        in (ps,env,eqs,f)

  | CtxNamecaseL (_, t, x1, x2, e, e2, e3) ->
      let ps' = add_typname_binder ps t in
      let ps'' = add_ident_binder ps' x1 in
      let ps''' = add_ident_binder ps'' x2 in
      let f () =
        fprintf ppf "@[namecase %a with@;  @[<hv>{%a,(%a,%a)} when %a = %a -> %a@]  @[<hv>otherwise -> %a@]@]"
          (pp_print_hole ps) ()
          (pp_print_typname_use ps') t
          (pp_print_ident_use ps'') x1
          (pp_print_ident_use ps''') x2
          (pp_print_ident_use ps'') x1
          (pp_print_prim_exb 32 ps) e
          (pp_print_prim_exb 70 ps''') e2
          (pp_print_prim_exb 70 ps) e3
      in (ps,env,eqs,f)

  | CtxNamecaseR (e1, t, x1, x2, _, e2, e3) ->
      let ps' = add_typname_binder ps t in
      let ps'' = add_ident_binder ps' x1 in
      let ps''' = add_ident_binder ps'' x2 in
      let f () =
        fprintf ppf "@[namecase %a with@;  @[<hv>{%a,(%a,%a)} when %a = %a -> %a@]  @[<hv>otherwise -> %a@]@]"
          (pp_print_prim_exb 80 ps) e1
          (pp_print_typname_use ps') t
          (pp_print_ident_use ps'') x1
          (pp_print_ident_use ps''') x2
          (pp_print_ident_use ps'') x1
          (pp_print_hole ps) ()
          (pp_print_prim_exb 70 ps''') e2
          (pp_print_prim_exb 70 ps) e3
      in (ps,env,eqs,f)

  (* Expressions ending with type *)
    | CtxPack(t1,_,t2) ->
        let f () =
          fprintf ppf "@[{%a@ ,@ %a}@ as@ %a@]"
            (pp_print_typ ps) t1
            (pp_print_hole ps) ()
            (pp_print_typ ps) t2
        in (ps,env,eqs,f)

    | CtxMarshalL (_, e2, t) ->
        let f () =
          fprintf ppf "@[<hv>marshal %a%a%a@ : %a@]"
            (pp_print_hole ps) ()
            pp_ibreak ()
            (pp_print_prim_ex 0 ps) e2
            (pp_print_typ ps) t
        in (ps,env,eqs,f)

    | CtxMarshalR (v1, _, t) ->
        let f () =
          fprintf ppf "@[<hv>marshal %a%a%a@ : %a@]"
            (pp_print_prim_ex 0 ps) v1
            pp_ibreak ()
            (pp_print_hole ps) ()
            (pp_print_typ ps) t
        in (ps,env,eqs,f)

    | CtxMarshalz (eqs', mk, _, t) ->
        assert ((eqs = []));
        let f () =
          fprintf ppf "@[<hv>marshalz %a%a%a@ : %a@]"
            (pp_print_esc_string) mk
            pp_ibreak ()
            (pp_print_hole ps) ()
            (pp_print_typ ps) t
        in (ps,env,eqs',f)

    | CtxUnmarshal(_, t) ->
        let f () =
          fprintf ppf "@[<hv>unmarshal%a%a@ as %a@]"
            pp_ibreak ()
            (pp_print_hole ps) ()
            (pp_print_typ ps) t
        in (ps,env,eqs,f)

   (* Other constructs *)
    | CtxInEnv (env', _) ->  (* NB: the *outer* env *)
        let f () = fprintf ppf "@[{:INENV: ";
          pp_print_etysubst_short ps ppf env;   (* TODO: Why this is different than pp InEnv? *)
          fprintf ppf ",@ %a:}@]" (pp_print_hole ps) ()
        in
        let ps' = add_etysubst ps env'
        in (ps',env',eqs,f)

    | CtxCol (_, eqs', t) ->
        let f () = fprintf ppf "@[<hv>[ %a@;<1 2>@[<hv>]_{ %a@;<1 2>}^{ %a@ }@]@]"  (* seem to be enough brackets, maybe don't need more *)
            (pp_print_hole ps) ()
            (pp_print_eqs_naked ps) eqs
            (pp_print_typ ps) t
        in (ps,env,eqs',f)

    | CtxStrVal _ -> (* TODO *)
        let f () = fprintf ppf "CtxStrVal: TODO"
        in (ps,env,eqs,f)

    | CtxImodule _ -> (* TODO *)
        let f () = fprintf ppf "CtxImodule: TODO"
        in (ps,env,eqs,f)

and pp_print_eval_ctx_pri (ps : printer_state) (ppf : formatter) env eqs ec =
  (* internal eqs passed as arg *)
  let ecs = List.map (fun f -> f CtxTop) (flatten_context ec) in
  fprintf ppf "@[<v>< ";
  let rec go (ps,env,eqs) =
    function
        [] -> (ps,env,eqs)
      | (ec::ecs) ->
          let (ps',env',eqs',pf) = pp_print_one_eval_ctx_pri (ps,env,eqs) ppf ec in
          let (ps'',env'',eqs'') = go (ps',env',eqs') ecs in
          (match ecs with
            [] -> ()
          | (_::_) -> fprintf ppf "@  ");
          pf ();
          (ps'',env'',eqs'')
  in
  let (ps',env',eqs') = go (ps,env,eqs) ecs in
  fprintf ppf "@ >@]";
  (ps',env',eqs')

(* print patterns *)

(* pp_print_pat_pri' ppf prints a pattern under the assumption that all the
   variables in the pattern are defined in ps.  When pp_print_pat_pri ppf calls
   pp_print_pat_pri' ppf it guarantees this precondition.
*)

and pp_print_pat_pri' ps ppf p =
  match p with
    PWild(t)   -> fprintf ppf "(_ : %a)"
                    (pp_print_typ ps) t
  | PVar(i,t)  ->  Debug.print (fun () -> ("pretty: pp_print_pat_pri': PVar"));
                   fprintf ppf "(%a : %a)"
                      (pp_print_ident_use ps) i
                      (pp_print_typ ps) t
  | PC0(c)  -> pp_print_con0 ps ppf c
  | PC1(c,p)-> fprintf ppf "(%a%a%a)"  (* FZ FZ CHECK *)
                 (pp_print_con1 ps) c
                 pp_ibreak ()
                 (pp_print_pat_pri'b ps) p
  | PCons(p1,p2) -> fprintf ppf "%a::%a" (pp_print_pat_pri'b ps) p1 (pp_print_pat_pri'b ps) p2
  | PTup(pl) -> fprintf ppf "(@[";
                pp_concat ppf fprintf ",@ " (pp_print_pat_pri'b ps) pl;
                fprintf ppf "@])"
  | PTyped(p,t) -> fprintf ppf "(%a : %a)"
                     (pp_print_pat_pri'b ps) p
                     (pp_print_typ ps) t

and pp_print_pat_pri'b ps ppf p = fprintf ppf "@[%a@]" (pp_print_pat_pri' ps) p

and pp_print_pat_pri ps ppf p =
  Debug.print (fun () -> ("pretty: pp_print_pat_pri: entry"));
  let vars = binding_vars_of_pat p in
  Debug.print (fun () -> ("pretty: pp_print_pat_pri: vars has length " ^ string_of_int (List.length vars)));
  let ps' = add_ident_binders ps vars in
  fprintf ppf "@[%a@]" (pp_print_pat_pri' ps') p;
  ps'

and pp_print_pat ps ppf x = ignore (pp_print_pat_pri ps ppf x)
and pp_print_pat_ps ps ppf x = pp_print_pat_pri ps ppf x

(* FZ* *)
and pp_print_pat_list_pri ps ppf l =
  fprintf ppf "@[<v>";
  let ps' = pp_concat_ps ps ppf fprintf "" pp_print_pat_ps l in
  fprintf ppf "@]";
  ps'

and pp_print_pat_list ps ppf l = ignore (pp_print_pat_list_pri ps ppf l)
and pp_print_pat_list_ps ps ppf l = pp_print_pat_list_pri ps ppf l

(* Print type annotations:  %[T] *)
and pp_print_typ_argument ps ppf t =
  if ps.ps_mode.pm_human then
    pp_print_string ppf "%[]"
  else
    fprintf ppf "%%[@[%a@]]" (pp_print_typ ps) t

(* Print tuple types: T1 * T2 * T3 * ... * Tn   *)
and pp_print_ttuple ps ppf (ts : (Ast.typ list)) =
  pp_concat ppf fprintf " *@ " (pp_print_tpb 1 ps) ts

(* Print sum types: T1 + T2 + T3 + ... + Tn   *)
and pp_print_tsum ps ppf (ts : (Ast.typ list)) =
  pp_concat ppf fprintf " +@ " (pp_print_tpb 1 ps) ts

(* The precedence argument takes values
   1: simple_core_type (TTyCon0, TTyCon1, TVar, TXDot, TTyName)
   2: typ_core_type    (TTup, TSum)
   3: fun_core_type    (TFunc)
   4: core_type_pri    (TForall, TExists)
 *)
and pp_print_tp precedence ps ppf t =
  match t with
  | TTyCon0 tc0 -> pp_print_tycon0 ppf tc0
  | TTyCon1(tc1,t1) -> fprintf ppf "%a%a%a"
                         (pp_print_tp 1 ps) t1
                         pp_ibreak ()
                         (pp_print_tycon1 ps) tc1
  | TTup(tl) ->  opt_parens (precedence < 2) ppf (pp_print_ttuple ps) tl
  | TSum(tl) ->  opt_parens (precedence < 2) ppf (pp_print_tsum ps) tl
  | TFunc(t1,t2) -> opt_parens (precedence < 3) ppf (fun ppf () ->
                      fprintf ppf "%a ->%a%a"
                        (pp_print_tpb 2 ps) t1
                        pp_ibreak ()
                        (pp_print_tp 3 ps) t2) ()
  | TVar(c)    -> pp_print_typname_use ps ppf c
  | TXDot(x,n)     -> fprintf ppf "%a.%a"
                         (pp_print_hash_or_modname ps) x
                         (pp_print_typname_ext ps) n
  | TForall(n,t)  -> let ps' = add_typname_binder ps n in
                     opt_parens (precedence < 4) ppf (fun ppf () ->
                       fprintf ppf "@[forall@ %a.@]@ %a"
                         (pp_print_typname_use ps') n
                         (pp_print_tp 4 ps') t) ()
  | TExists(n,t)  -> let ps' = add_typname_binder ps n in
                     opt_parens (precedence < 4) ppf (fun ppf () ->
                       fprintf ppf "@[exists@ %a.@]@ %a"
                         (pp_print_typname_use ps') n
                         (pp_print_tp 4 ps') t) ()
  | TTyName(abstract_name) -> pp_print_abstract_name ps ppf abstract_name

and pp_print_tpb precedence ps ppf t =
  fprintf ppf "@[<hov>%a@]" (pp_print_tp precedence ps) t

and pp_print_typ  ps ppf t = pp_print_tpb 4 ps ppf t
and pp_print_typb ps ppf t = pp_print_tpb 1 ps ppf t

and pp_print_kind ps ppf = function
    KType   -> pp_print_string ppf "Type"
  | KEq(ty) -> fprintf ppf "Eq(%a)" (pp_print_typ ps) ty

and pp_print_eq ps ppf = function
    EHash (h,etn,ty) ->
      fprintf ppf "@[%a.%a =%a%a@]"
        (pp_print_hash ps) h
        (pp_print_typname_ext ps) etn
        pp_ibreak ()
        (pp_print_typ ps) ty
  | EMod (mn,etn, ty) ->
      fprintf ppf "@[%a.%a =%a%a@]"
        (pp_print_modname ps) mn
        (pp_print_typname_ext ps) etn
        pp_ibreak ()
        (pp_print_typ ps) ty

(* equation set is canonicalised (sorted and duplicates removed) before printing;
   if this is not done, hashes are not well-defined. *)
and pp_print_eqs_naked ps ppf eqs =
  let canonicalise_eqs = match !canonicalise_eqs with
                           Some f -> f
                         | None -> raise (Util.Never_happen "pp_print_eqs_naked: canonicalise_eqs not set") in
  let eqs = canonicalise_eqs eqs in
  fprintf ppf "@[<hov>";
  pp_concat ppf fprintf ",@ " (pp_print_eq ps) eqs;
  fprintf ppf "@]"

and pp_print_eqs ps ppf eqs =
  if not(ps.ps_mode.pm_tex) then
    fprintf ppf "{%a}" (pp_print_eqs_naked ps) eqs
  else
    fprintf ppf "\\{%a\\}" (pp_print_eqs_naked ps) eqs

and pp_print_signature_item_pri ps ppf = function s -> match s.desc with
    SVal(ident,t) ->
      let ps' = add_ident_binder ps (ident_int ident) in
      fprintf ppf "@[<hv>val%a@[<hv>%a%a: %a@]@]"
        pp_ibreak ()
        (pp_print_ident_bind ps') ident
        pp_ibreak ()
        (pp_print_typ ps) t;
      ps'
  | STyp(typname,kind)  ->
      let ps' = add_typname_binder ps (typname_int typname) in
      fprintf ppf "@[<hv>type%a@[<hv>%a%a: %a@]@]"
        pp_ibreak ()
        (pp_print_typname_bind ps') typname
        pp_ibreak ()
        (pp_print_kind ps) kind;
      ps'

and pp_print_signature_item ps ppf x = ignore (pp_print_signature_item_pri ps ppf x)

(* bitrot
and pp_print_namesiglist ps ppf = function
  | [] -> ""
  | (m,sgn)::nsl ->
      let ps' = add_modname_binder ps m in
      "module " ^ pp_print_modname ppf ps' mn ^ " : sig " ^ pp_print_signature ps ppf sgn ^ " end "
      ^ pp_print_namesiglist ppf ps' ngl
*)

and pp_print_structure_item ps ppf = function s -> match s.desc with
  | StrVal(ident, v) ->
      let ps' = add_ident_binder ps (ident_int ident) in
      fprintf ppf "@[<hov>let%a%a =%a%a@]"
        pp_ibreak ()
        (pp_print_ident_bind ps') ident
        pp_ibreak ()
        (pp_print_expr ps) v;
      ps'
  | StrValMulti(ident, pats, e) ->
      let ps' = add_ident_binder ps (ident_int ident) in
      fprintf ppf "@[<hv>let%a@[<hv>%a "
        pp_ibreak ()
        (pp_print_ident_bind ps') ident;
      let ps'' = pp_print_pat_list_ps ps' ppf pats in
      fprintf ppf " =%a%a@]@]"
        pp_ibreak()
        (pp_print_expr ps'') e;
      ps'
  | StrTyp(typname, t) ->
      let ps' = add_typname_binder ps (typname_int typname) in
      fprintf ppf "@[<hv>type %a =%a%a@]"
        (pp_print_typname_bind ps') typname
        pp_ibreak ()
        (pp_print_typ ps) t;
      ps'

and pp_print_signature_ps ps ppf sign =
  fprintf ppf (if List.length sign.desc <= 1 then format_of_string "@[<hv>" else format_of_string "@[<v>");
  let ps = pp_concat_ps ps ppf fprintf "@ " pp_print_signature_item_pri sign.desc in
  fprintf ppf "@]";
  ps

and pp_print_signature ps ppf sign =
  ignore (pp_print_signature_ps ps ppf sign)

and pp_print_structure ps ppf str =
      if List.length str.desc <=1 then
        fprintf ppf "@[<hv>%a@]"
          (fun ppf () ->
            ignore (pp_concat_ps ps ppf fprintf "@ " pp_print_structure_item str.desc)) ()
      else
        fprintf ppf "@[<v>%a@]"
          (fun ppf () ->
            ignore (pp_concat_ps ps ppf fprintf "@ " pp_print_structure_item str.desc)) ()

and pp_print_vliteral ps ppf = function
  | VNumber n -> pp_print_int ppf n
  | VHash h -> pp_print_hash ps ppf h

and pp_print_atomic_version ps ppf (avne : atomic_version) =
  match avne with
  | VLiteral vlit -> pp_print_vliteral ps ppf vlit
  | VMyname       -> pp_print_string ppf "myname"

and pp_print_version ps ppf (vne : version) =
  match vne with
  | VAtomic avne -> pp_print_atomic_version ps ppf avne
  | VCons (avne, vne') -> fprintf ppf "%a.%a"
                            (pp_print_atomic_version ps) avne
                            (pp_print_version ps) vne'

and pp_print_atomic_hash_version_constraint ps ppf (ahvce : atomic_hash_version_constraint) =
  match ahvce with
  | AVCHModname m -> pp_print_modname ps ppf m
  | AVCHHash h -> pp_print_hash ps ppf h

and pp_print_atomic_version_constraint ps ppf (avce : atomic_version_constraint) =
  match avce with
  | AVCHashVersion ahcve -> pp_print_atomic_hash_version_constraint ps ppf ahcve
  | AVCNumber n -> pp_print_int ppf n

and pp_print_tail_version_constraint ps ppf = function
  | TVCAtomic  avce    -> pp_print_atomic_version_constraint ps ppf avce
  | TVCBetween (n1,n2) -> fprintf ppf "%a-%a"
                                  pp_print_int n1
                                  pp_print_int n2
  | TVCBefore  n2      -> fprintf ppf "-%a" pp_print_int n2
  | TVCAfter   n1      -> fprintf ppf "%a-" pp_print_int n1
  | TVCAnything        -> pp_print_string ppf "*"

and pp_print_version_constraint ps ppf (vce : version_constraint) =
  match vce with
  | VCDotted (avce_list, tvce) ->
      pp_concat ppf pp_print_string "." (pp_print_atomic_version_constraint ps) avce_list;
      begin match avce_list with
      | [] -> ()
      | _ -> pp_print_string ppf "."
      end;
      pp_print_tail_version_constraint ps ppf tvce
  | VCNameEqual ahvce  -> fprintf ppf "name = %a" (pp_print_atomic_hash_version_constraint ps) ahvce

and pp_print_withspec ps ppf (ws : withspec) =
  let go ppf (modname,ext,ty) = fprintf ppf "%a.%a = %a" (pp_print_modname ps) modname (pp_print_typname_ext ps) ext (pp_print_typ ps) ty
  in
  fprintf ppf "{@[<hv>%a@]}"
    (fun ppf () -> pp_concat ppf fprintf ",@ " go ws) ()

and pp_print_resolvespec_item ps ppf =
  function Resolve_static_link  -> pp_print_string ppf "Static_Link"
         | Resolve_here_already -> pp_print_string ppf "Here_Already"
         | Resolve_url(url)     -> pp_print_esc_string ppf url

and pp_print_resolvespec ps ppf (rs : resolvespec) =
  pp_concat ppf
    fprintf ",@ "
    (pp_print_resolvespec_item ps)
    rs

and pp_print_ats ps ppf ats =
  (* canonicalise the ordering of the ats by sorting it *)
  let ats_sorted = List.sort etn_compare ats in
  fprintf ppf "{@[<hv>%a@]}"
    (fun ppf () -> pp_concat ppf fprintf ",@ " (pp_print_typname_ext ps) ats_sorted) ()

and pp_print_likespec ps ppf = function
  | LikeNone -> ()
  | LikeMod modname -> fprintf ppf "like %a" (pp_print_modname ps) modname
  | LikeStr str -> fprintf ppf "like @[<hv>struct%a%a@ end"
                     pp_ibreak ()
                     (pp_print_structure ps) str

and pp_print_mode ps ppf = function
    MHash   -> pp_print_string ppf "hash"
  | MFresh  -> pp_print_string ppf "fresh"
  | MCfresh -> pp_print_string ppf "cfresh"
  | MBangCfresh ->  pp_print_string ppf "cfresh!"
  | MBangHash -> pp_print_string ppf "hash!"

(* returns environment including module name *)
and pp_print_mod_user ps ppf (name, mode, (mubody: Ast.mod_user_body)) =
  let ps' = add_modname_binder ps name in
  fprintf ppf "@[<hv>module %a %a@,: sig@,@[<h 2>    %a@]@,  end@,  version %a@,= struct@,@[<h 2>    %a@]@,  end@]%a@]"
    (pp_print_mode ps) mode
    (pp_print_modname ps') name
    (pp_print_signature ps) mubody.mu_sign
    (pp_print_version ps) mubody.mu_vne
    (pp_print_structure ps) mubody.mu_str
    (fun ppf () ->
      match mubody.mu_withspec with
        [] -> ()
      | _  -> fprintf ppf "@ with!%a%a"
                pp_ibreak () (pp_print_withspec ps) mubody.mu_withspec) ();
  fprintf ppf "@ ";
  ps'

(* returns environment including module name *)
and pp_print_import_user ps ppf (name, mode, (iubody: Ast.import_user_body)) =
  let ps' = add_modname_binder ps name in
  fprintf ppf "@[<hv>import %a %a @,: sig@,@[<h 2>    %a@]@,  end@,  version %a@,  %a@,  by %a@,  = %a@]"
    (pp_print_mode ps) mode
    (pp_print_modname ps') name
    (pp_print_signature ps) iubody.iu_sign
    (pp_print_version_constraint ps) iubody.iu_vce
    (pp_print_likespec ps) iubody.iu_likespec
    (pp_print_resolvespec ps) iubody.iu_resolvespec
    (fun ppf () ->
      match iubody.iu_mo with
        None -> pp_print_string ppf "unlinked"
      | Some(mn) -> pp_print_modname ps ppf mn) ();
  fprintf ppf "@ ";
  ps'

and pp_print_valuability ppf vub =
  match vub with
  | Valuable    -> pp_print_string ppf "valuable"
  | CValuable   -> pp_print_string ppf "cvaluable"
  | Nonvaluable -> pp_print_string ppf "nonvaluable"

and pp_print_valuabilities ppf vubs =
  fprintf ppf "@[(%a,@ %a)@]"
    pp_print_valuability vubs.term_valuability
    pp_print_valuability vubs.type_valuability

and pp_print_definition_pri ps ppf (def : definition) =
  let (ignore_this, pm_ignore_defns') =
    match ps.ps_mode.pm_ignore_defns with
      Some n -> if n>=1 then (true,Some (n-1)) else (false, Some 0)
    | None -> (true,None) in
  let ps = {ps with ps_mode = {ps.ps_mode with pm_ignore_defns=pm_ignore_defns'}} in
  if ignore_this then
    (match def with
    | Mod_compile (name, mcbody) ->
        let ps' = add_modname_binder ps name in ps'
    | Mod_imod (name, mibody) ->
        let ps' = add_modname_binder ps name in ps'
       | Import_compile (name, icbody) ->
        let ps' = add_modname_binder ps name in ps'
    | Mod_fresh (name, mfbody) ->
        let ps' = add_modname_binder ps name in ps'
    | Import_fresh (name, ifbody) ->
        let ps' = add_modname_binder ps name in ps'
    | Mark_compile s ->
        ps
    )
  else
    match def with
    | Mod_compile (name, mcbody) ->
        let ps' = add_modname_binder ps name in
        if ps'.ps_mode.pm_wide then (
          fprintf ppf "@[<hv>cmodule %a@ %a@ : %a@ sig %a%a@ end%a%a sig %a%a@ end version%a%a@ = struct%a%a@ end@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) mcbody.mc_hash
            (pp_print_eqs ps) mcbody.mc_eqs
            pp_ibreak()
            (pp_print_signature ps) mcbody.mc_sign0
            pp_ibreak ()
            pp_print_valuabilities mcbody.mc_vubs
            pp_ibreak ()
            (pp_print_signature ps) mcbody.mc_sign1
            pp_ibreak ()
            (pp_print_version ps) mcbody.mc_vn
            pp_ibreak ()
            (pp_print_structure ps) mcbody.mc_str;
          fprintf ppf "@ ";
          ps')
        else (
          fprintf ppf "@[<hv>cmodule %a %a : %a@,  sig@,@[<h 2>    %a@]@,  end %a@,  sig@,@[<h 2>    %a@]@,  end@,  version %a@,= struct@,@[<h 2>    %a@]@,  end@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) mcbody.mc_hash
            (pp_print_eqs ps) mcbody.mc_eqs
            (pp_print_signature ps) mcbody.mc_sign0
            pp_print_valuabilities mcbody.mc_vubs
            (pp_print_signature ps) mcbody.mc_sign1
            (pp_print_version ps) mcbody.mc_vn
            (pp_print_structure ps) mcbody.mc_str;
          fprintf ppf "@ ";
          ps')
    | Mod_imod (name, mibody) ->
        let ps' = add_modname_binder ps name in
        if ps'.ps_mode.pm_wide then (
          fprintf ppf "@[<hv>cmodule %a@ %a@ : %a@ sig %a%a@ end%a%a sig %a%a@ end version%a%a@ = struct%a%a@ end@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) mibody.mi_hash
            (pp_print_eqs ps) mibody.mi_eqs
            pp_ibreak()
            (pp_print_signature ps) mibody.mi_sign0
            pp_ibreak ()
            pp_print_valuabilities mibody.mi_vubs
            pp_ibreak ()
            (pp_print_signature ps) mibody.mi_sign1
            pp_ibreak ()
            (pp_print_version ps) mibody.mi_vn
            pp_ibreak ()
            (pp_print_structure ps) ({desc=mibody.mi_str_done @ mibody.mi_str_todo; loc=mibody.mi_str_loc});
          fprintf ppf "@ ";
          ps')
        else (
          fprintf ppf "@[<hv>cmodule %a %a : %a@, sig@,@[<h 2>    %a@]@,  end %a@, sig@,@[<h 2>    %a@]@,  end@,  version %a@,= struct@,@[<h 2>    %a@]@,  end@]@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) mibody.mi_hash
            (pp_print_eqs ps) mibody.mi_eqs
            (pp_print_signature ps) mibody.mi_sign0
            pp_print_valuabilities mibody.mi_vubs
            (pp_print_signature ps) mibody.mi_sign1
            (pp_print_version ps) mibody.mi_vn
            (pp_print_structure ps) ({desc=mibody.mi_str_done @ mibody.mi_str_todo; loc=mibody.mi_str_loc});
          fprintf ppf "@ ";
          ps')

    | Import_compile (name, icbody) ->
        let ps' = add_modname_binder ps name in
        if ps'.ps_mode.pm_wide then (
          fprintf ppf "@[<hv>cimport %a %a@ : sig%a%a@ end %a%a sig%a%a@ end version%a%a@ like struct%a%a@ end@ by %a@ = %a@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) icbody.ic_hash
            pp_ibreak ()
            (pp_print_signature ps) icbody.ic_sign0
            pp_ibreak ()
            pp_print_valuabilities icbody.ic_vubs
            pp_ibreak ()
            (pp_print_signature ps) icbody.ic_sign1
            pp_ibreak ()
            (pp_print_version_constraint ps) icbody.ic_vc
            pp_ibreak ()
            (pp_print_structure ps) icbody.ic_likestr
            (pp_print_resolvespec ps) icbody.ic_resolvespec
            (fun ppf () ->
              match icbody.ic_mo with
                None -> pp_print_string ppf "unlinked"
              | Some(mn) -> pp_print_modname ps ppf mn) ();
          fprintf ppf "@ ";
          ps')
        else (
          fprintf ppf "@[<hv>cimport %a %a@,: sig@,@[<h 2>    %a@]@,  end %a@, sig@,@[<h 2>    %a@]@,  end@,  version %a@,  like@[<hov 2>  struct@,@[<h 2>    %a@]@,  end@]@,  by %a@,  = %a@]"
            (pp_print_modname ps') name
            (pp_print_hash ps) icbody.ic_hash
            (pp_print_signature ps) icbody.ic_sign0
            pp_print_valuabilities icbody.ic_vubs
            (pp_print_signature ps) icbody.ic_sign1
            (pp_print_version_constraint ps) icbody.ic_vc
            (pp_print_structure ps) icbody.ic_likestr
            (pp_print_resolvespec ps) icbody.ic_resolvespec
            (fun ppf () ->
              match icbody.ic_mo with
                None -> pp_print_string ppf "unlinked"
              | Some(mn) -> pp_print_modname ps ppf mn) ();
          fprintf ppf "@ ";
          ps')
    | Mod_fresh (name, mfbody) -> (* syntax is the same as for Mod_user *)
        pp_print_mod_user ps ppf (name, MFresh, mfbody)
    | Import_fresh (name, ifbody) -> (* syntax is the same as for Import_user *)
        pp_print_import_user ps ppf (name, MFresh, ifbody)
    | Mark_compile mk ->
        fprintf ppf "mark \"%s\"" (String.escaped mk);
        fprintf ppf "@ ";
        ps

and pp_print_source_definition_pri ps ppf (def : source_definition) =
  let (ignore_this, pm_ignore_defns') =
    match ps.ps_mode.pm_ignore_defns with
      Some n -> if n>=1 then (true,Some (n-1)) else (false, Some 0)
    | None -> (true,None) in
  let ps = {ps with ps_mode = {ps.ps_mode with pm_ignore_defns=pm_ignore_defns'}} in
  if ignore_this then
    (match def with
    | Mod_user (name, mode, mubody) ->
        let ps' = add_modname_binder ps name in ps'
    | Mod_alias (name, mabody)  ->
        let ps' = add_modname_binder ps name in ps'
    | Import_user (name, mode, iubody) ->
        let ps' = add_modname_binder ps name in ps'
    | Mark_user s ->
        ps
    )
  else
    match def with
    | Mod_user (name, mode, mubody) ->
        pp_print_mod_user ps ppf (name, mode, mubody)
    | Mod_alias (name, mabody)  ->
        let ps' = add_modname_binder ps name in
        fprintf ppf "@[<hv>amodule %a@ : sig%a%a@ end@ = %a@]"
          (pp_print_modname ps') name
          pp_ibreak ()
          (pp_print_signature ps) mabody.ma_sign
          (pp_print_modname ps) mabody.ma_modname;
        fprintf ppf "@ ";
        ps'
    | Import_user (name, mode, iubody) ->
        pp_print_import_user ps ppf (name, mode, iubody)
    | Mark_user mk ->
        fprintf ppf "mark \"%s\"" (String.escaped mk);
        fprintf ppf "@ ";
        ps

and printer_state_after_source_definitions ps ds =
  pp_print_source_definitions_pri ps null_formatter ds

and printer_state_after_definitions ps ds =
  pp_print_definitions_pri ps null_formatter ds

and pp_print_definition ps ppf x = ignore (pp_print_definition_pri ps ppf x)
and pp_print_definition_ps ps ppf x = pp_print_definition_pri ps ppf x

and pp_print_source_definition ps ppf x = ignore (pp_print_source_definition_pri ps ppf x)
and pp_print_source_definition_ps ps ppf x = pp_print_source_definition_pri ps ppf x

(* pp_print_definitions_pri ppf expects the definitions to already be freshened *)
and pp_print_definitions_pri ps ppf ds =
  fprintf ppf "@[<v>";
  let ps' = pp_concat_ps ps ppf fprintf ""
      pp_print_definition_pri ds in
  fprintf ppf "@]";
  ps'

(* pp_print_source_definitions_pri ppf expects the definitions to already be freshened *)
and pp_print_source_definitions_pri ps ppf ds =
  fprintf ppf "@[<v>";
  let ps' = pp_concat_ps ps ppf fprintf ""
      pp_print_source_definition_pri ds in
  fprintf ppf "@]";
  ps'

(* pp_print_definitions ppf expects the definitions to already be freshened *)
and pp_print_definitions ps ppf ds = ignore (pp_print_definitions_pri ps ppf ds)

(* pp_print_compiled_defintions_ps ppf expects the definitions to already be freshened *)
and pp_print_definitions_ps ps ppf ds = pp_print_definitions_pri ps ppf ds

(* pp_print_source_definitions ppf expects the definitions to already be freshened *)
and pp_print_source_definitions ps ppf ds = ignore (pp_print_source_definitions_pri ps ppf ds)

(* pp_print_source_defintions_ps ppf expects the definitions to already be freshened *)
and pp_print_source_definitions_ps ps ppf ds = pp_print_source_definitions_pri ps ppf ds


(* FZ* : do we need to modify the printer state?  The code below assumes yes *)
and pp_print_compilation_unit_definition ps ppf cud =
   match cud with
   | CUDef sd_fun -> fprintf ppf "<source definition>" ; ps  (* pp_print_source_definition_ps ps ppf sd *)
   | CUSourceString s -> raise (Util.Never_happen ("FZ: pp of CUSourceString" ^ s))
   | CUIncludeSource (s, None) ->
        (fprintf ppf "includesource %a" pp_print_esc_string s) ; ps
   | CUIncludeSource (s, Some s') ->
        (fprintf ppf "includesource %a (* %a *)"
                     pp_print_esc_string s
                     pp_print_esc_string s') ; ps
   | CUIncludeCompiled s ->
        (fprintf ppf "includecompiled %a" pp_print_esc_string s); ps

and pp_print_compilation_unit_definitions ps ppf cuds =
   fprintf ppf "@[<v>";
   let ps' = pp_concat_ps ps ppf fprintf ""
     pp_print_compilation_unit_definition cuds in
   fprintf ppf "@]";
  ps'

(* FZ* : pp_print_compilation_unit assumes that the compilation unit
         has already been freshened : is this THE RIGHT THING ? *)
and pp_print_compilation_unit ps ppf (cuds, eo) =
  fprintf ppf "@[<v>";
  let ps' = pp_print_compilation_unit_definitions ps ppf cuds in
   ( match eo with
   | None -> fprintf ppf "@]"
   | Some e_fun -> fprintf ppf "@ <expression>@]" ) ; (* fprintf ppf "@ %a@]" (pp_print_expr ps') e ) ; *)
   ps'

and pp_print_compiled_unit = fun ps ppf (nenv, defs, eo) ->
 match eo with
   None -> fprintf ppf "@[%a,%a%a@]"
             (pp_print_nameenv ps) nenv
             pp_ibreak () (pp_print_definitions ps) defs
 | Some e -> fprintf ppf "@[%a,%a" (pp_print_nameenv ps) nenv pp_ibreak ();
             let ps' = pp_print_definitions_pri ps ppf defs in
             fprintf ppf "%a%a@]" pp_ibreak () (pp_print_expr ps') e

and pp_print_program ps ppf (ds,e) =
  if ps.ps_mode.pm_dumplevel = 0 then () else
  Debug.print (fun () -> "pretty: pp_print_program");
  fprintf ppf "@[<v>";
  let ps' = pp_print_definitions_pri ps ppf ds in
  Debug.print (fun () -> "pretty: pp_print_program: calculated dsstring");
  Debug.print (fun () -> ("pretty: pp_print_program: ps' = <unavail>" (*print_printer_state ps'*) ));
  fprintf ppf "@ %a@]" (pp_print_prim_expr ps') e

and pp_print_store ps ppf store =
  let go ppf (l,v) =
    fprintf ppf "@[<hv>(%a :=%a%a)@]"
      (pp_print_location ps) l  pp_ibreak ()  (pp_print_prim_expr ps) v
  in
  fprintf ppf "@[<hov>";
  pp_concat ppf fprintf ",@ " go store;
  fprintf ppf "@]"

and pp_print_loctyplist ps ppf storeenv =
  let go ppf (l,ty) =
    fprintf ppf "@[<hv>(%a :%a%a)@]"
      (pp_print_location ps) l  pp_ibreak ()  (pp_print_typ ps) ty
  in
  fprintf ppf "@[<hov>";
  pp_concat ppf fprintf ",@ " go storeenv;
  fprintf ppf "@]";

and pp_print_esubst ps ppf re =   (* brutally *)
  let go ppf (iid,v) = fprintf ppf "(%a = %a)"
                         (pp_print_ident_use ps) iid
                         (pp_print_prim_expr ps) v
  in
  pp_concat ppf fprintf "@ " go (iidmap_to_list re)

and pp_print_esubst_short ps ppf re =
  (* assumes ps has bindings already *)
  if !Opts.printenv then
    if !Opts.printenvbodies then
      let go ppf (iid,v) =
        fprintf ppf "%a=%a"
          (pp_print_ident_use ps) iid
          (pp_print_prim_expr ps) v
      in
      pp_concat ppf fprintf ",@ " go (iidmap_to_list re)
    else
      let go ppf (iid,v) =
        fprintf ppf "%a=.." (pp_print_ident_use ps) iid
      in
      pp_concat ppf fprintf ",@ " go (iidmap_to_list re)
  else
    ()

and pp_print_tysubst ps ppf re =   (* brutally *)
  let go ppf (itn,ty) = fprintf ppf "(%a = %a)"
                          (pp_print_typname_use ps) itn
                          (pp_print_typ ps) ty
  in
  pp_concat ppf fprintf "@ " go (itnmap_to_list re)

and pp_print_tysubst_short ps ppf re =
  (* assumes ps has bindings already *)
  if !Opts.printenv then
    if !Opts.printenvbodies then
      let go ppf (itn,ty) =
        fprintf ppf "%a=%a"
          (pp_print_typname_use ps) itn
          (pp_print_typ ps) ty
      in
      pp_concat ppf fprintf ",@ " go (itnmap_to_list re)
    else
      let go ppf (itn,ty) =
        fprintf ppf "%a=.." (pp_print_typname_use ps) itn
      in
      pp_concat ppf fprintf ",@ " go (itnmap_to_list re)
  else
    ()

and pp_print_etysubst ps ppf (menv,tenv) =
  fprintf ppf "%a,@ %a" (pp_print_tysubst ps) tenv (pp_print_esubst ps) menv

and pp_print_etysubst_short ps ppf (menv,tenv) =
  fprintf ppf "%a,@ %a" (pp_print_tysubst_short ps) tenv (pp_print_esubst_short ps) menv

and pp_print_prim_expr_or_bs ps ppf v =
  match v with
    Util.Inl e -> pp_print_prim_expr ps ppf e
  | Util.Inr bs -> fprintf ppf "%a.*" (pp_print_bs ps) bs

and pp_print_flatten_env_short ps ppf re =
  (* assumes ps has bindings already *)
  if !Opts.printenv then
    if !Opts.printenvbodies then
      let go ppf (iid,v) =
        fprintf ppf "%a=%a"
          (pp_print_ident_use ps) iid
          (pp_print_prim_expr_or_bs ps) v
      in
      pp_concat ppf fprintf ",@ " go (iidmap_to_list re)
    else
      let go ppf (iid,v) =
        fprintf ppf "%a=.." (pp_print_ident_use ps) iid
      in
      pp_concat ppf fprintf ",@ " go (iidmap_to_list re)
  else
    ()

and pp_print_typeenv_pri ps ppf env =
  let go ps ppf tee is_val_prefix  = match tee with
  | (Env_val(id,ty) ) -> let ps' = add_ident_binder ps id in
    if true && (* (!Opts.suppress_econst_in_pp_print_typeenv)  *)
       true &&   (* (try ignore(ps.econst_string_of_ident id); true with _ -> false) *)
       is_val_prefix
    then
      (is_val_prefix,false,ps')
    else (
      fprintf ppf "%a:%a" (pp_print_ident_use ps') id  (pp_print_typ ps) ty;
      (is_val_prefix,true,ps'))
  | (Env_typ(tn,k)  ) -> let ps' = add_typname_binder ps tn in
                         fprintf ppf "%a:%a" (pp_print_typname_use ps') tn (pp_print_kind ps) k;
                         (false,true,ps')
  | (Env_loc(l,ty)  ) -> fprintf ppf "%a:%a" (pp_print_location ps) l (pp_print_typ ps) ty;
                         (false,true,ps)
  | (Env_mod(m,sign)) -> let ps' = add_modname_binder ps m in
                         fprintf ppf "@[%a@ : sig%a%a@ end@]"
                           (pp_print_modname ps') m
                           pp_ibreak ()
                          (pp_print_signature ps) sign;
                         (false,true,ps')
  in
  match env with
    [] -> pp_print_string ppf "empty"; ps
  | _  -> (
      pp_print_string ppf "[val prefix of typeenv elided]";
      pp_force_newline ppf ();
      snd (pp_concat_ps_b ps ppf fprintf ",@ " go env true) )

and pp_print_typeenv_ps ps ppf env =
  (* env is in reverse order *)
  pp_print_typeenv_pri ps ppf (List.rev (typeenv_list_of_typeenv env))

and pp_print_nameenv_entry ps ppf (ne: nameenv_entry) =
  match ne with
    NEnv_nmod (name_abs, ext, mhbody) ->
      if ps.ps_mode.pm_wide then (
        fprintf ppf "@[<hv>%a@ : nmodule %a@ : %a@  sig%a%a@ end version%a%a@ = struct%a%a@ end@]"
          (pp_print_abstract_name ps) name_abs
          pp_print_string (external_modname_to_string ext)
          (pp_print_eqs ps) mhbody.mh_eqs
          pp_ibreak ()
          (pp_print_signature ps) mhbody.mh_sign0
          pp_ibreak ()
          (pp_print_version ps) mhbody.mh_vne
          pp_ibreak ()
          (pp_print_structure ps) mhbody.mh_str)
      else (
        fprintf ppf "@[<hv>%a@ : nmodule %a : %a@,  sig@,@[<h 2>    %a@]@,  end@,  version %a@,= struct@,@[<h 2>    %a@]@,  end@]"
          (pp_print_abstract_name ps) name_abs
          pp_print_string (external_modname_to_string ext)
          (pp_print_eqs ps) mhbody.mh_eqs
          (pp_print_signature ps) mhbody.mh_sign0
          (pp_print_version ps) mhbody.mh_vne
          (pp_print_structure ps) mhbody.mh_str)
  | NEnv_nimp (name_abs, ext, ihbody) ->
      if ps.ps_mode.pm_wide then (
        fprintf ppf "@[<hv>%a@ : nimport %a@ : sig%a%a@ end version%a%a@ like struct%a%a@ end@]"
          (pp_print_abstract_name ps) name_abs
          pp_print_string (external_modname_to_string ext)
          pp_ibreak ()
          (pp_print_signature ps) ihbody.ih_sign0
          pp_ibreak ()
          (pp_print_version_constraint ps) ihbody.ih_vce
          pp_ibreak ()
          (pp_print_structure ps) ihbody.ih_likestr)
      else (
        fprintf ppf "@[<hv>%a@ : nimport %a@,: sig@,@[<h 2>    %a@]@,  end@,  version %a@,  like@[<hov 2>  struct@,@[<h 2>    %a@]@,  end@]@]"
          (pp_print_abstract_name ps) name_abs
          pp_print_string (external_modname_to_string ext)
          (pp_print_signature ps) ihbody.ih_sign0
          (pp_print_version_constraint ps) ihbody.ih_vce
          (pp_print_structure ps) ihbody.ih_likestr)
  | NEnv_type (n) ->
        fprintf ppf "@[<hv>%a@ : Type@]"
          (pp_print_abstract_name ps) n
  | NEnv_tname (n,t) ->
        fprintf ppf "@[<hv>%a@ : %a name@]"
          (pp_print_abstract_name ps) n
          (pp_print_typb ps) t

and pp_print_nameenv_list ps ppf l =
  pp_concat ppf fprintf ",@ " (pp_print_nameenv_entry ps) (List.rev l)

and pp_print_nameenv ps ppf (ne: nameenv) =
  if ps.ps_mode.pm_dumplevel = 0 then ()
  else
    (if not ps.ps_mode.pm_tex then
      fprintf ppf "@[{@ %a@ }@]" (pp_print_nameenv_list ps) (nameenv_list_of_nameenv ne)
    else
      fprintf ppf "@[\\{@ %a@ \\}@]" (pp_print_nameenv_list ps) (nameenv_list_of_nameenv ne))

and pp_print_nametypeenv_ps ps ppf (ne, te) =
  fprintf ppf "%a, " (pp_print_nameenv ps) ne;
  pp_print_typeenv_ps ps ppf te

and pp_print_name_value ps ppf (n : name_value) =
  if not ps.ps_mode.pm_full then
    pp_print_string ppf "name_value(...)"  (* FZ, I believe this is never used *)
  else if ps.ps_mode.pm_lithash then begin
    let an = abstract_name_of_name_value n in
    let t = type_of_name_value n in
        fprintf ppf "name_value(@[<hv>%a@ %a@])"
          (pp_print_abstract_name_really ps) an
          (pp_print_typ_argument ps) t                (* TODO *)
  end else if ps.ps_mode.pm_trail then begin
    let an = abstract_name_of_name_value n in
    let t = type_of_name_value n in
    let hstr =
      try
        HashAbstrName.find ps.abstrnamemap.an_map an
      with
        Not_found ->
          let hstr = ps.abstrnamemap.an_genname an in
          HashAbstrName.add ps.abstrnamemap.an_map an hstr;
          let (ppfsubtrail,ppfsubtrailbuf) = opentrail ps.ps_mode in (* FZ CHECK *)
          let ps' = { ps with ppftrail = ppfsubtrail; ppftrailbuf = ppfsubtrailbuf } in
          if ps.ps_mode.pm_tex then
            fprintf ps.ppftrail "    @[@<0>%s%a@<0>%s = %a@]@ " "\\textit{"
              pp_print_string hstr "}"
              (pp_print_name_value_really ps') n
          else
            fprintf ps.ppftrail "@[%a = %a@]@ "
              pp_print_string hstr
              (pp_print_name_value_really ps') n;
          flushtrail ps' ps.ppftrail;
          hstr
    in
    if ps.ps_mode.pm_tex then fprintf ppf "name_value(@[@<0>%s%s@<0>%s@ %a@])"
                                          "\\textit{" hstr "}" (pp_print_typ_argument ps) t
                         else fprintf ppf "name_value(@[%s@ %a@])" hstr (pp_print_typ_argument ps) t
  end else
    fprintf ppf "name_value(%a)" (pp_print_name_value_really ps) n

(* without name_value( )  *)
and pp_print_name_value_really ps ppf (n : name_value) =
  ( match n with
    | VHashMvf (_,h,x,t) ->
        fprintf ppf "@[<hv>hash(%a.%a)%a%a@]"
          (pp_print_hash ps) h
          (pp_print_ident_ext ps) x
          pp_ibreak ()
          (pp_print_typ_argument ps) t
    | VHashTs (_,t1,e) ->
        fprintf ppf "@[<hv>hash(%a,@ %a)@]"
          (pp_print_typ ps) t1
          (pp_print_esc_string) e
    | VHashHts (_,t1,e1,e2) ->
        fprintf ppf "@[<hv>hash(%a,@ %a,@ %a)@]"
          (pp_print_typ ps) t1
          (pp_print_esc_string) e1
          (pp_print_name_value ps) e2
    | VHashName (an,t) ->
        fprintf ppf "@[<hv>%a@ %a@]"
          (pp_print_abstract_name_really ps) an
          (pp_print_typ_argument ps) t )

and pp_print_thread_state ps ppf s = match s with
   TsRunnable       -> pp_print_string ppf "[RUNNABLE]"
 | TsSlowcall       -> pp_print_string ppf "[SLOWCALL]"
 | TsResolveBlocked       -> pp_print_string ppf "[RESOLVE_BLOCKED]"
 | TsMutexBlocked n -> fprintf ppf "[MUTEX_BLOCKED %a]" (pp_print_name_value ps) n
 | TsCVarWaiting n  -> fprintf ppf "[CVAR_WAITING %a]"  (pp_print_name_value ps) n

(* n: [state] defs ;; exp *)
and pp_print_thread_info ps ppf (ti: Ast.thread_info) =
  if ps.ps_mode.pm_dumplevel = 0 then () else begin
    pp_print_abstract_name ps ppf (abstract_name_of_name_value ti.ti_name);
    if ps.ps_mode.pm_dumplevel >=1 then begin
      fprintf ppf "@[: %a@ " (pp_print_thread_state ps) ti.ti_state;
      if ps.ps_mode.pm_enbrace_defns then fprintf ppf "{";
      let ps' = pp_print_definitions_pri ps ppf ti.ti_defs in
      if ps.ps_mode.pm_enbrace_defns then fprintf ppf "}";
      fprintf ppf  ";;@ %a@]" (pp_print_prim_expb ps') ti.ti_expr
    end
  end

(* MX (locked; waiting_list) *)
and pp_print_mutex_info ps ppf (mtxi: Ast.mutex_info) =
  fprintf ppf "@[MX(%a"
    (pp_print_abstract_name ps) (abstract_name_of_name_value mtxi.mtxi_name);
  ( match mtxi.mtxi_state with
      | None   -> fprintf ppf ";@ _)@]"
      | Some s -> fprintf ppf ";@ %a@ :@ %a)@]"
          (pp_print_name_value ps) s.lmtxi_owner
          (pp_print_queue (pp_print_name_value ps)) s.lmtxi_waiting )

(* CV (waiting_list) *)
and pp_print_cvar_info ps ppf (cvi: Ast.cvar_info) =
  fprintf ppf "@[CV(%a)@]"
    (pp_print_queue (pp_print_name_value ps)) cvi.cvi_waiting

(* n: MX(...) *)
and pp_print_mutex ps ppf (n, mtxi) =
  fprintf ppf "@[%a:%a@]"
    (pp_print_abstract_name ps) (abstract_name_of_name_value n)
    (pp_print_mutex_info ps) mtxi

(* n: CV(...) *)
and pp_print_cvar ps ppf ((n: name_value), cvi) =
  fprintf ppf "@[%a:%a@]"
    (pp_print_abstract_name ps) (abstract_name_of_name_value n)
    (pp_print_cvar_info ps) cvi

and pp_print_thread ps ppf ((n: name_value), ti) =
  pp_print_thread_info ps ppf ti

and pp_print_thread_name ps ppf ((n: name_value), ti) =
fprintf ppf "@[%a@]"
    (pp_print_abstract_name ps) (abstract_name_of_name_value n)

and pp_print_threaded_inner ps ppf ((n: name_value), (ti: Ast.threaded_smallstep_inner)) =
fprintf ppf "@[%a:%a@]"
    (pp_print_abstract_name ps) (abstract_name_of_name_value n)
    (pp_print_threaded_smallstep_inner ps) ti


(* - this doesn't surround the configuration by angle brackets *)
and pp_print_configuration ps ppf (cfg : Ast.configuration) =
  if ps.ps_mode.pm_dumplevel = 0 then () else
  fprintf ppf "@[<hv>";
  (if ps.ps_mode.pm_dumplevel >= 3 then
     fprintf ppf "EN:%a,%a" (pp_print_nameenv ps) cfg.cfg_nenv pp_ibreak ());
  if ps.ps_mode.pm_enbrace_defns then fprintf ppf "{";
  let ps' = pp_print_definitions_pri ps ppf cfg.cfg_defs in
  if ps.ps_mode.pm_enbrace_defns then fprintf ppf "}";
  (if ps.ps_mode.pm_dumplevel >=2 then
     fprintf ppf ",@ @[Es:{%a},@ s:{%a}@]"
       (pp_print_loctyplist ps') cfg.cfg_senv
       (pp_print_store ps') cfg.cfg_store);
  fprintf ppf ",@ @[@[threads:{%a}@],@ @[(RUNNABLE: %a,@ SLOWCALL: %a)@],@ MX:{%a}, CV:{%a}@]@]"
    (pp_print_namevaluemap ps' pp_print_thread) cfg.cfg_threads
    (pp_print_queue (pp_print_name_value ps')) cfg.cfg_runnable
    (pp_print_namevaluemap ps' pp_print_thread_name) cfg.cfg_slowcall
    (pp_print_namevaluemap ps' pp_print_mutex) cfg.cfg_mutexes
    (pp_print_namevaluemap ps' pp_print_cvar) cfg.cfg_cvars



(* - this doesn't surround the configuration by angle brackets *)
and pp_print_smallstep_outer ps ppf (scfg : Ast.smallstep_outer) =
  if ps.ps_mode.pm_dumplevel = 0 then () else
  fprintf ppf "@[<hv>";
  (if ps.ps_mode.pm_dumplevel >= 3 then
     fprintf ppf "EN:%a,%a" (pp_print_nameenv ps) scfg.scfg_nenv pp_ibreak ());
  if ps.ps_mode.pm_enbrace_defns then fprintf ppf "{";
  let ps' = pp_print_definitions_pri ps ppf scfg.scfg_defs in
  if ps.ps_mode.pm_enbrace_defns then fprintf ppf "}";
  (if ps.ps_mode.pm_dumplevel >=2 then
     fprintf ppf ",@ @[Es:{%a},@ s:{%a}@]"
       (pp_print_loctyplist ps') scfg.scfg_senv
       (pp_print_store ps') scfg.scfg_store);
  fprintf ppf ",@ @[@[threads:{%a}@],@ @[(RUNNABLE: %a,@ SLOWCALL: %a)@],@ MX:{%a}, CV:{%a}@]@]"
    (pp_print_namevaluemap ps' pp_print_threaded_inner) scfg.scfg_threads
    (pp_print_queue (pp_print_name_value ps')) scfg.scfg_runnable
    (pp_print_namevaluemap ps' pp_print_thread_name) scfg.scfg_slowcall
    (pp_print_namevaluemap ps' pp_print_mutex) scfg.scfg_mutexes
    (pp_print_namevaluemap ps' pp_print_cvar) scfg.scfg_cvars

(* TODO: print new bits *)
and pp_print_threaded_smallstep_inner ps ppf (eee : Ast.threaded_smallstep_inner) =

  if ps.ps_mode.pm_dumplevel = 0 then () else begin
    pp_print_abstract_name ps ppf (abstract_name_of_name_value eee.tsc_name);
    if ps.ps_mode.pm_dumplevel >=1 then begin
      fprintf ppf "@[: %a@ " (pp_print_thread_state ps) eee.tsc_state;
      match eee.tsc_state with
        TsSlowcall when ps.ps_mode.pm_dumplevel <=2 ->
          fprintf ppf "@]"
      | _ ->
          if ps.ps_mode.pm_enbrace_defns then fprintf ppf "{";
          let ps' = pp_print_definitions_pri ps ppf eee.tsc_defs in
          if ps.ps_mode.pm_enbrace_defns then fprintf ppf "}";
          let ps' = add_etysubst ps eee.tsc_env in
          pp_print_etysubst_short ps' ppf eee.tsc_env;
          let (ps_top, env_top, eqs_top) = pp_print_eval_ctx_pri ps' ppf eee.tsc_env eee.tsc_eqs eee.tsc_ec in
      (*    assert ((eqs_top = []));
            assert ((env_top = (IIdentMap.empty,ITypnameMap.empty))); *)
          fprintf ppf  ",@,%a@," (pp_print_eqs ps') eee.tsc_eqs;
          fprintf ppf  "%a@ :}@]" (pp_print_prim_expr ps') eee.tsc_expr;
          match eee.tsc_next_expr with None ->
            fprintf ppf ", None"
          | Some e ->
            fprintf ppf  ",Some %a@ :}@]" (pp_print_prim_expr ps') e
    end
  end



and pp_print_hash ps ppf (hash : hash) =
  if not ps.ps_mode.pm_full then
    pp_print_string ppf "hash(...)"
  else if ps.ps_mode.pm_lithash then begin
    let h = hash_of_hash hash in
    pp_print_string ppf (string_of_lithash h)
  end else if ps.ps_mode.pm_trail then begin
    let h = hash_of_hash hash in
    let hstr =
      try
        HashHash.find ps.hashmap.hm_map h
      with
        Not_found ->
          let hstr = ps.hashmap.hm_genname hash in
          HashHash.add ps.hashmap.hm_map h hstr;
          (* deal with hashes in hashes; mustn't just use the same ppftrail because
             we'd get hash defs interleaved in hashes *)
          let (ppfsubtrail,ppfsubtrailbuf) = opentrail ps.ps_mode in
          let ps' = { ps with ppftrail = ppfsubtrail; ppftrailbuf = ppfsubtrailbuf } in
          if ps.ps_mode.pm_tex then
            fprintf ps.ppftrail "    @[@<0>%s%a@<0>%s @[= %a@ = %a@]@]@ " "\\textit{"
              pp_print_string hstr "}"
              (pp_print_hash_really ps') hash
              pp_print_string (string_of_lithash(hash_of_hash hash))
          else
            fprintf ps.ppftrail "    @[%a @[= %a@ = %a@]@]@ "
              pp_print_string hstr
              (pp_print_hash_really ps') hash
              pp_print_string (string_of_lithash(hash_of_hash hash));
          (* P wants that literal hash aligned with the symbolic one *)
          flushtrail ps' ps.ppftrail;
          hstr
    in
    if ps.ps_mode.pm_tex then fprintf ppf "@<0>%s%s@<0>%s" "\\textit{" hstr "}"
                         else pp_print_string ppf hstr
  end else
    pp_print_hash_really ps ppf hash

and pp_print_hash_really ps ppf (hash : hash) =
    (* MUST NOT LOOK AT LITHASH PART OF HashM/HashI *)
    let ps = empty_scope ps in
    (* NB: This used to change the mode to pm_hash as well; but really
    we only want to do that if this is actually being used for a hash,
    not when we're just displaying for the user/debugger.  In any
    case, one of the things pm_hash does is set pm_wide; but changing
    ps doesn't automatically change the margin of ppf - that is set
    only when ppf is initially constructed.  So it was broken anyway. *)
      begin match hash with
        HashM (_, ext, mhbody) ->
        if ps.ps_mode.pm_wide then (
          fprintf ppf "hash(@[<hv>hmodule %a@ : %a@  sig%a%a@ end version%a%a@ = struct%a%a@ end@])"
            pp_print_string (external_modname_to_string ext)
            (pp_print_eqs ps) mhbody.mh_eqs
            pp_ibreak ()
            (pp_print_signature ps) mhbody.mh_sign0
            pp_ibreak ()
            (pp_print_version ps) mhbody.mh_vne
            pp_ibreak ()
            (pp_print_structure ps) mhbody.mh_str)
        else (
          fprintf ppf "hash(@[<hv>hmodule %a : %a@,  sig@,@[<h 2>    %a@]@,  end@,  version %a@,= struct@,@[<h 2>    %a@]@,  end@])"
            pp_print_string (external_modname_to_string ext)
            (pp_print_eqs ps) mhbody.mh_eqs
            (pp_print_signature ps) mhbody.mh_sign0
            (pp_print_version ps) mhbody.mh_vne
            (pp_print_structure ps) mhbody.mh_str)
      | HashI (_, ext, ihbody) ->
        if ps.ps_mode.pm_wide then (
          fprintf ppf "hash(@[<hv>himport %a@ : sig%a%a@ end version%a%a@ like struct%a%a@ end@])"
            pp_print_string (external_modname_to_string ext)
            pp_ibreak ()
            (pp_print_signature ps) ihbody.ih_sign0
            pp_ibreak ()
            (pp_print_version_constraint ps) ihbody.ih_vce
            pp_ibreak ()
            (pp_print_structure ps) ihbody.ih_likestr)
        else (
          fprintf ppf "hash(@[<hv>himport %a@,: sig@,@[<h 2>    %a@]@,  end@,  version %a@,  like@[<hov 2>  struct@,@[<h 2>    %a@]@,  end@]@])"
            pp_print_string (external_modname_to_string ext)
            (pp_print_signature ps) ihbody.ih_sign0
            (pp_print_version_constraint ps) ihbody.ih_vce
            (pp_print_structure ps) ihbody.ih_likestr)
      | HashLit lithash ->
          pp_print_string ppf (string_of_lithash lithash)
      | HashName abstract_name ->
          pp_print_abstract_name ps ppf abstract_name
      end

(* == HASHING HASHES == *)
(* (in this file to avoid cyclic dependency with Pretty.pp_print_hash) *)

(* compute the 16*8=128-bit hash value of a hash, for comparison *)
and mkHashM : (internal_ident -> string) -> external_modname * mod_hash_body -> hash
  = fun econst_string_of_ident (emn,mh)
 -> let printer_state = initial_printer_state (Some econst_string_of_ident) pm_hash in
    let dummy = hash_of_int 0 in
    let canonical_string = wrap pp_print_hash_really printer_state (HashM(dummy,emn,mh)) in
    Debug.print' Opts.DBC_mkhash (fun () -> "\n>>>>mkHashM of:\n"^canonical_string^"\n<<<<\n");
    HashM(Digest.string canonical_string,emn,mh)

and mkHashI : (internal_ident -> string) -> external_modname * import_hash_body -> hash
  = fun econst_string_of_ident (emn,ih)
 -> let printer_state = initial_printer_state (Some econst_string_of_ident) pm_hash in
    let dummy = hash_of_int 0 in
    let canonical_string = wrap pp_print_hash_really printer_state (HashI(dummy,emn,ih)) in
    Debug.print' Opts.DBC_mkhash (fun () -> "\n>>>>mkHashI of:\n"^canonical_string^"\n<<<<\n");
    HashI(Digest.string canonical_string,emn,ih)

and hash_of_hash : hash -> Digest.t
  = fun h
 -> match h with
    | HashM(lithash,_,_) -> lithash
    | HashI(lithash,_,_) -> lithash
    | HashLit(lithash)   -> lithash
    | HashName(lithash)  -> lithash

(* compute the 16*8=128-bit hash value of a natural literal, for comparison *)
and hash_of_int : int -> Digest.t
  = fun n
 -> (if n<0 then raise (Util.Never_happen "hash_of_int of negative value") else ());
    List.fold_left (fun s i -> s ^ String.make 1 (char_of_int (i mod 256))) ""
      [0;0;0;0;0;0;0;0;n lsr 56;n lsr 48;n lsr 40;n lsr 32;n lsr 24;n lsr 16;n lsr 8;n]

let mkAbstrNameMvf (h, eid, t) = (* FZ: should we hash t here ??? *)  (* K: no, 'cos it's determined by h and eid *)
  let str = (string_of_lithash (hash_of_hash h)) ^ (external_ident_to_string eid) in
  VHashMvf(Digest.string str, h, eid, t)

let mkAbstrNameTs (t, s) =
  let ps = initial_printer_state None pm_hash in
  let str = ((wrap pp_print_typ) ps t) ^ s in
  VHashTs(Digest.string str, t, s)

let mkAbstrNameHts (t, s, n') =
  let ps = initial_printer_state None pm_hash in
  let str = ((wrap pp_print_typ) ps t) ^ s ^ abstract_name_of_name_value n' in
  VHashHts(Digest.string str, t, s, n')


let pp_print_name_use ps ppf n = pp_print_abstract_name_really ps ppf n

(* == Simple entry points (which return strings) == *)

let print_name_use = wrap pp_print_name_use
let print_ident_use = wrap pp_print_ident_use
let print_typname_use = wrap pp_print_typname_use
let print_name_value = wrap pp_print_name_value
let print_modname = wrap pp_print_modname
let print_ident_ext = wrap pp_print_ident_ext
let print_typname_ext = wrap pp_print_typname_ext
let print_ident_bind = wrap pp_print_ident_bind
let print_typname_bind = wrap pp_print_typname_bind
let print_program = wrap pp_print_program
let print_signature = wrap pp_print_signature
let print_signature_ps = wrap' pp_print_signature_ps
let print_structure = wrap pp_print_structure
let print_signature_item = wrap pp_print_signature_item
let print_atomic_version = wrap pp_print_atomic_version
let print_version = wrap pp_print_version
let print_version_constraint = wrap pp_print_version_constraint
let print_withspec = wrap pp_print_withspec
let print_likespec = wrap pp_print_likespec
let print_resolvespec_item = wrap pp_print_resolvespec_item
let print_resolvespec = wrap pp_print_resolvespec
let print_typeenv_ps = wrap' pp_print_typeenv_ps
let print_nameenv = wrap pp_print_nameenv
let print_nametypeenv_ps = wrap' pp_print_nametypeenv_ps
let print_store = wrap pp_print_store
let print_esubst = wrap pp_print_esubst
let print_tysubst = wrap pp_print_tysubst
let print_etysubst = wrap pp_print_etysubst
let print_typ = wrap pp_print_typ
let print_kind = wrap pp_print_kind
let print_eq = wrap pp_print_eq
let print_eqs = wrap pp_print_eqs
let print_prim_expr = wrap pp_print_prim_expr
let print_expr = wrap pp_print_expr
let print_expr_loc = wrap pp_print_expr_loc
let print_mtch = wrap pp_print_mtch
let print_pat = wrap pp_print_pat
let print_pat_ps = wrap' pp_print_pat_ps
let print_definition = wrap pp_print_definition
let print_definition_ps = wrap' pp_print_definition_ps
let print_definitions = wrap pp_print_definitions
let print_definitions_ps = wrap' pp_print_definitions_ps
let print_source_definition = wrap pp_print_source_definition
let print_source_definition_ps = wrap' pp_print_source_definition_ps
let print_source_definitions = wrap pp_print_source_definitions
let print_source_definitions_ps = wrap' pp_print_source_definitions_ps
let print_compilation_unit ps ppf = wrap pp_print_compilation_unit ps ppf
let print_compiled_unit ps ppf = wrap pp_print_compiled_unit ps ppf
let print_location = wrap pp_print_location
let print_configuration = wrap pp_print_configuration
(* let print_smallstep_inner = wrap pp_print_smallstep_inner *)
let print_thread_info = wrap pp_print_thread_info
let print_threaded_smallstep_inner = wrap pp_print_threaded_smallstep_inner
let print_smallstep_outer = wrap pp_print_smallstep_outer
let print_hash = wrap pp_print_hash
let print_marshalled_value = wrap pp_print_marshalled_value

let _ = Ast.basic_print_expr := function e -> print_expr (initial_printer_state None (pm_debug ())) e
let _ = Ast.basic_print_prim_expr := print_prim_expr (initial_printer_state None (pm_debug ()))
let _ = Ast.basic_print_esubst := fun env -> wrap pp_print_esubst_short (add_esubst (initial_printer_state None (pm_debug ())) env) env
