(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * Datatypes for Acute abstract syntax trees.                     *
 * (Returned by the parser and used everywhere).                  *
 *                                                                *
 * Version: $Id: ast.ml,v 1.637 2004/12/31 11:49:36 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Util
open Basecon

(* fresh names with a suggested string name *)

type 'a prettyname = 'a name * string

let string_of_prettyname : 'a prettyname -> string
  = fun (_,s) -> s

let fresh_prettyname : 'a prettyname -> 'a prettyname
  = fun (i,s) -> (fresh,s)

let swap_prettyname : 'a prettyname -> 'a prettyname -> 'b -> 'b
  = fun (i,_) (i',_) v -> swap i and i' in v

let name_of_prettyname : 'a prettyname -> 'a name
  = fun (i,_) -> i

let string_of_name : 'a name -> string
  = fun i -> Printf.sprintf "%08x:%06x" (Obj.magic i) (fst (Obj.magic i))
      (* first string is the address of the atom,
         second string is the unique integer it contains *)

let debug_string_of_prettyname : 'a prettyname -> string
  = fun x ->
    string_of_prettyname x ^ "(*" ^ string_of_name (name_of_prettyname x) ^ "*)"


(* various kinds of identifiers *)

type external_typname = string
type internal_typname_T and internal_typname = internal_typname_T prettyname
type typname = external_typname * internal_typname

type external_ident = string
type internal_ident_T and internal_ident = internal_ident_T prettyname
type ident = external_ident * internal_ident

type external_modname = string
type internal_modname_T and internal_modname = internal_modname_T prettyname
type modname = external_modname * internal_modname

let eid_compare = (String.compare)

module OrderedIIdent =
  struct
    type t = internal_ident
    let compare x y =
      let xn = name_of_prettyname x in
      let yn = name_of_prettyname y in
      let r = Pervasives.compare xn yn in
(*      Debug.print' Opts.DBC_evalstep (fun () -> "compare ("^string_of_name xn^", "^string_of_name yn^") = "^string_of_int r); *)  (* Pervasives.compare may not be the best thing for names, since by default in Fresh they are generated in sorted order *)
      r
  end

module IIdentMap = Map.Make(OrderedIIdent)
(* module IIdentMap :
  sig
    type key = internal_ident
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end
  = struct
    type key = internal_ident
    type 'a t = (key*'a) list
    let empty = []
    let add = fun k el l -> (k,el)::l
    let find k l =
      let rec find_int l =
	match l with
	| (xk,x)::xs -> if xk = k then x else find_int xs
        | _ -> raise Not_found
      in find_int l
    let remove k l =
      let rec remove_int l la =
	match l with
	| (xk,x)::xs -> if xk = k then (List.rev(la))@xs else remove_int xs ((xk,x)::la)
        | _ -> List.rev la
      in remove_int l []
    let mem k l =
      let rec find_int l =
	match l with
	| (xk,x)::xs -> if xk = k then true else find_int xs
        | [] -> false
      in find_int l
    let iter f l =
      let rec iter_int l =
	match l with
	| (xk,x)::xs -> f xk x; iter_int xs
	| [] -> ()
      in iter_int l
    let map : ('a -> 'b) -> 'a t -> 'b t = fun  (f: 'a -> 'b) (l : 'a t) ->
      let rec map_int l =
	match l with
	| (xk,x)::xs -> let r = f x in [] (* (r :: map_int xs) *)
	| [] -> []
      in map_int l
    let mapi f l =
      let rec mapi_int l =
	match l with
	| (xk,x)::xs -> let r = f xk x in [] (* (r :: mapi_int xs) *)
	| [] -> []
      in mapi_int l

    let fold f l e =
      let rec fold_int l =
	match l with
	| (xk,x)::xs -> f xk x (fold_int xs)
	| [] -> e
      in fold_int l


  end *)



module OrderedITypname =
  struct
    type t = internal_typname
    let compare x y = Pervasives.compare (name_of_prettyname x) (name_of_prettyname y)
  end

module ITypnameMap = Map.Make(OrderedITypname)

module OrderedModname =
  struct
    type t = modname
    let compare x y = Pervasives.compare x y  (* NB: ext *is* significant here *)
  end

module ModnameMap = Map.Make(OrderedModname)

module ModnameSet = Set.Make(OrderedModname)

module OrderedEModname =
  struct
    type t = external_modname
    let compare x y = Pervasives.compare x y  (* NB: ext *is* significant here *)
  end

module EModnameMap = Map.Make(OrderedEModname)

type location = int  (* reference cell names *)

let fresh_location =
  let counter = ref 0 in
    function () -> (incr counter; !counter)

module OrderedLoc =
  struct
    type t = location
    let compare x y = Pervasives.compare x y  (* built-in is fine *)
  end

module LocMap = Map.Make(OrderedLoc)

module OrderedEIdent =
  struct
    type t = external_ident
    let compare x y = Pervasives.compare x y
  end

module EIdentMap = Map.Make(OrderedEIdent)
(* module EIdentMap :
  sig
    type key = external_ident
    type 'a t
    val empty : 'a t
    val add : key -> 'a -> 'a t -> 'a t
    val find : key -> 'a t -> 'a
    val remove : key -> 'a t -> 'a t
    val mem : key -> 'a t -> bool
    val iter : (key -> 'a -> unit) -> 'a t -> unit
    val map : ('a -> 'b) -> 'a t -> 'b t
    val mapi : (key -> 'a -> 'b) -> 'a t -> 'b t
    val fold : (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
  end
  = struct
    type key = external_ident
    type 'a t = (key*'a) list
    let empty = []
    let add = fun k el l -> (k,el)::l
    let find k l =
      let rec find_int l =
	match l with
	| (xk,x)::xs -> if xk = k then x else find_int xs
        | _ -> raise Not_found
      in find_int l
    let remove k l =
      let rec remove_int l la =
	match l with
	| (xk,x)::xs -> if xk = k then (List.rev(la))@xs else remove_int xs ((xk,x)::la)
        | _ -> List.rev la
      in remove_int l []
    let mem k l =
      let rec find_int l =
	match l with
	| (xk,x)::xs -> if xk = k then true else find_int xs
        | [] -> false
      in find_int l
    let iter f l =
      let rec iter_int l =
	match l with
	| (xk,x)::xs -> f xk x; iter_int xs
	| [] -> ()
      in iter_int l
    let map : ('a -> 'b) -> 'a t -> 'b t = fun  (f: 'a -> 'b) (l : 'a t) ->
      let rec map_int l =
	match l with
	| (xk,x)::xs -> let r = f x in [] (* (r :: map_int xs) *)
	| [] -> []
      in map_int l
    let mapi f l =
      let rec mapi_int l =
	match l with
	| (xk,x)::xs -> let r = f xk x in [] (* (r :: mapi_int xs) *)
	| [] -> []
      in mapi_int l

    let fold f l e =
      let rec fold_int l =
	match l with
	| (xk,x)::xs -> f xk x (fold_int xs)
	| [] -> e
      in fold_int l


  end
*)
module OrderedETypname =
  struct
    type t = external_typname
    let compare x y = Pervasives.compare x y
  end

module ETypnameMap = Map.Make(OrderedETypname)

type 'a located = { desc : 'a; loc : Location.t }

let gh_loc : Location.t -> 'a -> 'a located
  = fun l x -> { desc = x; loc = {l with Location.loc_ghost=true} }


let at_loc : Location.t -> 'a -> 'a located
  = fun l x -> { desc = x; loc = l }

let no_loc : 'a -> 'a located
  = fun x -> at_loc Location.none x

(* basic types - mostly defined in Basecon FZ *)

(* types *)
type typ  =
  | TTyCon0 of tycon0                             (* nullary types *)
  | TTyCon1 of tycon1 * typ                       (* unary types *)
  | TTup of typ list                              (* tuple type *)
  | TSum of typ list                              (* sum type *)
  | TTyName of abstract_name                      (* type name *)
  | TXDot of hash_or_modname * external_typname   (* module or hash projection *)
  | TFunc of typ * typ                            (* function type *)
  | TVar of internal_typname                      (* type variable or metavariable *)
  | TForall of internal_typname * typ             (* universal quantification *)
  | TExists of internal_typname * typ             (* existential quantification *)

(* kinds *)
and  kind = KType                       (* kind of all types *)
          | KEq of typ                  (* singleton kind *)

(* hashes *)
(* use Pretty.mkHashM and Pretty.mkHashI *)
and  hash = HashM of Digest.t * external_modname * mod_hash_body
          | HashI of Digest.t * external_modname * import_hash_body
          | HashLit of Digest.t
          | HashName of abstract_name

and hash_or_modname = Hash of hash
                    | Modname of modname

(* equation annotations for coloured brackets *)
and eqn = EHash of hash * external_typname * typ
        | EMod of modname * external_typname * typ

and eqs = eqn list

and abstract_name = Digest.t


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(* Expressions and all related things *)

(* Constructors of arity 0 *)
and con0 =
        | Nil of typ   (* the type of the whole expression is: typ list *)
        | NONE of typ  (* the type of the whole expression is: typ option *)
        | B0 of basecon0

(* Constructors of arity 1 *)
and con1 =
  | Inj of int * typ
  | SOME
  | TieCon
  | Node               (* Node of a tree *)
  | B1 of basecon1

(* sequence of nested brackets (BC in the semantics).  The head is the *innermost* bracket. *)
and bracket_seq = (eqs * typ) list

(* either an Op or an Econst; appears in OP form *)
and op_or_econst = OEOp of op
                 | OEEconst of internal_ident

and lazy_op = LoAnd | LoOr

(* Name value (nnnn in the spec) *)
and name_value =
    VHashMvf  of abstract_name * hash * external_ident * typ
  | VHashTs   of abstract_name * typ * string
  | VHashHts  of abstract_name * typ * string * name_value
  | VHashName of abstract_name * typ

(* Expressions *)
and expr_desc =
          LocC0 of con0
        | LocC1 of con1 * expr
        | LocCons of expr * expr
        | LocTup of expr list
        | LocOp of op_or_econst * expr list
              (* this is a sugared form: the source form requires the
                 list to be empty, and the core form requries that the
                 list has length n, where n is the arity of the op *)
        | LocLoc of location  (* if you add a type to this, please fix Evalsupp.early_emap_fresh_prim_expr *)
        | LocFn  of mtch  (* this is a sugared form; the core form has mtch = [(PVar(x,t),e)] *)
        | LocFun  of pat list * expr   (* this is a sugared form; fun p1 .. pn -> e *)
        | LocTAbs of internal_typname * expr  (* type abstraction *)
        | LocInEnv of etysubst * expr
              (* this is a runtime-only form *)
              (* it is a delayed substitution, binding *all* vars in expr *)
              (* See flattenclos for the precise meaning of the term *)
        | LocClos of etysubst Lazy.t
              * internal_ident * typ
              * bracket_seq * expr
              * (internal_ident * typ * bracket_seq) option
              (* this is a runtime-only form *)
              (* remember to freshen_clos *)
              (* Clos(env,x2,ty2,bs2,e1,None)              - nonrecursive closure
                 Clos(env,x2,ty2,bs2,e1,Some(x1,ty1,bs1))  - recursive closure
                 env      - substitution environment
                 x2 : ty2 - argument
                 bs2      - brackets to apply around x2 in e1
                 e1       - body
                 x1 : ty1 - name of this closure in env
                 bs1      - brackets to apply around x1 in e1
              *)
              (* Closures carry an environment for their free variables.
                 This is (often) hidden from the printed
                 representation and from the typechecker.
                 See flattenclos for the precise meaning of the term. *)
        | LocTClos of etysubst
              * internal_typname
              * expr
              (* this is a runtime-only form *)
              (* remember to freshen *)
              (* TClos(menv,tenv,itn,e)   (type closures are never recursive!)
                 (menv,tenv)  - substitution environment
                 itn          - argument (always of kind Type)
                 e            - body
                 NB: bracket-pushing through closure is *weird*
                 because tyvar ends up at different colour. Hmm.
              *)
        | LocId of internal_ident
        | LocDot of modname * external_ident  (* module projection *)
        | LocHashDot of hash * external_ident (* hash projection *)
        | LocIf of expr * expr * expr
        | LocWhile of expr * expr
        | LocLazyOp of lazy_op * expr list
        | LocSeq of expr * expr
        | LocApp of expr * expr
        | LocTApp of expr * typ               (* type application *)
        | LocPack of typ * expr * typ       (* existential package *)
        | LocNamecase of expr * internal_typname * internal_ident * internal_ident * expr * expr * expr  (* unpackaging and name equality *)
        | LocUnpack of internal_typname * internal_ident * expr * expr (* unpackaging *)
        | LocMatch of expr * mtch
        | LocLet of expr * (pat * expr)
             (* this is a sugared-only form; we have that
                  let p = e' in e''
                is roughly represented by
                  Let (e', (p, e'')) *)
             (* remember to freshen_pat_expr *)
        | LocLetMulti of typ * (pat list * expr) * (internal_ident(*rtfif*) * expr)
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetMulti (T, ([p1;..; pn], e'), (x,e'')) *)
             (* remember to freshen_pats_foo *)
        | LocLetrec of typ * (internal_ident(*rtfif*) * (mtch * expr))
             (* this includes a sugared form;
                the core form has mtch = [(PVar(x,t),e)] *)
        | LocLetrecMulti of typ * (pat list * (internal_ident(*rtfif*) * (expr * expr)))
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let rec (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetrecMulti (T, ([p1;..; pn], (x,(e',e'')))) *)
             (* remember to freshen_pats_foo (ps,e') *)
        | LocRaise of expr  (* not fully type annotated, but it's too hard to maintain the type annotation :-( *)
        | LocTry of expr * mtch
        | LocMarshal of expr * expr * typ
        | LocMarshalz of string * expr * typ
        | LocUnmarshal of expr * typ
        | LocRET of typ
        | LocSLOWRET of typ
        | LocCol of expr * eqs * typ       (* coloured brackets *)
        | LocOP of int * op_or_econst * expr list  (* int is n, the total number of args.  I think (-K). *)
        | LocResolve of expr * modname * resolvespec  (* NB: expr is more general than the semantics allows *)
        | LocResolve_blocked of expr * modname * resolvespec  (* ditto *)
        | LocPar of expr * expr  (* e1 | e2 : sugared form *)
        | LocFresh of typ
        | LocCFresh of typ
        | LocHashMvf of hash_or_modname * external_ident * typ  (* hash(X.x)_T *)
        | LocHashTs of typ * expr * typ                         (* hash(T,s)_T' *)
        | LocHashHts of typ * expr * expr * typ                 (* hash(T,s,h)_T' *)
        | LocNameValue of name_value
        | LocSwap of expr * expr * expr
        | LocFreshfor of expr * expr
        | LocSupport of typ * expr
        | LocTie of modname * external_ident
        | LocNameOfTie of expr
        | LocValOfTie of expr

and prim_expr =
          C0 of con0
        | C1 of con1 * prim_expr
        | Cons of prim_expr * prim_expr
        | Tup of prim_expr list
        | Op of op_or_econst * prim_expr list
              (* this is a sugared form: the source form requires the
                 list to be empty, and the core form requries that the
                 list has length n, where n is the arity of the op *)
        | Loc of location
        | Fn  of prim_mtch  (* this is a sugared form; the core form has mtch = [(PVar(x,t),e)] *)
        | Fun of pat list * prim_expr (* this is a sugared form; fun p1 .. pn -> e *)
        | TAbs of internal_typname * prim_expr  (* type abstraction *)
        | InEnv of etysubst * prim_expr
              (* this is a runtime-only form *)
              (* it is a delayed substitution, binding *all* vars in expr *)
              (* See flattenclos for the precise meaning of the term *)
        | Clos of etysubst Lazy.t
              * internal_ident * typ
              * bracket_seq * prim_expr
              * (internal_ident * typ * bracket_seq) option
              (* this is a runtime-only form *)
              (* remember to freshen_clos *)
              (* Clos(env,x2,ty2,bs2,e1,None)              - nonrecursive closure
                 Clos(env,x2,ty2,bs2,e1,Some(x1,ty1,bs1))  - recursive closure
                 env      - substitution environment
                 x2 : ty2 - argument
                 bs2      - brackets to apply around x2 in e1
                 e1       - body
                 x1 : ty1 - name of this closure in env
                 bs1      - brackets to apply around x1 in e1
              *)
              (* Closures carry an environment for their free variables.
                 This is (often) hidden from the printed
                 representation and from the typechecker.
                 See flattenclos for the precise meaning of the term. *)
        | TClos of etysubst
              * internal_typname
              * prim_expr
              (* this is a runtime-only form *)
              (* remember to freshen *)
              (* TClos(menv,tenv,itn,e)   (type closures are never recursive!)
                 (menv,tenv)  - substitution environment
                 itn          - argument (always of kind Type)
                 e            - body
                 NB: bracket-pushing through closure is *weird*
                 because tyvar ends up at different colour. Hmm.
              *)
        | Id of internal_ident
        | Dot of modname * external_ident  (* module projection *)
        | HashDot of hash * external_ident (* hash projection *)
        | If of prim_expr * prim_expr * prim_expr
        | While of prim_expr * prim_expr
        | LazyOp of lazy_op * prim_expr list
        | Seq of prim_expr * prim_expr
        | App of prim_expr * prim_expr
        | TApp of prim_expr * typ               (* type application *)
        | Pack of typ * prim_expr * typ       (* existential package *)
        | Unpack of internal_typname * internal_ident * prim_expr * prim_expr (* unpackaging *)
        | Namecase of prim_expr * internal_typname * internal_ident * internal_ident * prim_expr * prim_expr * prim_expr  (* unpackaging and name equality *)
        | Match of prim_expr * prim_mtch
        | Let of prim_expr * (pat * prim_expr)
             (* this is a sugared-only form; we have that
                  let p = e' in e''
                is roughly represented by
                  Let (e', (p, e'')) *)
             (* remember to freshen_pat_expr *)
        | LetMulti of typ * (pat list * prim_expr) * (internal_ident (*rtfif*) * prim_expr)
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetMulti (T, ([p1;..; pn], e'), (x,e'')) *)
             (* remember to freshen_pats_foo *)
        | Letrec of typ * (internal_ident(*rtfif*) * (prim_mtch * prim_expr))
             (* this includes a sugared form;
                the core form has mtch = [(PVar(x,t),e)] *)
        | LetrecMulti of typ * (pat list * (internal_ident (*rtfif*) * (prim_expr * prim_expr)))
             (* this is a sugared-only form; pat list is nonempty. we have that
                  let rec (x:T) p1 .. pn = e' in e''
                is roughly represented by
                  LetrecMulti (T, ([p1;..; pn], (x,(e',e'')))) *)
             (* remember to freshen_pats_foo (ps,e') *)
        | Raise of prim_expr  (* not fully type annotated, but it's too hard to maintain the type annotation :-( *)
        | Try of prim_expr * prim_mtch
        | Marshal of prim_expr * prim_expr * typ
        | Marshalz of string * prim_expr * typ
        | Unmarshal of prim_expr * typ
        | RET of typ
        | SLOWRET of typ
        | Col of prim_expr * eqs * typ       (* coloured brackets *)
        | OP of int * op_or_econst * prim_expr list  (* int is n, the total number of args.  I think (-K). *)
        | Resolve of prim_expr * modname * resolvespec  (* NB: expr is more general than the semantics allows *)
        | Resolve_blocked of prim_expr * modname * resolvespec  (* ditto *)
        | Par of prim_expr * prim_expr (* e1 | e2 : sugared form *)
        | Fresh of typ
        | CFresh of typ
        | HashMvf of hash_or_modname * external_ident * typ
        | HashTs of typ * prim_expr * typ
        | HashHts of typ * prim_expr * prim_expr * typ
        | NameValue of name_value
        | Swap of prim_expr * prim_expr * prim_expr
        | Freshfor of prim_expr * prim_expr
        | Support of typ * prim_expr
        | Tie of modname * external_ident
        | NameOfTie of prim_expr
        | ValOfTie of prim_expr

and expr = expr_desc located

(* Evaluation contexts *)
and eval_ctx =  (* ...from the inside *)
    CtxTop
  | CtxC1 of con1 * eval_ctx
  | CtxConsL of eval_ctx * prim_expr
  | CtxConsR of prim_value * eval_ctx
  | CtxTup of prim_expr list * eval_ctx * prim_value list  (* Note 3 *)
  | CtxInEnv of etysubst * eval_ctx  (* Note 2 *)
  | CtxIf of eval_ctx * prim_expr * prim_expr
  | CtxSeq of eval_ctx * prim_expr
  | CtxAppL of eval_ctx * prim_expr
  | CtxAppR of prim_value * eval_ctx (* int is no of arguments needed by the value *)
  | CtxTApp of eval_ctx * typ
  | CtxUnpack of internal_typname * internal_ident * eval_ctx * prim_expr
  | CtxNamecaseL of eval_ctx * internal_typname * internal_ident * internal_ident * prim_expr * prim_expr * prim_expr
  | CtxNamecaseR of prim_value * internal_typname * internal_ident * internal_ident * eval_ctx * prim_expr * prim_expr
  | CtxMatch of eval_ctx * prim_mtch
  | CtxRaise of eval_ctx
  | CtxTry of eval_ctx * prim_mtch
  | CtxMarshalL of eval_ctx * prim_expr * typ
  | CtxMarshalR of prim_value * eval_ctx * typ
  | CtxMarshalz of eqs * string * eval_ctx * typ
  | CtxUnmarshal of eval_ctx * typ
  | CtxCol of eval_ctx * eqs * typ       (* coloured brackets *)
  | CtxOp of op_or_econst * prim_value list * eval_ctx * prim_expr list
  | CtxLazyOp of lazy_op * eval_ctx * prim_expr
  | CtxOP of eqs * int * op_or_econst * prim_expr list * eval_ctx * prim_value list  (* Note 3 *)
  | CtxHashTs of eqs * typ * eval_ctx * typ
  | CtxHashHtsL of eqs * typ * eval_ctx * prim_expr * typ
  | CtxHashHtsR of eqs * typ * prim_value * eval_ctx * typ
  | CtxSwapL of eval_ctx * prim_expr * prim_expr
  | CtxSwapM of prim_value * eval_ctx * prim_expr
  | CtxSwapR of prim_value * prim_value * eval_ctx
  | CtxFreshforL of eval_ctx * prim_expr
  | CtxFreshforR of prim_value * eval_ctx
  | CtxSupport of typ * eval_ctx
  | CtxNameOfTie of eval_ctx
  | CtxValOfTie of eval_ctx
  | CtxPack of typ * eval_ctx * typ
(* Module init stuff *)
  | CtxStrVal of ident * eval_ctx * Location.t
  | CtxImodule of eqs * modname * mod_compile_body_ctx * eval_ctx (* Note 1 *)


(* Note 1: in the colour-changing contexts CtxMarshalz, CtxCol, CtxOP,
           CtxHashTs, CtxHashHtsL, and CtxHashHtsR, the eqs records
           the *outer* colour *)
(* Note 2: in the environment-changing context CtxInEnv, the etysubst records the *outer* environment *)
(* Note 3: elements (e1,e2,e3,__,v5,v6,v7) stored as ([e3;e2;e1],__,[v5;v6;v7]) *)


and language =
  | SugaredSourceInternalForm   (* user source, abstractly - sugar permitted *)
  | SourceInternalForm          (* desugared abstract user source *)
  | SourceInternalFreshForm     (* cfresh not permitted; for Mod_fresh *)
  | CompiledForm                (* compiled / executing form *)
      (* TODO: could subdivide Compiled into compiled and executing *)


and value = expr  (* just for documentation *)
and prim_value = prim_expr (* just for documentation *)


(* Matches *)
and mtch = (pat * expr) list   (* remember to freshen_pat_expr *)

and prim_mtch = (pat * prim_expr) list located   (* remember to freshen_pat_expr *)


(* Patterns *)
and pat =
          PWild of typ
        | PVar of internal_ident * typ
        | PC0 of con0
        | PC1 of con1 * pat
        | PCons of pat * pat
        | PTup of pat list
        | PTyped of pat * typ

and esubst = prim_expr IIdentMap.t
(* simultaneous substitution: all domain idents are distinct and none
   binds in any of the range expressions *)
(* TODO: not sure if this is the semantics currently adopted in the bitrotted eval *)
and tysubst = typ ITypnameMap.t

and etysubst = esubst * tysubst

(* primitive ops *)

(* TODO: deref etc should be primitives *)
and op =
        | Ref of typ    (* ref *)
              (* Ref ty has language type: ty -> ty ref *)
        | Deref of typ  (* ! *)
              (* Deref ty has language type: ty ref -> ty *)
        | Assign of typ (* := *)
              (* Assign ty has language type: ty ref -> ty -> unit *)
	| Assign' of typ (* := *)
              (* Assign ty has language type: ty ref -> ty -> unit *)
	| Equal of typ  (* = *)
              (* Equal ty has language type: ty -> ty -> bool *)
        | Less          (* < *)
              (* Less ty has language type: int -> int -> bool *)
        | LessEq        (* <= *)
        | Greater       (* > *)
        | GreaterEq     (* >= *)
        | Mod
        | Land
        | Lor
        | Lxor
        | Lsl
        | Lsr
        | Asr
        | UMinus     (* - *)
        | Plus       (* + *)
        | Minus      (* - *)
        | Times      (* * *)
        | Divide     (* / *)
        | ListConcat of typ (* @ *)
              (* ListConcat ty has language type: ty list -> ty list -> ty list *)
        | StrConcat  (* ^ *)
        | CompareName of typ    (* comparison between names *)
        | NameToString of typ
        | CreateThread of typ   (* concurrency operators *)
        | Self
        | Kill
        | CreateMutex
        | Lock
        | TryLock
        | Unlock
        | CreateCVar
        | Wait
        | Waiting            (* waiting on a condition variable *)
        | Signal
        | Broadcast
        | Thunkify
        | Unthunkify
        | Exit of typ

(* -=-- ---------- ------------------------------------ ---------- --=- *)


and signature_item_desc =
       SVal of ident * typ
     | STyp of typname * kind

and signature_item = signature_item_desc located

and signature = signature_item list located (* remember to freshen_signature *)

(* and prim_signature = signature_item_dsec list *)

and structure_item_desc =
       StrVal of ident * expr
     | StrValMulti of ident * pat list * expr
        (* this is a sugared-only form.  pat list is nonempty. *)
        (* we have that
                  let x p1 .. pn = e'
           is roughly represented by
                  StrValMulti (x, [p1;..; pn], e') *)
        (* remember to freshen_pats_expr *)
     | StrTyp of typname * typ


and structure_item = structure_item_desc located

and structure = structure_item list located(* remember to freshen_structure *)


(* and prim_structure = prim_structure_item_desc list *)

and vliteral =
     | VNumber of int   (* nn *)
     | VHash of hash    (* NN, h *)

and atomic_version =  (* avne *)
     | VLiteral of vliteral
     | VMyname (* compiler will write in name of current module *)

and version =  (* vne *)
     | VAtomic of atomic_version
     | VCons of atomic_version * version

and atomic_hash_version_constraint =  (* ahvce *)
      | AVCHModname of modname
      | AVCHHash of hash

and atomic_version_constraint =  (* avce *)
      | AVCHashVersion of atomic_hash_version_constraint
      | AVCNumber of int

and tail_version_constraint =
        TVCAtomic  of atomic_version_constraint   (* atomic  *)
      | TVCBetween of int * int  (* closed interval *)
      | TVCBefore  of int        (* left-open interval *)
      | TVCAfter   of int        (* right-open interval *)
      | TVCAnything              (* wildcard, matches any version *)

and version_constraint = (* vce *)
      | VCDotted of atomic_version_constraint list * tail_version_constraint
      | VCNameEqual of atomic_hash_version_constraint

and withspec = (modname * external_typname * typ) list
and likespec =
      | LikeNone             (* empty likespec *)
      | LikeMod of modname   (* like a module *)
      | LikeStr of structure (* like an inline struct *)

and resolvespec_item =
         Resolve_static_link
       | Resolve_here_already
       | Resolve_url of string

and resolvespec = resolvespec_item list

and mode =
       | MHash
       | MFresh
       | MCfresh
       | MBangHash
       | MBangCfresh

and mod_user_body = { mu_sign: signature;
                      mu_vne: version;
                      mu_str: structure;
                      mu_withspec: withspec; }

and mod_compile_body = { mc_hash: hash;
                         mc_eqs: eqs;
                         mc_vubs: valuabilities;
                         mc_sign0: signature;
                         mc_sign1: signature;
                         mc_vn: version;
                         mc_str: structure; }

and mod_imod_body = { mi_hash: hash;
		      mi_vubs : valuabilities;
                      mi_eqs: eqs;
                      mi_sign0: signature;
                      mi_sign1: signature;
                      mi_vn: version;
                      mi_str_done: structure_item list;  (* in reverse order *)
		      mi_str_todo: structure_item list;  (* in forward order *)
		      mi_str_loc : Location.t
		    }

and mod_compile_body_ctx = { ctx_mc_hash: hash;
                             ctx_mc_vubs : valuabilities;
			     ctx_mc_eqs: eqs;
                             ctx_mc_sign0: signature;
                             ctx_mc_sign1: signature;
                             ctx_mc_vn: version;
			     ctx_mc_str_done: structure_item list;
			     ctx_mc_str_todo: structure_item list;
                             ctx_mc_str_loc: Location.t }


and mod_hash_body = { mh_eqs: eqs;
                      mh_sign0: signature;
                      mh_vne: version;
                      mh_str: structure; }

and mod_alias_body = { ma_sign: signature;
                       ma_modname: modname; }

and import_user_body = { iu_sign: signature;
                         iu_vce: version_constraint;
                         iu_likespec: likespec;
                         iu_resolvespec: resolvespec;
                         iu_mo: modname option; }

and import_compile_body = { ic_hash: hash;
                            ic_vubs: valuabilities;
                            ic_sign0: signature;
                            ic_sign1: signature;
                            ic_vc: version_constraint;
                            ic_likestr: structure;
                            ic_resolvespec: resolvespec;
                            ic_mo: modname option; }

and import_hash_body = { ih_sign0: signature;
                         ih_vce: version_constraint;
                         ih_likestr: structure; }


and source_definition =
       | Mod_user of modname * mode * mod_user_body
       | Mod_alias of modname * mod_alias_body
       | Import_user of modname * mode * import_user_body
       | Mark_user of string

and definition =
       | Mod_compile of modname * mod_compile_body
       | Mod_imod of modname * mod_imod_body
       | Import_compile of modname * import_compile_body
       | Mod_fresh of modname * mod_user_body
       | Import_fresh of modname * import_user_body
       | Mark_compile of string

and source_definitions = source_definition list
(* remember to freshen_source_definitions;
   use freshen_program or freshen_configuration if
   the definitions bind elsewhere *)

and definitions = definition list
(* remember to freshen_source_definitions;
   use freshen_program or freshen_configuration if
   the definitions bind elsewhere *)

and 'state compilation_unit_definition =
       | CUDef of ('state -> source_definition * 'state)
       | CUSourceString of string  (* implementation extension; internal use only *)
       | CUIncludeSource of string * string option  (* second argument is file to write object code to *)
       | CUIncludeCompiled of string

and 'state compilation_unit_definitions =  'state compilation_unit_definition list

and 'state compilation_unit = 'state compilation_unit_definitions * ('state -> expr * 'state) option

and compiled_unit = nameenv * definitions * expr option

and prim_compiled_unit = nameenv * definitions * prim_expr option


(* hm, does this remain true? *)
(* pre-post module-init? *)
and program = definitions * prim_expr  (* remember to freshen_compiled_program *)


and store = (location * prim_expr) list

(* if (l, ty) is in loctyplist then l has language type ty
   (thus we don't strip the refs) *)
and loctyplist = (location * typ) list  (* no repetitions in the domain *)

and namesiglist = (modname * signature) list  (* no repetitions in the domain *)

(* -- Type environments -- *)

(* Type environments store information about the type/kind of various identifiers,
   namely: value identifiers->types, locations->types, modulenames (external+internal)->signatures
   typeconstructor-names -> kinds. *)
(* NB: stored from right to left, so shadowing identifiers occur
   before shadowed ones (if there was shadowing) *)
and typeenv_entry =
  Env_val of internal_ident * typ
| Env_loc of location * typ  (* l : T ref is represented as Env_loc(l,T) *)
| Env_mod of modname * signature
| Env_typ of internal_typname * kind

and typeenv_list = typeenv_entry list

and module_info = {  (* for efficiency, we store the following info on a module: *)
    mod_sign : signature;             (* original signature *)
    mod_eidmap : typ EIdentMap.t;     (* _flattened_ type of each value field *)
    mod_etnmap : kind ETypnameMap.t;  (* _flattened_ kind of each type field *)
  }

and typeenv = {
    te_list : typeenv_list;   (* stored from right to left, so shadowing identifiers occur before shadowed ones *)
    te_iidmap : typ IIdentMap.t;
    te_locmap : typ LocMap.t;
    te_modmap : module_info ModnameMap.t;
    te_itnmap : kind ITypnameMap.t;
  }

and nameenv_entry =
  | NEnv_nmod of abstract_name * external_modname * mod_hash_body    (* n : nmodule... *)
  | NEnv_nimp of abstract_name * external_modname * import_hash_body (* n : nimport... *)
  | NEnv_type of abstract_name                    (* n : Type *)
  | NEnv_tname of abstract_name * typ             (* n : T name *)

and nameenv_list = nameenv_entry list

(* NB: this carries only *fresh* NEnv_tname names; others carry enough info already that
   there is no need to add them here. *)
and nameenv = {
    ne_list : nameenv_list;   (* stored from right to left, so shadowing identifiers occur before shadowed ones *)
  }

(* Valuability *)

and valuability =
  | Valuable
  | CValuable
  | Nonvaluable

and valuabilities = { term_valuability : valuability;
                      type_valuability : valuability;
                    }


let name_of_nameenv_entry = function
  | NEnv_nmod(n,_,_)
  | NEnv_nimp(n,_,_)
  | NEnv_type(n)
  | NEnv_tname(n,_) -> n

let emptynameenv = { ne_list = []; }

let lookup_nameenv n env =
  let lookup_nameenv_aux ne =
    ( if n = name_of_nameenv_entry ne then Some ne else None )
  in
  findfirst lookup_nameenv_aux env.ne_list

let addnameenv ne env = { ne_list = ne::(env.ne_list); }

let name_mem_nameenv n env =
  is_Some (lookup_nameenv n env)

let nameenv_list_of_nameenv env = env.ne_list
let nameenv_nameenv_of_list nl = { ne_list = nl; }

let limitnameenv ne ns =
  let ln = nameenv_list_of_nameenv ne in
  let ln' =
  List.filter (fun e ->
    List.mem (name_of_nameenv_entry e) ns) ln
  in nameenv_nameenv_of_list ln'

let names_of_modnames ne modnames =
  let mn_ref = ref [] in
  List.iter
    (function
	NEnv_nmod(n, mn, _) | NEnv_nimp(n, mn, _)
	-> if List.mem mn modnames then
	  mn_ref := (n::(!mn_ref))
	else ()
      |	_ -> ()) ne.ne_list; !mn_ref

(* often En and E go around together holding hands *)
(* we need to pull this apart and put it together again too often to use a record here *)
type nametypeenv = nameenv * typeenv

(* name values *)

(* name_value manipulation *)
let abstract_name_of_name_value : name_value -> abstract_name
  = function
      VHashMvf (n,_,_,_) -> n
    | VHashTs  (n,_,_  ) -> n
    | VHashHts (n,_,_,_) -> n
    | VHashName(n,_    ) -> n

let type_of_name_value : name_value -> typ
  = function
      VHashMvf (_,_ ,_,ty) -> ty
    | VHashTs  (_,ty,_   ) -> ty
    | VHashHts (_,ty,_,_ ) -> ty
    | VHashName(_,ty     ) -> ty

let name_value_compare : name_value ->  name_value -> int
  = fun n1 n2 -> String.compare (abstract_name_of_name_value n1) (abstract_name_of_name_value n2)

let name_value_eq : name_value ->  name_value -> bool
  = fun n1 n2 -> name_value_compare n1 n2 = 0


(* name value map *)
module OrderedNameValue =
  struct
    type t = name_value
    let compare x y = String.compare (abstract_name_of_name_value x) (abstract_name_of_name_value y)
  end

module NameValueMap = Map.Make(OrderedNameValue)

let nameValueMap_assocs : 'a NameValueMap.t -> (NameValueMap.key * 'a) list
  = fun m
 -> let r = ref [] in
    NameValueMap.iter (fun x y -> r := (x,y)::!r) m;
    !r

(* Marshalled data *)
type marshalled_body = { mb_ne : nameenv option;
                         mb_defs : definitions;
                         mb_store : store;
                         mb_storeenv : loctyplist;
                         mb_expr : expr;
                         mb_typ : typ; }

(* the following will be the result of unserialising a message string;
   by contrast, we use Marshalled foo as a subexpression of the
   expression language *)
and marshalled_value = Marshalled_value of marshalled_body

(*  A configuration is a quintuple
        En ; <definitions, store environment, store, proc > .
    The store happens to be represented as an association list
    from (location-)expressions to (value-)expressions.
    Proc is implemented keeping threads and synchronisation gadgets separated.
*)
(* NB: the loctyplist and store should have identical domains *)

(* Processes *)

and thread_state = TsRunnable
                 | TsSlowcall
		 | TsResolveBlocked
                 | TsMutexBlocked of name_value
                 | TsCVarWaiting of name_value

and threaded_smallstep_inner = {
    tsc_name   : name_value;
    tsc_state  : thread_state;
    tsc_defs   : definitions;
    tsc_ec     : eval_ctx;
    tsc_env    : etysubst;
    tsc_eqs    : eqs;
    tsc_def    : (structure_item list) option;
    tsc_expr   : prim_expr;
    tsc_next_expr : prim_expr option;
  }

and thread_info = {
    ti_name : name_value;
    ti_state: thread_state;
    ti_defs : definitions;
    ti_expr : prim_expr;
  }

and mutex_info = {
    mtxi_name : name_value;
    mtxi_state : locked_mutex_info option;
  }

and locked_mutex_info = {
    lmtxi_owner   : name_value;
    lmtxi_waiting : name_value Util.AQueue.t
  }

and cvar_info = {
    cvi_name : name_value;
    cvi_waiting : name_value Util.AQueue.t;
  }


(* OLD: Acute 1.500 *)
(* type outer_configuration = (definitions * loctyplist * store)
and configuration = outer_configuration * prim_expr remember to freshen_configuration *)

type configuration = { cfg_nenv : nameenv;
                       cfg_defs : definitions;
                       cfg_store : store;
                       cfg_senv : loctyplist;
                       cfg_runnable : name_value AQueue.t;
		       cfg_slowcall : unit NameValueMap.t;
                       cfg_threads : thread_info NameValueMap.t;
                       cfg_mutexes : mutex_info NameValueMap.t;
                       cfg_cvars : cvar_info NameValueMap.t; }

and smallstep_outer = { scfg_nenv : nameenv;
                        scfg_defs : definitions;
			scfg_store : store;
			scfg_senv : loctyplist;
			scfg_runnable : name_value AQueue.t;
			scfg_slowcall :  unit NameValueMap.t;
                        scfg_threads : threaded_smallstep_inner NameValueMap.t;
			scfg_mutexes : mutex_info NameValueMap.t;
			scfg_cvars : cvar_info NameValueMap.t; }


(* a smallstep configuration has:
   - definitions, store environment, store
   - evaluation context
   - current colour (at the hole)
   - expression under evaluation
*)

(*
and smallstep_inner_info =
    { sc_ec : eval_ctx;
      sc_env : etysubst;
      sc_eqs : eqs;
    }

(* only using this because it was in existence for refocus functions
   so easier than rewriting them *)

and smallstep_inner = {  ssi_ec  : eval_ctx;
                         ssi_env : etysubst;
                         ssi_eqs : eqs;
                         ssi_e   : prim_expr; }
*)



(* We use this in the smallstep evaluator *)
and collected =
    {
    cdefs : definitions;
    cstore : store;
    csenv : loctyplist;
    ceqs : eqs;
    cenv : etysubst;
    cctx : eval_ctx;
    cexpr : prim_expr
  }

(* OLD: Acute 1.500 *)
(* and small_step_configuration = nameenv * outer_configuration * small_step_inner *)
(* old comment (there isn't one): remember to freshen_configuration *)

(*and smallstep_configuration = nameenv * smallstep_outer * smallstep_inner
and smallstep_p_configuration = nameenv * smallstep_outer * threaded_smallstep_inner
*)

(* M has killed reduce_fast *)
(*
and runtime_env = etysubst
and eval_context = (runtime_env * eqs * (prim_expr->prim_expr)) list (* used in reduce_fast *)
*)

let append_iidmap : 'a IIdentMap.t -> 'a IIdentMap.t -> 'a IIdentMap.t
    (* m1 has priority *)
  = fun m1 m2 ->
    IIdentMap.fold (fun i e m -> IIdentMap.add i e m) m1 m2
let iidmap_to_list : 'a IIdentMap.t -> (internal_ident * 'a) list
  = fun m ->
    IIdentMap.fold (fun i e l -> (i,e)::l) m []

(* why oh why doesn't Ocaml support polymorphism in this way? *)
let append_itnmap : 'a ITypnameMap.t -> 'a ITypnameMap.t -> 'a ITypnameMap.t
    (* m1 has priority *)
  = fun m1 m2 ->
    ITypnameMap.fold (fun i e m -> ITypnameMap.add i e m) m1 m2
let itnmap_to_list : 'a ITypnameMap.t -> (internal_typname * 'a) list
  = fun m ->
    ITypnameMap.fold (fun i e l -> (i,e)::l) m []

let empty_esubst   :   esubst = IIdentMap.empty
let empty_tysubst  :  tysubst = ITypnameMap.empty
let empty_etysubst : etysubst = (empty_esubst,empty_tysubst)

let initial_store : store = []
let initial_store_env : loctyplist = []


(* -=-- ---------- ------------------------------------ ---------- --=- *)

(* basic name constructors  *)
let fresh_internal_typname s  = ((fresh,s) : internal_typname)
let fresh_internal_ident   s  = ((fresh,s) : internal_ident)
let fresh_internal_modname () = ((fresh,"M") : internal_modname)
let fresh_modname ext = (ext, fresh_internal_modname ())

let isinfix_op : op -> bool
  = fun op
 -> match op with
    | Assign _ | Assign' _
    | Equal _ | Less | LessEq | Greater | GreaterEq
    | Mod | Land | Lor | Lxor | Lsl | Lsr | Asr
    | Plus | Minus | Times | Divide
    | ListConcat _ | StrConcat
       -> true
    | UMinus | Ref _ | Deref _
    | CompareName _ | NameToString _
    | CreateThread _ | Self | Kill
    | CreateMutex | Lock | TryLock | Unlock
    | CreateCVar | Wait | Waiting | Signal
    | Broadcast | Thunkify | Unthunkify | Exit _
       -> false

let isinfix_op_or_econst oe =
  match oe with
  | OEOp op -> isinfix_op op
  | OEEconst iid -> false

(* information on primitive ops *)
let op_info : op -> (typ list * typ)
  = fun op
 -> match op with
    | Ref(ty)          -> ([ty]                                                                ,TTyCon1(TRef,ty))
    | Deref(ty)        -> ([TTyCon1(TRef,ty)]                                                  ,ty)
    | Assign(ty)       -> ([TTyCon1(TRef,ty);ty]                                               ,TTyCon0 TUnit)
    | Assign'(ty)      -> ([TTyCon1(TRef,ty);ty]                                               ,TTyCon0 TUnit)
    | Equal(ty)        -> ([ty;ty]                                                             ,TTyCon0 TBool)  (* TODO: should be restricted *)
    | Less             -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TBool)
    | LessEq           -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TBool)
    | Greater          -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TBool)
    | GreaterEq        -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TBool)
    | Mod              -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Land             -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)  (* these are infix in Ocaml *)
    | Lor              -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Lxor             -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Lsl              -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Lsr              -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Asr              -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | UMinus           -> ([TTyCon0 TInt]                                                      ,TTyCon0 TInt)
    | Plus             -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Minus            -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Times            -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | Divide           -> ([TTyCon0 TInt;TTyCon0 TInt]                                         ,TTyCon0 TInt)
    | ListConcat(ty)   -> ([TTyCon1(TList,ty);TTyCon1(TList,ty)]                               ,TTyCon1(TList,ty))
    | StrConcat        -> ([TTyCon0 TString;TTyCon0 TString]                                   ,TTyCon0 TString)
    | CompareName(ty)  -> ([TTyCon1(TName,ty);TTyCon1(TName,ty)]                               ,TTyCon0 TInt)
    | NameToString(ty) -> ([TTyCon1(TName,ty)]                                                 ,TTyCon0 TString)
    | CreateThread(ty) -> ([TTyCon1(TName,TTyCon0 TThread);TFunc(ty,TTyCon0 TUnit);ty]         ,TTyCon0 TUnit)
    | Self             -> ([TTyCon0 TUnit]                                                     ,TTyCon1(TName,TTyCon0 TThread))
    | Kill             -> ([TTyCon1(TName,TTyCon0 TThread)]                                    ,TTyCon0 TUnit)
    | CreateMutex      -> ([TTyCon1(TName,TTyCon0 TMutex)]                                     ,TTyCon0 TUnit)
    | Lock             -> ([TTyCon1(TName,TTyCon0 TMutex)]                                     ,TTyCon0 TUnit)
    | TryLock          -> ([TTyCon1(TName,TTyCon0 TMutex)]                                     ,TTyCon0 TBool)
    | Unlock           -> ([TTyCon1(TName,TTyCon0 TMutex)]                                     ,TTyCon0 TUnit)
    | CreateCVar       -> ([TTyCon1(TName,TTyCon0 TCVar)]                                      ,TTyCon0 TUnit)
    | Wait             -> ([TTyCon1(TName,TTyCon0 TCVar);TTyCon1(TName,TTyCon0 TMutex)]        ,TTyCon0 TUnit)
    | Waiting          -> ([TTyCon1(TName,TTyCon0 TCVar);TTyCon1(TName,TTyCon0 TMutex)]        ,TTyCon0 TUnit)
    | Signal           -> ([TTyCon1(TName,TTyCon0 TCVar)]                                      ,TTyCon0 TUnit)
    | Broadcast        -> ([TTyCon1(TName,TTyCon0 TCVar)]                                      ,TTyCon0 TUnit)
    | Thunkify         -> ([TTyCon1(TList,TTyCon0 TThunkkey)]                                  ,TFunc(TTyCon1(TList,TTyCon0 TThunkkey),TTyCon0 TUnit))
    | Unthunkify       -> ([TTyCon1(TList,TTyCon0 TThunklet)]                                  ,TTyCon0 TUnit)
    | Exit(ty)         -> ([TTyCon0 TInt]                                                      ,TTyCon0 TUnit)

(* arities of primitive ops *)
let arity_of_op = function (* keeping the same order as the datatype *)
  op -> List.length (fst (op_info op))

(* filled in by Econst once defined *)
let arity_of_econst_ref = ref (fun iid -> raise (Never_happen "arity_of_econst_ref: not filled in!"))

let arity_of_op_or_econst oe =
  match oe with
  | OEOp op -> List.length (fst (op_info op))
  | OEEconst iid -> !arity_of_econst_ref iid

(* freshening pattern-expression pairs *)

let binding_vars_of_pat pat =
  let rec binding_vars_of_pat' vars = function
      PVar (i,_) -> i :: vars
    | PC1 (_,p) -> binding_vars_of_pat' vars p
    | PCons(p1,p2) -> binding_vars_of_pat' (binding_vars_of_pat' vars p1) p2
    | PTup [] -> vars
    | PTup (p::pl) -> binding_vars_of_pat' (binding_vars_of_pat' vars p) (PTup pl)
    | PTyped (p,_) -> binding_vars_of_pat' vars p
    | PWild _ -> vars
    | PC0 _ -> vars
  in binding_vars_of_pat' [] pat

let binding_vars_of_pats pats =
  List.concat (List.map binding_vars_of_pat pats)

let swap_list : ('a name * 'a name) list -> 'b -> 'b
  = fun swaps e ->
(*    let (nl1,nl2) = List.split swaps
    in Freshness.swap_multiple_atoms nl1 nl2 e *)

  let f e (a1,a2) = (swap a1 and a2 in e) in
    List.fold_left f e swaps
;;

(* uncomment the following bit of ocaml and just run ./compile to see
   the swap bug.  We've (or at leas J has) been assuming that latter
   swaps override earlier ones, but it isn't true. imagine that we're
   trying to freshen a binding of the form

       bind x in bind x in x

   then it's the *second* x that binds.  usually, we extract the the
   binders and associate them to fresh names [(x,x1); (x,x2)] and then
   apply swap_list. but this is broken (we get the wrong one).

   J suggests we rewrite swap_list to use fold_right and document it
   as follows:

     ``The argument to swap list consists of an association list. The
     range (i.e. 2nd part of each pair) is required to consist of
     distinct fresh names.  If the domain contains duplicates, the
     last maplet in the list takes priority.''

*)

(*
let x = fresh in let x1 = fresh in let x2 = fresh in
let print z = List.assoc z [(x, "x"); (x1, "x1"); (x2, "x2")] in
Printf.printf "%s\n" (print (swap_list [(x,x1); (x,x2)] x));
exit 0;;
*)

(*
let freshen_iident_foo = id
let freshen_pat_expr = id
let freshen_pat_prim_expr = id
let freshen_pats_foo = id
let freshen_source_definitions = id
let freshen_definitions = id
let freshen_program = id
let freshen_structure = id
let freshen_signature = id
let freshen_mb mb l = mb
*)

let freshen_iident_foo : internal_ident * 'a -> internal_ident * 'a
  = fun (iid,e)
 -> let iid' = fresh_prettyname iid in
    (iid', swap (name_of_prettyname iid) and (name_of_prettyname iid') in e)

let freshen_pat_expr (pat, (expr:expr)) =
  let vars = binding_vars_of_pat pat in
  let swaps = List.map (fun (x,_) -> (x,fresh)) vars in
  let pat' = swap_list swaps pat in
  let expr' = swap_list swaps expr.desc in
  (pat', {expr with desc = expr'})

let freshen_pat_prim_expr (pat, (expr:prim_expr)) =
  let vars = binding_vars_of_pat pat in
  let swaps = List.map (fun (x,_) -> (x,fresh)) vars in
  let pat' = swap_list swaps pat in
  let expr' = swap_list swaps expr in
  (pat', expr')

let freshen_pats_foo (pats, foo) =
  let vars = List.concat (List.map binding_vars_of_pat pats) in
  let swaps = List.map (fun (x,_) -> (x,fresh)) vars in
  let pats' = List.map (swap_list swaps) pats in
  let foo' = swap_list swaps foo in
  (pats', foo')

let rec freshen_source_definitions' = function
  | [] -> ([], [])
  | Mod_user ((ext, (inter,s)), mode, body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_source_definitions' (swap inter and inter' in ds) in
      (Mod_user ((ext, (inter',s)), mode, body) :: ds', (inter,inter')::swaps')
  | Mod_alias ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_source_definitions' (swap inter and inter' in ds) in
      (Mod_alias ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')
  | Import_user ((ext, (inter,s)), mode, body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_source_definitions' (swap inter and inter' in ds) in
      (Import_user ((ext, (inter',s)), mode, body) :: ds', (inter,inter')::swaps')
  | Mark_user mk :: ds ->
      let (ds', swaps') = freshen_source_definitions' ds in
      (Mark_user mk :: ds',  swaps')

let rec freshen_definitions' = function
  | [] -> ([], [])
  | Mod_compile ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_definitions' (swap inter and inter' in ds) in
      (Mod_compile ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')
 | Mod_imod ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_definitions' (swap inter and inter' in ds) in
      (Mod_imod ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')


  | Import_compile ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_definitions' (swap inter and inter' in ds) in
      (Import_compile ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')
  | Mod_fresh ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_definitions' (swap inter and inter' in ds) in
      (Mod_fresh ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')
  | Import_fresh ((ext, (inter,s)), body) :: ds ->
      let inter' = fresh in
      let (ds', swaps') = freshen_definitions' (swap inter and inter' in ds) in
      (Import_fresh ((ext, (inter',s)), body) :: ds', (inter,inter')::swaps')
  | Mark_compile mk :: ds ->
      let (ds', swaps') = freshen_definitions' ds in
      (Mark_compile mk :: ds',  swaps')

let freshen_source_definitions ds =
  let (ds', swaps') = freshen_source_definitions' ds in ds'

let freshen_definitions ds =
  let (ds', swaps') = freshen_definitions' ds in ds'

let freshen_program (ds, e) =
  let (ds',swaps') = freshen_definitions' ds in
  let e' = swap_list swaps' e in
  (ds', e');;

(* NB: we don't touch the locations *)
(* let freshen_configuration : configuration -> configuration *)
(* = fun c -> *)
(*   let (ds',swaps') = freshen_definitions' c.cfg_defs in *)
(*   { cfg_defs     = ds'; *)
(*     cfg_store    = swap_list swaps' c.cfg_store; *)
(*     cfg_senv     = swap_list swaps' c.cfg_senv; *)
(*     cfg_runnable = swap_list swaps' c.cfg_runnable; *)
(*     cfg_mutexes  = swap_list swaps' c.cfg_mutexes; *)
(*     cfg_cvars    = swap_list swaps' c.cfg_cvars; *)
(*   } *)

let rec freshen_structure' = function
  | [] -> []
  | str::ss -> (match str.desc with
     StrVal ((ext, (inter,s)), e)  ->
      let inter' = fresh in
      let ss' = freshen_structure' (swap inter and inter' in ss) in
      {str with desc = StrVal ((ext, (inter',s)), e)} :: ss'
  | StrValMulti ((ext, (inter,s)), ps, e) ->
      let (ps,e) = freshen_pats_foo (ps,e) in
      let inter' = fresh in
      let ss' = freshen_structure' (swap inter and inter' in ss) in
{str with desc = StrValMulti ((ext, (inter',s)), ps, e)} :: ss'
  | StrTyp ((ext, (inter,s)), t) ->
      let inter' = fresh in
      let ss' = freshen_structure' (swap inter and inter' in ss) in
       {str with desc = StrTyp ((ext, (inter',s)), t)} :: ss');;

let freshen_structure str = {str with desc=freshen_structure' str.desc}

let rec freshen_signature' = function
  | [] -> []
  | s::ss -> (match s.desc with
      SVal ((ext, (inter,s_int)), t) ->
	let inter' = fresh in
	let ss' = freshen_signature' (swap inter and inter' in ss) in
({s with desc = SVal ((ext, (inter',s_int)), t)}) :: ss'
  | STyp ((ext, (inter,s_int)), k) ->
      let inter' = fresh in
      let ss' = freshen_signature' (swap inter and inter' in ss) in
      {s with desc=STyp ((ext, (inter',s_int)), k)} :: ss');;

let freshen_signature sign = {sign with desc=freshen_signature' sign.desc}

(* bitrot
 * let freshen_clos (sub,((x2i,x2s) as x2),ty2,bs2,e1,x1o) =
 *   let x2i' = fresh in
 *   let (sub',swaps) =
 *     IIdentMap.fold (fun (i,s) e (m,l) -> let i' = fresh in (IIdentMap.add (i',s) e m, (i,i')::l))
 *                    sub (IIdentMap.empty,[(x2i,x2i')]) in
 *   let (e1',x1o') = swap_list swaps (e1,x1o) in
 *   (sub',(x2i',x2s),ty2,bs2,e1',x1o');;
 *
 *   (* easy since the sub is guaranteed to have all domain internal
 *   names distinct and none of the domain elements bind in the range
 *   expressions. They may be coincidentally present in the range but
 *   they don't *bind*. *)
 *)


(* Constructor/inspector functions *)

let external_typname_of_string s = s
let external_typname_to_string s = s
let typname_ext (extern,intern) = extern
let typname_int (extern,intern) = intern

let etn_compare : external_typname -> external_typname -> int
  = fun etn1 etn2
 -> Pervasives.compare etn1 etn2

(* compare internal typnames for equality  *)
let tn_eq : internal_typname -> internal_typname -> bool
  = fun (tn1,_) (tn2,_)
 -> tn1 = tn2
let tn_compare : internal_typname -> internal_typname -> int
  = fun (tn1,_) (tn2,_)
 -> Pervasives.compare tn1 tn2

let external_ident_of_string s = s
let external_ident_to_string s = s
let ident_ext (extern,intern) = extern
let ident_int (extern,intern) = intern


(* compare internal idents for equality *)
let id_eq : internal_ident -> internal_ident -> bool
  = fun (id1,_) (id2,_)
 -> id1 = id2
let eid_eq : external_ident -> external_ident -> bool
  = fun eid1 eid2
 -> eid1 = eid2
let id_compare : internal_ident -> internal_ident -> int
  = fun (id1,_) (id2,_)
 -> Pervasives.compare id1 id2

let external_modname_of_string s = s
let external_modname_to_string s = s
let modname_ext (extern,intern) = extern
let modname_int (extern,intern) = intern

(* literal hash helpers stuff *)
let string_of_lithash : Digest.t -> string
  = fun h
 -> let hex c = Printf.sprintf "%02X" (int_of_char c) in
    String.concat "" ("0#"::stringmap hex h)

let lithash_of_string : string -> Digest.t
(* WARNING: doesn't do any error checking *)
  = fun s
 -> let unhex c = let n = int_of_char c - 48 in if n < 10 then n else ((n-7) mod 16) in
    let hdigits = stringmap unhex (String.sub s 2 32) in
    let rec tobytes ds =
      match ds with
        []         -> []
      | [_]        -> raise (Never_happen "lithash_of_string")
      | (n::m::tl) -> (n * 16 + m) :: tobytes tl in
    List.fold_left (fun s i -> s ^ String.make 1 (char_of_int (i mod 256))) ""
      (tobytes hdigits)

(* build a TDot or a THashDot: X.t *)
let txdot : hash_or_modname -> external_typname -> typ
  = fun x etn
 -> TXDot(x,etn)

(* build a Dot or a HashDot: X.x *)
let xdot : hash_or_modname -> external_ident -> prim_expr
  = fun x eid
 -> match x with
      Hash    h  -> HashDot(h,  eid)
    | Modname mn -> Dot    (mn, eid)

(*
let definition_is_compiled : definition -> bool
  = function
  | Mod_compile    _ -> true
  | Import_compile _ -> true
  | Mark           _ -> true
  | _                -> false

let definition_not_compiled : definition -> bool
  = function
  | Mod_compile    _ -> false
  | Import_compile _ -> false
  | _                -> true

*)

let modname_of_source_definition = function
  | Mod_user (m,_,_)
  | Mod_alias (m,_)
  | Import_user (m,_,_) -> m
  | Mark_user _ -> assert false

let modname_of_definition = function
  | Mod_compile (m, _)
  | Mod_imod (m, _)
  | Import_compile (m, _)
  | Mod_fresh (m, _)
  | Import_fresh (m, _) -> m
  | Mark_compile _ -> assert false

let sign0_of_definition = function
    Mod_compile (_, body) -> body.mc_sign0
  | Mod_imod (_, body) -> body.mi_sign0
  | Import_compile (_, body) -> body.ic_sign0
  | Mod_fresh _ -> raise (Never_happen "sign0_of_definition of Mod_fresh")
  | Import_fresh _ -> raise (Never_happen "sign0_of_definition of Import_fresh")
  | Mark_compile _ -> raise (Never_happen "sign0_of_definition of Mark")

let sign1_of_definition = function
    Mod_compile (_, body) -> body.mc_sign1
  | Mod_imod (_, body) -> body.mi_sign1
  | Import_compile (_, body) -> body.ic_sign1
  | Mod_fresh ((ext, _), body) -> body.mu_sign  (* FZ sign is sign1 for fresh modules *)
  | Import_fresh (_,body) -> body.iu_sign
  | Mark_compile _ -> raise (Never_happen "sign1_of_definition of Mark")

let sign_of_definition = function
    Mod_compile _ -> raise (Never_happen "sign_of_definition of Mod_compile")
  | Mod_imod _ -> raise (Never_happen "sign_of_definition of Mod_imod")
  | Import_compile _ -> raise (Never_happen "sign_of_definition of Mod_compile")
  | Mod_fresh (_, body) -> body.mu_sign
  | Import_fresh (_,body) -> body.iu_sign
  | Mark_compile _ -> raise (Never_happen "sign_of_definition of Mark")

let print_modname_unsafe (eid,iid) = eid ^ "[" ^ string_of_int (Obj.magic iid : int) ^ "]"

let external_modname_of_hash = function
  | HashM(_,emn,_) -> Some emn
  | HashI(_,emn,_) -> Some emn
  | HashLit _      -> None
  | HashName _     -> None

(* compare modnames for equality (examine internal and external names) *)
let mn_eq : modname -> modname -> bool
  = fun (emn1,(imn1,_)) (emn2,(imn2,_))
 -> emn1 = emn2 && imn1 = imn2
let mn_ext_eq : external_modname -> external_modname -> bool
  = fun emn1 emn2
 -> emn1 = emn2
let mn_compare : modname -> modname -> int
  = fun (emn1,(imn1,_)) (emn2,(imn2,_))
 -> Pervasives.compare (emn1,imn1) (emn2,imn2)


(* compare locations for equality *)
let l_eq : location -> location -> bool
  = fun l1 l2
 -> l1 = l2

let sign_typfieldnames : signature -> typname list
  = fun sign
 -> partial_map (function si ->
                 match si.desc with
                   SVal(_,_)  -> None
                 | STyp(tn,_) -> Some(tn)) sign.desc

let sign_valfieldnames : signature -> ident list
  = fun sign
 -> partial_map (function si ->
                 match si.desc with
                   SVal(id,_) -> Some(id)
                 | STyp(_,_)  -> None) sign.desc

(* lookup kind of etn in signature *)
(* WARNING: unless the signature is compiled, the result may contain references to sig typnames *)
let lookup_signature_etn : signature -> external_typname -> kind option
  = fun sign etn
 -> findfirst (function si ->
                 match si.desc with
                   SVal(_,_)  -> None
                 | STyp(tn,k) -> if etn = typname_ext tn then Some(k) else None) sign.desc

(* lookup type of eid in signature *)
(* WARNING: unless the signature is compiled, the result may contain references to sig typnames *)
let lookup_signature_eid : signature -> external_ident -> typ option
  = fun sign eid
 -> findfirst (function si ->
                 match si.desc with
                   SVal(id,ty) -> if eid = ident_ext id then Some(ty) else None
                 | STyp(_,_)   -> None) sign.desc


let rec limitdom : signature -> signature
  = fun sign
 -> {sign with desc =partial_map (function si ->
                   match si.desc with
                     SVal(_,_)  -> None
                   | STyp(tn,KType) -> Some(si)
                   | STyp(tn,KEq t) -> None) sign.desc}


let str_typfieldnames : structure -> typname list
  = fun sign
 -> partial_map (function si ->
                 match si.desc with
                   StrVal(_,_)  -> None
                 | StrValMulti(_,_,_) -> None
                 | StrTyp(tn,_) -> Some(tn)) sign.desc

let str_valfieldnames : structure -> ident list
  = fun sign
 -> partial_map (function si ->
                 match si.desc with
                   StrVal(id,_) -> Some(id)
                 | StrValMulti(id,_,_) -> Some(id)
                 | StrTyp(_,_)  -> None) sign.desc

let lookup_structure_etn : structure -> external_typname -> typ option
  = fun str etn
 -> findfirst (function sti ->
                 match sti.desc with
                   StrVal(_,_)   -> None
                 | StrValMulti(_,_,_) -> None
                 | StrTyp(tn,ty) -> if etn = typname_ext tn then Some(ty) else None) str.desc

let rec lookup_structure_eid : structure -> external_ident -> expr option
    = fun str eid
      -> findfirst (function sti->
        match sti.desc with
          StrVal((eid',iid'),v')  when (eid_eq eid eid') -> Some(v')
        | StrValMulti((eid',iid'),ps',e') when (eid_eq eid eid') -> raise (Unimplemented "lookup_structure_eid of sugared structure value")
        | _ -> None) str.desc


let lookup_store_loc : store -> location -> prim_expr option
  = fun s l0
 -> findfirst (function (l,e) -> if l_eq l l0 then Some(e) else None) s


(* basic type and name environment accessor; remainder defined elsewhere *)

let typeenv_list_of_typeenv : typeenv -> typeenv_list
  = fun env -> env.te_list

(* printers (filled in later) *)


let print_hack e =
  match e with
            C0        _ -> "C0"
        | C1        _ -> "C1"
        | Cons      _ -> "Cons"
        | Tup       _ -> "Tup"
        | Op        _ -> "Op"
        | Loc       _ -> "Loc"
        | Fn        _ -> "Fn"
        | Fun       _ -> "Fun"
        | TAbs      _ -> "TAbs"
        | InEnv     _ -> "InEnv"
        | Clos      _ -> "Clos"
        | TClos     _ -> "TClos"
        | Id        _ -> "Id"
        | Dot       _ -> "Dot"
        | HashDot   _ -> "HashDot"
        | If        _ -> "If"
        | While     _ -> "While"
        | LazyOp    _ -> "LazyOp"
        | Seq       _ -> "Seq"
        | App       _ -> "App"
        | TApp      _ -> "TApp"
        | Pack      _ -> "Pack"
        | Unpack    _ -> "Unpack"
        | Namecase  _ -> "Namecase"
        | Match     _ -> "Match"
        | Let       _ -> "Let"
        | LetMulti  _ -> "LetMulti"
        | Letrec    _ -> "Letrec"
	| LetrecMulti   _ -> "LetrecMulti"
        | Raise     _ -> "Raise"
        | Try       _ -> "Try"
        | Marshal   _ -> "Marshal"
	| Marshalz   _ -> "Marshalz"
	| Unmarshal   _ -> "Unmarshal"
        | RET       _ -> "RET"
        | SLOWRET   _ -> "SLOWRET"
        | Col       _ -> "Col"
        | OP        _ -> "OP"
        | Resolve   _ -> "Resolve"
	| Resolve_blocked   _ -> "Resolve_blocked"
        | Par       _ -> "Par"
        | Fresh     _ -> "Fresh"
        | CFresh    _ -> "CFresh"
        | HashMvf   _ -> "HashMvf"
        | HashTs    _ -> "HashTs"
        | HashHts   _ -> "HashHts"
	| NameValue   _ -> "NameValue"
        | Swap      _ -> "Swap"
        | Freshfor  _ -> "Freshfor"
        | Support   _ -> "Support"
        | Tie       _ -> "Tie"
      | NameOfTie   _ -> "NameOfTie"
        | ValOfTie _ -> "ValOfTie"

let rec print_hack_all : prim_expr -> string = fun e ->
  match e with
          C0        _ -> "C0"
        | C1        (c1, e') -> "C1(" ^ print_hack_all e' ^ ")"
        | Cons      (e1, e2) -> "Cons("
	    ^ print_hack_all e1
	    ^ ","
	    ^ print_hack_all e2
	    ^ ")"
        | Tup       es -> "Tup(" ^  List.fold_right (fun a -> fun b -> a ^ "," ^ b) (List.map (fun e -> print_hack_all e) es) "" ^ ")"
        | Op        (oe, es) -> "Op(" ^ List.fold_right (fun a -> fun b -> a ^ "," ^ b) (List.map (fun e -> print_hack_all e) es) "" ^ ")"
        | Loc       l -> "Loc (" ^ string_of_int l ^ ")"
        | Fn        m -> "Fn"
        | Fun       (ps, e') -> "Fun(" ^ print_hack_all e' ^ ")"
        | TAbs      (itn, e') -> "TAbs(" ^ string_of_prettyname itn ^ "," ^ print_hack_all e' ^ ")"
        | InEnv     (env, e') -> "InEnv(" ^ print_hack_all e' ^ ")"
        | Clos      (env, x2, ty2, bs2, e1, x1o) -> "Clos(" ^ string_of_prettyname x2 ^ "," ^ print_hack_all e1 ^ ")"
        | TClos     (env, itn, e') -> "TClos(" ^ string_of_prettyname itn ^ "," ^ print_hack_all e' ^ ")"
        | Id        i -> "Id(" ^ string_of_prettyname i ^ ")"
        | Dot       (mn, eid) -> "Dot"
        | HashDot   (h, eid) -> "HashDot"
        | If        (e1, e2, e3) -> "If(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ "," ^ print_hack_all e3 ^ ")"
        | While     (e1, e2) -> "While(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
        | LazyOp    (lo, e1) -> "LazyOp"
        | Seq       (e1, e2) -> "Seq(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
        | App       (e1, e2) -> "App(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
        | TApp      (e', t) -> "TApp(" ^ print_hack_all e' ^ ")"
        | Pack      (t1, e', t2) -> "Pack(" ^ print_hack_all e' ^ ")"
        | Unpack    (itn, iid, e1, e2) -> "Unpack"
        | Namecase  (e1, r, x1, x2, e', e2, e3) -> "Namecase"
        | Match     (e', m) -> "Match(" ^ print_hack_all e' ^ ")"
        | Let       (e1, (p, e2)) -> "Let"
        | LetMulti  (ty, (ps, e1), (iid, e)) -> "LetMulti"
        | Letrec    (ty, (iid, (m, e))) -> "Letrec"
    | LetrecMulti   (ty, (ps, (iid, (e1, e2)))) -> "LetrecMulti"
        | Raise     e'  -> "Raise(" ^ print_hack_all e' ^ ")"
        | Try       (e', m)  -> "Try(" ^ print_hack_all e' ^ ")"
        | Marshal   (e1, e2, ty) -> "Marshal(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
       | Marshalz   (mk, e', ty) -> "Raise(" ^ print_hack_all e' ^ ")"
      | Unmarshal   (e', ty)  -> "Unmarshal(" ^ print_hack_all e' ^  ")"
        | RET       _ -> "RET"
        | SLOWRET   _ -> "SLOWRET"
        | Col       (e', eqs, t) -> "Col(" ^ print_hack_all e' ^  ")"
        | OP        (i, oe, es) -> "OP"
        | Resolve   (e', mn, rs) -> "Resolve"
| Resolve_blocked   (e', mn, rs) -> "Resolve_blocked"
        | Par       (e1, e2) -> "Par(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
        | Fresh     _ -> "Fresh"
        | CFresh    _ -> "CFresh"
        | HashMvf   _ -> "HashMvf"
        | HashTs    _ -> "HashTs"
        | HashHts   _ -> "HashHts"
      | NameValue   _ -> "NameValue"
        | Swap      (e1, e2, e3) -> "If(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ "," ^ print_hack_all e3 ^ ")"
        | Freshfor  (e1, e2) -> "Freshfor(" ^ print_hack_all e1 ^ "," ^ print_hack_all e2 ^ ")"
        | Support   (ty, e') -> "Support(" ^ print_hack_all e' ^ ")"
        | Tie       _ -> "Tie"
      | NameOfTie   e'  -> "NameOfTie(" ^ print_hack_all e' ^ ")"
        | ValOfTie  e' -> "ValOfTie(" ^ print_hack_all e' ^ ")"



let basic_print_expr : (expr -> string) ref
  = ref (fun _ -> raise (Never_happen "basic_print_expr: not filled in"))
let basic_print_prim_expr : (prim_expr -> string) ref
  = ref (fun _ -> raise (Never_happen "basic_print_prim_expr: not filled in"))


let basic_print_esubst : (esubst -> string) ref
  = ref (fun _ -> raise (Never_happen "esubst: not filled in"))


let rec exprtoprim e =
  match e.desc with
      LocC0(c) -> C0(c)
    | LocC1(c1,e)        -> C1(c1,exprtoprim e)
    | LocCons(e1,e2)     -> Cons(exprtoprim e1, exprtoprim e2)
    | LocTup(es)         -> Tup(List.map (exprtoprim) es)
    | LocOp(oe,el)       -> Op(oe,(List.map (exprtoprim) el))
    | LocLoc(l)          -> Loc(l)
    | LocFn(m)           -> Fn(mtchtoprim e.loc m)
    | LocFun(ps,e)       -> Fun(ps,exprtoprim e)
    | LocInEnv(env,e)    -> InEnv(env, exprtoprim e)
    | LocClos(env,x2,ty2,bs2,e1,x1o) -> Clos(env, x2, ty2, bs2, exprtoprim e1, x1o)
    | LocTClos(env,itn,e)-> TClos(env,itn,exprtoprim e)
    | LocId(iid)         -> Id(iid)
    | LocDot(mn,eid)     -> Dot(mn,eid)
    | LocHashDot(h,eid)  -> HashDot(h,eid)
    | LocIf(e1,e2,e3)    -> If (exprtoprim e1, exprtoprim e2, exprtoprim e3)
    | LocWhile(e1,e2)    -> While (exprtoprim e1, exprtoprim e2)
    | LocLazyOp(lo,el)   -> LazyOp(lo,(List.map (exprtoprim) el))
    | LocSeq(e1,e2)      -> Seq (exprtoprim e1, exprtoprim e2)
    | LocApp(e1,e2)      -> App (exprtoprim e1, exprtoprim e2)
    | LocTAbs(itn,e)     -> TAbs (itn, exprtoprim e)
    | LocTApp(e,t)       -> TApp (exprtoprim e, t)
    | LocPack(t1,e,t2) -> Pack (t1, exprtoprim e, t2)
    | LocUnpack(itn,iid,e1,e2) -> Unpack(itn, iid, exprtoprim e1, exprtoprim e2)
    | LocNamecase(e1,r,x1,x2,e,e2,e3) -> Namecase (exprtoprim e1,r,x1,x2,exprtoprim e,exprtoprim e2,exprtoprim e3)
    | LocMatch(e,m)      -> Match (exprtoprim e, mtchtoprim e.loc m)
    | LocLet(e1,(p,e2))  -> Let (exprtoprim e1, (p, exprtoprim e2))
    | LocLetMulti(ty,(ps,e1),(iid,e))       -> LetMulti (ty,(ps,exprtoprim e1),(iid,(exprtoprim e)))
    | LocLetrec(ty,(iid,(m,e)))             -> Letrec (ty,(iid,(mtchtoprim e.loc m, exprtoprim e)))
    | LocLetrecMulti(ty,(ps,(iid,(e1,e2)))) -> LetrecMulti (ty,(ps,(iid,(exprtoprim e1, exprtoprim e2))))
    | LocRaise(e)        -> Raise (exprtoprim e)
    | LocTry(e,m)        -> Try (exprtoprim e, mtchtoprim e.loc m)
    | LocMarshal(e1,e2,ty) -> Marshal (exprtoprim e1, exprtoprim e2, ty)
    | LocMarshalz(mk,e2,ty) -> Marshalz (mk, exprtoprim e2, ty)
    | LocUnmarshal(e,ty) -> Unmarshal (exprtoprim e, ty)
    | LocRET(ty)         -> RET ty
    | LocSLOWRET(ty)     -> SLOWRET ty
    | LocCol(e,eqs,ty)   -> Col (exprtoprim e, eqs, ty)
    | LocOP(i,oe,es)     -> OP (i, oe, List.map (exprtoprim) es)
    | LocResolve(e,mn,rs) -> Resolve (exprtoprim e, mn, rs)
    | LocResolve_blocked(e,mn,rs) -> Resolve_blocked (exprtoprim e, mn, rs)
    | LocPar(e1,e2)     -> Par(exprtoprim e1, exprtoprim e2)
    | LocFresh(ty)       -> Fresh(ty)
    | LocCFresh(ty)      -> CFresh(ty)
    | LocHashMvf(x,eid,t) -> HashMvf(x,eid,t)
    | LocHashTs(t1,e,t2) -> HashTs(t1, exprtoprim e, t2)
    | LocHashHts(t1,e2,e1,t2) -> HashHts(t1, exprtoprim e2, exprtoprim e1, t2)
    | LocNameValue(n)    -> NameValue(n)
    | LocSwap(e1,e2,e3)  -> Swap (exprtoprim e1, exprtoprim e2, exprtoprim e3)
    | LocFreshfor(e1,e2) -> Freshfor(exprtoprim e1, exprtoprim e2)
    | LocSupport(ty,e)   -> Support(ty, exprtoprim e)
    | LocTie(mn,eid)     -> Tie(mn,eid)
    | LocNameOfTie(e)    -> NameOfTie(exprtoprim e)
    | LocValOfTie(e)     -> ValOfTie(exprtoprim e)

(* helper for mtchtoprim *)
and mtchtoprim' m =
  List.map (function (p,e) -> (p, exprtoprim e)) m

and mtchtoprim l m =
  let m' = mtchtoprim' m in {desc=m'; loc=l}

let rec primtoexpr e =
  match e with
      C0(c) -> no_loc (LocC0(c))
    | C1(c1,e)        -> no_loc (LocC1(c1,primtoexpr e))
    | Cons(e1,e2)     -> no_loc (LocCons(primtoexpr e1, primtoexpr e2))
    | Tup(es)         -> no_loc (LocTup(List.map (primtoexpr) es))
    | Op(op,el)       -> no_loc (LocOp(op,(List.map (primtoexpr) el)))
    | Loc(l)          -> no_loc (LocLoc(l))
    | Fn(m)           -> no_loc (LocFn(primtomtch m.desc))
    | Fun(ps,e1)      -> no_loc (LocFun(ps, primtoexpr e1))
    | InEnv(env,e)    -> no_loc (LocInEnv(env, primtoexpr e))
    | Clos(env,x2,ty2,bs2,e1,x1o)
      -> no_loc (LocClos(env, x2, ty2, bs2, primtoexpr e1, x1o))
    | TClos(env,itn,e)-> no_loc (LocTClos(env,itn,primtoexpr e))
    | Id(iid)         -> no_loc (LocId(iid))
    | Dot(mn,eid)     -> no_loc (LocDot(mn,eid))
    | HashDot(h,eid)  -> no_loc (LocHashDot(h,eid))
    | If(e1,e2,e3)    -> no_loc (LocIf (primtoexpr e1, primtoexpr e2, primtoexpr e3))
    | While(e1,e2)    -> no_loc (LocWhile (primtoexpr e1, primtoexpr e2))
    | LazyOp(lo,el)   -> no_loc (LocLazyOp(lo, (List.map (primtoexpr) el)))
    | Seq(e1,e2)      -> no_loc (LocSeq (primtoexpr e1, primtoexpr e2))
    | TAbs(itn,e)     -> no_loc (LocTAbs (itn, primtoexpr e))
    | TApp(e,t)       -> no_loc (LocTApp (primtoexpr e, t))
    | Pack(t1,e,t2)
      -> no_loc (LocPack(t1, primtoexpr e, t2))
    | Unpack(itn,iid,e1,e2)
      -> no_loc (LocUnpack(itn,iid, primtoexpr e1, primtoexpr e2))
    | Namecase(e1,r,x1,x2,e,e2,e3) -> no_loc (LocNamecase (primtoexpr e1,r,x1,x2,primtoexpr e,primtoexpr e2,primtoexpr e3))
    | App(e1,e2)      -> no_loc (LocApp (primtoexpr e1, primtoexpr e2))
    | Match(e,m)      -> no_loc (LocMatch (primtoexpr e, primtomtch m.desc))
    | Let(e1,(p,e2))  -> no_loc (LocLet (primtoexpr e1, (p, primtoexpr e2)))
    | LetMulti(ty,(ps,e1),(iid,e)) -> no_loc (LocLetMulti (ty,(ps,primtoexpr e1),(iid,(primtoexpr e))))
    | Letrec(ty,(iid,(m,e))) -> no_loc (LocLetrec (ty,(iid,(primtomtch m.desc, primtoexpr e))))
    | LetrecMulti(ty,(ps,(iid,(e1,e2)))) -> no_loc (LocLetrecMulti (ty,(ps,(iid,(primtoexpr e1, primtoexpr e2)))))
    | Raise(e)        -> no_loc (LocRaise (primtoexpr e))
    | Try(e,m)        -> no_loc (LocTry (primtoexpr e, primtomtch m.desc))
    | Marshal(e1,e2,ty) -> no_loc (LocMarshal (primtoexpr e1, primtoexpr e2, ty))
    | Marshalz(mk,e2,ty) -> no_loc (LocMarshalz (mk, primtoexpr e2, ty))
    | Unmarshal(e,ty) -> no_loc (LocUnmarshal (primtoexpr e, ty))
    | RET(ty)         -> no_loc (LocRET ty)
    | SLOWRET(ty)     -> no_loc (LocSLOWRET ty)
    | Col(e,eqs,ty)   -> no_loc (LocCol (primtoexpr e, eqs, ty))
    | OP(i,oe,es)     -> no_loc (LocOP (i, oe, List.map (primtoexpr) es))
    | Resolve(e,mn,rs) -> no_loc (LocResolve (primtoexpr e, mn, rs))
    | Resolve_blocked(e,mn,rs) -> no_loc (LocResolve_blocked (primtoexpr e, mn, rs))
    | Par(e1,e2)     -> no_loc (LocPar (primtoexpr e1, primtoexpr e2))
    | Fresh(ty)       -> no_loc (LocFresh (ty))
    | CFresh(ty)      -> no_loc (LocCFresh (ty))
    | HashMvf(x,eid,t) -> no_loc (LocHashMvf (x,eid,t))
    | HashTs(t1,e,t2) -> no_loc (LocHashTs (t1, primtoexpr e, t2))
    | HashHts(t1,e2,e1,t2) -> no_loc (LocHashHts (t1, primtoexpr e2, primtoexpr e1, t2))
    | NameValue(n)    -> no_loc (LocNameValue(n))
    | Swap(e1,e2,e3)  -> no_loc (LocSwap (primtoexpr e1, primtoexpr e2, primtoexpr e3))
    | Freshfor(e1,e2) -> no_loc (LocFreshfor(primtoexpr e1, primtoexpr e2))
    | Support(ty,e)   -> no_loc (LocSupport(ty, primtoexpr e))
    | Tie(mn,eid)     -> no_loc (LocTie(mn,eid))
    | NameOfTie(e)    -> no_loc (LocNameOfTie(primtoexpr e))
    | ValOfTie(e)     -> no_loc (LocValOfTie(primtoexpr e))

and primtomtch m =
     List.map (function (p,e) -> (p, primtoexpr e)) m

(* tmap_* f find the top-level type nodes and apply f to them; they
don't go inside hashes or Marshalled nodes. *)
(* NO FRESHENING *)

let rec tmap_con0 f c =
  match c with
  | Nil t -> Nil (f t)
  | NONE t -> NONE (f t)
  | B0 b0 -> c

and tmap_con1 f c =
  match c with
  | Inj(i,t) -> Inj (i, f t)
  | SOME -> c
  | TieCon -> c
  | Node -> c
  | B1 b1 -> c

and tmap_op f op =
  match op with
  | Ref t -> Ref (f t)
  | Deref t -> Deref (f t)
  | Assign t -> Assign (f t)
  | Assign' t -> Assign' (f t)
  | Equal t -> Equal (f t)
  | ListConcat t -> ListConcat (f t)
  | CreateThread t -> CreateThread (f t)
  | Exit t -> Exit (f t)
  | CompareName t -> CompareName (f t)
  | NameToString t -> NameToString (f t)
  | Less | LessEq | Greater | GreaterEq | Mod | Land | Lor | Lxor | Lsl | Lsr | Asr | UMinus
  | Plus | Minus | Times | Divide | StrConcat
  | Self | Kill | CreateMutex | Lock | TryLock | Unlock | CreateCVar | Wait | Waiting | Signal | Broadcast
  | Thunkify | Unthunkify
    -> op

and tmap_resolvespec_item f resolvespec_item =
  match resolvespec_item with
  | Resolve_static_link
  | Resolve_here_already
  | Resolve_url _ -> resolvespec_item

and tmap_resolvespec f resolvespec =
  List.map (tmap_resolvespec_item f) resolvespec

and tmap_esubst f esubst =
  IIdentMap.map (tmap_expr f) esubst

and tmap_bs f bs =
  List.map (fun (eqs,t) -> (tmap_eqs f eqs, f t)) bs

and tmap_expr f e =  (* NO FRESHENING *)
  at_loc e.loc
  (match e.desc with
  | LocC0 c          -> LocC0 (tmap_con0 f c)
  | LocC1 (c, e)     -> LocC1 (tmap_con1 f c, tmap_expr f e)
  | LocCons (e1,e2)  -> LocCons (tmap_expr f e1, tmap_expr f e2)
  | LocTup es        -> LocTup (List.map (tmap_expr f) es)
  | LocOp (oe,el)    -> LocOp ((tmap_op_or_econst f oe), List.map (tmap_expr f) el)
  | LocLoc l         -> e.desc
  | LocFn mtch       -> LocFn (tmap_mtch f mtch)
  | LocFun (ps,e1)   -> LocFun (List.map (tmap_pat f) ps, tmap_expr f e1)
  | LocInEnv _       -> raise (Never_happen "tmap_expr on LocInEnv")
  | LocClos _        -> raise (Never_happen "tmap_expr on LocClos")
                        (* NB: there's an implementation of this in the repo, v1.71 *)
  | LocTClos _       -> raise (Never_happen "tmap_expr on LocTClos")
  | LocId _            -> e.desc
  | LocDot _           -> e.desc
  | LocHashDot _       -> e.desc
  | LocIf (e1, e2, e3) -> LocIf (tmap_expr f e1, tmap_expr f e2, tmap_expr f e3)
  | LocWhile (e1, e2)  -> LocWhile (tmap_expr f e1, tmap_expr f e2)
  | LocLazyOp (lo,el)  -> LocLazyOp (lo, List.map (tmap_expr f) el)
  | LocSeq (e1, e2)    -> LocSeq (tmap_expr f e1, tmap_expr f e2)
  | LocApp (e1, e2)    -> LocApp (tmap_expr f e1, tmap_expr f e2)
  | LocTAbs(itn,e)     -> LocTAbs (itn, tmap_expr f e)
  | LocTApp(e,t)       -> LocTApp (tmap_expr f e, f t)
  | LocPack(t1,e,t2) -> LocPack (f t1, tmap_expr f e, f t2)
  | LocUnpack(itn,iid,e1,e2) -> LocUnpack (itn, iid, tmap_expr f e1, tmap_expr f e2)
  | LocNamecase(e1,r,x1,x2,e,e2,e3) -> LocNamecase(tmap_expr f e1, r, x1, x2, tmap_expr f e, tmap_expr f e2, tmap_expr f e3)
  | LocMatch (e, m)    -> LocMatch (tmap_expr f e, tmap_mtch f m)
  | LocLet (e, (p',e')) -> LocLet (tmap_expr f e, (tmap_pat f p', tmap_expr f e'))
  | LocLetMulti (ty,(ps,e1),(iid,e)) ->
                       LocLetMulti (f ty,
                                 (List.map (tmap_pat f) ps, tmap_expr f e1),
                                 (iid,(tmap_expr f e)))
  | LocLetrec (t,(i,(mtch,expr))) -> LocLetrec (f t, (i,(tmap_mtch f mtch, tmap_expr f expr)))
  | LocLetrecMulti(ty,(ps,(iid,(e1,e2)))) ->
                          LocLetrecMulti (f ty,
                                       (List.map (tmap_pat f) ps,
                                        (iid,(tmap_expr f e1, tmap_expr f e2))))
  | LocRaise (e)           -> LocRaise (tmap_expr f e)
  | LocTry (e, mtch)       -> LocTry (tmap_expr f e, tmap_mtch f mtch)
  | LocMarshal (e1, e2, t)  -> LocMarshal (tmap_expr f e1, tmap_expr f e2, f t)
  | LocMarshalz (mk, e2, t) -> LocMarshalz (mk, tmap_expr f e2, f t)
  | LocUnmarshal (e, t)    -> LocUnmarshal (tmap_expr f e, f t)
  | LocRET t               -> LocRET (f t)
  | LocSLOWRET t           -> LocSLOWRET (f t)
  | LocCol (e, eqs, t)     -> LocCol (tmap_expr f e, tmap_eqs f eqs, f t)
  | LocOP (n, oe, es)      -> LocOP (n, tmap_op_or_econst f oe, List.map (tmap_expr f) es)
  | LocResolve (e, mn, resolvespec ) -> LocResolve (tmap_expr f e, mn, tmap_resolvespec f resolvespec)
  | LocResolve_blocked (e, mn, resolvespec ) -> LocResolve_blocked (tmap_expr f e, mn, tmap_resolvespec f resolvespec)
  | LocPar (e1, e2)        -> LocPar (tmap_expr f e1, tmap_expr f e2)
  | LocFresh t             -> LocFresh (f t)
  | LocCFresh t            -> LocCFresh (f t)
  | LocHashMvf(x,eid,t)    -> LocHashMvf(x, eid, f t)
  | LocHashTs(t1,e,t2)     -> LocHashTs(f t1, e, f t2)
  | LocHashHts(t1,e2,e1,t2) -> LocHashHts(f t1, e2, e1, f t2)
  | LocNameValue(n)        -> LocNameValue(tmap_name_value f n)
  | LocSwap(e1,e2,e3)      -> LocSwap (e1, e2, e3)
  | LocFreshfor(e1,e2)     -> LocFreshfor (e1, e2)
  | LocSupport(t,e)        -> LocSupport (f t, e)
  | LocTie(mn,eid)         -> LocTie (mn,eid)
  | LocNameOfTie(e)        -> LocNameOfTie (e)
  | LocValOfTie(e)         -> LocValOfTie (e) )

(* fz does not go inside hashes *)
(* kw doesn't go inside name_values either *)
and tmap_name_value f = function
   nv ->  nv
(* kw thinks this is wrong, since the types had better be closed before we construct the name_value
 *    VHashMvf(n,h,eid,t)    -> VHashMvf(n,h,eid,f t)
 *  | VHashTs(n,t,s)         -> VHashTs(n,f t,s)
 *  | VHashHts(n,t,s,nv)     -> VHashHts(n,f t,s,tmap_name_value f nv)
 *  | VHashName(n,t)         -> VHashName(n,f t)
 *)



and tmap_pat f = function
  | PWild t -> PWild (f t)
  | PVar (internal_ident, t) -> PVar (internal_ident, f t)
  | PC0 c -> PC0 (tmap_con0 f c)
  | PC1 (c, pat) -> PC1 (tmap_con1 f c, tmap_pat f pat)
  | PCons (pat1, pat2) -> PCons (tmap_pat f pat1, tmap_pat f pat2)
  | PTup pats -> PTup (List.map (tmap_pat f) pats)
  | PTyped (pat, t) -> PTyped (tmap_pat f pat, f t)

and tmap_eqn f = function
    EHash (hash, external_typname, t) -> EHash (hash, external_typname, f t)
  | EMod (name, external_typname, t) -> EMod (name, external_typname, f t)

and tmap_eqs f eqs =
  List.map (tmap_eqn f) eqs

and tmap_kind f = function
  | KType -> KType
  | KEq t -> KEq (f t)

and tmap_op_or_econst f = function
  | OEOp(op)         -> OEOp(tmap_op f op)
  | OEEconst(_) as e -> e

and tmap_pat_expr f (pat, e) =
  (tmap_pat f pat, tmap_expr f e)

and tmap_mtch f mtch = List.map (tmap_pat_expr f) mtch

and tmap_signature_item f = function s ->
  let s' = match s.desc with
  | SVal (ident, t) -> SVal (ident, f t)
  | STyp (typname, k) -> STyp (typname, tmap_kind f k)
in {s with desc=s'}

and tmap_structure_item f = function s ->
  let s' = match s.desc with
  | StrVal (ident, e) -> StrVal (ident, tmap_expr f e)
  | StrValMulti (ident, ps, e) -> StrValMulti (ident, List.map (tmap_pat f) ps, tmap_expr f e)
  | StrTyp (typname, t) -> StrTyp (typname, f t)
  in at_loc s.loc s'

and tmap_signature' f sign =
 List.map (tmap_signature_item f) sign

and tmap_signature f sign =
  {sign with desc=List.map (tmap_signature_item f) sign.desc}

and tmap_structure' f str =
  List.map (tmap_structure_item f) str

and tmap_structure f str =
  {str with desc=List.map (tmap_structure_item f) str.desc}

(* emap_* f find all top-level term nodes and apply f them; they
don't go inside hashes. *)

let rec emap_structure_item f str_item =
  match str_item.desc with
  | StrVal (ident, e) -> at_loc str_item.loc (StrVal (ident, f e))
  | StrValMulti (ident, ps, e) -> let (ps,e) = freshen_pats_foo (ps,e) in
                                  at_loc str_item.loc (StrValMulti (ident, ps, f e))
  | StrTyp _ -> str_item

and emap_structure freshen f str =
  let str' = if freshen then freshen_structure str else str in
  {str' with desc=List.map (emap_structure_item f) str'.desc}

(* early_tmap_typ ft t tries to apply ft to t; if ft returns Some x
then the result is x; if None, early_tmap_typ traverses through the
smaller type expressions with the same ``quit early if possible''
strategy; it does not go inside hashes. *)

let rec early_tmap_typ ft t =
  begin match ft t with
    Some x -> x
  | None ->
      begin match t with
      | TTyCon0 tc0 -> t
      | TTyCon1 (tc1,t0) -> TTyCon1 (tc1,early_tmap_typ ft t0)
      | TTup ts -> TTup (List.map (early_tmap_typ ft) ts)
      | TSum ts -> TSum (List.map (early_tmap_typ ft) ts)
      | TFunc (t1, t2) -> TFunc (early_tmap_typ ft t1, early_tmap_typ ft t2)
      | TXDot _ -> t
      | TVar _ -> t
      | TTyName _ -> t
      | TForall (itn, t0) -> TForall (itn, early_tmap_typ ft t0)
      | TExists (itn, t0) -> TExists (itn, early_tmap_typ ft t0)
      end
  end

(* tsub_typ performs a simultaneous internal typname to type expression
substitution on a type; doesn't go inside hashes. *)
(* ASSUMES NO SHADOWING OF TYPE BINDERS!! *)
let rec tsub_typ tsubs t =
  let dosub = function
    | TVar i ->
        begin try Some (assoc_by tn_eq i tsubs) with Not_found -> None end
    | _ -> None
  in early_tmap_typ dosub t

(* early_emap_expr fe tries to apply fe to e; if fe returns Some x
then the result is x; if None, early_emap_expr traverses through the
smaller term expressions with the same ``quit early if possible''
strategy.  It does not go inside hashes, Resolve, or
Resolve_blocked.  (To process the inside of Resolve nodes, for
example, make fe explicitly handle Resolve nodes.)

It also does not go inside Clos or InEnv nodes - we need to carefully track
what early_emap_expr is applied to - sometimes it is to expressions
which are known not to contain closures, other times to expressions
which may contain closures but where those are expression-ident
closed.  The fe should handle any Clos, TClos, or InEnv, otherwise this raises an
exception.

Colour (eqs) is passed to fe if known.
*)

(* bool says whether to freshen or not; freshening is no longer supported *)
let rec early_emap_expr frp fe eqs e =
  (if frp then raise (Util.Unimplemented "early_emap_expr:freshening"));  (* use early_emap_fresh_* *)
  begin match fe eqs e with
    Some x -> x
  | None ->
      let loc e' = { desc = e'; loc = e.loc } in
      match e.desc with
      | LocC0 _ -> e
      | LocC1 (c1, e0) -> loc (LocC1 (c1, early_emap_expr frp fe eqs e0))
      | LocCons (e1, e2) -> loc (LocCons (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocTup es -> loc (LocTup (List.map (early_emap_expr frp fe eqs) es))
      | LocOp (oe,el) -> loc (LocOp (oe, (List.map (early_emap_expr frp fe eqs) el)))
      | LocLoc _ -> e
      | LocFn mtch -> loc (LocFn (early_emap_mtch frp fe eqs mtch))
      | LocFun (ps,e1) -> loc (LocFun (ps, early_emap_expr frp fe eqs e1))
      | LocInEnv _ ->
          raise (Never_happen "early_emap_expr reached an InEnv and doesn't know what to do.  The fe should have handled this case.")
      | LocClos _ ->
          raise (Never_happen "early_emap_expr reached a Clos and doesn't know what to do. The fe should have handled this case.")
      | LocTClos _ ->
          raise (Never_happen "early_emap_expr reached a TClos and doesn't know what to do. The fe should have handled this case.")
      | LocId _ -> Debug.print(function () -> "Returning from ID");e
      | LocDot _ -> e
      | LocHashDot _ -> e
      | LocIf (e1, e2, e3) ->
          loc (LocIf (early_emap_expr frp fe eqs e1,
                 early_emap_expr frp fe eqs e2,
                 early_emap_expr frp fe eqs e3))
      | LocWhile (e1, e2) -> loc (LocWhile (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocLazyOp (lo,el) -> loc (LocLazyOp (lo, (List.map (early_emap_expr frp fe eqs) el)))
      | LocSeq (e1, e2)   -> loc (LocSeq (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocApp (e1, e2)   -> loc (LocApp (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocTAbs(itn,e)  -> loc (LocTAbs (itn, early_emap_expr frp fe eqs e))
      | LocTApp(e,t)     -> loc (LocTApp (early_emap_expr frp fe eqs e, t))
      | LocPack(t1,e,t2) -> loc (LocPack (t1, early_emap_expr frp fe eqs e, t2))
      | LocUnpack(itn,iid,e1,e2) ->
          loc (LocUnpack (itn, iid, early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocNamecase(e1,r,x1,x2,e,e2,e3) -> loc (LocNamecase(early_emap_expr frp fe eqs e1, r, x1, x2, early_emap_expr frp fe eqs e, early_emap_expr frp fe eqs e2, early_emap_expr frp fe eqs e3))
      | LocMatch (e, m)   -> loc (LocMatch (early_emap_expr frp fe eqs e, early_emap_mtch frp fe eqs m))
      | LocLet (e1, pe)   -> loc (LocLet (early_emap_expr frp fe eqs e1, early_emap_pat_expr frp fe eqs pe))
      | LocLetMulti (ty,(ps,e1),ie) ->
          (let (iid,e) = ie in
          let (ps,e1) = (ps,e1) in
          loc (LocLetMulti (ty,
                            (ps, early_emap_expr frp fe eqs e1),
                            (iid,(early_emap_expr frp fe eqs e)))))
      | LocLetrec (t,ime) -> (let (i,(mtch, e0)) = ime in (*Debug.print_string_really "letrec matched ";*)
          let pair = (early_emap_mtch frp fe eqs mtch, early_emap_expr frp fe eqs e0) in loc (LocLetrec (t, (i,(pair)))))
      | LocLetrecMulti(ty,(ps,iee)) ->
          (let (iid,(e1,e2)) = iee in
          let (ps, e1) = (ps,e1) in
          loc (LocLetrecMulti (ty,
                               (ps,
                                (iid,(early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))))))
      | LocRaise (e0) -> loc (LocRaise (early_emap_expr frp fe eqs e0))
      | LocTry (e0, mtch) -> loc (LocTry (early_emap_expr frp fe eqs e0, early_emap_mtch frp fe eqs mtch))
      | LocMarshal (e1, e2, t) ->
          loc (LocMarshal (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2, t))
      | LocMarshalz (mk, e2, t) ->
          loc (LocMarshalz (mk, early_emap_expr frp fe (Some []) e2, t))
      | LocUnmarshal (e0, t) -> loc (LocUnmarshal (early_emap_expr frp fe eqs e0, t))
      | LocRET _ -> e
      | LocSLOWRET _ -> e
      | LocCol (e0, eqs', t) -> loc (LocCol (early_emap_expr frp fe (Some eqs') e0, eqs', t))
      | LocOP (n, oe, es) ->
          loc (LocOP (n, oe, List.map (early_emap_expr frp fe (Some [])) es))
      | LocResolve _ -> e
      | LocResolve_blocked _ -> e
      | LocPar (e1, e2) -> loc (LocPar (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocFresh _ -> e
      | LocCFresh _ -> e
      | LocHashMvf _ -> e
      | LocHashTs(t1,e,t2) -> loc (LocHashTs(t1, (early_emap_expr frp fe eqs e), t2))  (* FZ *)
      | LocHashHts(t1,e2,e1,t2)
	-> loc (LocHashHts(t1, early_emap_expr frp fe eqs e2, early_emap_expr frp fe eqs e1, t2)) (* FZ *)
      | LocNameValue _ -> e
      | LocSwap(e1,e2,e3)
        -> loc (LocSwap (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2, early_emap_expr frp fe eqs e3))
      | LocFreshfor(e1,e2) -> loc (LocFreshfor (early_emap_expr frp fe eqs e1, early_emap_expr frp fe eqs e2))
      | LocSupport(ty,e)   -> loc (LocSupport(ty, early_emap_expr frp fe eqs e))
      | LocTie _ -> e
      | LocNameOfTie(e)    -> loc (LocNameOfTie(early_emap_expr frp fe eqs e))
      | LocValOfTie(e)     -> loc (LocValOfTie(early_emap_expr frp fe eqs e))
 end

and early_emap_prim_expr frp fe eqs e =
  (if frp then raise (Util.Unimplemented "early_emap_expr:freshening"));  (* use early_emap_fresh_* *)
  begin match fe eqs e with
    Some x -> x
  | None ->
      (* Debug.print' Opts.DBC_reachable (fun () ->"doing " ^ print_hack e); *)
      match e with
      | C0 _ -> e
      | C1 (c1, e0) -> C1 (c1, early_emap_prim_expr frp fe eqs e0)
      | Cons (e1, e2) -> Cons (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | Tup es -> Tup (List.map (early_emap_prim_expr frp fe eqs) es)
      | Op (oe,el) -> Op (oe, (List.map (early_emap_prim_expr frp fe eqs) el))
      | Loc _ -> e
      | Fn m -> Fn (at_loc m.loc (early_emap_prim_mtch frp fe eqs m.desc))
      | Fun (ps,e1) -> Fun (ps, early_emap_prim_expr frp fe eqs e1)
      | InEnv _ ->
          raise (Never_happen "early_emap_prim_expr reached an InEnv and doesn't know what to do.  The fe should have handled this case.")
      | Clos _ ->
          raise (Never_happen "early_emap_prim_expr reached a Clos and doesn't know what to do. The fe should have handled this case.")
      | TClos _ ->
          raise (Never_happen "early_emap_prim_expr reached a TClos and doesn't know what to do. The fe should have handled this case.")
      | Id _ -> Debug.print(function () -> "Returning from Id");e
      | Dot _ -> e
      | HashDot _ -> e
      | If (e1, e2, e3) ->
          If (early_emap_prim_expr frp fe eqs e1,
              early_emap_prim_expr frp fe eqs e2,
              early_emap_prim_expr frp fe eqs e3)
      | While (e1, e2) -> While (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | LazyOp (lo,el) -> LazyOp (lo, (List.map (early_emap_prim_expr frp fe eqs) el))
      | Seq (e1, e2) -> Seq (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | App (e1, e2) -> App (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)

      | TAbs(itn,e)  -> TAbs (itn, early_emap_prim_expr frp fe eqs e)
      | TApp(e,t)     -> TApp (early_emap_prim_expr frp fe eqs e, t)
      | Pack(t1,e,t2) -> Pack (t1, early_emap_prim_expr frp fe eqs e, t2)
      | Unpack(itn,iid,e1,e2) ->
          Unpack (itn, iid, early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | Namecase(e1,r,x1,x2,e,e2,e3) -> Namecase(early_emap_prim_expr frp fe eqs e1, r, x1, x2, early_emap_prim_expr frp fe eqs e, early_emap_prim_expr frp fe eqs e2, early_emap_prim_expr frp fe eqs e3)
      | Match (e, m) -> Match (early_emap_prim_expr frp fe eqs e, at_loc m.loc (early_emap_prim_mtch frp fe eqs m.desc))
      | Let (e1, pe) -> Let (early_emap_prim_expr frp fe eqs e1, early_emap_prim_pat_expr frp fe eqs pe)
      | LetMulti (ty,(ps,e1),ie) ->
          (let (iid,e) = ie in
          let (ps,e1) = (ps,e1) in
          LetMulti (ty,
                    (ps, early_emap_prim_expr frp fe eqs e1),
                    (iid,(early_emap_prim_expr frp fe eqs e))))
      | Letrec (t,ime) -> (let (i,(mtch,e0)) = ime in (*Debug.print_string_really "letrec matched ";*)
          let pair = (at_loc mtch.loc (early_emap_prim_mtch frp fe eqs mtch.desc), early_emap_prim_expr frp fe eqs e0) in Letrec (t, (i,(pair))))
      | LetrecMulti(ty,(ps,iee)) ->
          (let (iid,(e1,e2)) = iee in
          let (ps, e1) = (ps,e1) in
          LetrecMulti (ty,
                       (ps,
                        (iid,(early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)))))
      | Raise (e0) -> Raise (early_emap_prim_expr frp fe eqs e0)
      | Try (e0, mtch) -> Try (early_emap_prim_expr frp fe eqs e0, at_loc mtch.loc (early_emap_prim_mtch frp fe eqs mtch.desc))
      | Marshal (e1, e2, t) ->
          Marshal (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2, t)
      | Marshalz (mk, e2, t) ->
          Marshalz (mk, early_emap_prim_expr frp fe (Some []) e2, t)
      | Unmarshal (e0, t) -> Unmarshal (early_emap_prim_expr frp fe eqs e0, t)
      | RET _ -> e
      | SLOWRET _ -> e
      | Col (e0, eqs', t) -> Col (early_emap_prim_expr frp fe (Some eqs') e0, eqs', t)
      | OP (n, oe, es) ->
          OP (n, oe, List.map (early_emap_prim_expr frp fe (Some [])) es)
      | Resolve _ -> e
      | Resolve_blocked _ -> e
      | Par (e1, e2) -> Par (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | Fresh _ -> e
      | CFresh _ -> e
      | HashMvf _ -> e
      | HashTs(t1,e,t2) -> HashTs(t1, (early_emap_prim_expr frp fe eqs e), t2)
      | HashHts(t1,e2,e1,t2)
        -> HashHts(t1, early_emap_prim_expr frp fe eqs e2, early_emap_prim_expr frp fe eqs e1, t2)
      | NameValue _ -> e
      | Swap(e1,e2,e3)
        -> Swap (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2, early_emap_prim_expr frp fe eqs e3)
      | Freshfor(e1,e2) -> Freshfor (early_emap_prim_expr frp fe eqs e1, early_emap_prim_expr frp fe eqs e2)
      | Support(ty,e)   -> Support(ty, early_emap_prim_expr frp fe eqs e)
      | Tie _ -> e
      | NameOfTie(e)    -> NameOfTie(early_emap_prim_expr frp fe eqs e)
      | ValOfTie(e)     -> ValOfTie(early_emap_prim_expr frp fe eqs e)
  end

and early_emap_mtch frp fe eqs mtch =
  List.map (early_emap_pat_expr frp fe eqs) mtch

and early_emap_prim_mtch frp fe eqs mtch =
  List.map (early_emap_prim_pat_expr frp fe eqs) mtch

and early_emap_pat_expr frp fe eqs (pat, expr) =
  let (pat, expr) = (pat, expr) in
  (pat, early_emap_expr frp fe eqs expr)

and early_emap_prim_pat_expr frp fe eqs (pat, expr) =
  let (pat, expr) = (pat, expr) in
  (pat, early_emap_prim_expr frp fe eqs expr)

and early_emap_esubst frp fe eqs subs =
  IIdentMap.map (early_emap_prim_expr frp fe eqs) subs

(* free module variables (with no duplicates in the output) *)

(* the fmv functions assume no Clos nodes in the input - reasonable as
we only use these for marshalling, which has to flatten anyway *)

let extend_set_ref eq set_ref x =
  if not (mem_by eq x !set_ref) then set_ref := x :: !set_ref

let rec fmv_prim_expr' ms_ref expr =
  let add_fmv _ e0 = match e0 with
      Dot (modname, _) -> extend_set_ref mn_eq ms_ref modname; Some e0
    | Clos _ -> raise (Never_happen "fmv_expr' called on a Clos - the argument should have been flattened ")
    | InEnv _ -> raise (Never_happen "fmv_expr' called on an InEnv - the argument should have been flattened ")
    | _ -> None
  in
  let _ = early_emap_prim_expr false add_fmv None expr in  (* FIXME: this is broken; it should walk over patterns too *)
  let _ = tmap_expr (fmv_typ' ms_ref) (primtoexpr expr) in expr  (* ick: primtoexpr *)


and fmv_typ' ms_ref ty =
  let add_fmv = function
    | TXDot (Modname(modname), _) as ty0 -> extend_set_ref mn_eq ms_ref modname; Some ty0
    | _ -> None
  in let _ = early_tmap_typ add_fmv ty in ty

and fmv_str' ms_ref str =
    let add_fmv = function s ->
      match s.desc with
      | StrVal (_, expr) -> let _ = fmv_prim_expr' ms_ref (exprtoprim expr) in ()
      | StrValMulti (_, _, expr) ->
	  let _ = fmv_prim_expr' ms_ref (exprtoprim expr) in ()
          (* FIXME: this is broken because it doesn't walk over the patterns too;
             but fmv_expr' above is broken likewise (consider what early_emap_prim_expr
             does to the mtch in Fn, for example). *)
      | StrTyp (_, ty) -> let _ = fmv_typ' ms_ref ty in ()
    in List.iter add_fmv str
;;

let fmv_typ ty =
  let ms_ref = ref [] in let _ = fmv_typ' ms_ref ty in !ms_ref
let fmv_expr expr =
  let ms_ref = ref [] in let _ = fmv_prim_expr' ms_ref (exprtoprim expr) in !ms_ref
let fmv_prim_expr expr =
  let ms_ref = ref [] in let _ = fmv_prim_expr' ms_ref  expr in !ms_ref
let fmv_str str =
  let ms_ref = ref [] in let _ = fmv_str' ms_ref str.desc in !ms_ref
let fmv_mo = function
  | None -> []
  | Some modname -> [modname]

(* locations of an expression *)
(* assumes no Clos nodes, similarly to fmv *)
let rec free_locs' locs_ref expr =
  let add_free_locs _ e = match e with
    | (Loc loc) as e0 ->  extend_set_ref l_eq locs_ref loc; Some e
    | InEnv _ -> raise (Never_happen "free_locs' called on an InEnv - the argument should have been flattened ")
    | Clos _ -> raise (Never_happen "free_locs' called on a Clos - the argument should have been flattened ")
    | _ -> None
  in
  let _ = early_emap_prim_expr false add_free_locs None expr in ();;




let prim_free_locs expr =
  let locs_ref = ref [] in let _ = free_locs' locs_ref expr in !locs_ref

let free_locs expr =
  let locs_ref = ref [] in let _ = free_locs' locs_ref (exprtoprim expr) in !locs_ref

let rec reachable_modules_locs =
  fun defs store (defs2, expr) ->
    let modnames_ref, locs_ref = ref [], ref [] in

    let rec f _ e0 =
      match e0 with
	Dot(modname, _) -> if not (mem_by mn_eq modname !modnames_ref)
	    then (modnames_ref := modname::(!modnames_ref);
		  f_def modname; Some e0)
	    else None
      |	Loc l -> if not (mem_by l_eq l !locs_ref)
	    then (locs_ref := l::(!locs_ref);
		  let _ = early_emap_prim_expr false f None (List.assoc l store) in ();
		  Some e0)
	    else None

      |	_ -> None

    and f_def mn =
      try
        let def = List.find (
          function m -> match m with
	      Mod_compile(mn', _) | Import_compile(mn', _)  -> mn' = mn
            |	_ -> false) defs2
        in
          match def with Import_compile(mn, ibody) ->
	    (match ibody.ic_mo with
	         None -> ()
	       | Some mn' ->
	           if not (mem_by mn_eq mn' !modnames_ref)
	           then (modnames_ref := mn'::(!modnames_ref);
		         f_def mn')
	           else ()
	    )
            |	Mod_compile(mn, mbody) -> (
	          let go s = match s.desc with
	              StrVal(_,e) | StrValMulti(_,_,e) ->  let _ = early_emap_prim_expr false f None (exprtoprim e) in ()
	            | StrTyp(_, t) ->
	                let defs' = fmv_typ t in
	                  List.iter f_def defs'
	          in List.iter go mbody.mc_str.desc
	        )
            |	_ -> ()
      with Not_found -> ()

(*     and f_def_def def = match def with *)
(* 	Mod_compile(mn, _) *)
(*       | Import_compile(mn, _) -> f_def mn *)
(*       |	_ -> () *)

    in let _ = early_emap_prim_expr false f None expr in
    (* let _ = List.map f_def_def defs2 in *)
    (!modnames_ref, !locs_ref)


(* calculate the maximum of a non-empty list *)
let max x y = if x < y then y else x
let rec max_list = function
  | [] -> assert false
  | [z] -> z
  | z::zs -> max z (max_list zs)

module IntMap = Map.Make (struct type t = int;; let compare = compare; end)
(* let rec make_loc_map min_loc = function *)
(*   | [] -> IntMap.empty *)
(*   | l::ls -> IntMap.add l min_loc (make_loc_map (min_loc+1) ls) *)
let rec make_loc_map = function
  | [] -> IntMap.empty
  | l::ls -> IntMap.add l (fresh_location()) (make_loc_map ls)

(* freshen a marshalled body, by giving all top-level definitions
distinct fresh names and locations distinct numbers different from the
ones in the argument *)



(* we lose the locations here ... FIX *)
(* FZ says: we do not pass location list around anymore, but we rely on fresh_location instead *)
let freshen_mb : marshalled_body -> marshalled_body
= fun mb ->
  (* freshen the top-level definitions *)
  let (defs, defs_swaps) = freshen_definitions' mb.mb_defs in
  let store = swap_list defs_swaps mb.mb_store in
  let storeenv = swap_list defs_swaps mb.mb_storeenv in
  let expr = swap_list defs_swaps mb.mb_expr in
  let ty = swap_list defs_swaps mb.mb_typ in

  (* pick a safe set of locs *)
(* FZ says: we do not need this anymore *)
(*   let minimum_safe_loc =             *)
(*     match locs_to_avoid with         *)
(*     | [] -> 0                        *)
(*     | _ -> max_list locs_to_avoid + 1 in *)

(*  let loc_map = make_loc_map minimum_safe_loc store_dom in *)

  let (store_dom, store_ran) = List.split store in
  let loc_map = make_loc_map store_dom in

  (* the substitution functions *)
  (* we need to go inside InEnv and Clos, because even though they are
     expression-closed, they are not location-closed.  But at the moment,
     marshalled values are flattened first, and so there is no need to
     implement the required fiddling.   (would be better to just use
     polytypic swaps, anyway...) *)

  let sub_loc l = try IntMap.find l loc_map with Not_found -> raise (Never_happen ("Couldn't find loc " ^ string_of_int l)) in

  let sub_expr eqs expr =
    let f = fun eqso e -> match e.desc with
      | LocLoc l -> Some (at_loc e.loc (LocLoc (sub_loc l)))
      | LocInEnv _ as e' -> raise (Unimplemented "freshen_mb: InEnv")
      | LocClos _ as e' -> raise (Unimplemented "freshen_mb: Clos")
      | _ -> None in
    early_emap_expr false f eqs expr in
  let sub_prim_expr eqs expr =
    let f = fun eqso e -> match e with
      | Loc l -> Some (Loc (sub_loc l))
      | InEnv _ as e' -> raise (Unimplemented "freshen_mb: InEnv")
      | Clos _ as e' -> raise (Unimplemented "freshen_mb: Clos")
      | _ -> None in
    early_emap_prim_expr false f eqs expr in

  (* rename all locs *)
  let store_dom' = List.map sub_loc store_dom in
  let store_ran' = List.map (sub_prim_expr (Some [])) store_ran in
  let store' = List.combine store_dom' store_ran' in
  let (storeenv_dom, storeenv_ran) = List.split storeenv in
  let storeenv_dom' = List.map sub_loc storeenv_dom in
  let storeenv' = List.combine storeenv_dom' storeenv_ran in
  let expr' = sub_expr (Some []) expr in
  { mb_ne = mb.mb_ne;
    mb_defs = defs;
    mb_store = store';
    mb_storeenv = storeenv';
    mb_expr = expr';
    mb_typ = ty;
  }


(* the filter function from the minicaml-rt:
 - the sign may not contain any val fields;
 - the str should be sufficiently type-flattened to avoid dangling
   names
*)
exception Filter_val_fields_in_sign_not_allowed
exception Filter_sign_missing_fields

let rec filter_str_sign' str sign =


  match (str, sign) with (_, []) -> []
  | (_, s::ss) ->
      match s.desc with SVal _ -> raise Filter_val_fields_in_sign_not_allowed
	  | STyp (tn', _ ) -> (
	      match str with [] -> raise Filter_sign_missing_fields
	      |	(str::strs) ->
		(
		match str.desc with
		  StrTyp(tn,t) ->  if typname_ext tn = typname_ext tn' then
		    str :: filter_str_sign' strs ss
		  else
		    filter_str_sign' strs sign
		| StrVal _ | StrValMulti _ -> filter_str_sign' strs sign
		      )
		  )

let filter_str_sign str sign =
  let str' = filter_str_sign' str.desc sign.desc
      in {str with desc=str'}

let find_val_field sign x =
  try List.find (fun s ->
    match s.desc with SVal((x',i), _) -> x = x'
      | STyp((x',i), _) -> false) sign.desc
with Not_found -> assert false

let find_typ_field sign x =
try
  List.find (fun s ->
    match s.desc with SVal((x',i), _) ->false
    | STyp((x',i), _) -> x=x') sign.desc
with Not_found -> assert false

(*
match (str,sign) with
  | (_, s:: _) when s.desc = (SVal) -> raise Filter_val_fields_in_sign_not_allowed
 | (StrTyp (tn, t) as str_item :: str', STyp (tn', _) :: sign') ->
      if typname_ext tn = typname_ext tn' then
        str_item :: filter_str_sign str' sign'
      else
        filter_str_sign str' sign
  | (StrVal _ :: str', _) -> filter_str_sign str' sign
  | (StrValMulti _ :: str', _) -> filter_str_sign str' sign
  | (_, []) -> []
  | ([], STyp _ :: _ ) -> raise Filter_sign_missing_fields
*)

(* existence test for type trees --- maintained by K *)
let rec texists f t =
  match t with
  | TTyCon0 _ -> f t
  | TTyCon1(_,tt) -> f t || texists f tt
  | TTup(ts)     -> List.exists (texists f) ts
  | TSum(ts)     -> List.exists (texists f) ts
  | TFunc(t1,t2) -> f t || texists f t1 || texists f t2
  | TVar(_)
  | TXDot(_,_)
  | TTyName(_) -> f t
  | TForall(_,tt) -> f t || texists f tt
  | TExists(_,tt) -> f t || texists f tt

let rec flatten_context : eval_ctx -> (eval_ctx->eval_ctx) list
    = fun ec ->
      match ec with
	CtxTop -> []
      |	CtxC1(c, ec') ->              (function ec ->  CtxC1(c, ec)             ) :: flatten_context ec'
      |	CtxConsL (ec', e2) ->         (function ec ->  CtxConsL(ec,e2)          ) :: flatten_context ec'
      |	CtxConsR (e1, ec') ->         (function ec ->  CtxConsR(e1,ec)          ) :: flatten_context ec'
      |	CtxTup (es, ec', vs) ->       (function ec ->  CtxTup(es,ec,vs)         ) :: flatten_context ec'
      | CtxInEnv (env, ec') ->        (function ec ->  CtxInEnv(env, ec)        ) :: flatten_context ec'
      | CtxIf (ec', e2, e3) ->        (function ec ->  CtxIf (ec, e2, e3)       ) :: flatten_context ec'
      |	CtxSeq (ec', e2) ->           (function ec ->  CtxSeq (ec, e2)          ) :: flatten_context ec'
      |	CtxAppL (ec', e2) ->          (function ec ->  CtxAppL (ec, e2)         ) :: flatten_context ec'
      |	CtxAppR (v1, ec') ->       (function ec ->  CtxAppR (v1, ec)      ) :: flatten_context ec'
      | CtxTApp (ec', t) ->           (function ec ->  CtxTApp (ec, t)          ) :: flatten_context ec'
      | CtxUnpack (itn, iid, ec', e) ->
                                      (function ec ->  CtxUnpack (itn,iid,ec,e) ) :: flatten_context ec'
      | CtxNamecaseL (ec', t, x1, x2, e, e2, e3) ->
                                      (function ec ->  CtxNamecaseL (ec, t, x1, x2, e, e2, e3) ) :: flatten_context ec'
      | CtxNamecaseR (v, t, x1, x2, ec', e2, e3) ->
                                      (function ec ->  CtxNamecaseR (v, t, x1, x2, ec, e2, e3) ) :: flatten_context ec'
      | CtxMatch (ec', m) ->          (function ec ->  CtxMatch (ec, m)         ) :: flatten_context ec'
      |	CtxRaise (ec') ->             (function ec ->  CtxRaise (ec)            ) :: flatten_context ec'
      |	CtxTry (ec', m) ->            (function ec ->  CtxTry (ec, m)           ) :: flatten_context ec'
      |	CtxMarshalL (ec', e, t) ->    (function ec ->  CtxMarshalL (ec, e, t)   ) :: flatten_context ec'
      |	CtxMarshalR (mk, ec', t) ->   (function ec ->  CtxMarshalR (mk, ec, t)  ) :: flatten_context ec'
      |	CtxMarshalz (eqs, mk, ec', t) ->
                                      (function ec ->  CtxMarshalz (eqs, mk, ec, t)) :: flatten_context ec'
      |	CtxUnmarshal(ec', t) ->       (function ec ->  CtxUnmarshal(ec, t)      ) :: flatten_context ec'
      |	CtxCol (ec', eqs', t) ->      (function ec ->  CtxCol (ec, eqs', t)     ) :: flatten_context ec'
      | CtxOp (oe, es, ec', vs) ->    (function ec ->  CtxOp (oe, es, ec, vs)   ) :: flatten_context ec'
      | CtxLazyOp (lo, ec', e') ->    (function ec ->  CtxLazyOp (lo, ec, e')   ) :: flatten_context ec'
      |	CtxOP (eqs, n, oe, es, ec', vs) -> (function ec ->  CtxOP (eqs, n, oe, es, ec, vs)) :: flatten_context ec'
      | CtxHashTs (eqs, t1, ec', t2) -> (function ec ->  CtxHashTs (eqs, t1, ec, t2) ) :: flatten_context ec'
      | CtxHashHtsL (eqs, t1, ec', e1, t2) ->
                                      (function ec ->  CtxHashHtsL (eqs, t1, ec, e1, t2) ) :: flatten_context ec'
      | CtxHashHtsR (eqs, t1, v2, ec', t2) ->
                                      (function ec ->  CtxHashHtsR (eqs, t2, v2, ec, t2) ) :: flatten_context ec'
      | CtxSwapL (ec', e2, e3) ->     (function ec ->  CtxSwapL (ec, e2, e3)    ) :: flatten_context ec'
      | CtxSwapM (v1, ec', e3) ->     (function ec ->  CtxSwapM (v1, ec, e3)    ) :: flatten_context ec'
      | CtxSwapR (v1, v2, ec') ->     (function ec ->  CtxSwapR (v1, v2, ec)    ) :: flatten_context ec'
      | CtxFreshforL (ec', e2) ->     (function ec ->  CtxFreshforL (ec, e2)    ) :: flatten_context ec'
      | CtxFreshforR (v1, ec') ->     (function ec ->  CtxFreshforR (v1, ec)    ) :: flatten_context ec'
      | CtxSupport (t, ec') ->        (function ec ->  CtxSupport (t, ec)       ) :: flatten_context ec'
      | CtxNameOfTie (ec') ->         (function ec ->  CtxNameOfTie (ec)        ) :: flatten_context ec'
      | CtxValOfTie (ec') ->          (function ec ->  CtxValOfTie (ec)         ) :: flatten_context ec'
      | CtxPack (t1, ec', t2) ->      (function ec ->  CtxPack (t1,ec,t2)       ) :: flatten_context ec'
      | CtxStrVal (id, ec', loc) ->         (function ec ->  CtxStrVal (id, ec, loc)        ) :: flatten_context ec'
      | CtxImodule (eqs, mn, mbody, ec') ->          (function ec ->  CtxImodule (eqs, mn, mbody, ec)         ) :: flatten_context ec'


let unflatten_context :(eval_ctx->eval_ctx) list -> eval_ctx
    = fun ecs -> List.fold_right (fun f x -> f x) ecs  CtxTop



(* convert a location to an expression *)
let expr_of_loc l =
  let (filename, lineno, char) = Location.get_pos_info l.Location.loc_start
      in
  Tup [C0(B0(String filename)); C0(B0(Int lineno)); C0(B0(Int char))]


(* ----------------------------- *)
(*  Generate decent fresh names  *)
(* ----------------------------- *)

let global_seed = ref None


(* seed the generator with our IP and this runtime's initialisation time *)
let seed = fun () -> global_seed := Some (
  let ip = try Sys.getenv("IP") with Not_found ->
    "127.0.0.1"
    (* raise (Failure "Must supply an IP environment variable") *)
  in
  let time = try Unix.gettimeofday () with e ->
    raise (Failure ("Couldn't get time:" ^ Printexc.to_string e))
  in
  let pid = try Unix.getpid () with e ->
    raise (Failure ("Couldn't get pid:" ^ Printexc.to_string e))
  in
  ip ^ " " ^ string_of_float time ^ " " ^ string_of_int pid ^ " ")

(* TODO: maybe add in a byte or two of /dev/random, but (i) that's
   Linux-specific and Francesco will be unhappy; and (ii) that's a bit
   wasteful of a limited entropy resource - one day someone will run
   Acute on a server (w/o mouse etc as entropy sources) and it will
   block on startup after a few starts. *)

(* keep a counter to index the sequence of fresh names *)
let freshcount = ref 0

(* generate a new one by hashing the seed and counter, and
   incrementing the counter *)
let new_abstract_name () =
  let h = Digest.string ((the !global_seed) ^ string_of_int (!freshcount))
  in
  (freshcount := !freshcount + 1);
  h


(* now use it: *)

(* add a new name_value to the map *)
let add_name_value : nameenv -> name_value -> nameenv
    = fun ne nv ->
      { ne with ne_list  = NEnv_tname(abstract_name_of_name_value nv, type_of_name_value nv)::ne.ne_list }

(* generate a new name_value of given type (i.e., T gives a new T name) *)
let new_name_value : nameenv -> typ -> nameenv * name_value
    = fun ne t ->
      let nv = VHashName(new_abstract_name(), t) in
      (add_name_value ne nv, nv)


(* FZ a better place for these? *)

let defs_below_mark defs =
  let rec defs_below_mark defs =
    match defs with [] -> None
    | (Mark_compile _)::defs' -> Some defs'
    | d::defs -> defs_below_mark defs
  in match defs_below_mark defs with
    None -> defs
  | Some ds -> ds


let mode_of_source_def s =
  match s with
    | Mod_user (_,m,_) -> m
    | Import_user (_,m,_) -> m
    | Mod_alias _ | Mark_user _ -> raise (Never_happen "no mode in mod_alias or mark_user")

let sdef_to_fresh_def : source_definition -> definition
  = fun d ->
  match d with
    | Mod_user (m,MFresh,b) -> Mod_fresh (m,b)
    | Import_user (m,MFresh,b) -> Import_fresh (m,b)
    | _ -> raise (Never_happen "sdef_to_fresh_def")

