/*
This is a parser implemented in BCPL based 
the Full Grammar Specification of python taken from Chater 10
of a python manual. This chapter says it was derived directly from the
grammar used to generate the CPthon Parser. It is essentially an
extended BNF specification with a few modifications allowing it to be
a specification of a recursive descent parser. Some of these
modifications are as follows.

Alternatives are tested in the order in which they appear.
( e )  matches e
e*     matches zero or more occurrences of e
e+     matches one or more occurrences of e
[ e ]  matches zero or one occurrences of e
e?     matches zero or one occurrences of e
s.e+   matches a sequence of es seperated by ses

&e     is successful if e can be matched, but does not advance the input
!e     is successful if e cannot be matched, but does not advance the input
~      disallows backtracking at this point

This file is a handwritten Python parsed based on this
specication. This parser creates a parse tree with an option to output the
tree in a readable form.

Implemented in BCPL by Martin Richards (c) 30 Jan 2025.


********************* The Rules *****************************
*/

GET "libhdr"
 
MANIFEST {  // Lexical tokens, parse tree operators and op-codes

Num=1; Name; String; True; False
Valof; Fnap; Lv; Ind; Vecap
Neg; Not; Mul; Div; Mod; Add; Sub
Eq; Ne; Le; Ge; Lt; Gt; Lsh; Rsh; And; Or; Xor
Comma; Fndef; Rtdef; Assign; Rtap; Resultis
Test; If; Unless; While; Until; For; Return; Seq
Let; Vec; Static; Statvec; Decl; Var
Lparen; Rparen; Lsquare; Rsquare; Lcurly; Rcurly
To; Do; Then; Else; Be; Eof; Semicolon
Rtrn; Fnrn; Addr; Local; Lab; Data; Jt; Jf; Jump;
Ln; Lp; Llp; Ll; Laddr; Sp; Sl; Stind; Lres

s_seq
s_statement
}
 
GLOBAL { 
rec_p:ug; rec_l; fin_p; fin_l
fatalerr; synerr; trnerr; errcount; errmax
progstream; tostream
mk1; mk2; mk3; mk4; mk5; mk6
newvec; treep; treet; treevec
optTokens; optTree; optCode; optTrace

// Globals used in LEX
chbuf; charv; ch; rch; lex; token; lexval; wordnode
wrchbuf; chcount; lineno
dsw; declsyswords; namestart; nametable; lookupword
rdstrch; rdtag
formtree
plist
opstr
inp
treep
statements
}

MANIFEST {                         //  Selectors
h1=0; h2=1; h3=2; h4=3; h5=4; h6=5
nametablesize = 541
c_tab         =   9
c_newline     =  10
}
 
LET lex() BE
{ SWITCHON ch INTO
  { CASE '*p': CASE '*n':
                 lineno := lineno + 1
    CASE '*c': CASE '*t': CASE '*s':
                 rch()
                 LOOP

    CASE '0':CASE '1':CASE '2':CASE '3':CASE '4':
    CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
                lexval := 0
                WHILE '0'<=ch<='9' DO
                { lexval := 10*lexval + ch - '0'
                  rch()
                }
                token := Num
                RETURN

    CASE '#':   lexval := 0  // Added 9/08/2021
                { rch()
		  IF  '0'<=ch<='9' DO
                  { lexval := (lexval<<4) + ch - '0'
		    LOOP
                  }
		  IF  'A'<=ch<='F' DO
                  { lexval := (lexval<<4) + ch - 'A' + 10
		    LOOP
                  }
		  IF  'a'<=ch<='f' DO
                  { lexval := (lexval<<4) + ch - 'a' + 10
		    LOOP
                  }
                  token := Num
                  RETURN
		} REPEAT
    CASE 'a':CASE 'b':CASE 'c':CASE 'd':CASE 'e':
    CASE 'f':CASE 'g':CASE 'h':CASE 'i':CASE 'j':
    CASE 'k':CASE 'l':CASE 'm':CASE 'n':CASE 'o':
    CASE 'p':CASE 'q':CASE 'r':CASE 's':CASE 't':
    CASE 'u':CASE 'v':CASE 'w':CASE 'x':CASE 'y':
    CASE 'z':
    CASE 'A':CASE 'B':CASE 'C':CASE 'D':CASE 'E':
    CASE 'F':CASE 'G':CASE 'H':CASE 'I':CASE 'J':
    CASE 'K':CASE 'L':CASE 'M':CASE 'N':CASE 'O':
    CASE 'P':CASE 'Q':CASE 'R':CASE 'S':CASE 'T':
    CASE 'U':CASE 'V':CASE 'W':CASE 'X':CASE 'Y':
    CASE 'Z':
                token := lookupword(rdtag())
                RETURN
 
    CASE '{': token := Lcurly;    BREAK
    CASE '}': token := Rcurly;    BREAK
    CASE '[': token := Lsquare;   BREAK
    CASE ']': token := Rsquare;   BREAK
    CASE '(': token := Lparen;    BREAK
    CASE ')': token := Rparen;    BREAK 
    CASE '!': token := Ind;       BREAK
    CASE '@': token := Lv;        BREAK
    CASE '+': token := Add;       BREAK
    CASE '-': token := Sub;       BREAK
    CASE ',': token := Comma;     BREAK
    CASE ';': token := Semicolon; BREAK
    CASE '&': token := And;       BREAK
    CASE '|': token := Or;        BREAK
    CASE '=': token := Eq;        BREAK
    CASE '**':token := Mul;       BREAK
    CASE '^': token := Xor;       BREAK
 
    CASE '/':   rch()
                IF ch='/' DO
                { rch() REPEATUNTIL ch='*n' | ch=endstreamch
                  LOOP
                }
                token := Div
                RETURN
 
    CASE '~':   rch()
                IF ch='=' DO { token := Ne;  BREAK }
                token := Not
                RETURN
 
    CASE '<':   rch()
                IF ch='=' DO { token := Le;  BREAK }
                IF ch='<' DO { token := Lsh; BREAK }
                token := Lt
                RETURN
 
    CASE '>':   rch()
                IF ch='=' DO { token := Ge;  BREAK }
                IF ch='>' DO { token := Rsh; BREAK }
                token := Gt
                RETURN
 
    CASE ':':   rch()
                IF ch='=' DO { token := Assign;  BREAK }
                synerr("'=' expected after ':'")
                RETURN
 
    CASE '"':
              { LET len = 0
                rch()
 
                UNTIL ch='"' DO
                { IF len=255 DO synerr("Bad string constant")
                  len := len + 1
                  charv%len := rdstrch()
                }
 
                charv%0 := len
                wordnode := newvec(len/bytesperword+2)
                h1!wordnode := String
                FOR i = 0 TO len DO (@h2!wordnode)%i := charv%i
                token := String
                BREAK
              }
 
    CASE '*'':  rch()
                lexval := rdstrch()
                token := Num
                UNLESS ch='*'' DO synerr("Bad character constant")
                BREAK

    DEFAULT:    UNLESS ch=endstreamch DO
                { LET badch = ch
                  ch := '*s'
                  synerr("Illegal character %x2", badch)
                }
                token := Eof
                RETURN
  } REPEAT
 
  rch()
}
 
LET lookupword(word) = VALOF
{ LET len, i = word%0, 0
  LET hashval = len
  FOR i = 1 TO len DO hashval := (13*hashval + word%i) & #xFF_FFFF
  hashval := hashval REM nametablesize
  wordnode := nametable!hashval
 
  WHILE wordnode & i<=len TEST (@h3!wordnode)%i=word%i
                          THEN i := i+1
                          ELSE wordnode, i := h2!wordnode, 0
  IF wordnode=0 DO
  { wordnode := newvec(len/bytesperword+3)
    h1!wordnode, h2!wordnode := Name, nametable!hashval
    FOR i = 0 TO len DO (@h3!wordnode)%i := word%i
    nametable!hashval := wordnode
  }
  RESULTIS h1!wordnode
}
 
AND dsw(word, tok) BE { lookupword(word); h1!wordnode := tok  }
 
AND declsyswords() BE
{ dsw("be", Be);             dsw("do", Do);         dsw("else", Else)
  dsw("false", False);       dsw("if", If);         dsw("for", For)
  dsw("let", Let);           dsw("mod", Mod);
  dsw("resultis", Resultis); dsw("return", Return); dsw("static", Static)
  dsw("test", Test);     dsw("to", To)
  dsw("true", True);         dsw("then", Then);     dsw("valof", Valof)
  dsw("vec", Vec);           dsw("unless", Unless); dsw("until", Until)
  dsw("while", While)  
  lookupword("start")
  namestart := wordnode
} 
 
LET rch() BE
{ ch := rdch()
  chcount := chcount+1
  chbuf%(chcount&63) := ch
}
 
AND wrchbuf() BE
{ writes("*n...")
  FOR p = chcount-63 TO chcount DO
  { LET k = chbuf%(p&63)
    IF 0<k<255 DO wrch(k)
  }
  newline()
}
 
AND rdtag() = VALOF
{ LET len = 0
  WHILE 'a'<=ch<='z' | 'A'<=ch<='Z' | '0'<=ch<='9' |  ch='_' DO
  { len := len+1
    IF len>255 DO synerr("Name too long")
    charv%len := ch
    rch()
  }
  charv%0 := len
  RESULTIS charv
}
 
AND rdstrch() = VALOF
{ LET res = ch
  IF ch='*n' | ch='*p' DO
  { lineno := lineno+1
    synerr("Unescaped newline character")
  }
  IF ch='\' DO
  { rch()
    SWITCHON ch INTO
    { DEFAULT:   synerr("Bad string or character constant")
      CASE '\': CASE '*'': CASE '"':  res := ch;        ENDCASE
      CASE 't': CASE 'T':             res := c_tab;     ENDCASE
      CASE 'n': CASE 'N':             res := c_newline; ENDCASE
    }
  }
  rch()
  RESULTIS res
}

LET newvec(n) = VALOF
{ abort(7654)
  treep := treep - n - 1;
  IF treep<=treevec DO fatalerr("More workspace needed")
  RESULTIS treep
}
 
AND mk1(a) = VALOF
{ LET p = newvec(0)
  p!0 := a
  RESULTIS p
}
 
AND mk2(a, b) = VALOF
{ LET p = newvec(1)
  p!0, p!1 := a, b
  RESULTIS p
}
 
AND mk3(a, b, c) = VALOF
{ LET p = newvec(2)
  p!0, p!1, p!2 := a, b, c
  RESULTIS p
}
 
AND mk4(a, b, c, d) = VALOF
{ LET p = newvec(3)
  p!0, p!1, p!2, p!3 := a, b, c, d
  RESULTIS p
}
 
AND mk5(a, b, c, d, e) = VALOF
{ LET p = newvec(4)
  p!0, p!1, p!2, p!3, p!4 := a, b, c, d, e
  RESULTIS p
}
 
AND mk6(a, b, c, d, e, f) = VALOF
{ LET p = newvec(5)
  p!0, p!1, p!2, p!3, p!4, p!5 := a, b, c, d, e, f
  RESULTIS p
}
 
AND formtree() = VALOF
{ LET res = 0
  rec_p, rec_l := level(), recover

  charv := newvec(256/bytesperword)     
  nametable := newvec(nametablesize)
  UNLESS charv & nametable DO fatalerr("More workspace needed")
  FOR i = 0 TO nametablesize DO nametable!i := 0
  declsyswords()
  lex()

  IF optTokens DO            // For debugging lex.
  { IF token=Eof RESULTIS 0
    writef("token = %i3 %s", token, opstr(token))
    IF token=Num    DO writef("       %n",  lexval)
    IF token=Name   DO writef("      %s",   charv)
    IF token=String DO writef("    *"%s*"", charv)
    newline()
    lex()
  } REPEAT

recover:
  res := statements()
  UNLESS token=Eof DO fatalerr("Incorrect termination")
  RESULTIS res
}
 
AND fatalerr(mess, a) BE
{ writef("*nFatal error:  ")
  writef(mess, a)
  writes("*nCompilation aborted*n")
  errcount := errcount+1
  longjump(fin_p, fin_l)
}

AND synerr(mess, a) BE
{ writef("*nError near line %n:  ", lineno)
  writef(mess, a)
  wrchbuf()
  errcount := errcount+1
  IF errcount >= errmax DO fatalerr("Too many errors")

  // Skip the rest of the input line 
  UNTIL ch='*n' | ch=endstreamch DO rch()
  lex()

  longjump(rec_p, rec_l)
}

LET checkfor(tok, mess) BE
{ UNLESS token=tok DO synerr(mess)
  lex()
}
 
LET opstr(op) = VALOF SWITCHON op INTO
{ DEFAULT:       RESULTIS "Unknown"

  CASE Assign:   RESULTIS "Assign";    CASE Add:     RESULTIS "Add"
  CASE And:      RESULTIS "And";       CASE Be:      RESULTIS "Be"
  CASE Comma:    RESULTIS "Comma";     CASE Data:    RESULTIS "Data"
  CASE Decl:     RESULTIS "Decl";      CASE Div:     RESULTIS "Div"
  CASE Do:       RESULTIS "Do";        CASE Else:    RESULTIS "Else"
  CASE Eq:      RESULTIS "Eq"
  CASE False:    RESULTIS "False";     CASE Fnap:    RESULTIS "Fnap"
  CASE For:      RESULTIS "For";       CASE Fndef:   RESULTIS "Fndef"
  CASE Fnrn:     RESULTIS "Fnrn";      CASE Ge:      RESULTIS "Ge"
  CASE Gt:       RESULTIS "Gt";
  CASE If:       RESULTIS "If";        CASE Ind:     RESULTIS "Ind"
  CASE Jf:       RESULTIS "Jf";        CASE Jt:      RESULTIS "Jt"
  CASE Jump:     RESULTIS "Jump";      CASE Lab:     RESULTIS "Lab"
  CASE Laddr:    RESULTIS "Laddr";     CASE Lcurly:  RESULTIS "Lcurly"
  CASE Le:       RESULTIS "Le";        CASE Let:     RESULTIS "Let"
  CASE Ll:       RESULTIS "Ll";        CASE Llp:     RESULTIS "Llp"
  CASE Ln:       RESULTIS "Ln";        CASE Lp:      RESULTIS "Lp"
  CASE Lparen:   RESULTIS "Lparen";    CASE Lres:    RESULTIS "Lres"
  CASE Lsh:      RESULTIS "Lsh";       CASE Lsquare: RESULTIS "Lsquare"
  CASE Lt:       RESULTIS "Lt";        CASE Lv:      RESULTIS "Lv"
  CASE Mod:      RESULTIS "Mod";       CASE Mul:     RESULTIS "Mul"
  CASE Name:     RESULTIS "Name";      CASE Ne:      RESULTIS "Ne"
  CASE Neg:      RESULTIS "Neg";       CASE Not:     RESULTIS "Not"
  CASE Num:      RESULTIS "Num";       CASE Or:      RESULTIS "Or"       
  CASE Rcurly:  RESULTIS "Rcurly"
  CASE Resultis: RESULTIS "Resultis";  CASE Return:  RESULTIS "Return"
  CASE Rparen:   RESULTIS "Rparen";    CASE Rsh:     RESULTIS "Rsh"
  CASE Rsquare:  RESULTIS "Rquare";    CASE Rtap:    RESULTIS "Rtap"
  CASE Rtdef:    RESULTIS "Rtdef";     CASE Rtrn:    RESULTIS "Rtrn"
  CASE Semicolon:RESULTIS "Semicolon"; CASE Seq:     RESULTIS "Seq"
  CASE Sl:       RESULTIS "Sl";        CASE Sp:      RESULTIS "Sp"
  CASE Static:  RESULTIS "Static"
  CASE Statvec:  RESULTIS "Statvec";   CASE String:  RESULTIS "String"
  CASE Stind:    RESULTIS "Stind";     CASE Sub:     RESULTIS "Sub"
  CASE Test:     RESULTIS "Test"
  CASE Then:     RESULTIS "Then";      CASE To:      RESULTIS "To"
  CASE True:     RESULTIS "True";      CASE Valof:   RESULTIS "Valof"
  CASE Vecap:    RESULTIS "Vecap";     CASE Vec:     RESULTIS "Vec"
  CASE Unless:   RESULTIS "Unless";    CASE Until:   RESULTIS "Until"
  CASE While:    RESULTIS "While";     CASE Xor:     RESULTIS "Xor"
}

LET plist(x, n, d) BE
{ writef("%i4: *n", x)
  plist1(x, n, d)
}

AND plist1(x, n, d) BE
{ LET s, size, ln = 0, 0, 0
  LET v = TABLE 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0

  IF x=0 DO { writes("Nil"); RETURN  }
 
  SWITCHON h1!x INTO
  { DEFAULT:
         size     := 1;        ENDCASE

    CASE Num:     writen(h2!x);         RETURN
    CASE Name:    writes(x+2);          RETURN
    CASE String:  writef("*"%s*"",x+1); RETURN

    CASE For:
         size, ln := 5, h6!x; ENDCASE

    CASE Fndef: CASE Rtdef:
         size, ln := 4, h5!x; ENDCASE

    CASE Let: CASE Vec: CASE Test:
         size, ln := 4, h5!x; ENDCASE

    CASE Vecap: CASE Mul: CASE Div: CASE Mod: CASE Add: CASE Sub:
    CASE Eq: CASE Ne: CASE Lt: CASE Gt: CASE Le: CASE Ge:
    CASE Lsh: CASE Rsh: CASE And: CASE Or: CASE Xor:
    CASE Comma: CASE Seq: CASE Decl: CASE Statvec:
         size     := 3;       ENDCASE

    CASE Assign: CASE Rtap: CASE Fnap:
    CASE If: CASE Unless: CASE While: CASE Until:
         size, ln := 3, h4!x; ENDCASE

    CASE Valof: CASE Lv: CASE Ind: CASE Neg: CASE Not:
         size     := 2;       ENDCASE

    CASE Static: CASE Resultis:
         size, ln := 2, h3!x; ENDCASE

    CASE True: CASE False:
         size     := 1;       ENDCASE
  }
 
  IF n=d DO { writes("Etc"); RETURN }
  writef("%s", opstr(h1!x))
  IF ln DO writef("  -- line %n", ln)
  FOR i = 2 TO size DO { newline()
                         FOR j=0 TO n-1 DO writes( v!j )
                         writes("**-")
                         v!n := i=size->"  ","! "
                         plist(h1!(x+i-1), n+1, d)
                       }
}


/*
statements: statement+
*/

LET statements() = VALOF
{ LET previnp, prevtreep = inp, treep
  LET a = statement()
  UNLESS a GOTO fail
  
  { LET b = statement()
    UNLESS b RESULTIS a
    a := mk3(s_seq, a, b)
  } REPEAT
  
fail:
  inp, treep := previnp, prevtreep
  RESULTIS 0
}

/*
statement: compound_stmt | simple_stmts
*/

AND statement() = VALOF
{ LET previnp, prevtreep = inp, treep
  RESULTIS mk1(s_statement)
  
fail:
  inp, treep := previnp, prevtreep
  RESULTIS 0
}




/*
statement_newline:
  | compound_stment NEWLINE
  | simple_stmts
  | NEWLINE
  | ENDMARKER

simple_stmts:
  | simple_stmt !';' NEWLINE
  | ';'.simple_stmt+ [';'] NEWLINE

simple_stmt:
  | assignment
  | type_alias
  | star_expressions
  | return_stmt
  | import_stmt
  | raise_stmt
  | 'pass'
  | del_stmt
  | yield_stmt
  | assert_stmt
  | 'break'
  | 'continue'
  | global_stmt
  | nonlocal_statement

assignment:
  | NAME ':' expression ['=' annotation_rhs']
  | ('(' single_target /)'
     | single_subscript_attribute_target) ':' expression ['=' annotated_rhs]
  | (start_targets '=' )+ (yield_expression | star_expressions) !'=' [TYPE_COMMENT]
  | single_target augassign ~ (yield_expr | star_expressions)

annotated_rhs: yield_expr | star_expressions

augassign:
  | '+=' | '-=' | '*=' | '@=' | '/=' | '%=' | '&=' | '|=' | '^='
  | '<<=' | '>>=' | '**=' | '//='

return_stmt:
| 'return' [star_expressions]

raise_stmt:
  | 'raise' expression ['from' expression ]
  | 'raise'

global_stmt:
  | 'global' ','.NAME+

nonlocal_stmt:
  | 'nonlocal' ','.NAME+

del_stmt:
  | 'del' del_targets &(';' | NEWLINE}

yield_stmt:
  | yield_expr

assert_stmt:
  | 'assert' expression [',' expression]

import_stmt:
  | import_name
  | import_from

import_name:
  | 'import' dotted_as_names

import_from:
  | 'from' ('.' | '...')* dotted_name 'import' import_from_targets
  | 'from' ('.' | '...')* 'import' import_from_targets

import_from_targets:
  | '(' import_from_as_names [','] ')'
  | import_from_as_names !','
  | '*'

import_from_as_names
  | ','.import_from_as_name+

import_from_as_name:
  | NAME ['as' NAME]

dotted_as_names:
  | ','.import_as_name+

dotted_as_name:
  | dotted_as_name ',' NAME
  | NAME

block:
  | NEWLINE INDENT statements DEDENT
  | simple_stmts

decorators:
  | ('@' named_expression NEWLINE)+

class_def:
  | decorators class_def_raw
  | class_def_raw

class_def_raw:
  | 'class' NAME [type_param] ['(' [arguments] ')'] ':' block

function_def:
  | decorators function_def_raw
  | function_def_raw

function_def_raw:
  | 'def' NAME [type_params] '(' [params] ')' ['->' expression] ':' [func_type_comment] block
  | 'async' 'def' NAME [type_params] '(' [params] ')' ['->' expression] ':' [func_type_comment] block

params:
  | parameters

parameters:
  | slash_no_default param_no_default* param_with_default* [star_etc]
  | slash_with_default param_with_default* [star_etc]
  | slash_no_default+ param_with_default* [star_etc]
  | param_with_default+ [star_etc]
  | star_etc

slash_no_default:
  | param_no_default+ '/' ','
  | param_no_default+ '/' &')'

slah_with_default:
  | param_no_default+ param_with_default+ '/' ','
  | param_no_default+ param_with_default+ '/' &')'

star_etc:
  | '*' param_no_default param_with_default* [kwds]
  | '*' param_no_default_star_annotation param_maybe_default* [kwds]
  | '*' ',' param_maybe_default* [kwds]
  | kwds

kwds:
  | '**' param_nodefault

param_nodefault:
  | param ',' TYPE_COMMENT?
  | param TYPE_COMMENR? &')'

param_no_default_star_annotation:
  | param_star_annotation ',' TYPE_COMMENT?
  | param_star_annotation TYPE_COMMENT? &')'

param_with_default:
  | param default ',' TYPE_COMMENT?
  | param default TYPE_COMMENR? &')'

param_maybe_default:
  | param default? ',' TYPE_COMMENT?
  | param default? TYPE_COMMENR? &')'

param:
  | NAME annotation?

param_star_annotation:
  | NAME startannotation

annotation:
  | ':' expression

star_annotation:
  | ':' start_expression

default:
  | '=' expression
  | invalid_default

if_stmt:
  | 'if' named_expression ':' block elif_stmt
  | 'if' named_expression ':' [block [else_block]

elif_stmt:
  | 'elif' named_expression ':' block elif_stmt
  | 'elif' named_expression ':' block [else_block]
  
else_block:
  | 'else' ';' block

while_stmt:
  | 'while' named_expression ':' block [else_block]

for_stmt:
  | 'for' star_targets 'in' ~ star_expressions ':' [TYPE_COMMENT] block [else_block]
  | async 'for' star_targets 'in' ~ star_expressions ':' [TYPE_COMMENT] block [else_block]

with_stmt:
  | 'with' '(' ','.with_item+ ','? ')' ':' [TYPE_COMMENT] block
  | 'with' ','.with_item+ ':' [TYPE_COMMENT] block
  | 'async' 'with' '(' ','.with_item+ ','? ')' ':' block 
  | 'async' 'with' ','.with_item+ ':' [TYPE_COMMENT] block 

with_item:
  | expression 'as' star_star_target &(',' | ')' | ':')
  | expression

try_stmt:
  | 'try' ':' block finally_block
  | 'try' ':' block except_block+ [else_block] [finally_block]
  | 'try' ':' block except_star_block+ [else_block] [finally_block]

except_block:
  | 'except' expression ['as' NAME] ':' block
  | 'except' ':' block

except_star_block:
  | 'except' '*' expression ['as' NAME] ':' block

finally_block:
  | 'finally' ':' block

match_stmt:
  | "match" subject_expr ':' NEWLINE INDENT case_block+ DEDENT

subject_expr:
  | star_named_expression ',' start_named_expressions?
  | named_expression

case_block:
  | "case" patterns guard? : block

guard:
  | 'f' named-expression

patterns:
  | open_sequence_pattern
  | pattern

pattern:
  | as_pattern
  | or_pattern

as_pattern:
  | or_pattern 'as' pattern_capture_target

or_pattern:
  | '|'.closed_pattern+

closed_pattern:
  | literal_pattern
  | capture_pattern
  | whildcard_pattern
  | value_pattern
  | group_pattern
  | sequence_pattern
  | mapping_pattern
  | class_pattern

literal_pattern:
  | signed_number !('+' | '-')
  | complex_number
  | strings
  | 'None'
  | 'True'
  | 'False'

literal_expr:
  | signed_number !('+' | '-')
  | complex_number
  | strings
  | 'None'
  | 'True'
  | 'False'

complex_number:
  | signed_real_number '+' imaginary_number
  | signed_real_number '-' imaginary_number

signed_number:
  | NUMBER
  | '-' NUMBER

real_number:
  | NUMBER

imaginary_number:
  | NUMBER

capture_pattern:
  | pattern_capture_target

pattern_capture_target:
  | !'_' NAME !(',' | '(' | '=')

whildcard_pattern:
  | '_'

value_pattern:
  | attr !('.' | ' (' | '=')

attr:
  | name_or_attr '.'NAME

name_or_attr:
  | attr
  | NAME

group_pattern:
  | '(' pattern ')'

sequence_pattern:
  | '[' maybe_sequence_pattern? ']'
  | '[' open_sequence_pattern? ']'

open_sequence_pattern:
  | maybe_star_pattern ',; maybe_sequence_pattern?

maybe_sequence_pattern:
  | ','.maybe_star_pattern+ ','?

maybe_star_pattern:
  | star_pattern
  | pattern

star_pattern:
  | '*' pattern_capture_target
  | '*' wildcard_pattern

mapping_pattern:
  | '{' '}'
  | '{' double_star_pattern ','? '}'
  | '{' items_pattern ',; double_star_pattern ','? '}'
  | '{' items_pattern ','? '}'

items_pattern:
  | ','.key_value_pattern+

key_value_pattern:
  | (literal_expr | attr) ':' pattern

double_star_pattern:
  | '**' pattern_capture_target

class_pattern:
  | name_or_attr '(' ')'
  | name_or_attr '(' positional_patterns ','? ')'
  | name_or_attr '(' keyword_patterns ','? ')'
  | name_or_attr '(' positional_patterns ','keyword_patterns ','? ')'

positional_patterns:
  | ','.pattern+

keyword_patterns:
  | ','.keyword_pattern+

keyword_pattern:
  | NAME '=' pattern

type_alias:
  | "type" NAME [type_params] '=' expression

type_params:
  | invalid-type_params
  | '[' type_params_seq ']'

type_param-seq:
  | ','.type_para+ [',']

type_param:
  | NAME [type_param_bound] [type_param_default]
  | '*' NAME [type_param_starred_default]
  | '**' NAME [type param_default]

type_param_bound:
  | ':' expression

type_param_default:
  | '=' expression

type_param_starred_default:
  | '=' star_expression

expressions:
  | expression (',' expression)+ [',']
  | expression ','
  | expression

expression:
  | disjunction 'if' disjunction 'else' expression
  | disjunction
  | lambdef

yield_expr:
  | 'yield' 'from' expression
  | 'yield' [star_expression]

star_expressions:
  | star_expression (',' start_expression )+ [',']
  | star_expression ','
  | star_expression

star_expression:
  | '*' bitwise_or
  | expression

star_named_expressions:
  | ','.star_named_expression+ [',']

star_named_expression:
  | '*' bitwise_or
  | name_expression

assignment_expression:
  | NAME ':=' ~ expression

named_expression:
  | assignment_expression
  | expression !':='

disjuction:
  | conjunction ('or' conjunction)+
  | conjunction

conjunction:
  | inversion ('and' inversion)+
  | inversion

inversion:
  | 'not' inversion
  | comparison

comparison:
  | bitwise_or compare_op_bitwase_or_pair+
  | bitwise_or

compare_op_bitwise_or_pair:
  | eq_bitwise_or
  | noteq_bitwise_or
  | lte_bitwise_or
  | lt_bitwise_or
  | gte_bitwise_or
  | gt_bitwise_or
  | notin_bitwise_or
  | in_bitwise_or
  | isnot_bitwise_or
  | is_bitwise_or

eq_bitwise_or:
  | '==' bitwise_or

noteq_bitwise_or:
  | ( '!=' ) bitwise_or

lte_bitwise_or:
  | '<=' bitwise_or

lt_bitwise_or:
  | '<' bitwise_or

gte_bitwise_or:
  | '>=' bitwise_or

gt_bitwise_or:
  | '>' bitwise_or

notin_bitwise_or:
  | 'not' 'in' bitwise_or

in_bitwise_or:
  | 'in' bitwise_or

isnot_bitwise_or:
  | 'is' 'not' bitwise_or

is_bitwise_or:
  | 'is' bitwise_or

bitwise_or:
  | bitwise_or '|' bitwise_xor
  | bitwise_xor

bitwise_xor:
  | bitwise_xor '^' bitwise_and
  | bitwise_and

bitwise_and:
  | bitwise_and '^' shift_expr
  | shift_expr

shift_expr:
  | shift_expr '<<' sum
  | shift_expr '>>' sum
  | sum

sum:
  | sum '+' term
  | sum '-' term
  | term

term:
  | term '*' factor
  | term '/' factor
  | term '//' factor
  | term '%' factor
  | term '@' factor
  | factor

factor:
  | '+' factor
  | '-' factor
  | '~' factor
  | power

power:
  | await_primary '**' factor
  | await_primary

await_primary:
  | 'await' primary
  | priamry

primary:
  | primary '.' NAME
  | primary '(' [arguments] ')'
  | primary '[' slices ']'
  | atom

slices:
  | slice !','
  | ','.(slices | starred_expression)+ [',']

slice:
  | [expression] ':' [expression] [':' [expression] ]
  | named_expression

atom:
  | NAME
  | 'True'
  | 'False'
  | 'None'
  | strings
  | NUMBER
  | (tuple | group | genexp)
  | (list | listcomp)
  | (dict | set | dictcomp | setcomp)
  | '...'

group:
  | '(' (yield_expr | named_expression) ')'

lambdef:
| 'lambda' [lambda_params] ':' expression

lambda_params:
  | lambda_parameters

lambda_parameters:
  | lambda_slash_no_default lambda_param_no default* lambda_param_with_default* [lambda_star_etc]
  | lambda_slash_with_default lambda_param_with_default* [lambda_star_etc]
  | lambda_param_no_default* lambda_param_with_default* [lambda_star_etc]
  | lambda_param_with_default* [lambda_star_etc]
  | lambda_star_etc  

lambda_slash_no_default:
  | lambda_param_no_default+ '/' ','
  | lambda_param_no_default+ '/' &':

lambda_slash_with_default:
  | lambda_param_no_default* lambda_param_with_default+ '/' ','
  | lambda_param_no_default* lambda_param_with_default+ '/' &':

lambda_star_etc:
  | '*' lambda_param_no_default lambda_param_maybe_default* [lambda_kwds]
  | '*' ',' lambda_param_maybe_default+ [lambda_kwds]
  | lambda_kwds

lambda_kwds:
  | '**' lambda_param_no_default

lambda_param_no_default:
  | lambda_param ','
  | lambda_param &':'

lambda_param_with_default:
  | lambda_param default ','
  | lambda_param default &':'

lambd_param_maybe_default:
  | lambda_param default? ','
  | lambda_param default? &':'

lambda_param:
  | NAME
  
  
fstring_middle:
  | fstring_replacement_field
  | FSTRING_MIDDLE

fstring_replacement_field:
  | '(' annotated_rhs '=' [fstring_conversion] [fstring_full_format_spec] '}'

fstring_conversion:
  | ':' NAME

fstring_full_format_spec:
  | ':' fstring_format_spec:

fstring_format_spec:
  | FSTRING_MIDDLE
  | fstring_replacement_field

fstring:
  | FSTRING_START fstring_middle* FSTRING_END

string:
  | STRING

strings:
  | (fstring | string)+

list:
  | '[' [star_named_expression] ']'



tuple:
  | '(' [star_named_expression ',' [star_named_expressions] ] ')'

set:
  | '{' star_named_expressions '}'

dict:
  | '{' [double_starred_kvpairs] '}'

double_starred-kvpair:
  | '**' bitwise_or
  | kvpair

kvpair:
  | expression ':' expression

for_if_clauses:
  | for_if_clause+

for_if_clause:
  | 'async' 'for' star_targets 'in' ~ disjunction ('if' disjuntion)*
  | 'for' star_targets 'in' ~ disjunction ('if' disjuntion)*

listcomp:
  | '[' named_expression for_if_clauses ']'

setcomp:
  | '{' named_expression for_if_clauses '}'
  s '}'

genexp:
  | '(' ( assignment_expression | expression !':=') for_if_clauses ')'

dictcomp:
  | '{' kvpair for_if_clauses '}'

arguments:
  | args [','] &')'

args:
  | ','.(starred_expression | ( assignment_expression | expression !':=') !'=')+ [',' kwargs]
  | kwargs

kwargs:
  | ',' kwarg_or starred+ ',' ','.kwarg_or_double_starred+
  | ','.kwarg_or_starred+
  | ','.kwarg_or_double_starred+

starred_expression_:
  | '*' expression

kwarg_or_starred:
  | NAME '=' expression
  | starred_expression

kwarg_or_double_starred:
  | NAME '=' expression
  | '**' expression

star_targets:
  | star_target !','
  | star_target (',' star_target)* [',']

star_targets_list_seq:
  | ','.star_target+ [',']

star_targets_tuple_seq:
  | star_target (',' star_target )+ [','] [',']
  | star_target ','

star_target:
  | '*' (!'*' star_target)
  | target_with_star_atom

target_with_star_atom:
  | t_primary ',' NAME !t_lookahead
  | t_primary '[' slices ']' !t_lookahead
  | star_atom

star_atom:
  | NAME
  | '(' target_with_star_atom ')'
  | '[' [star_targets_tuple_seq] ']'
  | '{' [star_targets_list_seq] '}'

single_target:
  | single_subscript_attribute_target
  | NAME
  | '(' single_target ')'

single_subscript_attribute_target:
  | t_primary ',' NAME !t_lookahead
  | t_primary '[' slices ']' !t_lookahead

t_primary:
  | t_primary ',' NAME &t_lookahead
  | t_primary '[' slices ']' &!t_lookahead
  | t_primary genexp &!t_lookahead
  | t_primary '(' [arguments] ')' &!t_lookahead
  | atom &!t_lookahead

t_lookahead:
  | '('
  | '['
  | '{'

del_targets:
  | ','.deltarget+ [',']

del_target:
  | t_primary ',' NAME !t_lookahead
  | t_primary '[' slices ']' !t_lookahead
  | del_atom

del_atom:
  | NAME
  | '(' del_target ')'
  | '(' [del_targets] ')'
  | '[' [del_targets] ']'

type_expression:
  | ','.expression+ ',' '*' expression ',' '**' expression
  | ','.expression+ ',' '*' expression
  | ','.expression+ ',' '**' expression
  | '*' expression ',' '**' expression
  | '*' expression
  | '**' expression
  | ','.expression+

func_type_comment:
  | NEWLINE TYPE_COMMENT &(NEWLINE INDENT)
  | TYPE_COMMENT

*/


LET start() = VALOF
{ LET treesize = 0
  AND memsize = 0
  AND argv = VEC 50
  AND argform =
        "PROG/A,TO=-o/K,TOKENS=-l/S,TREE=-p/S,TRACE=-t/S"
  LET stdout = output()

  errmax   := 2
  errcount := 0
  
  treevec := getvec(10000)
  treet   := treevec+10000
  treep   := treet
  
  fin_p, fin_l := level(), fin

  progstream, tostream := 0, 0
   
  writef("*nSynpython parser (30 Jan 2025*n")
 
  IF rdargs(argform, argv, 50)=0 DO fatalerr("Bad arguments*n")

  treesize := 10000
  memsize  := 50000

  progstream := findinput(argv!0)      // PROG

  IF progstream=0 DO fatalerr("Trouble with file %s*n", argv!0)

  selectinput(progstream)
 
  IF argv!1                            // TO      -o
  DO { tostream := findoutput(argv!1)
       IF tostream=0 DO fatalerr("Trouble with code file %s*n", argv!1)
     }

  optTokens := argv!2                  // TOKENS  -l
  optTree   := argv!3                  // TREE    -p
  optTrace  := argv!4                  // TRACE   -t

  treevec := getvec(treesize)

  UNLESS treevec DO
    fatalerr("Insufficient memory*n")
   
  UNLESS tostream DO tostream := stdout
  selectoutput(tostream)

  { LET tree = 0
    LET b = VEC 64/bytesperword
    chbuf := b
    FOR i = 0 TO 63 DO chbuf%i := 0
    chcount, lineno := 0, 1
    rch()
 
    treep := treevec + treesize

    tree := formtree()              // Perform Syntax Analysis
    IF optTokens GOTO fin

    IF optTree DO { writes("Parse Tree*n")
                    plist(tree, 0, 20)
                    newline()
                  }
  
    IF errcount GOTO fin

    IF errcount GOTO fin
  }
   
fin:
  IF treevec       DO freevec(treevec)
  IF progstream    DO { selectinput(progstream); endread()  }
  IF tostream      DO { selectoutput(tostream)
                        UNLESS tostream=stdout DO  endwrite() }

  selectoutput(stdout)
  result2 := 0 // No reason given
  RESULTIS errcount=0 -> 0, 20
}

