package lex;

import lib.*;
import err.*;
import syn.*;

public class Lex extends Lib {

    // The Lexical analyser.

    // The lexical tokens

    public final static int Num       =   1;
    public final static int Name      =   2;
    public final static int String    =   3;
    public final static int True      =   4;
    public final static int False     =   5;
    public final static int Valof     =   6;
    public final static int Lv        =   8;
    public final static int Ind       =   9;
    public final static int Vecap     =  10;
    public final static int Neg       =  11;
    public final static int Not       =  12;
    public final static int Mul       =  13;
    public final static int Div       =  14;
    public final static int Mod       =  15;
    public final static int Add       =  16;
    public final static int Sub       =  17;
    public final static int Eq        =  18;
    public final static int Ne        =  19;
    public final static int Le        =  20;
    public final static int Ge        =  21;
    public final static int Lt        =  22;
    public final static int Gt        =  23;
    public final static int Lsh       =  24;
    public final static int Rsh       =  25;
    public final static int And       =  26;
    public final static int Or        =  27;
    public final static int Xor       =  28;
    public final static int Comma     =  29;
    public final static int Assign    =  32;
    public final static int Return    =  34;
    public final static int Test      =  35;
    public final static int If        =  36;
    public final static int Unless    =  37;
    public final static int While     =  38;
    public final static int Until     =  39;
    public final static int For       =  40;
    public final static int Resultis  =  41;
    public final static int Let       =  43;
    public final static int Vec       =  44;
    public final static int Static    =  45;
    public final static int Lparen    =  49;
    public final static int Rparen    =  50;
    public final static int Lsquare   =  51;
    public final static int Rsquare   =  52;
    public final static int Lcurly    =  53;
    public final static int Rcurly    =  54;
    public final static int To        =  55;
    public final static int Do        =  56;
    public final static int Then      =  57;
    public final static int Else      =  58;
    public final static int Be        =  59;
    public final static int Eof       =  60;
    public final static int Semicolon =  61;
    public final static int Printf    =  82;
    public final static int Sys       =  83;


    public static String opstr(int tok) {
	switch(tok) {
	default:        //return "Unknown";

	case Assign:    return "Assign";    case Add:      return "Add";
	case And:       return "And";       case Be:       return "Be";
	case Comma:     return "Comma";     case Div:      return "Div";
	case Do:        return "Do";        case Else:     return "Else";
        case Eq:        return "Eq";        case False:    return "False";
	case For:       return "For";       case Ge:       return "Ge";
	case Gt:        return "Gt";        case If:       return "If";
        case Ind:       return "Ind";       case Lcurly:   return "Lcurly";
	case Le:        return "Le";        case Let:      return "Let";
	case Lparen:    return "Lparen";    case Lsh:      return "Lsh";
        case Lsquare:   return "Lsquare";   case Lt:       return "Lt";
        case Lv:        return "Lv";        case Mod:      return "Mod";
        case Mul:       return "Mul";       case Name:     return "Name";
        case Ne:        return "Ne";        case Neg:      return "Neg";
        case Not:       return "Not";       case Num:      return "Num";
        case Or:        return "Or";        case Printf:   return "Printf";
        case Rcurly:    return "Rcurly";    case Resultis: return "Resultis";
        case Return:    return "Return";    case Rparen:   return "Rparen";
        case Rsh:       return "Rsh";       case Rsquare:  return "Rquare";
	case Semicolon: return "Semicolon"; case Static:   return "Static";
        case Sub:      return "Sub";        case Sys:      return "Sys";
        case Test:     return "Test";       case Then:     return "Then";
        case To:       return "To";         case True:     return "True";
        case Valof:    return "Valof";      case Vec:      return "Vec";
	case Unless:    return "Unless";    case Until:    return "Until";
	case While:     return "While";     case Xor:      return "Xor";
	}
    }

    public static String charv;
    public static char[] chbuf;
    public static int ch;
    public static int token;
    public static int lexval;
    public static int lexstr;
    public static int chcount;
    public static int lineno;

    public static void init() {
	//writes("Lex.init called\n");
	Nametab.init();
	chbuf = new char[64];
	for(int i=0; i<=63; i++) chbuf[i] = 0;
	chcount = 0;
	lineno = 1;
	rch();

    }

    public static void lex() throws Softerror, Fatalerror {
	while(true) {
	    switch(ch) {
      
	    case '\f': case '\n':
		lineno++;

	    case '\r': case '\t': case ' ':
		rch();
		continue;

	    case '0':case '1':case '2':case '3':case '4':
	    case '5':case '6':case '7':case '8':case '9':
		lexval = 0;
		while('0'<=ch && ch<='9'){
		    lexval = 10*lexval + ch - '0';
		    rch();
		}
		token = Num;
		return;
 
	    case 'a':case 'b':case 'c':case 'd':case 'e':
	    case 'f':case 'g':case 'h':case 'i':case 'j':
	    case 'k':case 'l':case 'm':case 'n':case 'o':
	    case 'p':case 'q':case 'r':case 's':case 't':
	    case 'u':case 'v':case 'w':case 'x':case 'y':
	    case 'z':
            case 'A':case 'B':case 'C':case 'D':case 'E':
            case 'F':case 'G':case 'H':case 'I':case 'J':
            case 'K':case 'L':case 'M':case 'N':case 'O':
            case 'P':case 'Q':case 'R':case 'S':case 'T':
            case 'U':case 'V':case 'W':case 'X':case 'Y':
            case 'Z':
		token = Nametab.lookup(rdtag());
		return;
 
	    case '{': token = Lcurly;    break;
            case '}': token = Rcurly;    break;
            case '[': token = Lsquare;   break;
            case ']': token = Rsquare;   break;
            case '(': token = Lparen;    break;
            case ')': token = Rparen;    break; 
            case '!': token = Ind;       break;
            case '@': token = Lv;        break;
            case '+': token = Add;       break;
            case '-': token = Sub;       break;
            case ',': token = Comma;     break;
            case ';': token = Semicolon; break;
            case '&': token = And;       break;
            case '|': token = Or;        break;
            case '=': token = Eq;        break;
            case '*': token = Mul;       break;
            case '^': token = Xor;       break;
 
            case '/':
		rch();
		if(ch=='/'){
		    do rch(); while(ch!='\n' && ch!=c_eof);
		      continue;
                  }
                  token = Div;
                  return;
 
              case '~':   rch();
                  if(ch=='=') { token = Ne;  break; }
                  token = Not;
                  return;
 
              case '<':
                  rch();
                  if(ch=='=') { token = Le;  break; }
                  if(ch=='<') { token = Lsh; break; }
                  token = Lt;
                  return;
 
              case '>':
                  rch();
                  if(ch=='=') { token = Ge;  break; }
                  if(ch=='>') { token = Rsh; break; }
                  token = Gt;
                  return;
 
              case ':':
                  rch();
                  if(ch=='=') { token = Assign;  break; }
                  Syn.synerr("'=' expected after ':'");
                  return;
 
              case '"':
                  charv = "";
                  rch();
                  while(ch!='"' && ch!=c_eof)
		      charv += (char)rdstrch();
                  token = String;
                  break;
 
              case '\'':
                  rch();
                  lexval = rdstrch();
                  token = Num;
                  if(ch!='\'')  Syn.synerr("Bad character constant");
                  break;

              default:
                  if(ch!=c_eof) {
                    int badch = ch;
                    ch = ' ';
                    Syn.synerr("Illegal character "+ (char)badch);
                  }
                  token = Eof;
                  return;
	    }
 
	    // break in the above switch jumps here.
            rch();
	    return;
	}
    }

    public static void rch() {
	ch = rdch();
	//System.out.println("rch: ch = " + ch);
	chbuf[++chcount&63] = (char)ch;
    }

    public static void wrchbuf() {
	int p;
	System.out.print("\n...");
	for(p = chcount+1; p<=chcount+64; p++) {
	    char k = chbuf[p&63];
	    if(k>0) System.out.print(""+k);
	}
	System.out.println();
    }
 
    public static String rdtag() {
	charv = "";
	while('a'<=ch && ch<='z' ||
	      'A'<=ch && ch<='Z' ||
	      '0'<=ch && ch<='9' ||  ch=='_') {
	    charv += (char)ch;
	    rch();
	}
	return charv;
    }

    public static int rdstrch()  throws Softerror, Fatalerror {
	int res = ch;
	if(ch=='\n' || ch=='\f') {
	    lineno++;
	    Syn.synerr("Unescaped newline character");
	}
	if(ch=='\\') {
	    rch();
	    switch(ch) {
	    default:   Syn.synerr("Bad string or character constant");
	    case '\\':case '\'': case '"':  res = ch;        break;
	    case 't': case 'T':             res = c_tab;     break;
	    case 'n': case 'N':             res = c_newline; break;
	    }
	}
	rch();
	return res;
    }

    public static void skip_to_eol() {
	while(ch!='\n' && ch!=c_eof) rch();
    }

}
