/* A compiler and interpreter for VSPL usinf flex/bison

(c) Martin Richards 29 may 2000
*/


#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <setjmp.h>

#include "vspl.h"
#include "parser.h"

#define maxint       0x7FFFFFFF
#define nametablesize 541
#define c_tab           9
#define c_newline      10

jmp_buf *reclabel, *finlabel;

extern void fatalerr(char *);
extern void fatalerrs(char *, char *);
extern void synerr(char *);
extern void synerrn(char *, int);
extern void trnerr(char *);
extern void trnerrs(char *, char *);
extern void trnerrn(char *, int);
int  errcount, errmax;
FILE *progstream, *tostream;

int optTokens, optTree, optCode, optTrace;

char strv[10000];
char *strp, *strt;
int chcount, lineno;
extern char *yytext;
extern FILE *yyin;
int intval;
char *charv;

Tree nametable[nametablesize];
Tree lookupword(char *word);
Tree wordnode;
Tok token;
Tree parsetree;

void formtree(void);
void plist(Tree, int, int);

/* Globals used in TRN and the interpreter */
 
void trnext(int);
void trprog(Tree);
void trcom(Tree, int);
void decldyn(Tree);
void declstatnames(Tree);
void checkdistinct(Cell *);
void addname(Tree, Tok, int);
Cell *cellwithname(Tree);
void trdecl(Tree);
void undeclare(Cell *);
void jumpcond(Tree, int, int);
void assign(Tree, Tree);
void load(Tree);
void loadlv(Tree);
void fnbody(Tree);
void loadlist(Tree);
void transname(Tree, Tok, Tok, Tok, Tok);
Cell dvec[1000];
Cell *dvece, *dvecp, *dvect;
int  comline;
char *procname;
int  resultlab;
int  ssp;
void outf(Tok);
void outfn(Tok, int);
void outfl(Tok, int);
void outfs(Tok, char *);
void outentry(int, int);
void outlab(int);
void outvar(int);
void outstatvec(int, int);
void outstring(int lab, char *str);
char *opstr(Tok);
int  hasOperand(Tok);
int  mem[50000];
int  memt = 50000;
int  regs;
int  codev, codep, codet;
int  datav, datap, datat;
int  stack, stackt;
int  labv[1000], refv[1000];
int  labmax=999;
void putcode(int);
void putd(int);
void putref(int);
void setlab(int, int);
int  nextlab(void);
int  labnumber;
void resolvelabels(void);
int  interpret(int regs, int mem[]);
void prf(char *form, int mem[], int arg[]);



void lex() { token = yylex(); }

void newline() { printf("\n"); }

int main(int argc, char *argv[]) {
  int i;
  char *progfilename=0; 
  char *tofilename=0; 
  int treesize = 10000;
  jmp_buf finbuf;

  errmax   = 3;
  errcount = 0;
  /*  fin_p, fin_l = level(), fin */

  optTokens=optTree=optCode=optTrace=0;

  /*  treevec=codev=datav=labv=refv=stack=0; */
  progstream=tostream=0;
   
  printf("\nVSPL (29 May 2000) lex/yacc version\n");
 
  for(i=1; i<argc; i++) {
    if     (strcmp(argv[i], "-l")==0)  optTokens = 1;
    else if(strcmp(argv[i], "-p")==0)  optTree = 1;
    else if(strcmp(argv[i], "-c")==0)  optCode = 1;
    else if(strcmp(argv[i], "-t")==0)  optTrace = 1;
    else if(strcmp(argv[i], "-o")==0)  tofilename = argv[++i];
    else                               progfilename = argv[i];
  }

  if(setjmp(finbuf)!=0) goto fin;
  finlabel = &finbuf;

  if(progfilename) progstream = fopen(progfilename, "r");

  yyin = progstream;

  if(progstream==0) fatalerr("Trouble with input file\n");

  if(tofilename){
    tostream = fopen(tofilename, "w");
    if(tostream==0) fatalerr("Trouble with output file\n");
  }

  regs=10;
  codev=100;
  codep=codev;
  codet=10000;
  datav=codet;
  datap=datav;
  datat=memt;

  { lineno = 1;
 
    formtree();               // Perform Syntax Analysis
    if(optTokens) goto fin;
 
    if(optTree) { printf("Parse Tree\n");
                  plist(parsetree, 0, 20);
                  newline();
                }
    if(errcount) goto fin;
    trprog(parsetree);        // Translate the tree
    if(errcount) goto fin;
    stack = datap;
    stackt = memt;

    { int *rv = mem+regs;
      int *sv = mem+stack;
      rv[0] = 0;            // result register
      rv[1] = stack;        // p pointer
      rv[2] = stack+2;      // sp
      rv[3] = codev;        // pc
      rv[4] = maxint;       // count

      sv[0]=sv[1]=sv[2]=0;

      { int ret = interpret(regs, mem);   // Execute the interpreter
        if(ret) printf("Return code %d\n", ret);
        printf("\nInstructions executed: %d\n", maxint-rv[4]);
      }
    }
  }
   
fin:
  if(progstream) fclose(progstream);
  if(tostream)   fclose(tostream);

  return errcount==0 ? 0 : 20;
}


Tree lookupword(char *word) {
  char *p = word;
  int hashval = 0;
  while(*p) hashval = (13*hashval + *p++) & 0xFFFFFF;
  hashval %= nametablesize;
  wordnode = nametable[hashval];

  while(wordnode.I && strcmp(wordnode.I->str, word)!=0)
    wordnode = wordnode.I->link;

  if(wordnode.I==0) {
    wordnode = mkId(Name, nametable[hashval], strp);
    p = word;
    do *strp++ = *p; while(*p++);
    nametable[hashval] = wordnode;
  }
  return wordnode;
}

Tree mkId (Tok op, Tree link, char *str){
  Tree p;
  p.I = (Id*)malloc(sizeof(Id));
  p.I->op = op;
  p.I->link = link;
  p.I->str = str;
  return p;
}

Tree mkInt(Tok op, int k){
  Tree p;
  p.K = (Int*)malloc(sizeof(Int));
  p.K->op = op;
  p.K->k = k;
  return p;
}

Tree mkStr(Tok op, char *str){
  Tree p;
  p.S = (Str*)malloc(sizeof(Str));
  p.S->op = op;
  p.S->str = str;
  return p;
}

Tree mkNull() {
  Tree p;
  p.N1= (N1*)0;
  return p;
}

Tree mk1(Tok op) {
  Tree p;
  p.N1= (N1*)malloc(sizeof(N1));
  p.N1->op = op;
  return p;
}

Tree mk2(Tok op, Tree a) {
  Tree p;
  p.N2= (N2*)malloc(sizeof(N2));
  p.N2->op = op;
  p.N2->a  = a;
  return p;
}

Tree mk3  (Tok op, Tree a, Tree b){
  Tree p;
  p.N3= (N3*)malloc(sizeof(N3));
  p.N3->op = op;
  p.N3->a  = a;
  p.N3->b  = b;
  return p;
}

Tree mk4  (Tok op, Tree a, Tree b, Tree c){
  Tree p;
  p.N4= (N4*)malloc(sizeof(N4));
  p.N4->op = op;
  p.N4->a  = a;
  p.N4->b  = b;
  p.N4->c  = c;
  return p;
}

Tree mk1n (Tok op, int ln){
  Tree p;
  p.N1n= (N1n*)malloc(sizeof(N1n));
  p.N1n->op = op;
  p.N1n->ln = ln;
  return p;
}

Tree mk2n (Tok op, Tree a, int ln){
  Tree p;
  p.N2n= (N2n*)malloc(sizeof(N2n));
  p.N2n->op = op;
  p.N2n->a  = a;
  p.N2n->ln = ln;
  return p;
}

Tree mk3n (Tok op, Tree a, Tree b, int ln){
  Tree p;
  p.N3n= (N3n*)malloc(sizeof(N3n));
  p.N3n->op = op;
  p.N3n->a  = a;
  p.N3n->b  = b;
  p.N3n->ln = ln;
  return p;
}

Tree mk4n (Tok op, Tree a, Tree b, Tree c, int ln){
  Tree p;
  p.N4n= (N4n*)malloc(sizeof(N4n));
  p.N4n->op = op;
  p.N4n->a  = a;
  p.N4n->b  = b;
  p.N4n->c  = c;
  p.N4n->ln = ln;
  return p;
}

Tree mk5n (Tok op, Tree a, Tree b, Tree c, Tree d, int ln){
  Tree p;
  p.N5n= (N5n*)malloc(sizeof(N5n));
  p.N5n->op = op;
  p.N5n->a  = a;
  p.N5n->b  = b;
  p.N5n->c  = c;
  p.N5n->d  = d;
  p.N5n->ln = ln;
  return p;
}


void formtree() {
  int res, i;
  jmp_buf recbuf;
  if(setjmp(recbuf)!=0) goto recover;
  reclabel = &recbuf;
  parsetree = mkNull();

  for(i=0; i<nametablesize; i++) nametable[i] = mkNull();
  strp = strv;
  strt = strv+10000;

  while(optTokens) {            // For debugging lex.
    lex();
    if(token==0)    return;
    printf("token = %3d %s", token, opstr(token));
    if(token==Num) printf("      %s", yytext);
    if(token==Name) printf("     %s", yytext);
    if(token==String) printf("   %s", yytext);
    newline();
  }

recover:
  res = yyparse();
  if(token!=0) fatalerr("Incorrect termination");
  return;
}


void fatalerr(char *mess) {
  printf("\nFatal error:  ");
  printf(mess);
  printf("\nCompilation aborted\n");
  errcount++;
  longjmp(*finlabel, 1);
}

void fatalerrs(char *mess, char *str) {
  printf("\nFatal error:  ");
  printf(mess, str);
  printf("\nCompilation aborted\n");
  errcount++;
  longjmp(*finlabel, 1);
}

char *opstr(Tok op) {
  switch(op) {
    default:        return "Unknown";

    case Assign:    return "Assign";    case Add:      return "Add";
    case And:       return "And";       case Be:       return "Be";
    case Comma:     return "Comma";     case Data:     return "Data";
    case Decl:      return "Decl";      case Div:      return "Div";
    case Do:        return "Do";        case Else:     return "Else";
    case Entry:     return "Entry";     case Eq:       return "Eq";
    case False:     return "False";     case Fnap:     return "Fnap";
    case For:       return "For";       case Fndef:    return "Fndef";
    case Fnrn:      return "Fnrn";      case Ge:       return "Ge";
    case Gt:        return "Gt";        case Halt:     return "Halt";
    case If:        return "If";        case Ind:      return "Ind";
    case Jf:        return "Jf";        case Jt:       return "Jt";
    case Jump:      return "Jump";      case Lab:      return "Lab";
    case Laddr:     return "Laddr";     case Lcurly:   return "Lcurly";
    case Le:        return "Le";        case Let:      return "Let";
    case Ll:        return "Ll";        case Llp:      return "Llp";
    case Ln:        return "Ln";        case Lp:       return "Lp";
    case Lparen:    return "Lparen";    case Lres:     return "Lres";
    case Lsh:       return "Lsh";       case Lsquare:  return "Lsquare";
    case Lt:        return "Lt";        case Lv:       return "Lv";
    case Mod:       return "Mod";       case Mul:      return "Mul";
    case Name:      return "Name";      case Ne:       return "Ne";
    case Neg:       return "Neg";       case Not:      return "Not";
    case Num:       return "Num";       case Or:       return "Or";       
    case Printf:    return "Printf";    case Rcurly:   return "Rcurly";
    case Resultis:  return "Resultis";  case Return:   return "Return";
    case Rparen:    return "Rparen";    case Rsh:      return "Rsh";
    case Rsquare:   return "Rquare";    case Rtap:     return "Rtap";
    case Rtdef:     return "Rtdef";     case Rtrn:     return "Rtrn";
    case Semicolon: return "Semicolon"; case Seq:      return "Seq";
    case Sl:        return "Sl";        case Sp:       return "Sp";
    case Stack:     return "Stack";     case Static:   return "Static";
    case Statvec:   return "Statvec";   case String:   return "String";
    case Stind:     return "Stind";     case Sub:      return "Sub";
    case Sys:       return "Sys";       case Test:     return "Test";
    case Then:      return "Then";      case To:       return "To";
    case True:      return "True";      case Valof:    return "Valof";
    case Vecap:     return "Vecap";     case Vec:      return "Vec";
    case Unless:    return "Unless";    case Until:    return "Until";
    case While:     return "While";     case Xor:      return "Xor";
  }
}


char *v[20];

void plist(Tree x, int n, int d) {
  int i, j;
  int size = 0;
  int ln = 0;

  if(x.I==0) { printf("Nil"); return;  }
 
  switch(x.I->op) {
    default:
         size     = 1;        break;

    case Num:     printf("%d",     x.K->k);   return;
    case Name:    printf("%s",     x.I->str); return;
    case String:  printf("\"%s\"", x.S->str); return;

    case For:
         size = 5; ln = x.N5n->ln; break;

    case Fndef: case Rtdef:
    case Let: case Vec: case Test:
         size = 4; ln = x.N4n->ln; break;

    case Vecap: case Mul: case Div: case Mod: case Add: case Sub:
    case Eq: case Ne: case Lt: case Gt: case Le: case Ge:
    case Lsh: case Rsh: case And: case Or: case Xor:
    case Comma: case Seq: case Decl: case Statvec:
         size = 3;                 break;

    case Assign: case Rtap: case Fnap:
    case If: case Unless: case While: case Until:
         size = 3; ln = x.N3n->ln; break;

    case Valof: case Lv: case Ind: case Neg: case Not:
         size = 2;                 break;

    case Printf: case Sys: case Static: case Resultis:
         size = 2; ln = x.N2n->ln; break;

    case Return:
		 size = 1; ln = x.N1n->ln; break;

    case True: case False:
         size = 1;                 break;
  }
 
  if(n==d) { printf("Etc"); return; }
  printf("%s", opstr(x.I->op));
  if(ln) printf("  -- line %d", ln);
  for(i=2; i<=size; i++) {
    Tree ni = i==2 ? x.N2n->a :
              i==3 ? x.N3n->b :
              i==4 ? x.N4n->c :
                     x.N5n->d;

    newline();
    for(j=0; j<n; j++) printf("%s", v[j]);
	printf("*-");
    v[n] = i==size ? "  " : "! ";
    plist(ni, n+1, d);
  }
}


void trnerr(char *mess) {
  printf("Error");
  if(procname) printf(" in %s", procname);
  if(comline)  printf(" near line %d", comline);
  printf(":   ");
  printf(mess);
  newline();
  errcount++;
  if(errcount >= errmax) fatalerr("Too many errors");
}

void trnerrs(char *mess, char *s) {
  printf("Error");
  if(procname) printf(" in %s", procname);
  if(comline) printf(" near line %d", comline);
  printf(":   ");
  printf(mess, s);
  newline();
  errcount++;
  if(errcount >= errmax) fatalerr("Too many errors");
}

void trnerrn(char *mess, int a) {
  printf("Error");
  if(procname) printf(" in %s", procname);
  if(comline) printf(" near line %d", comline);
  printf(":   ");
  printf(mess, a);
  newline();
  errcount++;
  if(errcount >= errmax) fatalerr("Too many errors");
}


void trprog(Tree x) {
  int i;
  dvec[0].name.I = 0;
  dvec[0].k = 0;
  dvec[0].n = 0;
  dvece = dvec;
  dvect = dvec + 1000;
  dvece++;
  for(i=0; i<nametablesize; i++) {
    Tree name = nametable[i];
    while(name.I) {
      Tree next = name.I->link;
      name.I->link.I = 0; // Mark undeclared
      name = next;
    }
  }

  for(i=0; i<=labmax; i++) { labv[i] = -1; refv[i] = 0; }

  resultlab = -2;
  comline = 1;
  procname = 0;
  labnumber = 1;
  ssp = 2;

  outfl(Laddr, 1); ssp++;  // 1 = lab number of start
  outfn(Fnap, 3);  ssp--;
  outf(Halt);

  declstatnames(x);
  checkdistinct(dvec+1);
  while(x.I) { trdecl(x.N3->a); x=x.N3->b; }
  resolvelabels();

  printf("Program size: %d   Data size: %d\n", codep-codev, datap-datav);
}


void trnext(int next) {
  if(next<0) outf(Rtrn);
  if(next>0) outfl(Jump, next);
}
 
void trcom(Tree x, int next) { 
// x       is the command to translate
// next<0  compile x followed by Rtrn
// next>0  compile x followed by Jump next
// next=0  compile x only
  while(1) {
    Tok op = x.I->op;

    switch(op) {
      default:
             trnerr("Compiler error in Trans");
             return;
 
      case Let:
           { Cell *e = dvece;
	     int s = ssp;
             comline = x.N4n->ln;
             addname(x.N4n->a, Local, ssp+1);
             load(x.N4n->b);
             trcom(x.N4n->c, next);
             undeclare(e);
             outfn(Stack, s);
             ssp = s;
             return;
           }
  
      case Vec:
           { Cell *e = dvece;
       	     int s = ssp;
             comline = x.N4n->ln;
             addname(x.N4n->a, Vec, ssp+1);
             ssp += x.N4n->b.K->k;
             outfn(Stack, ssp);
             trcom(x.N4n->c, next);
             undeclare(e);
             outfn(Stack, s);
             ssp = s;
             return;
           }
  
      case Assign:
             comline = x.N3n->ln;
             assign(x.N3n->a, x.N3n->b);
             trnext(next);
             return;
 
      case Rtap:
           { int s = ssp;
             comline = x.N3n->ln;
             ssp += 3;
             outfn(Stack, ssp);
             loadlist(x.N3n->b);
             load(x.N3n->a);
             outfn(Rtap, s+1);
             ssp = s;
             trnext(next);
             return;
           }
 
      case Printf:
      case Sys:
           { int s = ssp;
             Tok op = x.N3n->op;
             comline = x.N3n->ln;
             loadlist(x.N3n->a);
             outfn(op, s+1);
             ssp = s;
             trnext(next);
             return;
           }
 
      case Unless:
      case If:
               comline = x.N4n->ln;
	       if(next>0) {
               jumpcond(x.N4n->a, op==Unless, next);
               trcom(x.N4n->b, next);
             } else {
               int l = nextlab();
               jumpcond(x.N4n->a, op==Unless, l);
               trcom(x.N4n->b, next);
               outlab(l);
               trnext(next);
             }
             return;
 
      case Test:
           { int l = nextlab();
             int m = 0;
             comline = x.N4n->ln;
             jumpcond(x.N4n->a, 0, l);
             if(next==0) {
               m = nextlab();
      	       trcom(x.N4n->b, m);
             } else {
               trcom(x.N4n->b, next);
             }
             outlab(l);
             trcom(x.N4n->c, next);
             if(m) outlab(m);
             return;
           }
 
      case Return:
             comline = x.N1n->ln;
             outf(Rtrn);
             return;
 
      case Resultis:
             comline = x.N2n->ln;
             if(resultlab=-1) { fnbody(x.N2n->a); return; }
             if(resultlab<=0) {
               trnerr("Resultis out of context");
               return;
             }
             load(x.N2n->a);
             outfl(Resultis, resultlab);
             ssp--;
             return;
 
      case Until:
      case While:
           { int l = nextlab();
             int m = next;
             comline = x.N3n->ln;
             if(next<=0) m = nextlab();
             jumpcond(x.N3n->a, op==Until, m);
             outlab(l);
             trcom(x.N3n->b, 0);
             comline = x.N3n->ln;
             jumpcond(x.N3n->a, op==While, l);
             if(next<=0) outlab(m);
             trnext(next);
             return;
           }
 
      case For:
           { Cell *e = dvece;
	     int s = ssp;
             int l = nextlab();
             int m = nextlab();
             comline = x.N5n->ln;
             addname(x.N5n->a, Local, ssp+1);
             load(x.N5n->b);  // The control variable at s+1
             load(x.N5n->c);  // The end limit        at s+2

             outfl(Jump, m);               // Jump to test

             outlab(l);                    // Start of body
             trcom(x.N5n->d, 0);

             outfn(Lp, s+1); ssp++;        // Inc control variable
             outfn(Ln, 1);   ssp++;
             outf(Add);      ssp--;
             outfn(Sp, s+1); ssp++;

             outlab(m);
             outfn(Lp, s+1); ssp++;        // Compare with limit
             outfn(Lp, s+2); ssp++;
             outf(Le);       ssp--;
             outfl(Jt, l);   ssp--;

             undeclare(e);
             outfn(Stack, s);
             ssp = s;
             trnext(next);
             return;
           }
  
      case Seq:
            trcom(x.N3->a, 0);
            x = x.N3->b;
	}
  }
}


void declstatnames(Tree x) {
  while(x.I) {
    Tree d = x.N3->a;
  
    switch(d.N3->op) {
      default:  trnerr("Compiler error in declstatnames");
                return;

      case Static: { Tree p = d.N3->a;
                     Tree np;
                     while(p.I) switch(p.N3->op) {
                       default:     trnerr("Bad STATIC declaration");
                                    return;

                       case Comma:  np = p.N3->b;
                                    p  = p.N3->a;
                                    continue;

                       case Name: { int lab = nextlab();
                                    outvar(lab);
                                    addname(p, Var, lab);
                                    p = np;
                                    np.I = 0;
                                    continue;
                                  }
                       case Statvec:
                                  { int lab = nextlab();
                                    int upb = p.N4->b.K->k;
                                    outstatvec(lab, upb);
                                    addname(p.N4->a, Addr, lab);
                                    p = np;
                                    np.I = 0;
                                    continue;
                                  }
                     }
                     break;      
                   }

      case Fndef:
      case Rtdef: 
              { Tree name = d.N4n->a;
                int lab = strcmp(name.I->str, "start")==0 ? 1 : nextlab();
                addname(name, Addr, lab);
                break;
              }
    }
    x = x.N3->b;
  }
}
  

void decldyn(Tree x) {
  if(x.I) {
    if(x.I->op==Name) { 
      addname(x, Local, ++ssp);
      return;
    }
 
    if(x.I->op==Comma) { 
      addname(x.N3->a, Local, ++ssp);
      decldyn(x.N3->b);
      return;
    }
 
    trnerr("Compiler error in Decldyn");
  }
}

void checkdistinct(Cell *p) {
  Cell *lim = dvece;
  Cell *q, *c;
  for(q=p; q<lim; q++) {
    Tree n = q->name;
    for(c=q+1; c<lim; c++) if(c->name.I==n.I)
        trnerrs("Name %s defined twice", n.I->str);
  }
}
 
void addname(Tree name, Tok k, int a) {
  Cell *p = dvece + 1;
  if(p>dvect) { trnerr("More workspace needed"); return; }
  dvece->name = name;
  dvece->k = k;
  dvece->n = a;
  name.I->link.Cell = dvece; // Remember the declaration
  dvece = p;
}
 
void undeclare(Cell *e) {
  Cell *t;
  for(t=e; t<dvece; t++) {
    Tree name = t->name;
    name.I->link.Cell = 0;   // Forget its declaration
  }
  dvece = e;
}

Cell *cellwithname(Tree n) {
  Cell *t = n.I->link.Cell;
  if(t) return t;  // It has been looked up before
  t = dvece;
  do t--; while( t->name.I!=n.I && t->name.I!=0);
  n.I->link.Cell = t;  // Associate the name with declaration item
  return t;
}

void trdecl(Tree x) {
  switch(x.N3->op) {

     case Fndef:
     case Rtdef:
             { Cell *e = dvece;
               Tree name = x.N4->a;
               Cell *t = cellwithname(name);
               int strlab = nextlab();

               resultlab = -2;
               procname = name.I->str;

               outstring(strlab, procname);
               outentry(t->n, strlab);
               ssp = 2;
               decldyn(x.N3->b);  // Declare the formal paramenters
               checkdistinct(e);
               outfn(Stack, ssp);
               if(x.N4->op==Rtdef) trcom(x.N4->c, -1);
               else                fnbody(x.N4->c);
 
               undeclare(e);
               procname = 0;
             }
 
    case Static:
    default:   return;
  }
}

void jumpcond(Tree x, int b, int l) {
  int sw = b;

  switch(x.N3->op) {
    case False:  b = !b;
    case True:   if(b) outfl(Jump, l);
                 return;
 
    case Not:    jumpcond(x.N3->a, !b, l);
                 return;
 
    case And: sw = !sw;
    case Or:  if(sw) {
                jumpcond(x.N3->a, b, l);
                jumpcond(x.N3->b, b, l);
                return;
              } else {
                int m = nextlab();
                jumpcond(x.N3->a, !b, m);
                jumpcond(x.N3->b, b,  l);
                outlab(m);
                return;
              }
 
    default:     load(x);
                 outfl(b ? Jt : Jf, l);
                 ssp--;
                 return;
  }
}

void load(Tree x) {
  Tok op = x.N3->op;

  switch(op) {
    default:      trnerrn("Compiler error in Load, op=%d", op);
                  outfl(Ln, 0);
                  ssp++;
                  return;
 
    case Vecap:
    case Mul: case Div: case Mod: case Add: case Sub:
    case Eq: case Ne: case Lt: case Gt: case Le: case Ge:
    case Lsh: case Rsh: case And: case Or: case Xor:
                  load(x.N3->a); load(x.N3->b); outf(op);
                  ssp--;
                  return;
 
    case Ind: case Neg: case Not:
                  load(x.N2->a);
                  outf(op);
                  return;

    case Lv:      loadlv(x.N2->a);
                  return;
 
    case Num:     outfn(Ln, x.K->k); ssp++; return;
    case True:    outfn(Ln, -1);     ssp++; return;
    case False:   outfn(Ln, 0);      ssp++; return;
 
    case String:  
                { int strlab = nextlab();
                  outstring(strlab, x.S->str);
                  outfl(Laddr, strlab);
                  ssp++;
                  return;
                }
 
    case Name:    transname(x, Lp, Ll, Llp, Laddr);
                  ssp++;
                  return;
 
    case Valof: { int rl = resultlab;
                  resultlab = nextlab();
                  trcom(x.N2->a, 0);
                  outlab(resultlab);
                  outfn(Stack, ssp);
                  outf(Lres); ssp++;
                  resultlab = rl;
                  return;
                }
 
    case Fnap:  { int s = ssp;
                  ssp += 3;
                  outfn(Stack, ssp);
                  loadlist(x.N3->b);
                  load(x.N3->a);
                  outfn(Fnap, s+1);
                  outf(Lres); ssp = s+1;
                  return;
                }
    case Printf:
    case Sys:
           { int s = ssp;
             Tok op = x.N2->op;
             comline = x.N2n->ln;
             loadlist(x.N2n->a);
             outfn(op, s+1);
             ssp = s;
             outf(Lres);
             ssp++;
             return;
           }
  }
}

void loadlv(Tree x) {
  switch(x.N3->op) {
    default:    trnerr("Bad operand to @");
                outf(Lres); ssp++;
                return;

    case Name:  transname(x, Llp, Laddr, 0, 0); ssp++;
                return;

    case Ind:   load(x.N2->a);
                return;

    case Vecap: load(x.N3->a); load(x.N3->b); outf(Add); ssp--;
                return;
  }
}

void fnbody(Tree x) {
  switch(x.N3->op) {
    default:      load(x);
                  outf(Fnrn);
                  ssp--;
                  return;
                   
    case Valof: { Cell *e = dvece;
                  int rl = resultlab;
                  resultlab = -1;
                  trcom(x.N2->a, -1);
                  resultlab = rl;
                  undeclare(e);
                  return;
                }
  }
}
 
void loadlist(Tree x) {
  if(x.I) if(x.N3->op==Comma) { loadlist(x.N3->a); loadlist(x.N3->b); }
          else                load(x);
}

void assign(Tree x, Tree y) {
  switch(x.N3->op) {
    default:    trnerr("Bad assignment");
                return;
    case Name:  load(y);
                transname(x, Sp, Sl, 0, 0);
                ssp--;
                return;
    case Vecap: load(y);
                load(x.N3->a); load(x.N3->b); outf(Add); ssp--;
                outf(Stind); ssp -= 2;
                return;
    case Ind:   load(y);
                load(x.N2->a);
                outf(Stind); ssp -= -2;
                return;
  }
}
 
void transname(Tree x, Tok p, Tok l, Tok v, Tok a) {
  Cell *c = cellwithname(x);
  Tok k = c->k;
  int n = c->n;
 
  switch(k) {
    default:      trnerrs("Name '%s' not declared", x.I->str);
   
    case Local:   outfn(p, n); return;
 
    case Var:     outfl(l, n); return;
 
    case Vec:     if(v==0) {
                    trnerrs("Misuse of local vector '%s'", x.I->str);
                    v = p;
                  }
                  outfn(v, n);
                  return;

    case Addr:    if(a==0) {
                    trnerrs("Misuse of entry name '%s'", x.I->str);
                    a = l;
                  }
                  outfl(a, n);
                  return;
  }
}
 

void wrf(char *form, int a, int b, int c) {
  if(optCode) printf(form, a, b, c);
}

void wrfs(char *form, char *a, int b, int c) {
  if(optCode) printf(form, a, b, c);
}

void outf(Tok op) {
  wrfs("%s\n", opstr(op), 0, 0);
  putcode(op);
}

void outfn(Tok op, int a) {
  wrfs("%s %d\n", opstr(op), a, 0);
  putcode(op); putcode(a);
}

void outfl(Tok op, int lab) {
  wrfs("%s L%d\n", opstr(op), lab, 0);
  putcode(op); putref(lab);
}

void outlab(int lab) {
  wrf("Lab L%d\n", lab, 0, 0);
  setlab(lab, codep);
}

void outentry(int l1, int l2) {
  wrf("Entry L%d L%d\n", l1, l2, 0);
  putref(l2);
  setlab(l1, codep);
}

void outstring(int lab, char *s) {
  int i = 0;
  char *sv = (char *)&mem[datap];
  wrf("String L%d", lab, 0, 0);
  wrfs(" %s\n", s, 0, 0);
  setlab(lab, datap);
  do {
    if(i++%4 == 0) putd(0);
    *sv++ = *s;
  } while(*s++);
}


void outstatvec(int lab, int a) {
  int i;
  wrf("Statvec L%d %d\n", lab, a, 0);
  setlab(lab, datap);
  for(i=0; i<a; i++) putd(0);
}

void outvar(int lab) {
  wrf("Var L%d\n", lab, 0, 0);
  setlab(lab, datap);
  putd(0);
}
 
void putcode(int w) {
  if(codep>codet)  trnerr("More code space needed");
  else             mem[codep++] = w;
}

void putd(int w) {
  if(datap>datat) trnerr("More data space needed");
  else            mem[datap++] = w;
}

void putref(int lab) {
  if(codep>codet) trnerr("More code space needed");
  else { mem[codep] = refv[lab];
         refv[lab] = codep++;
  }
}

void setlab(int lab, int addr) { labv[lab] = addr; }

int nextlab() {
  if(labnumber>=labmax) fatalerr("More label space needed");
  return ++labnumber;
}
 

void resolvelabels() {
  int lab;
  for(lab=1; lab<=labnumber; lab++) {
    int p = refv[lab];
    int labval = labv[lab];
    if(p && labval<0) if(lab==1) trnerr("start not defined");
                      else       trnerrn("Label %d unset", lab);
    while(p) { int np = mem[p];
               mem[p] = labval;
               p = np;
             }
  }
}



int interpret(int regs, int *mem) {
  int retcode = 0;
  int *rv = mem+regs;
  int res   = rv[0];
  int *pp   = mem + rv[1];
  int *sp   = mem + rv[2];
  int *pc   = mem + rv[3];
  int count = rv[4];

  while(1)
  { int op = *pc++;                // Fetch next instruction
    if(optTrace) {
      printf("p:%5d  sp:%5d %10d %10d  %5d: %8s",
              pp-mem,    sp-mem, sp[-1], sp[0], pc-mem-1, opstr(op));
      if(hasOperand(op)) printf(" %d", pc[0]);
      newline();
    }
    if(count<=0) { retcode = 3; break; } // Zero count
    count--;
    
    switch(op) {
      default:     retcode = 1;    goto ret;    // Unknown op code

      case Halt:   retcode = sp[0]; goto ret;

      case Laddr:
      case Ln:     sp++; sp[0] = *pc++;      continue;
      case Lp:     sp++; sp[0] = pp[*pc++]; continue;
      case Llp:    sp++; sp[0] = pp+*pc++ - mem;   continue;
      case Ll:     sp++; sp[0] = mem[*pc++]; continue;
      case Sp:     pp[*pc++] = sp[0]; sp--;  continue;
      case Sl:     mem[*pc++] = sp[0]; sp--; continue;

      case Rtap:
      case Fnap: { int *opp = pp;
                   int *retaddr = pc+1;
                   pp = pp+*pc;
                   pc = sp[0]+mem;
                   pp[0] = opp-mem;
                   pp[1] = retaddr-mem;
                   pp[2] = pc-mem;
                   sp = pp+2;
                   continue;
                 }

      case Lres:   sp++; sp[0] = res;                continue;

      case Fnrn:   res = sp[0];
      case Rtrn: { int *npp = mem + pp[0];
		   int *npc = mem + pp[1];
                   sp = pp-1;
                   pp = npp;
	       	   pc = npc;
                   continue;
                 }
      case Ind:    sp[0] =  mem[sp[0]];                    continue;
      case Neg:    sp[0] =  -  sp[0];                      continue;
      case Not:    sp[0] =  ~  sp[0];                      continue;
      case Stind:  sp = sp-2; mem[sp[2]] = sp[1];          continue;
      case Vecap:  sp--; sp[0] = mem[sp[0] + sp[1]];  continue;
      case Mul:    sp--; sp[0] = (sp[0])  *  (sp[1]); continue;
      case Div:    sp--; sp[0] = (sp[0])  /  (sp[1]); continue;
      case Mod:    sp--; sp[0] = (sp[0])  %  (sp[1]); continue;
      case Add:    sp--; sp[0] = (sp[0])  +  (sp[1]); continue;
      case Sub:    sp--; sp[0] = (sp[0])  -  (sp[1]); continue;
      case Eq:     sp--; sp[0] = (sp[0]) ==  (sp[1]); continue;
      case Ne:     sp--; sp[0] = (sp[0]) !=  (sp[1]); continue;
      case Le:     sp--; sp[0] = (sp[0]) <=  (sp[1]); continue;
      case Ge:     sp--; sp[0] = (sp[0]) >=  (sp[1]); continue;
      case Lt:     sp--; sp[0] = (sp[0])  <  (sp[1]); continue;
      case Gt:     sp--; sp[0] = (sp[0])  >  (sp[1]); continue;
      case Lsh:    sp--; sp[0] = (sp[0]) <<  (sp[1]); continue;
      case Rsh:    sp--; sp[0] = ((unsigned)sp[0]) >>  (sp[1]); continue;
      case And:    sp--; sp[0] = (sp[0])  &  (sp[1]); continue;
      case Or:     sp--; sp[0] = (sp[0])  |  (sp[1]); continue;
      case Xor:    sp--; sp[0] = (sp[0])  ^  (sp[1]); continue;
      case Jt:     sp--; pc = sp[1] ? *pc+mem : pc+1;     continue;
      case Jf:     sp--; pc = sp[1] ? pc+1 : *pc+mem;     continue;
      case Resultis: 
                   ssp--; res = sp[1];
      case Jump:   pc = *pc + mem;                    continue;
      case Stack:  sp = pp + *pc++;                   continue;
      case Printf: sp = pp + *pc++ - 1;
                   prf((char *)(mem+sp[1]), mem, &sp[2]);
                   continue;
      case Sys:    sp = pp + *pc++ - 1;
                   switch(sp[1]) {
                     default: printf("\nBad sys(%d,...) call\n", sp[1]);
                              retcode  = 2;                goto ret;   
                     case 0:  retcode  = sp[2];            goto ret;
                     case 1:  res = interpret(sp[2], mem); continue;
                     case 2:  optTrace = sp[2];            continue;
                     case 3:  res = count; count = sp[2];  continue;
                   }
    }
  }

ret:
  rv[0] = res;
  rv[1] = pp-mem;
  rv[2] = sp-mem;
  rv[3] = pc-mem;
  rv[4] = count;
  return retcode;
}

void prf(char *form, int mem[], int arg[]) {
  int *p = arg;
  while(*form) {
    int k = *form++;
    char fmt[50];
    if(k=='%') {
      char *f=fmt;
      *f++ = '%';
      while(*form) {
        k = *form++;
	if('0'>k || k>'9') break;
	*f++ = k;
      }
      switch(k) {
        default:  printf("%c",k);                  continue;
	case 'd': *f++='d'; *f=0; 
                  printf(fmt, (int)   *p++);       continue;
	case 'x': *f++='X'; *f=0;
                  printf(fmt, (int)   *p++);       continue;
	case 's': *f++='s'; *f=0;
                  printf(fmt, (char *)(mem+*p++)); continue;
      }
    }
    printf("%c", k);
  }
}

int hasOperand(Tok op) {
  switch(op) {
    case Fnrn:case Rtrn:case Lres:case Halt:
    case Vecap:case Ind:case Stind:case Neg:case Not:
    case Mul:case Div:case Mod:case Add:case Sub:
    case Eq:case Ne:case Le:case Ge:case Lt:case Gt:
    case Lsh:case Rsh:case And:case Or:case Xor:
              return 0;
    default:  return 1;
  }
}


