/*

This is a program test posible ways of determining how the amplitude
of a specified note varies with time. It taks input wither from a
.wav file or a microphone.

Implemented by Martin Richards (c) September 2008

Usage:

noterecog FROM/K,TO/K,TEST=-t/K/N,NOTE=-n/K/N,
          -a/K/N,-b/K/N,-c/K/N,MSECS=-ms/K/N,MIC/S,GEN/S

FROM/K      If neither MIC or GEN is given, this gives .wav file
            containing the samples. It skips the first msecs of
            samples.
TO.K        is result of the analysis (default: to stdout).
TEST=-t/K/N Selects which test to run on the samples (default 1).
NOTE=-n/K/N The the note being tested, eg 4c,  5gis or  3bes.
-a/K/N      Arg fot the test
-b/K/N      Arg fot the test
-c/K/N      Arg fot the test
MSECS=-ms/K/N   Length in msecs if using the mic or generating the smples
            For samples read from a .wav file, this is the amount of time
            to skip at the start. It will then read upto 20 second of
            samples, ie 20x44100=882000 samples.
MIC/S       Read msecs wort of samples from the microphone
GEN/S       Generate msecs worth of samples as a combination of
            square. triangular and sie waves.

The sample rate must be 44100 and the samples bit length must be 16.
Stereo .wav input is converted to mono samples by taking the average
of the two channels.

Change history

17/05/2023
Initial implementation based on anawav.b.

*/

GET "libhdr"
GET "sound.h"
GET "mc.h"

GLOBAL {
  sampsv:ug // Vector for 32-bit signed samples
            // obtained from a .wav file, microphone input
	    // or generated by program made by a combination
 	    // of square, triangle and sine waves.
  sampsvupb // UPB of sampv

  sxv_upb   // Self expanding vector to hold the samples
  sxv_v     // being tested.
  sxv       // This will point to sxv_upb, the control block.

  tostream
  stdin
  stdout

  fromname  // If non zero itis the .wav file name
  toname
  testno
  noteno    // The MIDI note number
  arg_a     // Arguments for the test
  arg_b
  arg_c
  arg_w
  msecs     // Number of milli-seconds of recorded data if from
            // the.wavfile, the microphone or generated by program.
  mic       // TRUE if the samples are from the microphone 
  gen       // TRUE if the samples are to be generated
  
  freqtab

  waveInCB
  devname
  format
  channels
  rate      // Samples per second

  dotest
  test1; test2; test3; test4; test5; test6; test7; test8

  getsamples
  rdmicsamples
  gensamples
  addsquarewav
  addtrianglewav
  addsinewav
  addwhitewav

  noterecog
  
  noteampco
  noteampcofn
}

LET start() = VALOF
{ LET argv = VEC 50

  stdout := output()
  tostream := 0

  writef("*nnoterecog entered*n")

  UNLESS rdargs("FROM/K,TO/K,TEST=-t/K/N,NOTE=-n/K/N,*
                *-a/K/N,-b/K/N,-c/K/N,-w/K/N,MSECS=-ms/N,*
                *MIC/S,GEN/S", argv, 50) DO
  { writef("Bad arguments for noterecog*n")
    RESULTIS 0
  }

  waveInCB := 0     // For mic input stream
  devname  := 0
  format   := 16    // S16_LE
  channels := 1     // Monoc
  rate     := 44100 // Samples per second
  
  fromname := "../23-1start.wav"  //0
  toname   := 0
  testno   := 0
  noteno   := 60
  arg_a    := 0
  arg_b    := 0
  arg_c    := 0
  arg_w    := 0
  msecs    := 0
  mic      := FALSE
  gen      := FALSE

  IF argv!0 DO fromname := argv!0      // FROM/K     .wav input
  IF argv!1 DO toname   := argv!1      // TO/K       result of analysis
  IF argv!2 DO testno   := !(argv!2)   // TEST=-t/K/N
  IF argv!3 DO noteno   := !(argv!3)   // NOTE=-n/K/N
  IF argv!4 DO arg_a    := !(argv!4)   // -a/K/N
  IF argv!5 DO arg_b    := !(argv!5)   // -b/K/N
  IF argv!6 DO arg_c    := !(argv!6)   // -c/K/N
  IF argv!7 DO arg_w    := !(argv!7)   // -w/K/N
  IF argv!8 DO msecs    := !(argv!8)   // MSECS/N
  mic                   := argv!9      // MIC/S
  gen                   := argv!10     // GEN/S

//abort(1278)
  //TEST sys(Sys_sound, snd_test)=-1
  //THEN writef("Sound is available*n")
  //ELSE writef("Sound is not available*n")

  setfreqtab()

  sxv_upb, sxv_v := 0, 0
  sxv := @sxv_upb
  
  UNLESS getsamples(sxv, freqtab!noteno, msecs) DO
  { writef("*nUnable to get the samples*n")
    GOTO fin
  }
  sampsv    := sxv_v
  sampsvupb := sampsv!0

  //IF FALSE DO
  { FOR i = 0 TO 500-1 DO
    { IF i MOD 10 = 0 DO writef("*n%i5: ", i)
      writef(" %i5", sampsv!(i+1))
    }
    newline()
    abort(1000)
  }
  
  dotest(testno)

fin:
  IF sampsv DO freevec(sampsv)
  IF tostream DO endstream(tostream)
  selectoutput(stdout)

  RESULTIS 0
}

AND noteampcofn(args) BE
{ // This is the main fuction of each nateamp coroutine.
  // The coroutine is created by a call:
  //     initco(fn, size, args)
  // where
  // fn    this function
  // size  The size of the coroutine
  // args  Vector of initialisation arguments
  // args!0=freq   The frequency to test as aa scaled integer
  //               eg 440_000
  // args!1=cycles The number of cycles to test.

  // The result is the estimated amplitude of this
  // frequency.
  LET freq   = args!0 // eg 440_000
  LET cycles = args!1
  LET amp = 0

  LET samplespercycle = muldiv(44100, 1_000_000, freq)
  LET q1 = samplespercycle  /8000
  LET q2 = samplespercycle*2/8000
  LET q3 = samplespercycle*3/8000
  LET q4 = samplespercycle*4/8000
  writef("freq=%5.3d cycles=%n*n", freq, cycles)
  //abort(1000)
  
  { // Start of the main loop
    LET p = cowait(amp)  // Return -1 first time
    LET a = 0
    UNLESS p BREAK // Test for close down
    // p is a position in samplesv
    amp := findamp(p,       cycles, samplespercycle) -
           findamp(p+q4,    cycles, samplespercycle) 
    a :=   findamp(p+q1,    cycles, samplespercycle) -
           findamp(p+q1+q4, cycles, samplespercycle)
    IF a > amp DO amp := a
    a :=   findamp(p+q2,    cycles, samplespercycle)
           findamp(p+q2+q4, cycles, samplespercycle)
    IF a > amp DO amp := a
    a :=   findamp(p+q3,    cycles, samplespercycle)
           findamp(p+q3+q4, cycles, samplespercycle)
    IF a > amp DO amp := a
  } REPEAT

  cowait(-1) REPEAT // Indicating the end of data
}

AND findamp(p, cycles, samplespercycle) = VALOF
{ // p      is the sbscript of the first sample in samplesv
  //        to look at.
  // cycles is the number ofcycle to test.
  // samplespercycle is a scale fixed point value
  //        with 3 digits after the decimal point
  //        giving the number of samples per cycle.
  LET amp, phase = 0, 0
  LET a0, a1, a2, a3 = 0, 0, 0, 0
  LET q0 = 0
  LET q1 = samplespercycle  /8000
  LET q2 = samplespercycle*2/8000
  LET q3 = samplespercycle*3/8000
  LET q4 = samplespercycle*4/8000
  //writef("findamp: p=%n cycles=%n samplespercycle=%5.3d*n",
  //                 p, cycles, samplespercycle)
  //writef("q1=%n q2=%n q3=%n q4=%n*n", q1, q2, q3, q4)
  FOR i = 0 TO cycles-1 DO
  { LET s = sampsv + p + samplespercycle*i/1000
    LET t = s + q4
    a0 := a0 + s!q0 - t!q0
    a1 := a1 + s!q1 - t!q1
    a2 := a2 + s!q2 - t!q2
    a3 := a3 + s!q3 - t!q3
    //writef("findamp: i=%i2 s=%i4 t=%i4 a0=%i4 a1=%i4 a2=%i4 a3=%i4*n",
    //                 i, s, t, a0, a1, a2, a3)

  }
  IF  a0 > amp DO amp, phase :=  a0, 0
  IF -a0 > amp DO amp, phase := -a0, 4
  IF  a1 > amp DO amp, phase :=  a1, 1
  IF -a1 > amp DO amp, phase := -a1, 5
  IF  a2 > amp DO amp, phase :=  a2, 2
  IF -a2 > amp DO amp, phase := -a2, 6
  IF  a3 > amp DO amp, phase :=  a3, 3
  IF -a3 > amp DO amp, phase := -a3, 7

  //writef("findamp: amp=%i4/%n*n", amp/cycles, phase)
//abort(997755)
  result2 := phase
  RESULTIS amp/cycles/2
}

AND dotest(n) BE
{ LET freq = 440_000
  LET cycles = 20
  LET count = 0
  noteampco := initco(noteampcofn, 1000, freq, cycles)
  writef("*ndotest:*n")
  abort(325476)

  //IF FALSE DO
  { FOR i = 1 TO 10 DO
    { LET days, msecs, filler = 0, 0, 0
      LET t0, t1 = 0, 0
      datstamp(@days)
      t0 := msecs
      FOR i = 1 TO 1000 DO
      { callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
        callco(noteampco, 100)
      }
      datstamp(@days)
      t1 := msecs
      writef("Time for callco(noteampco, 100) is %n usecs*n", (t1-t0)/10)
    }
    abort(1000)
  }
  
  FOR p = 1 TO 1*44100/10 BY 10 DO
  { LET val = callco(noteampco, p) // Ask for another amplitude.
    LET phase = result2
    IF count MOD 10 = 0 DO writef("*n%i6: ", count)
    writef(" %i4/%n", val, phase)
    count := count+1
  }
  newline()
}

AND getmic(v, upb) BE
{ LET count = 0 // Count of samples read

  LET waveInCB = sys(Sys_sound, snd_waveInOpen, 0, 16, 1, 44100)

  writef("waveInCB = %n*n", waveInCB)

  IF waveInCB=-1 DO
  { writef("Cannot open wave input device*n")
    v!0 := 0
    RETURN
  }

  writef("*nRecording %n samples*n", upb)

  UNTIL count>=upb DO
  { LET len = sys(Sys_sound,
                  snd_waveInRead, waveInCB, v+count, upb-count, 0)
    count := count+len
    //IF len DO writef("len = %i5 count = %i7 bufsize=%i7*n",
    //                 len, count, bufsize)
  }

  v!0 := upb

  //writef("Closing waveInCB*n")

  waveInCB := sys(Sys_sound, snd_waveInClose, waveInCB)
}

AND playsamples(v, t0, t1) BE
{ LET s0 = muldiv(44100, t0, 1_000) 
  LET s1 = muldiv(44100, t1, 1_000)
  sawritef("Playing from time %4.3d to %4.3d*n", t0, t1)
  playvec(v+s0, s1-s0+1)
}

AND wrwavfile(name, v) BE
{ // Output the .wav file
  LET upb = v!0
  LET wavout = findoutput(name)

  UNLESS wavout DO
  { writef("Unable to open %s for output*n", name)
    RETURN
  }
  
  selectoutput(wavout)

  wrriffhdr(1,         // mode = mono
            44100,     // typically 44100
            16,        // bits per sample
            upb*2)     // number of bytes of data
  FOR i = 1 TO upb DO
  { LET w = v!i
    binwrch(w)
    binwrch(w>>8)
  }
  endstream(wavout)
  selectoutput(stdout)
}

AND getsamples(sxv, freq, msecs) = VALOF
{ // Return TRUE if successful.
  writef("getsamples: freq=%5.3d length = %5.3d secs*n", freq, msecs)
  abort(776623)
  
  IF mic DO
  { // Read msecs worth of data from the microphone.
    writef("Reading %5.3d secs of samples from the microphone*n", msecs)
    RESULTIS rdmicsamples(sxv, msecs)
  }

  IF gen DO
  { // Generate data as a combination of square, triangle
    // and sine waves of amplitudes arg_a, arg_b and arg_c.
    writef("Generating %5.3d secs of samples*n", msecs)
    RESULTIS gensamples(sxv, freq, msecs)
  }

  IF fromname DO
  { // Read data from a .wav file, skipping the first msecs
    // worth of samples.
    writef("*nLoading wav file %s*n", fromname)
    RESULTIS rdwavfile(sxv, fromname, msecs)
  }

  writef("Unable to obtain samples*n")
  RESULTIS FALSE
}

AND rdmicsamples(msecs) = VALOF
{
  writef("rdmicsamples: entered*n")
  RESULTIS 0
}

AND gensamples(sxv, freq, msecs) = VALOF
{ LET p = 1
  LET q = muldiv(44100, msecs, 1000)
  writef("gensamples: freq=%5.3d secs=%5.3d samples=%n*n",
          freq, msecs, q)

  IF arg_a DO addsquarewav  (freq, arg_a, sxv, p, q)
  IF arg_b DO addtrianglewav(freq, arg_b, sxv, p, q)
  IF arg_c DO addsinewav    (freq, arg_c, sxv, p, q)
  IF arg_w DO addwhitewav   (freq, arg_c, sxv, p, q)
  
  RESULTIS TRUE
}

AND addsquarewav(freq, amp, sxv, p, q) = VALOF
{ // Add a square wave of amplitude amp to positions
  // p to q in the self expanding vector.
  LET samplespercycle = muldiv(44100_000, 1000, freq)
  LET samplesperhalfcycle = samplespercycle/2
  // Note both samplespercycle and freq are scaled values
  // with 3 digits after the decimal point.
  LET v = sxv!1
  writef("addsquarewav: freq=%5.3d amp=%n p=%m q=%n*n", freq, amp, p, q)
  writef("addsquarewav: samplespercycle=%5.3d samplesperhalfcycle=%5.3d*n",
          samplespercycle, samplesperhalfcycle)
  abort(12678)
  UNLESS v    DO { sxpushval(sxv, 0); v := sxv!1 }
  WHILE v!0<q DO { sxpushval(sxv, 0); v := sxv!1 }
  FOR i = p TO q-1 DO
  { LET phase = (i*1000) MOD samplespercycle
    LET sample =  phase < samplesperhalfcycle -> amp, -amp
    writef("addsquarewav: adding sample %i6 at position %n*n", sample, i+1)
    v!(i+1) := v!(i+1) + sample
  }
  abort(12679)
  RESULTIS TRUE
}
  
AND addtrianglewav(freq, amp, sxv, p, q) = VALOF
{ // Add a square wave of amplitude amp to positions
  // p to q in the self expanding vector.
  LET samplespercycle = muldiv(44100_000, 1000, freq)
  LET samplesperhalfcycle = samplespercycle/2
  // Note both samplespercycle and freq are scaled values
  // with 3 digits after the decimal point.
  LET v = sxv!1
  writef("addtrianglewav: freq=%5.3d amp=%n p=%m q=%n*n", freq, amp, p, q)
  writef("addtrianglewav: samplespercycle=%5.3d samplesperhalfcycle=%5.3d*n",
          samplespercycle, samplesperhalfcycle)
  abort(12678)
  UNLESS v    DO { sxpushval(sxv, 0); v := sxv!1 }
  WHILE v!0<q DO { sxpushval(sxv, 0); v := sxv!1 }
  FOR i = p TO q-1 DO
  { LET phase = (i*1000) MOD samplespercycle
    LET sample =  phase < samplesperhalfcycle -> amp, -amp
    writef("addtrianlewav: adding sample %i6 at position %n*n", sample, i+1)
    v!(i+1) := v!(i+1) + sample
  }
  abort(12679)
  RESULTIS TRUE
}
  
AND addsinewav(freq, amp, sxv, p, q) = VALOF
{ // Add a square wave of amplitude amp to positions
  // p to q in the self expanding vector.
  LET samplespercycle = muldiv(44100_000, 1000, freq)
  LET FLT fsamplespercycle = FLOAT samplespercycle
  LET famp = FLOAT amp
  // Note both samplespercycle and freq are scaled values
  // with 3 digits after the decimal point.
  LET FLT pi = sys(Sys_flt, fl_mk, 314158, -5)
  LET FLT pi2 = pi + pi
  LET v = sxv!1
  //writef("addsinewav: freq=%5.3d amp=%n p=%n q=%n*n", freq, amp, p, q)
  //writef("addsinewav: samplespercycle=%5.3d*n", samplespercycle)
  //writef("addsinewav: pi=%5.3e pi2=%5.3e*n", pi, pi2)
  //abort(12678)
  UNLESS v    DO { sxpushval(sxv, 0); v := sxv!1 }
  WHILE v!0<q DO { sxpushval(sxv, 0); v := sxv!1 }
  FOR i = p TO q-1 DO
  { LET phase = (i*1000) MOD samplespercycle
    LET FLT angle = pi2 * FLOAT phase / fsamplespercycle
    LET sample =  FIX (sys(Sys_flt, fl_sin, angle) * famp)
    //writef("addsinewav: phase=%5.3d samplespercycle=%5.3d*n",
    //        phase, samplespercycle)
    //writef("addsinewav: adding angle=%5.3e sample %i6 at position %n*n",
    //        angle, sample, i+1)
    //abort(82883)
    v!(i+1) := v!(i+1) + sample
  }
  abort(12679)
  RESULTIS TRUE
}
  
AND addwhitewav(freq, amp, sxv, p, q) = VALOF
{ // Add a square wave of amplitude amp to positions
  // p to q in the self expanding vector.
  LET samplespercycle = muldiv(44100_000, 1000, freq)
  LET samplesperhalfcycle = samplespercycle/2
  // Note both samplespercycle and freq are scaled values
  // with 3 digits after the decimal point.
  LET v = sxv!1
  writef("addwhitewav: freq=%5.3d amp=%n p=%m q=%n*n", freq, amp, p, q)
  writef("addwhitewav: samplespercycle=%5.3d samplesperhalfcycle=%5.3d*n",
          samplespercycle, samplesperhalfcycle)
  abort(12678)
  UNLESS v    DO { sxpushval(sxv, 0); v := sxv!1 }
  WHILE v!0<q DO { sxpushval(sxv, 0); v := sxv!1 }
  FOR i = p TO q-1 DO
  { LET sample =  randno(2*arg_w+1) - arg_w - 1
    writef("addwhitewav: adding sample %i6 at position %n*n", sample, i+1)
    v!(i+1) := v!(i+1) + sample
  }
  abort(12679)
  RESULTIS TRUE
}
  
AND noterecog(v, upb) BE
{writef("sampsv=%n*n", sampsv)
//abort(1234)
  //test1(sampv, 0, 100)
  test2(sampsv, 440_000, 1_000, 2_500)
}

AND test1(v, p, len) BE
{ //LET upb = v!0

  FOR i = 0 TO len-1 DO
  { IF i MOD 10 = 0 DO writef("*n%i8: ", p+i)
    writef(" %i7", v!(p+i))
  }
  newline()

}

AND test2(v, noteno, t0, t1) BE
{ // frq is the freqency as a scaled integer eq 440_000
  // t0 and t1 are times in seconds as scale integers
  // eg 1_000 1_500.
  LET upb = v!0
  LET x, y = 0, 0
  LET w = 0

  writef("*nAnalysing notes around  %n from time %4.3d to %4.3d*n*n",
          noteno, t0, t1)

  FOR t = t0 TO t1 BY 100 DO
  { LET sn = muldiv(44100,  t, 1000) // Starting sample number
    LET len = 44100/20               // 1/20 sec sample period
    UNLESS  0 < sn < upb DO sn := 1

    writef("%5.3d:", t)
    FOR d = -8 TO 8 BY 2 DO
    { LET frq = freqtab!(noteno+d)
      // eg frq from 420_000 to 450_000
      // For each frq analyse 1/20 sec of samples
      // sn is the sample number at time t0
      writef(" %i5", amp(v+sn, upb-sn, frq, len))
    }
    newline()
  }

  newline() 
}

AND amp(v, upb, frq, len) = VALOF
{ LET bestx = 0

  FOR i = 0 TO 44100_000/frq DO
  { LET x = amp1(v+i, upb-i, frq, len)
    IF x>bestx DO bestx := x
  }
  RESULTIS bestx
} 

AND amp1(v, upb, frq, len) = VALOF
{ LET p, x, y = 0, 0, 0
  LET w = 0

  { LET a, b, c, d = 0, 0, 0, 0

    { a := a+v!p    // Quadrant 1
      w, p := w+frq, p+1
    } REPEATWHILE w<11025_000
    w := w-22050_000

    { b := b+v!p    // Quadrant 2
      w, p := w+frq, p+1
    } REPEATWHILE w<0

    { c := c+v!p    // Quadrant 3
      w, p := w+frq, p+1
    } REPEATWHILE w<11025_000
    w := w-22050_000

    { d := d+v!p    // Quadrant 4
      w, p := w+frq, p+1
    } REPEATWHILE w<0

    x := x + a + b - c - d
    y := y - a + b + c - d

    //writef("p=%n abcd= %i6 %i6 %i6 %i6  xy=%i6 %i6*n", p, a, b, c, d, x, y)
    //abort(1000)
  } REPEATWHILE p<len

  x, y := x/p, y/p
  //writef("*nfrq=%4.3d:p=%i5 x=%i6 y=%i6 power=%n*n", frq, p, x, y, x)
//abort(1000)
  RESULTIS x
}

AND famp(p, f) = VALOF
{ // Return the average amplitude of 1/10 of a second
  // of samples starting at p multiplied by a square wave
  // of frequency f.
  // f is scaled so that f=440_000 represents 440.000 Hz.
  LET a, p0 = 0, p
  LET x = 0

//  writef("*nfamp: p=%n f=%9.3d*n", p, f)
  { x := x - 22_050_000
    { //writef("p=%n x=%i9 a=%i6 + %i6 => %i6*n",
      //        p, x, a, !p, a+!p)
      x, a := x+f, a+!p
      p := p+1
    } REPEATUNTIL x >= 0

    x := x + 22_050_000
    { //writef("p=%n x=%i9 a=%i6 - %i6 => %i6*n",
      //        p, x, a, !p, a-!p)
      x, a := x-f, a-!p
      p := p+1
    } REPEATUNTIL x <= 0
    //abort(1001)
  } REPEATUNTIL p-p0 > 44100/10
  RESULTIS a / (p- p0 + 1)
}

AND initcolourtab(ctab) BE
{ LET colours = TABLE       10,
      //  red     green    blue
        255<<16 | 255<<8 | 255,  // White
        150<<16 | 150<<8 | 150,  // Light grey
          0<<16 | 150<<8 | 150,  // 
          0<<16 |   0<<8 | 190,  // Blue
        130<<16 |   0<<8 | 130,  //
        150<<16 |   0<<8 |   0,  // Red
        140<<16 | 140<<8 |   0,  // 
          0<<16 | 180<<8 |   0,  // Green
        100<<16 | 100<<8 | 100,  // Dark grey
          0<<16 |   0<<8 |   0   // Black

  FOR i = 1 TO colours!0-1 DO
  { LET n = colours!0
    LET p = (256*(i-1))/(n-1)
    LET q = (256*i)/(n-1)
    LET c1, c2 = colours!i, colours!(i+1)
    LET r1, g1, b1 = c1>>16&255, c1>>8&255, c1&255
    LET r2, g2, b2 = c2>>16&255, c2>>8&255, c2&255
//sawritef("i=%i2 p=%i3  q=%i3  c1=%x6 c2=%x6*b*n", i, p, q, c1, c2)
    FOR t = p TO q DO
    { LET r = (r1*(q-t)+r2*(t-p))/(q-p)
      LET g = (g1*(q-t)+g2*(t-p))/(q-p)
      LET b = (b1*(q-t)+b2*(t-p))/(q-p)
      ctab!t := r<<16 | g<<8 | b
      //sawritef("%i3: %x6*n", t, ctab!t)
    }
    //sawritef("*n")
    //abort(1000)
  }
//sawritef("*nColour table*n")
  //FOR i = 0 TO 256 DO
  //{ IF i MOD 8 = 0 DO sawrch('*n')
  //  sawritef(" %x6", ctab!i)
  //}
  //sawrch('*n')    
}

/*
AND cmplampfn(x, freq, sv, svupb, wv, k) = VALOF
{ // Compile function 10+x to compute the amplitude of frequency freq
  // at the mid point of samples held held in sv (UPB svupb).
  // wv is the same size as sv and is set to the average waveform.
  // k is the number of cycles to average. k is reduced appropriately
  // if k cycles has more than svupb samples.
  // The result is TRUE if compilation is successful.
  mcKKK(mc_entry, 10+x, 3, 0)

  mcRK(mc_mv, mc_a, x/2)       // Dummy function

  mcF(mc_rtn)
  mcF(mc_endfn)
//abort(1111)
} 

AND mcfamp4(note) = VALOF
{ LET amp = mcCall(10+note, 0, 0, 0)
  RESULTIS amp
}
*/

/*
AND getwav(name, v, upb) BE
{ LET instream = 0

  IF FALSE DO
  { //Set a step up from -20000 to +20000 at t = 1 sec
    //FOR i = 0 TO upb DO v!i := i<44100 -> -20000, +20000

    //Set a slope up from -20000 to +20000 at t = 1 sec for 1/20 sec
    //FOR i = 0 TO 44100 DO v!i := -20000
    //FOR i = 0 TO 2205  DO v!(i+44100) := -20000 + muldiv(40000, i, 2205)
    //FOR i = 44100+2205 TO upb DO v!i := +20000

    // Set a 440Hz sine wave with varying amplitude
    //IF FALSE DO
    { LET x      = 1_000_000_000
      LET xdot   = 0
      LET k      = 3925     // 440Hz
      LET amp    = 20_000
      LET ampdiv = 1_000_000_000/amp

      FOR i = 0 TO 44100 DO // Increase volume for 1 sec
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!i := muldiv(x/ampdiv, i, 44100)
        //v!i := muldiv(x/ampdiv, 44100, 44100)
      }
      FOR i = 44100 TO 88200 DO // Steady volume for 1 sec
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!i := x/ampdiv
      }
      FOR i = 0 TO 44100 DO // Decrease volume to the end
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!(88200+i) := muldiv(x/ampdiv, 44100-i, 44100)
      }
      FOR i = 3*44100 TO upb DO v!i := 0
    }

   IF FALSE DO
     FOR i = 0 TO 44100+4410 DO
     { IF i MOD 4410 = 0 DO abort(1000)
       IF i MOD 10 = 0 DO sawritef("*n%i5:", i)
       sawritef(" %i6", v!i)
     }

    RETURN
  }

  instream := findinput(name)

  UNLESS instream DO
  { writef("Trouble with .wav file %s*n", name)
    v!0 := 0
    RETURN
  }
  writef("Getting wav data from %s*n", name)

  selectinput(instream)
  wav2v(v, upb)
  endstream(instream)
  writef("Number of samples = %n (%5.3d secs)*n",
          v!0, muldiv(v!0, 1000, 44100))
}
*/

AND wav2v(v, upb) = 0

AND rdwavfile(filename) = VALOF
{ // Read samples from the .wav file with name filename. It returns
  // a vector containing the samples as 32-bit signed integers.
  // The zeroth element holds the upb of the vector. If there is a
  // fault the result is zero.
  // If the .wav file is in stereo the left and right samples
  // are averaged.
  LET wavin = findinput(filename)
  LET startpos = muldiv(44100, msecs, 1000)
  LET endpos = startpos + 20*44100
  
  LET sxv_upb, sxv_v = 0, 0 // A self expanding vector
  LET sxv = @sxv_upb
  
  LET riff  = 0        //  0 RIFF
  LET size  = 0        //  4 filesize - 8
  LET wave  = 0        //  8 WAVE
  LET fmt   = 0        // 12 fmt
  LET chksz = 0        // 16 16
  LET quant = 0        // 20 1 = linear
  LET mode  = 0        // 22 1=mono 2=stereo
  LET rate  = 0        // 24 Typically 44100
  LET brate = 0        // 28 byte rate
  LET bytePerSample=0  // 32 1, 2 or 4  = bits/8 * mode
  LET bits  = 0        // 34 bits per sample = 8 or 16
  LET filler= 0        // 36 filler

  LET data  = 0        // 36 data
  LET len   = 0        // 40 bytes of data or -1
  LET count = 0

  UNLESS wavin RESULTIS 0
  selectinput(wavin)
  
  riff  := rd4()              //  0 RIFF
  size  := rd4()              //  4 filesize - 8
  wave  := rd4()              //  8 WAVE
  fmt   := rd4()              // 12 fmt
  chksz := rd4()              // 16 18?
  quant := rd2()              // 20 1 = linear
  mode  := rd2()              // 22 1=mono 2=stereo
  rate  := rd4()              // 24 Typically 44100
  brate := rd4()              // 28 byte rate
  bytePerSample:=rd2()        // 32 1, 2 or 4  = bits/8 * mode
  bits  := rd2()              // 34 bits per sample = 8 or 16
  filler:= chksz=18->rd2(), 0 // 36 filler

  data  := rd4()              // 36 data
  len   := rd4()              // 40 bytes of data or -1

  UNLESS riff=#x46464952 DO writef("Bad RIFF word %x8*n", riff)
  UNLESS wave=#x45564157 DO writef("Bad WAVE word %x8*n", wave)
  UNLESS fmt =#x20746D66 DO writef("Bad fmt  word %x8*n", fmt)
  UNLESS chksz=18        DO writef("Bad subchunk size %n*n", chksz)
  UNLESS mode=1 | mode=2 DO writef("Bad mode %n*n", mode)
  UNLESS rate=44100      DO writef("Bad rate %n*n", rate)
  UNLESS bits=16         DO writef("Bad bits per sample %n*n", bits)
  UNLESS data=#x61746164 DO writef("Bad data word %x8*n", data)

writef("rate=%n %s*n", rate, mode=2->"stereo", "mono")

  { LET w = rd2()
    count := count+1
    IF count < startpos LOOP
    IF count > endpos BREAK
    IF w<0 BREAK               // End of .wav file
    IF mode=2 DO w := (w+rd2())/2
    IF (w & #x8000)~=0 DO w := w | #xFFFF0000 // Sign extend
    sxpushval(sxv, w)
    LOOP
    TEST sxv_v
    THEN IF sxv_v!0 MOD 12 = 1 DO writef("*n%i5: ", sxv_v!0) 
    ELSE writef("*n%i5: ", 0)
    writef(" %i6", w)
    //IF sxv_v!0 MOD 4096 = 0 DO abort(9283)
    //IF sxv_v!0 > 40_000 BREAK
  } REPEAT

  RESULTIS sxv_v
}

AND wrriffhdr(mode, rate, bits, databytes) BE
{ LET bytes_per_sample = bits/8 * mode
  LET byte_rate = bytes_per_sample * rate
  writes("RIFF")        //  0: R I F F
  wr4(36+0)             //  4: size of this file - 8
  writes("WAVE")        //  8: W A V E
  writes("fmt ")        // 12: f m t
  wr4(16)               // 16: fmt subchunk size is 16
  wr2(1)                // 20: 1 = linear quantisation
  wr2(mode)             // 22: 1 = mono, 2=stereo
  wr4(rate)             // 24: samples per second
  wr4(byte_rate)        // 28: bytes per second
  wr2(bytes_per_sample) // 32: block align -- bits/8 * mode  = 1, 2 or 4
  wr2(bits)             // 34: bits per sample  = 8 or 16
  writes("data")        // 36: d a t a
  //wr4(byte_rate * 1)    // 40: number of bytes of data or zero
  wr4(databytes)        // 40: number of bytes of data or -1
}

AND wr1(b) BE
{ binwrch(b)
}

AND wr2(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
}

AND rd2() = VALOF
{ LET w = 0
  LET s = @w
  LET a = binrdch()
  LET b = binrdch()
  IF a<0 | b<0 DO
  { writef("rd2: a=%n b=%n*n", a, b)
    abort(102233)
    RESULTIS -1
  }
  s%0 := a
  s%1 := b
  RESULTIS w
}

AND wr4(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
  binwrch(s%2)
  binwrch(s%3)
}

AND rd4() = VALOF
{ LET w = 0
  LET s = @w
  LET a = binrdch()
  LET b = binrdch()
  LET c = binrdch()
  LET d = binrdch()
  IF a<0 | b<0 | c<0 | d<0 RESULTIS -1
  s%0 := a
  s%1 := b
  s%2 := c
  s%3 := d
  RESULTIS w
}

AND playvec(buf, upb) BE
{ LET waveOutCB = sys(Sys_sound,
                      snd_waveOutOpen, 0, 16, 1, 44100)

  //writef("waveOutCB = %n format=%n channels=%n rate=%n*n",
  //       waveOutCB, format, channels, rate)

  IF waveOutCB=-1 DO
  { writef("Cannot open wave output device*n")
    RETURN
  }

  //writef("*nPlaying %n samples*n", upb)

  { LET count = 0 // Count of samples sent played

    UNTIL count>=upb DO
    { LET len = sys(Sys_sound,
                    snd_waveOutWrite, waveOutCB, buf+1+count, upb-count)
      count := count+len
      //IF len DO writef("len = %i5 count = %i7 bufsize=%i7*n",
      //                 len, count, bufsize)
      IF len<0 BREAK
//writef("Delaying 1 second*n")
      sys(Sys_delay, 1)
    }

  }

  sys(Sys_sound, snd_waveOutClose, waveOutCB)
}

AND setfreqtab() BE
{ // Set freqtab so that freqtab!n = 1000 * the note frequency
  // where n is the MIDI note number. n=60 for middle C (C4).

  // Only notes with non zero frequencies will have recognisers.

  freqtab := TABLE
     8_176,   8_662,   9_178,   9_723,  10_301,  10_914, //   0 c-1.. b-1
    11_563,  12_250,  12_979,  13_750,  14_568,  15_434,

    16_352,  17_324,  18_355,  19_446,  20_602,  21_827, //  12 c0 .. b0
    23_125,  24_500,  25_957,  27_500,  29_136,  30_868,

    32_703,  34_648,  36_709,  38_891,  41_204,  43_654, //  24 c1 .. b1
    46_250,  49_000,  51_914,  55_000,  58_271,  61_736,
  
    65_406,  69_296,  73_417,  77_782,  82_407,  87_308, //  36 c2 .. b2
    92_499,  97_999, 103_827, 110_000, 116_541, 123_471,

   130_812, 138_592, 146_833, 155_564, 164_814, 174_615, //  48 c3 .. b3
   184_998, 195_998, 207_653, 220_000, 233_082, 246_942,

   261_623, 277_183, 293_665, 311_127, 329_628, 349_229, //  60 c4 .. b4
   369_995, 391_996, 415_305, 440_000, 466_164, 493_884,

   523_245, 554_366, 587_330, 622_254, 659_255, 698_457, //  72 c5 .. b5
   739_989, 783_991, 830_610, 880_000, 932_328, 987_767,

  1046_489,1108_731,1174_659,1244_508,1318_510,1396_913, //  84 c6 .. b6
  1479_978,1567_982,1661_219,1760_000,1864_655,1975_533,

  2092_978,2217_461,2349_318,2489_016,2637_020,2793_826, //  96 c7 .. b7
  2959_955,3135_963,3322_438,3520_000,3729_310,3951_066,

  4185_955,4434_922,       0,       0,       0,       0, // 108 c8 .. b8
         0,       0,       0,       0,       0,       0,

         0,       0,       0,       0,       0,       0, // 120 c9 .. g9
         0,       0

  // Check the table
  checktab( 98, 2349_318)
  checktab( 99, 2489_016)
  checktab(100, 2637_020)
  checktab(101, 2793_826)
  checktab(102, 2959_955)
  checktab(103, 3135_963)
  checktab(104, 3322_438)
  checktab(105, 3520_000)
  checktab(106, 3729_310)
  checktab(107, 3951_066)
  checktab(108, 4185_955)
  checktab(109, 4434_922)
}

AND checktab(n, f) BE WHILE n>=0 DO
  { UNLESS freqtab!n = f DO
    { writef("note=%i3 change %8.3d to %8.3d*n", n, freqtab!n, f)
      abort(1000)
    }
    n, f := n-12, (f+1)/2
  }


AND wrnotename(n) BE
{ LET oct = n/12 - 1
  LET note = n MOD 12
  UNLESS 0<=n<=127 DO
  {  writef("Bad")
     RETURN
  }

  SWITCHON note INTO
  { DEFAULT: writef("Bad");           RETURN
    CASE  0:  writef( "%nc",  oct);   RETURN
    CASE  1:  writef( "%nc#", oct);   RETURN
    CASE  2:  writef( "%nd",  oct);   RETURN
    CASE  3:  writef( "%nd#", oct);   RETURN
    CASE  4:  writef( "%ne",  oct);   RETURN
    CASE  5:  writef( "%nf",  oct);   RETURN
    CASE  6:  writef( "%nf#", oct);   RETURN
    CASE  7:  writef( "%ng",  oct);   RETURN
    CASE  8:  writef( "%ng#", oct);   RETURN
    CASE  9:  writef( "%na",  oct);   RETURN
    CASE 10:  writef( "%na#", oct);   RETURN
    CASE 11:  writef( "%nb",  oct);   RETURN
  }
}           


