/* This is a version of the benchmark program written by Duncan Lissett
   in March 2003.

His results are:

Language:      MS C#
Test:          bench
Processor:     2GHz Pentium 4
O/S:           Windows XP
Date:          7 Mar 03
Tester:        D. Lissett
Size:          9728
CPU Time:      0.0298 (wall-clock count=10,000)
*/

using System;
using System.Collections.Specialized;
using System.Text;
using System.IO;

/// <remarks>
/// C# version of the systems programming language benchmark
/// Author: Duncan Lissett, May 2003
/// 
/// In this interpretation of the test, each scheduler function
/// is associated with it's datatypes in a class, and works 
/// directly on those values in the Tcb without restoring them
/// to registers.
/// 
/// compile with:
///    csc /target:exe /optimize bench.cs
/// or  
///    csc /target:exe /optimize /define:bench100 bench.cs
/// </remarks>
/// 

internal class Bench
{
   const bool TraceOn = true; 

   // Use Encoding to explicitly convert bytes to ascii chars
   static ASCIIEncoding Encoding = new ASCIIEncoding();
   static int Layout = 0;


   static void Main(string[] args)
   {
      #if bench100
         const int count = 10000*100;
         const int qpktcountval = 2326410;
         const int holdcountval = 930563;
      #else
         const int count = 10000;
         const int qpktcountval = 23246;
         const int holdcountval = 9297;
      #endif

      Console.WriteLine("Bench mark starting, Count={0}", count); 
      
      Scheduler s = new Scheduler(TraceOn);
      Packet wkq;

      s.AddIdleTask(Scheduler.Idle, 0, null, count);

      wkq = new Packet(null, Scheduler.Worker, Packet.Work);
      wkq = new Packet(wkq, Scheduler.Worker,  Packet.Work);
      s.AddWorkerTask(Scheduler.Worker, 1000, wkq);

      wkq = new Packet(null, Scheduler.DeviceA, Packet.Device);
      wkq = new Packet(wkq, Scheduler.DeviceA, Packet.Device);
      wkq = new Packet(wkq, Scheduler.DeviceA, Packet.Device);
      s.AddHandlerTask(Scheduler.HandlerA, 2000, wkq);

      wkq = new Packet(null, Scheduler.DeviceB, Packet.Device);
      wkq = new Packet(wkq, Scheduler.DeviceB, Packet.Device);
      wkq = new Packet(wkq, Scheduler.DeviceB, Packet.Device);
      s.AddHandlerTask(Scheduler.HandlerB, 3000, wkq);

      s.AddDeviceTask(Scheduler.DeviceA, 4000, null);

      s.AddDeviceTask(Scheduler.DeviceB, 5000, null);

      Console.WriteLine("starting"); 

      s.Schedule();

      Console.WriteLine("\nfinished"); 
      Console.Write("qkt count = {0}  ", s.queueCount);
      Console.WriteLine("holdcount = {0}", s.holdCount);     

      Console.Write("These results are ");
      if (s.queueCount == qpktcountval && s.holdCount == holdcountval)
         Console.WriteLine("correct");
      else 
         Console.WriteLine("incorrect");

      Console.WriteLine("end of run");  
   }


   internal static void Trace(byte aByte)
   {
      --Layout;
      if (Layout <= 0)
      {
         Console.WriteLine();
         Layout = 50;
      }
      // need to explicitly convert byte to ascii char
      byte[] b = new Byte[1];
      b[0] = aByte;
      char[] c = Encoding.GetChars(b);
      Console.Write(c[0]);
   }

}


internal class Scheduler
{
   protected Tcb[] table = new Tcb[MaxTasks];
   protected Tcb list = null;
   protected Tcb currentTcb;
   protected TaskId currentId;

   internal int queueCount = 0;
   internal int holdCount = 0;
   protected bool traceOn = false;


   // Idler, Worker, HandlerA, HandlerB, DeviceA & 
   // DeviceB alias the enum TaskId values.
   internal enum TaskId 
   {
      Idle, Worker, HandlerA, HandlerB, 
      DeviceA, DeviceB}
   internal const TaskId Idle = TaskId.Idle;
   internal const TaskId Worker = TaskId.Worker;
   internal const TaskId HandlerA = TaskId.HandlerA;
   internal const TaskId HandlerB = TaskId.HandlerB;
   internal const TaskId DeviceA = TaskId.DeviceA;
   internal const TaskId DeviceB = TaskId.DeviceB;
   protected const int MaxTasks = 10;


   internal Scheduler(bool trace)
   {
      traceOn = trace;
   }


   internal interface ISchedulerTask 
   {
      Tcb run(Packet aPacket);
   }


   internal void AddIdleTask(TaskId anId, int aPriority, 
      Packet aWorkQueue, int aCount)
   {
      AddRunningTask(anId, aPriority, aWorkQueue, 
         new IdleTask(this, 1, aCount) );
   }


   /// <summary>
   /// The Idle task counts how often it is activated ending
   /// the simulation when the aCount limit is reached.
   /// Normally it releases either DeviceA or DeviceB.
   /// </summary>
   private class IdleTask : ISchedulerTask
   {
      int v1, v2;
      Scheduler s;

      public IdleTask(Scheduler aScheduler, int aValue1, int aValue2)
      {
         v1 = aValue1;        
         v2 = aValue2;  
         s = aScheduler;
      }

      public Tcb run(Packet aPacket)
      {
         v2--;
         if (v2 == 0) 
            return s.HoldCurrent();

         if ( (v1 & 1) == 0 )
         {
            v1 = v1 >> 1;
            return s.Release(DeviceA);
         }
         else
         {
            v1 = (v1 >> 1) ^ 0xD008;
            return s.Release(DeviceB);
         }
      }
   }


   internal void AddWorkerTask(TaskId anId, int aPriority, 
      Packet aWorkQueue)
   {
      AddTask(anId, aPriority, aWorkQueue, 
         new WorkerTask(this, HandlerA, 0) );
   }


   /// <summary>
   /// The worker task packs characters into Packets and
   /// sends them alternately to HandlerA or HandlerB.
   /// </summary>
   private class WorkerTask : ISchedulerTask
   {
      TaskId v1; 
      int v2;
      Scheduler s;

      public WorkerTask(Scheduler aScheduler, TaskId aValue1, int aValue2)
      {
         v1 = aValue1;        
         v2 = aValue2;  
         s = aScheduler;
      }

      public Tcb run(Packet aPacket)
      {
         if (aPacket == null) 
         {   return s.SuspendCurrent(); }
         else 
         {
            if (v1 == HandlerA)
               v1 = HandlerB;
            else
               v1 = HandlerA;
            aPacket.Id  = v1;
            aPacket.A1 = 0;
            for (int i=0; i < Packet.dataSize; i++)
            {
               v2++;
               if (v2 > 26) v2 = 1;
               aPacket.A2[i] = (byte)('A' + v2 - 1);               
            } 
            return s.Queue(aPacket);
         }
      }
   }


   internal void AddHandlerTask(TaskId anId, int aPriority,
      Packet aWorkQueue)
   {
      AddTask(anId, aPriority, aWorkQueue,
         new HandlerTask(this) );
   }


   /// <summary>
   /// The HandlerA task receives Packets from Worker and
   /// sends the characters they contain to DeviceA task
   /// one at a time.
   /// The HandlerB task receives Packets from Worker and
   /// sends the characters they contain to DeviceB task
   /// one at a time.
   /// </summary>
   private class HandlerTask : ISchedulerTask
   {
      Packet v1, v2;
      Scheduler s;

      public HandlerTask(Scheduler aScheduler)
      {  
         s = aScheduler;
      }

      public Tcb run(Packet aPacket)
      {
         if (aPacket != null) 
         {
            if (aPacket.Kind == Packet.Work)
               v1 = aPacket.AddTo(v1);
            else 
               v2 = aPacket.AddTo(v2);
         }
         if (v1 != null) 
         {
            byte count = v1.A1;
            Packet v;
            if (count < Packet.dataSize )
            {
               if (v2 != null ) 
               {
                  v = v2;
                  v2 = v2.Link;
                  v.A1 = v1.A2[count];
                  v1.A1 = (byte)(count + 1);
                  return s.Queue(v);
               }
            }
            else
            {
               v = v1;
               v1 = v1.Link;
               return s.Queue(v);
            }
         }
         return s.SuspendCurrent(); 
      }
   }


   internal void AddDeviceTask(TaskId anId, int aPriority,
      Packet aWorkQueue
      )
   {
      AddTask(anId, aPriority, aWorkQueue,  
         new DeviceTask(this) );
   }


   /// <summary>
   /// The device task simulates a single character device. 
   /// On receiving a Packet is suspends itself. 
   /// When DeviceA is released by the Idle task it returns the
   /// packet to HanderA. When DeviceB is released by the Idle 
   /// task it returns the packet to HanderB.
   /// </summary>
   private class DeviceTask : ISchedulerTask
   {
      Packet v1;
      Scheduler s;

      public DeviceTask(Scheduler aScheduler)
      {  
         s = aScheduler;
      }

      public Tcb run(Packet aPacket)
      {
         if (aPacket == null) 
         {
            if (v1 == null) 
               return s.SuspendCurrent();
            Packet v = v1;
            v1 = null;
            return s.Queue(v);
         }
         else 
         {
            v1 = aPacket;
            if (s.traceOn) Bench.Trace(aPacket.A1);
            return s.HoldCurrent();
         }
      }
   }


   protected void AddTask(TaskId anId, int aPriority,
      Packet aWorkQueue, ISchedulerTask aTask)
   {
      currentTcb = new Tcb(list, anId, aPriority, aWorkQueue, aTask);
      list = currentTcb;
      table[(int)anId] = currentTcb;
   }


   protected void AddRunningTask(TaskId anId, int aPriority,
      Packet aWorkQueue, ISchedulerTask aTask)
   {
      AddTask(anId, aPriority, aWorkQueue, aTask);
      currentTcb.SetRunning(); // Set the newly created task to run
   }


   internal void Schedule()
   {
      currentTcb = list;
      while (currentTcb != null)
      {
         if (currentTcb.IsHeldOrSuspended())
            currentTcb = currentTcb.Link;
         else
         {
            currentId = currentTcb.Id;
            if (traceOn) Bench.Trace((byte)('0' + currentId + 1));

            // no restore store of register values in this version

            Tcb nextTcb = currentTcb.Run();
            currentTcb = nextTcb;
         }
      }
   }


   protected Tcb Find(TaskId anId)
   {
      try { return table[(int)anId]; }
      catch (IndexOutOfRangeException){ return null; }
   }


   protected Tcb Queue(Packet aPacket)
   {
      Tcb t = Find(aPacket.Id);
      if (t == null) return t;
      queueCount++;
      aPacket.Link = null;
      aPacket.Id = currentId;
      return t.CheckPriorityAdd(currentTcb, aPacket);
   }


   protected Tcb Release(TaskId anId)
   {
      Tcb t = Find(anId);
      if (t == null) return t;
      t.IsHeld = false;
      if (t.Priority > currentTcb.Priority) 
         return t;
      else 
         return currentTcb;
   }


   protected Tcb HoldCurrent()
   {
      ++holdCount;
      currentTcb.IsHeld = true;
      return currentTcb.Link;
   }


   protected Tcb SuspendCurrent()
   {
      currentTcb.IsSuspended = true;
      return currentTcb;
   }

}


/// <remarks>Each simulated task has a task control block Tcb.</remarks>
internal class Tcb 
{
   // Variables from spec
   protected Tcb link;       // pointer to another tcb or nil
   protected Scheduler.TaskId id;  // identifier (a small integer)
   protected int pri;        // priority (a positive integer)
   protected Packet wkq;      // list of Packets in the tasks work queue
   protected int state;      // a 3 bit value giving the state
   protected Scheduler.ISchedulerTask task;  
   // the function called by the scheduler
   // when it transfers control to this task
   // and the data the function operates on


   internal Tcb(Tcb aTcb, Scheduler.TaskId anId, int aPriority, 
      Packet aWorkQueue, Scheduler.ISchedulerTask aTask
      )
   {
      link = aTcb;
      id = anId;
      pri = aPriority;
      wkq = aWorkQueue;   
      task = aTask;
      if (wkq == null) 
         state = Suspended;
      else 
         state = SuspendedRunnable;
   }

   ///<value>The next Tcb or null.</value>
   internal Tcb Link
   {
      get { return link; }
   }


   ///<value>The identifier of the task.</value>
   internal Scheduler.TaskId Id
   {
      get { return id; }
   }


   ///<value>The priority of the task.</value>
   internal int Priority
   {
      get { return pri; }
   }


   internal Tcb CheckPriorityAdd(Tcb aTask, Packet aPacket)
   {
      if (wkq == null )
      {
         wkq = aPacket;
         IsRunnable = true;
         if (pri > aTask.Priority) { return this; }
      }
      else 
      {
         wkq = aPacket.AddTo(wkq);
      }
      return aTask;
   }


   internal Tcb Run()
   {
      Packet packet;
      if (IsSuspendedRunnable())
      {
         packet = wkq;
         wkq = packet.Link;
         if (wkq == null) 
            state = 0; //SetRunning();
         else 
            SetRunnable();
      }
      else
      {
         packet = null;
      }
      return task.run(packet);
   }


   // Named bit masks to access 3 bit state value.
   protected static readonly int Runnable = BitVector32.CreateMask();
   protected static readonly int Suspended = BitVector32.CreateMask(Runnable);
   protected static readonly int Held = BitVector32.CreateMask(Suspended);
   protected static readonly int SuspendedRunnable = Suspended | Runnable;

   protected static readonly int NotRunnable = ~Runnable;
   protected static readonly int NotSuspended = ~Suspended;
   protected static readonly int NotHeld = ~Held;


   internal void SetRunning()
   {
      state = 0;
   }

   protected void SetRunnable()
   {
      state = Runnable;
   }

   protected bool IsRunnable
   {
      get { return (state & Runnable) != 0; }
      set { state = value ? state|Runnable : state&NotRunnable; }
   }

   internal bool IsSuspended
   {
      get { return (state & Suspended) != 0; }
      set { state = value ? state|Suspended : state&NotSuspended; }
   }

   internal bool IsHeld
   {
      get { return (state & Held) != 0; }
      set { state = value ? state|Held : state&NotHeld; }
   }

   internal bool IsHeldOrSuspended()
   {
      return (state & Held) != 0 || (state == Suspended); 
   }

   protected bool IsSuspendedRunnable()
   {
      return state == SuspendedRunnable;
   }

}


/// <remarks>Communication between tasks is by means of Packets.</remarks>
internal class Packet
{
   internal const int dataSize = 4;
   // Variables from spec
   protected Packet link;     // pointer to the next Packet or nil
   protected Scheduler.TaskId id;  // task or device that sent the packet
   protected Kinds kind;      // part of the message
   protected byte a1 = 0;     // part of the message
   protected byte[] a2 = new byte[dataSize];  
               

   // Device & Work alias the enum Kinds values.
   internal enum Kinds {Device, Work};
   internal const Kinds Device = Kinds.Device;
   internal const Kinds Work = Kinds.Work;

   
   internal Packet(Packet aLink, Scheduler.TaskId anId, Kinds aKind)
   {
      link = aLink;
      id = anId;
      kind = aKind;
   }


   /// <summary>Append this Packet to aQueue of Packets.</summary>
   internal Packet AddTo(Packet aQueue)
   {
      Packet next, peek;
      link = null;
      if (aQueue == null) return this;
      next = aQueue;
      while ((peek = next.Link) != null)
         next = peek;
      next.Link = this;
      return aQueue;
   }

   ///<value>The next Packet or null.</value>
   internal Packet Link 
   {
      get { return link; }
      set { link = value; }
   }

   ///<value>The id of the task or device that sent the Packet.</value>
   internal Scheduler.TaskId Id
   {
      get { return id; }
      set { id = value; }
   }

   ///<value>Encodes the message contained in the Packet.</value>
   internal Kinds Kind
   {
      get { return kind; }
   }

   ///<value>Encodes the message contained in the Packet.</value>
   internal byte A1
   {
      get { return a1; }
      set { a1 = value; }
   }

   ///<value>Encodes the message contained in the Packet.</value>
   internal byte[] A2
   {
      get { return a2; }
   }
   }

--0-1118989296-1053376746=:51524--
