/*

This will eventually be a BCPL implementation of an emulator of the
Z80 microprocessor. Its human interaction will be based on the
interactive debugger of the BCPL Cintcode system. This emulator has
features that the debugging of code generated by the BCPL to Z80
compiler z80bcpl.b. Only implements the Z80 instructions needed for
this purpose. Additional instructions can easily be added if needed.

Implemented by Martin Richards (c) 5 Oct 2020

Usage: FROM,TO/K,GLBSIZE,STKSIZE,d/S
FROM    The file name of an optional Intel Hex file to load.
TO      Output to file rather than the screen
GLBSIZE The size in words of the root global vector, default size 768.
STKSIZE The size in words of the root stack, default size 4000.
-d      Turn on debugging output


All memory locations are first initialised with the value #xDEAD.
The global vector elements are initialised with #xC000+gn where gn is
the global number, except global zero is initialise with GLBSIZE.
All stack elements are initialised with the value #xABCD.
If FROM is specified the register PC is set to the value of global 1.

IY is normally set to the byte address of global 64.
IX is normally set to the byte address of word 64 relative to the
   current BCPL P pointer.

The BCPL function stack frame is as follows:

          ----------------------------------------------------
         |   P' |  L | A1 | A2 | A3 | more arguments and locals  
          ----------------------------------------------------
             ^
             |
             P

P     The BCPL stack frame pointer of the current function
P'    The BCPL stack frame pointer of the enclosing function or zero.
L     The return byte address.
A1, A2,... The function arguments and locals
      A1 is passed in HL'  ie in the alternate register set
      A2 is passed in DE'
      A3 is passed in BC'

HL is the function entry address

The first instruction of a function is:
RST #x38 SETLINK0   if there are no arguments
RST #x38 SETLINK1   if there awas one argument
RST #x38 SETLINK2   if there were two arguments
RST #x38 SETLINK3   if there were three or more arguments

The function result is returned in HL

History

09/09/2022
Making extensive changes, still in process.

05/10/2020
Started the initial implementation.

*/

GET "libhdr"

/*
// All the instructions compiled by bcplcgz80
// Clearly z80emu.b must implement all of them.

        CASE i_inc    :  RESULTIS  "    INC   %S"
        CASE i_dec    :  RESULTIS  "    DEC   %S"
        CASE i_and    :  RESULTIS  "    AND   %S"
        CASE i_or     :  RESULTIS  "    OR    %S"
        CASE i_xor    :  RESULTIS  "    XOR   %S"
        CASE i_sbchl  :  RESULTIS  "    SBC   HL,%S"
        CASE i_addhl  :  RESULTIS  "    ADD   HL,%S"
        CASE i_ldrr   :  RESULTIS  "    LD    %S,%S"
        CASE i_cpl    :  RESULTIS  "    CPL"
        CASE i_ldrn   :  RESULTIS  "    LD    %S,#X%X2"
        CASE i_ldrnn  :  RESULTIS  "    LD    %S,#X%X4"
        CASE i_ldrll  :  RESULTIS  "    LD    %S,L%N"
        CASE i_push   :  RESULTIS  "    PUSH  %S"
        CASE i_pop    :  RESULTIS  "    POP   %S"
        CASE i_ret    :  RESULTIS  "    RET"
        CASE i_jpihl  :  RESULTIS  "    JP    (HL)"
        CASE i_exx    :  RESULTIS  "    EXX"
        CASE i_jr     :  RESULTIS  "    JR    L%N-$"
        CASE i_jrz    :  RESULTIS  "    JR    Z,L%N-$"
        CASE i_jrnz   :  RESULTIS  "    JR    NZ,L%N-$"
        CASE i_jrc    :  RESULTIS  "    JR    C,L%N-$"
        CASE i_jrnc   :  RESULTIS  "    JR    NC,L%N-$"
        CASE i_jp     :  RESULTIS  "    JP    L%N"
        CASE i_jpz    :  RESULTIS  "    JP    Z,L%N"
        CASE i_jpnz   :  RESULTIS  "    JP    NZ,L%N"
        CASE i_jpc    :  RESULTIS  "    JP    C,L%N"
        CASE i_jpnc   :  RESULTIS  "    JP    NC,L%N"
        CASE i_jpp    :  RESULTIS  "    JP    P,L%N"
        CASE i_jpm    :  RESULTIS  "    JP    M,L%N"
        CASE i_ldri   :  RESULTIS  "    LD    %S,(%S%S%N)"
        CASE i_ldir   :  RESULTIS  "    LD    (%S%S%N),%S"
        CASE i_ldhll  :  RESULTIS  "    LD    HL,(L%N)"
        CASE i_ldrl   :  RESULTIS  "    LD    %S,(L%N)"
        CASE i_srl    :  RESULTIS  "    SRL   %S"
        CASE i_rr     :  RESULTIS  "    RR    %S"
        CASE i_ldlhl  :  RESULTIS  "    LD    (L%N),HL"
        CASE i_ldlr   :  RESULTIS  "    LD    (L%N),%S"
        CASE i_nop    :  RESULTIS  "    NOP"
        CASE i_cpir   :  RESULTIS  "    CPIR"

        CASE i_rst38  :  RESULTIS  "    RST   #X38      " // MR 20/08/2022

// RST arguments used by bcplcgz80.b
        CASE m_apply      :  RESULTIS  "M_APPLY"
        CASE m_fixup      :  RESULTIS  "M_FIXUP"
        CASE m_loadix     :  RESULTIS  "M_LOADIX"
        CASE m_loadiy     :  RESULTIS  "M_LOADIY"
        CASE m_storeix    :  RESULTIS  "M_STOREIX"
        CASE m_storeiy    :  RESULTIS  "M_STOREIY"  
        CASE m_setlink0   :  RESULTIS  "M_SETLINK0"
        CASE m_setlink1   :  RESULTIS  "M_SETLINK1"
        CASE m_setlink2   :  RESULTIS  "M_SETLINK2"
        CASE m_setlink3   :  RESULTIS  "M_SETLINK3"
        CASE m_return     :  RESULTIS  "M_RETURN"
        CASE m_finish     :  RESULTIS  "M_FINISH"
        CASE m_loadlvix   :  RESULTIS  "M_LOADLVIX"
        CASE m_loadlviy   :  RESULTIS  "M_LOADLVIY"
        CASE m_multiply   :  RESULTIS  "M_MULTIPLY"
        CASE m_divide     :  RESULTIS  "M_DIVIDE"
        CASE m_rem        :  RESULTIS  "M_REM"
        CASE m_lshift     :  RESULTIS  "M_LSHIFT"
        CASE m_rshift     :  RESULTIS  "M_RSHIFT"
        CASE m_eq         :  RESULTIS  "M_EQ"
        CASE m_ne         :  RESULTIS  "M_NE"
        CASE m_ls         :  RESULTIS  "M_LS"
        CASE m_gr         :  RESULTIS  "M_GR"
        CASE m_le         :  RESULTIS  "M_LE"
        CASE m_ge         :  RESULTIS  "M_GE"
        CASE m_rdivide    :  RESULTIS  "M_RDIVIDE"
        CASE m_rrem       :  RESULTIS  "M_RREM"
        CASE m_rlshift    :  RESULTIS  "M_RLSHIFT"
        CASE m_rrshift    :  RESULTIS  "M_RRSHIFT"
        CASE m_abs        :  RESULTIS  "M_ABS"
        CASE m_linsearch  :  RESULTIS  "M_LINSEARCH"

        CASE m_default    :  RESULTIS  "M_DEFAULT"
*/

MANIFEST
$( // Taken from com/bcplcgz80.b to help show all the Z80 istructions
   // that the compiler might generate.
   
    dupl0                 =  #B000000000 
    dupl1                 =  #B100000000 

    i_inc                 =  #B00000011 // INC   %S
    i_dec                 =  #B00001011 // DEC   %S
    i_and                 =  #B10100000 // AND   %S
    i_or                  =  #B10110000 // OR    %S
    i_xor                 =  #B10101000 // XOR   %S
    i_sbchl               =  #B01000010 // SBC   HL,%S
    i_addhl               =  #B00001001 // ADD   HL,%S
    i_ldrr                =  #B01000000 // LD    %S,%S
    i_cpl                 =  #B00101111 // CPL
    i_ldrn                =  #B00000110 // LD    %S,#X%X2
    i_ldrnn               =  #B00000001 + dupl0  // LD    %S,#X%X4
    i_ldrll               =  #B00000001 + dupl1  // LD    %S,L%N
    i_push                =  #B11000101 // PUSH  %S
    i_pop                 =  #B11000001 // POP   %S
    i_ret                 =  #B11001001 // RET
    i_jpihl               =  #B11101001 // JP    (HL)
    i_exx                 =  #B11011001 // EXX
    i_jr                  =  #B00011000 + dupl0  // JR    L%N-$
    i_jrz                 =  #B00101000 // JR    Z,L%N-$
    i_jrnz                =  #B00100000 // JR    NZ,L%N-$
    i_jrc                 =  #B00111000 + dupl0  // JR    C,L%N-$
    i_jrnc                =  #B00110000 // JR    NC,L%N-$
    i_jp                  =  #B11000011 // JP    L%N
    i_jpz                 =  #B11001010 // JP    Z,L%N
    i_jpnz                =  #B11000010 // JP    NZ,L%N
    i_jpnc                =  #B11010010 // JP    NC,L%N
    i_jpc                 =  #B11011010 // JP    C,L%N"
    i_jpp                 =  #B11110010 // JP    P,L%N"
    i_jpm                 =  #B11111010 // JP    M,L%N"
    i_ldri                =  #B01000110 // LD    %S,(%S%S%N)
    i_ldir                =  #B01110000 // LD    (%S%S%N),%S
    i_ldhll               =  #B00101010 // LD    HL,(L%N)
    i_ldrl                =  #B01001011 // LD    %S,(L%N)
    i_srl                 =  #B00111000 + dupl1  // SRL   %S
    i_rr                  =  #B00011000 + dupl1  // RR    %S
    i_ldlhl               =  #B00100010 // LD    (L%N),HL
    i_ldlr                =  #B01000011 // LD    (L%N),%S
    i_cpir                =  #B10110001 // CPIR
    i_nop                 =  #B00000000 // NOP
    i_rst38               =  #B11111111 // RST   #X38

    m_jpentrysize         =  3

    m_apply               =  0  * m_jpentrysize
    m_fixup               =  1  * m_jpentrysize
    m_loadix              =  2  * m_jpentrysize
    m_loadiy              =  3  * m_jpentrysize
    m_storeix             =  4  * m_jpentrysize
    m_storeiy             =  5  * m_jpentrysize
    m_setlink0            =  6  * m_jpentrysize
    m_setlink1            =  7  * m_jpentrysize
    m_setlink2            =  8  * m_jpentrysize
    m_setlink3            =  9  * m_jpentrysize        
    m_return              =  10 * m_jpentrysize         
    m_finish              =  11 * m_jpentrysize         
    m_loadlvix            =  12 * m_jpentrysize        
    m_loadlviy            =  13 * m_jpentrysize        
    m_multiply            =  14 * m_jpentrysize        
    m_divide              =  15 * m_jpentrysize        
    m_rem                 =  16 * m_jpentrysize        
    m_lshift              =  17 * m_jpentrysize        
    m_rshift              =  18 * m_jpentrysize        
    m_eq                  =  19 * m_jpentrysize        
    m_ne                  =  20 * m_jpentrysize        
    m_ls                  =  21 * m_jpentrysize        
    m_gr                  =  22 * m_jpentrysize        
    m_le                  =  23 * m_jpentrysize        
    m_ge                  =  24 * m_jpentrysize        
    m_rdivide             =  25 * m_jpentrysize        
    m_rrem                =  26 * m_jpentrysize        
    m_rlshift             =  27 * m_jpentrysize        
    m_rrshift             =  28 * m_jpentrysize        
    m_abs                 =  29 * m_jpentrysize        
    m_linsearch           =  30 * m_jpentrysize

    m_sys                 =  31 * m_jpentrysize // Added for z80emu
$)

GLOBAL {
stdin:ug
stdout
debug       // = TRUE or FALSE

z80memupbb  // Typically = #xFFFF
z80memupb   // Typically = #x7FFF
mem         // A vector holding the 64K bytes of the Z80 memory

glbsize
stksize
glbupb
stkupb
rootgvec
rootstack
codevec
codep

gptr
pptr

pflag         // =TRUE for printing, =FALSE for executing
doz80instr    // (a)   Print or execute one instruction
prz80instr    // (pc)  Print the instruction at pc

faultcode     // Value returned by doz80instr
indentpos     // Used by prstate()

zerror
notyet        // (pc) the instruction at pc is not yet implemented
              // It returns the size of the offending instruction.
r2str
cc2str

// The variables holding the Z80 registers
r_a;  r_b;  r_c;  r_d;  r_e;  r_f;  r_h;  r_l   // The main registers
r_a1; r_b1; r_c1; r_d1; r_e1; r_f1; r_h1; r_l1  // The alternate set
r_ix; r_iy; r_sp; r_pc
r_i;  r_r   // Interrupt page register and the memory refresh register

// Bits in F
r_fval;  r_fval1   // Used to detemine S and Z
r_fpval; r_fpval1  // Used to detemine P/V
r_fhval; r_fhval1  // Used to detemine H
r_fn;    r_fn1
r_fc;    r_fc1

fromfilename
tofilename
checkaddr
cont
error

getbb
getsbb
getwb
getswb
getww
getsww

putbb
putwb
putww

signextb
signextw

getr
putr

praddr

rdval
rdvaraddr
rch

bpt        // The current breakpoint number or -1
bpt_addr   // Vector of breakpoint PC values
bpt_instr  // Vector of breakpoint first bytes (op codes)
brkstep    // =TRUE when resuming from a breakpoint
singlestep   // =TRUE when single stepping (\ command)
tracing

gptr       // Currently selected G pointer = (IY-128 >> 2
pptr       // Currently selected P pointer = (IX-128) >> 2

count      // A count decremented every time a Z80 instruction is executed.
           // Fault 2 occurs when count reaches zero.

trcount    // Set by the TR command to trace the exeution of a number
           // of instructions. When not tracing its value is -1

finval     // The value to be returned by start

finp       // Finish label
finl

recp       // Recovery label
recl

retp       // The label to cause a return from sadebug.
retl       // This will cause the next Z80 instruction to
           // be executed.

style
fullstatus
val        // Current value
vars       // Vector of variables (V1 .. V9)

ch         // Current charater, capitalised
lch        // Current charater

loadhex    // (filename)  Load an Intel Hex file and set the globals defined.

z80dumpmem // Dump the Z80 registers and memory to Z80DUMP in a compacted
           // form using rn length encoding.

prflags    // (f) print the flag bits

// Execution functions that print or execute the instruction
// at a specified location depending on the setting of pflag.
// The result is the size in bytes of the instruction.

doz80instrCB   // (a)   Print or execute one instruction with op BC
doz80instrED   // (a)   Print or execute one instruction with op EB
doz80instrXD   // (a)   Print or execute one instruction with op DD or FD
doz80instrXDCB // (a)   Print or execute one instruction with op XD CB

// Fuctions to print or execute specific instructions depending
// on the setting of pflag.

// The op is in capitals, followed by w for 16 bit operations
// and operands f the form:
//    r    r_a, r_b etc
//    ir   (BC), (DE) (HL), (SP)
//    ird  (IX+d) (IY+d)
//    n    n in range 0 to 255
//    nn   nn in range 0 to 65535

z80ADCrird
z80ADCrn
z80ADCrr
z80ADCr
z80ADDrird
z80ADDrn
z80ADDrr
z80ADD16rr
z80ADDr
z80ANDird
z80ANDn
z80ANDrr
z80ANDr
z80BITbird
z80BITbr
z80BITbro
z80CALLnn
z80CCF
z80CCP
z80CPD
z80CPDR
z80CPI
z80CPIR
z80CPL
z80CPird
z80CPn
z80CPr
z80DAA
z80DAAr
z80DECird
z80DECr
z80DEC16r
z80DI
z80DJNZo
z80EI
z80EXX
z80EXrr
z80HALT
z80IM
z80IMk
z80INCird
z80INCr
z80INC16r
z80IND
z80INDR
z80INI
z80INIR
z80INrin
z80INrr
z80JPr
z80JPnn
z80JPccnn
z80JPccd
z80JPd
z80JPir
z80JRra
z80LDD
z80LDDR
z80LDI
z80LDIR
z80LDirdr
z80LDinnr
z80LDrird
z80LDrinn
z80LDrn
z80LDrr
z80LD16rnn
z80LDwr
z80MULUWrr
z80NEG
z80NOP
z80ORn
z80ORr
z80OTDR
z80OTIR
z80OUTD
z80OUTI
z80OUTrr
z80POP16r
z80PUSH16r
z80RCCird
z80RRCr
z80RESbird
z80RESbr
z80RET
z80RETI
z80RETN
z80RETr
z80RLA
z80RLCA
z80RLCird
z80RLCr
z80RLD
z80RLird
z80RLr
z80RRA
z80RRCA
z80RRCird
z80RRD
z80RRird
z80RRr
z80RSTn
z80SBCrn
z80SBCrr
z80SBCr
z80SCF
z80SETbird
z80SETbr
z80SLAird
z80SLAr
z80SRAird
z80SRAr
z80SRLird
z80SRLr
z80SUBn
z80SUBrn
z80SUBrr
z80SUBr
z80XORn
z80XORr


// Execution functions (do not change r_pc)
// All 8 bit operations unless 16 is specified.

x_adc
x_adc16
x_add
x_add16
x_and
x_neg
x_or
x_rl
x_rlc
x_rr
x_rra
x_rrc
x_sbc
x_sbc16
x_sla
x_srl
x_sub
x_xor

x_exx


}

MANIFEST
{ // Register numbers
  n_b=0     // B
  n_c       // C
  n_d       // D
  n_e       // E
  n_h       // H
  n_l       // L
  n_f       // F
  n_a       // A

  n_af      // AF
  n_bc      // BC
  n_de      // DE
  n_hl      // HL
  n_sp      // SP
  n_pc      // PC
  n_ix      // IX
  n_iy      // IY

  n_af1     // AF'
  n_bc1     // BC'
  n_de1     // DE'
  n_hl1     // HL'

  n_ic      // (C)    Used by IN and OUT
  n_ibc     // (BC)
  n_ide     // (DE)
  n_ihl     // (HL)
  n_isp     // (SP)
  n_inn     // (nn)   used by eg LD A,(nn)
  n_i       // I
  n_r       // R

  n_a1;  n_b1; n_c1; n_d1; n_e1; n_f1; n_h1; n_l1  // The alternate set

  // Condition codes
  cc_c=1     // C       These are needed by eg RET M  or RET NZ
  cc_m       // M
  cc_nc      // NC
  cc_nz      // NZ
  cc_p       // P
  cc_pe      // PE
  cc_po      // PO
  cc_z       // Z

  f_brk = #xEF     // RST #x28 The instruction used for breakpoints
  f_sys = #xF7     // RST #x30 The instruction used the BCPL sys function

  tagbits    = #b_1111_1100_0000_0000
  nontagbits = #b_0000_0011_1111_1111
  globword   = #b_1010_0000_0000_0000
}


LET start() = VALOF
{ LET argv = VEC 50
  finp  := level()   // Label to be used by longjump to cause
  finl  := fin       // z80emu to terminate.

  writef("*nZ80 Emulator for BCPL code compiled by z80bcpl*n")
  writef("Type ? for help*n")

  stdin   := input()
  stdout  := output()
  debug   := FALSE
  
  z80memupbb := #xFFFF                       // The byte upb of the Z80 memory
  z80memupb  := z80memupbb>>1                // The word upb of the Z80 memory
  mem     := getvec(z80memupbb/bytesperword) // 64K bytes of the Z80 memory

  // Initialise the whole Z80 memory
  FOR a = 0 TO z80memupbb-1 BY 2 DO putwb(a, #xDEAD)

  UNLESS rdargs("FROM,TO/K,GLBSIZE/K/N,STKSIZE/K/N,-s/S,-d/S", argv, 50) DO
  { writef("Bad arguments for z80emu*n")
    RESULTIS 0
  }

  fromfilename := 0
  glbsize := 1000  // Default sizes
  stksize := 4000
  
  IF argv!0 DO fromfilename := argv!0                    // FROM

  IF argv!1 DO tofilename := argv!1                      // TO/K

  IF argv!2 DO glbsize := !(argv!2)                      // GLBSIZE/K/N
  glbupb := glbsize-1

  IF argv!3 DO stksize := !(argv!3)                      // GLBSIZE/K/N
  stkupb := stksize-1

  singlestep := argv!4                                   // -s/S
  tracing := singlestep
  
  IF argv!5 DO debug := ~debug                           // -d/S

  vars := 100 // word address of the variables vector
  FOR i = 0 TO 9 DO putww(vars+i, 0) // Set all variables to zero
  
  bpt_addr   := TABLE 0,0,0,0,0,0,0,0,0,0
  bpt_instr  := TABLE 0,0,0,0,0,0,0,0,0,0
  style      := 'F'                   // Default printing style
  fullstatus := FALSE
  val        := 0                     // Current value
  brkstep    := FALSE
  //singlestep, tracing := FALSE, FALSE
  faultcode  := 0
  
  // Set the default values of the z80 registers.
  r_a,  r_b,  r_c,  r_d,  r_e,  r_h,  r_l  := 0, 0, 0, 0, 0, 0, 0
  r_a1, r_b1, r_c1, r_d1, r_e1, r_h1, r_l1 := 0, 0, 0, 0, 0, 0, 0
  r_ix, r_iy, r_sp, r_pc := 0, 0, 0, 0
  r_i,  r_r := 0, 0  // Interrupt page register and the memory refresh register

  // Individual bits of registers F and F', typically zero or set to eg fb_s
  r_fval,  r_fpval,  r_fhval,  r_fn,  r_fc  := 0, 0, 0, 0, 0
  r_fval1, r_fpval1, r_fhval1, r_fn1, r_fc1 := 0, 0, 0, 0, 0
  
  count := 1_000_000_000 // Decremented every time a Z80 intruction is executed.

// Allocate and initialise the root global vector
  rootgvec := 1000   // The root global vector starts at word addr 1000
  gptr := rootgvec
  putww(gptr, glbupb)
  FOR gn = 1 TO glbupb DO putww(gptr+gn, globword+gn)

  // Allocate and initialise the root stack
  rootstack := rootgvec+glbsize
    FOR a = 0 TO stkupb DO putww(rootstack+a, #xEEEE)

  // Set the location for code loaded by loadcode.
  codevec := rootstack+stksize
  codep := codevec<<1 // Byte address of where to load Intel Hex
  FOR i = 0 TO 1000 DO putww(codevec+i, #xC0DE)
  writef("gvec=%n stackbase=%n codebase=%n*n",
          rootgvec, rootstack, codevec)
  
  IF fromfilename DO loadhex(fromfilename)

  r_iy := (rootgvec <<1) + 128
  pptr := rootstack
  r_ix := (pptr<<1) + 128
  r_sp := 64000-2
  putwb(r_sp, 0)  // Set return address zero to cause a return from
                  // the z80emu command.
//  abort(4444)
  r_pc := 100          // Execution starts at byte address 100
  
  putbb(100, #xFD)     // LD L,(IY-124)    Set HL to G1 ie start
  putbb(101, #x6E)
  putbb(102, #x82)
  putbb(103, #xFD)     // LD H,(IY-125)
  putbb(104, #x66)
  putbb(105, #x83)
  putbb(106, #xFF)     // Apply 2          Enter start with return
  putbb(107, m_apply)  //                  address zero.
  putbb(108, #x02)
  putbb(109, #xFF)     // Exit from z80emu
  putbb(110, m_finish)
  
  putbb(111, #x00)
  putbb(112, #x00)

  putww(gptr+3, 113)       // Initialise sys
  
  putbb(113, #xFF)
  putbb(114, m_setlink3)   // SETLINK3
  putbb(115, #xFF)
  putbb(116, m_sys)        // SYS          Calls z80sys()
  putbb(117, #xFF)
  putbb(118, m_return)     // RETURN
  
  putbb(119, #x00)
  putbb(120, #x00)
  putbb(121, #x00)
  putbb(122, #x00)
  putbb(123, #x00)

  // Start execution in the interactive debugger.
  faultcode := 0
  brkstep := FALSE
  //singlestep, tracing := TRUE, TRUE  // Normally set by the -s option
  val := 0
  pflag := FALSE
  
  writef("*nEntering sadebug before executing the *
         *first Z80 instruction of the program*n*n")
  indentpos := 0
//db("Just after initialisation")

  UNLESS singlestep GOTO execute
  
  { sadebug()

    // singlestep is TRUE if single stepping
    // brkstep is TRUE if executin the instruction at a breakpoint
    // If neither are TRUE call doz80instr repeatedly while faultcode is zero
execute:
//abort(6598)
    { IF trcount=0 DO { faultcode := 18; BREAK }
      IF count=0   DO { faultcode := 1;  BREAK }
      pflag := FALSE
      faultcode := doz80instr(r_pc)
      IF tracing DO prstate()
      IF count  >0 DO count   := count-1
      IF trcount>0 DO trcount := trcount-1
    } REPEATUNTIL faultcode | singlestep
  } REPEAT
  
fin:
  newline()
  IF mem DO freevec(mem)
}

AND parity(w) = VALOF
{ // This returns 1 if the 8-bit pattern w has even parity.
  LET t = TABLE 1,0,0,1, 0,1,1,0, 0,1,1,0,1,0,0,1
  // If bbbb is an even parity 4-bit pattern then t!bbbb returns 1
  RESULTIS t!(w%#xF) EQV t!((w>>4)&#xF)
}

AND nsize(n) = VALOF
{ LET res = 1
  IF n<0 DO { res := res+1; n := -n }
  WHILE n > 9 DO { res := res+1; n := ABS(n/10) }
  RESULTIS res
}

AND wrd(n, d) BE
{ FOR i = nsize(n)+1 TO d DO wrch(' ')
  wrn(n)
}

AND wrn(n) BE
{ IF n<0 DO { wrch('-'); n := -n }
  wrpn(n)
}

AND wrpn(n) BE
{ IF n > 9 DO wrpn((n>>1)/5)   // Unsigned division by 10
  wrch((n + 10) MOD 10 + '0')  // Unsigned MOD 10
}

AND sadebug() BE
// Enter the standalone debugger. What it does depends on the settings
// of the global variables faultcode trcount brkstep and singlestep.

// faultcode holds the fault code returned doz80instr when it last
//           executed an instruction. If non zero the fault is reported
//           and the execution state output followed by the debugger
//           prompt.

// brkstep   is only TRUE when the instruction at a breakpoint has just
//           been executed when resuming execution after a breakpoint.
//           If faultcode is non zero the fault is reported and the
//           execution state output followed by the debugger prompt.
//           If faultcode is zero and singlestep is FALSE, normal
//           execution of Z80 instructions resumes.

// singlstep If TRUE the execution state output followed by the
//           debugger prompt. If singlestep is FALSE only the
//           debugger prompt is output.

// When the debugger prompt has been output this function reads and
// performs debugging commands from the user. It only returns when
// another Z80 instruction needs to be executed.

{ recp := level()    // For longjump to goto the place where the 
  recl := recover    // debugger command prompt is output.

  retp := level()    // For longjump to use to cause a return from
  retl := ret        // sadebug causing another Z80 instruction to
                     // be executed.

  bpt   := -1        // No breakpoint number yet

  unsetbreakpoints()

  IF brkstep & faultcode=0 & ~singlestep DO
  { // We are resuming normal execution after the first instruction at
    // a breakpoint has just been executed.
    setbreakpoints()
    brkstep := FALSE
    RETURN
  }

  IF faultcode DO singlestep := TRUE

  // Now report the fault if faultcode is non zero
  
  IF faultcode=2 DO  // Fault caused by the BRK instruction
    FOR i = 0 TO 9 IF bpt_addr!i=val DO
    { bpt := i
      brkstep := TRUE // This must be TRUE when resuming from a breakpoint.
      writef("*n## BPT %n*n", bpt)
      indentpos := 0
      faultcode := 0
      prstate()
      BREAK
    }

  // Attempt to give a description of the current fault
  IF faultcode DO
  { LET mess =  VALOF SWITCHON faultcode INTO
    { DEFAULT:  RESULTIS "Unknown fault"
    
      CASE   1: RESULTIS "Illegal instruction"
      CASE   2: RESULTIS "BRK instruction"
      CASE   3: RESULTIS "Zero count"
      CASE   4: RESULTIS "pc out of range"
      CASE   5: RESULTIS "Division by zero"
      CASE  11: RESULTIS "Watch addr: %+%i7 value: %i8"
      CASE  12: RESULTIS "Indirect address out of range: %+%+%+%n"
      CASE  13: RESULTIS "SIGINT received"
      CASE  15: RESULTIS "PC out of range"
      CASE  16: RESULTIS "P pointer out of range"
      CASE  17: RESULTIS "SIGSEGV occurred"
      CASE  18: RESULTIS "Zero trcount"
      CASE  99: RESULTIS "User requested"
      CASE 110: RESULTIS "Callco fault"
      CASE 111: RESULTIS "Resumeco fault"
      CASE 112: RESULTIS "Deleteco fault"
      CASE 186: RESULTIS "Bad input stream"
      CASE 187: RESULTIS "Bad output stream"
      CASE 188: RESULTIS "Unable to replenish input"
      CASE 189: RESULTIS "Wrch fault"
      CASE 190: RESULTIS "Endread fault"
      CASE 191: RESULTIS "Endwrite fault"
      CASE 197: RESULTIS "Store chain fault"
    }
    TEST faultcode=2
    THEN { sawritef("*n## Breakpoint at PC=%n*n", r_pc)
           brkstep := TRUE // This causes the next instruction to be
	                   // executed with all breakpoints removed
			   // and with singlestep set to TRUE.
			   // If brkstep is TRUE and singlestep is FALSE
			   // after an instruction has been executed
			   // normsl execution of instructions resumes
			   // with all current breakpoint set.
			   // brkstep is always set to FALSE except when
			   // resuming execution after a breakpoint.
         }
    ELSE { sawritef("*n## ABORT %n: ", faultcode)
           sawritef(mess, 0,0,0,0)
	 }
    sawrch('*n')
    indentpos := 0
  }

  IF singlestep & tracing DO prstate()

recover:
  ch := '*n'
nxt:                       // Main loop for debug commands
  IF ch='*n' DO sawritef("$ ")
  rch()
sw:
  singlestep := FALSE // Only set to TRUE by the '\' command.
  SWITCHON ch INTO

  { DEFAULT: error()

    CASE endstreamch:
    CASE 'Q': sawritef("*n"); sys(Sys_quit, 0)   // Quit
         
    CASE '*s':
    CASE '*t':
    CASE '*n': GOTO nxt

    CASE '?':
      newline()
      writes("?       Print this list of debug commands*n")
      writes("L name      Load an Intel Hex program into memory*n")
      writes("123 #o377 #FF03 'c  Constants*n")
      writes("Vn      Variables 0 to 9*n")
      writes("G Gn    Global vector and global variables*n")
      writes("P Pn    Stack frame pointer and local variables*n")
      writes("R<reg>  Where <reg> is one of*N")
      writes("        A B C D E F H L AF BC DE HL SP PC IX IY*n")
      writes("U W     Unsigned byte or word, equive to &#FF or #FFFF*n")
      writes("!e                  Subscription*n")
      writes("**e /e %e +e -e      Arithmetic operators*n")
      writes("|e &e ^e            Bit pattern operators OR, AND and XOR*n")
      writes("< >                 Logical shift left/right one place*n")
      writes("=           Print current value*n")
      writes("F           Toggle the fullstatus flag*n")
      writes("$b $c $d $e $f $o $s $u $x Set the print style*n")
      writes("Sn SR<reg>  Store val in memory or a register*n")
      writes("SPn SGn SVn Store val in a local, global or variable*n")
      writes("TRn         Trace the next n instructions*n")
      writes("TBn         Print n bytes from byte address val*n")
      writes("TS          Print the registers and current instruction*n")
      writes("TF          Print the flags register F*n")
      writes("Tn          Print n words from word address val*n")
      writes("I           Print instruction at address val*n")
      writes("N           Print instruction at address val, increment val*n")
      writes("D           Dump all of Z80 memory to DUMP.mem*n")
      writes("Q           Quit -- leave z80emu*n")
      writes("M           Set/Reset memory watch address*n")
      writes("B 0Bn eBn   List, Unset or Set breakpoints*n")
      writes("X           Execute code until the next RET instruction*n")
      writes("C           Continue normal execution*n")
      writes("\           Execute one Z80 instruction*n")
      GOTO recover

    CASE 'L':  // L <filename>
    { LET len = 0
      LET name = VEC 50
      rch() REPEATWHILE ch='*s'
      WHILE 'A'<=ch<='Z' |
            '0'<=ch<='9' |
	    ch='-' | ch='_' | ch='.' | ch='/' DO
      { len := len+1
        IF len <= 50 * bytesperword DO name%len := lch
	rch()
      }
      name%0 := len
      writef("Loading Intel Hex file: %s*n", name)
      loadhex(name)
      GOTO sw
    }
    
    CASE '0': CASE '1': CASE '2':
    CASE '3': CASE '4': CASE '5':
    CASE '6': CASE '7': CASE '8':
    CASE '9': CASE '#': CASE '*'':
    CASE 'A': CASE 'G': CASE 'P': CASE 'R':
    CASE 'V':
              val := rdval();                     GOTO sw

    CASE 'U': val := val & #xFF;                  GOTO recover
    CASE 'W': val := val & #xFFFF;                GOTO recover

    CASE '!': rch(); val := cont(val + rdval());  GOTO sw
    CASE '+': rch(); val :=      val + rdval();   GOTO sw
    CASE '-': rch(); val :=      val - rdval();   GOTO sw
    CASE '**':rch(); val :=      val * rdval();   GOTO sw
    CASE '/': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val / a
                       GOTO sw
                     }
    CASE '%': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val REM a
                       GOTO sw
                     }
    CASE '|': rch(); val := val  |  rdval();  GOTO sw
    CASE '&': rch(); val := val  &  rdval();  GOTO sw
    CASE '^': rch(); val := val XOR rdval();  GOTO sw

    CASE '<': val := val << 1;                GOTO nxt
    CASE '>': val := val >> 1;                GOTO nxt

    CASE '=': prW(val); newline();            GOTO recover

    CASE '$': rch()
              UNLESS ch='B' | ch='C' | ch='D' | ch='F' |
                     ch='O' | ch='E' | ch='S' | ch='U' |
                     ch='X' | ch='Y' DO
              { writef("Valid style letters are: BCDEFOSUXY*n")
                GOTO recover
              }
              style := ch
              GOTO nxt

    CASE 'S': // Sn  SR<reg>  SPn  SGn  SVn
      rch()
      SWITCHON ch INTO
      { DEFAULT:
          GOTO sw

	CASE '0': CASE '1': CASE '2': CASE '3': CASE '4':
	CASE '5': CASE '6': CASE '7': CASE '8': CASE '9':
        { LET a = rdn()
          UNLESS 0<=a<z80memupb DO error() 
          putww(a, val)
          GOTO sw
        }

        CASE 'B': rch()
	{ LET n = rdn()
          UNLESS 0<=n<z80memupbb DO error() 
          putbb(n, val)
          GOTO sw
	}
	
        CASE 'R': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}
	
	CASE 'P': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'G': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'V': rch()
	{ LET n = rdn()
          UNLESS 0<=n<=9 DO error() 
          vars!n := val
          GOTO sw
        }
      }

    CASE '\':               // Single step execution.
      //db("CASE '\':")
      indentpos := 3        // Used by prstate()
      singlestep := TRUE
      tracing := TRUE
      RETURN                // Execute one Z80 intruction

    CASE 'X':  // Xn   Trace n instruction executions
      error()

    CASE 'H':
      writef("*nAF'=%x4 BC'=%x4 DE'=%x4 HL'=%x4*n",
              getr(n_af1), getr(n_bc1), getr(n_de1), getr(n_hl1))
	      GOTO recover

    CASE 'F':
      fullstatus := ~fullstatus
      writef(" Full Status is %s*n", fullstatus -> "on", "off")
      GOTO recover
      
    CASE 'T': // TS  TF  TRn  TBn  Tn
      rch()
      SWITCHON ch INTO
      { DEFAULT:
          error()

        CASE 'S':         // TS     Print the current state
        { newline()
          prstate()
          GOTO recover
        }

        CASE 'F':         // TF     Print the flags
        { writef(" ")
          prflags()
	  newline()
          GOTO recover
        }

        CASE 'R': rch()    // TRn    Trace n instructions
        { trcount := rdn()
          tracing := TRUE
	  singlestep := FALSE
	  GOTO ret         // Resume execution while
	                   // trcount>0 or faultcode~=0
			   // tracing each instruction.
        }

        CASE 'B': rch()    // TBn    Print n bytes from byte address val
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO writef("*n%i5:   ", val+i)
            prB(mem%(val+i))
	    wrch(' ')
          }
	  newline()
	  GOTO sw
        }

        CASE '0':CASE '1':CASE '2':CASE '3':CASE '4': // Tn
        CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO writef("*n%i5:   ", val+i)
            prW(getww(val+i))
	    wrch(' ')
          }
	  newline()
          GOTO recover
        }
      }

    CASE 'D': writef("*nCintcode memory dumped to DUMP.mem*n")
              sys(Sys_dumpmem, 6) // Dump the memory (context=6)
              GOTO recover

    CASE 'I': writef(" %i5: ", val)
              val := val+prz80instr(val)
 	      newline()
	      GOTO recover
 
    CASE 'Z':  // Equivalent to P1B9C
      error()
      //val := pptr!1  // set breakpoint 9 to current return address

    caseb:
    CASE 'B':  // Set, clear or display breakpoints.
    { LET comch = ch
      TEST comch='B' THEN rch()       // For B
                     ELSE ch := '9'   // For X or Z
      IF '0'<=ch<='9' DO
      { LET n = ch - '0'  // Set or Clear break point n.
        bpt_addr!n := 0
        IF val=0 GOTO nxt
        checkaddr(val>>B2Wsh)
	// Clear any breakpoint for this address
        FOR i = 0 TO 9 IF bpt_addr!i=val DO bpt_addr!i := 0
	// Set breakpoint n for this address
        bpt_addr!n, bpt_instr!n := val, mem%val
        GOTO comch='B' -> nxt, resume
      }
      // B is not followed by a digit so list the breakpoints.
      UNLESS ch='*n' DO newline()
      FOR i = 0 TO 9 DO
      { LET ba=bpt_addr!i
        IF ba DO
        { writef("%n:  ", i)
          writearg(ba)
          newline()
        }
      }
      GOTO recover
    }

    CASE 'M':  // Set or clear memory watch address
      checkaddr(val)
      TEST val THEN writef("*nWatch address: %n*n", val)
               ELSE writef("*nWatch unset*n")
      sys(Sys_watch, val) // ###### Wrong ######
      GOTO recover

    CASE 'C': // Continue normal execution.
      setbreakpoints()
      IF getbb(r_pc)=f_brk DO
      { brkstep := TRUE       // Resuming from a breakpoint
        unsetbreakpoints()
	RETURN                // Obey one instruction before restoring
	                      // the breakpoints.
      }
      brkstep := FALSE
      singlestep := FALSE
      tracing := FALSE
      newline()
      //db("CASE 'C':")
      RETURN

resume:
    { LET pc = r_pc
      newline()
      FOR i = 0 TO 9 IF pc=bpt_addr!i DO
      { // We are resuming at a break point
        writef("Resuming after breakpoint %n*n", i)
        brkstep := TRUE       // Execute one instruction before reinstating
        RETURN                // the BRK instructions
      }
      writef("Resuming normal execution*n")
    }
  }

ret:
  UNLESS brkstep DO setbreakpoints()
  RETURN
}

AND setbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO
  { bpt_instr!i := getbb(ba) // Save the previous instruction at the
    putbb(ba, f_brk)         // breakpoint. and replace it by BRK (=RST #x28)
  }
}

AND unsetbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO putbb(ba, bpt_instr!i) // Restore the original instruction.
}

AND prprompt() BE
{ 
  writef("** ")  // Standalone prompt
}


AND writearg(n) BE
{ LET tag = n & tagbits
//abort(2398)
  IF tag=globword DO
  { //abort(9898)
    writef("  #G%z3#", n & nontagbits)
    RETURN
  }
  IF n=#xC0DE DO
  { writef("  #x%x4", n)
    RETURN
  }
  writef("  %6i", n)
}

AND rdn() = VALOF
{ LET res = 0
  WHILE '0'<=ch<='9' DO { res := res*10 + ch - '0'; rch() }
  RESULTIS res
}

AND rdreg() = VALOF
{ // Read a register name and return its register number.
  // The result is -1 if the register in not recognised.
  // On entry ch is the first character of the name.
  // On exit ch is the character following the name.
  // The registers are:
  //  B C D E H L F A AF BC DE HL SP PC IX IY
  
  SWITCHON ch INTO
  { DEFAULT:
      rch(); RESULTIS -1
    CASE 'A':
      rch()
      UNLESS ch='F' RESULTIS n_a
      rch(); RESULTIS n_af
    CASE 'B':
      rch()
      UNLESS ch='C' RESULTIS n_b
      rch(); RESULTIS n_bc
    CASE 'C':
      rch(); RESULTIS n_c
    CASE 'D':
      rch()
      UNLESS ch='E' RESULTIS n_d
      rch(); RESULTIS n_de
    CASE 'E':
      rch(); RESULTIS n_e
    CASE 'H':
      rch()
      UNLESS ch='L' RESULTIS n_h
      rch(); RESULTIS n_hl
    CASE 'L':
      rch(); RESULTIS n_l
    CASE 'F':
      rch(); RESULTIS n_f
    CASE 'S':
      rch()
      UNLESS ch='P' RESULTIS -1
      rch(); RESULTIS n_sp
    CASE 'P':
      rch()
      UNLESS ch='C' RESULTIS -1
      rch(); RESULTIS n_pc
    CASE 'I':
      rch()
      IF ch='X' DO { rch(); RESULTIS n_ix }
      IF ch='Y' DO { rch(); RESULTIS n_iy }
      RESULTIS -1
  }
}

AND rdvaradd1r(type) = VALOF
{ LET base, lim, n = ?, ?, ?
  UNLESS '0'<=ch<='9' DO error()
  n := rdn()
  SWITCHON type INTO
  { DEFAULT:   error()
    //CASE 'P': base, lim := pptr,   fsize;           ENDCASE
    //CASE 'G': base, lim := gptr,   gptr!g_globsize; ENDCASE
    //CASE 'R': base, lim := regs,   r_upb;           ENDCASE
    CASE 'V': base, lim := vars,   9;               ENDCASE
    CASE 'A': base, lim := 0,      z80memupb;         ENDCASE
  }
  UNLESS 0<=n<=lim DO error()
  RESULTIS base + n
}

AND rdargval(str) = VALOF
{ // str is either a decimal number ddd..d
  // or        a hexadecimal number #hh..h
  LET res = 0
  LET radix = 0
  FOR i = 1 TO str%0 DO
  { LET ch = capitalch(str%i)
    SWITCHON ch INTO
    { DEFAULT:
      CASE '#': UNLESS i=1 RESULTIS -1
                radix := 16
		LOOP

      CASE '0':CASE '1':CASE '2':CASE '3':CASE '4':
      CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
                res := res*radix + ch - '0'
		LOOP

      CASE 'A':CASE 'B':CASE 'C':CASE 'D':CASE 'E':CASE 'F':
                UNLESS radix=16 RESULTIS -1
                res := res*radix + ch - 'A'
		LOOP

    }
  }
  RESULTIS res
  
}

AND rdval() = VALOF
{ // Read a val from the current input stream with first
  // character in ch.
  // Possible values:
  // Vn V Gn G Pn P 'c' #<hex> #O<oct> n
  // RA  RB  RC  RD  RE  RF  RH  RL
  // RIX RIY RSP RPC RAF RBC RDE RHL
  LET res, radix = 0, 10

  WHILE ch='*s' DO rch()
  
  SWITCHON ch INTO
  { DEFAULT:   error()

    CASE 'V':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         RESULTIS vars!n
	       }
	       error()
               RESULTIS 0

    CASE 'A':  rch()        // A n   Return the contents of word address n
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(n)
	       }
               RESULTIS 0

    CASE 'G':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(gptr+n)
	       }
               RESULTIS gptr

    CASE 'P':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(pptr+n)
	       }
               RESULTIS pptr

    CASE '*'': rch(); res := lch; rch();  RESULTIS res

    CASE '#':  radix := 16
               rch()
               IF ch='O' DO { radix := 8; rch() }

    CASE '0': CASE '1': CASE '2': CASE '3': CASE '4': 
    CASE '5': CASE '6': CASE '7': CASE '8': CASE '9': 
             { LET d = 100
               IF '0'<=ch<='9' DO d := ch-'0'
               IF 'A'<=ch<='F' DO d := ch-'A'+10
               IF d>=radix RESULTIS res
               res := res*radix+d
               rch()
             } REPEAT

    CASE 'R':
      rch()
      RESULTIS getr(rdreg()) 
  }
}

AND praddr(a) BE
{ LET type, base = 'A', 0
//  IF pptr <= a <= pptr+fsize & a <= pptr+100 DO type, base := 'P', pptr
//  IF gptr <= a <= gptr+gptr!g_globsize       DO type, base := 'G', gptr
  IF vars <= a <= vars+9                     DO type, base := 'V', vars
//  IF regs <= a <= regs+r_upb                 DO type, base := 'R', regs
  writef("*n%c%i5:", type, a-base)
}

AND prch(ch) BE
{ IF 32 < ch <= 127 DO
  { writef("   %c", ch)
    RETURN
  }
  SWITCHON ch INTO
  { DEFAULT:   writef(" %i3", ch); RETURN
    CASE '*s': writef("  SP");     RETURN
    CASE '*b': writef("  BS");     RETURN
    CASE '*n': writef("  NL");     RETURN
    CASE '*c': writef("  CR");     RETURN
    CASE '*p': writef("  FF");     RETURN
  }
}

AND prB(b) BE
{ // Print a byte in the current style
  b := b & #xFF
  SWITCHON style INTO
  { DEFAULT:   error();              RETURN
    CASE 'C':  prch(b);              RETURN
    CASE 'B':  writef("%8b", b);     RETURN
    CASE 'F':
    CASE 'S':
    CASE 'D':  writef("%4i", signextb(b));     RETURN
    CASE 'O':  writef("%3o", b);     RETURN
    CASE 'U':  writef("%3u", b);     RETURN
    CASE 'X':  writef("%2x", b);     RETURN
    CASE 'Y':  writef("%2x", b);     RETURN
  }

}

AND prW(w) BE
{ // Print a word in the current style
  SWITCHON style INTO
  { DEFAULT:   error();                 RETURN
    CASE 'C':  { LET ch = w & #xFF
                 //writes(" ")
                 wrch(32<=ch<=127 -> ch, '?')
	         ch := w>>8 & #xFF
                 wrch(32<=ch<=127 -> ch, '?')
                 RETURN
               }
    CASE 'B':  writef("%16b", w);           RETURN
    CASE 'D':  writef("%6i",  signextw(w)); RETURN
    CASE 'F':  writearg(w);                 RETURN
    CASE 'O':  writef("%6o",  w&#xFFFF);    RETURN
    CASE 'S':  checkaddr(w)
               writef("%S",   w);           RETURN
    CASE 'U':  writef("%6u",  w&#xFFFF);    RETURN
    CASE 'X':  writef("%4x",  w);           RETURN
    CASE 'Y':  writef("%2x",  w);           RETURN
  }
}

AND checkaddr(a) = VALOF // a is a BCPL pointer in the Z80 memory
{ UNLESS 0<=a<=z80memupb DO error()
  RESULTIS a
}

AND cont(a) = getww(a) // a is a BCPL pointer

AND error() BE { writes("  ??*n"); longjump(recp, recl) }

AND prstate() BE
{ // indentpos is the current indentation position
  LET prevpflag = pflag // pflag must be preserved
  IF fullstatus DO
  { FOR i = indentpos TO 3 DO wrch(' ')
    writef("P=%i5: %i5 %i5", pptr, getww(pptr), getww(pptr+1))
    FOR i = 2 TO 5 DO { wrch(' '); prW(getww(pptr+i)) }
    writef("  (SP)="); prW(getwb(r_sp))
    writef("  "); prflags()
    newline()
    indentpos := 0
  }
  FOR i = indentpos TO 3 DO wrch(' ')
  writef("A="); prB(r_a)
  writef(" BC="); prW(getr(n_bc))
  writef(" DE="); prW(getr(n_de))
  writef(" HL="); prW(getr(n_hl))
  writef(" IX="); prW(r_ix)
  writef(" IY="); prW(r_iy)
  writef(" sp=%i5 %i5:", r_sp, r_pc)
  prz80instr(r_pc)
  newline()
  pflag := prevpflag
  //abort(7687)
}

AND getbb(a) = mem%a

AND getsbb(a) = signextb(mem%a)

AND getwb(a) = mem%a | mem%(a+1)<<8

AND getswb(a) = signextw(mem%a | mem%(a+1)<<8)

AND getww(a) = getwb(a+a)

AND getsww(a) = signextw(getwb(a+a))

AND putbb(a, val) BE mem%a := val

AND putwb(a, val) BE mem%a, mem%(a+1) := val, val>>8

AND putww(a, val) BE putwb(a+a, val)

AND signextb(byte) = VALOF
{ LET res = byte & #x7F
  IF (byte & #x80)>0 DO res := res - #x80
  RESULTIS res
}

AND signextw(word) = VALOF
{ LET res = word & #x7FFF
  IF (word & #x8000)>0 DO res := res - #x8000
  RESULTIS res
}

AND putreg(n, val) BE SWITCHON n INTO
{ DEFAULT:
    error()
    RETURN

  CASE  n_a: r_a  := val & #xFF;    RETURN  
  CASE  n_b: r_b  := val & #xFF;    RETURN  
  CASE  n_c: r_c  := val & #xFF;    RETURN  
  CASE  n_d: r_d  := val & #xFF;    RETURN  
  CASE  n_e: r_e  := val & #xFF;    RETURN  
  CASE  n_f: r_f  := val & #xFF;    RETURN  
  CASE  n_h: r_h  := val & #xFF;    RETURN  
  CASE  n_l: r_l  := val & #xFF;    RETURN  
  CASE n_ix: r_ix := val & #xFFFF;  RETURN  
  CASE n_iy: r_iy := val & #xFFFF;  RETURN  
  CASE n_sp: r_sp := val & #xFFFF;  RETURN  
  CASE n_pc: r_pc := val & #xFFFF;  RETURN  
  CASE n_af: r_a, r_f := val>>8 & #xFF, val & #xFF; RETURN  
  CASE n_bc: r_b, r_c := val>>8 & #xFF, val & #xFF; RETURN
  CASE n_de: r_d, r_e := val>>8 & #xFF, val & #xFF; RETURN
  CASE n_hl: r_h, r_l := val>>8 & #xFF, val & #xFF; RETURN
}

AND rch() BE { lch := sardch(); ch := capitalch(lch) }

AND loadhex(filename) = VALOF
{ // Load BCPL Intel Hex from the given file starting at byte address codep.
  // Globals defined in this file update entries in the global vector
  // with word address rootgvec.
  // Relocation data is encoded by lines starting with $02000004 followed by
  // 4 byte hex values giving the addrsses of the words to relocate and
  // terminated by the checksum byte.


  LET prevstream = input()
  LET hexstream = findinput(filename)
  UNLESS hexstream DO
  { writef("Cannot open file: *"%s*"*n", filename)
    RESULTIS FALSE
  }
  selectinput(hexstream)

  { // Read an Intel Hex record
    LET ch = rdch()
//IF ch>0 DO writef("ch='%c'*n", ch)
//abort(1001)

    IF ch=endstreamch BREAK

    IF ch='$' DO
    { // Either a relocation record or an ingnorable record
      LET n    = rdhexbyte()    // Number of addresses in this record, or ignored
      LET ch1  = rdhexbyte()    // low byte of load address, ignored
      LET ch2  = rdhexbyte()    // low byte of load address, ignored
      LET type = rdhexbyte()    // The record's type, =04 for a relocation block
      LET addr = (ch1<<8) + ch2 // Big ender ordering, ignored
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

      IF type=4 DO
      {
//writef("$%x2%x2%x2%x2*n", n, ch1,ch2, type)
//writef("loadhex: type=4 ch=%c n=%n ch1=%n ch2=%n addr=%n*n",
//                        ch,   n,   ch1,   ch2,   addr)
        FOR i = 1 TO n DO
        { LET p = codep//+6
	  ch1 := rdhexbyte()
          ch2 := rdhexbyte()
	  addr := (ch1<<8) + ch2
	  p := p + addr             // Need not be 2 byte aligned.
          sum := sum+ch1+ch2
          putwb(p, codep+getwb(p))

//writef("codep=%n*n",codep)
//writef("Word at %n changed from %n to %n*n", p, getwb(p), codep+getwb(p))
//abort(1009)
        }
      
//        newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
      
      UNTIL ch='*n' | ch=endstreamch DO ch := rdch()
      IF ch='*n' LOOP
    }


    IF ch=':' DO
    { // Format: : nn aaaa tt   followed by bytes to load  and the checksum byte
      LET n    = rdhexbyte()  // Number of code bytes in this record
      LET ch1  = rdhexbyte()  // low byte of load address
      LET ch2  = rdhexbyte()  // low byte of load address
      LET type = rdhexbyte()  // The record's type
      LET addr = ch1<<8 | ch2 // Big ender ordering
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

//      writef("n=%i2 addr=%i5 type=%n*n", n, addr, type)
      
      IF type=0 DO
      { FOR i = 1 TO n DO
        { LET byte = rdhexbyte()
          sum := sum+byte
          putbb(codep+addr, byte)
//writef(" %x2", byte)
          addr := addr+1
        }
      
        //newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
    }
    
    IF ch=endstreamch BREAK
    ch:=ch // To get round a compiler bug
  } REPEAT


//writef("Init the globals*n")
//abort(1003)
  // Initialise the globals
  
  // BCPL sections start with bytes 'B' 'C''P' 'L' followed by 2 bytes
  // giving the size of the section in bytes.
  { LET size = getwb(codep+4)
    LET p = codep+size-6
//    writef("codep=%n size=%n*n", codep, size)
//    FOR a = 0 TO size*2 -1 DO writef("%x2: %x2*n", a, getbb(codep+a))
//    newline()


    { LET gn    = getwb(p)>>1
      LET entry = getwb(p+2)
      IF entry=0 | p < codep BREAK
      putww(rootgvec+gn, entry)
      //writef("gn=%i3 entry=%n*n", gn, entry)
      p := p-4
    } REPEAT
    
    //abort(1234)
//    FOR a = codep TO codep+size-2 BY 2 DO writef("%i5: %x4*n", a, getwb(a))
  }
  
  IF hexstream DO endstream(hexstream)
  selectinput(prevstream)
  RESULTIS TRUE
}

AND rdhexbyte() = VALOF
{ // Read tw hex characters and retun the specified byte
  LET res = hexval(rdch())
  RESULTIS res<<4 | hexval(rdch())
}

AND hexval(ch) = '0'<=ch<='9' -> ch-'0',
                 'A'<=ch<='F' -> ch-'A'+10,
		 'a'<=ch<='f' -> ch-'a'+10,
		                 -1

AND z8dumpmem() BE
{ LET dumpstream = findoutput("Z80DUMP")
  selectoutput(dumpstream)
  writef("AF=%x4*n", r_a<<8 | r_f)
  writef("BC=%x4*n", r_c<<8 | r_b)
  writef("DE=%x4*n", r_e<<8 | r_d)
  writef("HL=%x4*n", r_l<<8 | r_h)
  writef("IX=%x4*n", r_ix)
  writef("IY=%x4*n", r_iy)
  writef("SP=%x4*n", r_sp)
  writef("PC=%x4*n", r_pc)
  writef("MEM=*n")
  FOR i = 0 TO z80memupbb DO
  { IF i MOD 32 = 0 DO newline()
    writef("%x2", getbb(i))
  }
  newline()
  endstream(dumpstream)
}

AND prird(r, d) BE // eg (SP-114)  or  (HL+12)
{ writef("(%s", r2str(r))
  IF d>0 DO writef("+%n", d)
  IF d<0 DO writef("-%n", -d)
  writef(")")
}

AND prflags(flags) BE
{ writef("F=")
  IF r_fval<0        DO wrch('S')
  IF r_fval=0        DO wrch('Z')
  IF r_fhval         DO wrch('H')
  IF parity(r_fpval) DO wrch('V')
  IF r_fn            DO wrch('N')
  IF r_fc            DO wrch('C')
}

AND setr_f() BE
{ // Use r_fval, r_fpval, r_fh, r_fn, r_fc to set r_f
  r_f := 0                      //  SZ_H _PNC   Flagregister bit positions
  IF r_fval<0        DO r_f :=     #b1000_0000   // S
  IF r_fval=0        DO r_f :=     #b0100_0000   // Z
  IF r_fhval         DO r_f := r_f+#b0001_0000   // H
  IF parity(r_fpval) DO r_f := r_f+#b0000_0100   // P/V
  IF r_fn            DO r_f := r_f+#b0000_0010   // N
  IF r_fc            DO r_f := r_f+#b0000_0001   // C
}

AND getr(rn) = VALOF SWITCHON rn INTO
{ DEFAULT:    writef("*ngetr: Bad register number %n, %s*n", rn, r2str(rn))
              RESULTIS 0

  CASE n_a:   RESULTIS r_a & #xFF
  CASE n_b:   RESULTIS r_b & #xFF
  CASE n_c:   RESULTIS r_c & #xFF
  CASE n_d:   RESULTIS r_d & #xFF
  CASE n_e:   RESULTIS r_e & #xFF
  CASE n_h:   RESULTIS r_h & #xFF
  CASE n_l:   RESULTIS r_l & #xFF
  CASE n_f:   RESULTIS r_f & #xFF
  
  CASE n_bc:  RESULTIS r_b<<8 | r_c
  CASE n_de:  RESULTIS r_d<<8 | r_e
  CASE n_hl:  RESULTIS r_h<<8 | r_l
  CASE n_af:  RESULTIS r_a<<8 | r_f
  
  CASE n_bc1: RESULTIS r_b1<<8 | r_c1
  CASE n_de1: RESULTIS r_d1<<8 | r_e1
  CASE n_hl1: RESULTIS r_h1<<8 | r_l1
  CASE n_af1: RESULTIS r_a1<<8 | r_f1
    
  CASE n_ibc: RESULTIS getbb(r_b<<8 | r_c)
  CASE n_ide: RESULTIS getbb(r_d<<8 | r_e)
  CASE n_ihl: RESULTIS getbb(r_h<<8 | r_l)
  
  CASE n_ix:  RESULTIS r_ix
  CASE n_iy:  RESULTIS r_iy
  CASE n_sp:  RESULTIS r_sp
  CASE n_pc:  RESULTIS r_pc
  CASE n_isp: RESULTIS getbb(r_sp)
}

AND putr(rn, val) BE SWITCHON rn INTO
{ DEFAULT:    writef("*nBad register number %n, %s*n", rn, r2str(rn))
              RETURN

  CASE n_a:   r_a := val; RETURN
  CASE n_b:   r_b := val; RETURN
  CASE n_c:   r_c := val; RETURN
  CASE n_d:   r_d := val; RETURN
  CASE n_e:   r_e := val; RETURN
  CASE n_h:   r_h := val; RETURN
  CASE n_l:   r_l := val; RETURN
  CASE n_f:   r_f := val; RETURN
  
  CASE n_bc:  r_b, r_c := val>>8, val & #xFF; RETURN
  CASE n_de:  r_d, r_e := val>>8, val & #xFF; RETURN
  CASE n_hl:  r_h, r_l := val>>8, val & #xFF; RETURN
  
  
  CASE n_ibc: putwb(r_b<<8 | r_c & #xFF, val); RETURN
  CASE n_ide: putwb(r_d<<8 | r_d & #xFF, val); RETURN
  CASE n_ihl: putwb(r_h<<8 | r_l & #xFF, val); RETURN
  
  CASE n_ix:  r_ix := val;      RETURN
  CASE n_iy:  r_iy := val;      RETURN
  CASE n_sp:  r_sp := val;      RETURN
  CASE n_isp: putwb(r_sp, val); RETURN
}

AND zerror(pc, n) = VALOF
{ writef("*nUnimplemented instruction at %n: ", pc)
  //pflag := TRUE
  //n := prz80instr(pc)
  newline()
  RESULTIS n
}

AND notyet(pc) = VALOF
{ // This prints an error message and
  // returns the size of the offending insruction.
  // It is only called when trying to execute an insruction.
  LET size = 0
  writef("*nInstruction is not yet implemented at %n: ", pc)
  FOR i = 0 TO 3 DO writef("%x2 ", getbb(pc+i))
  writef("   ")
  //r_pc := pc+
  size := prz80instr(pc)
  newline()
  abort(999)
  singlestep := TRUE
  tracing := TRUE
  RESULTIS size
}


AND z80ADCrr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, x_adc(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80ADC16rr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, x_adc(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80ADDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, x_add(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADD16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, x_add16(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ANDrr(pc,r,s) = VALOF TEST pflag
THEN { writef("AND %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, x_and(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80XORr(pc,r) = VALOF TEST pflag
THEN { writef("XOR %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(n_a, x_xor(r_a, getr(r)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80XORrird(pc,r, s) = VALOF
{ LET d = getbb(pc+1)
  TEST pflag
  THEN { writef("XOR %s,(%s+%s)", r2str(r), r2str(s), d)
         RESULTIS 2
       }
  ELSE { notyet(pc)
         putr(n_a, x_xor(r_a, getr(r)))
         r_pc := r_pc+2
         RESULTIS 0
       }
}

AND z80CCP(pc) = VALOF TEST pflag
THEN { writef("CCP")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPL(pc) = VALOF TEST pflag
THEN { writef("CPL")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPr(pc, r) = VALOF TEST pflag
THEN { writef("CP %s", r2str(r))
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80DAA(pc) = VALOF TEST pflag
THEN { writef("DAA")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80DAAr(pc, r) = VALOF TEST pflag
THEN { writef("DAA %s", r2str(r))
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80DECr(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, getr(r)-1)
       RESULTIS 0
     }

AND z80DEC16r(pc, r, size) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS size
     }
ELSE { notyet(pc)
       putr(r, getr(r)-1)
       r_pc := pc+size
       RESULTIS 0
     }

AND z80DI(pc) = VALOF TEST pflag
THEN { writef("DI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80EI(pc) = VALOF TEST pflag
THEN { writef("EI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80EXX(pc) = VALOF TEST pflag
THEN { writef("EXX")
       RESULTIS 1
     }
ELSE { notyet(pc)
       x_exx(pc)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80EXrr(pc, r, s) = VALOF TEST pflag
THEN { writef("EX %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { LET t = ?
       // This implements one of
       // EB:      EX DE,HL
       // 08:      EX AF,AF'
       // E3:      EX (SP),HL
       // DD E3:   EX (SP),IX
       // FD E3:   EX (SP),IY     
       notyet(pc)
       t := getr(s); putr(s, getr(r)); putr(r, t)
       TEST s=n_ix | s=n_iy THEN r_pc := r_pc+2
                            ELSE r_pc := r_pc+1
       RESULTIS 0
     }
     
AND z80HALT(pc) = VALOF TEST pflag
THEN { writef("HALT")
       RESULTIS 1
     }
ELSE { notyet(pc)
       writef("*nHALT instructions encountered*n")
       abort(999)
       r_pc := pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80INCr(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80INC16r(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80LDinnr(pc, r) = VALOF
{ LET nn = getwb(pc+1)
  TEST pflag THEN { writef("LD (%n),%s", nn, r2str(r))
                    RESULTIS 3
		  }
             ELSE { notyet(pc)
                    putwb(nn, getr(r))
                    RESULTIS 0
		  }
}

AND z80LD16innr(pc, r) = VALOF
{ LET nn = getwb(pc+1)
  TEST pflag THEN { writef("LD (%n),%s", nn, r2str(r))
                    RESULTIS 3
                  }
             ELSE { notyet(pc)
                    putwb(nn, getr(r))
                    RESULTIS 0
		  }
}

AND z80LDirdn(pc, r, d) = VALOF
{ // Instruction: LD (IX+d),n  or  LD (IY+d),n
  // Code:        DD 36  d  n  or  FD 36  d  n
  // r=n_ix or n_iy
  LET n = getbb(pc+3)
  TEST pflag
  THEN { TEST d<0
         THEN writef("LD (%s-%n),%n", r2str(r), -d, n)
         ELSE writef("LD (%s+%n),%n", r2str(r),  d, n)
         RESULTIS 4
       }
  ELSE { LET a = getr(r)+d
         notyet(pc)
         putbb(a, n)
	 r_pc := pc+4
	 RESULTIS 0
       }
}

AND z80LDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, getr(s))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80LD16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { notyet(pc)
       putr(r, getr(s))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80NOP(pc) = VALOF TEST pflag
THEN { writef("NOP")
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80POP16r(pc, r, size) = VALOF TEST pflag
THEN { writef("POP %s", r2str(r))
       RESULTIS size
     }
ELSE { putr(r, getwb(r_sp))
       r_sp := r_sp+2
       r_pc := pc+size
       RESULTIS 0
     }

AND z80RRA(pc) = VALOF TEST pflag
THEN { writef("RRA")
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80RRCA(pc) = VALOF TEST pflag
THEN { writef("RRCA")
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80RET(pc) = VALOF TEST pflag
THEN { writef("RET")
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETcc(pc, cc) = VALOF TEST pflag
THEN { writef("RET")
       RESULTIS 1
     }
ELSE { notyet(pc)
       TEST evalcc(cc)
       THEN r_pc := getwb(r_sp)
       ELSE r_pc := pc+1
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETN(pc) = VALOF TEST pflag
THEN { writef("RETN")
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80ADDrn(pc,r) = VALOF TEST pflag
THEN { writef("ADD %s,%n", r2str(r), getbb(pc+1))
       RESULTIS 2
     }
ELSE { notyet(pc)
       putr(r, getr(r) + getbb(pc+1))
       r_pc :=r_pc+2 & #xFFFF
       RESULTIS 0
     }

AND z80SUBrird(pc, r, s) = VALOF
{ LET d = getsbb(pc+2)
  TEST pflag
  THEN { LET rstr = r2str(r)
         LET sstr = r2str(s)
         TEST d<0
         THEN writef("SUB %s,(%s%n", rstr, sstr, d)
         ELSE writef("SUB %s,(%s+%n", rstr, sstr, d)
         RESULTIS 3
       }
  ELSE { LET res = getr(r)+getr(s)
         notyet(pc)
         putr(r, res)
         r_pc := pc+3
         RESULTIS 0
       }
}

AND z80SBC16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { LET res = x_sbc(getr(r), getr(s), r_fc)
       notyet(pc)
       res := res-1
       putr(r, res)
       r_pc := pc+2
       RESULTIS 0
     }

AND z80ANDn(pc) = VALOF TEST pflag
THEN { writef("AND %n", getbb(pc+1))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80ANDr(pc,r) = VALOF TEST pflag
THEN { writef("AND %s", r2str(r))
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80BITbr(pc, b, r) = VALOF TEST pflag
THEN { writef("BIT %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPD(pc) = VALOF TEST pflag
THEN { writef("CPD")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPDR(pc) = VALOF TEST pflag
THEN { writef("CPDR")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPI(pc) = VALOF TEST pflag
THEN { writef("CPI")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPIR(pc) = VALOF TEST pflag
THEN { writef("CPIR")
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80CPn(pc) = VALOF TEST pflag
THEN { writef("CP %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80DJNZo(pc) = VALOF TEST pflag
THEN { writef("DJNZo %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80IMk(pc, k) = VALOF TEST pflag
THEN { writef("IM %n", k)
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80IND(pc) = VALOF TEST pflag
THEN { writef("IND")
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80INDR(pc) = VALOF TEST pflag
THEN { writef("INDR")
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80INI(pc) = VALOF TEST pflag
THEN { writef("INI")
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80INIR(pc) = VALOF TEST pflag
THEN { writef("INIR")
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80INCird(pc, r) = VALOF TEST pflag
THEN { writef("INC "); prird(r, getsbb(pc+2))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80INrr(pc, r, s) = VALOF TEST pflag
THEN { writef("IN %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80LDrn(pc, r) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), getbb(pc+1))
       RESULTIS 2
     }
ELSE { notyet(pc)
       putr(r, getbb(pc+1))
       r_pc := r_pc+2 & #xFFFF
       RESULTIS 2
     }

AND z80LD16rnn(pc, r) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), getswb(pc+1))
       RESULTIS 3
     }
ELSE { putr(r, getswb(pc+1))
       r_pc := r_pc+3 & #xFFFF
       RESULTIS 0
     }

AND z80ORr(pc, r) = VALOF TEST pflag
THEN { writef("OR %s", r2str(r))
       RESULTIS 1
     }
ELSE { notyet(pc)
       r_a := r_a | getr(r)
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80ORird(pc, r) = VALOF
{ LET d = getsbb(pc+2)
  TEST pflag
  THEN { TEST d<0
         THEN writef("OR (%s%n)", r2str(r), d)
         ELSE writef("OR (%s+%n)", r2str(r), d)
         RESULTIS 3
       }
  ELSE { notyet(pc)
         r_a := r_a | getr(r)
         r_pc := r_pc+3 & #xFFFF
         RESULTIS 0
       }
}

AND z80OUTrr(pc, r, s) = VALOF TEST pflag
THEN { writef("OUT %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80PUSHr(pc, r) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80PUSH16r(pc, r, size) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80RRCr(pc, r) = VALOF TEST pflag
THEN { writef("RRC %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80RESbr(pc, b, r) = VALOF TEST pflag
THEN { writef("RES %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80RLCr(pc, r) = VALOF TEST pflag
THEN { writef("RLC %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80RLr(pc, r) = VALOF TEST pflag
THEN { writef("RL %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80RRr(pc, r) = VALOF TEST pflag
THEN { writef("RR %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET w = getr(r)
       LET c = r_fc
       notyet(pc)
       r_fc := w&1
       w := w>>1
       IF c DO w := w + #x80
       putr(r, w)
       r_fval := w
       r_fhval := w
       r_fn := 0
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80SLAr(pc, r) = VALOF TEST pflag
THEN { writef("SLA %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80SRAr(pc, r) = VALOF TEST pflag
THEN { writef("SRA %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80SRLr(pc, r) = VALOF TEST pflag
THEN { writef("SRL %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET w = getr(r)
       notyet(pc)
       r_fc := w&1
       w := (w>>1)&#x7F
       r_fval := 0
       r_fhval := w
       r_fn := 0
       putr(r, w)
       r_pc := r_pc+2
       RESULTIS 0
     }


AND z80SETbr(pc, b, r) = VALOF TEST pflag
THEN { writef("SET %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }


AND z80ADCrird(pc, r, x) = VALOF TEST pflag
THEN { writef("ADC %s,", r2str(r)); prird(x, getsbb(pc+2))
       RESULTIS 3
     }
ELSE { LET a = getr(x) + getsbb(pc+2)
       notyet(pc)
       putr(r, getr(r) + getbb(a) + r_fc)
       r_pc := r_pc+3
       RESULTIS 0
     }



AND z80ADDrixd(pc,r,d) = VALOF TEST pflag
THEN { writef("ADC %s,(IX+%n)", r2str(r), d)
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }


AND z80ADDriyd(pc,r,d) = VALOF TEST pflag
THEN { writef("ADC %s,(IX+%n)", r2str(r), d)
       RESULTIS 1
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

AND z80LDirdr(pc, r, d, s) = VALOF TEST pflag
THEN { writef("LD "); prird(r, d); writef(",%s", r2str(s)); 
       RESULTIS 3
     }
ELSE { LET a = getr(r)+d
       putbb(a, getr(s))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80LDrird(pc, r, s, d) = VALOF TEST pflag
THEN { writef("LD %s,", r2str(r)); prird(s, d)
       RESULTIS 3
     }
ELSE { LET a = getbb(getr(s)+d)
//db("Loading reg %s with (%s+%n) value %n*n", r2str(r), r2str(s), d, a)
       putr(r, a)
       r_pc := pc+3
       //db("Returning from z80LDrird: r_pc=%n*n", r_pc)
       RESULTIS 0
     }

AND z80LDrinn(pc, r) = VALOF
{ LET w = getwb(pc+1)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), w)
         RESULTIS 3
       }
  ELSE { notyet(pc)
         putr(r, getwb(w))
         r_pc := pc+3 & #xFFFF
         RESULTIS 0
       }
}

AND z80LD16rinn(pc, r) = VALOF
{ LET w = getwb(pc+1)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), w)
         RESULTIS 3
       }
  ELSE { notyet(pc)
         putr(r, getwb(w))
         r_pc := r_pc+3 & #xFFFF
         RESULTIS 0
       }
}

AND z80LDIr(pc, r) = VALOF
{ TEST pflag
  THEN { writef("LDI %s", r2str(r))
         RESULTIS 3
       }
  ELSE { notyet(pc)
         putr(r, r_i)
         r_pc := r_pc+2 & #xFFFF
         RESULTIS 0
       }
}

AND z80LDRr(pc, r) = VALOF
{ TEST pflag
  THEN { writef("LDI %s", r2str(r))
         RESULTIS 3
       }
  ELSE { notyet(pc)
         putr(r, r_r)
         r_pc := r_pc+2 & #xFFFF
         RESULTIS 0
       }
}

AND z80RLCird(pc, r, d) = VALOF TEST pflag
THEN { writef("RLC "); prird(r, d)
       RESULTIS 4
     }
ELSE { r_pc := pc+notyet(pc)
       RESULTIS 0
     }

// Arithemetic and logical functions

AND evalcc(cc) = VALOF SWITCHON cc INTO
{ DEFAULT:    writef("evalcc: Bad cc=%n*n", cc)
              abort(999)
	      RESULTIS FALSE
  CASE cc_m:  RESULTIS r_fval<0
  CASE cc_c:  RESULTIS r_fc>0
  CASE cc_nc: RESULTIS r_fc=0
  CASE cc_nz: RESULTIS r_fval~=0
  CASE cc_p:  RESULTIS parity(r_fval)=0
  CASE cc_pe: RESULTIS parity(r_fval)=0
  CASE cc_po: RESULTIS parity(r_fval)=0
  CASE cc_z:  RESULTIS r_fval=0

}

AND x_adc(a, b) = VALOF
{ LET res = a + b
  IF r_fc DO res := res + 1
  RESULTIS res
}

AND x_adc16(a, b) = VALOF
{ LET res = a + b
  IF r_fc DO res := res + 1
  RESULTIS res
}

AND x_add(a, b) = VALOF
{ LET res = a + b
  RESULTIS res
}

AND x_xor(a, b) = VALOF
{ LET res = (a XOR b) &#xFFFF
  RESULTIS res
}

AND x_add16(a, b) = VALOF
{ LET res = a + b
  RESULTIS res
}

AND x_exx(pc) BE
{ LET t = ?
  t := r_a1; r_a1:=r_a; r_a:=t 
  t := r_f1; r_f1:=r_f; r_f:=t 
  t := r_b1; r_b1:=r_b; r_b:=t 
  t := r_c1; r_c1:=r_c; r_c:=t 
  t := r_d1; r_d1:=r_d; r_d:=t 
  t := r_e1; r_e1:=r_e; r_e:=t 
  t := r_h1; r_h1:=r_h; r_h:=t 
  t := r_l1; r_l1:=r_l; r_l:=t
}

AND db(mes, a,b,c,d,e) BE
{ writef("*ndb: "); writef(mes, a,b,c,d,e)
  writef("*nr_pc=%n pptr=%n faultcode=%n singlestep=%n brkstep=%n mem=%n pflag=%n",
            r_pc,   pptr,   faultcode,   singlestep,   brkstep,   mem,   pflag)
  writef("*n  100: ")
  FOR a = 100 TO 118 DO writef(" %x2", getbb(a))
  FOR i = 0 TO 19 DO
  { IF i MOD 10 = 0 DO writef("*n%i5: ", pptr+i)
    writef(" %x4", getww(pptr+i))
  }
  newline()
  indentpos := 0
  prstate()
  abort(1000)
}
