/*
Written by M. Richards  (c) 11 Sep 2016

History

10/01/2025
Minor changes


Typical usage when run under rastsys:

> raster                   Cause the next CLI command to generate raster data
> bcpl com/bcpl.b to junk  This creates the file RASTER

The RASTER file starts something like:

F1034504           The total number of Cintcode instructions obeyed
M734000            Highest byte address accessed
K1000              Set kval to 1000 instruction per raster line
S8                 Set sval to 8 bytes per unit
W10B3W1345B1N      10 white 3 black 1345 white 1 black newline
W13B3W12B2N        etc
...
Note that F1034504 M734000 are written to the start of the raster file
after all the raster data has been written.

The rastar data can be converted to a PostScript image using the
rast2ps command.  It can also be converted to a .wav file using
rast2wav.

If the bits option is given, raster will generate a bit stream
file derived from the fifth bit of every memory reference. This can
be converted to a .wav file by a command sequence such as the following:


rastsys
raster bits
bcpl com/bcpl.b to junk
bits2wav to junk.wav

This is an approximation of how sound was generated by EDSAC 2 in
1963.  Edsac 2 sent the 5th bit of the effective address suitably
filtered to the loud speaker every time a machine instruction was
executed. The sound generated was very useful to the user.

History

10/01/2020
Made minor changes.
*/


SECTION "RASTER"

GET "libhdr"

LET start() = VALOF
{ LET argv = VEC 40
  AND outstream = 0
  AND tallyv = rootnode!rtn_tallyv
  AND oldout = output()
  AND kval, sval = 1000, 8
  AND tofilename = 0            // The default filename
  AND bits = FALSE

  // Test whether rastering is available.
  UNLESS sys(Sys_setraster, 3, 0) DO
  { writes("Rastering is not available, use rastsys instead.*n")
    RESULTIS 0
  }

  UNLESS rdargs("KVAL=-k/N,SVAL=-s/N,TO/K,BITS/S,-h=HELP/S",argv, 40) DO
  { writes("Bad arguments for RASTER*n")
    RESULTIS 20
  }

  IF argv!0 DO kval := !(argv!0)   // KVAL=-k/N Num of instruction per line
  IF argv!1 DO sval := !(argv!1)    // SVAL=s/N  Num of byte addrs per mem unit
  IF argv!2 DO tofilename := argv!2 // TO/K
  bits := argv!3                    // BITS/S
  
  IF argv!4 DO                      // -h=HELP/S
  { writes("*nTypical usage:*n*n")
    writes("    rastersys*n")
    writes("    raster*n")
    writes("    origbcpl com/origbcpl.b to junk*n")
    writes("    rast2ps to origbcpl.ps incl origbcplps.h*n")
    writes("    ctrl-c*n")
    writes("    ps2pdf origbcpl.ps origbcpl.pdf*n")
    writes("    okular origbcpl.pdf*n*n")

    writes("    raster kval 1000 sval 12 to rastdata*n")
    writes("    bcpl com/bcpl.b to junk*n")
    writes("    raster2wav to origbcpl.wav*n*n")

    writes("    raster bits*n")
    writes("    bcpl com/bcpl.b to junk*n")
    writes("    bits2wav to bcpl.wav*n*n")

    writes("*nRemember to use rastsys (NOT cintsys)*n")
    RESULTIS 0
  }

  sys(Sys_setcount, maxint)  // Select the slow interpreter
  
  IF rootnode!rtn_fast DO
  { writef("Rastering is only available when using the slow interpreter*n")
    writef("Execute the CLI command: slow*n")
    RESULTIS 0
  }
  
//writef("args: %n %n %n %n*n", argv!0, argv!1, argv!2, argv!3)
//writef("tofilename=%s kval=%n sval=%n bits=%n*n",
//        tofilename, kval, sval, bits)

  TEST bits
  THEN { // Generate a file containing a bit stream of the
         // fifth bit of the address of every memory reference
	 UNLESS tofilename DO tofilename := "RASTER.bits"
         writef("*nBit stream data will be written to fle: %s*n",
	         tofilename)
         writef("The data is generated while the next CLI command is being executed*n")
	 writef("Each generated bit is the fifth bit of each memory address referenced*n")
         sys(Sys_setraster, 4, 1)  // Set bit stream generation
       }
  ELSE { UNLESS tofilename DO tofilename := "RASTER"
         writef("*nRaster data will be written to fle: %s*n", tofilename)
         writef("while the next CLI command is being executed*n")
         sys(Sys_setraster, 4, 0)  // Set raster generation (not bits)
         sys(Sys_setraster, 1, kval)
         sys(Sys_setraster, 2, sval)
       }

  // bits = 1
  // bits = 0 and kval and sval have been set.

  // Specify the raster file and cause the next CLI command to
  // generate raster or bit stream data.

  UNLESS sys(Sys_setraster, 0, tofilename) DO // Attempt to open raster file
  { writef("Trouble opening raster file: %s*n", tofilename)
    RESULTIS 0
  }

  //sawritef("raster: setting cli_tallyflag to TRUE*n")
  cli_tallyflag := TRUE  // Tell the CLI to start generating
                         // rastering or bit stream data.
  //abort(4455)
  RESULTIS 0
}
