/*
This is a compiler and interpreter for the language VSPL
implemented in MCPL

(c) Martin Richards 25 May 2000
*/


GET "mcpl.h"
 
MANIFEST   // Lexical tokens, parse tree operators and op-codes

Num=1, Name, String, True, False,
Valof, Fnap, Lv, Ind, Vecap,
Neg, Not, Mul, Div, Mod, Add, Sub,
Eq, Ne, Le, Ge, Lt, Gt, Lsh, Rsh, And, Or, Xor,
Comma, Fndef, Rtdef, Assign, Rtap, Resultis,
Test, If, Unless, While, Until, For, Return, Seq,
Let, Vec, Static, Statvec, Decl, Var,
Lparen, Rparen, Lsquare, Rsquare, Lcurly, Rcurly,
To, Do, Then, Else, Be, Eof, Semicolon,
Rtrn, Fnrn, Addr, Local, Lab, Data, Jt, Jf, Jump,
Ln, Lp, Llp, Ll, Laddr, Sp, Sl, Stind, Lres,
Entry, Stack, Printf, Sys, Halt
 
MANIFEST                         // Exception names
  E_fatal, E_lex, E_syn, E_trn, E_int, E_fin

GLOBAL
fatalerr:Ug, synerr, trnerr, errcount, errmax,
progstream, tostream, stdout,
mk1, mk2, mk3, mk4, mk5, mk6,
newvec, treep, treet, treevec,
optTokens, optTree, optCode, optTrace,

// Globals used in LEX
chbuf, charv, ch, nch, rch, lex, token, lexval,
wrchbuf, chcount, lineno,
dsw, declsyswords, namestart, nametable, lookup,
rdstrch, rdtag,

// Globals used in SYN
checkfor, rdprog, rdblockbody,
rnamelist, rstatlist, rname,
rdef, rncom, rcom,
formtree, plist,
rexplist, rdseq,
nexp, rexp, rbexp,
 
// Globals used in TRN and the interpreter
 
trnext:300, trprog, trcom, decldyn,
declstatnames, checkdistinct, addname, cellwithname,
trdecl, undeclare, jumpcond,
assign, load, fnbody, loadlist, transname,
dvec, dvece, dvecp, dvect,
comline, procname, resultlab, ssp,
outf, outfn, outfl, outfs, outentry,
outlab, outvar, outstatvec, outstring, opstr, hasOperand,
mem, memt, regs,
codev, codep, codet, datav, datap, datat, stack, stackt,
labv, refv, labmax, putc, putd, putref,
setlab, nextlab, labnumber, resolvelabels,
interpret, printf

MANIFEST                         //  Selectors
H1=0, H2=1, H3=2, H4=3, H5=4, H6=5,
Nametablesize = 541,
Nametableupb  = Nametablesize-1,
C_tab         =   9,
C_newline     =  10


FUN start : =>
{ LET treesize = 0
  LET memsize = 0
  LET argv = VEC 50
  LET argform = "PROG/A,TO=-o/K,TOKENS=-l/S,TREE=-p/S,CODE=-c/S,TRACE=-t/S"
  stdout := output()

  errmax   := 3
  errcount := 0

  treevec, labv, refv, mem := 0, 0, 0, 0
  progstream, tostream := 0, 0
   
  writef "\nVSPL (29 May 2000) MCPL Version\n"

  IF rdargs(argform, argv, 50)=0 DO fatalerr "Bad arguments\n" 

  treesize := 10000
  memsize  := 50000

  progstream := findinput(argv!0)      // PROG
  IF progstream=0 DO fatalerr("Trouble with file %s\n", argv!0)

  selectinput(progstream)
 
  IF argv!1                            // TO      -o
  DO { tostream := findoutput(argv!1)
       IF tostream=0 DO fatalerr("Trouble with code file %s\n", argv!1)
     }

  optTokens := argv!2                  // TOKENS  -l
  optTree   := argv!3                  // TREE    -p
  optCode   := argv!4                  // CODE    -c
  optTrace  := argv!5                  // TRACE   -t

  treevec := getvec treesize
  mem     := getvec memsize
  memt    := memsize
  labv    := getvec 1000
  refv    := getvec 1000
  labmax  := 1000

  UNLESS treevec AND mem AND labv AND refv DO
     fatalerr "Insufficient memory\n"
   
  UNLESS tostream DO tostream := stdout
  selectoutput tostream

  mk_init treesize

  { LET tree = formtree()            // Perform Syntax Analysis

    IF optTokens RAISE E_fin

    IF optTree DO { writes "Parse Tree\n"
                    plist(tree, 0, 20)
                    newline()
                  }
  
    IF errcount RAISE E_fin

    regs  := 10
    codev := 100
    codep := codev
    codet := 10000
    datav := codet
    datap := datav
    datat := memt

    trprog tree                     // Translate the tree

    stack := datap
    stackt := memt

    IF errcount RAISE E_fin

    { LET rv = @mem!regs
      LET sv = @mem!stack
      rv!0 := 0        // result register
      rv!1 := stack    // p pointer
      rv!2 := stack+2  // sp
      rv!3 := codev    // pc
      rv!4 := Maxint   // count

      sv!0, sv!1, sv!2 := 0, 0, 0
 
      { LET ret = interpret(regs, mem)    // Execute the interpreter
        IF ret DO writef("Return code %d\n", ret)
        writef("\nInstructions executed: %d\n", Maxint-rv!4)
      }
    }
  }
} HANDLE
  : E_lex|E_syn, mess, a => writef("Syntax error near line %d: ",
                                                          lineno)
                            writef(mess, a); newline()
  : E_fatal              => writef "Compilation aborted\n"
  : E_fin                =>
  .

  mk_close()
  IF treevec       DO freevec treevec
  IF mem           DO freevec mem
  IF labv          DO freevec labv
  IF refv          DO freevec refv
  IF progstream    DO { selectinput progstream; endread()  }
  IF tostream      DO { selectoutput tostream
                        UNLESS tostream=stdout DO  endwrite() }

  selectoutput stdout
  RETURN errcount=0 -> 0, 20
.

/************************************************************
*
*             Lexical Analyser Functions and Data
*
*  lex_init()        initialise the lexical analyser
*  lex()             set token and possibly lexval to next
*  lineno            line number of current character
*
*  dsw(token, str)   put a reserved word into the symbol table
*  declsyswords()    put all the reserved word into the table
*  rdtag()           read an identifier into charv
*  rdnumber()        read a decimal number, returns it value
*  rdstring()        read a VSPL string, return a string node
*  rch()             update ch and nch
*  wrchbuf()         write the contents of the input buffer
*
************************************************************/


FUN lex_init : =>
  // Set up the symbol table
  charv := mkvec(256/Bytesperword)
  chbuf := mkvec(64/Bytesperword)
  UNLESS charv AND chbuf DO fatalerr("More workspace needed")
  FOR i = 0 TO 63 DO chbuf%i := 0
  lineno := 1
  nametable_init()
  declsyswords()
  rch()
  rch()
.
FUN lex : =>
{ MATCH (ch, nch)

  : '\n' | '\f'         => lineno++; rch();         LOOP

  : '\t' | '\r' | ' '   => rch();                   LOOP

  : 'A'..'Z' | 'a'..'z' =>
                lexval := lookup(rdtag());
                token := lexval!H1;                 RETURN
 
  : '0'..'9' => lexval := rdnumber(); token := Num; RETURN
 
  : '"'      => lexval := rdstring(); token := String; RETURN
 
  : '\''     => rch()
                lexval := rdstrch()
                UNLESS ch='\'' DO synerr("Bad character constant")
                token := Num;       rch();          RETURN

  : '{'      => token := Lcurly;    rch();          RETURN
  : '}'      => token := Rcurly;    rch();          RETURN
  : '['      => token := Lsquare;   rch();          RETURN
  : ']'      => token := Rsquare;   rch();          RETURN
  : '('      => token := Lparen;    rch();          RETURN
  : ')'      => token := Rparen;    rch();          RETURN 
  : '!'      => token := Ind;       rch();          RETURN
  : '@'      => token := Lv;        rch();          RETURN
  : '+'      => token := Add;       rch();          RETURN
  : '-'      => token := Sub;       rch();          RETURN
  : ','      => token := Comma;     rch();          RETURN
  : ';'      => token := Semicolon; rch();          RETURN
  : '&'      => token := And;       rch();          RETURN
  : '|'      => token := Or;        rch();          RETURN
  : '='      => token := Eq;        rch();          RETURN
  : '*'      => token := Mul;       rch();          RETURN
  : '^'      => token := Xor;       rch();          RETURN
  : '/', '/' => rch() REPEATUNTIL ch='\n' OR ch=Endstreamch
                LOOP
  : '/'      => token := Div;       rch();          RETURN
 
  : '~', '=' => token := Ne;        rch(); rch();   RETURN
  : '~'      => token := Not;       rch();          RETURN

  : '<', '=' => token := Le;        rch(); rch();   RETURN
  : '<', '<' => token := Lsh;       rch(); rch();   RETURN
  : '<'      => token := Lt;        rch();          RETURN
 
  : '>', '=' => token := Ge;        rch(); rch();   RETURN
  : '>', '>' => token := Rsh;       rch(); rch();   RETURN
  : '>'      => token := Gt;        rch();          RETURN
 
  : ':', '=' => token := Assign;    rch(); rch();   RETURN
  : ':'      => synerr("'=' expected after ':'");   RETURN
 
  :          => UNLESS ch=Endstreamch DO
                { LET badch = ch
                  ch := ' '
                  synerr("Illegal character %x2", badch)
                }
                token := Eof
                RETURN
  :  ?       => RAISE (E_lex, "Bad character %c", ch)
  .
} REPEAT
.
FUN rdtag : =>
  LET len = 0
  WHILE 'a'<=ch<='z' OR 'A'<=ch<='Z' OR '0'<=ch<='9' OR  ch='_' DO
  { IF len>255 DO synerr("Name too long")
    charv%len++ := ch
    rch()
  }
  charv%len := 0
  RETURN charv
. 
FUN rdnumber : => 
  LET res = 0
  WHILE '0'<=ch<='9' DO { res := 10*res + ch - '0'; rch() }
  RETURN res
.
FUN rdstring : =>
  LET len = 0
  rch()

  { IF ch='"'  OR ch=Endstreamch BREAK
    charv%len++ := rdstrch()
  } REPEAT

  charv%len := 0
  rch()

  LET w = mkvec((len+1)/Bytesperword + 1)
  w!0 := String
  copystr(charv, @ w!1)
  RETURN w
.
FUN rdstrch : =>
  LET res = ch
  IF ch='\n' OR ch='\f' DO
  { lineno++
    synerr("Unescaped newline character")
  }
  IF ch='\\' DO
  { rch()
    MATCH ch
    : '\\' | '\'' | '"' => res := ch
    : 't'  | 'T' =>        res := C_tab
    : 'n'  | 'N' =>        res := C_newline
    :            =>        synerr("Bad string or character constant")
    .
  }
  rch()
  RETURN res
.
FUN dsw : word, tok => LET t = lookup(word); t!H1 := tok
.
FUN declsyswords : =>
{ dsw("be", Be);             dsw("do", Do);         dsw("else", Else)
  dsw("false", False);       dsw("if", If);         dsw("for", For)
  dsw("let", Let);           dsw("mod", Mod);       dsw("printf", Printf)
  dsw("resultis", Resultis); dsw("return", Return); dsw("static", Static)
  dsw("sys", Sys);           dsw("test", Test);     dsw("to", To)
  dsw("true", True);         dsw("then", Then);     dsw("valof", Valof)
  dsw("vec", Vec);           dsw("unless", Unless); dsw("until", Until)
  dsw("while", While)  
  namestart := lookup "start"
} 
.
FUN rch : =>
  ch := nch
  nch := rdch()
  chbuf%(chcount++&63) := ch
.
FUN wrchbuf : =>
  writes("\n...")
  FOR p = chcount TO chcount+63 DO
  { LET k = chbuf%(p&63)
    IF 0<k<255 DO wrch(k)
  }
  newline()
.
//********** End of Lexical Analyser code *********************


 
/************************************************************
*
*            Symbol Table Functions and Data
*
*    nametable_init()
*    nametable_close()
*    hash str          returns a 31 bit hash of the string str
*    lookup str        return a node for identifier str, creating
*                      one if necessary
*
************************************************************/


FUN nametable_init : =>
  nametable := mkvec Nametablesize
  UNLESS nametable DO fatalerr("More workspace needed")
  FOR i = 0 TO Nametableupb DO nametable!i := 0
.
FUN nametable_close : => 
  freevec nametable
  nametable := 0
.
FUN hashstr : str =>
  LET hashval = 0
  WHILE %str DO hashval := (13*hashval XOR %str++) & #xFF_FFFF
  TEST hashval>=0 THEN RETURN hashval
                  ELSE RETURN hashval>>1
.
FUN lookup : str =>
  LET hash = hashstr str
  LET i    = hash MOD Nametablesize
  LET w    = nametable!i
 
  // A namenode has form [ token, hashchain, hash, chars ... ]
  // where token is either Name or a reserved word token and 
  // hash is a 31 bit hash of the identifier (to reduce the number
  // string comparisons).

  WHILE w MATCH w
          : [tok, chain, whash, chars] =>
              IF hash=whash AND eqstr(str, @chars) RETURN w
              w := chain
          .

  // matching node not found, so make one.
 
  w := mkvec((lenstr str + 1)/ Bytesperword + 3)
  MATCH w
  : [tok, chain, whash, chars] =>
      tok, chain, whash, nametable!i := Name, nametable!i, hash, w
      copystr(str, @chars)
  .
  RETURN w
.
FUN clearhints : =>
  FOR i = 0 TO Nametableupb DO
  { LET p = nametable!i
    WHILE p MATCH p
            : [tok, chain, hint, chars] => hint, p := 0, chain
            .
  }
.

//********** End of Id Table code ***************************

 
/************************************************************
*
*            Space Allocation Functions and Data
*
*   mk_init upb
*   mk_close
*   mkvec(upb)
*   mk1(op)
*   mk2(op, a)
*   mk3(op, a, b)
*   mk4(op, a, b, c)
*   mk5(op, a, b, c, d)
*   mk6(op, a, b, c, d, e)
*  
*
************************************************************/

FUN mk_init : upb =>
  treevec := getvec upb 
  UNLESS treevec RAISE (E_fatal, "Unable to allocate work space")
  treet := treevec + upb
  treep := treet
.
FUN mk_close : => IF treevec DO { freevec treevec; treevec := 0 }
.
FUN mkvec : upb =>
  treep := @ treep!-(upb+1)
  IF treep<treevec RAISE (E_fatal, "More space needed")
  RETURN treep
.
FUN mk1 : op => 
  LET p = mkvec 0
  !p := op
  RETURN p
.
FUN mk2 : op, a =>
  LET p = mkvec 1
  !p, p!1 := op, a
  RETURN p
.
FUN mk3 : op, a, b =>
  LET p = mkvec 2
  !p, p!1, p!2 := op, a, b
  RETURN p
.
FUN mk4 : op, a, b, c =>
  LET p = mkvec 3
  !p, p!1, p!2, p!3 := op, a, b, c
  RETURN p
.
FUN mk5 : op, a, b, c, d =>
  LET p = mkvec 4
  !p, p!1, p!2, p!3, p!4 := op, a, b, c, d
  RETURN p
.
FUN mk6 : op, a, b, c, d, e =>
  LET p = mkvec 5
  !p, p!1, p!2, p!3, p!4, p!5 := op, a, b, c, d, e
  RETURN p
.

//********** End of Space Allocation code *******************



/************************************************************
*
*           String Functions
*
*    eqstr(s1, s2)     returns TRUE if equal
*    lenstr s          return length not including final 0
*    copystr(s1, s2)   copies s1 to s2
*
************************************************************/

FUN eqstr : s1, s2 =>
{ LET ch1=%s1++, ch2=%s2++
  IF ch1=0 AND ch2=0 RETURN TRUE
  IF ch1 ~= ch2      RETURN FALSE
} REPEAT
.
FUN lenstr : s =>
  LET len = 0
  WHILE %s++ DO len++
  RETURN len
.
FUN copystr : s1, s2 =>
{ LET ch = %s1++
  %s2++ := ch
  IF ch=0 RETURN
} REPEAT
.
//********** End String Functions code **********************



FUN formtree : =>

  lex_init()
  lex()

  IF optTokens DO            // For debugging lex.
  { IF token=Eof RETURN 0
    writef("token = %3d %s", token, opstr(token))
    IF token=Num DO writef("        %d",     lexval)
    IF token=Name DO writef("       %s",     charv)
    IF token=String DO writef("     \"%s\"", charv)
    newline()
    lex()
  } REPEAT

  LET res = 0

  { UNLESS token=Eof DO
    { res := rdprog()
      BREAK
    } HANDLE : E_syn | E_lex => LOOP .
  } REPEAT

  UNLESS token=Eof DO fatalerr("Incorrect termination")
  RETURN res
.
FUN fatalerr : mess, a =>
  writef "\nFatal error:  "
  writef(mess, a)
  writes "\nCompilation aborted\n"
  errcount++
  RAISE E_fin
.
FUN synerr : mess, a =>
  writef("\nError near line %d:  ", lineno)
  writef(mess, a)
  wrchbuf()
  errcount++
  IF errcount >= errmax DO fatalerr("\nCompilation aborted\n")

  // Skip the rest of the input line 
  UNTIL ch='\n' OR ch=Endstreamch DO rch()
  lex()
  RAISE E_syn
.
FUN checkfor
: tok, mes => UNLESS token=tok RAISE (E_syn, mes)
              lex()
.
FUN rdprog : =>
  LET ln = lineno

  MATCH token

  : Eof    => RETURN 0

  : Static => LET d = ?
              lex()
              d := mk3(Static, rstatlist(), ln)
              RETURN  mk3(Decl, d, rdprog())

  : Let    => LET n=0, args=0
              lex()
              n := rname()
              checkfor(Lparen, "'(' missing")
              IF token=Name DO args := rnamelist()
              checkfor(Rparen, "')' missing")
 
              IF token=Be DO
              { LET d = mk5(Rtdef, n, args, rncom(), ln)
                RETURN mk3(Decl, d, rdprog())
              }
 
              IF token=Eq DO
              { LET d = mk5(Fndef, n, args, nexp 0, ln)
                RETURN mk3(Decl, d, rdprog())
              }
 
              synerr("Bad procedure heading")

  :        => synerr("Bad outer level declaration\n")
  .
.
FUN rdblockbody : =>
{ LET res = 0
  LET op  = token

  MATCH op
 
  : Let | Vec =>
             LET n=0, e=0, ln= lineno
             lex()
             n := rname()
             TEST op=Let
             THEN { checkfor(Eq, "Missing '='")
                    e := rexp 0
                  }
             ELSE { checkfor(Lsquare, "Missing '['")
                    e := rexp 0
                    UNLESS e!H1=Num DO synerr("Bad 'vec' declaration")
                    checkfor(Rsquare, "Missing ']'")
                  }
             checkfor(Semicolon, "';' expected")
             RETURN mk5(op, n, e, rdblockbody(), ln)
           
  :     =>   RETURN rdseq()
  .
  HANDLE : E_syn => LOOP .
} REPEAT
.
FUN rdseq : =>
  LET a = rcom()
  IF token=Rcurly OR token=Eof RETURN a
  checkfor(Semicolon, "';' expected")
  RETURN mk3(Seq, a, rdseq())
.
FUN rnamelist : =>
  LET a = rname()
  UNLESS token=Comma RETURN a
  lex()
  RETURN mk3(Comma, a, rnamelist())
.
FUN rexplist : =>
  LET a = rexp 0
  UNLESS token=Comma RETURN a
  lex()
  RETURN mk3(Comma, a, rexplist())
.
FUN rstatlist : =>
  LET a = rname()
  IF token=Lsquare DO
  { LET b = nexp 0 
    UNLESS b!H1=Num DO synerr("Number expected")
    checkfor(Rsquare, "']' expected")
    a := mk3(Statvec, a, b)
  }
  UNLESS token=Comma RETURN a
  lex()
  RETURN mk3(Comma, a, rstatlist())
.
FUN rname : =>
  LET a = lexval
  checkfor(Name, "Name expected")
  RETURN a
.
FUN rbexp : =>
  LET a=0, op=token, ln=lineno

  MATCH op
 
  : True | False | Name | String =>
                a := lexval
                lex()
                RETURN a
 
  : Num      => a := mk2(Num, lexval)
                lex()
                RETURN a
 
  : Printf | Sys =>
                lex()
                checkfor(Lparen, "'(' missing")
                a := 0
                UNLESS token=Rparen DO a := rexplist()
                checkfor(Rparen, "')' missing")
                RETURN mk3(op, a, ln)

  : Lparen   => a := nexp 0
                checkfor(Rparen, "')' missing")
                RETURN a
 
  : Valof    => RETURN mk2(Valof, rncom())
 
  : Ind | Lv => RETURN mk2(op, nexp 8)
 
  : Add      => RETURN nexp 6
 
  : Sub      => a := nexp 6
                TEST a!H1=Num THEN a!H2 := - a!H2
                              ELSE a := mk2(Neg, a)
                RETURN a
 
  : Not      => RETURN mk2(Not, nexp 3)
  :          => synerr "Error in expression"
  .
.
FUN nexp : n =>  lex();
                 RETURN rexp n
.
FUN rexp : n =>
  LET a = rbexp()

  { LET op=token, ln=lineno
    MATCH (op, n)
 
    : Lparen              => lex()
                             LET b = 0
                             UNLESS token=Rparen DO b := rexplist()
                             checkfor(Rparen, "')' missing")
                             a := mk4(Fnap, a, b, ln)
 
    : Lsquare             => LET b = nexp(0)
                             checkfor(Rsquare, "']' missing")
                             a := mk3(Vecap, a, b)

    : Mul | Div | Mod,  <7 => a := mk3(op, a, nexp 7)
    : Add | Sub,        <6 => a := mk3(op, a, nexp 6)
    : Lsh | Rsh,        <5 => a := mk3(op, a, nexp 5)
    : Eq | Le | Lt | Ne | Ge | Gt,
                        <4 => a := mk3(op, a, nexp 4)
    : And,              <3 => a := mk3(op, a, nexp 3)
    : Or,               <2 => a := mk3(op, a, nexp 2)
    : Xor,              <1 => a := mk3(op, a, nexp 1)
    :                      => RETURN a
    .
  } REPEAT
.
FUN rcom : =>
  LET op=token, ln=lineno
 
  MATCH token
 
  : Name | Num | Lparen | Ind | Sys | Printf =>
  // All tokens that can start an expression.
                 LET a = rexp 0
 
                 IF token=Assign DO
                 { UNLESS a!H1=Name OR a!H1=Vecap OR a!H1=Ind DO
                     synerr("Bad assigment statement")
                   RETURN mk4(Assign, a, nexp 0, ln)
                 }
 
                 IF a!H1=Fnap DO
                 { a!H1 := Rtap
                   RETURN a
                 }
 
                 UNLESS a!H1=Sys OR a!H1=Printf DO
                   synerr("Error in command")
                 RETURN a
 
  : Resultis  => RETURN mk3(op, nexp 0, ln)
 
  : If | Unless | While | Until =>
                 LET a = nexp 0
                 checkfor(Do, "'do' missing")
                 RETURN mk4(op, a, rcom(), ln)
 
  : Test      => LET a = nexp(0)
                 checkfor(Then, "'then' missing")
                 LET b = rcom()
                 checkfor(Else, "'else' missing")
                 RETURN mk5(Test, a, b, rcom(), ln)
 
  : For       => lex()
                 LET n = rname()
                 checkfor(Eq, "'=' expected")
                 LET a = rexp 0
                 checkfor(To, "'to' expected")
                 LET b = rexp 0
                 checkfor(Do, "'do' missing")
                 RETURN mk6(For, n, a, b, rcom(), ln)

  : Return    => lex()
                 RETURN mk2(op, ln)
 
  : Lcurly    => lex()
                 LET a = rdblockbody()
                 checkfor(Rcurly, "'}' expected")
                 RETURN a

  :           => synerr("Command expected")
  .
.
FUN rncom : => lex()
               RETURN rcom()
.
FUN opstr
: Assign    => "Assign"    : Add      => "Add"
: And       => "And"       : Be       => "Be"
: Comma     => "Comma"     : Data     => "Data"
: Decl      => "Decl"      : Div      => "Div"
: Do        => "Do"        : Else     => "Else"
: Entry     => "Entry"     : Eq       => "Eq"
: False     => "False"     : Fnap     => "Fnap"
: For       => "For"       : Fndef    => "Fndef"
: Fnrn      => "Fnrn"      : Ge       => "Ge"
: Gt        => "Gt"        : Halt     => "Halt"
: If        => "If"        : Ind      => "Ind"
: Jf        => "Jf"        : Jt       => "Jt"
: Jump      => "Jump"      : Lab      => "Lab"
: Laddr     => "Laddr"     : Lcurly   => "Lcurly"
: Le        => "Le"        : Let      => "Let"
: Ll        => "Ll"        : Llp      => "Llp"
: Ln        => "Ln"        : Lp       => "Lp"
: Lparen    => "Lparen"    : Lres     => "Lres"
: Lsh       => "Lsh"       : Lsquare  => "Lsquare"
: Lt        => "Lt"        : Lv       => "Lv"
: Mod       => "Mod"       : Mul      => "Mul"
: Name      => "Name"      : Ne       => "Ne"
: Neg       => "Neg"       : Not      => "Not"
: Num       => "Num"       : Or       => "Or"    
: Printf    => "Printf"    : Rcurly   => "Rcurly"
: Resultis  => "Resultis"  : Return   => "Return"
: Rparen    => "Rparen"    : Rsh      => "Rsh"
: Rsquare   => "Rsquare"   : Rtap     => "Rtap"
: Rtdef     => "Rtdef"     : Rtrn     => "Rtrn"
: Semicolon => "Semicolon" : Seq      => "Seq"
: Sl        => "Sl"        : Sp       => "Sp"
: Stack     => "Stack"     : Static   => "Static"
: Statvec   => "Statvec"   : String   => "String"
: Stind     => "Stind"     : Sub      => "Sub"
: Sys       => "Sys"       : Test     => "Test"
: Then      => "Then"      : To       => "To"
: True      => "True"      : Valof    => "Valof"
: Vecap     => "Vecap"     : Vec      => "Vec"
: Unless    => "Unless"    : Until    => "Until"
: While     => "While"     : Xor      => "Xor"
:           => "Unknown"
.

FUN plist : x, n, d =>
  LET s=0, size=0, ln=0
  LET v = TABLE [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]
  IF x=0 DO { writes("Nil"); RETURN  }
 
  MATCH x
  : [Num, k]    => writef("%d", k);   RETURN
  : [Name, ?, ?, chars]   => writef("%s", @chars);          RETURN
  : [String, chars] => writef("*"%s*"", @chars); RETURN

  : [For, n, a, b, c, l] => size, ln := 5, l

  : [Fndef | Rtdef | Let | Vec | Test,
          n, a, b, l]    => size, ln := 4, l

  : [Vecap | Mul | Div | Mod | Add | Sub |
     Eq | Ne | Lt | Gt | Le | Ge |
     Lsh | Rsh | And | Or | Xor |
     Comma | Seq | Decl | Statvec,
           a, b]         => size     := 3

  : [ Assign | Rtap | Fnap | If | Unless | While | Until,
           a, b, l]      => size, ln := 3, l

  : [Valof | Lv | Ind | Neg | Not,
           a]            => size     := 2

  : [Printf | Sys | Static | Resultis,
           a, l]         => size, ln := 2, l

  : [True | False]       => size     := 1
  :                      => size     := 1
  .
 
  IF n=d DO { writes("Etc"); RETURN }
  writef("%s", opstr(x!H1))
  IF ln DO writef("  -- line %d", ln)
  FOR i = 2 TO size DO { newline()
                         FOR j=0 TO n-1 DO writes( v!j )
                         writes("*-")
                         v!n := i=size->"  ","! "
                         plist(x!(i-1), n+1, d)
                       }
.

FUN trnerr : mess, a =>
  writes("Error")
  IF procname DO writef(" in %s", @procname!H4)
  IF comline DO writef(" near line %d", comline)
  writes(":   ")
  writef(mess, a)
  newline()
  errcount++
abort 999
  IF errcount >= errmax DO fatalerr("\nCompilation aborted\n")
.
FUN trprog : x =>
  dvec, dvect := treevec, treep
  dvec!H1, dvec!H2, dvec!H3 := 0, 0, 0
  dvece := dvec + 3*Bpw
  clearhints() // Mark all names as undeclared

  FOR i = 0 TO labmax DO labv!i, refv!i := -1, 0

  resultlab := -2
  comline, procname, labnumber := 1, 0, 1
  ssp := 2
  outfl(Laddr, 1); ssp++  // 1 = lab number of start
  outfn(Fnap, 3);  ssp++
  outf Halt
  declstatnames x
  checkdistinct(@dvec!H4)

  WHILE x DO { trdecl(x!H2); x:=x!H3 }
  resolvelabels()
  writef("Program size: %d   Data size: %d\n", codep-codev, datap-datav)
.

FUN trnext : next =>  IF next<0 DO outf Rtrn
                      IF next>0 DO outfl(Jump, next)
.
FUN trcom //arguments (x, next)
// x       is the command to translate
// next<0  compile x followed by Rtrn
// next>0  compile x followed by Jump next
// next=0  compile x only

: [Let, name, exp, body, ln], next =>
             LET e=dvece, s=ssp
             comline := ln
             addname(name, Local, ssp+1)
             load exp
             trcom(body, next)
             undeclare e 
             outfn(Stack, s)
             ssp := s
  
: [Vec, name, [Num,upb], body, ln], next =>
             LET e=dvece, s=ssp
             comline := ln
             addname(name, Vec, ssp+1)
             ssp := ssp + upb
             outfn(Stack, ssp)
             trcom(body, next)
             undeclare e
             outfn(Stack, s)
             ssp := s
  
: [Assign, lhs, rhs, ln], next =>
             comline := ln
             assign(lhs, rhs)
             trnext next
 
: [Rtap, rt, args, ln] , next=>
             LET s = ssp
             comline := ln
             ssp +:= 3
             outfn(Stack, ssp)
             loadlist(args)
             load rt
             outfn(Rtap, s+1)
             ssp := s
             trnext next 
 
: [op(Printf|Sys), args, ln], next =>
             LET s = ssp
             comline := ln
             loadlist args
             outfn(op, s+1)
             ssp := s
             trnext next
 
: [op(If|Unless), cond, body, ln], next =>
             comline := ln
             TEST next>0
             THEN { jumpcond(cond, op=Unless, next)
                    trcom(body, next)
                  }
             ELSE { LET l = nextlab()
                    jumpcond(cond, op=Unless, l)
                    trcom(body, next)
                    outlab l
                    trnext next
                  }
 
: [Test, cond, c1, c2, ln], next =>
             LET l=nextlab(), m=0
             comline := ln
             jumpcond(cond, FALSE, l)
         
             TEST next=0
             THEN { m := nextlab(); trcom(c1, m) }
             ELSE trcom(c1, next)
                     
             outlab l
             trcom(c2, next)
             UNLESS m=0 DO outlab m
 
: [Return, ln], next =>
             comline := ln
             outf Rtrn
 
: [Resultis, exp, ln], next =>
             comline := ln
             IF resultlab=-1 DO { fnbody exp; RETURN }
             UNLESS resultlab>0 DO
             { trnerr("RESULTIS out of context")
               RETURN
             }
             load exp
             outfl(Resultis, resultlab)
             ssp--
 
: [op(While|Until), cond, body, ln], next =>
             LET l=nextlab(), m=next
             comline := ln
             IF next<=0 DO m := nextlab()
             jumpcond(cond, op=Until, m)
             outlab l
             trcom(body, 0)
             comline := ln
             jumpcond(cond, op=While, l)
             IF next<=0 DO outlab m
             trnext next
 
: [For, name, a, b, body, ln], next =>
             LET e=dvece, s=ssp
             LET l=nextlab(), m=nextlab()
             comline := ln
             addname(name, Local, ssp+1)
             load a  // The control variable at s+1
             load b  // The end limit        at s+2

             outfl(Jump, m)               // Jump to test

             outlab l                     // Start of body
             trcom(body, 0)

             outfn(Lp, s+1); ssp++ // Inc control variable
             outfn(Ln, 1);   ssp++
             outf(Add);      ssp--
             outfn(Sp, s+1); ssp--

             outlab m
             outfn(Lp, s+1); ssp++ // Compare with limit
             outfn(Lp, s+2); ssp++
             outf(Le);       ssp--
             outfl(Jt, l);   ssp--

             undeclare e
             outfn(Stack, s)
             ssp := s
             trnext next
  
: [Seq, c1, c2], next =>
             trcom(c1, 0)
             trcom(c2, next)

:         => trnerr("Compiler error in Trans")
.

FUN declstatnames : x => WHILE x DO
{ LET d = x!H2
//writef("declstatnames: %d\n", x)
//plist(x, 0, 5); newline()
  MATCH d

  : [Static, slist] =>
              LET p=slist, np=0
              WHILE p MATCH p

              : [Comma, a, b] => p, np := a, b

              : name[Name]    => LET lab = nextlab()
                                 outvar lab
                                 addname(name, Var, lab)
                                 p, np := np, 0

              : [Statvec, name, [?,upb]] =>
                                 LET lab = nextlab()
                                 outstatvec(lab, upb)
  //  writef("returned from outstatvec\n")
                                 addname(name, Addr, lab)
                                 p, np := np, 0

              :               => trnerr "Bad STATIC declaration"
              .

  : [Fndef | Rtdef, name] =>
              LET lab = name=namestart -> 1, nextlab()
              addname(name, Addr, lab)

  :        => trnerr("Compiler error in declstatnames")
  .

  x := x!H3
}
.

FUN decldyn
: 0                   => RETURN
 
: name[Name]          => addname(name, Local, ++ssp)
 
: [Comma, name, rest] => addname(name, Local, ++ssp)
                         decldyn rest
 
:                     =>   trnerr "Compiler error in Decldyn"
.

FUN checkdistinct : p =>
  LET lim = @dvece!-3
  LET q = p
  WHILE q<lim DO
  { LET n = q!H1
    LET c = @q!3
    WHILE c<=lim DO
    { IF c!H1=n DO trnerr("Name %s defined twice", @n!H4)
      c +:= 3*Bpw
    }
    q +:= 3*Bpw
  }
.
 
FUN addname : name, k, a =>
  LET p = dvece + 3*Bpw
  IF p>dvect DO { trnerr("More workspace needed"); RETURN }
  dvece!H1, dvece!H2, dvece!H3 := name, k, a
  name!H2 := dvece // Remember the declaration
//writef("addname: %s %d %d  cell=%d\n", @name!H4, k, a, dvece)
  dvece := p
.
 
FUN undeclare : e => 
  FOR t = e TO dvece-3*Bpw BY 3*Bpw DO
  { LET name = t!H1
    name!H2 := 0   // Forget its declaration
  }
  dvece := e
.

FUN cellwithname : n =>
  LET t = n!H2
//writef("cellwithname: %s\n", @n!H4)
//abort 2222
  //IF t RETURN t  // It has been looked up before
  t := dvece
  t -:= 3*Bpw REPEATUNTIL t!H1=n OR t!H1=0
  n!H2 := t  // Associate the name with declaration item
//writef("cellwithname: %s %d %d  cell=%d\n", @n!H4, t!H2, t!H3, t)
  RETURN t
.

FUN trdecl
: [Static, slist] =>  // Static declarations are compiled in declstatnames
               RETURN

: [op(Fndef|Rtdef), name, args, body] =>
               LET e = dvece
               LET t = cellwithname name
               LET strlab = nextlab()

               resultlab := -2
               procname := name

               outstring(strlab, @procname!H4)
               outentry(t!H3, strlab)
               ssp := 2
               decldyn args  // Declare the formal paramenters
               checkdistinct e 
               outfn(Stack, ssp)
               TEST op=Rtdef THEN trcom(body, -1)
                             ELSE fnbody body
 
               undeclare e
               procname := 0
 
: =>
writef("default in trdecl\n")
           RETURN
.

FUN jumpcond

: [False],     FALSE, l => outfl(Jump, l)
: [False],     TRUE,  l => RETURN

: [True],      TRUE,  l => outfl(Jump, l)
: [True],      FALSE, l => RETURN

: [Not, x],    b,    l  => jumpcond(x, NOT b, l)

: [And, x, y], FALSE, l => jumpcond(H2!x, FALSE, l)
                           jumpcond(H2!x, FALSE, l)

: [And, x, y], TRUE,  l => LET m = nextlab()
                           jumpcond(x, FALSE,  m)
                           jumpcond(y, TRUE, l)
                           outlab m

: [Or,  x, y], TRUE,  l => jumpcond(H2!x, TRUE, l)
                           jumpcond(H2!x, TRUE, l)

: [Or,  x, y], FALSE, l => LET m = nextlab()
                           jumpcond(x, TRUE,  m)
                           jumpcond(y, FALSE, l)
                           outlab m
 
: x,           b,     l => load(x)
                           outfl(b -> Jt, Jf, l)
                           ssp--
.

FUN load
 
: [op(Vecap | Mul | Div | Mod | Add | Sub |
      Eq | Ne | Lt | Gt | Le | Ge |
      Lsh | Rsh | And | Or | Xor), a, b] =>
                   load a; load b; outf op
                   ssp--
 
: [op(Ind | Neg | Not), a] =>
                   load a
                   outf op

: [Lv, a]  =>      loadlv a
 
: [Num, k]  =>     outfn(Ln,  k); ssp++
: [True]  =>       outfn(Ln, -1); ssp++
: [False]  =>      outfn(Ln,  0); ssp++
 
: [String, chars] =>  
                   LET strlab = nextlab()
                   outstring(strlab, @chars)
                   outfl(Laddr, strlab)
                   ssp++
 
: name[Name]    => transname(name, Lp, Ll, Llp, Laddr)
                   ssp++
 
: [Valof, body] => LET rl = resultlab
                   resultlab := nextlab()
                   trcom(body, 0)
                   outlab resultlab 
                   outfn(Stack, ssp)
                   outf Lres; ssp++
                   resultlab := rl
 
: [Fnap, a, b]  => LET s = ssp
                   ssp +:= 3
                   outfn(Stack, ssp)
                   loadlist b
                   load a
                   outfn(Fnap, s+1)
                   outf Lres; ssp := s+1

: [op(Printf | Sys), arglist, ln] =>
                   LET s = ssp
                   comline := ln
                   loadlist arglist
                   outfn(op, s+1); ssp := s
                   outf Lres;      ssp++

: [op]          => trnerr("Compiler error in Load, op=%d", op)
                   outfl(Ln, 0); ssp++
.

FUN loadlv

: name[Name]    => transname(name, Llp, Laddr, 0, 0); ssp++

: [Ind, a]      => load a

: [Vecap, a, b] => load a; load b; outf Add; ssp--

:               => trnerr("Bad operand to @")
                   outf Lres; ssp++
.

FUN fnbody
                   
: [Valof, body] => LET e=dvece, rl=resultlab
                   resultlab := -1
                   trcom(body, -1)
                   resultlab := rl
                   undeclare e

: x             => load x; outf Fnrn; ssp--
.

FUN loadlist
: 0             => RETURN

: [Comma, a, b] => loadlist a; loadlist b

: x             => load x
.

FUN assign

: name[Name],    rhs => load rhs
                        transname(name, Sp, Sl, 0, 0); ssp--

: [Vecap, x, y], rhs => load rhs
                        load x; load y; outf Add; ssp--
                        outf Stind;               ssp -:= 2

: [Ind, x],      rhs => load rhs
                        load x
                        outf Stind ; ssp -:= 2

:                    => trnerr("Bad assignment")
.
 
FUN transname : name[?,?,?,chars], p, l, v, a =>

  MATCH cellwithname name
   
  : [?, Local, n] => outfn(p, n)
 
  : [?, Var,   n] => outfl(l, n)
 
  : [?, Vec,   n] => IF v=0 DO
                     { trnerr("Misuse of local vector '%s'", @chars)
                       v := p
                     }
                     outfn(v, n)

  : [?, Addr,  n] => IF a=0 DO
                     { trnerr("Misuse of entry name '%s'", @chars)
                       a := l
                     }
                     outfl(a, n)

  :               => trnerr("Name '%s' not declared", @chars)
  .
.

FUN wrf : form, a, b, c => IF optCode DO writef(form, a, b, c)
.
FUN outf : op =>
  wrf("%s\n", opstr op)
  putc op
.
FUN outfn : op, a =>
  wrf("%s %d\n", opstr op, a)
  putc op; putc a
.
FUN outfl : op, lab =>
  wrf("%s L%d\n", opstr op, lab)
  putc op; putref lab
.
FUN outlab : lab =>
  wrf("Lab L%d\n", lab)
  setlab(lab, codep)
.
FUN outentry : l1, l2 =>
  wrf("Entry L%d L%d\n", l1, l2)
  putref l2 
  setlab(l1, codep)
.
FUN outstring : lab, s =>
  LET sv = @mem!datap
  wrf("String L%d %s\n", lab, s)
  setlab(lab, datap)
  FOR i = 0 TO s%0 DO
  { IF i MOD 4 = 0 DO putd 0
    sv%i := s%i
  }
.
FUN outstatvec : lab, a =>
  wrf("Statvec L%d %d\n", lab, a)
  setlab(lab, datap)
  FOR i = 0 TO a-1 DO putd 0
.
FUN outvar : lab =>
  wrf("Var L%d\n", lab)
  setlab(lab, datap)
  putd 0
.
FUN putc : w => TEST codep>codet
                THEN trnerr("More code space needed")
                ELSE mem!codep++ := w
.
FUN putd : w => TEST datap>datat
                THEN trnerr("More data space needed")
                ELSE mem!datap++ := w
.
FUN putref : lab => TEST codep>codet
                    THEN trnerr("More code space needed")
                    ELSE { mem!codep := refv!lab
                           refv!lab := codep++
                         }
.
FUN setlab : lab, addr => labv!lab := addr
.
FUN nextlab : =>
  TEST labnumber>=labmax
  THEN fatalerr("More label space needed")
  ELSE labnumber++
  RETURN labnumber
.

FUN resolvelabels : => FOR lab = 1 TO labnumber DO
{ LET p = refv!lab
  LET labval = labv!lab
  IF p AND labval<0 TEST lab=1 THEN trnerr "start not defined"
                               ELSE trnerr("Label %d unset", lab)
  WHILE p DO { LET np = mem!p
               mem!p, p := labval, np
             }
}
.

FUN interpret : regs, mem => 
  LET retcode = 0
  LET rv = @mem!regs
  LET res=rv!0, pp=rv!1, sp=rv!2, pc=rv!3, count=rv!4

  { LET op = mem!pc                // Fetch next instruction
    IF optTrace DO
    { writef("p:%5d  sp:%5d %10d %10d  %5d: %8s",
              pp, sp, mem!(sp-1), mem!sp, pc, opstr op)
      IF hasOperand op DO writef(" %d", mem!(pc+1))
      newline()
    }
    IF count<=0 DO { retcode := 3; BREAK } // Zero count
    count--
    pc++
    IF sp>mem+stackt DO abort 9999
    MATCH op
    : Halt =>   retcode := sp!0; BREAK

    : Laddr |
      Ln =>     mem!++sp := mem!pc++;           LOOP
    : Lp =>     mem!++sp := mem!(pp+mem!pc++);  LOOP
    : Llp =>    mem!++sp := pp+mem!pc++;        LOOP
    : Ll =>     mem!++sp := mem!(mem!pc++);     LOOP
    : Sp =>     mem!(pp+mem!pc++) := mem!sp--;  LOOP
    : Sl =>     mem!(mem!pc++):= mem!sp--;      LOOP

    : Rtap |
      Fnap => LET opp=pp, retaddr=pc+1
              pp, pc := pp+mem!pc, mem!sp
              mem!pp, mem!(pp+1), mem!(pp+2) := opp, retaddr, pc
              sp := pp+2
              LOOP

    : Lres =>  mem!++sp := res;              LOOP

    : Fnrn |
      Rtrn =>   res := mem!sp
                LET npp=mem!pp, npc=mem!(pp+1)
                sp := pp-1
                pp, pc := npp, npc
                LOOP
    : Ind =>  mem!sp :=  !  mem!sp;                      LOOP
    : Neg =>  mem!sp :=  -  mem!sp;                      LOOP
    : Not =>  mem!sp := NOT mem!sp;                      LOOP
    : Stind =>sp -:= 2; mem!(mem!(sp+2)) := mem!(sp+1);        LOOP
    : Vecap =>sp--; mem!sp := mem!(mem!sp + mem!(sp+1)); LOOP
    : Mul =>  sp--; mem!sp := (mem!sp)  *  (mem!(sp+1)); LOOP
    : Div =>  sp--; mem!sp := (mem!sp)  /  (mem!(sp+1)); LOOP
    : Mod =>  sp--; mem!sp := (mem!sp) MOD (mem!(sp+1)); LOOP
    : Add =>  sp--; mem!sp := (mem!sp)  +  (mem!(sp+1)); LOOP
    : Sub =>  sp--; mem!sp := (mem!sp)  -  (mem!(sp+1)); LOOP
    : Eq =>   sp--; mem!sp := (mem!sp)  =  (mem!(sp+1)); LOOP
    : Ne =>   sp--; mem!sp := (mem!sp) ~=  (mem!(sp+1)); LOOP
    : Le =>   sp--; mem!sp := (mem!sp) <=  (mem!(sp+1)); LOOP
    : Ge =>   sp--; mem!sp := (mem!sp) >=  (mem!(sp+1)); LOOP
    : Lt =>   sp--; mem!sp := (mem!sp)  <  (mem!(sp+1)); LOOP
    : Gt =>   sp--; mem!sp := (mem!sp)  >  (mem!(sp+1)); LOOP
    : Lsh =>  sp--; mem!sp := (mem!sp) <<  (mem!(sp+1)); LOOP
    : Rsh =>  sp--; mem!sp := (mem!sp) >>  (mem!(sp+1)); LOOP
    : And =>  sp--; mem!sp := (mem!sp)  &  (mem!(sp+1)); LOOP
    : Or =>   sp--; mem!sp := (mem!sp)  |  (mem!(sp+1)); LOOP
    : Xor =>  sp--; mem!sp := (mem!sp) XOR (mem!(sp+1)); LOOP
    : Jt =>   sp--; TEST mem!(sp+1) THEN pc := mem!pc
                                    ELSE pc++;           LOOP
    : Jf =>   sp--; TEST mem!(sp+1) THEN pc++
                                    ELSE pc := mem!pc;   LOOP
    : Resultis => 
                res := mem!sp--; pc := mem!pc;           LOOP
    : Jump   => pc := mem!pc;                            LOOP
    : Stack  => sp := pp+mem!pc++;                       LOOP
    : Printf => sp := pp+mem!pc++ - 1
                printf(mem!(sp+1), mem, sp+2)
                LOOP
    : Sys =>  sp := pp+mem!pc++ - 1
              MATCH mem!(sp+1)
              : 0 => retcode  := mem!(sp+2);            BREAK
              : 1 => res := interpret(mem!(sp+2), mem); LOOP
              : 2 => optTrace := mem!(sp+2);            LOOP
              : 3 => res := count; count := mem!(sp+2); LOOP
              :   => writef("\nBad sys(%d,...) call\n", mem!(sp+1))
                     retcode  := 2;               BREAK   
              .

    :      => retcode := 1;    BREAK    // Unknown op code
    .
  } REPEAT

  rv!0, rv!1, rv!2, rv!3, rv!4 := res, pp, sp, pc, count
  RETURN retcode
.

FUN printf : form, mem, p =>
  LET fmt = @mem!form
  { LET k = %fmt++
    UNLESS k RETURN
    IF k='%' DO
    { LET n = 0;
      { k := %fmt++
	UNLESS '0'<=k<='9' BREAK
	n := 10*n + k - '0'
      } REPEAT
      MATCH k
      : 'd' => writed  (mem!p++,        n); LOOP
      : 's' => wrs     (@mem!(mem!p++), n); LOOP
      : 'x' => writehex(mem!p++,        n); LOOP
      :     => wrch(k);                     LOOP
      .
    }
    wrch(k)
  } REPEAT
.

FUN wrs : s, n =>
  LET len = 0
  WHILE s%len DO len := len+1
  FOR i = len+1 TO n DO wrch(' ')
  FOR i = 0 TO len-1 DO wrch(s%i)
.

FUN hasOperand
  : Fnrn | Rtrn | Lres | Halt |
    Vecap | Ind | Stind | Neg | Not |
    Mul | Div | Mod | Add | Sub |
    Eq | Ne | Le | Ge | Lt | Gt |
    Lsh | Rsh | And | Or | Xor => FALSE
  :                            => TRUE
  .
