/*

This is intended as a naive direct translation of the C version of the
VSPL system. It uses the bare minimum of object oriented features and
thus cannot be regarded as a good example of Java programming. The C
functions all appear as static methonds in the class VSPL and all
the tree nodes are subclasses of the class Tree and have no methods
other than their constructors. All the arrays are static.

Implemented in Java by Martin Richards (c) March 2006
*/

import java.io.*;

class Tree {
    int op;

    Tree(int op) {
	this.op = op;
    }
}

class Id extends Tree {
    Id link;
    int hint; // Either zero or the position in dvec of this cell.
    String str;

    Id(int op, Id link, String str) {
	super(op);
	this.link = link;
	this.str = str;
    }
}

class Int extends Tree {
    int k;

    Int(int op, int k) {
	super(op);
	this.k = k;
    }
}

class Str extends Tree {
    String str;

    Str(int op, String str) {
	super(op);
	this.str = str;
    }
}

class N1 extends Tree {
    Tree a;

    N1(int op, Tree a) {
	super(op);
	this.a = a;
    }
}

class N2 extends Tree {
    Tree a, b;

    N2(int op, Tree a, Tree b) {
	super(op);
	this.a = a;
	this.b = b;
    }
}

class N3 extends Tree {
    Tree a, b, c;

    N3(int op, Tree a, Tree b, Tree c) {
	super(op);
	this.a = a;
	this.b = b;
	this.c = c;
    }
}

class N0n extends Tree {
    Tree a;
    int ln;

    N0n(int op, int ln) {
	super(op);
	this.ln = ln;
    }
}


class N1n extends Tree {
    Tree a;
    int ln;

    N1n(int op, Tree a, int ln) {
	super(op);
	this.a = a;
	this.ln = ln;
    }
}

class N2n extends Tree {
    Tree a, b;
    int ln;

    N2n(int op, Tree a, Tree b, int ln) {
	super(op);
	this.a = a;
	this.b = b;
	this.ln = ln;
    }
}

class N3n extends Tree {
    Tree a, b, c;
    int ln;

    N3n(int op, Tree a, Tree b, Tree c, int ln) {
	super(op);
	this.a = a;
	this.b = b;
	this.c = c;
	this.ln = ln;
    }
}

class N4n extends Tree {
    Tree a, b, c, d;
    int ln;

    N4n(int op, Tree a, Tree b, Tree c, Tree d, int ln) {
	super(op);
	this.a = a;
	this.b = b;
	this.c = c;
	this.d = d;
	this.ln = ln;
    }
}


class Cell {
    Id name;
    int k;
    int n;

    Cell(Id name, int k, int n) {
	this.name = name;
	this.k = k;
	this.n = n;
    }
}

class Fatalerror extends Exception {
}

class Error extends Exception {
}


public class VSPL {

    // Lexical tokens, parse tree operators and op-codes

    final static int Num       =   1;
    final static int Name      =   2;
    final static int String    =   3;
    final static int True      =   4;
    final static int False     =   5;
    final static int Valof     =   6;
    final static int Fnap      =   7;
    final static int Lv        =   8;
    final static int Ind       =   9;
    final static int Vecap     =  10;
    final static int Neg       =  11;
    final static int Not       =  12;
    final static int Mul       =  13;
    final static int Div       =  14;
    final static int Mod       =  15;
    final static int Add       =  16;
    final static int Sub       =  17;
    final static int Eq        =  18;
    final static int Ne        =  19;
    final static int Le        =  20;
    final static int Ge        =  21;
    final static int Lt        =  22;
    final static int Gt        =  23;
    final static int Lsh       =  24;
    final static int Rsh       =  25;
    final static int And       =  26;
    final static int Or        =  27;
    final static int Xor       =  28;
    final static int Comma     =  29;
    final static int Fndef     =  30;
    final static int Rtdef     =  31;
    final static int Assign    =  32;
    final static int Rtap      =  33;
    final static int Return    =  34;
    final static int Test      =  35;
    final static int If        =  36;
    final static int Unless    =  37;
    final static int While     =  38;
    final static int Until     =  39;
    final static int For       =  40;
    final static int Resultis  =  41;
    final static int Seq       =  42;
    final static int Let       =  43;
    final static int Vec       =  44;
    final static int Static    =  45;
    final static int Statvec   =  46;
    final static int Decl      =  47;
    final static int Var       =  48;
    final static int Lparen    =  49;
    final static int Rparen    =  50;
    final static int Lsquare   =  51;
    final static int Rsquare   =  52;
    final static int Lcurly    =  53;
    final static int Rcurly    =  54;
    final static int To        =  55;
    final static int Do        =  56;
    final static int Then      =  57;
    final static int Else      =  58;
    final static int Be        =  59;
    final static int Eof       =  60;
    final static int Semicolon =  61;
    final static int Rtrn      =  62;
    final static int Fnrn      =  63;
    final static int Addr      =  64;
    final static int Local     =  65;
    final static int Lab       =  66;
    final static int Data      =  67;
    final static int Jt        =  68;
    final static int Jf        =  69;
    final static int Jump      =  70;
    final static int Ln        =  71;
    final static int Lp        =  72;
    final static int Llp       =  73;
    final static int Ll        =  74;
    final static int Laddr     =  75;
    final static int Sp        =  76;
    final static int Sl        =  77;
    final static int Stind     =  78;
    final static int Lres      =  79;
    final static int Entry     =  80;
    final static int Stack     =  81;
    final static int Printf    =  82;
    final static int Sys       =  83;
    final static int Halt      =  84;

    final static int maxint        = 0x7FFFFFFF;
    final static int nametablesize = 541;
    final static int c_tab         =   9;
    final static int c_newline     =  10;


    static int  errcount, errmax;

    static DataInputStream progstream;
    static DataOutputStream tostream;

    static boolean optTokens, optTree, optCode, optTrace;


    // Globals used in LEX

    static String charv;
    static char[] chbuf;
    static int ch;
    static int token;
    static int lexval;
    static Str lexstr;
    static Id wordnode;
    static Id[] nametable;
    static Id namestart;
    static int chcount, lineno;

    // Globals used in SYN
 
    static String[] plindentv;  // Used in plist(..)

    // Globals used in TRN and the interpreter
 
    static Cell dvec[];
    static int dvece, dvecp, dvect;
    static int  comline;
    static String procname;
    static int  resultlab;
    static int  ssp;

    static int[] mem;
    final static int  memt = 50000;
    static int  regs;
    static int  codev, codep, codet;
    static int  datav, datap, datat;
    static int  stack, stackt;
    static int[]  labv, refv;
    final static int  labmax=999;
    static int  labnumber;

    static void newline() { System.out.println(""); }



    // Private i/o library

    private static DataInputStream cis;
    private static DataOutputStream cos;

    private static DataInputStream findinput(String name)
    {
	try {
	    return new DataInputStream(new FileInputStream(name));
	} catch(IOException e) {
	    return null;
	}
    }

    private static DataOutputStream findoutput(String name)
    {
	try {
	    return new DataOutputStream(new FileOutputStream(name));
	} catch(IOException e) {
	    return null;
	}
    }

    private static void selectinput(DataInputStream stream) {
	cis = stream;
    }

    private static void selectoutput(DataOutputStream stream) {
	cos = stream;
    }

    private static int rdch() {
	try {
	    return cis.read();
	} catch(IOException e) {
	    return -1;
	}
    }

    private static void wrch(char ch) {
	try {
	    cos.write(ch);
	} catch(IOException e) {
	}
    }


    private static void endread() {
	try {
	    cis.close();
	    cis = null;
	} catch(IOException e) {
	}
    }

    private static void endwrite() {
	try {
	    cos.close();
	    cos = null;
	} catch(IOException e) {
	}
    }

    // End of private i/o library


    public static void main(String[] args) {

	try {
	    String progfilename=null; 
	    String tofilename=null; 

	    cis = new DataInputStream(System.in);
	    cos = new DataOutputStream(System.out);

	    errmax   = 2;
	    errcount = 0;

	    optTokens=optTree=optCode=optTrace=false;

	    wrs("\nVSPL (1 Mar 2006) Java Flat Version\n");
 
	    if (args.length==0) {
		wrs(
                  "Usage: java VSPL primes.vp {-l} {-p} {-c} {-t} {-o file}\n");
		return;
	    }

	    progfilename = null;
	    tofilename = null;

	    for(int i=0; i<args.length; i++) {
		//System.out.println("arg " + i + ": " + args[i]);
		if     (args[i].equals("-l"))  optTokens = true;
		else if(args[i].equals("-p"))  optTree = true;
		else if(args[i].equals("-c"))  optCode = true;
		else if(args[i].equals("-t"))  optTrace = true;
		else if(args[i].equals("-o"))  tofilename = args[++i];
		else                           progfilename = args[i];
	    }

	    if(progfilename==null) {
		wrs("No source filename given\n");
		return;
	    }

	    selectinput(findinput(progfilename));
	    if(cis==null) fatalerr("Trouble with input file "+progfilename);

	    plindentv = new String[20];

	    regs=10;
	    codev=100;
	    codep=codev;
	    codet=10000;
	    datav=codet;
	    datap=datav;
	    datat=memt;


	    Tree tree=null;
	    chbuf = new char[64];
	    for(int i=0; i<=63; i++) chbuf[i] = 0;
	    chcount = 0;
	    lineno = 1;
	    rch();

	    tree = formtree();  // Perform syntax analysis
	    if(optTokens) return;

	    if(optTree) {
		wrs("Parse Tree\n", 0);
		plist(tree, 0, 20);
		newline();
	    }
	    if(errcount>0) throw new Error();

	    mem = new int[memt+1];
	    for(int i=0; i<=memt; i++) mem[i] = 0;

	    trprog(tree);           // Translate the tree
	    if(errcount>0) return;
	    stack = datap;
	    stackt = memt;
	    
	    {
		mem[regs+0] = 0;            // result register
		mem[regs+1] = stack;        // p pointer
		mem[regs+2] = stack+2;      // sp
		mem[regs+3] = codev;        // pc
		mem[regs+4] = maxint;       // count

		mem[stack+0]=mem[stack+1]=mem[stack+2]=0;

		{
		    int ret = interpret(regs, mem);   // Execute the interpreter
		    if (ret!=0)
			wrs("Return code "+ ret+"\n");
		    wrs("\nInstructions executed: "+(maxint-mem[regs+4])+"\n");
		}
	    }
	} catch (Exception e) {
	    //System.out.println("\nAn exception was raised");
	}

	if(progstream!=null) {
	    selectinput(progstream);
            endread();
        }
	if(tostream!=null) {
	    selectoutput(tostream);
	    endwrite();
	}
    }


    public static void lex() throws Error, Fatalerror {
	while(true) {
	    switch(ch) {
      
	    case '\f': case '\n':
		lineno++;

	    case '\r': case '\t': case ' ':
		rch();
		continue;

	    case '0':case '1':case '2':case '3':case '4':
	    case '5':case '6':case '7':case '8':case '9':
		lexval = 0;
		while('0'<=ch && ch<='9'){
		    lexval = 10*lexval + ch - '0';
		    rch();
		}
		token = Num;
		return;
 
	    case 'a':case 'b':case 'c':case 'd':case 'e':
	    case 'f':case 'g':case 'h':case 'i':case 'j':
	    case 'k':case 'l':case 'm':case 'n':case 'o':
	    case 'p':case 'q':case 'r':case 's':case 't':
	    case 'u':case 'v':case 'w':case 'x':case 'y':
	    case 'z':
            case 'A':case 'B':case 'C':case 'D':case 'E':
            case 'F':case 'G':case 'H':case 'I':case 'J':
            case 'K':case 'L':case 'M':case 'N':case 'O':
            case 'P':case 'Q':case 'R':case 'S':case 'T':
            case 'U':case 'V':case 'W':case 'X':case 'Y':
            case 'Z':
		token = lookupword(rdtag());
		return;
 
	    case '{': token = Lcurly;    break;
            case '}': token = Rcurly;    break;
            case '[': token = Lsquare;   break;
            case ']': token = Rsquare;   break;
            case '(': token = Lparen;    break;
            case ')': token = Rparen;    break; 
            case '!': token = Ind;       break;
            case '@': token = Lv;        break;
            case '+': token = Add;       break;
            case '-': token = Sub;       break;
            case ',': token = Comma;     break;
            case ';': token = Semicolon; break;
            case '&': token = And;       break;
            case '|': token = Or;        break;
            case '=': token = Eq;        break;
            case '*': token = Mul;       break;
            case '^': token = Xor;       break;
 
            case '/':
		rch();
		if(ch=='/'){
		    do rch(); while(ch!='\n' && ch!=Eof);
		      continue;
                  }
                  token = Div;
                  return;
 
              case '~':   rch();
                  if(ch=='=') { token = Ne;  break; }
                  token = Not;
                  return;
 
              case '<':
                  rch();
                  if(ch=='=') { token = Le;  break; }
                  if(ch=='<') { token = Lsh; break; }
                  token = Lt;
                  return;
 
              case '>':
                  rch();
                  if(ch=='=') { token = Ge;  break; }
                  if(ch=='>') { token = Rsh; break; }
                  token = Gt;
                  return;
 
              case ':':
                  rch();
                  if(ch=='=') { token = Assign;  break; }
                  synerr("'=' expected after ':'");
                  return;
 
              case '"':
                  charv = "";
                  rch();
                  while(ch!='"' && ch!=Eof)
		      charv += (char)rdstrch();
                  token = String;
                  break;
 
              case '\'':
                  rch();
                  lexval = rdstrch();
                  token = Num;
                  if(ch!='\'')  synerr("Bad character constant");
                  break;

              default:
                  if(ch!=-1) {
                    int badch = ch;
                    ch = ' ';
                    synerr("Illegal character "+ (char)badch);
                  }
                  token = Eof;
                  return;
	    }
 
	    // break in the above switch jumps here.
            rch();
	    return;
	}
    }

    public static void rch() {
	ch = rdch();
	//System.out.println("rch: ch = " + ch);
	chbuf[++chcount&63] = (char)ch;
    }

    public static void wrchbuf() {
	int p;
	System.out.print("\n...");
	for(p = chcount+1; p<=chcount+64; p++) {
	    char k = chbuf[p&63];
	    if(k>0) System.out.print(""+k);
	}
	System.out.println();
    }
 
    public static String rdtag() {
	charv = "";
	while('a'<=ch && ch<='z' ||
	      'A'<=ch && ch<='Z' ||
	      '0'<=ch && ch<='9' ||  ch=='_') {
	    charv += (char)ch;
	    rch();
	}
	return charv;
    }

    public static int rdstrch()  throws Error, Fatalerror {
	int res = ch;
	if(ch=='\n' || ch=='\f') {
	    lineno++;
	    synerr("Unescaped newline character");
	}
	if(ch=='\\') {
	    rch();
	    switch(ch) {
	    default:   synerr("Bad string or character constant");
	    case '\\':case '\'': case '"':  res = ch;        break;
	    case 't': case 'T':             res = c_tab;     break;
	    case 'n': case 'N':             res = c_newline; break;
	    }
	}
	rch();
	return res;
    }


    public static int lookupword(String word) {
	int hashval = 0;
        int len = word.length();

	for(int i=0; i<len; i++)
	    hashval = (13*hashval + word.charAt(i)) & 0xFFFFFF;
	hashval = (hashval&maxint) % nametablesize;
	wordnode = nametable[hashval];

	while(wordnode!=null && !wordnode.str.equals(word))
	    wordnode = wordnode.link;

	if(wordnode==null) {
	    wordnode = new Id(Name, nametable[hashval], word);
	    nametable[hashval] = wordnode;
	}
	//System.out.println("lookupword: "+word+
        //                   " hashval="+hashval+
        //                   " tok="+wordnode.op);
	return wordnode.op;
    }
 
    public static void dsw(String word, int tok) {
	lookupword(word);
	wordnode.op = tok;
    }
 
    public static void declsyswords() {
	dsw("be", Be);             dsw("do", Do);         dsw("else", Else);
	dsw("false", False);       dsw("if", If);         dsw("for", For);
	dsw("let", Let);           dsw("mod", Mod);       dsw("printf", Printf);
	dsw("resultis", Resultis); dsw("return", Return); dsw("static", Static);
	dsw("sys", Sys);           dsw("test", Test);     dsw("to", To);
	dsw("true", True);         dsw("then", Then);     dsw("valof", Valof);
	dsw("vec", Vec);           dsw("unless", Unless); dsw("until", Until);
	dsw("while", While);
	lookupword("start");
	namestart = wordnode;
    }


    public static Tree formtree() throws Fatalerror, Error {
	Tree res=null;

	try {

	    nametable = new Id[nametablesize];

	    for(int i=0; i<nametablesize; i++) nametable[i] = null;

	    declsyswords();
	    lex();

	    while(optTokens) {            // For debugging lex.
		if(token==Eof)    return res;
		System.out.print("token = "+token+" "+opstr(token));
		if(token==Num) System.out.print("        "+  lexval);
		if(token==Name) System.out.print("       "+wordnode.str);
		if(token==String) System.out.print("     \""+ charv+"\"");
		newline();
		lex();
	    }
	} catch (Error e) {
	    //System.out.println("\nException Error caught in formtree");
	}

recover:
	res = rdprog();
	if (token!=Eof) fatalerr("Incorrect termination");
	return res;
    }

    static void fatalerr(String mess) throws Fatalerror {
	System.out.print("\nFatal error:  ");
	System.out.print(mess);
	System.out.print("\nCompilation aborted\n");
	errcount++;
	throw new Fatalerror();
    }

    static void synerr(String mess) throws Fatalerror, Error {
	System.out.print("\nError near line "+ lineno + ": ");
	System.out.print(mess);
	wrchbuf();
	errcount++;
	if(errcount >= errmax) fatalerr("Too many errors");

	// Skip the rest of the input line 
	while(ch!='\n' && ch!=Eof) rch();
	lex();

	throw new Error();
    }
   
    static void trnerr(String mess) throws Fatalerror {
	System.out.println("Error");
	if(procname!=null) System.out.println(" in "+procname);
	if(comline!=0)  System.out.println(" near line "+comline);
	System.out.println(":   ");
	System.out.println(mess);
	newline();
	errcount++;
	if(errcount >= errmax) fatalerr("Too many errors");
    }

    public static void checkfor(int tok, String mess) throws Fatalerror, Error {
	if(token!=tok) synerr(mess);
	lex();
    }
 
    public static Tree rdprog() throws Fatalerror, Error {
	Tree res=null;
	int ln = lineno;
	switch(token) {
	default:
	    synerr("Bad outer level declaration\n");

	case Eof:
	    res = null; break;

	case Static:
	    {
		Tree d;
		lex();
		d = new N1n(Static, rstatlist(), ln);
		return  new N2(Decl, d, rdprog());
	    }

	case Let:
	    {
		Tree n;
		Tree args;
		lex();
		n = rname();
		checkfor(Lparen, "'(' missing");
		if(token==Name) args = rnamelist();
		else            args = null;
		checkfor(Rparen, "')' missing");
 
		if(token==Be) {
		    Tree d = new N3n(Rtdef, n, args, rncom(), ln);
		    return new N2(Decl, d, rdprog());
		}
 
		if(token==Eq) {
		    Tree d = new N3n(Fndef, n, args, rnexp(0), ln);
		    return new N2(Decl, d, rdprog());
		}
 
		synerr("Bad procedure heading");
	    }
	}
	return res;
    }

    public static Tree rdblockbody() throws Fatalerror, Error {
	Tree res;
	//jmp_buf *rl = reclabel;

	while(true) {
	    try {
		int op = token;

		switch(op) {
		default:
		    res = rdseq();
		    break;

		case Let:
		case Vec:
		    {
			Tree n;
			Tree e;
			int ln = lineno;
			lex();
			n = rname();
			if(op==Let) {
			    checkfor(Eq, "Missing '='");
			    e = rexp(0);
			} else {
			    checkfor(Lsquare, "Missing '['");
			    e = rexp(0);
			    if(e.op!=Num) synerr("Bad 'vec' declaration");
			    checkfor(Rsquare, "Missing ']'");
			}
			checkfor(Semicolon, "';' expected");
			res = new N3n(op, n, e, rdblockbody(), ln);
			break;
		    }
		}
	    } catch (Error e) {
		continue;
	    }
	    return res;
	}
    }
 
    private static Tree rdseq() throws Fatalerror, Error {
	Tree a = rcom();
	if(token==Rcurly || token==Eof) return a;
	checkfor(Semicolon, "';' expected");
	return new N2(Seq, a, rdseq());
    }

    private static Tree rnamelist() throws Fatalerror, Error {
	Tree a = rname();
	if(token!=Comma) return a;
	lex();
	return new N2(Comma, a, rnamelist());
    }

    private static Tree rexplist() throws Fatalerror, Error {
	Tree a = rexp(0);
	if(token!=Comma) return a;
	lex();
	return new N2(Comma, a, rexplist());
    }
 
    private static Tree rstatlist() throws Fatalerror, Error {
	Tree a = rname();
	if(token==Lsquare) {
	    Tree b = rnexp(0);
	    if(b.op!=Num) synerr("Number expected");
	    checkfor(Rsquare, "']' expected");
	    a = new N2(Statvec, a, b);
	}
	if(token!=Comma) return a;
	lex();
	return new N2(Comma, a, rstatlist());
    }

    private static Tree rname() throws Fatalerror, Error {
	Tree a = wordnode;
	checkfor(Name, "Name expected");
	return a;
    }
 
    private static Tree rbexp() throws Fatalerror, Error {
	Tree a;
	int op = token;
	int ln = lineno;
 
	switch(op) {
 
	default:
	    synerr("Error in Expression");

	case True:
	case False:
	case Name:
	    a = wordnode;
	    lex();
	    return a;
 
	case String:
	    a = new Str(String, charv);
	    lex();
	    return a;
 
	case Num:
	    a = new Int(Num, lexval);
	    lex();
	    return a;
 
	case Printf:
	case Sys:
	    lex();
	    checkfor(Lparen, "'(' missing");
	    a = rexplist();
	    checkfor(Rparen, "')' missing");
	    return new N1n(op, a, ln);

	    case Lparen:
		a = rnexp(0);
		checkfor(Rparen, "')' missing");
		return a;
 
	case Valof:
	    return new N1(Valof, rncom());
 
	case Ind:
	case Lv:
	    return new N1(op, rnexp(8));

	case Add:
	    return rnexp(6);
 
	case Sub:
	    a = rnexp(6);
	    if(a.op==Num)
		((Int)a).k = - ((Int)a).k;
	    else
		a = new N1(Neg, a);
	    return a;
 
	case Not:
	    return new N1(Not, rnexp(3));
	}
    }
 
    private static Tree rnexp(int n) throws Fatalerror, Error {
	lex(); return rexp(n);
    }
 
    private static Tree rexp(int n) throws Fatalerror, Error {
	Tree a=rbexp();
	Tree b;
	int p = 0;

	while(true) {
	    int op = token;
	    int ln = lineno;
	    switch(op) {
 
	    default:
		return a;
 
	    case Lparen:
		lex();
		if(token!=Rparen) b = rexplist();
		else b = null;
		checkfor(Rparen, "')' missing");
		a = new N2n(Fnap, a, b, ln);
		continue;
 
	    case Lsquare:
		b = rnexp(0);
		checkfor(Rsquare, "']' missing");
		a = new N2(Vecap, a, b);
		continue;
 
	    case Mul:case Div:case Mod:
		p = 7;              break;
	    case Add:case Sub:
		p = 6;              break;
	    case Lsh:case Rsh:
		p = 5;              break;
	    case Eq:case Le:case Lt:case Ne:case Ge:case Gt:
		p = 4;              break;
	    case And:
		p = 3;              break;
	    case Or:
		p = 2;              break;
	    case Xor:
		p = 1;              break;
	    }
      
	    if(n>=p) return a;
	    a = new N2(op, a, rnexp(p));
	}
    }
  
    private static Tree rcom() throws Fatalerror, Error {
	Tree n;
	Tree a;
	Tree b;
	int op= token;
	int ln = lineno;
 
	switch(token) {
	default:
	    synerr("Command expected");
 
	case Name:case Num:case Lparen:case Ind:
	case Sys:case Printf:
	    // All tokens that can start an expression.
	    a = rexp(0);
 
	    if(token==Assign) {
		if(a.op!=Name && a.op!=Vecap && a.op!=Ind)
		    synerr("Bad assigment statement");
		return new N2n(Assign, a, rnexp(0), ln);
	    }
 
	    if(a.op==Fnap) {
		a.op = Rtap;
		return a;
	    }
 
	    if(a.op!=Sys && a.op!=Printf)
		synerr("Error in command");
	    return a;
 
	case Resultis:
	    return new N1n(op, rnexp(0), ln);
 
	case If:    case Unless:
	case While: case Until:
	    a = rnexp(0);
	    checkfor(Do, "'do' missing");
	    return new N2n(op, a, rcom(), ln);
 
	case Test:
	    a = rnexp(0);
	    checkfor(Then, "'then' missing");
	    b = rcom();
	    checkfor(Else, "'else' missing");
	    return new N3n(Test, a, b, rcom(), ln);
 
	case For:
	    lex();
	    n = rname();
	    checkfor(Eq, "'=' expected");
	    a = rexp(0);
	    checkfor(To, "'to' expected");
	    b = rexp(0);
	    checkfor(Do, "'do' missing");
	    return new N4n(For, n, a, b, rcom(), ln);

	case Return:
	    lex();
	    return new N0n(op, ln);
 
	case Lcurly:
	    lex();
	    a = rdblockbody();
	    checkfor(Rcurly, "'}' expected");
	    return a;
	}
    }

    private static Tree rncom() throws Fatalerror, Error {
	lex(); return rcom();
    }

    public static String opstr(int op) {
	switch(op) {
	default:        //return "Unknown";

	case Assign:    return "Assign";    case Add:      return "Add";
	case And:       return "And";       case Be:       return "Be";
	case Comma:     return "Comma";     case Data:     return "Data";
	case Decl:      return "Decl";      case Div:      return "Div";
	case Do:        return "Do";        case Else:     return "Else";
	case Entry:     return "Entry";     case Eq:       return "Eq";
	case False:     return "False";     case Fnap:     return "Fnap";
	case For:       return "For";       case Fndef:    return "Fndef";
	case Fnrn:      return "Fnrn";      case Ge:       return "Ge";
	case Gt:        return "Gt";        case Halt:     return "Halt";
	case If:        return "If";        case Ind:      return "Ind";
	case Jf:        return "Jf";        case Jt:       return "Jt";
	case Jump:      return "Jump";      case Lab:      return "Lab";
	case Laddr:     return "Laddr";     case Lcurly:   return "Lcurly";
	case Le:        return "Le";        case Let:      return "Let";
	case Ll:        return "Ll";        case Llp:      return "Llp";
	case Ln:        return "Ln";        case Lp:       return "Lp";
	case Lparen:    return "Lparen";    case Lres:     return "Lres";
	case Lsh:       return "Lsh";       case Lsquare:  return "Lsquare";
	case Lt:        return "Lt";        case Lv:       return "Lv";
	case Mod:       return "Mod";       case Mul:      return "Mul";
	case Name:      return "Name";      case Ne:       return "Ne";
	case Neg:       return "Neg";       case Not:      return "Not";
	case Num:       return "Num";       case Or:       return "Or";  
	case Printf:    return "Printf";    case Rcurly:   return "Rcurly";
	case Resultis:  return "Resultis";  case Return:   return "Return";
	case Rparen:    return "Rparen";    case Rsh:      return "Rsh";
	case Rsquare:   return "Rquare";    case Rtap:     return "Rtap";
	case Rtdef:     return "Rtdef";     case Rtrn:     return "Rtrn";
	case Semicolon: return "Semicolon"; case Seq:      return "Seq";
	case Sl:        return "Sl";        case Sp:       return "Sp";
	case Stack:     return "Stack";     case Static:   return "Static";
	case Statvec:   return "Statvec";   case String:   return "String";
	case Stind:     return "Stind";     case Sub:      return "Sub";
	case Sys:       return "Sys";       case Test:     return "Test";
	case Then:      return "Then";      case To:       return "To";
	case True:      return "True";      case Valof:    return "Valof";
	case Vecap:     return "Vecap";     case Vec:      return "Vec";
	case Unless:    return "Unless";    case Until:    return "Until";
	case While:     return "While";     case Xor:      return "Xor";
	}
    }

    public static void plist(Tree x, int n, int d) {
	int i, j;
	int size = 0;
	Tree a1=null, a2=null, a3=null, a4=null;
	int ln = 0;

	if(x==null) { System.out.print("Nil"); return;  }
 
	switch(x.op) {
	default:
	    size     = 1;        break;

	case Num:
	    System.out.print(""  +((Int)x).k);        return;
	case Name:
	    System.out.print(""  +((Id)x).str);       return;
	case String:
	    System.out.print("\""+((Str)x).str+"\""); return;

	case For:
	    size = 5;
	    a1 = ((N4n)x).a;
	    a2 = ((N4n)x).b;
	    a3 = ((N4n)x).c;
	    a4 = ((N4n)x).d;
	    ln = ((N4n)x).ln;
	    break;

	case Fndef: case Rtdef:
	case Let: case Vec: case Test:
	    size = 4;
	    a1 = ((N3n)x).a;
	    a2 = ((N3n)x).b;
	    a3 = ((N3n)x).c;
	    ln = ((N3n)x).ln;
	    break;

	case Vecap:
	case Mul: case Div: case Mod: case Add: case Sub:
	case Eq: case Ne: case Lt: case Gt: case Le: case Ge:
	case Lsh: case Rsh:
	case And: case Or: case Xor:
	case Comma: case Seq: case Decl: case Statvec:
	    size = 3;
	    a1 = ((N2)x).a;
	    a2 = ((N2)x).b;
	    break;

	case Assign: case Rtap: case Fnap:
	case If: case Unless: case While: case Until:
	    size = 3;
	    a1 = ((N2n)x).a;
	    a2 = ((N2n)x).b;
	    ln = ((N2n)x).ln;
	    break;

	case Valof: case Lv: case Ind: case Neg: case Not:
	    size = 2;
	    a1 = ((N1)x).a;
	    break;

	case Printf: case Sys: case Static: case Resultis:
	    size = 2;
	    a1 = ((N1n)x).a;
	    ln = ((N1n)x).ln;
	    break;

	case Return:
	    size = 1;
	    ln = ((N0n)x).ln;
	    break;

	case True: case False:
	    size = 1;                 break;
	}
 
	if(n==d) { System.out.print("Etc"); return; }
	System.out.print(""+ opstr(x.op));
	if(ln>0) System.out.print("  -- line "+ ln);
	for(i=2; i<=size; i++) {
	    Tree ni = i==2 ? a1 :
		i==3 ? a2 :
		i==4 ? a3 :
		a4;

	    newline();
	    for(j=0; j<n; j++) System.out.print("" + plindentv[j]);
	    System.out.print("*-");
	    plindentv[n] = i==size ? "  " : "! ";
	    plist(ni, n+1, d);
	}
    }



    public static void trprog(Tree x) throws Fatalerror {
	dvec = new Cell[1000];
	dvec[0] = null;
	dvece = 0;
	dvect = 1000;
	dvece++;

        nametable = new Id[nametablesize];
	for(int i=0; i<nametablesize; i++) {
	    Id name = nametable[i];
	    while(name!=null) {
		Id next = name.link;
		name.hint = 0; // Mark undeclared
		name = next;
	    }
	}

	labv = new int[labmax+1];
	refv = new int[labmax+1];
	for(int i=0; i<=labmax; i++) { labv[i] = -1; refv[i] = 0; }

	resultlab = -2;
	comline = 1;
	procname = null;
	labnumber = 1;
	ssp = 2;

	outfl(Laddr, 1); ssp++;  // 1 = lab number of start
	outfn(Fnap, 3);  ssp--;
	outf(Halt);

	declstatnames(x);
	checkdistinct(1);
	while(x!=null) { trdecl(((N2)x).a); x=((N2)x).b; }
	resolvelabels();

	System.out.println("Program size: "+(codep-codev)+
                           "   Data size: "+(datap-datav));
    }


    static void trnext(int next) throws Fatalerror {
	if(next<0) outf(Rtrn);
	if(next>0) outfl(Jump, next);
    }
 
    static void trcom(Tree x, int next) throws Fatalerror {
	// x       is the command to translate
	// next<0  compile x followed by Rtrn
	// next>0  compile x followed by Jump next
	// next=0  compile x only
	while(true) {
	    int op = x.op;

	    switch(op) {
	    default:
		trnerr("Compiler error in Trans");
		return;
 
	    case Let:
		{
		    int e = dvece;
		    int s = ssp;
		    comline = ((N3n)x).ln;
		    addname((Id)(((N3n)x).a), Local, ssp+1);
		    load(((N3n)x).b);
		    trcom(((N3n)x).c, next);
		    undeclare(e);
		    outfn(Stack, s);
		    ssp = s;
		    return;
		}
  
	    case Vec:
		{
		    int e = dvece;
		    int s = ssp;
		    comline = ((N3n)x).ln;
		    addname((Id)(((N3n)x).a), Vec, ssp+1);
		    ssp += ((Int)((N3n)x).b).k;
		    outfn(Stack, ssp);
		    trcom(((N3n)x).c, next);
		    undeclare(e);
		    outfn(Stack, s);
		    ssp = s;
		    return;
		}

	    case Assign:
		    comline = ((N2n)x).ln;
		    assign(((N2n)x).a, ((N2n)x).b);
		    trnext(next);
		    return;
 
	    case Rtap:
		{
		    int s = ssp;
		    comline = ((N2n)x).ln;
		    ssp += 3;
		    outfn(Stack, ssp);
		    loadlist(((N2n)x).b);
		    load(((N2n)x).a);
		    outfn(Rtap, s+1);
		    ssp = s;
		    trnext(next);
		    return;
		}
 
	    case Printf:
	    case Sys:
		{
		    int s = ssp;
		    comline = ((N1n)x).ln;
		    loadlist(((N1n)x).a);
		    outfn(op, s+1);
		    ssp = s;
		    trnext(next);
		    return;
		}
      
 
	    case Unless:
	    case If:
		comline = ((N2n)x).ln;
		if(next>0) {
		    jumpcond(((N2n)x).a, op==Unless, next);
		    trcom(((N2n)x).b, next);
		} else {
		    int l = nextlab();
		    jumpcond(((N2n)x).a, op==Unless, l);
		    trcom(((N2n)x).b, next);
		    outlab(l);
		    trnext(next);
		}
		return;
 
	    case Test:
		{
		    int l = nextlab();
		    int m = 0;
		    comline = ((N3n)x).ln;
		    jumpcond(((N3n)x).a, false, l);
		    if(next==0) {
			m = nextlab();
			trcom(((N3n)x).b, m);
		    } else {
			trcom(((N3n)x).b, next);
		    }
		    outlab(l);
		    trcom(((N3n)x).c, next);
		    if(m>0) outlab(m);
		    return;
		}
 
	    case Return:
		comline = ((N0n)x).ln;
		outf(Rtrn);
		return;
 
	    case Resultis:
		comline = ((N1n)x).ln;
		if(resultlab==-1) {
		    fnbody(((N1n)x).a);
		    return;
		}
		if(resultlab<=0) {
		    trnerr("Resultis out of context");
		    return;
		}
		load(((N1n)x).a);
		outfl(Resultis, resultlab);
		ssp--;
		return;
 
	    case Until:
	    case While:
		{
		    int l = nextlab();
		    int m = next;
		    comline = ((N2n)x).ln;
		    if(next<=0) m = nextlab();
		    jumpcond(((N2n)x).a, op==Until, m);
		    outlab(l);
		    trcom(((N2n)x).b, 0);
		    comline = ((N2n)x).ln;
		    jumpcond(((N2n)x).a, op==While, l);
		    if(next<=0) outlab(m);
		    trnext(next);
		    return;
		}
 
	    case For:
		{
		    int e = dvece;
		    int s = ssp;
		    int l = nextlab();
		    int m = nextlab();
		    comline = ((N4n)x).ln;
		    addname((Id)(((N4n)x).a), Local, ssp+1);
		    load(((N4n)x).b);  // The control variable at s+1
		    load(((N4n)x).c);  // The end limit        at s+2

		    outfl(Jump, m);         // Jump to test

		    outlab(l);              // Start of body
		    trcom(((N4n)x).d, 0);

		    outfn(Lp, s+1); ssp++;  // Inc control variable
		    outfn(Ln, 1);   ssp++;
		    outf(Add);      ssp--;
		    outfn(Sp, s+1); ssp++;

		    outlab(m);
		    outfn(Lp, s+1); ssp++;  // Compare with limit
		    outfn(Lp, s+2); ssp++;
		    outf(Le);       ssp--;
		    outfl(Jt, l);   ssp--;

		    undeclare(e);
		    outfn(Stack, s);
		    ssp = s;
		    trnext(next);
		    return;
		}
  
	    case Seq:
		trcom(((N2)x).a, 0);
		x = ((N2)x).b;
	    }
	}
    }


    static void declstatnames(Tree x) throws Fatalerror {
	while(x!=null) {
	    Tree d = ((N2)x).a;
  
	    switch(d.op) {
	    default:
		trnerr("Compiler error in declstatnames");
                return;

	    case Static: {
		Tree p = ((N1n)d).a;
		Tree np=null;
		while(p!=null)
		    switch(p.op) {
		    default:
			trnerr("Bad STATIC declaration");
			return;

		    case Comma:
			np = ((N2)p).b;
			p  = ((N2)p).a;
			continue;

		    case Name: {
			int lab = nextlab();
			outvar(lab);
			addname((Id)p, Var, lab);
			p = np;
			np = null;
			continue;
		    }

		    case Statvec: {
			int lab = nextlab();
			int upb = ((Int)(((N2)p).b)).k;
			outstatvec(lab, upb);
			addname((Id)(((N2)p).a), Addr, lab);
			p = np;
			np = null;
			continue;
		    }
		    }
		break;
	    } 

	    case Fndef:
	    case Rtdef: {
		Tree name = ((N3n)d).a;
		int lab = (name==namestart) ? 1 : nextlab();
		addname((Id)name, Addr, lab);
		break;
	    }
	    }
	    x = ((N2)x).b;
	}
    }
  

    static void decldyn(Tree x) throws Fatalerror {
	if(x!=null) {
	    if(x.op==Name) { 
		addname((Id)x, Local, ++ssp);
		return;
	    }
 
	    if(x.op==Comma) { 
		addname((Id)(((N2)x).a), Local, ++ssp);
		decldyn(((N2)x).b);
		return;
	    }
 
	    trnerr("Compiler error in Decldyn");
	}
    }

    static void checkdistinct(int p) throws Fatalerror {
	int lim = dvece;
	for(int q=p; q<lim; q++) {
	    Id n = dvec[q].name;
	    for(int c=q+1; c<lim; c++) if(n.equals(dvec[c].name))
		trnerr("Name "+n.str+" defined twice");
	}
    }
 
    static void addname(Id name, int k, int a) throws Fatalerror {
	if(dvece>=dvect) { trnerr("More workspace needed"); return; }
	name.hint = dvece; // Remember the declaration
	dvec[dvece++] = new Cell(name, k, a);
    }
 
    static void undeclare(int e) {
	for(int t=e; t<dvece; t++) {
	    Id name = dvec[t].name;
	    name.hint = 0;   // Forget its declaration
	}
	dvece = e;
    }

    static Cell cellwithname(Id n) {
	int t = n.hint;
	if(t!=0) return dvec[t];  // It has been looked up before
	t = dvece;
	do t--; while( t>0 && n.str.equals(dvec[t].name));
	n.hint = t;  // Associate the name with declaration item
	return dvec[t];
    }

    static void trdecl(Tree x) throws Fatalerror {
	switch(x.op) {

	case Fndef:
	case Rtdef:
	    {
		int e = dvece;
		Id name = (Id)(((N3n)x).a);
		Cell t = cellwithname(name);
		int strlab = nextlab();

		resultlab = -2;
		procname = name.str;

		outstring(strlab, procname);
		outentry(t.n, strlab);
		ssp = 2;
		decldyn(((N3n)x).b);  // Declare the formal paramenters
		checkdistinct(e);
		outfn(Stack, ssp);
		if(x.op==Rtdef) trcom(((N3n)x).c, -1);
		else                  fnbody(((N3n)x).c);
 
		undeclare(e);
		procname = null;
	    }
 
	case Static:
	default:   return;
	}
    }

    static void jumpcond(Tree x, boolean b, int l) throws Fatalerror {
	boolean sw = b;

	switch(x.op) {
	case False:
	    b = !b;
	case True:
	    if(b) outfl(Jump, l);
	    return;
 
	case Not:
	    jumpcond(((N2)x).a, !b, l);
	    return;
 
	case And:
	    sw = !sw;
	case Or:
	    if(sw) {
                jumpcond(((N2)x).a, b, l);
                jumpcond(((N2)x).b, b, l);
                return;
	    } else {
		int m = nextlab();
                jumpcond(((N2)x).a, !b, m);
                jumpcond(((N2)x).b, b,  l);
                outlab(m);
                return;
	    }
 
	default:
	    load(x);
	    outfl(b ? Jt : Jf, l);
	    ssp--;
	    return;
  }
}

static void load(Tree x) throws Fatalerror {
  int op = x.op;

  switch(op) {
    default:      trnerr("Compiler error in Load, op="+op);
                  outfl(Ln, 0);
                  ssp++;
                  return;
 
    case Vecap:
    case Mul: case Div: case Mod: case Add: case Sub:
    case Eq: case Ne: case Lt: case Gt: case Le: case Ge:
    case Lsh: case Rsh: case And: case Or: case Xor:
                  load(((N2)x).a); load(((N2)x).b); outf(op);
                  ssp--;
                  return;
 
    case Ind: case Neg: case Not:
                  load(((N1)x).a);
                  outf(op);
                  return;

    case Lv:      loadlv(((N1)x).a);
                  return;
 
    case Num:     outfn(Ln, ((Int)x).k); ssp++; return;
    case True:    outfn(Ln, -1);         ssp++; return;
    case False:   outfn(Ln, 0);          ssp++; return;
 
    case String:  
                { int strlab = nextlab();
                  outstring(strlab, ((Str)x).str);
                  outfl(Laddr, strlab);
                  ssp++;
                  return;
                }
 
    case Name:    transname((Id)x, Lp, Ll, Llp, Laddr);
                  ssp++;
                  return;
 
    case Valof: { int rl = resultlab;
                  resultlab = nextlab();
                  trcom(((N1)x).a, 0);
                  outlab(resultlab);
                  outfn(Stack, ssp);
                  outf(Lres); ssp++;
                  resultlab = rl;
                  return;
                }
 
    case Fnap:  { int s = ssp;
                  ssp += 3;
                  outfn(Stack, ssp);
                  loadlist(((N2n)x).b);
                  load(((N2n)x).a);
                  outfn(Fnap, s+1);
                  outf(Lres); ssp = s+1;
                  return;
                }
    case Printf:
    case Sys:
           { int s = ssp;
             comline = ((N1n)x).ln;
             loadlist(((N1n)x).a);
             outfn(op, s+1);
             ssp = s;
             outf(Lres);
             ssp++;
             return;
           }
  }
}

static void loadlv(Tree x) throws Fatalerror {
  switch(x.op) {
    default:    trnerr("Bad operand to @");
                outf(Lres); ssp++;
                return;

    case Name:  transname((Id)x, Llp, Laddr, 0, 0); ssp++;
                return;

    case Ind:   load(((N1)x).a);
                return;

    case Vecap: load(((N2)x).a); load(((N2)x).b); outf(Add); ssp--;
                return;
  }
}

static void fnbody(Tree x) throws Fatalerror {
  switch(x.op) {
    default:      load(x);
                  outf(Fnrn);
                  ssp--;
                  return;
                   
    case Valof: {
	int e = dvece;
                  int rl = resultlab;
                  resultlab = -1;
                  trcom(((N1)x).a, -1);
                  resultlab = rl;
                  undeclare(e);
                  return;
                }
  }
}
 
static void loadlist(Tree x) throws Fatalerror {
  if(x!=null) if(x.op==Comma) { loadlist(((N2)x).a); loadlist(((N2)x).b); }
          else                  load(x);
}

static void assign(Tree x, Tree y) throws Fatalerror {
  switch(x.op) {
    default:    trnerr("Bad assignment");
                return;
    case Name:  load(y);
                transname((Id)x, Sp, Sl, 0, 0);
                ssp--;
                return;
    case Vecap: load(y);
                load(((N2)x).a); load(((N2)x).b); outf(Add); ssp--;
                outf(Stind); ssp -= 2;
                return;
    case Ind:   load(y);
                load(((N1)x).a);
                outf(Stind); ssp -= -2;
                return;
  }
}
 
static void transname(Id x, int p, int l, int v, int a) throws Fatalerror {
  Cell c = cellwithname(x);
  int k = c.k;
  int n = c.n;
 
  switch(k) {
    default:      trnerr("Name '"+x.str+"' not declared");
   
    case Local:   outfn(p, n); return;
 
    case Var:     outfl(l, n); return;
 
    case Vec:     if(v==0) {
                    trnerr("Misuse of local vector '+x.str+'");
                    v = p;
                  }
                  outfn(v, n);
                  return;

    case Addr:    if(a==0) {
                    trnerr("Misuse of entry name '"+x.str+"'");
                    a = l;
                  }
                  outfl(a, n);
                  return;
  }
}
 

static void wrf(String form) {
  if(optCode) System.out.println(form);
}

static void wrfs(String form) {
  if(optCode) System.out.println(form);
}

static void outf(int op) throws Fatalerror {
  wrf(""+opstr(op));
  putcode(op);
}

static void outfn(int op, int a) throws Fatalerror {
  wrf(opstr(op)+" "+a);
  putcode(op); putcode(a);
}

static void outfl(int op, int lab) throws Fatalerror {
  wrf(opstr(op)+" L"+lab);
  putcode(op); putref(lab);
}

static void outlab(int lab) {
  wrf("Lab L"+lab);
  setlab(lab, codep);
}

static void outentry(int l1, int l2) throws Fatalerror {
  wrf("Entry L"+l1+" L"+l2);
  putref(l2);
  setlab(l1, codep);
}

static void outstring(int lab, String s) throws Fatalerror {
    int d = datap;
    int len = s.length();
    wrf("String L"+lab+" "+s);
    setlab(lab, datap);
    for(int i=0; i<=len; i++) {
	char ch = (i==len) ? 0 : s.charAt(i);
	if(i%4 == 0) putd(0);
	putbyte(d, i, ch);
    }
}


static void outstatvec(int lab, int a) throws Fatalerror {
  int i;
  wrf("Statvec L"+lab+" "+a);
  setlab(lab, datap);
  for(i=0; i<a; i++) putd(0);
}

static void outvar(int lab) throws Fatalerror {
  wrf("Var L"+lab);
  setlab(lab, datap);
  putd(0);
}
 
static void putcode(int w) throws Fatalerror {
  if(codep>codet)  trnerr("More code space needed");
  else             mem[codep++] = w;
}

static void putd(int w) throws Fatalerror {
  if(datap>datat) trnerr("More data space needed");
  else            mem[datap++] = w;
}

static void putref(int lab) throws Fatalerror {
  if(codep>codet) trnerr("More code space needed");
  else { mem[codep] = refv[lab];
         refv[lab] = codep++;
  }
}

static void setlab(int lab, int addr) { labv[lab] = addr; }

static int nextlab() throws Fatalerror {
  if(labnumber>=labmax) fatalerr("More label space needed");
  return ++labnumber;
}
 

    static void resolvelabels() throws Fatalerror {
	int lab;
	for(lab=1; lab<=labnumber; lab++) {
	    int p = refv[lab];
	    int labval = labv[lab];
	    if(p!=0 && labval<0)
		if(lab==1)
		    trnerr("start not defined");
		else
		    trnerr("Label "+lab+" unset");
	    while(p!=0) {
		int np = mem[p];
		mem[p] = labval;
		p = np;
	    }
	}
    }


    static int interpret(int regs, int[]mem) {
	int retcode = 0;
	int res   = mem[regs+0];
	int pp    = mem[regs+1];
	int sp    = mem[regs+2];
	int pc    = mem[regs+3];
	int count = mem[regs+4];

	while(true)
	    { int op = mem[pc++];        // Fetch next instruction
	    if(optTrace) {
		System.out.print("p:"); wrd(pp, 5);
		System.out.print("  sp:"); wrd(sp, 5);
		System.out.print(" "); wrd(mem[sp-1], 10);
		System.out.print(" "); wrd(mem[sp], 10);
		System.out.print("  "); wrd(pc-1, 5);
		System.out.print(": "); wrs(opstr(op), 8);

		if(hasOperand(op)) System.out.print(" "+mem[pc]);
		newline();
	    }
	    if(count<=0) { retcode = 3; break; } // Zero count
	    count--;
    
	    switch(op) {
	    default:     retcode = 1;       break;    // Unknown op code

	    case Halt:   retcode = mem[sp]; break;

      case Laddr:
      case Ln:     sp++; mem[sp] = mem[pc++];      continue;
      case Lp:     sp++; mem[sp] = mem[pp+mem[pc++]]; continue;
      case Llp:    sp++; mem[sp] = pp+mem[pc++];   continue;
      case Ll:     sp++; mem[sp] = mem[mem[pc++]]; continue;
      case Sp:     mem[pp+mem[pc++]] = mem[sp]; sp--;  continue;
      case Sl:     mem[mem[pc++]] = mem[sp]; sp--; continue;

      case Rtap:
      case Fnap: { int opp = pp;
                   int retaddr = pc+1;
                   pp += mem[pc];
                   pc = mem[sp];
                   mem[pp+0] = opp;
                   mem[pp+1] = retaddr;
                   mem[pp+2] = pc;
                   sp = pp+2;
                   continue;
                 }

      case Lres:   sp++; mem[sp] = res;                continue;

      case Fnrn:   res = mem[sp];
      case Rtrn: { int npp = mem[pp];
		   int npc = mem[pp+1];
                   sp = pp-1;
                   pp = npp;
	       	   pc = npc;
                   continue;
                 }
      case Ind:    mem[sp] =  mem[mem[sp]];                    continue;
      case Neg:    mem[sp] =  -  mem[sp];                      continue;
      case Not:    mem[sp] =  ~  mem[sp];                      continue;
      case Stind:  sp = sp-2; mem[mem[sp+2]] = mem[sp+1];          continue;
      case Vecap:  sp--; mem[sp] = mem[mem[sp] + mem[sp+1]];  continue;
      case Mul:    sp--; mem[sp] = (mem[sp])  *  (mem[sp+1]); continue;
      case Div:    sp--; mem[sp] = (mem[sp])  /  (mem[sp+1]); continue;
      case Mod:    sp--; mem[sp] = (mem[sp])  %  (mem[sp+1]); continue;
      case Add:    sp--; mem[sp] = (mem[sp])  +  (mem[sp+1]); continue;
      case Sub:    sp--; mem[sp] = (mem[sp])  -  (mem[sp+1]); continue;
      case Eq:     sp--; mem[sp] = ((mem[sp]) ==  (mem[sp+1]))?-1:0; continue;
      case Ne:     sp--; mem[sp] = ((mem[sp]) !=  (mem[sp+1]))?-1:0; continue;
      case Le:     sp--; mem[sp] = ((mem[sp]) <=  (mem[sp+1]))?-1:0; continue;
      case Ge:     sp--; mem[sp] = ((mem[sp]) >=  (mem[sp+1]))?-1:0; continue;
      case Lt:     sp--; mem[sp] = ((mem[sp])  <  (mem[sp+1]))?-1:0; continue;
      case Gt:     sp--; mem[sp] = ((mem[sp])  >  (mem[sp+1]))?-1:0; continue;
      case Lsh:    sp--; mem[sp] = (mem[sp]) <<  (mem[sp+1]); continue;
      case Rsh:    sp--; mem[sp] = (mem[sp]) >>> (mem[sp+1]); continue;
      case And:    sp--; mem[sp] = (mem[sp])  &  (mem[sp+1]); continue;
      case Or:     sp--; mem[sp] = (mem[sp])  |  (mem[sp+1]); continue;
      case Xor:    sp--; mem[sp] = (mem[sp])  ^  (mem[sp+1]); continue;
      case Jt:     sp--; pc = (mem[sp+1]!=0) ? mem[pc] : pc+1;     continue;
      case Jf:     sp--; pc = (mem[sp+1]==0) ? mem[pc] : pc+1;     continue;
      case Resultis: 
                   sp--; res = mem[sp+1];
      case Jump:   pc = mem[pc];                          continue;
      case Stack:  sp = pp + mem[pc++];                   continue;
      case Printf: sp = pp + mem[pc++] - 1;
                   prf(mem[sp+1], sp+2);
                   continue;
      case Sys:    sp = pp + mem[pc++] - 1;
                   switch(mem[sp+1]) {
                     default: System.out.println(
                            "\nBad sys("+mem[sp+1]+",...) call");
                              retcode  = 2;                break;   
                     case 0:  retcode  = mem[sp+2];            break;
                     case 1:  res = interpret(mem[sp+2], mem); continue;
                     case 2:  optTrace = mem[sp+2]!=0;         continue;
                     case 3:  res = count; count = mem[sp+2];  continue;
                   }
		   break;
    }
    break; // Leave the while(true) loop
  }
 
ret:
  mem[regs+0] = res;
  mem[regs+1] = pp;
  mem[regs+2] = sp;
  mem[regs+3] = pc;
  mem[regs+4] = count;
  return retcode;
}

static void prf(int form, int argp) {
    int i=0;
    while(true) {
	char k = getbyte(form, i++);
	if(k==0) return;
	if(k=='%') {
	  int n=0;
          while(true) {
	      k = getbyte(form, i++);
              if ('0'>k || k>'9') break;
	      n = 10*n + k - '0';
	  }
	  switch(k) {
	  default:
	      wrch(k);               continue;
	  case 'd':
	      wrd(mem[argp++], n);   continue;
	  case 'x':
	      wrx(mem[argp++], n);   continue;
	  case 's':
	      wrs(mem[argp++], n);   continue;
	  }
      }
      wrch(k);
  }
}

static char getbyte(int s, int i) {
    int p = s + i/4;
    int sh = 8*(i%4);
    return (char)((mem[p]>>sh) & 255);
}

static void putbyte(int s, int i, int ch) {
    int p = s + i/4;
    int sh = 8*(i%4);
    int mask = 0xFFFFFFFF - (0xFF<<sh);
    mem[p] = mem[p] & mask | (ch<<sh);
}

    public static void wrd(int n, int d) {
        if(n<0) d--;
        if (n<1000000000 && d>=10) wrch(' ');
        if (n<100000000 && d>=9) wrch(' ');
        if (n<10000000 && d>=8) wrch(' ');
        if (n<1000000 && d>=7) wrch(' ');
        if (n<100000 && d>=6) wrch(' ');
        if (n<10000 && d>=5) wrch(' ');
        if (n<1000 && d>=4) wrch(' ');
        if (n<100 && d>=3) wrch(' ');
        if (n<10 && d>=2) wrch(' ');
        if(n<0) { wrch('-'); n = -n; }
        wrpn(n);
    }

public static void wrpn(int n) {
    if (n>9) wrpn(n/10);
    wrch((char)('0'+(n%10)));
}

    public static void wrx(int n, int d) {
        if (n>>>28 == 0 && d>=8) wrch(' ');
        if (n>>>24 == 0 && d>=7) wrch(' ');
        if (n>>>20 == 0 && d>=6) wrch(' ');
        if (n>>>16 == 0 && d>=5) wrch(' ');
        if (n>>>12 == 0 && d>=4) wrch(' ');
        if (n>>> 8 == 0 && d>=3) wrch(' ');
        if (n>>> 4 == 0 && d>=2) wrch(' ');
        wrhex(n);
    }

public static void wrhex(int n) {
    int dig = n & 15;
    if (n>>>4 != 0) wrhex(n>>>4);
    wrch((char)((0<=dig && dig<=9) ? '0'+dig : 'A'+dig-10));
}


static void wrs(int s, int n) {
    int len = 0;
    while(getbyte(s, len)!=0) len++;
    for(int i=0; i<len; i++) wrch(getbyte(s, i));
    for(int i=len+1; i<=n; i++) wrch(' ');
}

static void wrs(String s, int n) {
    int len = s.length();
    for(int i=0; i<len; i++) wrch(s.charAt(i));
    for(int i=len+1; i<=n; i++) wrch(' ');
}

static void wrs(String s) {
    wrs(s, 0);
}

static boolean hasOperand(int op) {
  switch(op) {
    case Fnrn:case Rtrn:case Lres:case Halt:
    case Vecap:case Ind:case Stind:case Neg:case Not:
    case Mul:case Div:case Mod:case Add:case Sub:
    case Eq:case Ne:case Le:case Ge:case Lt:case Gt:
    case Lsh:case Rsh:case And:case Or:case Xor:
              return false;
    default:  return true;
  }
}

}






