package code;

import lib.*;
import args.*;
import trn.*;
import err.*;

public class Code extends Lib {

    public final static int Name      =   2;
    public final static int String    =   3;
    public final static int True      =   4;
    public final static int False     =   5;
    public final static int Fnap      =   7;
    public final static int Lv        =   8;
    public final static int Ind       =   9;
    public final static int Vecap     =  10;
    public final static int Neg       =  11;
    public final static int Not       =  12;
    public final static int Mul       =  13;
    public final static int Div       =  14;
    public final static int Mod       =  15;
    public final static int Add       =  16;
    public final static int Sub       =  17;
    public final static int Eq        =  18;
    public final static int Ne        =  19;
    public final static int Le        =  20;
    public final static int Ge        =  21;
    public final static int Lt        =  22;
    public final static int Gt        =  23;
    public final static int Lsh       =  24;
    public final static int Rsh       =  25;
    public final static int And       =  26;
    public final static int Or        =  27;
    public final static int Xor       =  28;

    public final static int Fndef     =  30;
    public final static int Rtdef     =  31;
    public final static int Assign    =  32;
    public final static int Rtap      =  33;

    public final static int Resultis  =  41;
    public final static int Vec       =  44;
    public final static int Static    =  45;
    public final static int Statvec   =  46;
    public final static int Decl      =  47;

    public final static int Rtrn      =  62;
    public final static int Fnrn      =  63;

    public final static int Lab       =  66;
    public final static int Data      =  67;
    public final static int Jt        =  68;
    public final static int Jf        =  69;
    public final static int Jump      =  70;
    public final static int Ln        =  71;
    public final static int Lp        =  72;
    public final static int Llp       =  73;
    public final static int Ll        =  74;
    public final static int Laddr     =  75;
    public final static int Sp        =  76;
    public final static int Sl        =  77;
    public final static int Stind     =  78;
    public final static int Lres      =  79;
    public final static int Entry     =  80;
    public final static int Stack     =  81;
    public final static int Printf    =  82;
    public final static int Sys       =  83;
    public final static int Halt      =  84;


    public static String opstr(int tok) {
	switch(tok) {
	default:        return "Unknown";

	case Add:       return "Add";       case And:      return "And";
        case Div:       return "Div";       case Eq:       return "Eq";
	case False:     return "False";     case Fnap:     return "Fnap";
	case Fnrn:      return "Fnrn";      case Ge:       return "Ge";
	case Gt:        return "Gt";        case Halt:     return "Halt";
        case Ind:       return "Ind";       case Jf:       return "Jf";
        case Jt:        return "Jt";        case Jump:     return "Jump";
	case Laddr:     return "Laddr";     case Le:       return "Le";
	case Ll:        return "Ll";        case Llp:      return "Llp";
	case Ln:        return "Ln";        case Lp:       return "Lp";
        case Lres:      return "Lres";      case Lsh:      return "Lsh";
	case Lt:        return "Lt";        case Mod:      return "Mod";
        case Mul:       return "Mul";       case Ne:       return "Ne";
	case Neg:       return "Neg";       case Not:      return "Not";
        case Or:        return "Or";        case Printf:   return "Printf";
	case Resultis:  return "Resultis";  case Rsh:      return "Rsh";
        case Rtap:      return "Rtap";      case Rtrn:     return "Rtrn";
	case Sl:        return "Sl";        case Sp:       return "Sp";
	case Stack:     return "Stack";     case Stind:     return "Stind";
        case Sub:       return "Sub";       case Sys:       return "Sys";
	case True:      return "True";      case Vecap:     return "Vecap";
        case Xor:       return "Xor";
	}
    }



    final static int regs   =    10;
    public final static int codev  =   100;
    final static int codet  = 10000;
    final static int memt   = 50000;
    public final static int labmax =  1000;

    private static int[] mem;
    private static int[] labv;
    private static int[] refv;
    public  static int datav;
    public  static int codep;
    public  static int datap;
    private static int datat;
    private static int stack;
    private static int stackt;

    private static boolean optCode;
    private static boolean optTrace;

    public static void init() {
	optCode  = Args.optCode;
	optTrace = Args.optTrace;
	mem = new int [memt+1];
	for (int i = 0; i<=memt; i++) mem[i] = 0;
	labv = new int [labmax+1];
	refv = new int [labmax+1];
	for (int i = 0; i<=labmax; i++) labv[i] = refv[i] = 0;

	codep = codev;
	datav = codet;
	datap = datav;
	datat = memt;
	Trn.labnumber = 0;

    }

    public static void start_interpreting() {
	try {
	    stack = datap;
	    stackt = memt;
	    
	    mem[regs+0] = 0;            // result register
	    mem[regs+1] = stack;        // p pointer
	    mem[regs+2] = stack+2;      // sp
	    mem[regs+3] = codev;        // pc
	    mem[regs+4] = Lib.maxint;       // count

	    mem[stack+0]=mem[stack+1]=mem[stack+2]=0;

	    int ret = interpret(regs, mem);   // Execute the interpreter
	    if (ret!=0)
		wrs("Return code "+ ret+"\n");
	    wrs("\nInstructions executed: "+(Lib.maxint-mem[regs+4])+"\n");

	} catch (Exception e) {
	    System.out.println("\nAn exception was raised while interpreting");
	}
    }

    private static void wrf(String form) {
	if(optCode) System.out.println(form);
    }

    private static void wrfs(String form) {
	if(optCode) System.out.println(form);
    }

    public static void outf(int op) throws Fatalerror {
	wrf(""+opstr(op));
	putcode(op);
    }

    public static void outfn(int op, int a) throws Fatalerror {
	wrf(opstr(op)+" "+a);
	putcode(op); putcode(a);
    }

    public static void outfl(int op, int lab) throws Fatalerror {
	wrf(opstr(op)+" L"+lab);
	putcode(op); putref(lab);
    }

    public static void outlab(int lab) {
	wrf("Lab L"+lab);
	setlab(lab, codep);
    }

    public static void outentry(int l1, int l2) throws Fatalerror {
	wrf("Entry L"+l1+" L"+l2);
	putref(l2);
	setlab(l1, codep);
}

    public static void outstring(int lab, String s) throws Fatalerror {
	int d = datap;
	int len = s.length();
	wrf("String L"+lab+" "+s);
	setlab(lab, datap);
	for(int i=0; i<=len; i++) {
	    char ch = (i==len) ? 0 : s.charAt(i);
	    if(i%4 == 0) putd(0);
	    putbyte(d, i, ch);
	}
    }


    public static void outstatvec(int lab, int a) throws Fatalerror {
	int i;
	wrf("Statvec L"+lab+" "+a);
	setlab(lab, datap);
	for(i=0; i<a; i++) putd(0);
    }

    public static void outvar(int lab) throws Fatalerror {
	wrf("Var L"+lab);
	setlab(lab, datap);
	putd(0);
    }
 
    private static void putcode(int w) throws Fatalerror {
	if(codep>codet)  Trn.trnerr("More code space needed");
	else             mem[codep++] = w;
    }

    private static void putd(int w) throws Fatalerror {
	if(datap>datat) Trn.trnerr("More data space needed");
	else            mem[datap++] = w;
    }

    private static void putref(int lab) throws Fatalerror {
	if(codep>codet) Trn.trnerr("More code space needed");
	else { mem[codep] = refv[lab];
	refv[lab] = codep++;
	}
    }

    public static void setlab(int lab, int addr) { labv[lab] = addr; }

    public static void resolvelabels() throws Fatalerror {
	//writes("Code.resolvelabel called, labnumber="+Trn.labnumber+"\n");
	for(int lab=1; lab<=Trn.labnumber; lab++) {
	    int p = refv[lab];
	    int labval = labv[lab];
	    if(p!=0 && labval<0)
		if(lab==1)
		    Trn.trnerr("start not defined");
		else
		    Trn.trnerr("Label "+lab+" unset");
	    while(p!=0) {
		int np = mem[p];
		mem[p] = labval;
		p = np;
	    }
	}
    }


    public static int interpret(int regs, int[]mem) {
	int retcode = 0;
	int res   = mem[regs+0];
	int pp    = mem[regs+1];
	int sp    = mem[regs+2];
	int pc    = mem[regs+3];
	int count = mem[regs+4];

	while(true) {
	    int op = mem[pc++];        // Fetch next instruction
	    if(optTrace) {
		writes("p:");    wrd(pp, 5);
		writes("  sp:"); wrd(sp, 5);
		writes(" ");     wrd(mem[sp-1], 10);
		writes(" ");     wrd(mem[sp], 10);
		writes("  ");    wrd(pc-1, 5);
		writes(": ");    wrs(opstr(op), 8);

		if(hasOperand(op)) writes(" "+mem[pc]);
		newline();
	    }
	    if(count<=0) { retcode = 3; break; } // Zero count
	    count--;
    
	    switch(op) {
	    default:     retcode = 1;       break;    // Unknown op code

	    case Halt:   retcode = mem[sp]; break;

	    case Laddr:
	    case Ln:     sp++; mem[sp] = mem[pc++];      continue;
	    case Lp:     sp++; mem[sp] = mem[pp+mem[pc++]]; continue;
	    case Llp:    sp++; mem[sp] = pp+mem[pc++];   continue;
	    case Ll:     sp++; mem[sp] = mem[mem[pc++]]; continue;
	    case Sp:     mem[pp+mem[pc++]] = mem[sp]; sp--;  continue;
	    case Sl:     mem[mem[pc++]] = mem[sp]; sp--; continue;

	    case Rtap:
	    case Fnap: {
		int opp = pp;
		int retaddr = pc+1;
		pp += mem[pc];
		pc = mem[sp];
		mem[pp+0] = opp;
		mem[pp+1] = retaddr;
		mem[pp/+2] = pc;
		sp = pp+2;
		continue;
	    }

	    case Lres:   sp++; mem[sp] = res;                continue;

	    case Fnrn:   res = mem[sp];
	    case Rtrn: {
		int npp = mem[pp];
		int npc = mem[pp+1];
		sp = pp-1;
		pp = npp;
		pc = npc;
		continue;
	    }
	    case Ind:
		mem[sp] =  mem[mem[sp]];                          continue;
	    case Neg:
		mem[sp] =  -  mem[sp];                            continue;
	    case Not:
		mem[sp] =  ~  mem[sp];                            continue;
	    case Stind:
		sp = sp-2; mem[mem[sp+2]] = mem[sp+1];            continue;
	    case Vecap:
		sp--; mem[sp] = mem[mem[sp] + mem[sp+1]];         continue;
	    case Mul:
		sp--; mem[sp] = (mem[sp])  *  (mem[sp+1]);        continue;
	    case Div:
		sp--; mem[sp] = (mem[sp])  /  (mem[sp+1]);        continue;
	    case Mod:
		sp--; mem[sp] = (mem[sp])  %  (mem[sp+1]);        continue;
	    case Add:
		sp--; mem[sp] = (mem[sp])  +  (mem[sp+1]);        continue;
	    case Sub:
		sp--; mem[sp] = (mem[sp])  -  (mem[sp+1]);        continue;
	    case Eq:
		sp--; mem[sp] = ((mem[sp]) ==  (mem[sp+1]))?-1:0; continue;
	    case Ne:
		sp--; mem[sp] = ((mem[sp]) !=  (mem[sp+1]))?-1:0; continue;
	    case Le:
		sp--; mem[sp] = ((mem[sp]) <=  (mem[sp+1]))?-1:0; continue;
	    case Ge:
		sp--; mem[sp] = ((mem[sp]) >=  (mem[sp+1]))?-1:0; continue;
	    case Lt:
		sp--; mem[sp] = ((mem[sp])  <  (mem[sp+1]))?-1:0; continue;
	    case Gt:
		sp--; mem[sp] = ((mem[sp])  >  (mem[sp+1]))?-1:0; continue;
	    case Lsh:
		sp--; mem[sp] = (mem[sp]) <<  (mem[sp+1]);        continue;
	    case Rsh:
		sp--; mem[sp] = (mem[sp]) >>> (mem[sp+1]);        continue;
	    case And:
		sp--; mem[sp] = (mem[sp])  &  (mem[sp+1]);        continue;
	    case Or:
		sp--; mem[sp] = (mem[sp])  |  (mem[sp+1]);        continue;
	    case Xor:
		sp--; mem[sp] = (mem[sp])  ^  (mem[sp+1]);        continue;
	    case Jt:
		sp--; pc = (mem[sp+1]!=0) ? mem[pc] : pc+1;       continue;
	    case Jf:
		sp--; pc = (mem[sp+1]==0) ? mem[pc] : pc+1;       continue;
	    case Resultis: 
		sp--; res = mem[sp+1];
	    case Jump:
		pc = mem[pc];                                     continue;
	    case Stack:
		sp = pp + mem[pc++];                              continue;
	    case Printf:
		sp = pp + mem[pc++] - 1;
		prf(mem[sp+1], sp+2);
		continue;
	    case Sys:sp = pp + mem[pc++] - 1;
		switch(mem[sp+1]) {
		default:
		    writes("\nBad sys("+mem[sp+1]+",...) call");
		    retcode  = 2;                    break;   
		case 0:
		    retcode  = mem[sp+2];            break;
		case 1:
		    res = interpret(mem[sp+2], mem); continue;
		case 2:
		    optTrace = mem[sp+2]!=0;         continue;
		case 3:
		    res = count; count = mem[sp+2];  continue;
		}
		break;
	    }
	    break; // Leave the while(true) loop
	}
 
	mem[regs+0] = res;
	mem[regs+1] = pp;
	mem[regs+2] = sp;
	mem[regs+3] = pc;
	mem[regs+4] = count;
	return retcode;
    }

    static void prf(int form, int argp) {
	int i=0;
	while(true) {
	    char k = getbyte(form, i++);
	    if(k==0) return;
	    if(k=='%') {
		int n=0;
		while(true) {
		    k = getbyte(form, i++);
		    if ('0'>k || k>'9') break;
		    n = 10*n + k - '0';
		}
		switch(k) {
		default:
		    wrch(k);               continue;
		case 'd':
		    wrd(mem[argp++], n);   continue;
		case 'x':
		    wrx(mem[argp++], n);   continue;
		case 's':
		    wrs(mem[argp++], n);   continue;
		}
	    }
	    wrch(k);
	}
    }

    static char getbyte(int s, int i) {
	int p = s + i/4;
	int sh = 8*(i%4);
	return (char)((mem[p]>>sh) & 255);
    }

    static void putbyte(int s, int i, int ch) {
	int p = s + i/4;
	int sh = 8*(i%4);
	int mask = 0xFFFFFFFF - (0xFF<<sh);
	mem[p] = mem[p] & mask | (ch<<sh);
    }

    public static void wrd(int n, int d) {
        if(n<0) d--;
        if (n<1000000000 && d>=10) wrch(' ');
        if (n<100000000 && d>=9) wrch(' ');
        if (n<10000000 && d>=8) wrch(' ');
        if (n<1000000 && d>=7) wrch(' ');
        if (n<100000 && d>=6) wrch(' ');
        if (n<10000 && d>=5) wrch(' ');
        if (n<1000 && d>=4) wrch(' ');
        if (n<100 && d>=3) wrch(' ');
        if (n<10 && d>=2) wrch(' ');
        if(n<0) { wrch('-'); n = -n; }
        wrpn(n);
    }

    public static void wrpn(int n) {
	if (n>9) wrpn(n/10);
	wrch((char)('0'+(n%10)));
    }

    public static void wrx(int n, int d) {
        if (n>>>28 == 0 && d>=8) wrch(' ');
        if (n>>>24 == 0 && d>=7) wrch(' ');
        if (n>>>20 == 0 && d>=6) wrch(' ');
        if (n>>>16 == 0 && d>=5) wrch(' ');
        if (n>>>12 == 0 && d>=4) wrch(' ');
        if (n>>> 8 == 0 && d>=3) wrch(' ');
        if (n>>> 4 == 0 && d>=2) wrch(' ');
        wrhex(n);
    }

    public static void wrhex(int n) {
	int dig = n & 15;
	if (n>>>4 != 0) wrhex(n>>>4);
	wrch((char)((0<=dig && dig<=9) ? '0'+dig : 'A'+dig-10));
    }


    static void wrs(int s, int n) {
	int len = 0;
	while(getbyte(s, len)!=0) len++;
	for(int i=0; i<len; i++) wrch(getbyte(s, i));
	for(int i=len+1; i<=n; i++) wrch(' ');
    }

    static void wrs(String s, int n) {
	int len = s.length();
	for(int i=0; i<len; i++) wrch(s.charAt(i));
	for(int i=len+1; i<=n; i++) wrch(' ');
    }

    static void wrs(String s) {
	wrs(s, 0);
    }

    static boolean hasOperand(int op) {
	switch(op) {

	default:
	    return false;

	case Fnap:
	case Jf:
        case Jt:
        case Jump:
	case Laddr:
	case Ll:
        case Llp:
	case Ln:
        case Lp:
        case Printf:
	case Resultis:
        case Rtap:
	case Sl:
        case Sp:
	case Stack:
	case Sys:
	    return true;
	}
    }
}
