/* This is a test program to experiment with simple spline
curves to control features like phrasing, rubato, tuning etc.

Implemented by Martin Richards (c) December 2006

The pline is controlled by a number of points of the form n:n,
as in

spline 0:1000 500:1200 1000:2000 4000:100

The program outputs a table of coordinates from the first to the
last using the given step.  

Summary of the algorithm used.

Suppose A, B, C and D are four consecutive points.  The average
gradients AB and BC is computed and placed in gb. Similarly average
gradients of BC and CD is placed in gc.  Point S is defined to have
the same x coordinate as C with the gradient of BS = gb.  Similarly,
point T is defined to have the same x coordinate as B with the
gradient of TC = gc.  Suppose alpha identifies a position on the x
axis between B and C, with 0<=alpha<=1. Denote the y coordinates of
the points on the lines BS and TC at position alpha by BS(alpha) and
TC(alpha), respectively.  The y coordinate of the point on the spline
at position alpha is the following combination of these two values.

BS(alpha) * (1-f(alpha)) + TC(alpha) * f(alpha)

where f(x) = 3x^2 - 2x^3

Note that
          f(0)    = 0          f'(0)   = 0
          f(1/2)  = 1/2        f'(1/2) = 3/2
          f(1)    = 1          f'(1)   = 0

Note that this spline is a cubic and that BS and TC are tangents and,
in some sense, the spline lies between these tangents.

*/

GET "libhdr"

LET start() = VALOF
{ LET argv = VEC 50
  LET n = 0        // The number of points given
  LET maxy = 60
  LET step = 1
  LET xv = VEC 11
  LET yv = VEC 11
 
  UNLESS rdargs(",,,,,,,,,,,step/k", argv, 50) DO
  { writef("Bad arguments for spline*n")
    RESULTIS 0
  }
  FOR i = 0 TO 3 DO
  { LET x, y = 0, 0
    UNLESS argv!i BREAK
    n := n+1
    xv!n := getnums(argv!i)
    yv!n := result2
    IF maxy<result2 DO maxy := result2
  }

  IF argv!11 & string.to.number(argv!11) DO step := result2

  xv!0     := 2*xv!1 - xv!2     // Set point 0 so that point 1 is
  yv!0     := 2*yv!1 - yv!2     // the midpoint of point 0 to point 2.
  xv!(n+1) := 2*xv!n - xv!(n-1) // Set point n+1 so that point n is
  yv!(n+1) := 2*yv!n - yv!(n-1) // the midpoint of point n-1 to point n+1.

  writef("*nSpline through points:")
  FOR i = 0 TO n+1 DO writef(" (%n %n)", xv!i, yv!i)
  newline()

  FOR i = 1 TO n-1 DO
  { LET ax,  ay  = xv!(i-1), yv!(i-1)
    LET bx,  by  = xv!(i+0), yv!(i+0)
    LET cx,  cy  = xv!(i+1), yv!(i+1)
    LET dx,  dy  = xv!(i+2), yv!(i+2)
    LET abx, aby = bx-ax,    by-ay
    LET bcx, bcy = cx-bx,    cy-by
    LET cdx, cdy = dx-cx,    dy-cy
    LET gb = (bcy + muldiv(aby,bcx,abx))/2 // Estimated gradient at b
    LET gc = (cdy + muldiv(bcy,cdx,bcx))/2 // Estimated gradient at c

    LET p = 0
    //writef("Spline from (%n %n) to (%n %n)*n", bx, by, cx, cy)
    WHILE p < bcx DO
    { LET y = spline(p, bcx, // 0 <= p <= bcx
                     by, gb, // Coordinates and gradient at b
                     cy, gc  // Coordinates and gradient at c
                    )
      writef("%i5: %i5 ", bx+p, y)
      FOR i = 0 TO 80 DO
      { wrch('**')
        IF i*maxy >= 50*y BREAK
      }
      newline()

      p := p+step
    }
    //newline()
  }

  RESULTIS 0
}

AND getnums(s) = VALOF
{ LET x, y = 0, 0
  LET p, len = 1, s%0
  LET neg = FALSE

  IF s%p = '-' DO neg, p := TRUE, p+1
  
  WHILE p<=len DO
  { LET ch = s%p
    p := p+1
    IF '0'<=ch<='9' DO { x := 10*x + ch - '0'; LOOP }
    IF ch=':' BREAK
  }
  IF neg DO x := -x

  neg := FALSE
  IF s%p = '-' DO neg, p := TRUE, p+1
  WHILE p<=len DO
  { LET ch = s%p
    p := p+1
    IF '0'<=ch<='9' DO { y := 10*y + ch - '0'; LOOP }
  }
  IF neg DO y := -y
//writef("x=%i5   y=%i5*n", x, y)


  result2 := y
  RESULTIS x
}

AND spline(p,r, by,gb, cy,gc) = VALOF
{ // Return the y coordinate corresponding to the 
  // proportion alpha (=p/r) of the x range the spline
  // of a curve that lies between the estimated tangents
  // at B and C. p is assumed to be between 0 and r.
  // This calculation rounds to the nearest integer.
  LET p1 = (muldiv(muldiv(30*r - 20*p, p, r), p, r) + 5)/10
  LET p1 = muldiv(muldiv(3*r - 2*p, p, r), p, r)

writef("p=%i5 r=%i5 p1=%i5 gb=%i5  gc=%i5", p, r, p1, gb, gc)
  
  //RESULTIS (muldiv(10*by + muldiv(10*gb,  p,  r), r-p1, r) +
  //          muldiv(10*cy - muldiv(10*gc, r-p, r),  p1,  r) + 5)/10
  RESULTIS muldiv(by + muldiv(gb,  p,  r), r-p1, r) +
           muldiv(cy - muldiv(gc, r-p, r),  p1,  r)
}

