/*
This is an experimental program that attempts to recognise the pitch
an amplitude of notes in digital sound input.  Currently, it expects
input in .wav format as 16-bit mono samples at 44100 samples per
second.

Implemented by Martin Richards (c) October 2007

See bcplprogs/accomp/doc/recogs for more details.

To simplify testing recogs.b has a builtin signal generator that can
generate a combination of square, triangle and sine waves and possibly
data from file. The resulting samples can then be fed directly into
the recogniser or written out as a .wav file, not both.  

Usage:

recogs "from,tempo/n,beats/n,rate/n,to/k,towav/k,*
       *a/n,b/n,c/n,n/n,m/n,msecs/n,plot/n"


from  <filename>    A .wav file of 16-bit mono source samples
tempo <num>         Number of beats per minute
beats <num>         Number of beats per bar
rate  <num>         The sample rate - should be 44100
to    <filename>    File for debugging output  
towav <filename>    File for .wav file output
a     <num>         MIDI note number of square wave source
b     <num>         MIDI note number of triangle wave source
c     <num>         MIDI note number of sine wave source
n     <num>         MIDI note number of start of glissando source
m     <num>         MIDI note number of end of glissando source
msecs <num>         Time to run if neither from nor n given
plot  <num>         Generate plotting data for the recogniser
                    with given MIDI number
*/

GET "libhdr"

GLOBAL {
  toname:ug
  tostream
  towavname      // If towav is given the samples are written to file
  towavstream
  fromname       // If from is given the samples are read from file
  fromstream
  dataname
  datastream
  stdin
  stdout
  format      // 16 = S16_LE
  channels    // 1=mono 2=stereo
  rate
  traceco
  polateco
  killerco

  oct0co; oct0p     // Octave coroutines
  oct1co; oct1p
  oct2co; oct2p
  oct3co; oct3p
  oct4co; oct4p
  oct5co; oct5p
  oct6co; oct6p
  oct7co; oct7p
  oct8co; oct8p

  done        // Set to TRUE during close down
  freqtab
  freqv       // Vector of recognised frequencies
  ampv        // Vector of amplitudes
  phasev      // Vector of phases
  rcgv        // Vector of note recognisers
  tempo       // Typically 160 beats per minute
  beats       // Typically 4 beats per bar
  beatpos     // Number of samples into current beat
  beatcount   // Number of beats since the start
  barnum      // An integer >=1
  samples_per_bar
  samples_per_beat
  sample_count // samples since the start of the current bar
  msecs        // Number of milli-seconds to generate, or zero
  current_amp  // Recent average amplitude
  notea        // Notes a, b and c are sustained if non zero
  noteb        // Positive notes are square wave, negative are
  notec        // sine waves.
  noten        // Notes n and m are the start and end of a
  notem        // slow glissando of quarter semitones. Each
               // note is on for 3/4 beat and off for 1/4 beat.
  dataco       // Coroutine getting samples from file FROM.
  vaco; vaparm // First (a) sustained note
  vbco; vbparm // Second (b) sustained note
  vcco; vcparm // Third (c) sustained note
  vnco; vnparm // Glissando note (from n to m)
  buf          // Buffer for output samples
  plotn        // MIDI number of note to plot
  plotcycles
  plotK        // Size of the smoothing buffer for note plotn
  plotA        // A and B values for note plotn
  plotB
  currentnote  // ddd.ddd the MIDI note in the glissando
}

MANIFEST {
  // Output buffer size for samples
  bufsize = 4096/4
  bufbytes = bufsize * bytesperword
}

LET start() = VALOF
{ LET argv = VEC 50
  LET form = "from,tempo/n,beats/n,rate/n,to/k,towav/k,*
             *a/n,b/n,c/n,n/n,m/n,msecs/n,plot/n"
  LET prevco = 0

  UNLESS rdargs(form, argv, 50) DO
  { writef("Bad arguments for recogs*n")
    RESULTIS 0
  }

  dataname := "plot/num.data"
  datastream := 0
  stdin := input()
  stdout := output()
  
  format := 16  // S16_LE
  channels := 1 // Mono
  rate := 44100

  // msecs is the data length if note a, b, or c is given
  // and neither FROM is given or a glissando requested.
  msecs := 20_000  // 20 seconds

  fromname := 0
  fromstream := 0
  toname := 0
  tostream := 0
  towavname := 0
  towavstream := 0

  tempo := 160    // Beats per minute
  //tempo := 60     // Beats per minute
  barnum := 1     // An integer >=1
  beats  := 4     // Typically 4
  sample_count := 0

  notea := 0      // Three sustained notes
  noteb := 0
  notec := 0

  noten := 0      // The lowest note of the glissando
  notem := 0      // The highest note of the glissando

  plotn := -1     // MIDI number of note to plot

  dataco := 0

  vaco, vaparm := 0, 0
  vbco, vbparm := 0, 0
  vcco, vcparm := 0, 0
  vnco, vnparm := 0, 0

  oct0co, oct0p := 0, 0    // Octave coroutines
  oct1co, oct0p := 0, 0
  oct2co, oct0p := 0, 0
  oct3co, oct0p := 0, 0
  oct4co, oct0p := 0, 0
  oct5co, oct0p := 0, 0
  oct6co, oct0p := 0, 0
  oct7co, oct0p := 0, 0
  oct8co, oct0p := 0, 0

  done := FALSE
//writef("calling setfreqtab*n")
  setfreqtab()
//abort(1000)

  IF argv!0  DO fromname :=   argv!0    // from
  IF argv!1  DO tempo    := !(argv!1)   // tempo
  IF argv!2  DO beats    := !(argv!2)   // beats
  IF argv!3  DO rate     := !(argv!3)   // rate
  IF argv!4  DO toname   :=   argv!4    // to
  IF argv!5  DO towavname:=   argv!5    // towav
  IF argv!6  DO notea    := !(argv!6)   // a
  IF argv!7  DO noteb    := !(argv!7)   // b
  IF argv!8  DO notec    := !(argv!8)   // c
  IF argv!9  DO noten    := !(argv!9)   // n
  IF argv!10 DO notem    := !(argv!10)  // m
  IF argv!11 DO msecs    := !(argv!11)  // msecs
  IF argv!12 DO plotn    := !(argv!12)  // plot

  samples_per_bar := rate * 60 * beats / tempo
  samples_per_beat := samples_per_bar / beats

  //sample_count := samples_per_bar * 5 / 8  // For beanbag.wav

  barnum := sample_count/samples_per_bar + 1

  writef("Samples per bar  = %n*n", samples_per_bar)
  writef("Samples per beat = %n*n", samples_per_beat)

  IF fromname DO
  { // One source comes from file
    fromstream := findinput(fromname)
    UNLESS fromstream DO
    { writef("Trouble with file: %s*n", fromname)
      GOTO fin
    }
    dataco := initco(datacofn, 500, fromstream)
    sawritef("Created dataco=%n for stream %s*n", dataco, fromname)
  }

  IF towavname DO
  { // Samples in .wav format to file
    towavstream := findoutput(towavname)
    UNLESS towavstream DO
    { writef("Trouble with file: %s*n", towavname)
      GOTO fin
    }
    writef("Sending samples to file %s*n", towavname)
    selectoutput(towavstream)
    wrhdr()
    selectoutput(stdout)
  }

  IF toname DO
  { // Output from the recognisers
    tostream := findoutput(toname)
    UNLESS tostream DO
    { writef("Trouble with file: %s*n", toname)
      GOTO fin
    }
    selectoutput(tostream)
  }

  IF notea DO
  { // Note source a
    writef("natea=%n*n", notea)
    TEST notea>0
    THEN vaco := initco(squarefn, 310, freqtab!notea,    5000)
//    THEN vaco := initco(squarefn, 300, 440_000,    5000)
    ELSE vaco := initco(sinefn,   310, freqtab!(-notea), 5000)
    vaparm := result2 // -> [freq, amplitude]
    sawritef("Created vaco=%n for note %n*n", vaco, notea)
  }

  IF noteb DO
  { // Note source b
    TEST noteb>0
    THEN vbco := initco(trianglefn, 320, freqtab!noteb,    5000)
    ELSE vbco := initco(sinefn,   320, freqtab!(-noteb), 5000)
    vbparm := result2 // -> [freq, amplitude]
    sawritef("Created vbco=%n for note %n*n", vbco, noteb)
  }

  IF notec DO
  { // Note source c
    TEST notec>0
    THEN vcco := initco(squarefn, 330, freqtab!notec,    5000)
    ELSE vcco := initco(sinefn,   330, freqtab!(-notec), 5000)
    vcparm := result2 // -> [freq, amplitude]
    sawritef("Created vcco=%n for note %n*n", vcco, notec)
  }

  IF noten DO
  { // Note source n for the glissando
    // Glissando one octave by default
    IF notem=0 DO notem := noten+12
    TEST noten>0
    THEN vnco := initco(squarefn, 400, freqtab!noten,    5000)
    ELSE vnco := initco(sinefn,   400, freqtab!(-noten), 5000)
    vnparm := result2 // -> [freq, amplitude]
    sawritef("Created vnco=%n for note %n*n", vnco, noten)
  }

  IF notea & vaco=0 | noteb & vbco=0 | notec & vcco=0 |
     noten & vnco=0 DO
  { writef("More space needed*n")
    GOTO fin
  }

  //                              n   rate      A   B
  oct0co := initco(octavefn, 302, 0, 44100/16, 85, 30)
  oct0p  := result2
  oct1co := initco(octavefn, 300, 1, 44100/ 8, 85, 30)
  oct1p  := result2
  oct2co := initco(octavefn, 300, 2, 44100/ 4, 85, 30)
  oct2p  := result2
  oct3co := initco(octavefn, 300, 3, 44100/ 2, 85, 30)
  oct3p  := result2
  oct4co := initco(octavefn, 300, 4, 44100,    85, 30)
  oct4p  := result2
  oct5co := initco(octavefn, 300, 5, 44100,    85, 30)
  oct5p  := result2
  oct6co := initco(octavefn, 300, 6, 44100,    85, 30)
  oct6p  := result2
  oct7co := initco(octavefn, 305, 7, 44100,    85, 30)
  oct7p  := result2
  oct8co := initco(octavefn, 310, 8, 44100,    85, 30)
  oct8p  := result2

  freqv      := getvec(127)
  ampv       := getvec(127)
  phasev     := getvec(127)
  buf        := getvec(bufsize-1)

  UNLESS freqv & ampv & phasev & buf &
         oct0co & oct1co & oct2co & oct3co & oct4co &
         oct5co & oct6co & oct7co & oct8co DO
  { writef("More memory needed*n")
    GOTO fin
  }

  FOR i = 0 TO 127 DO
    freqv!i, ampv!i, phasev!i := 0, 0, 0

  //tstfindamp()
  //GOTO fin

  newline()

  sawritef("Creating note recognisers*n")

  { // Create the note recogniser coroutines
    // and give them to their octave coroutines.

    IF oct0p FOR n = 21 TO 23 DO // Notes: a0 .. b0
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 0, 44100/16, n,  10,   95, 50)
      LET node = result2
      node!1 := !oct0p  // Insert this node at the head of the list
      !oct0p := node
    }

    IF oct1p FOR n = 24 TO 35 DO // Notes: c1 .. b1
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 1, 44100/8, n,    6,  85, 50)
      LET node = result2
      node!1 := !oct1p  // Insert this node at the head of the list
      !oct1p := node
    }

    IF oct2p FOR n = 36 TO 47 DO // Notes: c2 .. b2
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 2, 44100/4,  n,   8,  85, 50)
      LET node = result2
      node!1 := !oct2p  // Insert this node at the head of the list
      !oct2p := node
    }

    IF oct3p FOR n = 48 TO 59 DO // Notes: c3 .. b3
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 3, 44100/2,  n,  1,  85, 50)
      LET node = result2
      node!1 := !oct3p  // Insert this node at the head of the list
      !oct3p := node
    }

    IF oct4p FOR n = 60 TO 71 DO // Notes: c4 .. b4
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 4, 44100,    n,  24,  94, 50)
      LET node = result2
      node!1 := !oct4p  // Insert this node at the head of the list
      !oct4p := node
    }

    IF oct5p FOR n = 72 TO 83 DO // Notes: c5 .. b5
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 5, 44100,    n , 32,  95, 50)
      LET node = result2
      node!1 := !oct5p  // Insert this node at the head of the list
      !oct5p := node
    }

    IF oct6p FOR n = 84 TO 95 DO // Notes: c6 .. b6
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 6, 44100,    n,  36,  95, 50)
      LET node = result2
      node!1 := !oct6p  // Insert this node at the head of the list
      !oct6p := node
    }

    IF oct7p FOR n = 96 TO 107 DO // Notes: c7 .. b7
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 7, 44100,    n,  40,  95, 50)
      LET node = result2
      node!1 := !oct7p  // Insert this node at the head of the list
      !oct7p := node
    }

    IF oct8p FOR n = 108 TO 108 DO // Note: c8
    IF plotn<0 | n=plotn DO
    { //                            oct            n cycles  A   B
      LET co   = initco(recogfn, 300, 8, 44100,    n,  20,  85, 30)
      LET node = result2
      node!1 := !oct8p  // Insert this node at the head of the list
      !oct8p := node
    }
  }

  killerco := createco(deleteco, 100)
  traceco  := createco(tracefn, 300)
//  polateco := initco(polatefn, 300, prevco, 6)

  current_amp := 10

  done := FALSE

  // done becomes TRUE when either
  // or there are no remaining sample coroutines.
  // or dataco=0 and vnco=0 and msecs milli-seconds has elapsed.

  // dataco becomes zero when the source file is exhausted
  // vnco becomes zero when it has finished its glissando.

  UNLESS plotn<0 DO
  { // Generate files plot/eg1.gp and num.data
    // for use by gnuplot
    LET gpname = "plot/eg1.gp"
    LET gpstream = findoutput(gpname)

    UNLESS gpstream DO
    { writef("Trouble with file: %s*n", gpname)
      GOTO fin
    }
    writef("Creating plotting files: %s and %s*n", gpname, dataname)
    selectoutput(gpstream)
    
    writes("set terminal latex*n")
    writes("set output *"eg1.tex*"*n")

    writes("unset key*n")
    writes("set size 7/5.,6/3.*n")
    writes("set pointsize 0.1*n")
    writes("set format xy *"$%g$*"*n")
    writef("set title *"Note %n Cycles=%n K=%n A=%n B=%n*"*n",
            plotn, plotcycles, plotK, plotA, plotB)
    writef("set cntrparam cubicspline*n")
    writef("plot [%n:%n] [-500:20000] *"../%s*" with linespoints 1 12*n",
            noten-2, notem+2, dataname)
    //writef("plot [%9.3d:%9.3d] [-500:20000] *"../%s*" with linespoints 1 12*n",
    //        freqtab!noten, freqtab!notem, dataname)
    endstream(gpstream)

    datastream := findoutput("plot/num.data")
    UNLESS datastream DO
    { writef("Trouble with file: %s*n", "plot/num.data")
      GOTO fin
    }
  }

  IF datastream DO selectoutput(datastream)

sawritef("Calling dosamples*n")

  dosamples()

fin:
  IF datastream DO endstream(datastream)
  IF tostream UNLESS tostream=stdout DO endstream(tostream)
  IF towavstream UNLESS towavstream=stdout DO endstream(towavstream)
  IF fromstream UNLESS fromstream=stdin DO endstream(fromstream)
  IF freqv  DO freevec(freqv)
  IF ampv   DO freevec(ampv)
  IF phasev DO freevec(phasev)
  IF vaco DO deleteco(vaco)
  IF vbco DO deleteco(vbco)
  IF vcco DO deleteco(vcco)
   selectoutput(stdout)
  //deleteco(polateco)
  writef("*nEnd of run*n")


  RESULTIS 0
}

AND rdhdr() BE
{ LET v = VEC 10
  FOR i = 0 TO 10 DO v!i := 0
  readwords(v, 11) // Read 11 words into v!0 .. v!10

  //FOR i = 0 TO 10 DO writef("v!%i2: %x8*n", i, v!i)

  UNLESS v%0='R' & v%1='I' & v%2='F' & v%3='F' DO
    writef("Not RIFF format*n")
  UNLESS v%20=1 & v%21=0 DO writef("Not linear encoding*n")
  IF v%22=1 & v%23=0 DO writef("Mono*n")
  IF v%22=2 & v%23=0 DO writef("Stereo*n")
  UNLESS v%34=16 & v%35=0 DO writef("Not 16-bit samples*n")
  rate := v!6
  writef("Sample rate = %n*n", rate)
//abort(3000)  
}

AND wrhdr() BE
{   writes("RIFF")        //  0: R I F F
    wr4(36+0)             //  4: size of this file - 8
    writes("WAVE")        //  8: W A V E
    writes("fmt ")        // 12: f m t
    wr4(format)           // 16: fmt subchunk size is 16
    wr2(1)                // 20: 1 = linear quantisation
    wr2(channels)         // 22: 1 = mono, 2=stereo
    wr4(rate)             // 24: samples per second
    wr4(rate*channels*16) // 28: bytes per second
    wr2(2*channels)       // 32: bits/8 * mode  = 1, 2 or 4
    wr2(16  )             // 34: bits per sample  = 8 or 16
    writes("data")        // 36: d a t a
    //wr4(byte_rate * 1)    // 40: number of bytes of data or zero
    wr4(-1)               // 40: number of bytes of data or -1
}

AND datacofn(args) BE
{ LET datastream = args!0
  LET oldin = ?
  LET inbuf = VEC 255

//sawritef("datacofn: entered*n")

  cowait() // Return to initco call

  selectinput(datastream)
  rdhdr()

  { LET oldin = input()
    LET len = ?
    selectinput(datastream)
    len := readwords(inbuf, 256)
    selectinput(oldin)

//sawritef("datacofn: len=%n*n", len)
    IF len<=0 BREAK 
    FOR i = 0 TO bytesperword*len-2 BY 2 DO
    { LET x = inbuf%i | inbuf%(i+1)<<8
      UNLESS (x & #x8000) = 0 DO x := x - #x10000
//sawritef("datacofn: sample: %x4*n", x)
      cowait(x)
    }
  } REPEAT

sawritef("dataco committing suicide*n")
  done := TRUE
  dataco := 0
  die()
}

AND dosamples() BE UNTIL done | plotn>=0 & vnco=0 DO
{ LET x = 0

  IF done RETURN
  IF noten>0 & vnco=0 RETURN
  IF fromname=0 & noten<=0 & msecs>0 DO
  { //sawritef("dosamples: %i7 rate=%n msecs=%n*n", sample_count, rate, msecs)
    //abort(9999)
     IF sample_count > rate * msecs /1000 RETURN
  }

  IF dataco DO x := x + callco(dataco)

  IF vaco   DO x := x + callco(vaco)
  IF vbco   DO x := x + callco(vbco)
  IF vcco   DO x := x + callco(vcco)

  IF vnco   DO
  { LET beatpos = sample_count MOD samples_per_beat
    LET beatcount = sample_count / samples_per_beat
    LET n = noten + beatcount/4
    LET p = beatpos * 4 / samples_per_beat // = 0, 1, 2 or 3
    LET f  = ((4-p)*freqtab!n + p*freqtab!(n+1))/4 // Interpolate frequency
    vnparm!0 := f  // Set the current frequency
    currentnote := noten*1000 + 250*beatcount
    IF n > notem DO
    { // The glissando is exhausted
      deleteco(vnco)
      done, vnco := TRUE, 0
      sawritef("End of glissando*n")
      GOTO send
    }

    TEST p < 3
    THEN { vnparm!1 := 5000 // set amplitude 5000 for first 3/4
           IF plotn>=0 & beatpos = samples_per_beat/2 DO
           { writef("%8.3d %i7*n", currentnote, ampv!plotn)
             sawritef("%8.3d %i7*n", currentnote, ampv!plotn)
             //writef("%8.3d %i7*n", f, ampv!plotn)
             //sawritef("%8.3d %i7*n", f, ampv!plotn)
           }
         }
    ELSE { vnparm!1 := 0 // set amplitude to zero for last 1/b
         }
    x := x + callco(vnco)
//sawritef("%i6: currentnote=%8.3d f=%9.3d amp=%i5 x=%i6*n",
//          sample_count, currentnote, vnparm!0, vnparm!1, x)
//abort(5555)
  }

send:
  barnum := sample_count / samples_per_bar + 1

  //sawritef("sample %i7: %i6*n", sample_count, x)

  //IF sample_count MOD 20 = 0 DO abort(1000)

  IF towavstream DO
  { LET oldout = output()
    selectoutput(towavstream)
    wr2(x)
    selectoutput(oldout)
//writef("writing sample: %i7*n", x)
//abort(6666)
    // Don't send the sample to the recognisers
    GOTO nxt
  }

  IF oct8co DO callco(oct8co, x)
  IF oct7co DO callco(oct7co, x)
  IF oct6co DO callco(oct6co, x)
  IF oct5co DO callco(oct5co, x)
  IF oct4co DO callco(oct4co, x)
  IF sample_count MOD 2 GOTO nxt
  IF oct3co DO callco(oct3co, x)
  IF sample_count MOD 4 GOTO nxt
  IF oct2co DO callco(oct2co, x)
  IF sample_count MOD 8 GOTO nxt
  IF oct1co DO callco(oct1co, x)
  IF sample_count MOD 16 GOTO nxt
  IF oct0co DO callco(oct0co, x)


nxt:
  callco(traceco)
  sample_count := sample_count+1
}


/*
polatefn is the main function of an interpolation coroutine
that is probably not required in this version of the program
but is kept just in case.

a, b, c, d are four consecutive input samples
x in the range 0 to 1 is the position of a point between b and c
t1 is a line through b parallel to ac
t2 is a line through c parallel to bd
p and q are points on the lines t1 and t2, respectively, at position x
the interpolated point at position x is (r)p+(1-r)q
where r = 1 -3xx +2xxx
note that r(0)=1, r(1/2)=1/2, r(1)=0,
          r'(0)=r'(1)=0
and r(x) = 1 - r(1-x)
The interpolated curve is a cubic spline that has tangents t1 and t2
at b and c.
*/

AND polatefn(args) BE
// args!0 is the target coroutine for the interpolated samples.

// Once started this coroutine will send a number of interpolated values
// to the target coroutine for each value obtained by cowait().
// The samples returned by cowait() have a rate of 44100. Octave 4
// expects samples at this rate. The rate must be doubled for each
// higher octave.
{ LET target = args!0   // Target supplied by initco
  LET octnum = args!1   // Octnum 4 rate  44100
                        // Octnum 5 rate  88200   ie x2
                        // Octnum 6 rate 176400   ie x4
                        // Octnum 7 rate 352800   ie x8
  LET k = 1<<(octnum-4) // Number of interpolated values per input value.
  LET a, b, c, d = 0, 0, cowait(), 0  // c is the first sample
  // a, b, c and d are four consecutive input values. 
  LET k3 = k*k*k
  LET k_2 = 2*k

//writef("%i2: %i7*n", 0, c)
  callco(target, c) // Send first sample

  // Optimise the special case where no interpolation is required.
  IF k=1 UNTIL done DO callco(target, cowait())

  UNTIL done DO
  { d := cowait()       // d is the next sample

//writef("*nVal %i7*n", b)
//abort(1000)

    FOR i = 1 TO k UNLESS done DO
    { LET px  = b + ((c-a)*i    + k)/k_2
      LET qx  = c - ((d-b)*(k-i)+ k)/k_2
      LET yk3 = k3 - 3*i*i*k + 2*i*i*i
      LET val = (yk3*px + (k3-yk3)*qx + k3/2) / k3
//writef("%i7 %i2: %i7*n", sample_count, i, val)
//abort(4000)
      callco(target, val)
    }

    a := b
    b := c
    c := d
  }

  die() // Commit suicide
}

AND octavefn(args) BE
{ // args -> [ octnum, rate]
  LET colist = 0      // List of note recogniser coroutines
  LET octnum = args!0 // 0..9  -- The octave number
  LET rate   = args!1 // The sample rate for this octave

  // Octave 4 to 8 have a sample rate of 44100
  // octave 3 has a sample rate of 22050
  // The rate is halved for each lower octave.
  // This ensures that the sample buffer sizes for recognisers
  // in octave 0 to 4 are between about 80 and 200.
  // The buffer size for note in the higher octaves are smaller.

  cowait(@colist) // Return this pointer as the result2 value
                  // of the initco call.

  UNTIL done DO
  { LET x = cowait() // Get a sample
    LET p = colist   // p=0 or p->[next, co]
//IF octnum=4 DO
//        writef("octavefn %n: rate=%i8 x=%i7 sample=%n*n",
//                octnum, rate, x, sample_count)
//IF octnum=4 DO 
//        abort(2222)
    // Hand the sample to each of this octave's recognisers
    WHILE p DO
    { LET co, next = p!0, p!1
      callco(co, x)
      p := next
    }
  }
}

AND recogfn(args) BE
{ // args!0  is the octave number 
  // args!1  is sample rate for this octave
  // args!2  is the MIDI number of the note to listen for.
  // args!3  is cycles
  // args!4  is 
  LET co, next = currco, 0 // A node for insertion into colist
  LET octnum = args!0 // The octave number containing this recogniser
  LET rate   = args!1 // The sample rate for this recogniser
  LET n      = args!2 // 0..127 -- The MIDI note number 0..127
                      // actually only 21..108, the piano range
  LET cycles = 1 //args!3 // Number of cycles to merge before
                      // analysing the sample buffer
  LET A      = args!4 // A and B control the rate of decay and
  LET B      = args!5 // the positive feedback effect.

  LET freq    = freqtab!n // The frequency (ddd.ddd) of MIDI note n
  // Only recognise notes between freqlo and freqhi
  // The recognised note must be within a 2/5 of a semitone
  // of the in-tune note.
  LET freqlo  = (3*freq+2*freqtab!(n-1))/5 // Lower limit
  LET freqhi  = (3*freq+2*freqtab!(n+1))/5 // Lower limit

  LET upb = (1000*rate + freq/2) / freq // Sample buffer size
  LET buf = getvec(upb)     // The sample buffer for one cycle
  LET K = upb / 4 + 1       // The smoothing buffer size
  LET smoothbuf = getvec(K) // The smoothing buffer
  LET smoothbufp, smoothsum = 0, 0
  LET phase = 0     // ddd.ddd ie in units of 1/1000
                    // phase/1000 is the position in buf of the
                    // leading edge of the wave.
  LET phaserate = 0 // ddd.ddd
                    // phaserate is the smoothed amount by which phase
                    // changes each cycle.
                    // From upb, rate and phaserate we can calculate
                    // and accurate frequency of a actual detected note.

  IF plotn=n DO plotcycles, plotK, plotA, plotB := cycles, K, A, B

writef("recogfn: octnum=%n rate=%i8 n=%i3 freq=%8.3d *
       *cycles=%i2 A=%i2 B=%i4 upb=%i3 K=%i2  ",
        octnum, rate, n, freq, cycles, A, B, upb, K)
wrnotename(n)
newline()
//abort(1000)
  // Try to find a good phase rate for the given frequency
  FOR pr = -2000 TO +2000 IF frequency(rate, upb, pr)> freq DO
  { phaserate := pr
    BREAK
  }

//writef("recogfn: n=%i3 freq=%8.3d upb=%n phaserate=%8.3d*n",
//                 n,    freq,      upb, phaserate)

  cowait(@co) // Return a pointer to [currco, next] to the initco
              // call in result2.

  // Initialise the smoothing and sample buffers
  FOR i = 0 TO K-1 DO smoothbuf!i := 0
  FOR i = 1 TO upb DO buf!i := 0

  phase  := 1 // Set an initial phase value

  { // Start of main recognition loop
    LET amp, newphase = ?, ?
    LET phasediff = ?
    LET freq = ?    // The actual frequency detected.
    LET csum, oldcsum, csumdiff = 0, 0, 0

    // Merge in the next cycles worth of smoothed samples cycles
    FOR c = 1 TO cycles FOR i = 1 TO upb UNLESS done DO
    { LET x = cowait() // get a sample
      IF FALSE DO
      { // Conditionally apply smoothing
        smoothsum := smoothsum + x - smoothbuf!smoothbufp
        smoothbuf!smoothbufp := x
        smoothbufp := (smoothbufp+1) MOD K

        x := smoothsum / K  // x is now the next smoothed sample
      }


IF n=-69 DO
writef("recogfn: time=%9.3d n=%i3 i=%i3 x=%n*n",
       muldiv(sample_count,1000,44100), n, i, x)

      // Merge x into the cycle buffer using decay parameters A and B.
      buf!i := x //(A*buf!i + B*x)/100
    }

    // The cycle buffer is now ready to be analysed

    FOR i = 1 TO upb DO csum := csum+buf!i
    csumdiff := (csum - oldcsum) / cycles / upb
    oldcsum := csum

    amp       := findamp(n, buf, upb, csumdiff)
    newphase  := result2
    phasediff := (newphase-phase)/cycles // phase shift per cycle
    IF phasediff > upb*1000/2 DO phasediff := phasediff - upb*1000
    IF phasediff <-upb*1000/2 DO phasediff := phasediff + upb*1000
    phaserate    := (phaserate*49 + phasediff+25) / 50
    phase        := newphase 
    freq         := frequency(rate, upb, phaserate)

IF n=69 | n=-61 DO
{ prbuf(buf, upb, phase/1000, rate)
  writef("time=%10.4d csumdiff=%n*n", muldiv(sample_count, 10000, rate), csumdiff)
  writef("%10.3d: note=%i3 freq=%8.3d actual=%8.3d amp=%i5 *
         *phase=%8.3d phasediff=%8.3d phaserate=%8.3d*n",
         sample_count*10/441, n, freqtab!n, freq, amp,
         phase, phasediff, phaserate)
  abort(1111)
}
    UNLESS freqlo<freq<freqhi DO amp := 0
    freqv!n      := freq
    ampv!n       := amp
    phasev!n     := phase 

IF 62<=n<=61 DO
{ FOR i = 1 TO upb DO
  { writef(" %i5", buf!i)
    IF i MOD 10 = 0 DO newline()
  }
  newline()

  writef("recogfn: n=%i3 freq=%8.3d upb=%i3 amp=%i5 phase=%8.3d phaserate=%8.3d ",
                   n,    freqv!n,   upb,    amp,    phase,      phaserate)
  IF amp>300 DO wrnotename(n)
  newline()
  abort(2000)
}

  } REPEATUNTIL done

  freevec(buf)
  die()
}

AND tstfindamp() BE
{ tst(4, 100, 200, 300, 400)
  tst(5, 100, 200, 300, 400, 500)

  tst(4,  100,  100, -100, -100)
  tst(4, -100,  100,  100, -100)
  tst(4, -100, -100,  100,  100)
  tst(4,  100, -100, -100,  100)

  tst(5,  101,  102,    3, -104, -105)
  tst(5, -105,  101,  102,    3, -104)
  tst(5, -104, -105,  101,  102,    3)
  tst(5,    3, -104, -105,  101,  102)
  tst(5,  102,    3, -104, -105,  101)
}

AND tst(upb, a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,x,y,z) BE
{ LET amp = ?
  LET phase = ?
  newline()
  FOR i = 1 TO upb DO writef(" %i4", (@upb)!i)
  newline()
  amp := findamp(0, @upb, upb, 0)
  phase := result2
  writef("*nupb=%i2 amp=%i4 phase=%8.3d*n", upb, amp, phase)
}

AND findamp(n, buf, upb, csumdiff) = VALOF TEST (upb & 1) = 0
THEN { // upb is even
       LET h = upb/2            // upb = h+h 
       LET s1, s2 = 0, 0        // The left and right sums.
       LET p, q = buf+1, buf+h+1
       LET m, r = 0, p
       LET a, b = ?, ?
       LET lim = buf+upb

       // Initialise the left and right sums.
       WHILE q<=lim DO
       { s1, s2 := s1+!p, s2+!q
         s1 := s1 - (p-buf-1) * csumdiff / upb
         s2 := s2 - (q-buf-1) * csumdiff / upb
         p, q := p+1, q+1
       }
       p, q := buf+1, buf+h+1

       { IF m<s1-s2 DO m, r := s1-s2, p
         IF m<s2-s1 DO m, r := s2-s1, q
//writef("p=%i2 q=%i2 s1=%i4 s2=%i4 m=%i4 r=%i2*n", p-buf, q-buf, s1, s2, m, r-buf)
         IF q>=lim BREAK
         a, b := !p, !q
         a := a - (p-buf-1) * csumdiff / upb
         b := b - (q-buf-1) * csumdiff / upb
         s1, s2 := s1+b-a, s2+a-b
         p, q := p+1, q+1
       } REPEAT

       result2 := 1000*(r-buf)  // ddd.ddd scaled phase
       RESULTIS (m + h)/upb     // rounded average amplitude
     }
ELSE { // upb is odd
       LET h = upb/2 + 1        // upb = h+h-1 
       LET s1, s2 = 0, 0        // Left and right sums
       LET p, q = buf+1, buf+h
       LET m, r = 0, p
       LET lim = buf+upb
       LET a, b = !p, !q

       // Initialise s1 and s2
       WHILE q<lim DO
       { q := q+1
         s1, s2 := s1+!p, s2+!q
         s1 := s1 - (p-buf-1) * csumdiff / upb
         s2 := s2 - (q-buf-1) * csumdiff / upb
         p := p+1
       }
       p, q := buf+1, buf+h

       { IF m<s2-s1 DO m, r := s2-s1, q
//writef("p=%i2 q=%i2 s1=%i4 s2=%i4 m=%i4 r=%i2*n", p-buf, q-buf, s1, s2, m, r-buf)
         IF q>=lim BREAK
         q := q+1
         a := !p
         a := a - (p-buf-1) * csumdiff / upb
         s1 := s1+b-a
         IF m<s1-s2 DO m, r := s1-s2, p
//writef("p=%i2 q=%i2 s1=%i4 s2=%i4 m=%i4 r=%i2*n", p-buf, q-buf, s1, s2, m, r-buf)
         p := p+1
         b := !q
         b := b - (q-buf-1) * csumdiff / upb
         s2 := s2+a-b
       } REPEAT

       result2 := 1000*(r-buf)  // ddd.ddd scaled phase
       RESULTIS (m + h)/upb     // rounded average amplitude
     }

AND frequency(rate, upb, phaserate) = VALOF
{ LET f = muldiv(rate, 1_000_000, upb*1_000 + phaserate)
  //writef("*nfrequency: rate=%n upb=%n phaserate=%8.3d => f=%8.3d*n",
  //        rate, upb, phaserate, f)
  //abort(3000)
  RESULTIS f // ddd.ddd   scaled frequency
}

AND setfreqtab() BE
{ // Set freqtab so that freetab!n = 1000 * the note frequency
  // where n is the MIDI note number. n=60 for middle C (C4).
//writef("Setting freqtab*n")
  freqtab := TABLE
     8_176,    8_662,    9_178,    9_723, //   0 c-1  .. dis-1
    10_301,   10_914,   11_563,   12_250, //   4 e-1  .. g-1
    12_979,   13_750,   14_568,   15_434, //   8 gis-1.. b-1

    16_352,   17_324,   18_355,   19_446, //  12 c0   .. dis0
    20_602,   21_827,   23_125,   24_500, //  16 e0   .. g0
    25_957,   27_500,   29_136,   30_868, //  20 gis0 .. b0
                                          //  21 piano bottom note

    32_704,   34_648,   36_709,   38_891, //  24 c1   .. dis1
    41_204,   43_654,   46_250,   49_000, //  28 e1   .. g1
    51_914,   55_000,   58_271,   61_736, //  32 gis1 .. b1
  
    65_407,   69_296,   73_417,   77_782, //  36 c2 .. b2
    82_407,   87_308,   92_499,   97_999, //  40
   103_827,  110_000,  116_541,  123_471, //  44

   130_813,  138_592,  146_833,  155_564, //  48 c3 .. b3
   164_814,  174_615,  184_998,  195_998, //  52
   207_653,  220_000,  233_082,  246_942, //  56

   261_626,  277_183,  293_665,  311_127, //  60 c4 .. b4
   329_628,  349_229,  369_995,  391_996, //  64
   415_305,  440_000,  466_164,  493_884, //  68

   523_252,  554_366,  587_330,  622_254, //  72 c5 .. b5
   659_256,  698_457,  739_989,  783_991, //  76
   830_610,  880_000,  932_328,  987_767, //  80

  1046_503, 1108_731, 1174_660, 1244_508, //  84 c6 .. b6
  1318_511, 1396_913, 1479_978, 1567_982, //  88
  1661_219, 1760_000, 1864_655, 1975_534, //  92

  2093_005, 2217_461, 2349_319, 2489_016, //  96 c7 .. b7
  2637_021, 2793_826, 2959_956, 3135_964, // 100
  3322_438, 3520_000, 3729_310, 3951_067, // 104

  4186_009, 4434_922, 4698_637, 4978_032, // 108 c8 .. b8
                                          // 108 piano top note
  5274_041, 5587_652, 5919_911, 6271_927, // 112
  6644_875, 7040_000, 7458_620, 7902_133, // 116

  8372_018, 8869_844, 9397_273, 9956_063, // 120 c9 .. f9
 10548_082,11175_303,11839_821,12543_854  // 124 fis9 .. g9

  // Check the table
  checktab(116,  6644_875) //
  checktab(117,  7040_000) // A8 (A4=440_000)
  checktab(118,  7458_620) //
  checktab(119,  7902_133) // B8
  checktab(120,  8372_018) // C9
  checktab(121,  8869_844) //
  checktab(122,  9397_273) // D9
  checktab(123,  9956_063) //
  checktab(124, 10548_082) // E9
  checktab(125, 11175_303) // F9
  checktab(126, 11839_821) // 
  checktab(127, 12543_854) // G9

  FOR n = 1 TO 127 DO
  { LET f1 = freqtab!(n-1)
    LET f2 = freqtab!n
    LET f = (muldiv(f2*10, 0_943874313, 1_000000000)+5)/10
    LET err = ABS (f-f1)
    IF err>1 DO
    { writef("n=%i3 f2=%9.3d  f1=%9.3d f=%9.3d err=%n*n", n, f2, f1, f, f-f1)
      abort(999)
    }
  }

  IF FALSE DO
  { LET f = 12543_854

    UNLESS freqtab!69=440_000 DO
    { writef("A4 should have frequency 440.000 not %8.3d*n", freqtab!69)
      abort(999)
    }

    FOR n = 127 TO 0 BY -1 DO
    { LET f1 = freqtab!n
      UNLESS f1=f DO
      { writef("n=%i3 f=%9.3d  f1=%9.3d*n", n, f, f1)
        abort(998)
      }
      // Multiply by f by 2**-(1/12), rounding
      f := (muldiv(f*10, 0_943874313, 1_000000000)+5)/10
    }
  }
//abort(9999)
}

AND checktab(n, f) BE WHILE n>=0 DO
  { UNLESS freqtab!n = f DO
    { writef("note=%i3 change %8.3d to %8.3d*n", n, freqtab!n, f)
      abort(1000)
    }
    n, f := n-12, (f+1)/2
  }


AND tracefn() BE
{ // This is the body of the trace coroutine
  LET samplepos = sample_count MOD samples_per_bar
  LET beatpos = 1_00+samplepos*beats*100/samples_per_bar
  LET minamp, maxamp = 10, 10

  STATIC { lastpos=0 }

  UNLESS plotn<0 RETURN

  //UNLESS beatpos MOD 10 = 0 RETURN
  //IF beatpos=lastpos RETURN
  //lastpos := beatpos

  // Write a newline at the start of each bar
  //IF beatpos MOD (100*beats) = 100 DO
  //{ newline()
  //  sawritef("Bar number %n*n", barnum)
  //}

//  writef("bar %i3:%4.2d  ", barnum, beatpos)

//  writef("%10i: sample_count MOD %n = %i5*n",
//          sample_count, 44100/100, sample_count MOD (44100/100))

  UNLESS sample_count MOD (44100/20) = 0 RETURN
//abort(1234)
  IF sample_count=0 DO
  { newline()
    writef("     Time        Count     *
           *C1          C2          C3           C4          C5          *
           *C6          C7*n")
    writef("                           *
           *|           |           |            |           |           *
           *|           |*n")
  }
  writef("%10.4d: ", muldiv(sample_count, 10000, rate))
  writef("%10i: ", sample_count)

  FOR n = 21 TO 95 DO
  { LET amp = ampv!n
    IF amp>maxamp DO maxamp := amp
  }

  current_amp := current_amp * 99 / 100
  IF current_amp < maxamp DO current_amp := maxamp

  // Choose a suitable amp threshold
  minamp := 10
  WHILE minamp < maxamp/2 DO
  { LET k = 0
    FOR n = 21 TO 95 DO
    { LET amp = ampv!n
      IF amp>minamp DO
      { k := k+1
        IF k>5 BREAK
      }
    }
    IF k<=5 BREAK
    minamp := minamp + 10
  }

  FOR n = 21 TO 108 DO
  { LET note = n MOD 12
    LET black = FALSE
    LET amp = ampv!n
    LET ch = '#'
    LET a = 36*amp/current_amp
    IF  0 <= a < 10 DO ch := '0'+a
    IF 10 <= a < 36 DO ch := 'A'+a-10

    IF note=1 | note=3 | note=6 | note=8 | note=10 DO black := TRUE
    //IF note=0 | note=5 DO wrch(' ')
    IF n=60 DO wrch(' ')

    TEST a>=14
    THEN wrch(ch)
    ELSE wrch(black -> '=', '.')
  }
  newline()
//  writef("Note %i3 amp=%i5 minamp=%i5 maxamp=%i5 current_amp=%i5*n",
//          60, ampv!60, minamp, maxamp, current_amp)
//  abort(1001) 
}

AND wrnotename(n) BE
{ LET oct = n/12 - 1
  LET note = n MOD 12
  UNLESS 0<=n<=127 DO
  {  writef("Bad")
     RETURN
  }

  SWITCHON note INTO
  { DEFAULT: writef("Bad");           RETURN
    CASE  0:  writef( "c%n",  oct);   RETURN
    CASE  1:  writef( "c#%n", oct);   RETURN
    CASE  2:  writef( "d%n",  oct);   RETURN
    CASE  3:  writef( "d#%n", oct);   RETURN
    CASE  4:  writef( "e%n",  oct);   RETURN
    CASE  5:  writef( "f%n",  oct);   RETURN
    CASE  6:  writef( "f#%n", oct);   RETURN
    CASE  7:  writef( "g%n",  oct);   RETURN
    CASE  8:  writef( "g#%n", oct);   RETURN
    CASE  9:  writef( "a%n",  oct);   RETURN
    CASE 10:  writef( "a#%n", oct);   RETURN
    CASE 11:  writef( "b%n",  oct);   RETURN
  }
}           

AND die() BE resumeco(killerco, currco)

AND prbuf(buf, upb, phase, rate) BE
{ LET max, min = buf!1, buf!1
  LET range = ?

  FOR i = 2 TO upb DO
  { LET val = buf!i
    IF max<val DO max := val
    IF min>val DO min := val
  }
  min, max := -100000, +100000
  range := max-min+1

  //FOR i = 1 TO upb BY 4 DO
  FOR i = 1 TO upb BY 1 DO
  { //LET val = (buf!i + buf!(i+1) + buf!(i+2) + buf!(i+3))/4
    LET val = buf!i
    LET pos = (val-min)*80/range
    writef("%10.4d: %i3 %i6 ", muldiv(sample_count-upb+i, 10000, rate), i, val)
    FOR i = 1 TO pos-1 DO wrch(' ')
    TEST i=phase
    THEN writes("**---------------")
    ELSE wrch(val>=0 -> '+', '-')
    newline()
  }
}

// The signal generator

AND wr2(a) BE
{ binwrch(a&255)
  binwrch(a>>8)
//sawritef("wr2: %x4*n", a)
//abort(7777)
}

AND wr4(a) BE
{ wr2(a)
  wr2(a>>16)
}

/*
LET start() = VALOF
{
  writef("Generating %8.3d msecs of square waves to file %s*n",
          secs, toname)

  IF towavstream DO
  { selectoutput(towavstream)
    wrhdr()
  }

  //FOR n = 24 TO 84  DO
//  FOR n = 69 TO 69  DO
//  { LET f = freqtab!n
  //FOR f = 250_000 TO 520_000 BY 1_000 DO
  //FOR f = freqtab!21 TO freqtab!95 BY 1_000 DO
  FOR n = 21 TO 95-1 FOR p = 0 TO 9 DO
  { LET f1, f2 = freqtab!n, freqtab!(n+1)
    LET f = (f1*(10-p) + f2*p)/10

    selectoutput(stdout)
    writef("Time %8.3d: f=%8.3d  f1=%8.3d f2=%8.3d*n",
            sample_count*10/441, f, f1, f2)
    writef("Time %8.3d: f=%8.3d*n", sample_count*10/441, f)
    //writef("Starting note: %i3  freq %8.3d  name ", n, f)
    //wrnotename(n)
    //newline()
    selectoutput(tostream)
    vafa!0 := f
    //vbfa!0 := 440_000
    //abort(1000)
    //FOR i = 1 TO 44100/4 DO // Generate a wave for 1/4 secs
    FOR i = 1 TO 44100/10 DO // Generate a wave for 1/10 secs
    //FOR i = 1 TO 44100/20 DO // Generate a wave for 1/20 secs
    { LET a = callco(vaco)
      //LET b = callco(vbco)
      //LET c = callco(vcco)
      //LET d = callco(vnco)
      LET val = a//+b/2+c/2+d/2
      sample_count := sample_count+1
      wr2(val)
//sawritef("sample %i6: f=%8.3d val=%i5*n", sample_count, f, val)
//abort(1000)
    }
  }
*/

AND squarefn(args) = VALOF
{ // args!0 = ddd.ddd  scaled frequency
  // args!1 = the amplitude, typically 0..1000
  // The first call return a two word vector holding
  // the frequency and amplitude. These values can be changed
  // anytime by the caller.
  // Subsequent calls (assumed to be at 44100 per second) yield
  // samples representing the square wave at the current frequency
  // and amplitude.

  // ************                   ---- +amplitude
  // *           *
  // *           *
  // *           *
  // *           *
  // *           *
  // *-----------*-----------*-     ---- 0
  //             *           *
  //             *           *
  //             *           *
  //             *           *
  //             *           *
  //             ************       ---- -amplitude
  // ^                       ^
  // |                       |
  // sample 0                sample 44100/freq
  LET freq, amplitude = args!0, args!1
  LET lastsamp = 0
  LET x = 0
  //sawritef("squarefn: @freq=%n*n", @freq)
  cowait(@freq)  // Return a pointer to [freq, amplitude]
//abort(3000)

  { // Generate square wave samples
    LET q4 = 44100*1000
    LET q2 = q4/2
    UNTIL x > q2 DO
    { cowait(amplitude)
//sawritef("squarefn: freq=%8.3d x=%8.3d amp=%n*n", freq, x, amplitude)
//abort(2000)
      x := x + freq
    }
    UNTIL x > q4 DO
    { cowait(-amplitude)
//sawritef("squarefn: freq=%8.3d x=%8.3d amp=%n*n", freq, x, -amplitude)
//abort(2000)
      x := x + freq
    }
    x := x - q4
  } REPEAT
}

AND trianglefn(args) = VALOF
//AND squarefn1(args) = VALOF
{ // args!0 = ddd.ddd  scaled frequency
  // args!1 = the amplitude, typically 0..1000
  // The first call returns a two word vector holding
  // the frequency and amplitude. These values can be changed
  // anytime by the caller.
  // Subsequent calls (assumed to be at 44100 per second) yield
  // samples representing the square wave at the current frequency
  // and amplitude.

  //       *---------------------- +amplitude
  //      / \
  //     /   \
  //    /     \
  //   /       \
  //  /         \
  // /-----------\-----------/-
  // |            \         /|
  // |             \       / |
  // |              \     /  |
  // |               \   /   |
  // |                \ /    |
  // |                 *-----|---- -amplitude
  // * 0                     *---- sample 44100 / freq

  LET freq, amplitude = args!0, args!1
  LET lastsamp = 0
  LET x = 0
  //sawritef("trianglefn: @freq=%n*n", @freq)
  cowait(@freq)  // Return a pointer -> [freq, amplitude]
//abort(3000)

  { // Generate triangle wave samples
    LET q4 = 44_100*1000
    LET q2 = q4/2
    LET q1 = q2/2
    LET q3 = q1+q2
//sawritef("trianglefn: freq=%8.3d q1=%n q2=%n q3=%n q4=%n*n", freq, q1, q2, q3, q4)
//abort(2001)
    UNTIL x > q1 DO
    { LET val = muldiv(amplitude, x, q1)
      cowait(val)
//sawritef("trianglefn: freq=%8.3d x=%8.3d amp=%n*n", freq, x, val)
//abort(2000)
      x := x + freq
    }
    UNTIL x > q3 DO
    { LET val = - muldiv(amplitude, x-q2, q1)
      cowait(val)
//sawritef("trianglefn: freq=%8.3d x=%8.3d amp=%n*n", freq, x, val)
//abort(2000)
      x := x + freq
    }
    UNTIL x > q4 DO
    { LET val = muldiv(amplitude, x-q4, q1)
      cowait(val)
//sawritef("trianglefn: freq=%8.3d x=%8.3d amp=%n*n", freq, x, val)
//abort(2000)
      x := x + freq
    }
    x := x - q4
  } REPEAT
}

AND sinefn(args) = VALOF
{ // args!0 = ddd.ddd  scaled frequency
  // args!1 = the amplitude, typically 0..1000
  // The first call return a two word vector holding
  // the frequency and amplitude. These values can be changed
  // anytime by the caller.
  // Subsequent calls (assumed to be at 44100 per second) yield
  // samples representing the sine wave at the current frequency
  // and amplitude.
  LET freq, amplitude = args!0, args!1
  LET lastsamp = 0
  LET x = 0
  //sawritef("sinefn: @freq=%n*n", @freq)
  cowait(@freq)  // Return a pointer to [freq, amplitude]
//abort(3000)
  { // Generate sine wave samples -- one day!!
    UNTIL x > 44_100*1000/2 DO
    { cowait(amplitude)
      x := x + freq
//sawritef("sinefn: freq=%8.3d x=%8.3d*n", freq, x)
//abort(2000)
    }
    UNTIL x > 44_100*1000 DO
    { cowait(-amplitude)
      x := x + freq
//sawritef("sinefn: freq=%8.3d x=%8.3d*n", freq, x)
//abort(2000)
    }
    x := x - 44_100*1000
  } REPEAT
}
