/*
This is a program to analyse wave input from either a microphone
or a .wav file. It is derived from anawav.b but uses a coroutine
for each frequency it recognises. Each is woken up once per cycle
of the frequency it is listening to, and calculates the approximate
amplitude of the most recent 20 cycles of that frequency.

Implemented by Martin Richards (c) September 2010

Usage:

anawav "SOURCE/N,FROM/K,TO/K,
        A/K/N,T1/K/N,T2/K/N,SCALE/N,BMP/K,GRID/N"

SOURCE  =1  samples from a .wav file (the default).
        =2  samples from the microphone.
        =3  samples from a square wave generator.
        =4  samples from another generator.
FROM    name of the .wav file or microphone.
TO      is result of the analysis (default: to stdout).
A       First parameter for the test.
T1      Start time in msecs.
T2      End time is msecs.
SCALE   percentage amplification of the input samples -- default 100
BMP     is the filename for the .bmp picture -- default pic.bmp
GRID    is the number of msecs per grid line in the .bmp picture
        the default value gives about 10 grid lines per picture.

The sample rate must be 44100 and the samples bit length must be 16.
Stereo .wav input is converted to mono samples by averaging the two
inputs.

Change history

13/10/2010
Started on a version using coroutines to calculate the amplitudes of
each frequency scheduled using a priority queue.
*/

GET "libhdr"
GET "sound.h"

MANIFEST {
  cumsampvupb=#xFFFF  // UPB (=65535) of cumsampv
  priqupb=1000        // Priority queue upper bound
  root48of2 = 1_014_545_335  // Actually   1.01454533494
}

GLOBAL {
  priq:ug      // The vector holding the priority queue
               // heap structure for the priority queue
               // priq!1 is zero or points to [sampleno, co] 
               // representing the earliest event item in the
               // priority queue. Control is given to coroutine co
               // as soon as the specified sample has been reached.
  priqn        // Number of items in the priority queue
  tracing

  prq          // Print the priority queue
  insertevent  // Insert [sampleno, co] into the priority queue
  upheap       // heap operation
  downheap     // heap operation
  getevent     // Extract the earliest priority queue item from the heap
  waituntil    // Wait until a specified sample number has been reached

  cumsampv     // Circular buffer of cumulative sample data as
               // 32-bit signed values. Upb 65535 values ie sufficient
               // for more than 1 second of data.
  tostream
  stdin
  stdout
  debug        // Controls debugging trace
  bmpname      // the .bmp file name -- default pic.bmp
  bmpstream

  source       // Number giving the source of the samples
  fromname     // .wav file name
  fromstream
  toname
  waveInCB     // Microphone in stream
  parm_a
  time1        // Start time in msecs
  time2        // End time in msecs
  sampleno     // Latest sample number
  stopsampleno // Final sample number
  freqtab      // freqtab!(4*n) = frequency of midi note n from 0 to 127
  cotab        // Table of tuner coroutines
  bmpco        // Coroutine generating the .bmp image
  sampleco     // Coroutine giving the samples
  ampfactor    // A percentage
  gridmsecs    // msecs per grid line -- default gives 10 lines
  pos          // Sample number of latest sample
  ampv         // Current amplitudes for each frequency
}

/*
There will up to be 512 tuners set to the frequencies of midi notes
from 0 to 127 at intervals of 1/4 of a semitone.
*/


LET init_freqtab() = VALOF
{ LET afrq = 32*440_000 // Frequency of A9 (midi note 129!)
  LET frq  = afrq

  // Fill in the frequency of A for each octave
  FOR i = 4*129 TO 0 BY -4*12 DO
  { IF i<=511 DO freqtab!i := afrq
    afrq := afrq/2
  }
  // Fill in all other entries from freqtab!(4*0+0) to freqtab!(4*127+3)
  FOR i = 4*129 TO 0 BY -1 DO
  { IF i<=511 TEST freqtab!i
              THEN frq := freqtab!i
              ELSE freqtab!i := frq
    frq := (muldiv(10*frq, 2_000_000_000, 2*root48of2-10)+5)/10 
  }
  //newline()
  //writef("freqtab!0      = %5.3d*n", freqtab!0)
  //writef("freqtab!(4x69) = %5.3d*n", freqtab!(4*69))

  RESULTIS TRUE
}

LET start() = VALOF
{ LET devname = 0
  LET format = 16  // S16_LE
  LET channels = 1 // Mono
  LET rate = 44100 // Samples per second
  LET argv = VEC 50

  stdout := output()
  tostream := 0
  debug := 0
  bmpname := "pic.bmp"
  bmpstream := 0

  cumsampv := 0
  ampv := 0
  cotab := 0
  bmpco := 0
  freqtab := 0
  priq := 0
  tostream := 0
  source := 1  // Default -- samples from a .wav file
  fromname := "../23-1start.wav"
  fromstream := 0
  waveInCB := -1  // No microphone input
  toname := 0
  parm_a, time1, time2 := -1, 2_300, 3_300
  parm_a, time1, time2 := -1, 6_500, 9_000  
  parm_a, time1, time2 := -1, 0_000, 9_000  
  ampfactor := 20   // default is 20%
  gridmsecs := 0


  writef("coanawav entered*n")

  UNLESS rdargs("SOURCE/N,FROM/K,TO/K,*
                *A/N,T1/N,T2/N,SCALE/N,BMP/K,GRID/N",
                argv, 50) DO
  { writef("Bad arguments for anawav*n")
    RESULTIS 0
  }

  IF argv!0 DO source    := !(argv!0)  // SOURCE/N
  IF argv!1 DO fromname  := argv!1     // FROM/K     .wav input
  IF argv!2 DO toname    := argv!2     // TO/K       result of analysis
  IF argv!3 DO parm_a    := !(argv!3)  // A/K/N
  IF argv!4 DO time1     := !(argv!4)  // T1/K/N
  IF argv!5 DO time2     := !(argv!5)  // T2/K/N
  IF argv!6 DO ampfactor := !(argv!6)  // SCALE/N    amplitude scaling
  IF argv!7 DO bmpname   := argv!7     // BMP/K      BMP file name
  IF argv!8 DO gridmsecs := !(argv!8)  // GRID/N     msecs per grid line

  UNLESS gridmsecs DO
    gridmsecs := (time2-time1)/10 // default 10 grid lines per picture

  TEST sys(Sys_sound, snd_test)=-1
  THEN writef("Sound is available*n")
  ELSE writef("Sound is not available*n")

  cumsampv := getvec(cumsampvupb)
  UNLESS cumsampv DO
  { writef("Unable to allocate a vector with UPB=%n*n", cumsampvupb)
    RESULTIS 0
  }

  // Clear the circular buffer of cumulative sample data
  FOR i = 0 TO cumsampvupb DO cumsampv!i := 0
  pos := 0

  priq := getvec(priqupb)
  UNLESS priq DO
  { writef("Unable to allocate a vector with UPB=%n*n", priqupb)
    RESULTIS 0
  }

  // Clear the circular buffer of cumulative sample data
  FOR i = 0 TO priqupb DO priq!i := 0
  // Currently there are no event items in the heap.
  priqn := 0

  freqtab := getvec(511)
  cotab   := getvec(511)
  ampv    := getvec(511)

  UNLESS freqtab & cotab & ampv DO
  { writef("More space needed*n")
    RESULTIS FALSE
  }

  FOR i = 0 TO 511 DO freqtab!i, cotab!i, ampv!i := 0, 0, 0

  UNLESS init_freqtab() GOTO fin

  SWITCHON source INTO
  { DEFAULT:  writef("*nBad source number %n*n", source)
              GOTO fin

    CASE 1:  sampleco := initco(wavfilecofn, 500, fromname)
             ENDCASE

    CASE 2:  sampleco := initco(microphonecofn, 500)
             ENDCASE

    CASE 3:  sampleco := initco(squarecofn, 500,
                                4*69,   // A above middle C
                                20000   // Amplitude
                               )
             ENDCASE

    CASE 4:  sampleco := initco(samplecofn, 500) // Other source
             ENDCASE

  }

  // Create the tuner coroutines
  //FOR i = 0 TO 511 DO cotab!i := initco(tunerfn, 500, i)
  //FOR i = 270 TO 276 DO cotab!i := initco(tunerfn, 500, i)
  FOR i = 0 TO 511 BY 1 DO cotab!i := initco(tunerfn, 500, i)

  // Create the bmp coroutine
  bmpco := initco(bmpcofn, 500, 0)

  runtuners(parm_a, time1, time2)

fin:
  IF cumsampv DO freevec(cumsampv)
  IF ampv     DO freevec(ampv)
  IF freqtab  DO freevec(freqtab)
  IF cotab DO
  { FOR i = 0 TO 511 IF cotab!i DO deleteco(cotab!i)
    freevec(cotab)
  }
  IF bmpco    DO deleteco(bmpco)
  IF priq     DO freevec(priq)
  IF tostream DO endstream(tostream)
  IF fromstream DO endstream(fromstream)
  UNLESS waveInCB=-1 DO sys(Sys_sound, snd_waveInClose, waveInCB)


  selectoutput(stdout)

  RESULTIS 0
}

AND microphonecofn(args) BE
{ LET buf = VEC 2048

  // Open the microphone source
  waveInCB := sys(Sys_sound, snd_waveInOpen, 0, 16, 1, 44100)

  //writef("waveInCB = %n*n", waveInCB)

  IF waveInCB=-1 DO
  { writef("Cannot open wave input device*n")
    GOTO ret
  }

  writef("*nReading microphone samples*n")

  { LET len = sys(Sys_sound,
                  snd_waveInRead, waveInCB, buf, 2048, 0)
    FOR i = 0 TO 2047 DO cowait(buf!i)
  } REPEAT

ret:
  cowait(0) REPEAT // Pad with zeros
}

AND runtuners(a) BE
{ // Run the event loop
  LET val = 0
  sampleno := 0
  stopsampleno := muldiv(44100, time2, 1000)
sawritef("runtuners: stopping at %6.3d sample number %n*n",
          time2, stopsampleno)

  { LET event = getevent()      // Get the earliest event
    LET eventsampleno = ?
    UNLESS event BREAK

    eventsampleno := event!0
    WHILE sampleno<=eventsampleno DO
    { val := val + callco(sampleco)
      sampleno := sampleno+1
      cumsampv!(sampleno & #xFFFF) := val
//sawritef("runtuners: %i4 cumulative val = %i6*n", sampleno, val)
    }
    //IF tracing DO writef("%i8: calling co=%n*n", eventsampleno, event!1)
    IF eventsampleno > stopsampleno BREAK
    callco(event!1)
  } REPEAT
}

AND tunerfn(args) BE
{ LET note = args!0
  LET freq = freqtab!note
  LET inc1000 = muldiv(44100_000, 1000, freq)
  LET incn = inc1000/1000
  LET incf = inc1000 MOD 1000
  AND ph0v = VEC 19
  AND ph1v = VEC 19
  AND ph2v = VEC 19
  AND ph3v = VEC 19

  FOR i = 0 TO 19 DO
    ph0v!i, ph1v!i, ph2v!i, ph3v!i := 0, 0, 0, 0

  //writef("tunerfn: co=%i6 note %i3 freq %9.3d incn=%i4 incf=%i3  ",
  //        currco, note, freq, incn, incf)
  //wrnotename(note)
  //newline()

  calcloop(incn, incf, note, ph0v, ph1v, ph2v, ph3v)
}

AND calcloop(incn, incf, note, ph0v, ph1v, ph2v, ph3v) BE
{ LET prevn, prevf = 0, 0
  LET ph0, ph1, ph2, ph3 = 0, 0, 0, 0
  LET p = 0

  { // Calculate the sample number just after the end
    // of the next cycle of this note.
    LET nextn = prevn + incn
    LET nextf = prevf + incf
    LET n2, tot, t, val, amp = ?, ?, ?, ?, ?

    IF nextf>=1000 DO nextn, nextf := nextn+1, nextf-1000
    // Ensure nextn is even
    UNLESS (nextn&1)=0 TEST nextf>0
                       THEN nextn, nextf := nextn+1, nextf-1000
                       ELSE nextn, nextf := nextn-1, nextf+1000

    n2 := (nextn-prevn)/2

//sawritef("calcloop: note %i3 waiting until %n*n", note, nextn)
    waituntil(nextn)
//sawritef("calcloop: note %i3 woken up at   %n*n", note, nextn)

    tot := (cumsampv!(nextn&#xFFFF)-cumsampv!(prevn&#xFFFF))/2
//writef("calcloop: note %i3 tot=%i5 nextn=%i4 nextf=%i3 incn=%i4 incf=%i4*n",
//        note, tot, nextn, nextf, incn, incf)

    t := nextn
    val := (cumsampv!(t & #xFFFF)-cumsampv!(t-n2 & #xFFFF) - tot)/n2
//writef("calcloop: note %i3 t=%i4 val0=%i6 = (%n - %n - %n)/%n*n",
//        note, t, val, cumsampv!(t & #xFFFF), cumsampv!(t-n2 & #xFFFF), tot, n2)
    ph0 := ph0 - ph0v!p + val; ph0v!p := val

    //t := nextn - n2/4
    //val := (cumsampv!(t & #xFFFF)-cumsampv!(t-n2 & #xFFFF) - tot)/n2
//writef("calcloop: note %i3 t=%i4 val1=%i6 = (%n - %n - %n)/%n*n",
//        note, t, val, cumsampv!(t & #xFFFF), cumsampv!(t-n2 & #xFFFF), tot, n2)
    //ph1 := ph1 - ph1v!p + val; ph1v!p := val

    t := nextn - n2/2
    val := (cumsampv!(t & #xFFFF)-cumsampv!(t-n2 & #xFFFF) - tot)/n2
//writef("calcloop: note %i3 t=%i4 val2=%i6 = (%n - %n - %n)/%n*n",
//        note, t, val, cumsampv!(t & #xFFFF), cumsampv!(t-n2 & #xFFFF), tot, n2)
    ph2 := ph2 - ph2v!p + val; ph2v!p := val

    //t := nextn - (3*n2)/4
    //val := (cumsampv!(t & #xFFFF)-cumsampv!(t-n2 & #xFFFF) - tot)/n2
//writef("calcloop: note %i3 t=%i4 val3=%i6 = (%n - %n - %n)/%n*n",
//        note, t, val, cumsampv!(t & #xFFFF), cumsampv!(t-n2 & #xFFFF), tot, n2)
    //ph3 := ph3 - ph3v!p + val; ph3v!p := val

//FOR i = p+1 TO p+20 DO
//{ LET q = i MOD 20
//  writef("%i4: %i6 %i6 %i6 %i6*n", i, ph0v!q, ph1v!q, ph2v!q, ph3v!q)
//}
//writef("      %i6 %i6 %i6 %i6*n", ph0/20, ph1/20, ph2/20, ph3/20)

    // Increment circular buffer position
    p := (p+1) MOD 20
    prevn, prevf := nextn, nextf

  // Estimate the amplitude
//writef("calcloop: %i5 %i5 %i5 %i5 %i5 %i5 %i5 %i5",
//          ph0/20, ph1/20, ph2/20, ph3/20, -ph0/20, -ph1/20,-ph2/20,-ph3/20)
    //amp := ABS ph0
    //IF ABS ph1 > amp DO amp := ABS ph1
    //IF ABS ph2 > amp DO amp := ABS ph2
    //IF ABS ph3 > amp DO amp := ABS ph3

//    ampv!note := amp/20  // 20 cycles
    ampv!note := (ABS ph0 + ABS ph2)/20  // 20 cycles
//writef(" => %n", amp/20)
//writef("   %i6 %i6  note=%i3  ",
//       (ABS ph0 + ABS ph2)/20, (ABS ph1 + ABS ph3)/20, note)
//wrnotename(note)
//newline()
//abort(1000)
  } REPEAT
}

AND bmpcofn() BE
{ // This is the body of bmpco that generates a .bmp format file
  // a colour table and one byte per pixel.
  // The size of the image is 15x25cms
  // The amplitude data is in ampv being set by the tuner coroutines.

  LET n1 = 0 //12 //+24   // Lowest Midi note processed
  LET n2 = 127 //108//-12   // Highest Midi note processed

  LET xsize = 4*(n2-n1+1) // Allow for four frequencies per semitone
  LET ysize = 600   // Raster lines of frequency amplitudes 
  //LET ysize = 1200   // Raster lines of frequency amplitudes 
  LET xres  = muldiv(xsize,   100, 15)  // 15 cms horizontal 
  LET yres  = muldiv(ysize+4, 100, 24)  // 25 cms vertical 4 pixels for col map
  LET hdrsize     = 14
  LET infohdrsize = 40
  LET paletsize   = 4*256
  LET rowlen = (xsize+3) & -4 // Round up to a multiple of 4 bytes
  LET dataoffset = hdrsize + infohdrsize + paletsize
  LET pixeldatasize = ysize * rowlen
  LET colourtab = VEC 256
  LET workv = 0

  bmpstream := 0

  initcolourtab(colourtab)

  bmpstream := findoutput(bmpname)

  UNLESS bmpstream DO
  { writef("Trouble with file: %s*n", bmpname)
    RETURN
  }

  selectoutput(bmpstream)

  // Write the header
  wr1('B'); wr1('M') // "BM"
  //wr4(hdrsize + infohdrsize + pixeldatasize) // File size in bytes
  wr4(dataoffset + pixeldatasize) // File size in bytes
  wr4(0)             // Unused
  wr4(dataoffset)    // File offset of pixel data

  // Write the Info header
  wr4(40)             // Size of info header = 40
  wr4(xsize)          // Bitmap width
  wr4(ysize+4)          // Bitmap height
  wr2(1)              // Number of planes = 1
  wr2(8)              // 8 bits per pixel
  wr4(0)              // No compression
  //wr4(pixeldatasize)  // Size of image
  wr4(0)              // Size of image =0 valid if no compression
  wr4(xres)           // Horizontal resolution in pixels per meter
  wr4(yres)           // Vertical   resolution in pixels per meter
  wr4(256)            // Number of colours actually used
  wr4(0)              // All colours are important

  // Write the colour table
  FOR i = 0 TO 255 DO wr4(colourtab!i)

  { // Print the palette colours
    FOR y = 1 TO 4 DO // 4 raster lines for the palette
    { FOR  x = xsize-1 TO 0 BY -1 DO
      { LET col = 255 * x / (xsize-1)
        //IF y<=2 & x < xsize/2 DO col := 0
        wr1(x=0!y=4->255, col)
      }
      FOR x=xsize+1 TO rowlen DO wr1(0) // Pad to a multiple of 4
    }
  }
 
  FOR y = 0 TO ysize-1 DO
  { LET t = time1 + muldiv(time2-time1, y, ysize-1)
    LET s = muldiv(44100, t, 1_000) // Number of first sample
    LET gridline = FALSE
    LET len = 0
    LET layout = 0
    LET max = 0

    STATIC { prevt=0 }

    //sawritef("bmpcofn: waiting until %i5 time %6.3d*n", s, t)

    waituntil(s)

    sawritef("time %6.3d raster line %i4 of %i4*n", t, y+1, ysize)

    UNLESS t/gridmsecs = prevt/gridmsecs DO
    { // Write the time every 100 msecs
      prevt := t
      sawritef("Grid line at time: %8.3d sample pos=%n*n", t, s)
      gridline := TRUE
    }

    // Find maximum amplitude
    max := 0
    FOR i = 0 TO xsize-1 IF ampv!i>max DO max := ampv!i

    // Write a raster line
    FOR x = xsize-1 TO 0 BY -1 DO
    { LET a = ampv!x
      LET n = n1 + (x+1)/4

      // Rescale amplitudes if necessary
      IF max>255 DO a := (a * 255) / max

//sawritef("test4: y=%i4 n=%i3 x=%i4 a=%i5*n", y, n, x, a)
//abort(1002)
      //a := a/4
      IF a<  0 DO a := 0
      IF a>255 DO a := 255

      //IF layout=0 DO sawrch('*n')
      //layout:=(layout + 1) & 31

      //sawritef(" %x2", a)
//LOOP
      TEST gridline
      THEN { LET col = 0
             LET note = n MOD 12
             SWITCHON note INTO
             { DEFAULT:  col := 0             // White notes
                         ENDCASE
               CASE  0:  col := 40            // C        Light Gray
                         IF n=60 DO col :=200 // Middle C Green
                         ENDCASE
               CASE  1:                       // C#  Black
               CASE  3:                       // D#  Black
               CASE  6:                       // F#  Black
               CASE  8:                       // G#  Black
               CASE 10:  col := 255           // A#  Black
                         ENDCASE
             }
             wr1(col)
           }
      ELSE wr1(a)
      len := len + 1
    }
//sawrch('*n')
//abort(1002)
    FOR x = xsize+1 TO rowlen DO wr1(0) // Pad up to next 32-bit boundary
  }

fin:
  IF bmpstream DO endstream(bmpstream)
  cowait(0)
}

AND initcolourtab(ctab) BE
{ LET colours = TABLE       10,
      //  red     green    blue
        255<<16 | 255<<8 | 255,  // White
        200<<16 | 200<<8 | 200,  // Light grey
        100<<16 | 100<<8 | 150,  // 
          0<<16 |   0<<8 | 255,  // Blue
        160<<16 | 160<<8 | 200,  //
          0<<16 | 255<<8 |   0,  // Green
        180<<16 | 180<<8 |   0,  // 
        255<<16 |   0<<8 |   0,  // Red
        180<<16 |  60<<8 |  60,  // Dark grey
          0<<16 |   0<<8 |   0   // Black

  FOR i = 1 TO colours!0-1 DO
  { LET n = colours!0
    LET p = (256*(i-1))/(n-1)
    LET q = (256*i)/(n-1)
    LET c1, c2 = colours!i, colours!(i+1)
    LET r1, g1, b1 = c1>>16&255, c1>>8&255, c1&255
    LET r2, g2, b2 = c2>>16&255, c2>>8&255, c2&255
//sawritef("i=%i2 p=%i3  q=%i3  c1=%x6 c2=%x6*b*n", i, p, q, c1, c2)
    FOR t = p TO q DO
    { LET r = (r1*(q-t)+r2*(t-p))/(q-p)
      LET g = (g1*(q-t)+g2*(t-p))/(q-p)
      LET b = (b1*(q-t)+b2*(t-p))/(q-p)
      ctab!t := r<<16 | g<<8 | b
      //sawritef("%i3: %x6*n", t, ctab!t)
    }
    //sawritef("*n")
    //abort(1000)
  }
//sawritef("*nColour table*n")
  //FOR i = 0 TO 256 DO
  //{ IF i MOD 8 = 0 DO sawrch('*n')
  //  sawritef(" %x6", ctab!i)
  //}
  //sawrch('*n')    
}

AND squarecofn(args) BE
{ LET x         = 0
  LET rate      = 44100
  LET q4        = rate*1000
  LET q2        = q4/2
  LET note      = args!0
  LET amplitude = args!1
  LET freq      = freqtab!note

  { UNTIL x > q2 DO { cowait(+amplitude) // First half cycle
                      x := x + freq
                    }
    UNTIL x > q4 DO { cowait(-amplitude) // Second half cycle
                      x := x + freq
                    }
    x := x - q4
  } REPEAT
}

AND samplecofn(args) BE
{ // Return samples of a 440Hz sine wave with varying amplitude

  LET k      = 3925     // 440Hz
  LET amp    = 20_000
  LET ampdiv = 1_000_000_000/amp

  cowait(0) // Return to initco

  { // Start of main loop
    LET x    = 1_000_000_000
    LET xdot = 0

    FOR i = 0 TO 44100-1 DO // Increase volume for 1 sec
    { LET xdotdot = -muldiv(x, k, 1_000_000)
      xdot := xdot + xdotdot
      x := x + xdot
      cowait(muldiv(x/ampdiv, i, 44100))
    }
    FOR i = 0 TO 44100-1 DO // Steady volume for 1 sec
    { LET xdotdot = -muldiv(x, k, 1_000_000)
      xdot := xdot + xdotdot
      x := x + xdot
      cowait(x/ampdiv)
    }
    FOR i = 44100 TO 1 BY -1 DO // Decrease volume to the end
    { LET xdotdot = -muldiv(x, k, 1_000_000)
      xdot := xdot + xdotdot
      x := x + xdot
      cowait(muldiv(x/ampdiv, i, 44100))
    }
  } REPEAT
}

AND wavfilecofn(args) BE
{ LET name = args!0

  fromstream := findinput(name)

  UNLESS fromstream DO
  { sawritef("Trouble with .wav file %s*n", name)
    GOTO ret
  }
  sawritef("Getting wav data from %s*n", name)

  cowait(0)  // Return to initco

  selectinput(fromstream)
  getwavsamples()
  endstream(fromstream)
  fromstream := 0
ret:
  cowait(0) REPEAT // Pad with zeros
}

AND getwavsamples() BE
{ // This is called from wavcofn.
  // It reads samples from the .wav file which is the currently
  // selected input and return them as 32-bit signed integers.
  // If the .wav file is in stereo the left and right samples
  // are averaged.
  LET riff  = rd4()       //  0 RIFF
  LET size  = rd4()       //  4 filesize - 8
  LET wave  = rd4()       //  8 WAVE
  LET fmt   = rd4()       // 12 fmt
  LET chksz = rd4()       // 16 16
  LET quant = rd2()       // 20 1 = linear
  LET mode  = rd2()       // 22 1=mono 2=stereo
  LET rate  = rd4()       // 24 Typically 44100
  LET brate = rd4()       // 28 byte rate
  LET bytePerSample=rd2() // 32 1, 2 or 4  = bits/8 * mode
  LET bits  = rd2()       // 34 bits per sample = 8 or 16
  LET filler= chksz=18->rd2(), 0       // 36 filler

  LET data  = rd4()       // 36 data
  LET len   = rd4()       // 40 bytes of data or -1

  UNLESS riff=#x46464952 DO sawritef("Bad RIFF word %x8*n", riff)
  UNLESS wave=#x45564157 DO writef("Bad WAVE word %x8*n", wave)
  UNLESS fmt =#x20746D66 DO writef("Bad fmt  word %x8*n", fmt)
  //UNLESS chksz=16        DO writef("Bad subchunk size %n*n", chksz)
  UNLESS mode=1 | mode=2 DO writef("Bad mode %n*n", mode)
  UNLESS rate=44100      DO writef("Bad rate %n*n", rate)
  UNLESS bits=16         DO writef("Bad bits per sample %n*n", bits)
  UNLESS data=#x61746164 DO writef("Bad data word %x8*n", data)

//sawritef("rate=%n %s*n", rate, mode=2->"stereo", "mono")

  { LET w = rd2()
    IF w<0 BREAK      // End of file reached
    IF mode=2 DO w := (w+rd2())/2
    IF (w & #x8000)~=0 DO w := w | #xFFFF0000 // Sign extend
    cowait(w)
  } REPEAT
}

AND wr1(b) BE
{ binwrch(b)
}

AND wr2(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
}

AND rd2() = VALOF
{ LET w = 0
  LET s = @w
  s%0 := binrdch()
  s%1 := binrdch()
  RESULTIS w
}

AND wr4(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
  binwrch(s%2)
  binwrch(s%3)
}

AND rd4() = VALOF
{ LET w = 0
  LET s = @w
  s%0 := binrdch()
  s%1 := binrdch()
  s%2 := binrdch()
  s%3 := binrdch()
  RESULTIS w
}


AND wrnotename(n) BE
{ LET adj = n & 3
  LET midinote = n/4
  LET oct = ?
  LET note = ?

  IF adj=3 DO midinote := midinote+1
  note := midinote MOD 12
  oct := midinote/12 - 1

  UNLESS 0<=midinote<=128 DO
  { writef("Bad note")
    RETURN
  }

  SWITCHON note INTO
  { DEFAULT:  writef("Bad note");     RETURN
    CASE  0:  writef( "%nc",  oct);   ENDCASE
    CASE  1:  writef( "%nc#", oct); ENDCASE
    CASE  2:  writef( "%nd",  oct); ENDCASE
    CASE  3:  writef( "%nd#", oct); ENDCASE
    CASE  4:  writef( "%ne",  oct); ENDCASE
    CASE  5:  writef( "%nf",  oct); ENDCASE
    CASE  6:  writef( "%nf#", oct); ENDCASE
    CASE  7:  writef( "%ng",  oct); ENDCASE
    CASE  8:  writef( "%ng#", oct); ENDCASE
    CASE  9:  writef( "%na",  oct); ENDCASE
    CASE 10:  writef( "%na#", oct); ENDCASE
    CASE 11:  writef( "%nb",  oct); ENDCASE
  }
  IF adj=2  DO wrch('+')
  IF 0<adj<=2 DO wrch('+')
  IF adj=3  DO wrch('-')
}           

// ################### Priority Queue functions ######################

LET prq() BE
{ FOR i = 1 TO priqn DO writef(" %i4", priq!i!0)
  newline()
}

AND insertevent(event) BE
{ IF priqn>=priqupb DO
  { sawritef("*nPriority queue is full*n")
    abort(999)
  }
  priqn := priqn+1        // Increment number of events
  //writef("insertevent: at time: %n  co=%n*n", event!0, event!1)
  upheap(event, priqn)
}

AND upheap(event, i) BE
{ LET eventtime = event!0
//writef("upheap: eventtime=%n i=%n*n", eventtime, i)

  { LET p = i/2          // Parent of i
    UNLESS p & eventtime < priq!p!0 DO
    { priq!i := event
//prq()
      RETURN
    }
    priq!i := priq!p     // Demote the parent
//prq()
    i := p
  } REPEAT
}

AND downheap(event, i) BE
{ LET j, min = 2*i, ? // j is left child, if present
//writef("downheap: eventtime=%n i=%n*n", event!0, i)
//prq()
  IF j > priqn DO
  { upheap(event, i)
    RETURN
  }
  min := priq!j!0
  // Look at other child, if it exists
  IF j<priqn & min>priq!(j+1)!0 DO j := j+1
  // promote earlier child
  priq!i := priq!j
  i := j
} REPEAT

AND getevent() = VALOF
{ LET event = priq!1      // Get the earliest event
  LET last  = priq!priqn  // Get the event at the end of the heap
//writef("getevent: priq:")
//prq()
  UNLESS priqn>0 RESULTIS 0 // No events in the priority queue
//               ^0
  priqn := priqn-1        // Decrement the heap size
  downheap(last, 1)       // Re-insert last event
  RESULTIS event
}

AND waituntil(sampleno) BE
{ // Make an event item into the priority queue
  LET eventtime, co = sampleno, currco
//writef("waituntil: sampleno=%n*n", sampleno)
  insertevent(@eventtime) // Insert into the priority queue
  cowait()                // Wait for the specified number of ticks
}



