/*
This is a program to analyse wave input from either a microphone
or a .wav file.

Implemented by Martin Richards (c) September 2008

Usage:

anawav "SECS/N,FROM/K,TO/K,TEST/K/N,
        A/K/N,B/K/N,C/K/N,PLAY/S,SCALE/N,BMP/K,GRID/N"

SECS    is the number of second of samples to read (default 10)
FROM    if given is the .wav file containing the samples
        if not given samples are taken from the microphone.
TO      is result of the analysis (default: to stdout).
TEST    selects which test to run on the samples (default 1).
A       First parameter for the test.
B       Second parameter for the test.
C       Third parameter for the test.
PLAY    causes the selected samples to be played.
SCALE   percentage amplification -- default 100
BMP     is the filename for the .bmp picture -- default pic.bmp
GRID    is the number of msecs per grid line in the .bmp picture

The sample rate must be 44100.
Samples bit length must be 16.
Stereo .wav input is converted to mono samples.
The samples are stored in a vector of 32-bit signed integers for
processing.

Change history

28/01/2009
Started to use the MC package to speed up the amplitude
computation.

05/01/2009
Started to implement test3 to generate a coloured picture
of note amplitudes.
*/

GET "libhdr"
GET "sound.h"
GET "mc.h"

GLOBAL {
  sampv:ug  // Vector for 32-bit signed samples
  sampvupb  // UPB of sampv
  tostream
  stdin
  stdout
  debug
  picstream
  bmpname
  bmpstream

  secs      // Number of seconds of recorded data
  fromname  // .wav file name
  toname
  testno
  parm_a
  parm_b
  parm_c
  play
  freqtab
  wrpixel
  ampfactor  // A percentage
  gridmsecs
  currsampv  // to hold samples around the current sample position
  currsampvupb // UPB of currsampv - an even number
  currsampno   // Absolute sample number at the midpoint of currsampv
  ampv         // Amplitudes for each frequency at mid point of currsampv
  mcb          // Control Block for the MC package
}

LET start() = VALOF
{ LET waveInCB = 0
  LET devname = 0
  LET format = 16  // S16_LE
  LET channels = 1 // Mono
  LET rate = 44100 // Samples per second
  LET argv = VEC 50

  stdout := output()
  tostream := 0
  debug := 0
  play := FALSE
  bmpname := "pic.bmp"

  writef("anawav entered*n")

  UNLESS rdargs("SECS/N,FROM/K,TO/K,TEST/K/N,*
                *A/N,B/N,C/N,PLAY/S,SCALE/N,BMP/K,GRID/N",
                argv, 50) DO
  { writef("Bad arguments for anawav*n")
    RESULTIS 0
  }

  secs := 82
  fromname := "../23-1start.wav"  //0
  toname := 0
  testno := 4
  parm_a, parm_b, parm_c := -1, 2_300, 3_300
  parm_a, parm_b, parm_c := -1, 6_500, 9_000
  ampfactor := 20
  gridmsecs := 1000 // default 1000 msecs per grid line

  IF argv!0 DO secs     := !(argv!0)   // SECS/N
  IF argv!1 DO fromname := argv!1      // FROM/K     .wav input
  IF argv!2 DO toname   := argv!2      // TO/K       result of analysis
  IF argv!3 DO testno   := !(argv!3)   // TEST/K/N
  IF argv!4 DO parm_a   := !(argv!4)   // A/K/N
  IF argv!5 DO parm_b   := !(argv!5)   // B/K/N
  IF argv!6 DO parm_c   := !(argv!6)   // C/K/N
  play := argv!7                       // PLAY/S     play the samples
  IF argv!8 DO ampfactor := !(argv!8)  // SCALE/N    amplitude scaling
  IF argv!9 DO bmpname  := argv!9      // BMP/K      BMP file name
  IF argv!10 DO gridmsecs := !(argv!10)// GRID/N     msecs per grid line

  TEST sys(Sys_sound, snd_test)=-1
  THEN writef("Sound is available*n")
  ELSE writef("Sound is not available*n")

  sampvupb := 44100 * secs    // For 32-bit samples

  sampv := getvec(sampvupb)
  UNLESS sampv DO
  { writef("Unable to allocate a vector with UPB=%n*n", sampvupb)
    RESULTIS 0
  }

  //writef("Trying to open wave input device %n*n", devname)

  TEST fromname
  THEN getwav(fromname, sampv, sampvupb)
  //ELSE getmic(sampv, sampvupb)
  ELSE getsamps(sampv, sampvupb)

  IF play DO playsamples(sampv, parm_b, parm_c)

  IF testno DO dotest(testno, sampv, parm_a, parm_b, parm_c)


fin:
  IF sampv DO freevec(sampv)
  IF tostream DO endstream(tostream)
  selectoutput(stdout)
  //writef("return code = %n*n", 0)

  RESULTIS 0
}

AND getsamps(v, upb) BE
{ LET w = 0
  LET f = 440
  v!0 := upb
  FOR i = 1 TO upb DO
  { w := w+f
    v!i := w>0 -> 1000, -1000
    IF w>22050 DO w := w-44100
  }  
}

AND getmic(v, upb) BE
{ LET count = 0 // Count of samples read

  LET waveInCB = sys(Sys_sound, snd_waveInOpen, 0, 16, 1, 44100)

  //writef("waveInCB = %n*n", waveInCB)

  IF waveInCB=-1 DO
  { writef("Cannot open wave input device*n")
    v!0 := 0
    RETURN
  }

  writef("*nRecording %n samples*n", upb)

  UNTIL count>=upb DO
  { LET len = sys(Sys_sound,
                  snd_waveInRead, waveInCB, v+count, upb-count, 0)
    count := count+len
    //IF len DO writef("len = %i5 count = %i7 bufsize=%i7*n",
    //                 len, count, bufsize)
  }

  v!0 := upb

  //writef("Closing waveInCB*n")

  waveInCB := sys(Sys_sound, snd_waveInClose, waveInCB)
}

AND playsamples(v, t0, t1) BE
{ LET s0 = muldiv(44100, t0, 1_000) 
  LET s1 = muldiv(44100, t1, 1_000)
  sawritef("Playing from time %4.3d to %4.3d*n", t0, t1)
  playvec(v+s0, s1-s0+1)
}

AND wrwavfile(name, v) BE
{ // Output the .wav file
  LET upb = v!0
  LET wavout = findoutput(name)

  UNLESS wavout DO
  { writef("Unable to open %s for output*n", name)
    RETURN
  }
  
  selectoutput(wavout)

  wrriffhdr(1,         // mode = mono
            44100,     // typically 44100
            16,        // bits per sample
            upb*2)     // number of bytes of data
  FOR i = 1 TO upb DO
  { LET w = v!i
    binwrch(w)
    binwrch(w>>8)
  }
  endstream(wavout)
  selectoutput(stdout)
}

AND dotest(n, v, a, b, c) BE SWITCHON testno INTO
{ DEFAULT: writef("Unknown test %n(%n, %n, %n)*n", testno, a, b, c)
           ENDCASE

  CASE 1:  test1(v, a, b, c)
           ENDCASE
  CASE 2:  test2(v, a, b, c)
           ENDCASE
  CASE 3:  test3(v, a, b, c)
           ENDCASE
  CASE 4:  test4(v, a, b, c)
           ENDCASE
  CASE 5:  test2(v, a, b, c)
           ENDCASE
  CASE 6:  test2(v, a, b, c)
           ENDCASE
  CASE 7:  test2(v, a, b, c)
           ENDCASE
  CASE 8:  test2(v, a, b, c)
           ENDCASE
  CASE 9:  test2(v, a, b, c)
           ENDCASE
}

AND test1(v, a, b, c) BE
{ LET upb = v!0
  IF a>upb DO a := -1
  IF a>0 DO v, upb := v+a, upb-a
  IF b<0 DO b := upb

  FOR i = 1 TO b DO
  { IF i MOD 10 = 0 DO writef("*n%i8: ", a+i)
    writef(" %i6", v!i)
  }
  newline()

}

AND test2(v, frq, t0, t1) BE
{ LET upb = v!0
  LET x, y = 0, 0
  LET w = 0

  writef("*nAnalysing frequency %4.3d from time %4.3d to %4.3d*n*n",
          frq, t0, t1)

  FOR t = t0 TO t1 BY 10 DO
  { LET sn = muldiv(44100,  t, 1000) // Starting sample number
    LET len = 44100/10               // 1/20 sec sample period
    UNLESS  0 < sn < upb DO sn := 1

    IF FALSE DO
    { LET bestfrq, bestval = 0, -1
      FOR f = 100_000 TO 600_000 BY 1_000 DO
      { LET x = amp(v+sn, upb-sn, f, len)
        IF x>bestval DO frq, bestval := f, x
      }
      writef("Best frq=%4.3d val=%i6*n", frq, bestval)
    }

    writef("%9.3d:", t, frq)
    //writef("%i5 frq=%4.3d:", t, frq)
    FOR d = -8 TO 8 BY 2 DO
      writef(" %i5", amp(v+sn, upb-sn, frq+d*1000, len))
      //writef(" %i6", amp(v+t, upb-t, frq, len))
    newline()
  }

  newline() 
}

AND amp(v, upb, frq, len) = VALOF
{ LET bestx = 0

  FOR i = 0 TO 44100_000/frq DO
  { LET x = amp1(v+i, upb-i, frq, len)
    IF x>bestx DO bestx := x
  }
  RESULTIS bestx
} 

AND amp1(v, upb, frq, len) = VALOF
{ LET p, x, y = 0, 0, 0
  LET w = 0

  { LET a, b, c, d = 0, 0, 0, 0

    { a := a+v!p    // Quadrant 1
      w, p := w+frq, p+1
    } REPEATWHILE w<11025_000
    w := w-22050_000

    { b := b+v!p    // Quadrant 2
      w, p := w+frq, p+1
    } REPEATWHILE w<0

    { c := c+v!p    // Quadrant 3
      w, p := w+frq, p+1
    } REPEATWHILE w<11025_000
    w := w-22050_000

    { d := d+v!p    // Quadrant 4
      w, p := w+frq, p+1
    } REPEATWHILE w<0

    x := x + a + b - c - d
    y := y - a + b + c - d

    //writef("p=%n abcd= %i6 %i6 %i6 %i6  xy=%i6 %i6*n", p, a, b, c, d, x, y)
    //abort(1000)
  } REPEATWHILE p<len

  x, y := x/p, y/p
  //writef("*nfrq=%4.3d:p=%i5 x=%i6 y=%i6 power=%n*n", frq, p, x, y, x)
//abort(1000)
  RESULTIS x
}

AND test3(v, frq, t0, t1) BE
{ // This test generates a .ppm file which is rather bulky
  // being ASCII text
  LET s0 = muldiv(t0, 44100, 1000) // First sample number
  LET s1 = muldiv(t1, 44100, 1000) // Last sample number
  LET n1 =  12  // Lowest midi note
  LET n2 = 108  // Highest midi note
  LET picname = "pic.ppm"
  LET k = 100  // Samples per raster line
  picstream := findoutput(picname)

  UNLESS picstream DO
  { writef("Trouble with file: %s*n", picname)
    RETURN
  }

  setfreqtab()

  writef("*nAnalysing frequencies from time %4.3d to %4.3d*n*n",
          t0, t1)

  selectoutput(picstream)
  writef("P3*n")
  writef("%n %n*n", 
          n2-n1+1,           // x size = range of midi notes
          (s1-s0)/k+1)       // y size = number of raster lines
                             //          with one line per k samples
  writef("255*n")


  { LET s = s0
    WHILE s<= s1 DO
    { // generate a raster line
      LET layout = 0
      LET p = @v!s 

      FOR n = n1 TO n2 DO
      { LET f = freqtab!n
        LET a = famp(p, f) / 10
        //IF a<0 DO a := -a
        IF a<0 DO a := 0
        IF a>255 DO a := 255
        IF layout MOD 8 = 0 DO newline()
        layout := layout+1
        wrpixel(a)
      }
      newline()
      s := s+k
    }
  }
  endstream(picstream)
}

AND famp(p, f) = VALOF
{ // Return the average amplitude of 1/10 of a second
  // of samples starting at p multiplied by a square wave
  // of frequency f.
  // f is scaled so that f=440_000 represents 440.000 Hz.
  LET a, p0 = 0, p
  LET x = 0

//  writef("*nfamp: p=%n f=%9.3d*n", p, f)
  { x := x - 22_050_000
    { //writef("p=%n x=%i9 a=%i6 + %i6 => %i6*n",
      //        p, x, a, !p, a+!p)
      x, a := x+f, a+!p
      p := p+1
    } REPEATUNTIL x >= 0

    x := x + 22_050_000
    { //writef("p=%n x=%i9 a=%i6 - %i6 => %i6*n",
      //        p, x, a, !p, a-!p)
      x, a := x-f, a-!p
      p := p+1
    } REPEATUNTIL x <= 0
    //abort(1001)
  } REPEATUNTIL p-p0 > 44100/10
  RESULTIS a / (p- p0 + 1)
}

AND test4(v, frq, t0, t1) BE
{ // This test generates a .pbm format file that used
  // a colour table and one byte per pixel.
  // The size of the image is 15x25cms

  LET n1 = 12 +24   // Lowest Midi note processed
  LET n2 = 108-12   // Highest Midi note processed

  LET xsize = 3*(n2-n1+1) // Allow for three frequencies per semitone
  LET ysize = 2000   // Raster lines of frequency amplitudes 
  LET xres  = muldiv(xsize, 100, 15)  // 15 cms horizontal 
  LET yres  = muldiv(ysize+4, 100, 24)  // 25 cms vertical
  LET hdrsize     = 14
  LET infohdrsize = 40
  LET paletsize   = 4*256
  LET rowlen = (xsize+3) & -4 // Round up to a multiple of 4 bytes
  LET dataoffset = hdrsize + infohdrsize + paletsize
  LET pixeldatasize = ysize * rowlen
  LET colourtab = VEC 256
  LET workv = 0
  ///LET mcseg = globin(loadseg("mci386"))

  ///mc := 0        // No currently selected MC instance
  //mcb := 0
  currsampv := 0
  workv := 0
  ampv := 0
  bmpstream := 0

  ///UNLESS mcseg DO
  ///{ writef("Unable to load the MC package*n")
  ///  GOTO fin
  ///}

  // Create an MC instance for 10+xsize functions with a data space of
  // 10 words and code space of 40000 words
  ///mcb := mcInit(10+xsize, 10, 40000) // create an MC instance

  ///UNLESS mcb DO
  ///{ writef("Unable to create an mci386 instance*n")
  ///  GOTO fin
  ///}

  ///mc := 0          // Currently no selected MC instance
  ///mcSelect(mcb)

  currsampvupb := 2*44100/10 // 1/10 either side of the current position
  currsampv := getvec(currsampvupb) // Buffer for 1/5 sec of samples
  workv := getvec(currsampvupb)     // Buffer for 1/5 sec of samples
  ampv := getvec(xsize) // For the applitudes of each frequency
  currsampno := 0

  UNLESS currsampv & workv & ampv DO
  { writef("More space needed*n")
    GOTO fin
  }

  writef("*nAnalysing from time %4.3d to %4.3d*n*n", t0, t1)

  initcolourtab(colourtab)

  setfreqtab()

  // Create the amplitude functions
  ///mcK(mc_debug, 1)
  IF FALSE DO
  FOR x = 0 TO xsize-1 DO
  { LET xby3 = x/3
    LET xmod3 = x MOD 3
    LET n = n1 + x/3
    LET freq = freqtab!n
    LET k = 0
    IF xmod3 = 0 DO freq := (freqtab!(n-1)+2*freq)/3
    IF xmod3 = 2 DO freq := (freqtab!(n+1)+2*freq)/3
    k := 22 + freq/(44*1000)  // is 32 cycles for A (440Hz)
                              //    42 cycles for A (880Hz)
    //k :=  6 + freq/(44*1000)  // is 16 cycles for A (440Hz)
    //                          //    26 cycles for A (880Hz)

    //sawritef("Creating amplitude function for freq=%9.3d x=%i3*n", freq, x)
    ///cmplampfn(x,            // Note number, function number = 10+x
    ///          freq,         // 1000 times the frequency to sample
    ///          currsampv,    // buffer around the sample point
    ///          currsampvupb, // UPB of currsampv
    ///          workv,        // to hold the average of k cycles
    ///          k)            // take the average of k cycles
    //abort(1000)
  }

  bmpstream := findoutput(bmpname)

  UNLESS bmpstream DO
  { writef("Trouble with file: %s*n", bmpname)
    RETURN
  }

  selectoutput(bmpstream)

  // Write the header
  wr1('B'); wr1('M') // "BM"
  //wr4(hdrsize + infohdrsize + pixeldatasize) // File size in bytes
  wr4(dataoffset + pixeldatasize) // File size in bytes
  wr4(0)             // Unused
  wr4(dataoffset)    // File offset of pixel data

  // Write the Info header
  wr4(40)             // Size of info header = 40
  wr4(xsize)          // Bitmap width
  wr4(ysize+4)          // Bitmap height
  wr2(1)              // Number of planes = 1
  wr2(8)              // 8 bits per pixel
  wr4(0)              // No compression
  //wr4(pixeldatasize)  // Size of image
  wr4(0)              // Size of image =0 valid if no compression
  wr4(xres)           // Horizontal resolution in pixels per meter
  wr4(yres)           // Vertical   resolution in pixels per meter
  wr4(256)            // Number of colours actually used
  wr4(0)              // All colours are important

  // Write the colour table
  FOR i = 0 TO 255 DO wr4(colourtab!i)

  { // Print the palette colours
    FOR y = 1 TO 4 DO // 4 raster lines for the palette
    { FOR  x = 0 TO xsize-1 DO
      { LET col = 255 * x / (xsize-1)
        wr1(x=0!y=4->255, col)
      }
      FOR x=xsize+1 TO rowlen DO wr1(0) // Pad to a multiple of 4
    }
  }
 
  FOR y = 0 TO ysize-1 DO
  { LET t = t0 + muldiv(t1-t0, y, ysize-1)
    LET s = muldiv(44100, t, 1_000) // Number of first sample
    LET gridline = FALSE
    LET len = 0
    STATIC { prevt=0 }

    sawritef("time %6.3d raster line %i4 of %i4*n", t, y+1, ysize)

    UNLESS t/gridmsecs = prevt/gridmsecs DO
    { // Write the time every 100 msecs
      prevt := t
      sawritef("Grid line at time: %8.3d sample pos=%n*n", t, s)
      gridline := TRUE
    }

    // Copy 1/5 secs of samples into currsampv
    { LET p = s - 44100/10
      LET offset = 0
      IF p < 0 DO
      { offset := -p
        FOR i = 0 TO offset-1 DO currsampv!i := 0
      }
      FOR i = offset TO currsampvupb DO currsampv!i := v!(p+i)

IF FALSE & t>1000 DO
{ sawritef("sampv=%n currsampv=%n s=%n time=%5.3d*n",
          sampv, currsampv, s, t)
  abort(1000)
}
    }

    FOR x = 0 TO xsize-1 DO
    { LET xby3 = x/3
      LET xmod3 = x MOD 3
      LET n = n1 + x/3
      LET freq = freqtab!n
      LET k = 22 + freq/(44*1000)  // is 32 cycles for A (440Hz)
                                   //    42 cycles for A (880Hz)
      //LET k =  6 + freq/(44*1000)  // is 16 cycles for A (440Hz)
      //                             //    26 cycles for A (880Hz)

      IF xmod3 = 0 DO freq := (freqtab!(n-1)+2*freq)/3
      IF xmod3 = 2 DO freq := (freqtab!(n+1)+2*freq)/3
//debug := n=69 & 1000<=t<=1050
      //ampv!x := mcfamp4(x)
      ampv!x := famp4(freq,
                      currsampv,
                      currsampvupb,
                      workv,
                      k)
    }

    FOR x = xsize-1 TO 0 BY -1 DO
    { LET a = ampv!x
      LET n = n1 + x/3

//sawritef("test4: y=%i4 n=%i3 x=%i4 a=%i5*n", y, n, x, a)
//abort(1000)
      IF a<  0 DO a := 0
      IF a>127 DO a := 128 + (a-128)/4
      IF a>255 DO a := 255

      TEST gridline
      THEN { LET col = 0
             LET note = n MOD 12
             SWITCHON note INTO
             { DEFAULT:  col := 0          // White notes
                         ENDCASE
               CASE  0:  col := 30         // C   
                         ENDCASE
               CASE  1:                    // C#  Black
               CASE  3:                    // D#  Black
               CASE  6:                    // F#  Black
               CASE  8:                    // G#  Black
               CASE 10:  col := 255        // A#  Black
                         ENDCASE
             }
             wr1(col)
           }
      ELSE wr1(a)
      len := len + 1
    }
    FOR x = xsize+1 TO rowlen DO wr1(0) // Pad up to next 32-bit boundary
  }

fin:
  ///IF mc        DO mcClose()
  ///IF mcseg     DO unloadseg(mcseg)
  IF currsampv DO freevec(currsampv)
  IF workv     DO freevec(workv)
  IF ampv      DO freevec(ampv)
  IF bmpstream DO endstream(bmpstream)
}

AND initcolourtab(ctab) BE
{ LET colours = TABLE       10,
      //  red     green    blue
        255<<16 | 255<<8 | 255,  // White
        150<<16 | 150<<8 | 150,  // Light grey
          0<<16 | 150<<8 | 150,  // 
          0<<16 |   0<<8 | 190,  // Blue
        130<<16 |   0<<8 | 130,  //
        150<<16 |   0<<8 |   0,  // Red
        140<<16 | 140<<8 |   0,  // 
          0<<16 | 180<<8 |   0,  // Green
        100<<16 | 100<<8 | 100,  // Dark grey
          0<<16 |   0<<8 |   0   // Black

  FOR i = 1 TO colours!0-1 DO
  { LET n = colours!0
    LET p = (256*(i-1))/(n-1)
    LET q = (256*i)/(n-1)
    LET c1, c2 = colours!i, colours!(i+1)
    LET r1, g1, b1 = c1>>16&255, c1>>8&255, c1&255
    LET r2, g2, b2 = c2>>16&255, c2>>8&255, c2&255
//sawritef("i=%i2 p=%i3  q=%i3  c1=%x6 c2=%x6*b*n", i, p, q, c1, c2)
    FOR t = p TO q DO
    { LET r = (r1*(q-t)+r2*(t-p))/(q-p)
      LET g = (g1*(q-t)+g2*(t-p))/(q-p)
      LET b = (b1*(q-t)+b2*(t-p))/(q-p)
      ctab!t := r<<16 | g<<8 | b
      //sawritef("%i3: %x6*n", t, ctab!t)
    }
    //sawritef("*n")
    //abort(1000)
  }
//sawritef("*nColour table*n")
  //FOR i = 0 TO 256 DO
  //{ IF i MOD 8 = 0 DO sawrch('*n')
  //  sawritef(" %x6", ctab!i)
  //}
  //sawrch('*n')    
}

/*
AND cmplampfn(x, freq, sv, svupb, wv, k) = VALOF
{ // Compile function 10+x to compute the amplitude of frequency freq
  // at the mid point of samples held held in sv (UPB svupb).
  // wv is the same size as sv and is set to the average waveform.
  // k is the number of cycles to average. k is reduced appropriately
  // if k cycles has more than svupb samples.
  // The result is TRUE if compilation is successful.
  mcKKK(mc_entry, 10+x, 3, 0)

  mcRK(mc_mv, mc_a, x/2)       // Dummy function

  mcF(mc_rtn)
  mcF(mc_endfn)
//abort(1111)
} 

AND mcfamp4(note) = VALOF
{ LET amp = mcCall(10+note, 0, 0, 0)
  RESULTIS amp
}
*/


AND famp4(freq, sv, svupb, w, cycles) = VALOF
{ // This versions calculate the amplitude of frequency freq for
  // samples centred at position p. cycles is the number of complete
  // cycles to sample. cycles is reduced if this number of cycles exceeds
  // 1/5 second.
  LET wupb2 = (44_100_000 / freq)/2 // Length of a half cycle
  LET wupb  = 2*wupb2               // Length of a whole cycle
  LET total = 0
  LET ltot, rtot = 0, 0
  LET v = @w!(wupb2+1) // The righthand half of w
  LET amp = 0
  LET base = 0  // where the cycles in sv will start
  LET len = 0

IF debug DO
{ sawritef("*nfreq=%6.3d sv=%n svupb=%n w=%n cycles=%n*n",
           freq, sv, svupb, w, cycles)
}
  { len := muldiv(cycles, 44_100_000, freq) // samples in given cycles
//sawritef("freq=%5.3d len=%i5 currsampvupb=%n cycles=%n*n",
//          freq, len, currsampvupb, cycles)
    IF len<=currsampvupb DO { base := (currsampvupb-len)/2; BREAK }
    cycles := cycles-1
  } REPEAT

IF debug DO
{ sawritef("famp4: base=%n len=%n cycles=%n*n", base, len, cycles)
  sawritef("famp4: wupb2=%n wupb=%n*n", wupb2, wupb)
  abort(1001)
}

  // Ignore very low frequencies
  IF cycles=0 RESULTIS 0

  // So for freq = 440_000  (A above middle C)
  // * * * ... * | * * * ... *
  // ^         ^   ^         ^
  // 0         49  50        99
  // 0         wupb2         wupb
  FOR i = 0 TO wupb DO w!i := 0
  FOR k = 0 TO cycles-1 DO
  { LET posk = muldiv(k, 44_100_000, freq)
    LET q = @currsampv!(base+posk)
IF debug DO sawritef("%i2: posk=%i5*n", k, posk)
    FOR i = 0 TO wupb DO w!i := w!i + q!i
  }
  FOR i = 0 TO wupb DO w!i := w!i/cycles

  IF debug DO
  { // Write the average cycle
    FOR i = 0 TO wupb DO
    { IF i MOD 10 = 0 DO sawrch('*n')
      sawritef(" %i7", w!i)
    }
    sawrch('*n')
//abort(1002)
  }


  // w!0 ... w!wupb contains the average shape of (just over) one cycle.
  // Now calculate the amplitude and phase.
  { LET mean = 0
    LET wl, wr = w!0, w!wupb
    //sawritef("f=%8.3d cycles=%n wupb2=%n wupb=%n*n", f, cycles, wupb2, wupb)

    //Adjust the average slope
    //FOR i = 0 TO wupb DO w!i := w!i - muldiv(i, wr - wl, wupb)

  IF debug DO
  { // Write the average cycle
    sawritef("*nAfter slope adjustment*n")
    FOR i = 0 TO wupb DO
    { IF i MOD 10 = 0 DO sawrch('*n')
      sawritef(" %i7", w!i)
    }
    sawrch('*n')
//abort(1003)
  }

    // Adjust the mean
    FOR i = 0 TO wupb DO mean := mean + w!i
    mean := mean/(wupb+1)
    //FOR i = 0 TO wupb DO w!i := w!i - mean

    IF debug DO
    { // Write the average cycle
      sawritef("*nAfter mean adjustment, mean=%n*n", mean)
      FOR i = 0 TO wupb DO
      { IF i MOD 10 = 0 DO sawrch('*n')
        sawritef(" %i7", w!i)
      }
      sawrch('*n')
      abort(1004)
    }
  }

  // Initialise total, ltot, amp
  ltot, rtot, amp := 0, 0, 0
  FOR i = 0       TO wupb2 DO ltot := ltot + w!i
  FOR i = wupb2+1 TO wupb  DO rtot := rtot + w!i
  total := ltot+rtot

  { LET tot2 = total/2
    FOR i = 0 TO wupb2 DO
    { LET a = ABS(ltot - tot2) // = |ltot-rtot|/2
      IF amp<a DO amp :=  a
IF debug DO
{ sawritef("i=%i4 tot=%n ltot=%i8 rtot=%i8 diff/2=%i8 amp=%i8*n",
            i, total, ltot, total-ltot, ltot-tot2, amp)
  //abort(1002)
}
      ltot := ltot - w!i + v!i
    }
  }
//  amp := (2*amp) / (cycles * (wupb + 1))
  amp := (2*amp) / (wupb + 1)
IF debug DO
{ sawritef("freq=%8.3d amp=%i6 cycles=%n wupb=%n*n", freq, amp, cycles, wupb)
  abort(1111)
}
  RESULTIS  muldiv(amp, ampfactor, 100)
}

AND getwav(name, v, upb) BE
{ LET instream = 0

  IF FALSE DO
  { //Set a step up from -20000 to +20000 at t = 1 sec
    //FOR i = 0 TO upb DO v!i := i<44100 -> -20000, +20000

    //Set a slope up from -20000 to +20000 at t = 1 sec for 1/20 sec
    //FOR i = 0 TO 44100 DO v!i := -20000
    //FOR i = 0 TO 2205  DO v!(i+44100) := -20000 + muldiv(40000, i, 2205)
    //FOR i = 44100+2205 TO upb DO v!i := +20000

    // Set a 440Hz sine wave with varying amplitude
    //IF FALSE DO
    { LET x      = 1_000_000_000
      LET xdot   = 0
      LET k      = 3925     // 440Hz
      LET amp    = 20_000
      LET ampdiv = 1_000_000_000/amp

      FOR i = 0 TO 44100 DO // Increase volume for 1 sec
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!i := muldiv(x/ampdiv, i, 44100)
        //v!i := muldiv(x/ampdiv, 44100, 44100)
      }
      FOR i = 44100 TO 88200 DO // Steady volume for 1 sec
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!i := x/ampdiv
      }
      FOR i = 0 TO 44100 DO // Decrease volume to the end
      { LET xdotdot = -muldiv(x, k, 1_000_000)
        xdot := xdot + xdotdot
        x := x + xdot
        v!(88200+i) := muldiv(x/ampdiv, 44100-i, 44100)
      }
      FOR i = 3*44100 TO upb DO v!i := 0
    }

   IF FALSE DO
     FOR i = 0 TO 44100+4410 DO
     { IF i MOD 4410 = 0 DO abort(1000)
       IF i MOD 10 = 0 DO sawritef("*n%i5:", i)
       sawritef(" %i6", v!i)
     }

    RETURN
  }

  instream := findinput(name)

  UNLESS instream DO
  { writef("Trouble with .wav file %s*n", name)
    v!0 := 0
    RETURN
  }
  writef("Getting wav data from %s*n", name)

  selectinput(instream)
  wav2v(v, upb)
  endstream(instream)
  writef("Number of samples = %n (%5.3d secs)*n",
          v!0, muldiv(v!0, 1000, 44100))
}

AND wav2v(v, upb) = VALOF
{ // Read up to upb samples from the .wav file which is the
  // currently selected input and place them as 32-bit signed integers
  // in the vector v whose upperbound is upb.
  // The result is n the number of samples placed in v.
  // v!0=n and v!1 .. v!n are the samples.
  // If the .wav file is in stereo the left and right samples
  // are averaged.
  LET riff  = rd4()       //  0 RIFF
  LET size  = rd4()       //  4 filesize - 8
  LET wave  = rd4()       //  8 WAVE
  LET fmt   = rd4()       // 12 fmt
  LET chksz = rd4()       // 16 16
  LET quant = rd2()       // 20 1 = linear
  LET mode  = rd2()       // 22 1=mono 2=stereo
  LET rate  = rd4()       // 24 Typically 44100
  LET brate = rd4()       // 28 byte rate
  LET bytePerSample=rd2() // 32 1, 2 or 4  = bits/8 * mode
  LET bits  = rd2()       // 34 bits per sample = 8 or 16
  LET filler= chksz=18->rd2(), 0       // 36 filler

  LET data  = rd4()       // 36 data
  LET len   = rd4()       // 40 bytes of data or -1

  UNLESS riff=#x46464952 DO sawritef("Bad RIFF word %x8*n", riff)
  UNLESS wave=#x45564157 DO writef("Bad WAVE word %x8*n", wave)
  UNLESS fmt =#x20746D66 DO writef("Bad fmt  word %x8*n", fmt)
  //UNLESS chksz=16        DO writef("Bad subchunk size %n*n", chksz)
  UNLESS mode=1 | mode=2 DO writef("Bad mode %n*n", mode)
  UNLESS rate=44100      DO writef("Bad rate %n*n", rate)
  UNLESS bits=16         DO writef("Bad bits per sample %n*n", bits)
  UNLESS data=#x61746164 DO writef("Bad data word %x8*n", data)

writef("rate=%n %s*n", rate, mode=2->"stereo", "mono")
  v!0 := 0

  FOR i = 1 TO upb DO
  { LET w = rd2()
    IF w<0 BREAK
    IF mode=2 DO w := (w+rd2())/2
    IF (w & #x8000)~=0 DO w := w | #xFFFF0000 // Sign extend
    v!i := w
    v!0 := i
  }
  RESULTIS v!0
}

AND wrriffhdr(mode, rate, bits, databytes) BE
{ LET bytes_per_sample = bits/8 * mode
  LET byte_rate = bytes_per_sample * rate
  writes("RIFF")        //  0: R I F F
  wr4(36+0)             //  4: size of this file - 8
  writes("WAVE")        //  8: W A V E
  writes("fmt ")        // 12: f m t
  wr4(16)               // 16: fmt subchunk size is 16
  wr2(1)                // 20: 1 = linear quantisation
  wr2(mode)             // 22: 1 = mono, 2=stereo
  wr4(rate)             // 24: samples per second
  wr4(byte_rate)        // 28: bytes per second
  wr2(bytes_per_sample) // 32: block align -- bits/8 * mode  = 1, 2 or 4
  wr2(bits)             // 34: bits per sample  = 8 or 16
  writes("data")        // 36: d a t a
  //wr4(byte_rate * 1)    // 40: number of bytes of data or zero
  wr4(databytes)        // 40: number of bytes of data or -1
}

AND wr1(b) BE
{ binwrch(b)
}

AND wr2(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
}

AND rd2() = VALOF
{ LET w = 0
  LET s = @w
  s%0 := binrdch()
  s%1 := binrdch()
  RESULTIS w
}

AND wr4(w) BE
{ LET s = @w
  binwrch(s%0)
  binwrch(s%1)
  binwrch(s%2)
  binwrch(s%3)
}

AND rd4() = VALOF
{ LET w = 0
  LET s = @w
  s%0 := binrdch()
  s%1 := binrdch()
  s%2 := binrdch()
  s%3 := binrdch()
  RESULTIS w
}

AND playvec(buf, upb) BE
{ LET waveOutCB = sys(Sys_sound,
                      snd_waveOutOpen, 0, 16, 1, 44100)

  //writef("waveOutCB = %n format=%n channels=%n rate=%n*n",
  //       waveOutCB, format, channels, rate)

  IF waveOutCB=-1 DO
  { writef("Cannot open wave output device*n")
    RETURN
  }

  //writef("*nPlaying %n samples*n", upb)

  { LET count = 0 // Count of samples sent played

    UNTIL count>=upb DO
    { LET len = sys(Sys_sound,
                    snd_waveOutWrite, waveOutCB, buf+1+count, upb-count)
      count := count+len
      //IF len DO writef("len = %i5 count = %i7 bufsize=%i7*n",
      //                 len, count, bufsize)
      IF len<0 BREAK
//writef("Delaying 1 second*n")
      sys(Sys_delay, 1)
    }

  }

  sys(Sys_sound, snd_waveOutClose, waveOutCB)
}

AND setfreqtab() BE
{ // Set freqtab so that freqtab!n = 1000 * the note frequency
  // where n is the MIDI note number. n=60 for middle C (C4).

  // Only notes with non zero frequencies will have recognisers.

  freqtab := TABLE
     8_176,   8_662,   9_178,   9_723,  10_301,  10_914, //   0 c-1.. b-1
    11_563,  12_250,  12_979,  13_750,  14_568,  15_434,

    16_352,  17_324,  18_355,  19_446,  20_602,  21_827, //  12 c0 .. b0
    23_125,  24_500,  25_957,  27_500,  29_136,  30_868,

    32_703,  34_648,  36_709,  38_891,  41_204,  43_654, //  24 c1 .. b1
    46_250,  49_000,  51_914,  55_000,  58_271,  61_736,
  
    65_406,  69_296,  73_417,  77_782,  82_407,  87_308, //  36 c2 .. b2
    92_499,  97_999, 103_827, 110_000, 116_541, 123_471,

   130_812, 138_592, 146_833, 155_564, 164_814, 174_615, //  48 c3 .. b3
   184_998, 195_998, 207_653, 220_000, 233_082, 246_942,

   261_623, 277_183, 293_665, 311_127, 329_628, 349_229, //  60 c4 .. b4
   369_995, 391_996, 415_305, 440_000, 466_164, 493_884,

   523_245, 554_366, 587_330, 622_254, 659_255, 698_457, //  72 c5 .. b5
   739_989, 783_991, 830_610, 880_000, 932_328, 987_767,

  1046_489,1108_731,1174_659,1244_508,1318_510,1396_913, //  84 c6 .. b6
  1479_978,1567_982,1661_219,1760_000,1864_655,1975_533,

  2092_978,2217_461,2349_318,2489_016,2637_020,2793_826, //  96 c7 .. b7
  2959_955,3135_963,3322_438,3520_000,3729_310,3951_066,

  4185_955,4434_922,       0,       0,       0,       0, // 108 c8 .. b8
         0,       0,       0,       0,       0,       0,

         0,       0,       0,       0,       0,       0, // 120 c9 .. g9
         0,       0

  // Check the table
  checktab( 98, 2349_318)
  checktab( 99, 2489_016)
  checktab(100, 2637_020)
  checktab(101, 2793_826)
  checktab(102, 2959_955)
  checktab(103, 3135_963)
  checktab(104, 3322_438)
  checktab(105, 3520_000)
  checktab(106, 3729_310)
  checktab(107, 3951_066)
  checktab(108, 4185_955)
  checktab(109, 4434_922)
}

AND checktab(n, f) BE WHILE n>=0 DO
  { UNLESS freqtab!n = f DO
    { writef("note=%i3 change %8.3d to %8.3d*n", n, freqtab!n, f)
      abort(1000)
    }
    n, f := n-12, (f+1)/2
  }


AND wrnotename(n) BE
{ LET oct = n/12 - 1
  LET note = n MOD 12
  UNLESS 0<=n<=127 DO
  {  writef("Bad")
     RETURN
  }

  SWITCHON note INTO
  { DEFAULT: writef("Bad");           RETURN
    CASE  0:  writef( "c%n",  oct);   RETURN
    CASE  1:  writef( "c#%n", oct);   RETURN
    CASE  2:  writef( "d%n",  oct);   RETURN
    CASE  3:  writef( "d#%n", oct);   RETURN
    CASE  4:  writef( "e%n",  oct);   RETURN
    CASE  5:  writef( "f%n",  oct);   RETURN
    CASE  6:  writef( "f#%n", oct);   RETURN
    CASE  7:  writef( "g%n",  oct);   RETURN
    CASE  8:  writef( "g#%n", oct);   RETURN
    CASE  9:  writef( "a%n",  oct);   RETURN
    CASE 10:  writef( "a#%n", oct);   RETURN
    CASE 11:  writef( "b%n",  oct);   RETURN
  }
}           

AND wrpixel(val) BE
{ STATIC { ctab = 0 }
  LET col = 0

  UNLESS ctab DO
  { // Initialise ctab on first call
    LET colours = TABLE
        10,
        255<<16 | 255<<8 | 255,  // White
        125<<16 | 125<<8 | 125,  // Grey
        255<<16 |   0<<8 | 255,  // Violet
          0<<16 |   0<<8 | 255,  // Blue
          0<<16 | 255<<8 |   0,  // Green
        255<<16 | 255<<8 |   0,  // Yellow
        255<<16 | 148<<8 |   0,  // Orange
        255<<16 |   0<<8 |   0,  // Red
         60<<16 |   0<<8 |   0,  // Brown
          0<<16 |   0<<8 |   0   // Black

    ctab := TABLE
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//  0
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 16
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 32
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 48
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 64
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 80
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,// 96
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//112
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//128
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//144
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//160
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//176
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//192
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//208
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//224
      0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,//240
      0                                                               //256

    FOR i = 1 TO colours!0-1 DO
    { LET n = colours!0
      LET p = (256*(i-1))/(n-1)
      LET q = (256*i)/(n-1)
      LET c1, c2 = colours!i, colours!(i+1)
      LET r1, g1, b1 = c1>>16&255, c1>>8&255, c1&255
      LET r2, g2, b2 = c2>>16&255, c2>>8&255, c2&255
//sawritef("i=%i2 p=%i3  q=%i3  c1=%x6 c2=%x6*b*n", i, p, q, c1, c2)
      FOR t = p TO q DO
      { LET r = (r1*(q-t)+r2*(t-p))/(q-p)
        LET g = (g1*(q-t)+g2*(t-p))/(q-p)
        LET b = (b1*(q-t)+b2*(t-p))/(q-p)
        ctab!t := r<<16 | g<<8 | b
        //sawritef("%i3: %x6*n", t, ctab!t)
      }
      //sawritef("*n")
      //abort(1000)
    }
    FOR i = 0 TO 256 DO
    { IF i MOD 8 = 0 DO sawrch('*n')
      sawritef("", ctab!i)
    }
    sawrch('*n')    
  }

  IF val<  0 DO val :=   0
  IF val>256 DO val := 256
 
  col := ctab!val
  //sawritef("val=%i3  col=%x6*n", val, col)
  writef(" %n %n %n", col>>16 &255, col>>8&255, col&255)
}

