val junk = 1;

(*
// A Discrete event simulation benchmark 
// translated from BCPL into New Jersy SML by Martin Richards (c) July 2004 

// This is a benchmark test for a discrete event simulator using
// BCPL style coroutines. It simulates a network of n nodes which
// each receive, queue, process and transmit messages to other nodes.
// The nodes are uniformly spaced on a straight line and the network
// delay is assumed to be proportional to the linear distance between
// the source and the destination. On arrival, if the receiving node is
// busy the message is queued, otherwise it is processed immediately.
// After processing the message for random time it is sent to another
// random node. If the node has a non empty queue it dequeues its first
// message and starts to process it, otherwise it becomes suspended.
// Initially every node is processing a message and every queue is
// empty. There are n coroutines to simulate the  progress of each
// message and the discrete event priority queue is implemented using
// the heapsort heap structure. The simulation stops at a specified
// simulated time. The result is the number of messages that have been
// processed. A machine independent random number generator is used
// so the resulting value should be indepent of implementation
// language and machine being used.
*)


open SMLofNJ;
open StringCvt;
(* open CommandLine; *)
open Cont;
open Vector;

exception CoErr of string;

(* CoVal is the type of value passed from one coroutine to another *)
datatype CoVal = ValInt of int
               | ValCo  of CoVal cont ref *  (* The Continuation *)
                           CoVal ref         (* The Parent *)
               | ValRef of CoVal ref
               | ValNull


fun wrs x = print x;

fun nl() = wrs "\n";

fun wri(x, n) = print(StringCvt.padLeft #" " n (Int.toString x));


fun wrv (ValInt  x) = wri(x, 0)
|   wrv (ValCo   x) = wrs "ValCo"
|   wrv (ValRef  x) = wrs "ValRef"
|   wrv  ValNull    = wrs "ValNull";

val res = ref ValNull;

(* currco is the current coroutine *)
val currco = ref (callcc(fn me => throw me (ValCo(ref me, ref ValNull))));


fun getcontin (ValCo(ref con, _)) = con
|   getcontin _  = raise CoErr("getcontin");

fun setcontin ((ValCo(res, _)), x) = res := x
|   setcontin _  = raise CoErr("setcontin");

fun getparent (ValCo(_, ref res)) = res
|   getparent _  = raise CoErr("getparent");

fun setparent ((ValCo(_, res)), x) = res := x
|   setparent _  = raise CoErr("setparent");

fun getint (ValInt x) = x
|   getint _  = raise CoErr("getint");

fun getref (ValRef x) = x
|   getref _  = raise CoErr("getref");

fun changeco(co, x) =
  callcc(fn me =>
          ( setcontin(!currco, me);   (* save our continuation *)
            currco := co;
            throw (getcontin co) x;   (* give control to co *)
            x
          )
        );

fun callco(co, x) =
( setparent(co, !currco);
  changeco(co, x)
);

fun resumeco(co, x) =
( let val p = getparent(!currco)
  in setparent(!currco, ValNull);
     setparent(co, p);
     changeco(co, x)
  end
);

fun cowait x =
  let val co = getparent(!currco)
  in setparent(!currco, ValNull);
     changeco(co, x)
  end;

fun createco f = callcc(fn me =>
  let val c = ref (ValCo(ref me,          (* the continuation *)
                         ref (!currco)))  (* the parent *)
  in setcontin(!currco, me);
     currco := !c;                        (* Make it the current coroutine *)

     while true do c := f(cowait (!c));   (* The main coroutine loop *)
     !c   (* just to get the result type right *)
  end);


(* Cosim Benchmark program *)

val nodes    = ref 5;          (* The number of nodes *)
val stoptime = ref 1000;       (* Time to stop the simulation *)
val ptmax    = ref 500         (* The maximum processing time *)
val seed     = ref 0           (* The seed for the random number generator *)
val tracing  = ref false;

val wkqv     = ref [];         (* The vector of work queues *)
val count    = ref 0;          (* count of messages processed *)

val cov      = ref [];         (* Vector of message coroutines *)
val simtime  = ref 0;          (* Simulated time *)



(* ################### Random number generator #######################
//
// The following random number generator is based on one give
// in Knuth: The art of programming, vol 2, p 26.
*)
structure Rnd = struct

val a : int list ref = ref []
and b : int list ref = ref []
and c : int list ref = ref []
and d : int list ref = ref [];

fun seta  0 x y r = a := (y::r)
|   seta  n x y r = seta (n-1) y ((x+y) mod 0x10000000) (y::r)

(*
and seta n x y r =
( wri(n,3); wri(54-n, 3); wri(x,10); wri(y,10); nl(); seta1 n x y r )

and setc n x y r =
( wri(n,3); wri(54-n, 3); wri(x,10); wri(y,10); nl(); setc1 n x y r )
*)
and setc 24 x y r = ( c := (y::r); nl();
                      seta 23 y ((x+y) mod 0x10000000) []
                    )
|   setc n  x y r =   setc (n-1) y ((x+y) mod 0x10000000) (y::r);


fun init seed = 
  let val x = 0x2345678+seed
      and y = 0x5362781
  in a := [];
     b := [];
     c := [];
     d := [];
     setc 54 y ((x+y) mod 0xFFFFFFF) [] 
  end;

fun next n =
  case (!b, !d) of
    (x::xs, y::ys) => let val t = (x+y) mod 0x10000000
                      in a := t :: (!a);
                         b := xs;
                         c := y :: (!c);
                         d := ys;
(* wri(x, 10); wri(y,10); wri(t, 10); wri(t mod n, 10); nl(); *)
                         t mod n
                      end

  | (   [],    _ ) =>  ( b := rev (!c);
                         c := [];
                         next n
                       )
  | (    _,    _ ) =>  ( d := rev (!a);
                         a := [];
                         next n
                       );

fun test 0 = nl()
|   test n = ( if n mod 10 = 0 then nl() else ();
               wri(next 10000, 5);
               test(n-1)
             )
end;

(* ############################# Wkq #################################
//
// The following implements the FIFO work queues
*)
structure Wkq = struct

datatype Node = Nd of bool ref       * (* true if busy *)
                      CoVal list ref * (* list of coroutines *)
                      CoVal list ref;  (* list of coroutines *)

fun mkNodeList 0 R = R
|   mkNodeList n R = mkNodeList (n-1) (Nd(ref false, ref[], ref[]) :: R);

(* The vector holding the priority queue *)
val wkqv = ref (fromList(mkNodeList 0 []));

fun init n = 
(* Create the vector of nodes *)
  wkqv := fromList(mkNodeList (n+1) []);

fun len [] = 0
|   len (_::xs) = 1 + len xs;

fun prwaitq node =
  let val Nd(busy, a, b) = sub(!wkqv, node)
  in
     (* wrs "wkq for node ";wri(node, 0); wrs ":"; *)
     if !busy
     then wrs "busy"
     else ();
 
     wrs " len="; wri(len(!a)+len(!b), 0); nl()
  end;

fun qitem node =
(* The message has reached this node *)
(* If currently not busy, mark as busy and return immediately *)
(* to process the message, otherwise append it to the end of  *)
(* the work queue for this node. *)
  let val  Nd(busy, a, b) = sub(!wkqv, node)
  in
(* wrs "qitem: entered\n"; *)
(* prwaitq node; *)
  if !busy
  then ( (* Append item to the end of this queue *)

         if !tracing
         then ( wri(!simtime,8); wrs ": node "; wri(node,4);
                wrs ": busy so appending message to end of work queue\n"
              )
         else ();
         a := (!currco) :: !a;
         (* prwaitq(node); *)
         cowait ValNull (* Wait to be activated (by dqitem) *)
       )
  else ( (* The node was not busy *)
         busy := true;  (* Mark node as busy *)
         if !tracing
         then ( wri(!simtime,8); wrs ": node "; wri(node,4);
                wrs ": node not busy\n"
              )
         else ();
         (* prwaitq(node); *)
         ValNull
       )
  end;

fun dqitem node =
(* A message has just been processed by this node and is ready to *)
(* process the next, if any. *)

  let val Nd(busy, a, b) = sub(!wkqv, node) (* Current item (~=0) *)
(*writef("dqitem(%n): entered, item=%n\n", node, item) *)
(* prwaitq node; *)
  in
  ( if !b = []
    then ( b := rev(!a);
           a := []
         )
    else ();
    if !b = []
    then busy := false            (* The node is no longer busy *)
    else let val co::bs = !b
         in ( b := bs;
(*prwaitq node; *)
              callco(co, ValNull);   (* Process the next message *)
              ()
            )
         end
  )
  end;
end;



(* ######################## Priority Queue ###########################
//
// The following priority queue is implemented using a heapsort-style
// heap.
*)
structure Pri = struct
  datatype Ev = Event of int * CoVal (* time, corresponding coroutine *)
              | Empty;

  fun mkHeapList 0 R = R
    | mkHeapList n R = mkHeapList (n-1) (ref Empty :: R);

  (* The vector holding the priority queue *)
  val heap = ref (fromList(mkHeapList 0 []));

  val priqn = ref 0;         (* Current number of items in the preiority Q *)

  fun init n = 
  (* Create a heap vector with upb n *)
    heap := fromList(mkHeapList (n+1) []);

fun eventTime Empty = 0
  | eventTime (Event(t, _)) = t;

fun eventCo Empty = ValNull
  | eventCo (Event(_, co)) = co;

fun prpriq(i, n) = 
  if i>n then nl()
         else ( wrs " "; wri(eventTime(!(sub(!heap, i))), 4);
                prpriq(i+1, n)
              );

fun prq() =
( prpriq(1, !priqn);
  nl()
)

(* wrs "upheap: eventtime="; wri(evtime,5); wrs " i="; wri(i,5); nl() *)

fun upheap(ev, 1) = sub(!heap, 1) := ev

  | upheap(ev, i) =
      let val p = i div 2;          (* Parent of i *)
          val evp = !(sub(!heap, p))
      in if eventTime ev >= eventTime evp
         then sub(!heap, i) := ev
         else ( sub(!heap, i) := evp;     (* Demote the parent *)
                (* prq(); *)
                upheap(ev, p)
              )
      end;


(* fun downheap(ev, i) = *)
(* wrs "downheap: eventtime="; wri(evtime, 5); wrs " i="; wri(i,5); nl(); *)
(* prq(); *)

fun downheap(ev, i) =
  let val j = 2*i (* left child, if present *)
  in ( if j > !priqn
       then upheap(ev, i) (* no children *)
       else if (j < !priqn andalso
                eventTime(!(sub(!heap, j))) > eventTime(!(sub(!heap, j+1))))
            then ( (* promote right child *)
                   sub(!heap, i) := !(sub(!heap, j+1));
                   downheap(ev, j+1)
                 )
            else ( (* promote left child *)
                   sub(!heap, i) := !(sub(!heap,j)) ;
                   downheap(ev, j)
                 )
     )
  end;

fun insertevent(t, co) =
( priqn := !priqn+1;        (* Increment number of events *)
(*
  wrs "insertevent: for time: "; wri(t, 0); nl();
*)
  upheap(Event(t, co), !priqn)
);

fun getevent() =
(* Extract the earliest event returning
    a CoVal,  ValCo if there is an event
           or ValNull if not
*)
  let val event = !(sub(!heap, 1))      (* Get the earliest event *)
      and last  = !(sub(!heap, !priqn)) (* Get the highest element *)
(*writef("getevent: priq:") *)
(*prq() *)
  in
    if event=Empty
    then Empty     (* No events in the priority queue *)
    else ( priqn := !priqn-1;      (* Decrement the heap size *)
           downheap(last, 1);      (* Re-insert last event *)
           simtime := eventTime event;
           event
         )
  end;

fun waitfor ticks =
( (* Put an event item into the priority queue *)
(*
wrs "waitfor: simtime="; wri(!simtime, 8); wrs " ticks="; wri(ticks, 4); nl();
*)

  (* Insert into the priority queue *)
  insertevent(!simtime+ticks, !currco);

  (* Wait for the specified number of ticks *)
  cowait ValNull
)

end;


(* ######################## Coroutine Bodies ########################## *)


fun stopcofn (ValInt t) =
  ( Pri.waitfor t; (* suspend this coroutine for t ticks *)

    if !tracing
    then ( wri(!simtime, 8); wrs ": Stop time reached\n" )
    else ();

    ValNull
  )
| stopcofn _ = ValNull;
 
fun messcofn (ValInt n) =
  let val node = ref n
  in (* cowait(ValNull); *)

     Wkq.qitem(!node);   (* Put the message on the work queue for this node *)

  (  while true do
     ( (* Start processing the first message *)

       let val prtime   = Rnd.next(!ptmax);     (* a random processing time *)
           val dest     = Rnd.next(!nodes) + 1; (* a random destination node *)
           val netdelay = abs(!node-dest)       (* the network delay *)
       in
       (
          (* wrs "prtime="; wri(prtime,3); wrs "  "; wri(dest, 0); nl(); *)
          if !tracing then
          ( wri(!simtime, 8); wrs ": node "; wri(!node, 4);
            wrs ": processing message until "; wri(!simtime+prtime, 0); nl()
          )
          else ();

          Pri.waitfor prtime;
          count := !count + 1; (* One more message processed *)

          if !tracing then
          ( wri(!simtime, 8); wrs ": node "; wri(!node, 4);
            wrs ": message processed\n"
          )
          else ();

          (* Wkq.prwaitq(!node); *)
          (* De-queue current item and activate the next, if any *)
          Wkq.dqitem(!node);
          (* Wkq.prwaitq(!node); *)

          if(!tracing) then
          ( wri(!simtime, 8); wrs ": node "; wri(!node, 4);
            wrs ": sending message to node "; wri(dest, 0);
            wrs " to arrive at "; wri(!simtime+netdelay, 0); nl()
          )
          else ();

          Pri.waitfor netdelay;
          node := dest;  (* The message has arrived at the destination node *)

          if(!tracing)
          then ( wri(!simtime, 8); wrs ": node "; wri(!node, 4);
                 wrs ": message reached this node\n"
               )
          else ();
          Wkq.qitem(!node)   (* Queue the message if necessary *)

        )
        end
        (* The node can now process the first message on its work queue *)
     );
     ValNull
  )     
end

|   messcofn _ = ( wrs "messcofn called with bad arg\n"; ValNull);


fun mkmesscorts(i, n) =
  if i > n
  then ()
  else let val co = createco messcofn
       in ( (* wrs "creating messco: "; wri(i, 3); wrs " n="; wri(n, 0); nl();*)
            callco(co, ValInt i);
            (* Pri.insertevent(0, co); *)
            mkmesscorts(i+1, n)
          )
       end;

fun cosimfn (arg:CoVal) = 
  let val stopco =  createco stopcofn
  in (
       (* Initialise the Priority Queue *)
       Pri.init(!nodes + 1+100);

       (* Initialise the node work queues *)
       Wkq.init(!nodes);


       (* wrs "cosim: created stop coroutine\n"; *)
       callco(stopco, ValInt(!stoptime));

       if !tracing
       then ( wri(!simtime, 8); wrs ": Starting simulation\n" )
       else ();

       (* make the message coroutines and put events
          in the priority queue to activate each at time 0
       *)
       mkmesscorts(1, !nodes);

       simtime := 0;
       count := 0;  (* Count of message processed *)

       (* Run the event loop *)

       while !simtime <= !stoptime do
         let val ev = Pri.getevent()
         in
            case ev of
              Pri.Empty => simtime := !stoptime + 1
            | Pri.Event(t, co) =>
                (
(*
                  if !tracing
                  then ( wri(!simtime, 8); wrs ": calling co\n" )
                  else ();
*)
                  if !simtime <= !stoptime
                  then ( callco(co, ValNull); () )
                  else ()
                )
         end;

       if !tracing
       then wrs "\nSimulation stopped\n\n"
       else ();

       wrs "Messages processed: "; wri(!count, 0); nl();

       wrs "\nCobench done\n";
       ValNull
     )
  end;

(* xx


*)


fun currtime() = Time.now();

fun atoi s = case (Int.fromString s) of
               SOME k => k
             | NONE   => 0;

fun processargs [] = ()
  | processargs [s]  =
      if s="-t" then tracing := true else ()
  | processargs (s::arg::rest)  =
      if s="-t" then (tracing := true; processargs (arg::rest))
      else if s="-n" then (nodes    := atoi arg;  processargs rest)
      else if s="-s" then (stoptime := atoi arg;  processargs rest)
      else if s="-p" then (ptmax    := atoi arg;  processargs rest)
      else if s="-r" then (seed     := atoi arg;  processargs rest)
      else processargs rest;

fun rdargs() =
  let val alist = getArgs()
  in case alist of
     [] => ()
     | _::rest => processargs rest
  end;

fun main() =
(
  nodes    := 500;      (* Set the default values *)
  stoptime := 1000000;
  ptmax    := 1000;
  tracing  := false;


  rdargs();

  wrs "\nCosim entered\n\n";
  wrs "Network nodes:       "; wri(!nodes,    10); nl();
  wrs "Stop time:           "; wri(!stoptime, 10); nl();
  wrs "Max processing time: "; wri(!ptmax,    10); nl();
  wrs "Random number seed:  "; wri(!seed,     10); nl();
  nl();

Rnd.init(!seed);
(*
nl();
Rnd.test 100;
*)
  let val rootco = createco cosimfn;
      val t0 = currtime()
  in  callco(rootco, ValNull);  (* start the root coroutine *)
      wrs "\nTime taken is ";
      wrs (Time.toString(Time.-(currtime(),t0))); nl(); nl()
  end
  handle CoErr x => (wrs "CoErr: "; wrs x; wrs "\nCobench failed\n\n")
);


(* Original BCPL code


*)

main() handle CoErr x => (wrs "CoErr: "; wrs x; nl())
