// A Discrete event simulator benchmark 
// Translated into Java by Martin Richards (c) June 2004 

// This is a benchmark test for a discrete event simulator using
// BCPL style coroutines. It simulates a network of n nodes which
// each receive, queue, process and transmit messages to other nodes.
// The nodes are uniformly spaced on a straight line and the network
// delay is assumed to be proportional to the linear distance between
// the source and the destination. On arrival, if the receiving node is
// busy the message is queued, otherwise it is processed immediately.
// After processing the message for random time it is sent to another
// random node. If the node has a non empty queue it dequeues its first
// message and starts to process it, otherwise it becomes suspended.
// Initially every node is processing a message and every queue is
// empty. There are n coroutines to simulate the  progress of each
// message and the discrete event priority queue is implemented using
// the heapsort heap structure. The simulation stops at a specified
// simulated time. The result is the number of messages that have been
// processed. A machine independent random number generator is used
// so the resulting value should be indepent of implementation
// language and machine being used.


import Cortn.*;

public class Cosim
{
    public static int seed     = 0;
    public static int nodes    = 500;
    public static int stoptime = 1000000;
    public static int ptmax    = 1000;
    public static boolean tracing = false;

    public static void main(String[] args)
    {
	try {

	    for (int i = 0; i < args.length; i++) {
		// System.out.println("arg[i] = " + args[i]);
		if (args[i].equals("-n")) {
		    nodes = Integer.parseInt(args[++i]);
		    continue;
		}
		
		if (args[i].equals("-s")) {
		    stoptime = Integer.parseInt(args[++i]);
		    continue;
		}

		if (args[i].equals("-p")) {
		    ptmax = Integer.parseInt(args[++i]);
		    continue;
		}

		if (args[i].equals("-r")) {
		    seed = Integer.parseInt(args[++i]);
		    continue;
		}

		if (args[i].equals("-t")) {
		    tracing = true;
		    continue;
		}
	    }
	    
	} 
	catch (NumberFormatException e) {
	    System.err.println("Integer argument expected");
	}

        System.out.println("\nNetwork nodes:       " + nodes);
        System.out.println("Stop time:           " + stoptime);
        System.out.println("Max processing time: " + ptmax);
        System.out.println("Random number seed:  " + seed+"\n");

	// Create and start the root coroutine
	Cortn netsim = new Netsim("netsim");
	System.exit(0);
    }
}

abstract class SimCortn extends Cortn {
    public static int nodes;
    public static int stoptime;
    public static int ptmax;
    public static int seed;
    public static boolean tracing;

    public static Wqitem wkqv[];      // The vector of work queues

    public static int    count;       // count of messages processed
    public static int    simtime;     // Simulated time

    SimCortn(String name) {
        super(name);
    }

    public void waitfor(int time) {
       // Make an event item into the priority queue
       //currco = this;
       Pqitem event = new Pqitem(time, currco);
       //System.out.println("waitfor: time="+time+" co="+currco.coName);
       Priq.insertevent(event); // Insert into the priority queue
       cowait(null);            // Wait for the specified number of ticks
    }

    public void prwaitq(int node) {
      Wqitem p = wkqv[node];
      if (p==null) {
          System.out.println("wkq for node "+node+": p=null");
          return;
      }
      if (p.flag) {
          System.out.println("wkq for node "+node+": p.flag=true");
          return;
      }
      System.out.println("wkq for node "+node+": ");
      while (p!=null) {
          System.out.println(" "+((Cortn)p.co).coName);
          p = p.next;
      }
      System.out.println("");
    }

    public void qitem(int node) 
    // The message has reached this node
    // If currently not busy, mark as busy and return to process
    // the message, other append it to the end of the work queue
    // for this node.

    { // Make a queue item
      Wqitem item = new Wqitem(null, this);
      //System.out.println("qitem: entered, node="+node);
      Wqitem p = wkqv[node];
      //prwaitq(node);
      if (p==null)
      { // The node was not busy
        wkqv[node] = new Wqitem();  // Mark node as busy
        if (tracing)
          System.out.println(""+simtime+": node "+node+": node not busy");
        //System.out.println("qitem: wkqv["+node+"]="+wkqv[node]);
	//prwaitq(node);
        return;
      }
      // Append item to the end of this queue
      if (tracing)
        System.out.println(""+simtime+
                           ": node "+node+
                           ": busy so appending message to end of work queue");
      if (p.flag) {
               wkqv[node] = item;          // Form a unit list
      } else { while (p.next!=null) p = p.next;  // Find the end of the wkq
               p.next = item;              // Append to end of wkq
             }
      //prwaitq(node);
      cowait(null); // Wait to be activated (by dqitem)
    }

    public void dqitem(int node)
    // A message has just been processed by this node and is ready to process
    // the next, if any.
    { Wqitem item = wkqv[node]; // Current item (~=null)
      //System.out.println("dqitem("+node+"): entered, item="+item);
      if (item==null)
      { System.out.println("No item to de-queue\n");
        return;
      }
      //prwaitq(node);
      if (item.flag) {
               wkqv[node] = null;                // The node is no longer busy
      } else { Wqitem next = item.next;
               Cortn co = (Cortn)item.co;
               wkqv[node] = next!=null ? next : new Wqitem(); // De-queue the item
               //prwaitq(node);
               callco(co, null);                // Process the next message
             }
      }
}

class Netsim extends SimCortn
{   // This is the root coroutine.
 
    public Cortn stopco;

    public Netsim(String name) {
	super(name);
    }

    public Object fn(Object c) {
	// Body of the root coroutine.
        nodes = Cosim.nodes;
        stoptime = Cosim.stoptime;
        ptmax = Cosim.ptmax;
        seed = Cosim.seed;
        tracing = Cosim.tracing;

        Priq.init();
        Rnd.init();

        wkqv = new Wqitem[nodes+1];

        count = 0;    // Count of message processed
        simtime = 0;  // Simulated time
        //Rnd.test();   // Test the random number generator
        //System.out.println("rdn(10000)="+Rnd.next(10000));

        if (tracing)
	    System.out.println("\n"+simtime+": Starting simulation");

        //System.out.println("Creating stopco");
	stopco = new Stopco("stopco");
        //System.out.println("Coroutine: "+stopco.coName+" created");
        callco(stopco, new Integer(Cosim.stoptime));

        for(int i = 1; i<= nodes; i++) {
	    Cortn co = new Messageco("messageco_"+i);
            //System.out.println("Coroutine: "+co.coName+" created");
	    callco(co, new Integer(i));
	}

	if (tracing)
	    System.out.println("All coroutines created");

        // Run the event loop

        while(true)
        { Pqitem event = Priq.getevent();      // Get the earliest event
          if (event==null) break;
          simtime = event.time;          // Set the simulated time
          if (simtime > stoptime) break;
	  //System.out.println(""+simtime+"callco: "+
	  //	     coName+" => "+event.co.coName);
          callco(event.co, null);
        }

        if (tracing) System.out.println("\nSimulation stopped\n");
        System.out.println("\nMessages processed: "+count);

	System.exit(0);
	return null;
    }
}

// The following random number generator is based on one give
// in Knuth: The art of programming, vol 2, p 26.

class Rnd {
     // A vector used by the random number generator
     public static int[] v;
     public static int i;        // subscripts of v
     public static int j;

     public static void init() {
      int a=0x2345678+Cosim.seed;
      int b=0x5362781;
      v = new int[55];
      for(int k=0; k<=54; k++)
      { int t = (a+b) & 0xFFFFFFF;
        a = b;
        b = t;
        v[k] = t;
	//System.out.println("rnd: v["+k+"] = "+t);
      }
      i = 55-55;
      j = 55-24;
    }

    public static int next(int n) {
      int val = (v[i] + v[j]) & 0xFFFFFFF;
      //System.out.println("rnd: next => "+val%n+" i="+i+" j="+j);
      v[i] = val;
      i = (i + 1) % 55;
      j = (j + 1) % 55;
      return val % n;
    }

    public static void test() {
       System.out.println("");
       for (int k=1; k<=100; k++) // Test the random number generator
       { int val = next(10000);
         if (val<1000) System.out.print(" ");
         if (val<100)  System.out.print(" ");
         if (val<10)   System.out.print(" ");
         System.out.print(" "+ val);
         if (k % 10 == 0) System.out.println("");
       }
       System.out.println("");
       System.exit(0);
    }
}

class Messageco extends SimCortn
{
    public Messageco(String name) {
	super(name);
    }

    public Object fn(Object arg) {
	int node = ((Integer)arg).intValue();

        qitem(node);   // Put the message on the work queue for this node

        while(true)
        { // Start processing the first message at node: node.
          int prtime   = Rnd.next(ptmax);      // a random processing time
          int dest     = Rnd.next(nodes) + 1;  // a random destination node
          int netdelay = Math.abs(node-dest);  // the network delay
          //System.out.println("prtime="+prtime+" dest="+dest);
          if (tracing)
             System.out.println(""+simtime+
                       ": node "+node+
                       ": processing message until "+ (simtime+prtime));
          waitfor(simtime+prtime);
          count++;       // One more message processed
          if (tracing)
            System.out.println(""+simtime+
                               ": node "+node+": message processed");

          //prwaitq(node);
          dqitem(node); // De-queue current item and activate the next, if any
          //prwaitq(node);
          if(tracing)
             System.out.println(""+simtime+
                                ": node "+node+
                                ": sending message to node "+dest+
                                " to arrive at "+(simtime+netdelay));

          waitfor(simtime+netdelay);
          node = dest;      // The message has arrived at the destination node
          if(tracing)
            System.out.println(""+simtime+
                               ": node "+node+
                               ": message reached this node");
          qitem(node);   // Queue the message if necessary
          // The node can now process the first message on its work queue
       }
    }
}

class Stopco extends SimCortn
{
    public Stopco(String name) {
	super(name);
    }

    public Object fn(Object arg) {
	int stoptime = ((Integer)arg).intValue();
        waitfor(stoptime);
        if(Cosim.tracing)
          System.out.println(""+stoptime+": Stop time reached");
        return null;
    }
}

class Pqitem {
    public int time;
    public Cortn co;

    Pqitem(int time, Cortn co) {
	this.time = time;
        this.co = co;
    }
}

class Wqitem {
    public boolean flag;
    public Wqitem next;
    public Object co;

    Wqitem(Wqitem next, Object co) {
        this.flag = false;
	this.next = next;
        this.co = co;
    }

    Wqitem() {
	this.flag = true;
    }

}

class Priq {
    public static Pqitem priq[];   // The vector holding the priority queue
    public static int priqupb;     // The upper bound
    public static int priqn;       // Number of items in the priority queue

    public static void init() {
        priqupb = Cosim.nodes+1; //+1 for the stopco event
	priq = new Pqitem[priqupb+1];
        priqn = 0;
    }

    public static void prq() {
        System.out.print("priq: priqn="+priqn+": ");
        for(int i = 1; i<=priqn; i++)
            System.out.print(" " + priq[i].time);
        System.out.println("");
    }

    public static void insertevent(Pqitem event) {
	// No need to be synchronized since only one coroutine can
	// run at a time.
        priqn++;                // Increment number of events
        //System.out.println("insertevent: at time: "+ event.time);
        upheap(event, priqn);
    }

    public static void upheap(Pqitem event, int i) {
        int eventtime = event.time;
        //System.out.println("upheap: eventtime="+eventtime+" i="+i);

        while(true)
        { int p = i/2;          // Parent of i
          if(p==0 || eventtime >= priq[p].time)
          { priq[i] = event;
	    //prq();
            return;
          }
          priq[i] = priq[p];     // Demote the parent
          //prq();
          i = p;
      }
    }

    public static void downheap(Pqitem event, int i) {
        while(true)
        { int j = 2*i;  // j is left child, if present
	  //System.out.println("downheap: eventtime="+event.time+" i="+i);
	  //prq();
          if (j > priqn)
          { upheap(event, i);
            return;
          }
          int min = priq[j].time;
          // Look at other child, if it exists
          if (j<priqn && min>priq[j+1].time) j++;
          // promote earlier child
          priq[i] = priq[j];
          i = j;
        }
    }

    public static Pqitem getevent() {
	// No need to be synchronized since only one coroutine can
	// run at a time.
       Pqitem event = priq[1];      // Get the earliest event
       Pqitem last  = priq[priqn];  // Get the event at the end of the heap
       //System.out.print("getevent: priq: priqn="+priqn+" ");
       //prq();
       if (priqn<=0) return null;      // No events in the priority queue
       priqn--;                        // Decrement the heap size
       if (priqn>0) downheap(last, 1); // Re-insert last event
       return event;
    }
}



