// Implementation of BCPL style coroutines in Java
// with help from Eben Upton
// by Martin Richards (c) April 2005

package Cortn;

public abstract class Cortn extends Thread {
    // BCPL style coroutines each have their own runtime stack
    // so must be implemented as threads in Java.
    public String coName;

    // All coroutines have access to currco, the current coroutine.
    public static Cortn currco = null;  // The current coroutine

    // Each coroutine has a parent coroutine, ecept for the
    // root coroutine.
    protected Cortn parent; // This coroutine's parent
    // val holds the value passed between coroutines.
    protected Object val;   // The value given to this coroutine
                            // when it was last given control

    public Cortn(String name) {
    	setName("T_"+name); // Set the thread name
	coName = name;      // Set the coroutine name

	Cortn co = currco;
	currco = this;
	parent = co;
	start();
	// Note that the call of start() causes the current thread
	// to complete any pending assignment to main memory before
	// the new thread gains control. The new thread will thus
	// be able to read variables such as currco and parent
	// correctly.

	if (co==null) {
	    // The root coroutine has just been started,
	    // so wait for it to finish.

	    //System.out.println(coName+
	    //                   ": Wait for the root coroutine to finish");
	    try { join(); }        // Wait for the root coroutine to finish
	    catch(Exception e) {}
	    //System.out.println(coName+": This root coroutine has finished");

	} else {
	    // A non-root coroutine has been started,
	    // so wait until control returns to us.
	    // This will happen in the cowait of the
	    // while (true) c = fn(cowait(c)); loop
	    co.currcoWait();
	}
    }

    public void run() {
	// If parent is null, we are the root coroutine,
	// so just call fn.

	if (parent==null) {
	    // We are running as the root coroutine.
	    fn(null);
	} else {
	    // We are a non-root coroutine, so just enter the
	    // normal loop. This immediately suspends itself
	    // in cowait.

	    Object c = null;
	    while (true) c = fn(cowait(c));
	}
    }

    public abstract Object fn(Object c);

    // conventional coroutine API

    public Object callco(Cortn target, Object arg) {
	// Note that there is no need for this to be a
	// synchronized method. The necessary synchronization
	// is performed by the calls of wait and currcoWait.
	target.parent = this;
	target.val = arg;
	currco = target;

	// Transfer control to the target coroutine.
	// Java will ensure that the main memory locations
	// holding target.parent, target.val and currco are
	// properly updated before control is given to the
	//target.
	target.unwait();

	// Wait until we are the current coroutine
	currcoWait();

	return val;
    }

    public Object resumeco(Cortn target, Object arg) {
	// Note that there is no need for this to be
	// asynchronized method.
	Cortn p = parent;   // Ensure that resumeco can resume itself
	parent = null;      // ie when target==currco
	target.parent = p;
	target.val = arg;
	currco = target;

	System.out.println(coName+": resumeco called => "+target.coName);

	// Transfer control to the target coroutine.
	target.unwait();
	// Wait until we are the current coroutine
	currcoWait();

	return val;
    }

    public Object cowait(Object arg) {
	// Note that there is no need for this to be
	// asynchronized method.
	Cortn target = parent;

	parent = null;
	target.val = arg;
	currco = target;

	// Transfer control to the target coroutine.
	target.unwait();
	// Wait until we are again the current coroutine
	currcoWait();

	return val;
    }

    // Internal support functions

    public synchronized void currcoWait() {
	// This method is called in the thread of the current
	// coroutine, so only runs when it has obtained the
	// current coroutine's lock. To obtain the lock the
	// thread's private variables (typically held in
	// machine registers) are flushed so that currco will
	// be fetched from main memory.
	// wait releases the lock but when the thread suspends
	// itself, but will regain the lock when it resumes
	// execution, so each time round the loop currco
	// must be reloaded from main memory.
	// currco is updated by callco, resumeco and cowait
	// and is flushed to memory along with parent and val
	// when the calling coroutine next obtains a lock in
	// the call: target.unwait(). These quantities are
	// thus properly passed from the calling coroutine's
	// thread to the target.
	try { while (currco!=this) {
		wait();
	    }
	}
	catch(Exception e) {}
    }

    public synchronized void unwait() {
	// Release all this object's threads.
	// Note that there may be threads other than the waiting
	// coroutine, so notify() alone is not sufficient.
	// If in doubt, try it.
	notifyAll();
    }
}
