// A Discrete event simulator benchmark 
// Designed and implemented by Martin Richards (c) June 2004 

// This is a benchmark test for a discrete event simulator using
// BCPL style coroutines. It simulates a network of n nodes which
// each receive, queue, process and transmit messages to other nodes.
// The nodes are uniformly spaced on a straight line and the network
// delay is assumed to be proportional to the linear distance between
// the source and the destination. On arrival, if the receiving node is
// busy the message is queued, otherwise it is processed immediately.
// After processing the message for random time it is sent to another
// random node. If the node has a non empty queue it dequeues its first
// message and starts to process it, otherwise it becomes suspended.
// Initially every node is processing a message and every queue is
// empty. There are n coroutines to simulate the  progress of each
// message and the discrete event priority queue is implemented using
// the heapsort heap structure. The simulation stops at a specified
// simulated time. The result is the number of messages that have been
// processed. A machine independent random number generator is used
// so the resulting value should be indepent of implementation
// language and machine being used.

#include <stdio.h>
#include <stdlib.h>
#include "cohdr.h"

typedef struct pqitem {
  int time;
  cortn co;
} pqitem;

typedef struct wqitem {
  struct wqitem *next;
  cortn co;
} wqitem;

pqitem **priq;      // The vector holding the priority queue
int     priqupb;    // The upper bound
int     priqn;      // Number of items in the priority queue
wqitem **wkqv;      // The vector of work queues
int    count;       // count of messages processed
int    nodes;       // The number of nodes
int    ptmax;       // The maximum processing time
cortn  stopco;      // The stop coroutine
cortn  *cov;        // Vector of message coroutines
int    ranv[55];    // A vector used by the random number generator
int    rani;        // subscripts of ranv
int    ranj;
int    simtime;     // Simulated time
int    stoptime;    // Time to stop the simulation
int    tracing=0;   // Tracing flag

// Functions
int rdn(void);
int initrnd(int seed);
void closernd(void);
void prq(void);
void insertevent(pqitem*);
void upheap(pqitem*, int i);
void downheap(pqitem*, int i);
pqitem *getevent(void);
void waitfor(int delay);
void prwaitq(int node);
void qitem(int node);
void dqitem(int node);
coval stopcofn(coval arg);
coval messcofn(coval arg);


// ################### Random number generator #######################

// The following random number generator is based on one give
// in Knuth: The art of programming, vol 2, p 26.
int rnd(int n)
{ int val = (ranv[rani] + ranv[ranj]) & 0xFFFFFFF;
  ranv[rani] = val;
  rani = (rani + 1) % 55;
  ranj = (ranj + 1) % 55;
  //printf("rnd: val=%8x  res=%4i\n", val, val%n);
  return val % n;
}

int initrnd(int seed)
{ int a=0x2345678+seed;
  int b=0x5362781;
  int i;
  for(i=0; i<=54; i++)
  { int t = (a+b) & 0xFFFFFFF;
    a = b;
    b = t;
    ranv[i] = t;
  }
  rani = 55-55;  // ie 0
  ranj = 55-24;  // ie 31
  return 1;
}

void closernd() { return; }

// ################### Priority Queue functions ######################

void prq(void)
{ int i;
  for(i = 1; i<=priqn; i++) printf(" %4d", priq[i]->time);
  putchar('\n');
}

void insertevent(pqitem *event)
{ priqn = priqn+1;        // Increment number of events
//printf("insertevent: at time: %d  co=%d\n", event->time, (int)event->co);
  upheap(event, priqn);
}

void upheap(pqitem *event, int i)
{ int eventtime = event->time;
//printf("upheap: eventtime=%d i=%d\n", eventtime, i);

 while(1)
  { int p = i/2;          // Parent of i
    if(p==0 || eventtime >= priq[p]->time)
    { priq[i] = event;
//prq();
      return;
    }
    priq[i] = priq[p];     // Demote the parent
//prq();
    i = p;
  }
}

void downheap(pqitem *event, int i)
{ while(1)
  { int j = 2*i;  // j is left child, if present
    int min;
//printf("downheap: eventtime=%d i=%d\n", event!0, i)
//prq()
    if (j > priqn)
    { upheap(event, i);
      return;
    }
    min = priq[j]->time;
  // Look at other child, if it exists
   if (j<priqn && min>priq[j+1]->time) j++;
    // promote earlier child
    priq[i] = priq[j];
    i = j;
  }
}

pqitem *getevent(void)
{ pqitem *event = priq[1];      // Get the earliest event
  pqitem *last  = priq[priqn];  // Get the event at the end of the heap
//printf("getevent: priq:")
//prq()
  if (priqn<=0) return 0;       // No events in the priority queue
  priqn--;                      // Decrement the heap size
  downheap(last, 1);            // Re-insert last event
  return event;
}

void waitfor(int ticks)
{ // Make an event item into the priority queue
  pqitem item = {simtime+ticks, currco};
  //printf("waitfor: simtime=%d ticks=%d\n", simtime, item.time);
  insertevent(&item);     // Insert into the priority queue
  cowait(0);                   // Wait for the specified number of ticks
}

// ###################### Queueing functions #########################

void prwaitq(int node)
{ wqitem *p = wkqv[node];
//abort(997)
  if ((int)p==-1 || p==0) { printf("wkq for node %d: %d\n", node, (int)p); return; }
  printf("wkq for node %d:", node);
  while (p)
  { printf(" %d", (int)p->co);
    p = p->next;
  }
  putchar('\n');
}

void qitem(int node)
// The message has reached this node
// I currently not busy, mark as busy and return to process
// the message, other append it to the end of the work queue
// for this node.
{ // Make a queue item
  wqitem item = {0, currco};
  wqitem *p = wkqv[node];
//printf("qitem: entered\n")
//prwaitq(node)
  if (p==0)
  { // The node was not busy
    wkqv[node] = (wqitem*)-1;  // Mark node as busy
    if(tracing)
      printf("%8d: node %4d: node not busy\n", simtime, node);
//printf("qitem: wkqv[%d=%d\n", node, wkqv[node)
    //prwaitq(node)
//abort(998)
    return;
  }
  // Append item to the end of this queue
//abort(1000)
  if(tracing)
    printf("%8d: node %4d: busy so appending message to end of work queue\n",
	   simtime, node);
//abort(1000)
  if ((int)p == -1) {
           wkqv[node] = &item;           // Form a unit list
  } else { while (p->next) p = p->next;  // Find the end of the wkq
           p->next = &item;              // Append to end of wkq
         }
  //prwaitq(node);
  cowait(0); // Wait to be activated (by dqitem)
}

void dqitem(node)
// A message has just been processed by this node and is ready to process
// the next, if any.
{ wqitem *item = wkqv[node]; // Current item (~=0)
//printf("dqitem(%d): entered, item=%d\n", node, item)
  if (item==0)
  { printf("No item to de-queue\n");
    return;
  }
//prwaitq(node)
  if ((int)item==-1) {
           wkqv[node] = 0;                  // The node is no longer busy
  } else { wqitem *next = item->next;
           cortn co = item->co;
           wkqv[node] = next ? next : (wqitem*)-1; // De-queue the item
//prwaitq(node);
           callco(co, 0);                       // Process the next message
         }
}

// ######################## Coroutine Bodies ##########################

coval stopcofn(coval arg)
{ waitfor(stoptime);
  if(tracing)
    printf("%8d: Stop time reached\n", simtime);
  return 0;
}
 
coval messcofn(coval arg)
{ int node = (int)arg;
  qitem(node);   // Put the message on the work queue for this node

 while(1)
  { // Start processing the first message
    int prtime   = rnd(ptmax);     // a random processing time
    int dest     = rnd(nodes) + 1; // a random destination node
    int netdelay = abs(node-dest); // the network delay
    //printf("prtime=%3d dest=%3d\n", prtime, dest);
//abort(1001)
    if(tracing)
      printf("%8d: node %4d: processing message until %d\n",
	      simtime, node, simtime+prtime);
    waitfor(prtime);
    count++;       // One more message processed
    if(tracing)
      printf("%8d: node %4d: message processed\n",
	     simtime, node, dest, simtime+netdelay);

//prwaitq(node);
    dqitem(node); // De-queue current item and activate the next, if any
//prwaitq(node)
    if(tracing)
      printf("%8d: node %4d: sending message to node %d to arrive at %d\n",
	     simtime, node, dest, simtime+netdelay);

    waitfor(netdelay);
    node = dest;      // The message has arrived at the destination node
    if(tracing)
      printf("%8d: node %4d: message reached this node\n",
	     simtime, node);
    qitem(node);   // Queue the message if necessary
    // The node can now process the first message on its work queue
  }
}
// ######################### Main Program ############################

coval rootfn(coval);

int main(int argc, char *argv[])
{ cortn root = createco(rootfn, 3000);
  int seed = 0;
  int i;

  nodes    = 500;
  stoptime = 1000000;
  ptmax    = 1000;

  for(i=1; i<argc; i++) {
    if     (strcmp(argv[i], "-n")==0)  nodes = atoi(argv[++i]);
    else if(strcmp(argv[i], "-s")==0)  stoptime = atoi(argv[++i]);
    else if(strcmp(argv[i], "-p")==0)  ptmax = atoi(argv[++i]);
    else if(strcmp(argv[i], "-r")==0)  seed = atoi(argv[++i]);
    else if(strcmp(argv[i], "-t")==0)  tracing = 1;
    else { printf("Bad arguments for cosim\n"); exit(0); }
  }
  printf("\nCosim entered\n\n");
  printf("Network nodes:       %d\n", nodes);
  printf("Stop time:           %d\n", stoptime);
  printf("Max processing time: %d\n", ptmax);
  printf("Random number seed:  %d\n", seed);
  putchar('\n');

  if (initrnd(seed)==0)
  { printf("Can't initialise the random number generator\n");
    return 0;
  }

if(0)
  for (i=1; i<=100; i++) // Test the random number generator
  { printf(" %4d", rnd(10000));
    if (i % 10 == 0) putchar('\n');
  }

  callco(root, 0);

  return 0;
}

coval rootfn(coval x)
{ int i;
  stopco = 0;
  wkqv = (wqitem**)malloc(sizeof(wqitem*) * (nodes+1));
  priq = (pqitem**)malloc(sizeof(pqitem*) * (nodes+2));
  cov  = (cortn*)  malloc(sizeof(cortn)   * (nodes+1));

  if (wkqv==0 || priq==0 || cov==0)
  { printf("Can't allocate space for the node work queues\n");
    goto ret;
  }

  for(i = 1; i<=nodes; i++) { wkqv[i] = 0;  cov[i] = 0; }
  priqn = 0;  // Number of events in the priority queue
  count = 0;  // Count of message processed
  simtime = 0; // Simulated time

  if(tracing) printf("%8d: Starting simulation\n", simtime);
  //printf("rdn(10000)=%d\n", rnd(10000));

  //printf("creating stopco\n");
  // Create and start the stop coroutine
  stopco = createco(stopcofn, 2000);
  if (stopco) callco(stopco, 0);

  // Create and start the message coroutines
  for(i = 1; i<=nodes; i++)
  { cortn co = createco(messcofn, 2000);
//printf("creating message coroutine %d\n", i);
    if (co) callco(co, (coval)i);
    cov[i] = co;
  }

  // Run the event loop

  while(1)
  { pqitem *event = getevent();      // Get the earliest event
    if (event==0) break;
    simtime = event->time;          // Set the simulated time
    //if(tracing) printf("%8d: calling co=%d\n", simtime, event!1)
   if (simtime > stoptime) break;
    callco(event->co, 0);
  }

  if(tracing) printf("\nSimulation stopped\n\n");
  printf("Messages processed: %d\n", count);

ret:
  for(i = nodes; i>=1; i--) if (cov[i]) deleteco(cov[i]);
//  if (cov) free(cov);
//  if (wkqv) free(wkqv);
//  if (priq) free(priq);
//  if (stopco) deleteco(stopco);
  closernd();
  return 0;

fail:
  printf("Unable to initialise the simulator\n");
  goto ret;
}

