// A Discrete event simulation benchmark 
// Designed and implemented by Martin Richards (c) June 2004 
// Translated into MCPL April 2005

// This is a benchmark test for a discrete event simulator using
// BCPL style coroutines. It simulates a network of n nodes which
// each receive, queue, process and transmit messages to other nodes.
// The nodes are uniformly spaced on a straight line and the network
// delay is assumed to be proportional to the linear distance between
// the source and the destination. On arrival, if the receiving node is
// busy the message is queued, otherwise it is processed immediately.
// After processing the message for random time it is sent to another
// random node. If the node has a non empty queue it dequeues its first
// message and starts to process it, otherwise it becomes suspended.
// Initially every node is processing a message and every queue is
// empty. There are n coroutines to simulate the  progress of each
// message and the discrete event priority queue is implemented using
// the heapsort heap structure. The simulation stops at a specified
// simulated time. The result is the number of messages that have been
// processed. A machine independent random number generator is used
// so the resulting value should be indepent of implementation
// language and machine being used.

MODULE cosim

GET "mcpl.h"

GLOBAL
       priq:Ug,    // The vector holding the priority queue
       priqupb,    // The upper bound
       priqn,      // Number of items in the priority queue
       wkqv,       // The vector of work queues
       count,      // count of messages processed
       nodes,      // The number of nodes
       ptmax,      // The maximum processing time
       stopco,     // The stop coroutine
       cov,        // Vector of message coroutines
       ranv,       // A vector used by the random number generator
       rani, ranj, // subscripts of ranv
       simtime,    // Simulated time
       stoptime,   // Time to stop the simulation
       tracing,

// Functions
       rdn,
       initrnd,
       closernd,
       prq,
       insertevent,
       upheap,
       downheap,
       getevent,
       waitfor,
       prwaitq,
       qitem,
       dqitem,
       stopcofn,
       messcofn

// ################### Random number generator #######################

// The following random number generator is based on one give
// in Knuth: The art of programming, vol 2, p 26.
FUN rnd : n =>
  LET val = (ranv!rani + ranv!ranj) & #x_FFF_FFFF
  ranv!rani := val
  rani := (rani + 1) MOD 55
  ranj := (ranj + 1) MOD 55
  RETURN val MOD n
.

FUN initrnd : seed =>
  LET a = #x_234_5678+seed
  LET b = #x_536_2781
  ranv := getvec(54)
  UNLESS ranv RETURN FALSE
  FOR i = 0 TO 54 DO
  { LET t = (a+b) & #x_FFF_FFFF
    a := b
    b := t
    ranv!i := t
  }
  rani, ranj := 55-55, 55-24  // ie: 0, 31
  RETURN TRUE
.

FUN closernd : => IF ranv DO freevec ranv
.

// ################### Priority Queue functions ######################

FUN prq : =>
  FOR i = 1 TO priqn DO writef(" %4d", priq!i!0)
  newline()
.

FUN insertevent : event =>
  priqn++        // Increment number of events
    //writef("insertevent: at time: %d  co=%d\n", event!0, event!1)
  upheap(event, priqn)
.

FUN upheap : event, i =>
  LET eventtime = event!0
    //writef("upheap: eventtime=%d i=%d\n", eventtime, i)

  { LET p = i/2          // Parent of i
    UNLESS p AND eventtime < priq!p!0 DO
    { priq!i := event
      //prq()
      RETURN
    }
    priq!i := priq!p     // Demote the parent
    //prq()
    i := p
  } REPEAT
.

FUN downheap : event, i =>
{ LET j   = 2*i // j is left child, if present
  LET min = ?
//writef("downheap: eventtime=%d i=%d\n", event!0, i)
//prq()
  IF j > priqn DO
  { upheap(event, i)
    RETURN
  }
  min := priq!j!0
  // Look at other child, if it exists
  IF j<priqn AND min>priq!(j+1)!0 DO j := j+1
  // promote earlier child
  priq!i := priq!j
  i := j
} REPEAT
.

FUN getevent : =>
  LET event = priq!1      // Get the earliest event
  LET last  = priq!priqn  // Get the event at the end of the heap
//writef("getevent: priq:")
//prq()
  UNLESS priqn>0 RETURN 0 // No events in the priority queue
  priqn := priqn-1        // Decrement the heap size
  downheap(last, 1)       // Re-insert last event
  RETURN event
.

FUN waitfor : ticks =>
  // Make an event item into the priority queue
  LET eventtime=simtime+ticks,
      co=currco
  //writef("waitfor: simtime=%d ticks=%d\n", simtime, ticks)
  insertevent(@eventtime) // Insert into the priority queue
  cowait()                // Wait for the specified number of ticks
.

// ###################### Queueing functions #########################

FUN prwaitq : node =>
  LET p = wkqv!node
//abort(997)
  IF -1 <= p <= 0 DO { writef("wkq for node %d: %d\n", node, p); RETURN }
  writef("wkq for node %d:", node)
  WHILE p DO
  { writef(" %d", p!1)
    p := !p
  }
  newline()
.

FUN qitem : node =>
// The message has reached this node
// I currently not busy, mark as busy and return to process
// the message, other append it to the end of the work queue
// for this node.
  // Make a queue item
  LET link=0, co=currco
  LET p = wkqv!node
//writef("qitem: entered\n")
//prwaitq(node)
  UNLESS p DO
  { // The node was not busy
    wkqv!node := -1  // Mark node as busy
    IF tracing DO
      writef("%8d: node %4d: node not busy\n", simtime, node)
//writef("qitem: wkqv!%d=%d\n", node, wkqv!node)
    //prwaitq(node)
//abort(998)
    RETURN
  }
  // Append item to the end of this queue
//abort(1000)
  IF tracing DO
    writef("%8d: node %4d: busy so appending message to end of work queue\n",
            simtime, node)
//abort(1000)
  TEST p=-1
  THEN wkqv!node := @link     // Form a unit list
  ELSE { WHILE !p DO p := !p  // Find the end of the wkq
         !p := @link          // Append to end of wkq
       }
  //prwaitq(node)
  cowait() // Wait to be activated (by dqitem)
.

FUN dqitem : node =>
// A message has just been processed by this node and is ready to process
// the next, if any.
  LET item = wkqv!node // Current item (~=0)
  //writef("dqitem(%d): entered, item=%d\n", node, item)
  UNLESS item DO abort(999)
     //prwaitq(node)
  TEST item=-1
  THEN wkqv!node := 0                  // The node is no longer busy
  ELSE { LET next = item!0
         LET co   = item!1
         wkqv!node := next -> next, -1 // De-queue the item
	 //prwaitq(node)
         callco(co)                    // Process the next message
       }
.

// ######################## Coroutine Bodies ##########################

FUN stopcofn : arg =>
  waitfor(stoptime)
  IF tracing DO
    writef("%8d: Stop time reached\n", simtime)
  RETURN 0
.
 
FUN messcofn : node =>
  qitem(node)   // Put the message on the work queue for this node

  { // Start processing the first message
    LET prtime   = rnd(ptmax)     // a random processing time
    LET dest     = rnd(nodes) + 1 // a random destination node
    LET netdelay = ABS(node-dest) // the network delay

//writef("prtime=%i3 dest=%i3\n", prtime, dest)
//abort(1001)
    IF tracing DO
      writef("%8d: node %4d: processing message until %d\n",
              simtime, node, simtime+prtime)
    waitfor(prtime)
    count++        // One more message processed
    IF tracing DO
      writef("%8d: node %4d: message processed\n",
              simtime, node, dest, simtime+netdelay)
    //abort 1001
//prwaitq(node)
    dqitem(node) // De-queue current item and activate the next, if any
//prwaitq(node)
    IF tracing DO
      writef("%8d: node %4d: sending message to node %d to arrive at %d\n",
              simtime, node, dest, simtime+netdelay)

    waitfor(netdelay)
    node := dest      // The message has arrived at the destination node
    IF tracing DO
      writef("%8d: node %4d: message reached this node\n",
              simtime, node)
    qitem(node)   // Queue the message if necessary
    // The node can now process the first message on its work queue
  } REPEAT
.

// ######################### Main Program ############################

FUN start : =>
  LET seed = 0
  LET argv = VEC 50

  UNLESS rdargs("-n,-s,-p,-r,-t/S", argv, 50) DO
  { writef("Bad arguments for cosim\n")
    RETURN 0
  }

  nodes, stoptime, ptmax := 500, 1_000_000, 1000
  IF argv!0 AND string_to_number(argv!0) DO nodes    := result2 // -k
  IF argv!1 AND string_to_number(argv!1) DO stoptime := result2 // -s
  IF argv!2 AND string_to_number(argv!2) DO ptmax    := result2 // -p
  IF argv!3 AND string_to_number(argv!3) DO seed     := result2 // -r
  tracing := argv!4                                           // -t

  writef("\nCosim entered\n\n")
  writef("Network nodes:       %d\n", nodes)
  writef("Stop time:           %d\n", stoptime)
  writef("Max processing time: %d\n", ptmax)
  writef("Random number seed:  %d\n", seed)
  newline()

  UNLESS initrnd(seed) DO
  { writef("Can't initialise the random number generator\n")
    RETURN 0
  }

IF FALSE DO
  FOR i = 1 TO 100 DO // Test the random number generator
  { writef(" %4d", rnd(10000))
    IF i MOD 10 = 0 DO newline()
  }

  stopco := 0

  wkqv, priq, cov := getvec(nodes), getvec(nodes+1), getvec(nodes)
  UNLESS wkqv AND priq AND cov DO  
  { writef("Can't allocate space for the node work queues\n")
    IF wkqv   DO freevec(wkqv)
    IF priq   DO freevec(priq)
    IF cov    DO freevec(cov)
    closernd()
    RETURN 0
  }

  FOR i = 1 TO nodes DO wkqv!i, cov!i := 0, 0
  priqn := 0   // Number of events in the priority queue
  count := 0   // Count of message processed
  simtime := 0 // Simulated time

  IF tracing DO writef("%8d: Starting simulation\n", simtime)

     //writef("rnd(10000)=%d\n", rnd(10000))
  // Create and start the stop coroutine
  stopco := createco(stopcofn, 200)
  IF stopco DO callco(stopco)
  // Create and start the message coroutines
  FOR i = 1 TO nodes DO
  { LET co = createco(messcofn, 200)
    IF co DO callco(co, i)
    cov!i := co
  }

  // Run the event loop

  { LET event = getevent()      // Get the earliest event
    UNLESS event BREAK
    simtime := event!0          // Set the simulated time
      //IF tracing DO writef("%8d: calling co=%d\n", simtime, event!1)
    IF simtime > stoptime BREAK
    callco(event!1)
  } REPEAT

  IF tracing DO writef("\nSimulation stopped\n\n")
  writef("Messages processed: %d\n", count)

  FOR i = nodes TO 1 BY -1 IF cov!i DO deleteco(cov!i)
  IF cov    DO freevec(cov)
  IF wkqv   DO freevec(wkqv)
  IF priq   DO freevec(priq)
  IF stopco DO deleteco(stopco)
  closernd()
  RETURN 0
.

// Total number of Cintcode instructions executed: 435,363,350
// Number of coroutine changes:                      2,510,520
