/*
This is a benchmark test for TCP/IP communication under Cintpos
using the TCP Handler task. It is based on tcptest.b.

Written by Martin Richards (c) April 2003

The Cintpos command tcpbench has argument format:

      -n/k,-k/k,-s/k,-h/k,-t/s,master/s,slave/s

-n n    The number of blocks to transmit
-k k    The number of slave copy coroutines
-s s    The size of eack block in bytes
-h name The host name of the slave machine
-t      Cause tracing output to be generated
master  Create source, master copy and sink coroutines
slave   Create the slave copy coroutines

If neither master nor slave are specified both are assumed.  

Tchbench can be used to set up exercise TCP/IP communication links
between two machines, one called the master and the other the slave.

                MASTER m/c                SLAVE m/c

                source -----------------> 8000+1
                6000+1 <----------------- copy
                copy   -----------------> 8000+2
                6000+2 <----------------- copy
                copy   -----------------> 8000+3
                  .                         .
                  .                         .
                       -----------------> 8000+k
                6000+k <----------------- copy
                sink

On the master machine, the source coroutine generates a stream of n
blocks of k bytes sending them to port 8001 on the slave machine. The
copy coroutines copy data received on one port to a port on the other
machine, and sink just reads and discards data from its port.

If neither master nor slave are specified, they are both implied and
both the master and slave coroutines run on the same machine. So, for
example:

tcpbench -n 10 -k 50 -s 1000

will cause a stream of 10 blocks of 1000 bytes to be directed at port
8001 on the local machine. These will be copied back to port 6001 and
then to port 8002 until eventually the data reaches the sink coroutine
on port 6050 where it is discarded. When all the data has been
transferred, statistics of the run including the total time taken will
be output and the tcpbench command terminated.

To run the benchmark on two different machines xxx and yyy, say, first
setup a slave on machine xxx by typing for example:

tcpbench slave -k 50

This will just setup 50 copy coroutines on slave machine xxx to echo
data received on port 8000+i back to port 6000+i on the master
machine.

Now, on machine yyy, type:

tcpbench master -n 10 -k 50 -s 1000 -h xxx

This will create source, sink and 49 copy coroutines all running on
machine yyy, but no slave copy coroutines. At this stage 10 blocks of
1000 bytes will be following a route from the source to sink
coroutines passing through all the copy coroutines on both
machines. When all the data has been transferred, the tcpbench
commands on both machines will terminate after outputing appropriate
statistics.

The option -t causes trace output to be generated for debugging
purposes.

The default values for -n -k and -s are 10, 50 and 1000 respectively,
causing both master and slave coroutines to run on the same machine
with TCP/IP communication passing through the loopback interface. Thus
the parameterless command: tcpbench can be run on a machine without
a network connection.
*/


SECTION "tcpbench"

GET "libhdr"
GET "manhdr"

GLOBAL
{ mainco:ug      // Main controller not running as root
  maincowaiting
  sourceco       // The coroutine that creates the data stream 
  mcopycov       // Vector of master copy coroutines
  scopycov       // Vector of slave copy coroutines
  sinkco         // The coroutine that throws away the data
  killco         // The coroutine use to kill others
  initialising   // Number of coroutines currently still initialising
                 // or -1 when all initialisation has been completed
  cocount        // Number of existing source, sink and copy coroutines

  tracing        // Controls tracing output
  stdout         // Standard output for tracing info

  n              // Number of blocks of data generated by sourceco
  k              // Number of slave copy coroutines
  size           // The number of bytes in each block of data
  hostname
  master         // TRUE if this machine should run
                 // the sourceco, sinkco and the k-1 master copy coroutine(s).
  slave          // TRUE if this machine should run
                 // the k slave copy coroutine(s).

  pktlist
  cosendpkt
  findpkt
}

/* res := cosendpkt(link, id, act, r1, r2, a1, a2, a3, a4, a5, a6)

This routine is a substitute sendpkt for multi-event tasks.  It can
only be called when running as a non root coroutine of a task.  A
packet-coroutine pair is placed in pktlist before dispatching the
packet (using qpkt) so that when the packet returns to this task this
coroutine can be reactivated.  The globals cis, cos and currentdir are
saved and restored by cosendpkt.

Multi-event mode is entered by executing

     pktlist, sendpkt := 0, cosendpkt

Re-implemented by MR 7/6/02, futher modified MR 7/5/03
*/

LET cosendpkt(link, id, act, r1, r2, a1, a2, a3, a4, a5, a6) = VALOF
{ // The following local variables form the pktlist node.
  // Functions other than cosendpkt may manipulate it
  // so its format must not change.
  LET next,      pkt,     co, ocis, ocos, ocurrentdir =
      pktlist, @link, currco,  cis,  cos,  currentdir


//sawritef("DLIB cosendpkt: sending pkt=%n from %n to %n pktlist=%n*n",
//          @link, taskid, id, pktlist)
//abort(1000)

  // Safety check -- cosendpkt cannot be called from the root coroutine
  IF currco!co_parent<=0 DO
  { sawritef(
     "DLIB co=%n T%n: can't cosendpkt %n(id=%n,type=%n) from root coroutine*n",
       currco, taskid, pkt, id, act)
    abort(999)
  }

  pktlist := @next       // Insert it at the head of pktlist

//sawritef("DLIB:   co=%n: cosendpkt %n(id=%n,type=%n) calling cowait*n",
//  currco, pkt, pkt!pkt_id, pkt!pkt_type)

//sawritef(
//   "DLIB: co=%n T%n: cosendpkt calling qpkt %n(id=%n,type=%n, arg1=%n,arg2=%n)*n",
//     currco, taskid, pkt, id, act, a1, a2)

  UNLESS qpkt(pkt) DO
  { sawritef("DLIB co=%n: cosendpkt -- qpkt failure*n", currco)
    abort(181)
  }

  { LET p = cowait() // Safety check -- we must resume with the
    IF p=pkt BREAK   // expected packet.
    sawritef("DLIB co=%n T=%n: cosendpkt: received wrong pkt=%n should be %n*n",
              currco, taskid,  p, pkt)
    abort(182)
  } REPEAT

//sawritef("DLIB:   co=%n: cosendpkt %n(res1=%n,res2=%n) resumes*n",
//  currco, pkt, pkt!pkt_res1, pkt!pkt_res2)

  // Restore the saved globals
  cis, cos, currentdir := ocis, ocos, ocurrentdir 

  result2 := r2
  RESULTIS r1
}

/* cptr := findpkt(pkt)

This routine searches the pktlist for the specified packet. If found
the list item is dequeued and a pointer to the coroutine is
returned. Zero is returned if the packet is not found in pktlist.
*/

AND findpkt(pkt) = VALOF
{ LET a = @pktlist
//sawritef("DLIB findpkt: task=%n from %n => ", taskid, pkt!pkt_id)

  // Search pktlist for the relevant item
  { LET p = !a
    UNLESS p DO
    { //sawritef("not found*n")
      RESULTIS 0   // Not found
    }
    IF p!1 = pkt DO
    { //sawritef("%n*n", p)
      !a := !p            // Remove from pktlist
      RESULTIS p!2        // The coroutine
    }
    a := p
  } REPEAT
}


// Called by source, sink and copy coroutines committing suicide
LET die() BE resumeco(killco, currco)

LET trace(format, a,b,c,d,e,f,g) BE
{ LET out = output()
  //selectoutput(stdout)
  sawritef(format, a,b,c,d,e,f,g)
  //selectoutput(out)
}

LET start() BE
{ LET oldsendpkt = sendpkt
  LET argv = VEC 50
  stdout := output()

  n, k, size, hostname := 10, 50, 1000, 0
  master := FALSE
  slave  := FALSE

  UNLESS rdargs("-n/K,-k/K,-s/K,-h/K,-t/S,master/s,slave/s", argv, 50) DO
  { writef("Bad argument for TCPBENCH*n")
    stop(20)
  }

  IF argv!0 & string_to_number(argv!0) DO n    := result2
  IF argv!1 & string_to_number(argv!1) DO k    := result2
  IF argv!2 & string_to_number(argv!2) DO size := result2
  IF argv!3 DO hostname := argv!3
  tracing := argv!4
  master  := argv!5
  slave   := argv!6

  TEST hostname THEN master := TRUE
                ELSE hostname := "localhost"
  UNLESS master | slave DO master, slave := TRUE, TRUE

  writef("*n*nn=%n k=%n size=%n master=%n slave=%n hostname=%s*n",
          n, k, size, master, slave, hostname)

  mcopycov := getvec(k)  // Vector of master copy coroutines
  scopycov := getvec(k)  // Vector of slave copy coroutines

  UNLESS mcopycov & scopycov DO
  { writef("More space needed*n")
    IF mcopycov DO freevec(mcopycov)
    IF scopycov DO freevec(scopycov)
    stop(20)
  }

  FOR i = 0 TO k DO mcopycov!i, scopycov!i := 0, 0
  mainco, killco, sourceco, sinkco := 0, 0, 0, 0
  maincowaiting := FALSE
  cocount      := 0 // Number of source, sink and copy coroutines in existence
  initialising := 0 // Number of sink and copy coroutines still initialising

  mainco := createco(mainfn,   500)
  killco := createco(deleteco, 200)

  UNLESS mainco & killco DO
  { writef("Unable to create mainco and killco*n")
    IF mainco DO deleteco(mainco)
    IF killco DO deleteco(killco)
    RETURN
  }

  // Now go into Multi-event mode
  //writef("Entering multi-event mode*n")
  pktlist, sendpkt := 0, cosendpkt

  // Start the mainco coroutine
  callco(mainco)

  // Now run an event loop until all the coroutines have finished their work

  WHILE cocount DO
  { LET pkt = taskwait() // Hopefully a TCP packet owned by a coroutine
    LET co  = findpkt(pkt)
    UNLESS co DO
    { sawritef("Unexpected pkt=%n received in the event loop*n", pkt)
      BREAK
    }

    //sawritef("pkt %n(%n,%n) to co=%n initialising=%n cocount=%n*n",
    //          pkt, pkt!pkt_id, pkt!pkt_type, co, initialising, cocount)

    callco(co, pkt) // Give it to the owner

    //sawritef("back from co=%n initialising=%n cocount=%n*n",
    //          co, initialising, cocount)

    IF maincowaiting UNLESS initialising DO
    { initialising := -1
      callco(mainco, 0)  // Wake up mainco
    }
  }

  // Reinstate single-event mode
  sendpkt := oldsendpkt

  //writef("Closing down, initialising=%n cocount=%n*n", initialising, cocount)

  // Close down

  freevec(mcopycov)
  freevec(scopycov)

  deleteco(killco)

  writef("tcpbench finished*n")
}

AND mainfn() = VALOF
{ cocount := cocount + 1
  maincowaiting := FALSE

  IF slave DO
  { trace("Creating %n slave copy coroutines*n", k)
    FOR i = 1 TO k DO scopycov!i := initco(copyfn, 700, 8000+i, 6000+i)
  }

  IF master DO
  { IF k>1 DO trace("Creating %n master copy coroutines*n", k-1)
    FOR i = 1 TO k-1 DO mcopycov!i := initco(copyfn, 700, 6000+i, 8001+i)

    // At this stage all the coroutines except for sorceco will have been
    // created and will be suspended waiting for input connections.
  
    // We now create the source coroutine which will immediately
    // begin to transmit its data.
    trace("Creating the sink coroutine*n")
    sinkco := initco(sinkfn, 500, 6000+k)
  }

  // Wait for all (sink and copy) coroutines to say they are ready
  maincowaiting := TRUE
  cowait()
  maincowaiting := FALSE

  IF master DO
  { trace("Creating the source coroutine*n")
    sourceco := initco(sourcefn, 500, n, size, hostname, 8001)
  }

  delay(1000)  // 1 second
  trace("Main coroutine completed*n")
  cocount := cocount-1
  die()
}

AND appendstr(v, s) BE
{ LET len = v%0
  LET n = s%0
  FOR i = 1 TO n DO v%(len+i) := s%i
  v%0 := len + n
}

AND appendnum(v, num) BE
{ LET len = v%0
  LET n   = 0
  LET dig = VEC 20
  { n := n+1
    dig!n := num REM 10 + '0'
    num := num/10
  } REPEATWHILE num>0
  FOR i = 1 TO n DO v%(len+n-i+1) := dig!i
  v%0 := len + n
}

AND sourcefn(args) = VALOF
{ LET n, size, hostname, port = args!0, args!1, args!2, args!3
  LET remipaddr = 0
  LET buf = getvec((size-1)/bytesperword)
  LET outstream = 0
  LET tcpname = VEC 20
  STATIC { ch = 0 }

  cocount := cocount + 1

  UNLESS hostname DO hostname := "localhost"

  tcpname%0 := 0
  appendstr(tcpname, "tcp:")
  appendstr(tcpname, hostname)
  appendstr(tcpname, ":")
  appendnum(tcpname, port)

  // Put data into the buffer
  FOR i = 0 TO size-1 DO
  { UNLESS 'A'<=ch<='Z' DO ch := 'A'
    buf%i := ch
    ch := ch + 1
  }

  //IF tracing DO sawritef("sourcefn: co=%n outport %n*n", currco, port)
  
  // Now start transmitting the data

  IF tracing DO trace("sourcefn: co=%n attempting to connect to %s*n",
                          currco, tcpname)

  outstream := findoutput(tcpname)

//sawritef("sourcefn: co=%n returned from findoutput(%s)*n", currco, tcpname)

  UNLESS outstream DO
  { trace("sourcefn: co=%n failed to open %s*n", currco, tcpname)
    abort(999)
    RESULTIS -1
  }

  remipaddr := sendpkt(-1, Task_tcphandler, Action_getremipaddr, 0, 0, outstream)
  UNLESS remipaddr DO
  { trace("sourcefn: co=%n failed to connect to %s*n", currco, tcpname)
    RESULTIS -1
  }

  IF tracing DO
     trace("sourcefn: co=%n connection established to %s ipaddr=%x8*n",
                         currco, tcpname, remipaddr)

  trace("source about to write %n block(s) of %n bytes*n", n, size)
  selectoutput(outstream)

  FOR i = 1 TO n DO
  { LET sent = 0
    IF tracing DO trace("sourcefn: co=%n sending %n bytes to port %n*n",
                            currco, size, port)
    FOR i = 0 TO size-1 DO wrch(buf%i)
    deplete(outstream)
//sawritef("sourcefn: co=%n %n bytes sent*n", currco, size)
//    FOR i = 0 TO size-1 DO sawritef("sourcefn: buf!%i3 = '%c'*n", i, buf%i)    
  }

//sawritef("sourcefn: co=%n delaying 5 seconds*n", currco)
//delay(5000)

//sawritef("sourcefn: co=%n closing output stream %s*n", currco, tcpname)
  endwrite()

  IF tracing DO trace("sourcefn: co=%n stream %s closed*n", currco, tcpname)

  freevec(buf)

  IF tracing DO trace("sourcefn: co=%n committing suicide*n", currco)

  cocount := cocount-1
  die()
  RESULTIS 0
}

AND copyfn(args) = VALOF
{ LET iport, oport = args!0, args!1
  LET bytecount    = 0
  LET buf          = VEC 256
  LET ipaddr       = 0       // ip address of remote machine
  LET remipaddr    = 0       // ip address of remote machine after connection
  LET tcpinname    = VEC 20
  LET tcpoutname   = VEC 20
  LET instream, outstream = 0, 0

  initialising := initialising + 1
  cocount      := cocount + 1

  IF tracing DO trace("copyfn: co=%n inport %n outport %n*n",
                          currco, iport, oport)

  tcpinname%0 := 0
  appendstr(tcpinname, "tcp::")
  appendnum(tcpinname, iport)

//  IF tracing DO trace("copyfn: co=%n calling findinput(%s)*n",
//                          currco, tcpinname)
  instream := findinput(tcpinname)
  ipaddr := result2

  UNLESS instream DO
  { trace("copyfn: co=%n Cannot read from %s*n", currco, tcpinname)
    abort(999)
  }

//  IF tracing DO sawritef("copyfn: co=%n returned from findinput(%s)=> %n*n",
//                          currco, tcpinname, instream)

  initialising := initialising - 1

//trace("copyfn: co=%n waiting to read from %s*n", currco, tcpinname)

//trace("copyfn: calling sendpkt with getremipaddr pkt instream=%n*n",
//                          instream)
  remipaddr := sendpkt(-1, Task_tcphandler, Action_getremipaddr, 0, 0, instream)
//trace("copyfn: returned from  sendpkt remipaddr=%x8*n", remipaddr)
//  IF tracing DO trace("copyfn: co=%n remipaddr=%x8*n",
//                          currco, remipaddr)
//abort(1001)
  UNLESS remipaddr DO
  { trace("copyfn: co=%n failed to connect to %s*n", currco, tcpinname)
    RESULTIS -1
  }

  IF tracing DO trace("copyfn: co=%n accepted connection via port %n from %x8*n",
                         currco, iport, remipaddr)

  tcpoutname%0 := 0
  appendstr(tcpoutname, "tcp:")
  appendnum(tcpoutname, remipaddr>>24 & 255)
  appendstr(tcpoutname, ".")
  appendnum(tcpoutname, remipaddr>>16 & 255)
  appendstr(tcpoutname, ".")
  appendnum(tcpoutname, remipaddr>> 8 & 255)
  appendstr(tcpoutname, ".")
  appendnum(tcpoutname, remipaddr     & 255)
  appendstr(tcpoutname, ":")
  appendnum(tcpoutname, oport)

//  IF tracing DO trace("copyfn: co=%n attempting to connect to %s*n",
//                          currco, tcpoutname)
  outstream := findoutput(tcpoutname)
  UNLESS outstream DO
  { trace("copyfn: co=%n cannot write to %s*n", currco, tcpoutname)
    RESULTIS -1
  }

//trace("copyfn: co=%n getremipaddr outstream=%n*n",currco, outstream)
  remipaddr := sendpkt(-1, Task_tcphandler, Action_getremipaddr, 0, 0, outstream)
//trace("copyfn: co=%n getremipaddr remipaddr=%x8*n",currco, remipaddr)
  UNLESS remipaddr DO
  { trace("copyfn: co=%n failed to connect to %s*n", currco, tcpoutname)
    RESULTIS -1
  }

  IF tracing DO trace("copyfn: co=%n connection established to %s*n",
                          currco, tcpoutname)

  selectinput(instream)
  selectoutput(outstream)

  { LET ch = rdch()
    IF ch=endstreamch BREAK
//trace("copyfn: co=%n copying byte %i3  '%c'*n", currco, bytecount, ch)
    wrch(ch)
    bytecount := bytecount + 1
    IF tracing & bytecount REM 1000 = 0 DO
      trace("copyfn: co=%n %n bytes from port %n*n", currco, bytecount, iport)
  } REPEAT

  deplete(outstream)

  IF tracing DO
       trace("copyfn: co=%n %n->%n finishing after copying %n bytes*n", 
                          currco, iport, oport, bytecount)

  endread()
  endwrite()

  IF tracing DO trace("copy coroutine %n->%n committing suicide*n", iport, oport)

//trace("copyfn: co=%n calling die()*n", stackbase)
  cocount := cocount-1
  die()
  RESULTIS 0
}

AND sinkfn(args) = VALOF
{ LET iport     = args!0
  LET bytecount = 0
  LET ipaddr    = 0
  LET remipaddr = 0
  LET instream  = 0
  LET tcpinname = VEC 20

  initialising := initialising + 1
  cocount := cocount + 1

  tcpinname%0 := 0
  appendstr(tcpinname, "tcp::")
  appendnum(tcpinname, iport)

//trace("sinkfn: co=%n tcpinname=%s*n", currco, tcpinname)
//abort(1000)

  //IF tracing DO trace("sinkfn: co=%n on port %n*n", currco, iport)

  instream := findinput(tcpinname)
  ipaddr := result2

  UNLESS instream DO
  { trace("sinkfn: co=%n Cannot read from %s*n", currco, tcpinname)
    abort(999)
  }

  initialising := initialising-1

  IF tracing DO trace("sinkfn: sink coroutine waiting to be connected*n")

  remipaddr := sendpkt(-1, Task_tcphandler, Action_getremipaddr, 0, 0, instream)
  UNLESS remipaddr DO
  { trace("sinkfn: co=%n failed connection from %s*n", currco, tcpinname)
    RESULTIS -1
  }

  IF tracing DO trace("sinkfn: co=%n accepted connection via port %n from %x8*n",
                         currco, iport, remipaddr)

  selectinput(instream)


  { LET ch = rdch()
    IF ch=endstreamch BREAK
//trace("sinkfn: bytecount=%i3 received %x2 '%c'*n", bytecount, ch, ch)
    bytecount := bytecount + 1
    IF tracing & bytecount REM 1000 = 0 DO
       trace("sinkfn: co=%n received %n bytes*n", currco, bytecount)
  } REPEAT

  trace("sink finishing after reading %n bytes*n", bytecount)
//delay(2000)
  endread()

  IF tracing DO trace("Sink co=%n committing suicide*n", currco)

  cocount := cocount-1
  die()
  RESULTIS 0
}




