{$l remout :}
{$R-}
program benchmark;

(* Pascal version of Martin Richards' Benchmark program
   to test the suitability of high-level system languages.

   Originally converted to Pascal by J.M.Bishop March 81.

   Modified to work on 16 bit Pascals by Colin Whitby-Strevens and 
   Chris Leed, Inmos Ltd. Sept 1981
   Modified by Martin Richards   Novemeber 1996
*)

const { Comment out one of the following lines }
   {Count=10000;      Qpktcountval=23246;   Holdcountval=9297;}
   Count=10000*100;  Qpktcountval=2326410; Holdcountval=930563;

const taskmax   =  10;
      idler     =  1;
      worker    =  2;
      handlera  =  3;
      handlerb  =  4;
      devicea   =  5;
      deviceb   =  6;
      bufsize   =  3;

type identifiers    =  0..taskmax ;
     priorities     =  0..maxint;
     states         =  (packet, wait, hold) ;
     stateset       =  set of states;
     kinds          =  (work, device) ;
     funcs          =  (idlefn, workfn, handlerfn, devicefn) ;
     bytes          =  0..255 ;
     bufindex       =  0..bufsize ;
     buffers        =  packed array [bufindex] of bytes;

     topackets      =  ^packets ;
     packets        =  record
                         plink  : topackets;
                         id     : identifiers;
                         kind   : kinds;
                         count  : bytes;
		         buffer : buffers;
                       end;

     totasks        = ^tasks;
     tasks          = record
                        tlink     :  totasks;
                        id        :  identifiers;
                        priority  :  priorities;
                        workq     :  topackets;
		        state     :  stateset;
                        case func   :  funcs of
                          idlefn    :  (counter       : integer;
                                        hasher        : integer);

                          workfn    :  (sender        : identifiers;
                                        data          : bytes);

                          handlerfn :  (workpackets   : topackets;
                                        devicepackets : topackets);

                          devicefn  :  (workdone      : topackets)
                        end;

var  tasktable    : array  [identifiers]  of totasks;
     task         : totasks;
     qpacketcount : integer;
     holdcount    : integer;
     tracing      : boolean;
     layout       : 0..50;

procedure initialise;
  var  workqu   : topackets;
       tasklist : totasks;
       id       : identifiers;

procedure  createtask ( i : identifiers;  p : priorities;
                        w : topackets;    s : stateset;
                        f : funcs) ;
  begin
    new(tasktable[i]) ;
    with tasktable[i]^ do begin
      tlink     :=  tasklist;
      tasklist  :=  tasktable[i] ;
      id        :=  i;
      priority  :=  p;
      workq     :=  w;
      state     :=  s;
      func      :=  f;
    end;
  end; {createtask}

function newpacket ( l : topackets;  i : identifiers;
                     k : kinds )  :  topackets ;
  var p : topackets ;
  begin
    new(p) ;
    with p^ do begin
      plink :=  l ;
      id    :=  i ;
      kind  :=  k ;
      count :=  0 ;
    end;
    newpacket  :=  p ;
  end;  {newpacket}

begin {initialise}
  for id  := 1 to taskmax do tasktable[id] := nil ;
  tasklist := nil ;

  createtask (idler, 0, nil, [{run}], idlefn);
  tasktable [idler]^.hasher  := 1;
  tasktable [idler]^.counter := Count;

  workqu := newpacket (nil   , 0, work);
  workqu := newpacket (workqu, 0, work);
  createtask (worker, 1000, workqu, [wait, packet], workfn);
  tasktable  [worker]^.sender   := handlera;
  tasktable  [worker]^.data     := 0 ;

  workqu := newpacket (nil   , devicea, device);
  workqu := newpacket (workqu, devicea, device);
  workqu := newpacket (workqu, devicea, device);
  createtask (handlera, 2000, workqu, [wait, packet], handlerfn);
  tasktable  [handlera]^.workpackets   := nil;
  tasktable  [handlera]^.devicepackets := nil;

  workqu := newpacket (nil   , deviceb, device);
  workqu := newpacket (workqu, deviceb, device);
  workqu := newpacket (workqu, deviceb, device);
  createtask (handlerb, 3000, workqu, [wait, packet], handlerfn);
  tasktable  [handlerb]^.workpackets   :=  nil ;
  tasktable  [handlerb]^.devicepackets :=  nil ;

  createtask (devicea, 4000, nil, [wait] , devicefn) ;
  tasktable [devicea]^.workdone := nil ;

  createtask (deviceb, 5000, nil, [wait] , devicefn);
  tasktable [deviceb]^.workdone := nil ;

  task :=  tasklist;

  qpacketcount := 0;
  holdcount := 0;

  layout := 0;
  tracing := false;

end; {initialise}

procedure trace(ch :char) ;
  begin
    layout := layout - 1;
    if layout <= 0 then begin
      writeln; layout  := 50;
    end;
    write(ch:1) ;
  end; {trace}

function holdself : totasks ;
  begin
    holdcount := holdcount + 1;
    with task^ do state := state + [hold];
    holdself := task^.tlink ;
  end;

function waitself : totasks ;
  begin
    with task^ do state := state + [wait];
    waitself := task;
  end;

function findtask (id : identifiers) : totasks;
  begin
    findtask := tasktable[0] ;
    if id in [0..taskmax] then findtask  := tasktable [id]
                          else writeln ('Bad task') ;
  end;

function release (id : identifiers) : totasks;
  var  t : totasks;
  begin
    t := findtask (id) ;
    if t = nil then release := nil else
    with t^ do begin
      state := state - [hold] ;
      if priority > task^.priority then
        release := t
      else release := task;
  end;
end; {release}

procedure append (p : topackets; var queue : topackets) ;
  var lq : topackets ; 
  begin
    lq := queue;
    p^.plink := nil;
    if  lq = nil then
      queue := p
    else begin
      while lq^.plink <> nil do lq := lq^.plink;
      lq^.plink := p;
  end;
end;

function queuepacket (p : topackets ) : totasks ;
  var t : totasks ;
  begin
    t  :=  findtask (p^.id) ;
    if  t = nil then queuepacket := nil
    else begin
      qpacketcount  :=  qpacketcount + 1 ;
      p^.plink  :=  nil;
      p^.id  :=  task^.id ;
      with t^  do
        if workq  =  nil then begin
          workq     :=  p ;
          state     :=  state + [packet] ;
          if priority  >  task^.priority  then
            queuepacket  := t
          else  queuepacket := task ;
        end
        else begin
          append (p, workq) ;
          queuepacket := task;
        end;
    end;
  end; {queuepacket}

function switchtask (packet : topackets) : totasks ;
  type bits  =  set of 0..31 ;
  var  thisworkp   :  topackets ;
       thisdevicep :  topackets;
       i           :  bufindex;
       shifter     :  record case boolean of
                        true  :  (i : integer) ;
                        false :  (s : bits) ;
                      end;
       D008        :  bits;
  begin
    with task^  do
    case func of
      idlefn  : begin
                  counter := counter - 1 ;
                  if counter = 0 then switchtask := holdself
                  else begin
                    if not odd(hasher) then begin
                      hasher  := hasher div  2 ;
                      shifter.i := hasher ;
		      {shifter.s := shifter.s - [15]; {turn sign bit off}
                      hasher := shifter.i ;
                      switchtask := release (devicea)  ;
                    end
                    else begin
                      D008  := [15, 14, 12, 3];
                      {if hasher < 0 then hasher := hasher - 1;}
                      {Correct for DIV to make it round away from zero}
		      shifter.i := hasher div 2;
                      {shifter.s := shifter.s - [15];}
                      shifter.s := (shifter.s - D008) + (D008 - shifter.s);
                      hasher := shifter.i ;
                      switchtask := release (deviceb) ;
                      end;
                    end;
                  end; {idlefn}
     workfn    :  begin
                    if packet = nil then switchtask := waitself
                    else begin
                      if sender = handlera then sender := handlerb
                                           else sender := handlera;
                      with packet^ do begin
                        id := sender; {destination task}
                        count := 0; {buffer subscript}
                        for  i := 0 to bufsize  do begin
                          buffer[i] := data+ord('A');
                          data := (data+1) mod 26;
                        end;
                      end;
                      switchtask := queuepacket (packet);
                    end;
		  end;  {workfn}
      handlerfn : begin  
                    if  packet  <>  nil then
                      case packet^.kind of
                        device :  append (packet, devicepackets);
                        work   :  append (packet, workpackets);
                      end;
                    if workpackets <> nil then begin
                      thisworkp := workpackets;
                      with  thisworkp^ do
                        if count > bufsize then begin
                          workpackets := workpackets^.plink;
                          switchtask := queuepacket (thisworkp);
                        end
                        else
                          if devicepackets <> nil then begin
                            thisdevicep := devicepackets;
                            devicepackets := devicepackets^.plink ;
                            thisdevicep^.count := buffer[count] ;
                            count  :=  count + 1 ;
                            switchtask := queuepacket (thisdevicep) ;
                          end
                        else switchtask  := waitself;
                    end
                    else switchtask := waitself;
                  end; {handlerfn}
        devicefn: begin
                    if packet = nil then begin
                      if workdone = nil then switchtask := waitself
                      else begin
                        packet := workdone ;
                        workdone := packet^.plink ;
                        switchtask := queuepacket (packet)
                      end
                    end
                    else begin
                      append (packet, workdone) ;
        	      if tracing then trace(packet^.count);
                      switchtask := holdself ;
                    end;
                 end; {devicefn}
    end;  {case}
  end; {switchtask}

procedure schedule;
  var currentpacket  :  topackets ;
  begin
    while task <> nil do begin
      currentpacket := nil ;
      if task^.state = [wait,packet] then
        with task^ do begin  
          currentpacket  := workq;
          workq  :=  workq^.plink;
          if workq = nil then state  :=  [{run}]
                         else state  :=  [{run,}packet];
      end; {with and runpacket}
      if task^.state - [packet] = [] {i.e. no hold or wait} then begin
	 if tracing then trace(chr(task^.id+ord('0')));;
        task := switchtask (currentpacket) ;
      end
      else task := task^.tlink;
    end; {all task}
  end; {schedule}

begin {Main Program}
writeln ('Bench mark starting, Count = ', Count	:1 ) ;
  initialise;

  writeln ( 'starting');

  schedule;

  writeln;
  writeln ( 'finished');
  writeln ( 'qkt count = ', qpacketcount :1, '   ',
            'holdcount = ', holdcount :1) ;
  write("these results are ");
  if (qpacketcount = Qpktcountval) and (holdcount = Holdcountval) then 
     writeln ('correct')
  else writeln ('incorrect') ;
  writeln ('end of run') ;
end.

	     



                    

