/*

This is a simple implementation of the AES-128 (Rijndael) standard.
It has a block size of 128 bits, a key size of 128 bit and uses 10 rounds.

Implemented in BCPL by Martin Richards June 2001
*/

GET "libhdr"

GLOBAL {
  count:ug
  key;  data;  code;  res

  s0; s1; s2; s3   // The state
  kv               // The key schedule
}

LET start() = VALOF
{ writef("Testing the AES-128 encryption algorithm*n*n")

//key  := TABLE #x2b7e1516, #x28aed2a6, #xabf71588, #x09cf4f3c
//data := TABLE #x3243f6a8, #x885a308d, #x313198a2, #xe0370734

  key  := TABLE #x00010203, #x04050607, #x08090a0b, #x0c0d0e0f
  data := TABLE #x00112233, #x44556677, #x8899aabb, #xccddeeff
  code := TABLE #x00000000, #x00000000, #x00000000, #x00000000
  res  := TABLE #x00000000, #x00000000, #x00000000, #x00000000

  kv := TABLE 0,0,0,0,0,0,0,0,0,0,0,  // To hold kv!0 .. kv!43
              0,0,0,0,0,0,0,0,0,0,0,
              0,0,0,0,0,0,0,0,0,0,0,
              0,0,0,0,0,0,0,0,0,0,0

  pr("key:  ", key)
  SetKey(key, kv)         // Set the key schedule
  pr("data: ", data)
  encrypt(data, code, kv) // Encrypt data => code
  pr("code: ", code)
  decrypt(code, res,  kv) // Decrypt code => res
  pr("res:  ", res)

  RESULTIS 0
}

AND pr(str, v) BE writef("%s %x8 %x8 %x8 %x8*n", str, v!0, v!1, v!2, v!3)

AND prst() BE pr("S:    ", @s0)

AND SetKey(key, kv) BE // Set the key schedule in kv
{ LET Rcon = TABLE 0, #x01000000, #x02000000, #x04000000, #x08000000,
                      #x10000000, #x20000000, #x40000000, #x80000000,
                      #x1b000000, #x36000000

  kv!0, kv!1, kv!2, kv!3 := key!0, key!1, key!2, key!3

  FOR i = 4 TO 43 DO
  { LET t = kv!(i-1)
    IF i REM 4 = 0 DO t := SubWord(RotWord(t)) NEQV Rcon!(i/4)
    kv!i := kv!(i-4) NEQV t
  }
}

AND RotWord(w) = w<<8 | w>>24

AND encrypt(data, code, kv) BE
{ GetState(data)                ;  prst()
  AddRoundKey(kv)               ;  prst();  newline()

  FOR round = 1 TO 9 DO
  { SubBytes()                  ;  prst()
    ShiftRows()                 ;  prst()
    MixColumns()                ;  prst()
    AddRoundKey(kv + 4*round)   ;  prst();  newline()
  }

  SubBytes()                    ;  prst()
  ShiftRows()                   ;  prst()
  AddRoundKey(kv + 4*10)        ;  prst();  newline()
  PutState(code)
}

AND decrypt(code, res, kv) BE
{ GetState(code)                ;  prst()
  AddRoundKey(kv + 4*10)        ;  prst();  newline()

  FOR round = 9 TO 1 BY -1 DO
  { InvShiftRows()              ;  prst()
    InvSubBytes()               ;  prst()
    AddRoundKey(kv + 4*round)   ;  prst();  newline()
    InvMixColumns()             ;  prst()
  }

  InvShiftRows()                ;  prst()
  InvSubBytes()                 ;  prst()
  AddRoundKey(kv)               ;  prst();  newline()

  PutState(res)
}

AND GetState(v) BE  s0, s1, s2, s3 := v!0, v!1, v!2, v!3

AND PutState(v) BE  v!0, v!1, v!2, v!3 := s0, s1, s2, s3 

AND AddRoundKey(v) BE
{ pr("K:    ", v)
  s0 := s0 NEQV v!0
  s1 := s1 NEQV v!1
  s2 := s2 NEQV v!2
  s3 := s3 NEQV v!3
}  

AND SubBytes() BE
{ s0 := SubWord(s0)
  s1 := SubWord(s1)
  s2 := SubWord(s2)
  s3 := SubWord(s3)
}

AND SubWord(w) = SubByte(w>>24)<<24 |
                 SubByte(w>>16)<<16 |
                 SubByte(w>> 8)<< 8 |
                 SubByte(w    )

AND SubByte(b) = VALOF
{ LET w = (b>>2 & 63) ! TABLE
            #x637c777b,#xf26b6fc5,#x3001672b,#xfed7ab76,
            #xca82c97d,#xfa5947f0,#xadd4a2af,#x9ca472c0,
            #xb7fd9326,#x363ff7cc,#x34a5e5f1,#x71d83115,
            #x04c723c3,#x1896059a,#x071280e2,#xeb27b275,
            #x09832c1a,#x1b6e5aa0,#x523bd6b3,#x29e32f84,
            #x53d100ed,#x20fcb15b,#x6acbbe39,#x4a4c58cf,
            #xd0efaafb,#x434d3385,#x45f9027f,#x503c9fa8,
            #x51a3408f,#x929d38f5,#xbcb6da21,#x10fff3d2,
            #xcd0c13ec,#x5f974417,#xc4a77e3d,#x645d1973,
            #x60814fdc,#x222a9088,#x46eeb814,#xde5e0bdb,
            #xe0323a0a,#x4906245c,#xc2d3ac62,#x9195e479,
            #xe7c8376d,#x8dd54ea9,#x6c56f4ea,#x657aae08,
            #xba78252e,#x1ca6b4c6,#xe8dd741f,#x4bbd8b8a,
            #x703eb566,#x4803f60e,#x613557b9,#x86c11d9e,
            #xe1f89811,#x69d98e94,#x9b1e87e9,#xce5528df,
            #x8ca1890d,#xbfe64268,#x41992d0f,#xb054bb16
  w := w >> ( (3 - (b&3)) * 8)
  RESULTIS w & 255
}

AND InvSubBytes() BE
{ s0 := InvSubWord(s0)
  s1 := InvSubWord(s1)
  s2 := InvSubWord(s2)
  s3 := InvSubWord(s3)
}

AND InvSubWord(w) = InvSubByte(w>>24)<<24 |
                    InvSubByte(w>>16)<<16 |
                    InvSubByte(w>> 8)<< 8 |
                    InvSubByte(w    )


AND InvSubByte(b) = VALOF
{ LET w = (b>>2 & 63) ! TABLE
            #x52096AD5, #x3036A538, #xBF40A39E, #x81F3D7FB,
            #x7CE33982, #x9B2FFF87, #x348E4344, #xC4DEE9CB,
            #x547B9432, #xA6C2233D, #xEE4C950B, #x42FAC34E,
            #x082EA166, #x28D924B2, #x765BA249, #x6D8BD125,
            #x72F8F664, #x86689816, #xD4A45CCC, #x5D65B692,
            #x6C704850, #xFDEDB9DA, #x5E154657, #xA78D9D84,
            #x90D8AB00, #x8CBCD30A, #xF7E45805, #xB8B34506,
            #xD02C1E8F, #xCA3F0F02, #xC1AFBD03, #x01138A6B,
            #x3A911141, #x4F67DCEA, #x97F2CFCE, #xF0B4E673,
            #x96AC7422, #xE7AD3585, #xE2F937E8, #x1C75DF6E,
            #x47F11A71, #x1D29C589, #x6FB7620E, #xAA18BE1B,
            #xFC563E4B, #xC6D27920, #x9ADBC0FE, #x78CD5AF4,
            #x1FDDA833, #x8807C731, #xB1121059, #x2780EC5F,
            #x60517FA9, #x19B54A0D, #x2DE57A9F, #x93C99CEF,
            #xA0E03B4D, #xAE2AF5B0, #xC8EBBB3C, #x83539961,
            #x172B047E, #xBA77D626, #xE1691463, #x55210C7D
  w := w >> ( (3 - (b&3)) * 8)
  RESULTIS w & 255
}

AND ShiftRows() BE
{ LET a, b, c, d = s0, s1, s2, s3
  s0 := a&#xFF000000 | b&#x00FF0000 | c&#x0000FF00 | d&#x000000FF
  s1 := b&#xFF000000 | c&#x00FF0000 | d&#x0000FF00 | a&#x000000FF
  s2 := c&#xFF000000 | d&#x00FF0000 | a&#x0000FF00 | b&#x000000FF
  s3 := d&#xFF000000 | a&#x00FF0000 | b&#x0000FF00 | c&#x000000FF
}

AND InvShiftRows() BE
{ LET a, b, c, d = s0, s1, s2, s3
  s0 := a&#xFF000000 | d&#x00FF0000 | c&#x0000FF00 | b&#x000000FF
  s1 := b&#xFF000000 | a&#x00FF0000 | d&#x0000FF00 | c&#x000000FF
  s2 := c&#xFF000000 | b&#x00FF0000 | a&#x0000FF00 | d&#x000000FF
  s3 := d&#xFF000000 | c&#x00FF0000 | b&#x0000FF00 | a&#x000000FF
}

AND MixColumns() BE
{ s0 := MixCol(s0)
  s1 := MixCol(s1)
  s2 := MixCol(s2)
  s3 := MixCol(s3)
}

AND MixCol(w) = VALOF
{ LET a, b, c, d = w>>24, w>>16 & 255, w>>8 & 255, w & 255
  LET t = a
  LET u = a NEQV b NEQV c NEQV d
  a := a NEQV u NEQV FFmulX(t NEQV b)
  b := b NEQV u NEQV FFmulX(b NEQV c)
  c := c NEQV u NEQV FFmulX(c NEQV d)
  d := d NEQV u NEQV FFmulX(d NEQV t)
  RESULTIS a<<24 | b<<16 | c<<8 | d
}

AND FFmulX(b) = b<<1 NEQV (b>127 -> #x11b, 0)

AND InvMixColumns() BE
{ s0 := InvMixCol(s0)
  s1 := InvMixCol(s1)
  s2 := InvMixCol(s2)
  s3 := InvMixCol(s3)
}

AND InvMixCol(w) = VALOF
{ LET a, b, c, d = w>>24, w>>16 & 255, w>>8 & 255, w & 255
  LET t = a
  LET u = a NEQV b NEQV c NEQV d
  LET v = ?

  u := u NEQV FFmulX(FFmulX(FFmulX(u)))
  v := u NEQV FFmulX(FFmulX(b NEQV d))
  u := u NEQV FFmulX(FFmulX(a NEQV c))

  a := a NEQV u NEQV FFmulX(t NEQV b)
  b := b NEQV v NEQV FFmulX(b NEQV c)
  c := c NEQV u NEQV FFmulX(c NEQV d)
  d := d NEQV v NEQV FFmulX(d NEQV t)

  RESULTIS a<<24 | b<<16 | c<<8 | d
}

/*  Output:

Testing the AES-128 encryption algorithm

key:   00010203 04050607 08090A0B 0C0D0E0F
data:  00112233 44556677 8899AABB CCDDEEFF
S:     00112233 44556677 8899AABB CCDDEEFF
K:     00010203 04050607 08090A0B 0C0D0E0F
S:     00102030 40506070 8090A0B0 C0D0E0F0

S:     63CAB704 0953D051 CD60E0E7 BA70E18C
S:     6353E08C 0960E104 CD70B751 BACAD0E7
S:     5F726415 57F5BC92 F7BE3B29 1DB9F91A
K:     D6AA74FD D2AF72FA DAA678F1 D6AB76FE
S:     89D810E8 855ACE68 2D1843D8 CB128FE4

S:     A761CA9B 97BE8B45 D8AD1A61 1FC97369
S:     A7BE1A69 97AD739B D8C9CA45 1F618B61
S:     FF879684 31D86A51 645151FA 773AD009
K:     B692CF0B 643DBDF1 BE9BC500 6830B3FE
S:     4915598F 55E5D7A0 DACA94FA 1F0A63F7

S:     3B59CB73 FCD90EE0 5774222D C067FB68
S:     3BD92268 FC74FB73 5767CBE0 C0590E2D
S:     4C9C1E66 F771F076 2C3F868E 534DF256
K:     B6FF744E D2C2C9BF 6C590CBF 0469BF41
S:     FA636A28 25B339C9 40668A31 57244D17

S:     2DFB0234 3F6D12DD 09337EC7 5B36E3F0
S:     2D6D7EF0 3F33E334 093602DD 5BFB12C7
S:     6385B79F FC538DF9 97BE478E 7547D691
K:     47F7F7BC 95353E03 F96C32BC FD058DFD
S:     24724023 6966B3FA 6ED27532 88425B6C

S:     36400926 F9336D2D 9FB59D23 C42C3950
S:     36339D50 F9B53926 9F2C092D C4406D23
S:     F4BCD454 32E554D0 75F1D6C5 1DD03B3C
K:     3CAAA3E8 A99F9DEB 50F3AF57 ADF622AA
S:     C81677BC 9B7AC93B 25027992 B0261996

S:     E847F565 14DADDE2 3F77B64F E7F7D490
S:     E8DAB690 1477D465 3FF7F5E2 E747DD4F
S:     9816EE74 00F87F55 6B2C049C 8E5AD036
K:     5E390F7D F7A69296 A7553DC1 0AA31F6B
S:     C62FE109 F75EEDC3 CC79395D 84F9CF5D

S:     B415F801 6858552E 4BB6124C 5F998A4C
S:     B458124C 68B68A01 4B99F82E 5F15554C
S:     C57E1C15 9A9BD286 F05F4BE0 98C63439
K:     14F9701A E35FE28C 440ADF4D 4EA9C026
S:     D1876C0F 79C4300A B45594AD D66FF41F

S:     3E175076 B61C0467 8DFC2295 F6A8BFC0
S:     3E1C22C0 B6FCBF76 8DA85067 F6170495
S:     BAA03DE7 A1F9B56E D5512CBA 5F414D23
K:     47438735 A41C65B9 E016BAF4 AEBF7AD2
S:     FDE3BAD2 05E5D0D7 3547964E F1FE37F1

S:     5411F4B5 6BD9700E 96A0902F A1BB9AA1
S:     54D990A1 6BA09AB5 96BBF40E A111702F
S:     E9F74EEC 023020F6 1BF2CCF2 353C21C7
K:     549932D1 F0855768 1093ED9C BE2C974E
S:     BD6E7C3D F2B5779E 0B61216E 8B10B689

S:     7A9F1027 89D5F50B 2BEFFD9F 3DCA4EA7
S:     7AD5FDA7 89EF4E27 2BCA100B 3D9FF59F
K:     13111D7F E3944A17 F307A78B 4D2B30C5
S:     69C4E0D8 6A7B0430 D8CDB780 70B4C55A

code:  69C4E0D8 6A7B0430 D8CDB780 70B4C55A
S:     69C4E0D8 6A7B0430 D8CDB780 70B4C55A
K:     13111D7F E3944A17 F307A78B 4D2B30C5
S:     7AD5FDA7 89EF4E27 2BCA100B 3D9FF59F

S:     7A9F1027 89D5F50B 2BEFFD9F 3DCA4EA7
S:     BD6E7C3D F2B5779E 0B61216E 8B10B689
K:     549932D1 F0855768 1093ED9C BE2C974E
S:     E9F74EEC 023020F6 1BF2CCF2 353C21C7

S:     54D990A1 6BA09AB5 96BBF40E A111702F
S:     5411F4B5 6BD9700E 96A0902F A1BB9AA1
S:     FDE3BAD2 05E5D0D7 3547964E F1FE37F1
K:     47438735 A41C65B9 E016BAF4 AEBF7AD2
S:     BAA03DE7 A1F9B56E D5512CBA 5F414D23

S:     3E1C22C0 B6FCBF76 8DA85067 F6170495
S:     3E175076 B61C0467 8DFC2295 F6A8BFC0
S:     D1876C0F 79C4300A B45594AD D66FF41F
K:     14F9701A E35FE28C 440ADF4D 4EA9C026
S:     C57E1C15 9A9BD286 F05F4BE0 98C63439

S:     B458124C 68B68A01 4B99F82E 5F15554C
S:     B415F801 6858552E 4BB6124C 5F998A4C
S:     C62FE109 F75EEDC3 CC79395D 84F9CF5D
K:     5E390F7D F7A69296 A7553DC1 0AA31F6B
S:     9816EE74 00F87F55 6B2C049C 8E5AD036

S:     E8DAB690 1477D465 3FF7F5E2 E747DD4F
S:     E847F565 14DADDE2 3F77B64F E7F7D490
S:     C81677BC 9B7AC93B 25027992 B0261996
K:     3CAAA3E8 A99F9DEB 50F3AF57 ADF622AA
S:     F4BCD454 32E554D0 75F1D6C5 1DD03B3C

S:     36339D50 F9B53926 9F2C092D C4406D23
S:     36400926 F9336D2D 9FB59D23 C42C3950
S:     24724023 6966B3FA 6ED27532 88425B6C
K:     47F7F7BC 95353E03 F96C32BC FD058DFD
S:     6385B79F FC538DF9 97BE478E 7547D691

S:     2D6D7EF0 3F33E334 093602DD 5BFB12C7
S:     2DFB0234 3F6D12DD 09337EC7 5B36E3F0
S:     FA636A28 25B339C9 40668A31 57244D17
K:     B6FF744E D2C2C9BF 6C590CBF 0469BF41
S:     4C9C1E66 F771F076 2C3F868E 534DF256

S:     3BD92268 FC74FB73 5767CBE0 C0590E2D
S:     3B59CB73 FCD90EE0 5774222D C067FB68
S:     4915598F 55E5D7A0 DACA94FA 1F0A63F7
K:     B692CF0B 643DBDF1 BE9BC500 6830B3FE
S:     FF879684 31D86A51 645151FA 773AD009

S:     A7BE1A69 97AD739B D8C9CA45 1F618B61
S:     A761CA9B 97BE8B45 D8AD1A61 1FC97369
S:     89D810E8 855ACE68 2D1843D8 CB128FE4
K:     D6AA74FD D2AF72FA DAA678F1 D6AB76FE
S:     5F726415 57F5BC92 F7BE3B29 1DB9F91A

S:     6353E08C 0960E104 CD70B751 BACAD0E7
S:     63CAB704 0953D051 CD60E0E7 BA70E18C
S:     00102030 40506070 8090A0B0 C0D0E0F0
K:     00010203 04050607 08090A0B 0C0D0E0F
S:     00112233 44556677 8899AABB CCDDEEFF

res:   00112233 44556677 8899AABB CCDDEEFF
*/

