/*
************** UNDER DEVELOPMENT ***************************

This will be an implementataion of the buddy system for space
allocation, based on an idea of Eben Upton.

Implemented in BCPL by Martin Richards (c) January 2003
*/

GET "testbuddy.b"

MANIFEST {
  K = 6               // K = lg of the number of cells available.
  N = 1<<K            // N = the number of cells = the maximum block size
  TreeUpb = 2*(N/8)-1 // Upper bound of the tree vector, each leaf node
                      // holds a bit map for the allocation of 8 leaf cells
  CellSize = 4        // Number of words per cell.

  // If p is a pointer to an allocated block
  // p!0 = the size of the block in words
  // p!1 = subscript of the tree element corresponding to this block.
  // p!2 = the bit mask for the block if its size is <= 8, 0 otherwise.
  // The size of allocated block is always >= 4 words.
}

GLOBAL {
  tree:300 // byte vector holding the heap structured allocation info

  tab      // If x is a leaf element of the tree it is a bit map
           // indicating which of the 8 cells corresponding to this
           // element are currently allocated. It can be used as a
           // subscript of tab to yield a 32-bit word containing packed 
           // information about these 8 memory cells.

           // If tab!x =
           //     rrrrcccc_bbbbbbbb_aaaaaaaa_qq_ppp_sss (in binary)

           //      sss = 0 if all 8 cells are allocated
           //         or 1 + lg largest block available
           // aaaaaaaa = bit pattern of a free blk of size 1  
           // bbbbbbbb = bit pattern of a free blk of size 2  
           // rrrrcccc = bit pattern of a free blk of size 4  
           //      ppp = position of the free blk of size 1  
           //      qq0 = position of the free blk of size 2
           //      r00 = position of the free blk of size 4

  space    // Vector holding CellSize*N words of free store.
}

// In the following function
// lsz = 1 + lg size of block corresponding to node i 
LET inittree(i, lsz) BE TEST lsz=4       // ie region size = 8
                        THEN tree%i := 0  // Set a leaf bitmap to zeroes
                        ELSE { tree%i := lsz
                               inittree(2*i,   lsz-1)
                               inittree(2*i+1, lsz-1)
                             }

AND init() = VALOF
{ tree  := getvec(TreeUpb/bytesperword)
  // tree is a byte vector, subscripts 0..TreeUpb
  space := getvec(N*CellSize-1)

  UNLESS tree & space RESULTIS 0

  tab := mktab() // Initialise the leaf node lookup table

  // Initialise the heap structure in tree
  tree%0 := 255
  inittree(1, K+1)

  // The children of node i are 2i and 2i+1, if any.
  // For a non leaf node i,
  // if tree%i=k+1 > 0  the largest free block in region i has size (1<<k).
  // if tree%i=0        the region is fully allocated.

  // Node 0 is special, it behaves like both the parent and sibling
  // of node 1 and holds a sentinel value to simplify programming.

  // Example:

  // For K=6,   N=64  Upb=15, and the elements tree are:

  //                       0:255                        sentinel node
  //                       1:7                          region  of size 64
  //           2:6                     3:6              regions of size 32
  //     4:5         5:5         6:5         7:5        regions of size 16
  //  8:0   9:0  10:0  11:0  12:0  13:0  14:0  15:0     leaf 8-bit maps

  // Note the space overhead is 2 bits per memory cell.

  RESULTIS 1
}

AND uninit() BE IF tree DO freevec(tree)

// alloc(size) -> res
// res  = -1   insuficient free space available
// res ~= -1   address of a block of given size (rounded up to a power of 2)

AND alloc(size) = VALOF // size = requested size in cells
{ LET i, j, pos = 1, 0, 0
  LET currsz = N  // Size of the entire allocatable memory in cells.
  LET sz     = 1  // sz will be the rounded up requested block size in cells
  LET szlim  = 16 // szlim controls the depth of tree search
  LET rlsz = 1    // rlsz = 1 + lg sz
  LET lsza = 1    // lsza = 1 + lg sz
  LET s, t, bits = ?, 0, 0
  
  // Find smallest sz that is large enough
  UNTIL sz>=size DO sz, rlsz := sz<<1, rlsz+1
  IF szlim<sz DO szlim := sz

  // Can a block of this size be allocated
  IF rlsz > tree%1 RESULTIS -1 // No suitable block available

  // Yes, allocation will be successful
  // Find the block of size sz with smallest address.

  WHILE currsz>szlim DO // szlim is at least 16
  { // Go down one level in the binary heap
    LET a, b = ?, ?
    i := i + i          // Find left child
    currsz := currsz>>1 // Blocks at this level are half the size

    a, b := tree%i, tree%(i+1)
    // Choose (hopefully) best fit
    IF a<rlsz | a>b>=rlsz DO
      i, pos := i+1, pos+currsz  // Right child is better.
  }

  j := i // Remember the position of the non leaf node

  IF sz<=8 DO
  { // Allocate a block of size 1, 2, 4 or 8
    LET k    = i+i
    LET map  = tree%k
    LET map1 = tree%(k+1)
    LET w    = tab!map
    LET w1   = tab!map1
    LET a, b = w&7, w1&7

    IF a<rlsz | a>b>=rlsz DO
    { k   := k+1         // Allocate space from right child
      map := map1
      w   := w1
      pos := pos+8
    }

    j := k
    // j = subscript of the leaf node to be used
    // map is the allocation bit map for that node
    // w is the information about this bit map

    SWITCHON sz INTO
    { CASE 1: bits := w>> 8 & 255; pos := pos + ((w>> 3) & 7); ENDCASE
      CASE 2: bits := w>>16 & 255; pos := pos + ((w>> 5) & 6); ENDCASE
      CASE 4: bits := w>>24 & 255; pos := pos + ((w>>28) & 4); ENDCASE
      CASE 8: bits := 255
    }
    //writef("bits=%b8 pos=%n*n", bits, pos) 
    map := map + bits
    ///writef("%i3: %b8*n", j, map)
    tree%j := map                   // Update leaf node bitmap
    t := tab!map & 7                // 1 + lg sz of one child
    s := tab!(tree%(j XOR 1)) & 7   // 1 + lg sz of other child
    IF t<s DO t := s
    // t = 1 + lg largest free block in region corresponding to tree%i
    //  or 0 is all 16 cells are allocated
  }

  // Update the ascestor nodes in the heap as far as necessary

  { tree%i := t           // Put t in current non leaf node
    ///writef("%i3: %n*n", i, t)
    s := tree%(i XOR 1)   // lg size of sibling
    t := s > t -> s, t    // t = lg size of larger sibling
    i := i>>1             // find parent node
  } REPEATWHILE t<tree%i

  pr(1, N, 0, pos, pos+sz, '**')

  pos   := @space!(pos*CellSize)
  pos!0 := rlsz
  pos!1 := j
  pos!2 := bits
  
  RESULTIS pos
}

AND free(p) BE
{ // free the block of size 1>>n at position p
  LET lsz  = p!0 // 1 + ls size of block being freed
  LET lsza = ?
  LET i    = p!1 // tree node relating to the block
  LET size = 1<<(lsz-1)
  LET a    = (p-space)/CellSize

  pr(1, N, 0, a, a+size, '-')

  IF lsz<=4 DO // The block has size 1, 2, 4 or 8
  { LET map = tree%i - p!2 // new leaf node bit map
    ///writef("%i3: %b8*n", i, map)
    tree%i := map
    lsz  := tab!map & 7              // 1 + lg size of region i
    lsza := tab!(tree%(i XOR 1)) & 7 // 1 + lg size of buddy region

    TEST lsz=4=lsza
    THEN { // Coallesce two 8 cell buddies
           i := i>>1              // find the parent
           lsz := 5
         }
    ELSE { // Freeing this small block has not freed a block of size 16
           IF lsz<lsza DO lsz := lsza
           // lsz = 1 + size of largest free block in region i

           // Node i cannot be coalesced with its buddy

           // Move up the tree increasing entries
           { i := i>>1                 // Select parent
             IF lsz <= tree%i RETURN
             ///writef("%i3: %n*n", i, lsz)
             tree%i := lsz             // Increase its size, if necessary.
           } REPEAT
         }
  }  

  // Mark it free, and move up the tree coalescing buddy blocks

  { // Region i is completely free.
    ///writef("%i3: %n*n", i, lsz)
    tree%i := lsz      // lsz corresponds to the full size of region i
    UNLESS tree%(i XOR 1) = lsz BREAK
    // The buddy is completely free so coallesce the blocks
    lsz, i := lsz+1, i>>1
  } REPEAT

  // Node i cannot be coalesced with its buddy

  // Make all anscestors at least lsz
  { i := i>>1                 // Select parent
    IF lsz <= tree%i RETURN
    ///writef("%i3: %n*n", i, lsz)
    tree%i := lsz             // Increase its size, if necessary.
  } REPEAT
}

AND pr(i, sz, a, p, q, ch) BE
{ LET x = tree%i
  TEST sz=8
  THEN prmap(x, 8, a, p, q, ch)
  ELSE { IF x=0 DO
         { // The region of size sz is fully allocated
           prmap(-1, sz, a, p, q, ch)
           RETURN
         }
         sz := sz>>1
         pr(i+i,   sz, a,    p, q, ch)
         pr(i+i+1, sz, a+sz, p, q, ch)
       }
}

AND prmap(bits, sz, a, p, q, ch) BE
{ //writef("bits=%i3  sz=%i2  a=%i2 p=%i2 q=%i2 ch=%c*n", bits, sz, a, p, q, ch)
  FOR i = 0 TO sz-1 DO
  { LET addr = a+i
    IF i REM 8 = 0 DO wrch(' ')
    TEST p <= addr < q
    THEN wrch(ch)
    ELSE wrch(((bits&1)=0) -> '.', 'x')
    //writef("bits=%n bits/2=%n*n", bits, bits/2)
    IF bits>0 DO bits := bits>>1
  }
  IF (a+sz) REM 64 = 0 DO newline()
}

AND start() = VALOF
{ LET a,b,c,d,e,f,g,h,i,j,k,l,m,n = 0,0,0,0,0,0,0,0,0,0,0,0,0,0
  init()
  a := alloc(32)
  b := alloc(16)
  free(a)
  m := alloc(1)
  n := alloc(2)

  a := alloc(8)
  c := alloc(4)
  d := alloc(8)
  e := alloc(1)
  h := alloc(4)
  i := alloc(4)
  free(h)
  free(c)
  c := alloc(2)
  f := alloc(2)
  free(a)
  g := alloc(1)
  free(c)
  free(b)
  free(d)
  free(i)
  free(e)
  free(g)
  free(f)
  free(m)
  free(n)
  uninit()
  RESULTIS 0
}

AND mktab() = TABLE // Function used to initialise tab
 #b00001111_00000011_00000001_00_000_100, // 00000000
 #b11110000_00001100_00000010_01_001_011, // 00000001
 #b11110000_00001100_00000001_01_000_011, // 00000010
 #b11110000_00001100_00000100_01_010_011, // 00000011
 #b11110000_00000011_00001000_00_011_011, // 00000100
 #b11110000_00110000_00000010_10_001_011, // 00000101
 #b11110000_00110000_00000001_10_000_011, // 00000110
 #b11110000_00110000_00001000_10_011_011, // 00000111
 #b11110000_00000011_00000100_00_010_011, // 00001000
 #b11110000_00110000_00000010_10_001_011, // 00001001
 #b11110000_00110000_00000001_10_000_011, // 00001010
 #b11110000_00110000_00000100_10_010_011, // 00001011
 #b11110000_00000011_00000001_00_000_011, // 00001100
 #b11110000_00110000_00000010_10_001_011, // 00001101
 #b11110000_00110000_00000001_10_000_011, // 00001110
 #b11110000_00110000_00010000_10_100_011, // 00001111
 #b00001111_11000000_00100000_11_101_011, // 00010000
 #b00000000_00001100_00000010_01_001_010, // 00010001
 #b00000000_00001100_00000001_01_000_010, // 00010010
 #b00000000_00001100_00100000_01_101_010, // 00010011
 #b00000000_00000011_00001000_00_011_010, // 00010100
 #b00000000_11000000_00000010_11_001_010, // 00010101
 #b00000000_11000000_00000001_11_000_010, // 00010110
 #b00000000_11000000_00001000_11_011_010, // 00010111
 #b00000000_00000011_00000100_00_010_010, // 00011000
 #b00000000_11000000_00000010_11_001_010, // 00011001
 #b00000000_11000000_00000001_11_000_010, // 00011010
 #b00000000_11000000_00000100_11_010_010, // 00011011
 #b00000000_00000011_00100000_00_101_010, // 00011100
 #b00000000_11000000_00000010_11_001_010, // 00011101
 #b00000000_11000000_00000001_11_000_010, // 00011110
 #b00000000_11000000_00100000_11_101_010, // 00011111
 #b00001111_11000000_00010000_11_100_011, // 00100000
 #b00000000_00001100_00000010_01_001_010, // 00100001
 #b00000000_00001100_00000001_01_000_010, // 00100010
 #b00000000_00001100_00010000_01_100_010, // 00100011
 #b00000000_00000011_00001000_00_011_010, // 00100100
 #b00000000_11000000_00000010_11_001_010, // 00100101
 #b00000000_11000000_00000001_11_000_010, // 00100110
 #b00000000_11000000_00001000_11_011_010, // 00100111
 #b00000000_00000011_00000100_00_010_010, // 00101000
 #b00000000_11000000_00000010_11_001_010, // 00101001
 #b00000000_11000000_00000001_11_000_010, // 00101010
 #b00000000_11000000_00000100_11_010_010, // 00101011
 #b00000000_00000011_00010000_00_100_010, // 00101100
 #b00000000_11000000_00000010_11_001_010, // 00101101
 #b00000000_11000000_00000001_11_000_010, // 00101110
 #b00000000_11000000_00010000_11_100_010, // 00101111
 #b00001111_11000000_01000000_11_110_011, // 00110000
 #b00000000_00001100_00000010_01_001_010, // 00110001
 #b00000000_00001100_00000001_01_000_010, // 00110010
 #b00000000_00001100_00000100_01_010_010, // 00110011
 #b00000000_00000011_00001000_00_011_010, // 00110100
 #b00000000_11000000_00000010_11_001_010, // 00110101
 #b00000000_11000000_00000001_11_000_010, // 00110110
 #b00000000_11000000_00001000_11_011_010, // 00110111
 #b00000000_00000011_00000100_00_010_010, // 00111000
 #b00000000_11000000_00000010_11_001_010, // 00111001
 #b00000000_11000000_00000001_11_000_010, // 00111010
 #b00000000_11000000_00000100_11_010_010, // 00111011
 #b00000000_00000011_00000001_00_000_010, // 00111100
 #b00000000_11000000_00000010_11_001_010, // 00111101
 #b00000000_11000000_00000001_11_000_010, // 00111110
 #b00000000_11000000_01000000_11_110_010, // 00111111
 #b00001111_00110000_10000000_10_111_011, // 01000000
 #b00000000_00001100_00000010_01_001_010, // 01000001
 #b00000000_00001100_00000001_01_000_010, // 01000010
 #b00000000_00001100_10000000_01_111_010, // 01000011
 #b00000000_00000011_00001000_00_011_010, // 01000100
 #b00000000_00110000_00000010_10_001_010, // 01000101
 #b00000000_00110000_00000001_10_000_010, // 01000110
 #b00000000_00110000_00001000_10_011_010, // 01000111
 #b00000000_00000011_00000100_00_010_010, // 01001000
 #b00000000_00110000_00000010_10_001_010, // 01001001
 #b00000000_00110000_00000001_10_000_010, // 01001010
 #b00000000_00110000_00000100_10_010_010, // 01001011
 #b00000000_00000011_10000000_00_111_010, // 01001100
 #b00000000_00110000_00000010_10_001_010, // 01001101
 #b00000000_00110000_00000001_10_000_010, // 01001110
 #b00000000_00110000_10000000_10_111_010, // 01001111
 #b00001111_00000011_00100000_00_101_011, // 01010000
 #b00000000_00001100_00000010_01_001_010, // 01010001
 #b00000000_00001100_00000001_01_000_010, // 01010010
 #b00000000_00001100_00100000_01_101_010, // 01010011
 #b00000000_00000011_00001000_00_011_010, // 01010100
 #b00000000_00000000_00000010_00_001_001, // 01010101
 #b00000000_00000000_00000001_00_000_001, // 01010110
 #b00000000_00000000_00001000_00_011_001, // 01010111
 #b00000000_00000011_00000100_00_010_010, // 01011000
 #b00000000_00000000_00000010_00_001_001, // 01011001
 #b00000000_00000000_00000001_00_000_001, // 01011010
 #b00000000_00000000_00000100_00_010_001, // 01011011
 #b00000000_00000011_00100000_00_101_010, // 01011100
 #b00000000_00000000_00000010_00_001_001, // 01011101
 #b00000000_00000000_00000001_00_000_001, // 01011110
 #b00000000_00000000_00100000_00_101_001, // 01011111
 #b00001111_00000011_00010000_00_100_011, // 01100000
 #b00000000_00001100_00000010_01_001_010, // 01100001
 #b00000000_00001100_00000001_01_000_010, // 01100010
 #b00000000_00001100_00010000_01_100_010, // 01100011
 #b00000000_00000011_00001000_00_011_010, // 01100100
 #b00000000_00000000_00000010_00_001_001, // 01100101
 #b00000000_00000000_00000001_00_000_001, // 01100110
 #b00000000_00000000_00001000_00_011_001, // 01100111
 #b00000000_00000011_00000100_00_010_010, // 01101000
 #b00000000_00000000_00000010_00_001_001, // 01101001
 #b00000000_00000000_00000001_00_000_001, // 01101010
 #b00000000_00000000_00000100_00_010_001, // 01101011
 #b00000000_00000011_00010000_00_100_010, // 01101100
 #b00000000_00000000_00000010_00_001_001, // 01101101
 #b00000000_00000000_00000001_00_000_001, // 01101110
 #b00000000_00000000_00010000_00_100_001, // 01101111
 #b00001111_00000011_10000000_00_111_011, // 01110000
 #b00000000_00001100_00000010_01_001_010, // 01110001
 #b00000000_00001100_00000001_01_000_010, // 01110010
 #b00000000_00001100_10000000_01_111_010, // 01110011
 #b00000000_00000011_00001000_00_011_010, // 01110100
 #b00000000_00000000_00000010_00_001_001, // 01110101
 #b00000000_00000000_00000001_00_000_001, // 01110110
 #b00000000_00000000_00001000_00_011_001, // 01110111
 #b00000000_00000011_00000100_00_010_010, // 01111000
 #b00000000_00000000_00000010_00_001_001, // 01111001
 #b00000000_00000000_00000001_00_000_001, // 01111010
 #b00000000_00000000_00000100_00_010_001, // 01111011
 #b00000000_00000011_10000000_00_111_010, // 01111100
 #b00000000_00000000_00000010_00_001_001, // 01111101
 #b00000000_00000000_00000001_00_000_001, // 01111110
 #b00000000_00000000_10000000_00_111_001, // 01111111
 #b00001111_00110000_01000000_10_110_011, // 10000000
 #b00000000_00001100_00000010_01_001_010, // 10000001
 #b00000000_00001100_00000001_01_000_010, // 10000010
 #b00000000_00001100_01000000_01_110_010, // 10000011
 #b00000000_00000011_00001000_00_011_010, // 10000100
 #b00000000_00110000_00000010_10_001_010, // 10000101
 #b00000000_00110000_00000001_10_000_010, // 10000110
 #b00000000_00110000_00001000_10_011_010, // 10000111
 #b00000000_00000011_00000100_00_010_010, // 10001000
 #b00000000_00110000_00000010_10_001_010, // 10001001
 #b00000000_00110000_00000001_10_000_010, // 10001010
 #b00000000_00110000_00000100_10_010_010, // 10001011
 #b00000000_00000011_01000000_00_110_010, // 10001100
 #b00000000_00110000_00000010_10_001_010, // 10001101
 #b00000000_00110000_00000001_10_000_010, // 10001110
 #b00000000_00110000_01000000_10_110_010, // 10001111
 #b00001111_00000011_00100000_00_101_011, // 10010000
 #b00000000_00001100_00000010_01_001_010, // 10010001
 #b00000000_00001100_00000001_01_000_010, // 10010010
 #b00000000_00001100_00100000_01_101_010, // 10010011
 #b00000000_00000011_00001000_00_011_010, // 10010100
 #b00000000_00000000_00000010_00_001_001, // 10010101
 #b00000000_00000000_00000001_00_000_001, // 10010110
 #b00000000_00000000_00001000_00_011_001, // 10010111
 #b00000000_00000011_00000100_00_010_010, // 10011000
 #b00000000_00000000_00000010_00_001_001, // 10011001
 #b00000000_00000000_00000001_00_000_001, // 10011010
 #b00000000_00000000_00000100_00_010_001, // 10011011
 #b00000000_00000011_00100000_00_101_010, // 10011100
 #b00000000_00000000_00000010_00_001_001, // 10011101
 #b00000000_00000000_00000001_00_000_001, // 10011110
 #b00000000_00000000_00100000_00_101_001, // 10011111
 #b00001111_00000011_00010000_00_100_011, // 10100000
 #b00000000_00001100_00000010_01_001_010, // 10100001
 #b00000000_00001100_00000001_01_000_010, // 10100010
 #b00000000_00001100_00010000_01_100_010, // 10100011
 #b00000000_00000011_00001000_00_011_010, // 10100100
 #b00000000_00000000_00000010_00_001_001, // 10100101
 #b00000000_00000000_00000001_00_000_001, // 10100110
 #b00000000_00000000_00001000_00_011_001, // 10100111
 #b00000000_00000011_00000100_00_010_010, // 10101000
 #b00000000_00000000_00000010_00_001_001, // 10101001
 #b00000000_00000000_00000001_00_000_001, // 10101010
 #b00000000_00000000_00000100_00_010_001, // 10101011
 #b00000000_00000011_00010000_00_100_010, // 10101100
 #b00000000_00000000_00000010_00_001_001, // 10101101
 #b00000000_00000000_00000001_00_000_001, // 10101110
 #b00000000_00000000_00010000_00_100_001, // 10101111
 #b00001111_00000011_01000000_00_110_011, // 10110000
 #b00000000_00001100_00000010_01_001_010, // 10110001
 #b00000000_00001100_00000001_01_000_010, // 10110010
 #b00000000_00001100_01000000_01_110_010, // 10110011
 #b00000000_00000011_00001000_00_011_010, // 10110100
 #b00000000_00000000_00000010_00_001_001, // 10110101
 #b00000000_00000000_00000001_00_000_001, // 10110110
 #b00000000_00000000_00001000_00_011_001, // 10110111
 #b00000000_00000011_00000100_00_010_010, // 10111000
 #b00000000_00000000_00000010_00_001_001, // 10111001
 #b00000000_00000000_00000001_00_000_001, // 10111010
 #b00000000_00000000_00000100_00_010_001, // 10111011
 #b00000000_00000011_01000000_00_110_010, // 10111100
 #b00000000_00000000_00000010_00_001_001, // 10111101
 #b00000000_00000000_00000001_00_000_001, // 10111110
 #b00000000_00000000_01000000_00_110_001, // 10111111
 #b00001111_00110000_00010000_10_100_011, // 11000000
 #b00000000_00001100_00000010_01_001_010, // 11000001
 #b00000000_00001100_00000001_01_000_010, // 11000010
 #b00000000_00001100_00000100_01_010_010, // 11000011
 #b00000000_00000011_00001000_00_011_010, // 11000100
 #b00000000_00110000_00000010_10_001_010, // 11000101
 #b00000000_00110000_00000001_10_000_010, // 11000110
 #b00000000_00110000_00001000_10_011_010, // 11000111
 #b00000000_00000011_00000100_00_010_010, // 11001000
 #b00000000_00110000_00000010_10_001_010, // 11001001
 #b00000000_00110000_00000001_10_000_010, // 11001010
 #b00000000_00110000_00000100_10_010_010, // 11001011
 #b00000000_00000011_00000001_00_000_010, // 11001100
 #b00000000_00110000_00000010_10_001_010, // 11001101
 #b00000000_00110000_00000001_10_000_010, // 11001110
 #b00000000_00110000_00010000_10_100_010, // 11001111
 #b00001111_00000011_00100000_00_101_011, // 11010000
 #b00000000_00001100_00000010_01_001_010, // 11010001
 #b00000000_00001100_00000001_01_000_010, // 11010010
 #b00000000_00001100_00100000_01_101_010, // 11010011
 #b00000000_00000011_00001000_00_011_010, // 11010100
 #b00000000_00000000_00000010_00_001_001, // 11010101
 #b00000000_00000000_00000001_00_000_001, // 11010110
 #b00000000_00000000_00001000_00_011_001, // 11010111
 #b00000000_00000011_00000100_00_010_010, // 11011000
 #b00000000_00000000_00000010_00_001_001, // 11011001
 #b00000000_00000000_00000001_00_000_001, // 11011010
 #b00000000_00000000_00000100_00_010_001, // 11011011
 #b00000000_00000011_00100000_00_101_010, // 11011100
 #b00000000_00000000_00000010_00_001_001, // 11011101
 #b00000000_00000000_00000001_00_000_001, // 11011110
 #b00000000_00000000_00100000_00_101_001, // 11011111
 #b00001111_00000011_00010000_00_100_011, // 11100000
 #b00000000_00001100_00000010_01_001_010, // 11100001
 #b00000000_00001100_00000001_01_000_010, // 11100010
 #b00000000_00001100_00010000_01_100_010, // 11100011
 #b00000000_00000011_00001000_00_011_010, // 11100100
 #b00000000_00000000_00000010_00_001_001, // 11100101
 #b00000000_00000000_00000001_00_000_001, // 11100110
 #b00000000_00000000_00001000_00_011_001, // 11100111
 #b00000000_00000011_00000100_00_010_010, // 11101000
 #b00000000_00000000_00000010_00_001_001, // 11101001
 #b00000000_00000000_00000001_00_000_001, // 11101010
 #b00000000_00000000_00000100_00_010_001, // 11101011
 #b00000000_00000011_00010000_00_100_010, // 11101100
 #b00000000_00000000_00000010_00_001_001, // 11101101
 #b00000000_00000000_00000001_00_000_001, // 11101110
 #b00000000_00000000_00010000_00_100_001, // 11101111
 #b00001111_00000011_00000001_00_000_011, // 11110000
 #b00000000_00001100_00000010_01_001_010, // 11110001
 #b00000000_00001100_00000001_01_000_010, // 11110010
 #b00000000_00001100_00000100_01_010_010, // 11110011
 #b00000000_00000011_00001000_00_011_010, // 11110100
 #b00000000_00000000_00000010_00_001_001, // 11110101
 #b00000000_00000000_00000001_00_000_001, // 11110110
 #b00000000_00000000_00001000_00_011_001, // 11110111
 #b00000000_00000011_00000100_00_010_010, // 11111000
 #b00000000_00000000_00000010_00_001_001, // 11111001
 #b00000000_00000000_00000001_00_000_001, // 11111010
 #b00000000_00000000_00000100_00_010_001, // 11111011
 #b00000000_00000011_00000001_00_000_010, // 11111100
 #b00000000_00000000_00000010_00_001_001, // 11111101
 #b00000000_00000000_00000001_00_000_001, // 11111110
 #b00000000_00000000_00000000_00_000_000  // 11111111




/*
The output of this program is:

 ******** ******** ******** ******** ........ ........ ........ ........
 xxxxxxxx xxxxxxxx xxxxxxxx xxxxxxxx ******** ******** ........ ........
 -------- -------- -------- -------- xxxxxxxx xxxxxxxx ........ ........
 ........ ........ ........ ........ xxxxxxxx xxxxxxxx *....... ........
 ........ ........ ........ ........ xxxxxxxx xxxxxxxx x.**.... ........
 ........ ........ ........ ........ xxxxxxxx xxxxxxxx x.xx.... ********
 ........ ........ ........ ........ xxxxxxxx xxxxxxxx x.xx**** xxxxxxxx
 ******** ........ ........ ........ xxxxxxxx xxxxxxxx x.xxxxxx xxxxxxxx
 xxxxxxxx ........ ........ ........ xxxxxxxx xxxxxxxx x*xxxxxx xxxxxxxx
 xxxxxxxx ****.... ........ ........ xxxxxxxx xxxxxxxx xxxxxxxx xxxxxxxx
 xxxxxxxx xxxx**** ........ ........ xxxxxxxx xxxxxxxx xxxxxxxx xxxxxxxx
 xxxxxxxx ----xxxx ........ ........ xxxxxxxx xxxxxxxx xxxxxxxx xxxxxxxx
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxx---- xxxxxxxx
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxx**.. xxxxxxxx
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxxxx** xxxxxxxx
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxxxxxx --------
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxxxxxx *.......
 xxxxxxxx ....xxxx ........ ........ xxxxxxxx xxxxxxxx xxxx--xx x.......
 xxxxxxxx ....xxxx ........ ........ -------- -------- xxxx..xx x.......
 -------- ....xxxx ........ ........ ........ ........ xxxx..xx x.......
 ........ ....---- ........ ........ ........ ........ xxxx..xx x.......
 ........ ........ ........ ........ ........ ........ x-xx..xx x.......
 ........ ........ ........ ........ ........ ........ x.xx..xx -.......
 ........ ........ ........ ........ ........ ........ x.xx..-- ........
 ........ ........ ........ ........ ........ ........ -.xx.... ........
 ........ ........ ........ ........ ........ ........ ..--.... ........
*/
