/*

This will eventually be a BCPL implementation of an emulator of the
Z80 microprocessor. Its human interaction will be based on the
interactive debugger of the BCPL Cintcode system. This emulator has
features that the debugging of code generated by the BCPL to Z80
compiler z80bcpl.b. Only implements the Z80 instructions needed for
this purpose. Additional instructions can easily be added if needed.

Implemented by Martin Richards (c) 5 Oct 2020

Usage: FROM,TO/K,GLBSIZE,STKSIZE,d/S
FROM    The file name of an optional Intel Hex file to load.
TO      Output to file rather than the screen
GLBSIZE The size in words of the root global vector, default size 768.
STKSIZE The size in words of the root stack, default size 4000.
-d      Turn on debugging output


All memory locations are first initialised with the value #xDEAD.
The global vector elements are initialised with #xC000+gn where gn is
the global number, except global zero is initialise with GLBSIZE.
All stack elements are initialised with the value #xABCD.
If FROM is specified the register PC is set to the value of global 1.

IY is normally set to the byte address of global 64.
IX is normally set to the byte address of word 64 relative to the
   current BCPL P pointer.

The BCPL function stack frame is as follows:

          ----------------------------------------------------
         |   P' |  L | A1 | A2 | A3 | more arguments and locals  
          ----------------------------------------------------
             ^
             |
             P

P     The BCPL stack frame pointer of the current function
P'    The BCPL stack frame pointer of the enclosing function or zero.
L     The return byte address.
A1, A2,... The function arguments and locals
      A1 is passed in HL'  ie in the alternate register set
      A2 is passed in DE'
      A3 is passed in BC'

HL is the function entry address

The first instruction of a function is:
RST #x38 SETLINK0   if there are no arguments
RST #x38 SETLINK1   if there awas one argument
RST #x38 SETLINK2   if there were two arguments
RST #x38 SETLINK3   if there were three or more arguments

The function result is returned in HL

History

09/09/2022
Making extensive changes, still in process.

05/10/2020
Started the initial implementation.

*/

GET "libhdr"

/*
// All the instructions compiled by bcplcgz80
// Clearly z80emu.b must implement all of them.

        CASE i_inc    :  RESULTIS  "    INC   %S"
        CASE i_dec    :  RESULTIS  "    DEC   %S"
        CASE i_and    :  RESULTIS  "    AND   %S"
        CASE i_or     :  RESULTIS  "    OR    %S"
        CASE i_xor    :  RESULTIS  "    XOR   %S"
        CASE i_sbchl  :  RESULTIS  "    SBC   HL,%S"
        CASE i_addhl  :  RESULTIS  "    ADD   HL,%S"
        CASE i_ldrr   :  RESULTIS  "    LD    %S,%S"
        CASE i_cpl    :  RESULTIS  "    CPL"
        CASE i_ldrn   :  RESULTIS  "    LD    %S,#X%X2"
        CASE i_ldrnn  :  RESULTIS  "    LD    %S,#X%X4"
        CASE i_ldrll  :  RESULTIS  "    LD    %S,L%N"
        CASE i_push   :  RESULTIS  "    PUSH  %S"
        CASE i_pop    :  RESULTIS  "    POP   %S"
        CASE i_ret    :  RESULTIS  "    RET"
        CASE i_jpihl  :  RESULTIS  "    JP    (HL)"
        CASE i_exx    :  RESULTIS  "    EXX"
        CASE i_jr     :  RESULTIS  "    JR    L%N-$"
        CASE i_jrz    :  RESULTIS  "    JR    Z,L%N-$"
        CASE i_jrnz   :  RESULTIS  "    JR    NZ,L%N-$"
        CASE i_jrc    :  RESULTIS  "    JR    C,L%N-$"
        CASE i_jrnc   :  RESULTIS  "    JR    NC,L%N-$"
        CASE i_jp     :  RESULTIS  "    JP    L%N"
        CASE i_jpz    :  RESULTIS  "    JP    Z,L%N"
        CASE i_jpnz   :  RESULTIS  "    JP    NZ,L%N"
        CASE i_jpc    :  RESULTIS  "    JP    C,L%N"
        CASE i_jpnc   :  RESULTIS  "    JP    NC,L%N"
        CASE i_jpp    :  RESULTIS  "    JP    P,L%N"
        CASE i_jpm    :  RESULTIS  "    JP    M,L%N"
        CASE i_ldri   :  RESULTIS  "    LD    %S,(%S%S%N)"
        CASE i_ldir   :  RESULTIS  "    LD    (%S%S%N),%S"
        CASE i_ldhll  :  RESULTIS  "    LD    HL,(L%N)"
        CASE i_ldrl   :  RESULTIS  "    LD    %S,(L%N)"
        CASE i_srl    :  RESULTIS  "    SRL   %S"
        CASE i_rr     :  RESULTIS  "    RR    %S"
        CASE i_ldlhl  :  RESULTIS  "    LD    (L%N),HL"
        CASE i_ldlr   :  RESULTIS  "    LD    (L%N),%S"
        CASE i_nop    :  RESULTIS  "    NOP"
        CASE i_cpir   :  RESULTIS  "    CPIR"

        CASE i_rst38  :  RESULTIS  "    RST   #X38      " // MR 20/08/2022

// RST arguments used by bcplcgz80.b
        CASE m_apply      :  RESULTIS  "M_APPLY"
        CASE m_fixup      :  RESULTIS  "M_FIXUP"
        CASE m_loadix     :  RESULTIS  "M_LOADIX"
        CASE m_loadiy     :  RESULTIS  "M_LOADIY"
        CASE m_storeix    :  RESULTIS  "M_STOREIX"
        CASE m_storeiy    :  RESULTIS  "M_STOREIY"  
        CASE m_setlink0   :  RESULTIS  "M_SETLINK0"
        CASE m_setlink1   :  RESULTIS  "M_SETLINK1"
        CASE m_setlink2   :  RESULTIS  "M_SETLINK2"
        CASE m_setlink3   :  RESULTIS  "M_SETLINK3"
        CASE m_return     :  RESULTIS  "M_RETURN"
        CASE m_finish     :  RESULTIS  "M_FINISH"
        CASE m_loadlvix   :  RESULTIS  "M_LOADLVIX"
        CASE m_loadlviy   :  RESULTIS  "M_LOADLVIY"
        CASE m_multiply   :  RESULTIS  "M_MULTIPLY"
        CASE m_divide     :  RESULTIS  "M_DIVIDE"
        CASE m_rem        :  RESULTIS  "M_REM"
        CASE m_lshift     :  RESULTIS  "M_LSHIFT"
        CASE m_rshift     :  RESULTIS  "M_RSHIFT"
        CASE m_eq         :  RESULTIS  "M_EQ"
        CASE m_ne         :  RESULTIS  "M_NE"
        CASE m_ls         :  RESULTIS  "M_LS"
        CASE m_gr         :  RESULTIS  "M_GR"
        CASE m_le         :  RESULTIS  "M_LE"
        CASE m_ge         :  RESULTIS  "M_GE"
        CASE m_rdivide    :  RESULTIS  "M_RDIVIDE"
        CASE m_rrem       :  RESULTIS  "M_RREM"
        CASE m_rlshift    :  RESULTIS  "M_RLSHIFT"
        CASE m_rrshift    :  RESULTIS  "M_RRSHIFT"
        CASE m_abs        :  RESULTIS  "M_ABS"
        CASE m_linsearch  :  RESULTIS  "M_LINSEARCH"

        CASE m_default    :  RESULTIS  "M_DEFAULT"
*/

MANIFEST
$( // Taken from bcplchz80.b to help show all the Z80 istructions
   // that the compiler might generate.
   
    dupl0                 =  #B000000000 
    dupl1                 =  #B100000000 

    i_inc                 =  #B00000011 // INC   %S
    i_dec                 =  #B00001011 // DEC   %S
    i_and                 =  #B10100000 // AND   %S
    i_or                  =  #B10110000 // OR    %S
    i_xor                 =  #B10101000 // XOR   %S
    i_sbchl               =  #B01000010 // SBC   HL,%S
    i_addhl               =  #B00001001 // ADD   HL,%S
    i_ldrr                =  #B01000000 // LD    %S,%S
    i_cpl                 =  #B00101111 // CPL
    i_ldrn                =  #B00000110 // LD    %S,#X%X2
    i_ldrnn               =  #B00000001 + dupl0  // LD    %S,#X%X4
    i_ldrll               =  #B00000001 + dupl1  // LD    %S,L%N
    i_push                =  #B11000101 // PUSH  %S
    i_pop                 =  #B11000001 // POP   %S
    i_ret                 =  #B11001001 // RET
    i_jpihl               =  #B11101001 // JP    (HL)
    i_exx                 =  #B11011001 // EXX
    i_jr                  =  #B00011000 + dupl0  // JR    L%N-$
    i_jrz                 =  #B00101000 // JR    Z,L%N-$
    i_jrnz                =  #B00100000 // JR    NZ,L%N-$
    i_jrc                 =  #B00111000 + dupl0  // JR    C,L%N-$
    i_jrnc                =  #B00110000 // JR    NC,L%N-$
    i_jp                  =  #B11000011 // JP    L%N
    i_jpz                 =  #B11001010 // JP    Z,L%N
    i_jpnz                =  #B11000010 // JP    NZ,L%N
    i_jpnc                =  #B11010010 // JP    NC,L%N
    i_jpc                 =  #B11011010 // JP    C,L%N"
    i_jpp                 =  #B11110010 // JP    P,L%N"
    i_jpm                 =  #B11111010 // JP    M,L%N"
    i_ldri                =  #B01000110 // LD    %S,(%S%S%N)
    i_ldir                =  #B01110000 // LD    (%S%S%N),%S
    i_ldhll               =  #B00101010 // LD    HL,(L%N)
    i_ldrl                =  #B01001011 // LD    %S,(L%N)
    i_srl                 =  #B00111000 + dupl1  // SRL   %S
    i_rr                  =  #B00011000 + dupl1  // RR    %S
    i_ldlhl               =  #B00100010 // LD    (L%N),HL
    i_ldlr                =  #B01000011 // LD    (L%N),%S
    i_cpir                =  #B10110001 // CPIR
    i_nop                 =  #B00000000 // NOP
    i_rst38               =  #B11111111 // RST   #X38

    m_jpentrysize         =  3

    m_apply               =  0  * m_jpentrysize
    m_fixup               =  1  * m_jpentrysize
    m_loadix              =  2  * m_jpentrysize
    m_loadiy              =  3  * m_jpentrysize
    m_storeix             =  4  * m_jpentrysize
    m_storeiy             =  5  * m_jpentrysize
    m_setlink0            =  6  * m_jpentrysize
    m_setlink1            =  7  * m_jpentrysize
    m_setlink2            =  8  * m_jpentrysize
    m_setlink3            =  9  * m_jpentrysize        
    m_return              =  10 * m_jpentrysize         
    m_finish              =  11 * m_jpentrysize         
    m_loadlvix            =  12 * m_jpentrysize        
    m_loadlviy            =  13 * m_jpentrysize        
    m_multiply            =  14 * m_jpentrysize        
    m_divide              =  15 * m_jpentrysize        
    m_rem                 =  16 * m_jpentrysize        
    m_lshift              =  17 * m_jpentrysize        
    m_rshift              =  18 * m_jpentrysize        
    m_eq                  =  19 * m_jpentrysize        
    m_ne                  =  20 * m_jpentrysize        
    m_ls                  =  21 * m_jpentrysize        
    m_gr                  =  22 * m_jpentrysize        
    m_le                  =  23 * m_jpentrysize        
    m_ge                  =  24 * m_jpentrysize        
    m_rdivide             =  25 * m_jpentrysize        
    m_rrem                =  26 * m_jpentrysize        
    m_rlshift             =  27 * m_jpentrysize        
    m_rrshift             =  28 * m_jpentrysize        
    m_abs                 =  29 * m_jpentrysize        
    m_linsearch           =  30 * m_jpentrysize

    m_sys                 =  31 * m_jpentrysize // Added for z80emu
$)

GLOBAL {
stdin:ug
stdout
debug       // = TRUE or FALSE

z80memupbb  // Typically = #xFFFF
z80memupb   // Typically = #x7FFF
mem         // A vector holding the 64K bytes of the Z80 memory

glbsize
stksize
glbupb
stkupb
rootgvec
rootstack
codevec
codep

gptr
pptr

pflag         // =TRUE for printing, =FALSE for executing
doz80instr    // (a)   Print or execute one instruction
prz80instr    // (pc)  Print the instruction at pc

faultcode     // Value returned by doz80instr
indentpos     // Used by prstate()

zerror
r2str

// The variables holding the Z80 registers
r_a;  r_b;  r_c;  r_d;  r_e;  r_f;  r_h;  r_l   // The main registers
r_a1; r_b1; r_c1; r_d1; r_e1; r_f1; r_h1; r_l1  // The alternate set
r_ix; r_iy; r_sp; r_pc
r_i;  r_d   // Interrupt page register and the memory refresh register

// Bits in F
r_fs; r_fs1
r_fz; r_fz1
r_fh; r_fh1
r_fv; r_fv1
r_fn; r_fn1
r_fc; r_fc1

fromfilename
tofilename
checkaddr
cont
error

getbb
getsbb
getwb
getww

putbb
putwb
putww

signextb
signextw

getr
putr

praddr

rdval
rdvaraddr
rch

bpt        // The current breakpoint number or -1
bpt_addr   // Vector of breakpoint PC values
bpt_instr  // Vector of breakpoint first bytes (op codes)
brkstep    // =TRUE when resuming from a breakpoint
singlestep   // =TRUE when single stepping (\ command)

gptr       // Currently selected G pointer = (IY-128 >> 2
pptr       // Currently selected P pointer = (IX-128) >> 2

count      // A count decremented every time a Z80 instruction is executed.
           // Fault 2 occurs when count reaches zero.
oldcount

finval     // The value to be returned by start

finp       // Finish label
finl

recp       // Recovery label
recl

retp       // The label to cause a return from sadebug.
retl       // This will cause the next Z80 instruction to
           // be executed.

style
fullstatus
val        // Current value
vars       // Vector of variables (V1 .. V9)

ch         // Current charater, capitalised
lch        // Current charater

loadhex    // (filename)  Load an Intel Hex file and set the globals defined.

z80dumpmem // Dump the Z80 registers and memory to Z80DUMP in a compacted
           // form using rn length encoding.

prflags    // (f) print the flag bits

// Execution functions that print or execute the instruction
// at a specified location depending on the setting of pflag.
// The result is the size in bytes of the instruction.

doz80instrCB  // (a)   Print or execute one instruction with op BC
doz80instrED  // (a)   Print or execute one instruction with op EB
doz80instrXD  // (a)   Print or execute one instruction with op DD or FD
doz80RST38    // {a}   Print or execute the instruction RST #x38

// Fuctions to print or execute specific instructions depending
// on the setting of pflag.

// The op is in capitals, followed by w for 16 bit operations
// and operands f the form:
//    r    r_a,
//    ir   (BC), (DE) (HL), (SP)
//    ird  (IX+d) (IY+d)
//    n    n in range 0 to 255
//    nn   nn in range 0 to 65535

z80ADCrird
z80ADCrn
z80ADCrr
z80ADCr
z80ADDrird
z80ADDrn
z80ADDrr
z80ADD16rr
z80ADDr
z80ANDird
z80ANDn
z80ANDrr
z80ANDr
z80BITbird
z80BITbr
z80BITbro
z80CALLnn
z80CCF
z80CCP
z80CPD
z80CPDR
z80CPI
z80CPIR
z80CPL
z80CPird
z80CPn
z80CPr
z80DAA
z80DAAr
z80DECird
z80DECr
z80DEC16r
z80DI
z80DJNZo
z80EI
z80EXX
z80EXrr
z80HALT
z80IM
z80IMk
z80INCird
z80INCr
z80INC16r
z80IND
z80INDR
z80INI
z80INIR
z80INrin
z80INrr
z80JPr
z80JPnn
z80JRra
z80LDD
z80LDDR
z80LDI
z80LDIR
z80LDirdr
z80LDinnr
z80LDrird
z80LDrinn
z80LDrn
z80LDrr
z80LD16rnn
z80LDwr
z80MULUWrr
z80NEG
z80NOP
z80ORn
z80ORr
z80OTDR
z80OTIR
z80OUTD
z80OUTI
z80OUTrr
z80POP16r
z80PUSH16r
z80RCCird
z80RCCr
z80RESbird
z80RESbr
z80RET
z80RETI
z80RETN
z80RETr
z80RLA
z80RLCA
z80RLCird
z80RLCr
z80RLD
z80RLird
z80RLr
z80RRA
z80RRCA
z80RRCird
z80RRD
z80RRird
z80RRr
z80RSTn
z80SBCrn
z80SBCrr
z80SBCr
z80SCF
z80SETbird
z80SETbr
z80SLAird
z80SLAr
z80SRAird
z80SRAr
z80SRLird
z80SRLr
z80SUBn
z80SUBrn
z80SUBrr
z80SUBr
z80XORn
z80XORr


// Execution functions (do not change r_pc)
// All 8 bit operations unless 16 is specified.

x_adc
x_adc16
x_add
x_add16
x_and
x_neg
x_or
x_rl
x_rlc
x_rr
x_rra
x_rrc
x_sbc
x_sbc16
x_sla
x_srl
x_sub
x_xor

x_exx


}

MANIFEST
{ // Register numbers
  n_b=0     // B
  n_c       // C
  n_d       // D
  n_e       // E
  n_h       // H
  n_l       // L
  n_f       // F
  n_a       // A

  n_af      // AF
  n_bc      // BC
  n_de      // DE
  n_hl      // HL
  n_sp      // SP
  n_pc      // PC
  n_ix      // IX
  n_iy      // IY

  n_af1     // AF'
  n_bc1     // BC'
  n_de1     // DE'
  n_hl1     // HL'

  n_ic      // (C)    Used by IN and OUT
  n_ibc     // (BC)
  n_ide     // (DE)
  n_ihl     // (HL)
  n_isp     // (SP)
  n_inn     // (nn)   used by eg LD A,(nn)

  n_a1;  n_b1; n_c1; n_d1; n_e1; n_f1; n_h1; n_l1  // The alternate set

  n_m       // M       These are needed by eg RET M  or RET NZ
  n_nz      // NZ
  n_nc      // NC
  n_p       // P
  n_pe      // PE
  n_po      // PO
  n_z       // Z
  n_i       // I
  n_r       // R

  // F register bit patterns

  fb_c = #b00000001  // carry
  fb_n = #b00000001  // add/subtract
  fb_v = #b00000100  // parity/overflow
  fb_p = #b00000100  // parity/overflow
  fb_h = #b00010000  // half carry
  fb_z = #b01000000  // zero
  fb_s = #b10000000  // sign
  
  fb_nc = #b00000001 XOR #xFF // not carry
  fb_nn = #b00000001 XOR #xFF // not add/subtract
  fb_nv = #b00000100 XOR #xFF // not parity/overflow
  fb_np = #b00000100 XOR #xFF // not parity/overflow
  fb_nh = #b00010000 XOR #xFF // not half carry
  fb_nz = #b01000000 XOR #xFF // not zero
  fb_ns = #b10000000 XOR #xFF // not sign
  
  f_brk = #xEF     // RST #x28 The instruction used for breakpoints
  f_sys = #xF7     // RST #x30 The instruction used the BCPL sys function

  tagbits    = #b_1111_1100_0000_0000
  nontagbits = #b_0000_0011_1111_1111
  globword   = #b_1010_0000_0000_0000
}


LET start() = VALOF
{ LET argv = VEC 50
  finp  := level()   // Label to be used by longjump to cause
  finl  := fin       // z80emu to terminate.

  writef("*nZ80 Emulator for BCPL code compiled by z80bcpl*n")
  writef("Type ? for help*n")

  stdin   := input()
  stdout  := output()
  debug   := FALSE
  
  z80memupbb := #xFFFF                       // The byte upb of the Z80 memory
  z80memupb  := z80memupbb>>1                // The word upb of the Z80 memory
  mem     := getvec(z80memupbb/bytesperword) // 64K bytes of the Z80 memory

  // Initialise the whole Z80 memory
  FOR a = 0 TO z80memupbb-1 BY 2 DO putwb(a, #xDEAD)

  UNLESS rdargs("FROM,TO/K,GLBSIZE/K/N,STKSIZE/K/N,-s/S,-d/S", argv, 50) DO
  { writef("Bad arguments for z80emu*n")
    RESULTIS 0
  }

  fromfilename := 0
  glbsize := 1000  // Default sizes
  stksize := 4000
  
  IF argv!0 DO fromfilename := argv!0                    // FROM

  IF argv!1 DO tofilename := argv!1                      // TO/K

  IF argv!2 DO glbsize := !(argv!2)                      // GLBSIZE/K/N
  glbupb := glbsize-1

  IF argv!3 DO stksize := !(argv!3)                      // GLBSIZE/K/N
  stkupb := stksize-1

  singlestep := argv!4                                   // -s/S

  IF argv!5 DO debug := ~debug                           // -d/S

  vars := 100 // word address of the variables vector
  FOR i = 0 TO 9 DO putww(vars+i, 0) // Set all variables to zero
  
  bpt_addr   := TABLE 0,0,0,0,0,0,0,0,0,0
  bpt_instr  := TABLE 0,0,0,0,0,0,0,0,0,0
  style      := 'F'                   // Default printing style
  fullstatus := FALSE
  val        := 0                     // Current value
  brkstep    := FALSE
  //singlestep := FALSE
  faultcode  := 0
  
  // Set the default values of the z80 registers.
  r_a,  r_b,  r_c,  r_d,  r_e,  r_h,  r_l  := 0, 0, 0, 0, 0, 0, 0
  r_a1, r_b1, r_c1, r_d1, r_e1, r_h1, r_l1 := 0, 0, 0, 0, 0, 0, 0
  r_ix, r_iy, r_sp, r_pc := 0, 0, 0, 0
  r_i,  r_d := 0, 0    // Interrupt page register and the memory refresh register

  // Individual bits of registers F and F', typically zero or set to eg fb_s
  r_fs,  r_fz,  r_fh,  r_fv,  r_fn,  r_fc  := 0, 0, 0, 0, 0, 0
  r_fs1, r_fz1, r_fh1, r_fv1, r_fn1, r_fc1 := 0, 0, 0, 0, 0, 0
  
  count := 1_000_000_000 // Decremented every time a Z80 intruction is executed.

// Allocate and initialise the root global vector
  rootgvec := 1000   // The root global vector starts at word addr 1000
  gptr := rootgvec
  putww(gptr, glbupb)
  FOR gn = 1 TO glbupb DO putww(gptr+gn, globword+gn)

  // Allocate and initialise the root stack
  rootstack := rootgvec+glbsize
    FOR a = 0 TO stkupb DO putww(rootstack+a, #xEEEE)

  // Set the location for code loaded by loadcode.
  codevec := rootstack+stksize
  codep := codevec<<1 // Byte address of where to load Intel Hex
  FOR i = 0 TO 1000 DO putww(codevec+i, #xC0DE)
  writef("gvec=%n stackbase=%n codebase=%n*n",
          rootgvec, rootstack, codevec)
  
  IF fromfilename DO loadhex(fromfilename)

  r_iy := (rootgvec <<1) + 128
  pptr := rootstack
  r_ix := (pptr<<1) + 128
  r_sp := 64000-2
  putwb(r_sp, 0)  // Set return address zero to cause a return from
                  // the z80emu command.
//  abort(4444)
  r_pc := 100          // Execution starts at byte address 100
  
  putbb(100, #xFD)     // LD L,(IY-124)    Set HL to G1 ie start
  putbb(101, #x6E)
  putbb(102, #x82)
  putbb(103, #xFD)     // LD H,(IY-125)
  putbb(104, #x66)
  putbb(105, #x83)
  putbb(106, #xFF)     // Apply 2          Enter start with return
  putbb(107, m_apply)  //                  address zero.
  putbb(108, #x02)
  putbb(109, #xFF)     // Exit from z80emu
  putbb(110, m_finish)
  
  putbb(111, #x00)
  putbb(112, #x00)

  putww(gptr+3, 113)       // Initialise sys
  
  putbb(113, #xFF)
  putbb(114, m_setlink3)   // SETLINK3
  putbb(115, #xFF)
  putbb(116, m_sys)        // SYS          Calls z80sys()
  putbb(117, #xFF)
  putbb(118, m_return)     // RETURN
  
  putbb(119, #x00)
  putbb(120, #x00)
  putbb(121, #x00)
  putbb(122, #x00)
  putbb(123, #x00)

  // Start execution in the interactive debugger.
  faultcode := 0
  brkstep := FALSE
  //singlestep := TRUE  // Normally set by the -s option
  val := 0
  pflag := FALSE
  
  writef("*nEntering sadebug before executing the *
         *first Z80 instruction of the program*n*n")
  indentpos := 0
//db("Just after initialisation")

  UNLESS singlestep GOTO execute
  
  { sadebug()

    // singlestep is TRUE if single stepping
    // brkstep is TRUE if executin the instruction at a breakpoint
    // If neither are TRUE call doz80instr repeatedly while faultcode is zero
execute:
//abort(6598)
    { //singlestep := TRUE
      //db("calling doz80instr()")
      pflag := FALSE
      faultcode := doz80instr(r_pc)
      //db("returned from doz80instr()")
    } REPEATUNTIL faultcode | singlestep | brkstep
  } REPEAT
  
fin:
  newline()
  IF mem DO freevec(mem)
}

AND parity(w) = VALOF
{ //This returns fb_v if te 8-bit pattern w has even parity
  MANIFEST { e=fb_v }
  LET t = TABLE e,0,0,e, 0,e,e,0, 0,e,e,0,e,0,0,e
  // If bbbb is an even parity 4-bit pattern then t!bbbb returns fb_v
  RESULTIS t!(w%#xF) EQV t!((w>>4)&#xF)
}

AND nsize(n) = VALOF
{ LET res = 1
  IF n<0 DO { res := res+1; n := -n }
  WHILE n > 9 DO { res := res+1; n := ABS(n/10) }
  RESULTIS res
}

AND wrd(n, d) BE
{ FOR i = nsize(n)+1 TO d DO wrch(' ')
  wrn(n)
}

AND wrn(n) BE
{ IF n<0 DO { wrch('-'); n := -n }
  wrpn(n)
}

AND wrpn(n) BE
{ IF n > 9 DO wrpn((n>>1)/5)   // Unsigned division by 10
  wrch((n + 10) MOD 10 + '0')  // Unsigned MOD 10
}

AND sadebug() BE
// Enter the standalone debugger. What it does depends on the settings
// of the global variables faultcode brkstep and singlesep.

// faultcode holds the fault code returned doz80instr when it last
//           executed an instruction. If non zero the fault is reported
//           and the execution state output followed by the debugger
//           prompt.

// brkstep   is only TRUE when the instruction at a breakpoint has just
//           been executed when resuming execution after a breakpoint.
//           If faultcode is non zero the fault is reported and the
//           execution state output followed by the debugger prompt.
//           If faultcode is zero and singlestep is FALSE, normal
//           execution of Z80 instructions resumes.

// singlstep If TRUE the execution state output followed by the
//           debugger prompt. If singlestep is FALSE only the
//           debugger prompt is output.

// When the debugger prompt has been output this function reads and
// performs debugging commands from the user. It only returns when
// another Z80 instruction needs to be executed.

{ recp := level()    // For longjump to goto the place where the 
  recl := recover    // debugger command prompt is output.

  retp := level()    // For longjump to use to cause a return from
  retl := ret        // sadebug causing another Z80 instruction to
                     // be executed.

  bpt   := -1        // No breakpoint number yet

  unsetbreakpoints()

  IF brkstep & faultcode=0 & ~singlestep DO
  { // We are resuming normal execution after the first instruction at
    // a breakpoint has just been executed.
    setbreakpoints()
    brkstep := FALSE
    RETURN
  }

  IF faultcode DO singlestep := TRUE

  // Now report the fault if faultcode is non zero
  
  IF faultcode=2 DO  // Fault caused by the BRK instruction
    FOR i = 0 TO 9 IF bpt_addr!i=val DO
    { bpt := i
      brkstep := TRUE // This must be TRUE when resuming from a breakpoint.
      writef("*n## BPT %n*n", bpt)
      indentpos := 0
      faultcode := 0
      BREAK
    }

  // Attempt to give a description of the current fault
  IF faultcode DO
  { LET mess =  VALOF SWITCHON faultcode INTO
                { CASE  -3: RESULTIS "Break from rasterp"
		  CASE   1: RESULTIS "Illegal instruction"
                  CASE   2: RESULTIS "BRK instruction"
                  CASE   3: RESULTIS "Zero count"
                  CASE   4: RESULTIS "pc out of range"
                  CASE   5: RESULTIS "Division by zero"
                  CASE  10: RESULTIS "Cintasm single step"
                  CASE  11: RESULTIS "Watch addr: %+%i7 value: %i8"
                  CASE  12: RESULTIS "Indirect address out of range: %+%+%+%n"
                  CASE  13: RESULTIS "SIGINT received"
                  CASE  14: RESULTIS "Unknown FLT op %+%n"
                  CASE  15: RESULTIS "PC out of range"
                  CASE  16: RESULTIS "P pointer out of range"
                  CASE  17: RESULTIS "SIGSEGV occurred"
                  CASE  18: RESULTIS "End of instruction tracing"
                  CASE  99: RESULTIS "User requested"
                  CASE 110: RESULTIS "Callco fault"
                  CASE 111: RESULTIS "Resumeco fault"
                  CASE 112: RESULTIS "Deleteco fault"
                  CASE 180: RESULTIS "Unable to delete a task"
                  CASE 181: RESULTIS "Unable to send a packet"
                  CASE 182: RESULTIS "Unexpected pkt received"
                  CASE 186: RESULTIS "Bad input stream"
                  CASE 187: RESULTIS "Bad output stream"
                  CASE 188: RESULTIS "Unable to replenish input"
                  CASE 189: RESULTIS "Wrch fault"
                  CASE 190: RESULTIS "Endread fault"
                  CASE 191: RESULTIS "Endwrite fault"
                  CASE 197: RESULTIS "Store chain fault"
                  DEFAULT:  RESULTIS "Unknown fault"
                }
    TEST faultcode=2
    THEN { sawritef("*n## Breakpoint at PC=%n*n", r_pc)
           brkstep := TRUE
         }
    ELSE { sawritef("*n## ABORT %n: ", faultcode)
           sawritef(mess, 0,0,0,0)
	 }
    sawrch('*n')
    indentpos := 0
  }

  prstate()

recover:
  ch := '*n'
nxt:                       // Main loop for debug commands
  IF ch='*n' DO sawritef("$ ")
  rch()
sw:
  singlestep := FALSE // Only set to TRUE by the '\' command.
  SWITCHON ch INTO

  { DEFAULT: error()

    CASE endstreamch:
    CASE 'Q': sawritef("*n"); sys(Sys_quit, 0)   // Quit
         
    CASE '*s':
    CASE '*t':
    CASE '*n': GOTO nxt

    CASE '?':
      newline()
      writes("?       Print this list of debug commands*n")
      writes("L name      Load an Intel Hex program into memory*n")
      writes("123 #o377 #FF03 'c  Constants*n")
      writes("Vn      Variables 0 to 9*n")
      writes("G Gn    Global vector and global variables*n")
      writes("P Pn    Stack frame pointer and local variables*n")
      writes("R<reg>  Where <reg> is one of*N")
      writes("        A B C D E F H L AF BC DE HL SP PC IX IY*n")
      writes("U W     Unsigned byte or word, equive to &#FF or #FFFF*n")
      writes("!e                  Subscription*n")
      writes("**e /e %e +e -e      Arithmetic operators*n")
      writes("|e &e ^e            Bit pattern operators OR, AND and XOR*n")
      writes("< >                 Logical shift left/right one place*n")
      writes("=           Print current value*n")
      writes("F           Toggle the fullstatus flag*n")
      writes("$b $c $d $e $f $o $s $u $x Set the print style*n")
      writes("Sn SR<reg>  Store val in memory or a register*n")
      writes("SPn SGn SVn Store val in a local, global or variable*n")
      writes("TRn         Trace the next n instructions*n")
      writes("TBn         Print n bytes from byte address val*n")
      writes("TS          Print the registers and current instruction*n")
      writes("TF          Print the flags register F*n")
      writes("Tn          Print n words from word address val*n")
      writes("I           Print instruction at address val*n")
      writes("N           Print instruction at address val, increment val*n")
      writes("D           Dump all of Z80 memory to DUMP.mem*n")
      writes("Q           Quit -- leave z80emu*n")
      writes("M           Set/Reset memory watch address*n")
      writes("B 0Bn eBn   List, Unset or Set breakpoints*n")
      writes("X           Execute code until the next RET instruction*n")
      writes("C           Continue normal execution*n")
      writes("\           Execute one Z80 instruction*n")
      GOTO recover

    CASE 'L':  // L <filename>
    { LET len = 0
      LET name = VEC 50
      rch() REPEATWHILE ch='*s'
      WHILE 'A'<=ch<='Z' |
            '0'<=ch<='9' |
	    ch='-' | ch='_' | ch='.' | ch='/' DO
      { len := len+1
        IF len <= 50 * bytesperword DO name%len := lch
	rch()
      }
      name%0 := len
      writef("Loading Intel Hex file: %s*n", name)
      loadhex(name)
      GOTO sw
    }
    
    CASE '0': CASE '1': CASE '2':
    CASE '3': CASE '4': CASE '5':
    CASE '6': CASE '7': CASE '8':
    CASE '9': CASE '#': CASE '*'':
    CASE 'G': CASE 'P': CASE 'R':
    CASE 'V':
              val := rdval();                     GOTO sw

    CASE 'U': val := val & #xFF;                  GOTO recover
    CASE 'W': val := val & #xFFFF;                GOTO recover

    CASE '!': rch(); val := cont(val + rdval());  GOTO sw
    CASE '+': rch(); val :=      val + rdval();   GOTO sw
    CASE '-': rch(); val :=      val - rdval();   GOTO sw
    CASE '**':rch(); val :=      val * rdval();   GOTO sw
    CASE '/': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val / a
                       GOTO sw
                     }
    CASE '%': rch(); { LET a = rdval()
                       UNLESS a DO error()
                       val := val REM a
                       GOTO sw
                     }
    CASE '|': rch(); val := val  |  rdval();  GOTO sw
    CASE '&': rch(); val := val  &  rdval();  GOTO sw
    CASE '^': rch(); val := val XOR rdval();  GOTO sw

    CASE '<': val := val << 1;                GOTO nxt
    CASE '>': val := val >> 1;                GOTO nxt

    CASE '=': prW(val); newline();            GOTO recover

    CASE '$': rch()
              UNLESS ch='B' | ch='C' | ch='D' | ch='F' |
                     ch='O' | ch='E' | ch='S' | ch='U' |
                     ch='X' | ch='Y' DO
              { writef("Valid style letters are: BCDEFOSUXY*n")
                GOTO recover
              }
              style := ch
              GOTO nxt

    CASE 'S': // Sn  SR<reg>  SPn  SGn  SVn
      rch()
      SWITCHON ch INTO
      { DEFAULT:
          GOTO sw

	CASE '0': CASE '1': CASE '2': CASE '3': CASE '4':
	CASE '5': CASE '6': CASE '7': CASE '8': CASE '9':
        { LET a = rdn()
          UNLESS 0<=a<z80memupb DO error() 
          putww(a, val)
          GOTO sw
        }

        CASE 'B': rch()
	{ LET n = rdn()
          UNLESS 0<=n<z80memupbb DO error() 
          putbb(n, val)
          GOTO sw
	}
	
        CASE 'R': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}
	
	CASE 'P': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'G': rch()
	{ LET n = rdreg()
	  IF n<0 DO error()
	  putreg(n, val)
	  GOTO sw
	}

	CASE 'V': rch()
	{ LET n = rdn()
          UNLESS 0<=n<=9 DO error() 
          vars!n := val
          GOTO sw
        }
      }

    CASE '\':               // Single step execution.
      //db("CASE '\':")
      indentpos := 3        // Used by prstate()
      singlestep := TRUE
      RETURN                // Execute one Z80 intruction

    CASE 'X':  // Xn   Trace n instruction executions
      error()

    CASE 'H':
      writef("*nAF'=%x4 BC'=%x4 DE'=%x4 HL'=%x4*n",
              getr(n_af1), getr(n_bc1), getr(n_de1), getr(n_hl1))
	      GOTO recover

    CASE 'F':
      fullstatus := ~fullstatus
      writef(" Full Status is %s*n", fullstatus -> "on", "off")
      GOTO recover
      
    CASE 'T': // TRn  TBn  TS  TF  Tn
      rch()
      //writef(" CASE 'T': ch=%i3='%c'*n", ch, ch)
      SWITCHON ch INTO
      { DEFAULT:
          error()

        CASE 'S':         // TS     Print the current state
        { newline()
          prstate()
          GOTO recover
        }

        CASE 'F':         // TF     Print the flags
        { writef(" ")
          prflags()
	  newline()
          GOTO recover
        }

        CASE 'R': rch()    // TRn    Trace n instructions
        { LET n = rdn()
 	  sys(Sys_settracing, n+1)
	  GOTO ret
        }

        CASE 'B': rch()    // TBn    Print n bytes from byte address val
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO writef("*n%i5:   ", val+i)
            prB(mem%(val+i))
	    wrch(' ')
          }
	  newline()
	  GOTO sw
        }

        CASE '0':CASE '1':CASE '2':CASE '3':CASE '4': // Tn
        CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
        { LET n = rdn()
          IF n<=0 DO n := 1
          FOR i=0 TO n-1 DO
          { IF i MOD 8 = 0 DO writef("*n%i5:   ", val+i)
            prW(getww(val+i))
	    wrch(' ')
          }
	  newline()
          GOTO recover
        }
      }

    CASE 'D': writef("*nCintcode memory dumped to DUMP.mem*n")
              sys(Sys_dumpmem, 6) // Dump the memory (context=6)
              GOTO recover

    CASE 'I': writef(" %i5: ", val)
              val := val+prz80instr(val)
 	      newline()
	      GOTO recover
 
    CASE 'Z':  // Equivalent to P1B9C
      error()
      //val := pptr!1  // set breakpoint 9 to current return address

    caseb:
    CASE 'B':  // Set, clear or display breakpoints.
    { LET comch = ch
      TEST comch='B' THEN rch()       // For B
                     ELSE ch := '9'   // For X or Z
      IF '0'<=ch<='9' DO
      { LET n = ch - '0'  // Set or Clear a break point.
        bpt_addr!n := 0
        IF val=0 GOTO nxt
        checkaddr(val>>B2Wsh)
        FOR i = 0 TO 9 IF bpt_addr!i=val DO bpt_addr!i := 0
        bpt_addr!n, bpt_instr!n := val, 0%val
        GOTO comch='B' -> nxt, resume
      }
      UNLESS ch='*n' DO newline()
      FOR i = 0 TO 9 DO  // List all breakpoints.
      { LET ba=bpt_addr!i
        UNLESS ba=0 DO
        { writef("%n:  ", i)
          writearg(ba)
          newline()
        }
      }
      GOTO recover
    }

    CASE 'M':  // Set or clear memory watch address
      checkaddr(val)
      TEST val THEN writef("*nWatch address: %n*n", val)
               ELSE writef("*nWatch unset*n")
      sys(Sys_watch, val) // ###### Wrong ######
      GOTO recover

    CASE 'C': // Continue normal execution.
      setbreakpoints()
      IF getbb(r_pc)=f_brk DO
      { brkstep := TRUE
        unsetbreakpoints()
	RETURN
      }
      brkstep := FALSE
      singlestep := FALSE
      newline()
      //db("CASE 'C':")
      RETURN

resume:
    { LET pc = r_pc
      newline()
      FOR i = 0 TO 9 IF pc=bpt_addr!i DO
      { // We are resuming at a break point
        writef("Resuming after breakpoint %n*n", i)
        brkstep := TRUE       // Execute one instruction before reinstating
        RETURN                // the BRK instructions
      }
      writef("Resuming normal execution*n")
    }
  }

ret:
  UNLESS brkstep FOR i = 0 TO 9 DO
  { LET ba=bpt_addr!i         // If brkstep=FALSE insert BRK at every breakpoint.
    IF ba DO putbb(ba, f_brk)
  }
  RETURN
}

AND setbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO
  { bpt_instr!i := getbb(ba) // Save the previous instruction at the breakpoint.
    putbb(ba, f_brk)         // Replace it by BRK (=RST #x28)
  }
}

AND unsetbreakpoints() BE FOR i = 0 TO 9 DO
{ LET ba = bpt_addr!i
  IF ba DO putbb(ba, bpt_instr!i)
}

AND prprompt() BE
{ 
  writef("** ")  // Standalone prompt
}


AND writearg(n) BE
{ LET tag = n & tagbits
//abort(2398)
  IF tag=globword DO
  { //abort(9898)
    writef("  #G%z3#", n & nontagbits)
    RETURN
  }
  IF n=#xC0DE DO
  { writef("  #x%x4", n)
    RETURN
  }
  writef("  %6i", n)
}

AND rdn() = VALOF
{ LET res = 0
  WHILE '0'<=ch<='9' DO { res := res*10 + ch - '0'; rch() }
  RESULTIS res
}

AND rdreg() = VALOF
{ // Read a register name and return its register number.
  // The result is -1 if the register in not recognised.
  // On entry ch is the first character of the name.
  // On exit ch is the character following the name.
  // The registers are:
  //  B C D E H L F A AF BC DE HL SP PC IX IY
  
  SWITCHON ch INTO
  { DEFAULT:
      rch(); RESULTIS -1
    CASE 'A':
      rch()
      UNLESS ch='F' RESULTIS n_a
      rch(); RESULTIS n_af
    CASE 'B':
      rch()
      UNLESS ch='C' RESULTIS n_b
      rch(); RESULTIS n_bc
    CASE 'C':
      rch(); RESULTIS n_c
    CASE 'D':
      rch()
      UNLESS ch='E' RESULTIS n_d
      rch(); RESULTIS n_de
    CASE 'E':
      rch(); RESULTIS n_e
    CASE 'H':
      rch()
      UNLESS ch='L' RESULTIS n_h
      rch(); RESULTIS n_hl
    CASE 'L':
      rch(); RESULTIS n_l
    CASE 'F':
      rch(); RESULTIS n_f
    CASE 'S':
      rch()
      UNLESS ch='P' RESULTIS -1
      rch(); RESULTIS n_sp
    CASE 'P':
      rch()
      UNLESS ch='C' RESULTIS -1
      rch(); RESULTIS n_pc
    CASE 'I':
      rch()
      IF ch='X' DO { rch(); RESULTIS n_ix }
      IF ch='Y' DO { rch(); RESULTIS n_iy }
      RESULTIS -1
  }
}

AND rdvaradd1r(type) = VALOF
{ LET base, lim, n = ?, ?, ?
  UNLESS '0'<=ch<='9' DO error()
  n := rdn()
  SWITCHON type INTO
  { DEFAULT:   error()
    //CASE 'P': base, lim := pptr,   fsize;           ENDCASE
    //CASE 'G': base, lim := gptr,   gptr!g_globsize; ENDCASE
    //CASE 'R': base, lim := regs,   r_upb;           ENDCASE
    CASE 'V': base, lim := vars,   9;               ENDCASE
    CASE 'A': base, lim := 0,      z80memupb;         ENDCASE
  }
  UNLESS 0<=n<=lim DO error()
  RESULTIS base + n
}

AND rdargval(str) = VALOF
{ // str is either a decimal number ddd..d
  // or        a hexadecimal number #hh..h
  LET res = 0
  LET radix = 0
  FOR i = 1 TO str%0 DO
  { LET ch = capitalch(str%i)
    SWITCHON ch INTO
    { DEFAULT:
      CASE '#': UNLESS i=1 RESULTIS -1
                radix := 16
		LOOP

      CASE '0':CASE '1':CASE '2':CASE '3':CASE '4':
      CASE '5':CASE '6':CASE '7':CASE '8':CASE '9':
                res := res*radix + ch - '0'
		LOOP

      CASE 'A':CASE 'B':CASE 'C':CASE 'D':CASE 'E':CASE 'F':
                UNLESS radix=16 RESULTIS -1
                res := res*radix + ch - 'A'
		LOOP

    }
  }
  RESULTIS res
  
}

AND rdval() = VALOF
{ // Read a val from the current input stream with first
  // character in ch.
  // Possible values:
  // Vn V Gn G Pn P 'c' #<hex> #O<oct> n
  // RA  RB  RC  RD  RE  RF  RH  RL
  // RIX RIY RSP RPC RAF RBC RDE RHL
  LET res, radix = 0, 10

  WHILE ch='*s' DO rch()
  
  SWITCHON ch INTO
  { DEFAULT:   error()

    CASE 'V':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         RESULTIS vars!n
	       }
	       error()
               RESULTIS 0

    CASE 'A':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(n)
	       }
               RESULTIS 0

    CASE 'G':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(gptr+n)
	       }
               RESULTIS gptr

    CASE 'P':  rch()
               IF '0'<=ch<='9' DO
	       { LET n = rdn()
	         IF 0 <= n < z80memupb RESULTIS getww(pptr+n)
	       }
               RESULTIS pptr

    CASE '*'': rch(); res := lch; rch();  RESULTIS res

    CASE '#':  radix := 16
               rch()
               IF ch='O' DO { radix := 8; rch() }

    CASE '0': CASE '1': CASE '2': CASE '3': CASE '4': 
    CASE '5': CASE '6': CASE '7': CASE '8': CASE '9': 
             { LET d = 100
               IF '0'<=ch<='9' DO d := ch-'0'
               IF 'A'<=ch<='F' DO d := ch-'A'+10
               IF d>=radix RESULTIS res
               res := res*radix+d
               rch()
             } REPEAT

    CASE 'R':
      rch()
      RESULTIS getr(rdreg()) 
  }
}

AND praddr(a) BE
{ LET type, base = 'A', 0
//  IF pptr <= a <= pptr+fsize & a <= pptr+100 DO type, base := 'P', pptr
//  IF gptr <= a <= gptr+gptr!g_globsize       DO type, base := 'G', gptr
  IF vars <= a <= vars+9                     DO type, base := 'V', vars
//  IF regs <= a <= regs+r_upb                 DO type, base := 'R', regs
  writef("*n%c%i5:", type, a-base)
}

AND prch(ch) BE
{ IF 32 < ch <= 127 DO
  { writef("   %c", ch)
    RETURN
  }
  SWITCHON ch INTO
  { DEFAULT:   writef(" %i3", ch); RETURN
    CASE '*s': writef("  SP");     RETURN
    CASE '*b': writef("  BS");     RETURN
    CASE '*n': writef("  NL");     RETURN
    CASE '*c': writef("  CR");     RETURN
    CASE '*p': writef("  FF");     RETURN
  }
}

AND prB(b) BE
{ // Print a byte in the current style
  b := b & #xFF
  SWITCHON style INTO
  { DEFAULT:   error();              RETURN
    CASE 'C':  prch(b);              RETURN
    CASE 'B':  writef("%8b", b);     RETURN
    CASE 'F':
    CASE 'S':
    CASE 'D':  writef("%4i", signextb(b));     RETURN
    CASE 'O':  writef("%3o", b);     RETURN
    CASE 'U':  writef("%3u", b);     RETURN
    CASE 'X':  writef("%2x", b);     RETURN
    CASE 'Y':  writef("%2x", b);     RETURN
  }

}

AND prW(w) BE
{ // Print a word in the current style
  SWITCHON style INTO
  { DEFAULT:   error();                 RETURN
    CASE 'C':  { LET ch = w & #xFF
                 //writes(" ")
                 wrch(32<=ch<=127 -> ch, '?')
	         ch := w>>8 & #xFF
                 wrch(32<=ch<=127 -> ch, '?')
                 RETURN
               }
    CASE 'B':  writef("%16b", w);           RETURN
    CASE 'D':  writef("%6i",  signextw(w)); RETURN
    CASE 'F':  writearg(w);                 RETURN
    CASE 'O':  writef("%6o",  w&#xFFFF);    RETURN
    CASE 'S':  checkaddr(w)
               writef("%S",   w);           RETURN
    CASE 'U':  writef("%6u",  w&#xFFFF);    RETURN
    CASE 'X':  writef("%4x",  w);           RETURN
    CASE 'Y':  writef("%2x",  w);           RETURN
  }
}

AND checkaddr(a) = VALOF // a is a BCPL pointer in the Z80 memory
{ UNLESS 0<=a<=z80memupb DO error()
  RESULTIS a
}

AND cont(a) = getww(a) // a is a BCPL pointer

AND error() BE { writes("  ??*n"); longjump(recp, recl) }

AND prstate() BE
{ // indentpos is the current indentation position
  LET prevpflag = pflag // pflag must be preserved
  IF fullstatus DO
  { FOR i = indentpos TO 3 DO wrch(' ')
    writef("P=%i5: %i5 %i5", pptr, getww(pptr), getww(pptr+1))
    FOR i = 2 TO 6 DO { wrch(' '); prW(getww(pptr+i)) }
    writef("  (SP)="); prW(getwb(r_sp))
    newline()
    indentpos := 0
  }
  FOR i = indentpos TO 3 DO wrch(' ')
  writef("A="); prB(r_a)
  writef(" BC="); prW(getr(n_bc))
  writef(" DE="); prW(getr(n_de))
  writef(" HL="); prW(getr(n_hl))
  writef(" IX="); prW(r_ix)
  writef(" IY="); prW(r_iy)
  writef(" sp=%i5 %i5:", r_sp, r_pc)
  prz80instr(r_pc)
  newline()
  pflag := prevpflag
  //abort(7687)
}

AND getbb(a) = mem%a

AND getsbb(a) = VALOF
{ LET byte = mem%a
  IF  byte<=127 RESULTIS byte
  RESULTIS byte-256
}

AND getwb(a) = mem%a | mem%(a+1)<<8

AND getww(a) = getwb(a+a)

AND putbb(a, val) BE mem%a := val

AND putwb(a, val) BE mem%a, mem%(a+1) := val, val>>8

AND putww(a, val) BE putwb(a+a, val)

AND signextb(byte) = VALOF
{ LET res = byte & #x7F
  IF (byte & #x80)>0 DO res := res - #x80
  RESULTIS res
}

AND signextw(word) = VALOF
{ LET res = word & #x7FFF
  IF (word & #x8000)>0 DO res := res - #x8000
  RESULTIS res
}

AND putreg(n, val) BE SWITCHON n INTO
{ DEFAULT:
    error()
    RETURN

  CASE  n_a: r_a  := val & #xFF;    RETURN  
  CASE  n_b: r_b  := val & #xFF;    RETURN  
  CASE  n_c: r_c  := val & #xFF;    RETURN  
  CASE  n_d: r_d  := val & #xFF;    RETURN  
  CASE  n_e: r_e  := val & #xFF;    RETURN  
  CASE  n_f: r_f  := val & #xFF;    RETURN  
  CASE  n_h: r_h  := val & #xFF;    RETURN  
  CASE  n_l: r_l  := val & #xFF;    RETURN  
  CASE n_ix: r_ix := val & #xFFFF;  RETURN  
  CASE n_iy: r_iy := val & #xFFFF;  RETURN  
  CASE n_sp: r_sp := val & #xFFFF;  RETURN  
  CASE n_pc: r_pc := val & #xFFFF;  RETURN  
  CASE n_af: r_a, r_f := val>>8 & #xFF, val & #xFF; RETURN  
  CASE n_bc: r_b, r_c := val>>8 & #xFF, val & #xFF; RETURN
  CASE n_de: r_d, r_e := val>>8 & #xFF, val & #xFF; RETURN
  CASE n_hl: r_h, r_l := val>>8 & #xFF, val & #xFF; RETURN
}

AND rch() BE { lch := sardch(); ch := capitalch(lch) }

AND loadhex(filename) = VALOF
{ // Load BCPL Intel Hex from the given file starting at byte address codep.
  // Globals defined in this file update entries in the global vector
  // with word address rootgvec.
  // Relocation data is encoded by lines starting with $02000004 followed by
  // 4 byte hex values giving the addrsses of the words to relocate and
  // terminated by the checksum byte.


  LET prevstream = input()
  LET hexstream = findinput(filename)
  UNLESS hexstream DO
  { writef("Cannot open file: *"%s*"*n", filename)
    RESULTIS FALSE
  }
  selectinput(hexstream)

  { // Read an Intel Hex record
    LET ch = rdch()
//IF ch>0 DO writef("ch='%c'*n", ch)
//abort(1001)

    IF ch=endstreamch BREAK

    IF ch='$' DO
    { // Either a relocation record or an ingnorable record
      LET n    = rdhexbyte()    // Number of addresses in this record, or ignored
      LET ch1  = rdhexbyte()    // low byte of load address, ignored
      LET ch2  = rdhexbyte()    // low byte of load address, ignored
      LET type = rdhexbyte()    // The record's type, =04 for a relocation block
      LET addr = (ch1<<8) + ch2 // Big ender ordering, ignored
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

      IF type=4 DO
      {
//writef("$%x2%x2%x2%x2*n", n, ch1,ch2, type)
//writef("loadhex: type=4 ch=%c n=%n ch1=%n ch2=%n addr=%n*n",
//                        ch,   n,   ch1,   ch2,   addr)
        FOR i = 1 TO n DO
        { LET p = codep//+6
	  ch1 := rdhexbyte()
          ch2 := rdhexbyte()
	  addr := (ch1<<8) + ch2
	  p := p + addr             // Need not be 2 byte aligned.
          sum := sum+ch1+ch2
          putwb(p, codep+getwb(p))

//writef("codep=%n*n",codep)
//writef("Word at %n changed from %n to %n*n", p, getwb(p), codep+getwb(p))
//abort(1009)
        }
      
//        newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
      
      UNTIL ch='*n' | ch=endstreamch DO ch := rdch()
      IF ch='*n' LOOP
    }


    IF ch=':' DO
    { // Format: : nn aaaa tt   followed by bytes to load  and the checksum byte
      LET n    = rdhexbyte()  // Number of code bytes in this record
      LET ch1  = rdhexbyte()  // low byte of load address
      LET ch2  = rdhexbyte()  // low byte of load address
      LET type = rdhexbyte()  // The record's type
      LET addr = ch1<<8 | ch2 // Big ender ordering
      LET sum  = n+ch1+ch2+type // Initial setting of the checksum

//      writef("n=%i2 addr=%i5 type=%n*n", n, addr, type)
      
      IF type=0 DO
      { FOR i = 1 TO n DO
        { LET byte = rdhexbyte()
          sum := sum+byte
          putbb(codep+addr, byte)
//writef(" %x2", byte)
          addr := addr+1
        }
      
        //newline()
//writef("about to read the checksum byte*n")
        sum := sum+rdhexbyte()
        IF (sum&#xFF)>0 DO
        { writef("Checksum error*n")
          BREAK
        }
//        writef("end of : record*n")
//        abort(1002)
      }
    }
    
    IF ch=endstreamch BREAK
    ch:=ch // To get round a compiler bug
  } REPEAT


//writef("Init the globals*n")
//abort(1003)
  // Initialise the globals
  
  // BCPL sections start with bytes 'B' 'C''P' 'L' followed by 2 bytes
  // giving the size of the section in bytes.
  { LET size = getwb(codep+4)
    LET p = codep+size-6
//    writef("codep=%n size=%n*n", codep, size)
//    FOR a = 0 TO size*2 -1 DO writef("%x2: %x2*n", a, getbb(codep+a))
//    newline()


    { LET gn    = getwb(p)>>1
      LET entry = getwb(p+2)
      IF entry=0 | p < codep BREAK
      putww(rootgvec+gn, entry)
      //writef("gn=%i3 entry=%n*n", gn, entry)
      p := p-4
    } REPEAT
    
    //abort(1234)
//    FOR a = codep TO codep+size-2 BY 2 DO writef("%i5: %x4*n", a, getwb(a))
  }
  
  IF hexstream DO endstream(hexstream)
  selectinput(prevstream)
  RESULTIS TRUE
}

AND rdhexbyte() = VALOF
{ // Read tw hex characters and retun the specified byte
  LET res = hexval(rdch())
  RESULTIS res<<4 | hexval(rdch())
}

AND hexval(ch) = '0'<=ch<='9' -> ch-'0',
                 'A'<=ch<='F' -> ch-'A'+10,
		 'a'<=ch<='f' -> ch-'a'+10,
		                 -1

AND z8dumpmem() BE
{ LET dumpstream = findoutput("Z80DUMP")
  selectoutput(dumpstream)
  writef("AF=%x4*n", r_a<<8 | r_f)
  writef("BC=%x4*n", r_c<<8 | r_b)
  writef("DE=%x4*n", r_e<<8 | r_d)
  writef("HL=%x4*n", r_l<<8 | r_h)
  writef("IX=%x4*n", r_ix)
  writef("IY=%x4*n", r_iy)
  writef("SP=%x4*n", r_sp)
  writef("PC=%x4*n", r_pc)
  writef("MEM=*n")
  FOR i = 0 TO z80memupbb DO
  { IF i MOD 32 = 0 DO newline()
    writef("%x2", getbb(i))
  }
  newline()
  endstream(dumpstream)
}

AND prird(r, d) BE // eg (SP-114)  or  (HL+12)
{ writef("(%s", r2str(r))
  IF d>0 DO writef("+%n", d)
  IF d<0 DO writef("-%n", -d)
  writef(")")
}

AND prflags(flags) BE
{ writef("F=")
  IF r_fs DO wrch('S')
  IF r_fz DO wrch('Z')
  IF r_fh DO wrch('H')
  IF r_fv DO wrch('V')
  IF r_fn DO wrch('N')
  IF r_fc DO wrch('C')
}

AND prz80instr(pc) = VALOF
{ // Print the Z80 instruction at location pc
  // and return its size in bytes.
  pflag := TRUE
  RESULTIS doz80instr(pc)
}

AND doz80instr(pc) = VALOF
{ // If pflag=TRUE output the instruction at location pc and
  //               return its size in bytes.
  // If pflag=FALSE execute the instruction at location pc (=r_pc) and
  // return zero if the instruction does not cause a fault or
  // return a non zero fault code.
  
  LET op = getbb(pc)

UNLESS pflag DO
{ //db("*nEntered doz80instr() pc=%n op=%x2", pc, op)
}
  SWITCHON op INTO
  { CASE #x00: RESULTIS z80NOP()                   // NOP
    CASE #x01: RESULTIS z80LD16rnn(pc, n_bc)       // LD BC,nn
    CASE #x02: RESULTIS z80LDrr(pc, n_ibc, n_a)    // LD (BC),A
    CASE #x03: RESULTIS z80INC16r(pc, n_bc)        // INC BC
    CASE #x04: RESULTIS z80INCr(pc, n_b)           // INC B
    CASE #x05: RESULTIS z80DECr(pc, n_b)           // DEC B
    CASE #x06: RESULTIS z80LDrn(pc, n_b)           // LD B,n
    CASE #x07: RESULTIS prbadop(pc)
    CASE #x08: RESULTIS z80EXrr(pc, n_af, n_af1)   // EX AF,AF'
    CASE #x09: RESULTIS z80ADD16rr(pc, n_hl, n_bc) // ADD HL,BC
    CASE #x0A: RESULTIS z80LDrr(pc, n_a, n_ibc)    // LD A,(BC)
    CASE #x0B: RESULTIS z80DEC16r(pc, n_bc)        // DEC BC
    CASE #x0C: RESULTIS z80INCr(pc, n_c)           // INC C
    CASE #x0D: RESULTIS z80DECr(pc, n_c)           // DEC C
    CASE #x0E: RESULTIS z80LDrn(pc, n_c)           // LD C,n
    CASE #x0F: RESULTIS z80RRCA(pc)                // RRCA

    CASE #x10: RESULTIS prbadop(pc)
    CASE #x11: RESULTIS z80LD16rnn(pc, n_de)       // LD DE,nn
    CASE #x12: RESULTIS z80LDrr(pc, n_ide, n_a)    // LD (DE),A
    CASE #x13: RESULTIS z80INC16r(pc, n_de)        // INC DE
    CASE #x14: RESULTIS z80INCr(pc, n_d)           // INC D
    CASE #x15: RESULTIS z80DECr(pc, n_d)           // DEC D
    CASE #x16: RESULTIS z80LDrn(pc, n_d)           // LD D,n
    CASE #x17: RESULTIS prbadop(pc)
    CASE #x18: RESULTIS prbadop(pc)
    CASE #x19: RESULTIS z80ADD16rr(pc, n_hl, n_de) // ADD HL,DE
    CASE #x1A: RESULTIS z80LDrr(pc, n_a, n_ide)    // LD A,(DE)
    CASE #x1B: RESULTIS z80DEC16r(pc, n_de)        // DEC DE
    CASE #x1C: RESULTIS z80INCr(pc, n_e)           // INC E
    CASE #x1D: RESULTIS z80DECr(pc, n_e)           // DEC E
    CASE #x1E: RESULTIS z80LDrn(pc, n_e)           // LD E,n
    CASE #x1F: RESULTIS prbadop(pc)

    CASE #x20: RESULTIS z80JRra(pc, n_nz)          // JR NZ,rel
    CASE #x21: RESULTIS z80LD16rnn(pc, n_hl)       // LD r,nn
    CASE #x22: RESULTIS prbadop(pc)
    CASE #x23: RESULTIS z80INC16r(pc, n_hl)        // INC HL
    CASE #x24: RESULTIS z80INCr(pc, n_h)           // INC H
    CASE #x25: RESULTIS z80DECr(pc, n_h)           // DEC H
    CASE #x26: RESULTIS z80LDrn(pc, n_h)           // LD H,n
    CASE #x27: RESULTIS z80DAAr(pc)                // DAA
    CASE #x28: RESULTIS prbadop(pc)
    CASE #x29: RESULTIS z80ADD16rr(pc, n_hl, n_hl) // ADD HL,HL
    CASE #x2A: RESULTIS prbadop(pc)
    CASE #x2B: RESULTIS z80DEC16r(pc, n_hl)        // DEC HL
    CASE #x2C: RESULTIS z80INCr(pc, n_l)           // INC L
    CASE #x2D: RESULTIS z80DECr(pc, n_l)           // DEC L
    CASE #x2E: RESULTIS z80LDrn(pc, n_l)           // LD L,n
    CASE #x2F: RESULTIS z80CPL(pc)                 // CPL

    CASE #x30: RESULTIS prbadop(pc)
    CASE #x31: RESULTIS prbadop(pc)
    CASE #x32: RESULTIS z80LDinnr(pc, n_a)         // LD (nn),A
    CASE #x33: RESULTIS z80INC16r(pc, n_sp)        // INC SP
    CASE #x34: RESULTIS z80INCr(pc, n_ihl)         // INC (HL)
    CASE #x35: RESULTIS z80DECr(pc, n_ihl)         // DEC (HL)
    CASE #x36: RESULTIS z80LDrn(pc, n_ihl)         // LD (HL),n
    CASE #x37: RESULTIS z80SCF(pc)                 // SCF
    CASE #x38: RESULTIS prbadop(pc)
    CASE #x39: RESULTIS z80ADD16rr(pc, n_hl, n_sp) // ADD HL,SP
    CASE #x3A: RESULTIS z80LDrinn(pc, n_a)         // LD A,(nn)
    CASE #x3B: RESULTIS z80DEC16r(pc, n_sp)        // DEC SP
    CASE #x3C: RESULTIS z80INCr(pc, n_a)           // INC A
    CASE #x3D: RESULTIS z80DECr(pc, n_a)           // DEC A
    CASE #x3E: RESULTIS z80LDrn(pc, n_a)           // LD A,n
    CASE #x3F: RESULTIS z80CCF(pc)                 // CCF

    CASE #x40: RESULTIS z80LDrr(pc, n_b, n_b)      // LD B,B
    CASE #x41: RESULTIS z80LDrr(pc, n_b, n_c)      // LD B,C
    CASE #x42: RESULTIS z80LDrr(pc, n_b, n_d)      // LD B,D
    CASE #x43: RESULTIS z80LDrr(pc, n_b, n_e)      // LD B,E
    CASE #x44: RESULTIS z80LDrr(pc, n_b, n_h)      // LD B,H
    CASE #x45: RESULTIS z80LDrr(pc, n_b, n_l)      // LD B,L
    CASE #x46: RESULTIS z80LDrr(pc, n_b, n_ihl)    // LD B,(HL)
    CASE #x47: RESULTIS z80LDrr(pc, n_b, n_a)      // LD B,A
    CASE #x48: RESULTIS z80LDrr(pc, n_c, n_b)      // LD C,B
    CASE #x49: RESULTIS z80LDrr(pc, n_c, n_c)      // LD C,C
    CASE #x4A: RESULTIS z80LDrr(pc, n_c, n_d)      // LD C,D
    CASE #x4B: RESULTIS z80LDrr(pc, n_c, n_e)      // LD C,E
    CASE #x4C: RESULTIS z80LDrr(pc, n_c, n_h)      // LD C,H
    CASE #x4D: RESULTIS z80LDrr(pc, n_c, n_l)      // LD C,L
    CASE #x4E: RESULTIS z80LDrr(pc, n_c, n_ihl)    // LD C,(HL)
    CASE #x4F: RESULTIS z80LDrr(pc, n_c, n_a)      // LD C,A

    CASE #x50: RESULTIS z80LDrr(pc, n_d, n_b)      // LD D,B
    CASE #x51: RESULTIS z80LDrr(pc, n_d, n_c)      // LD D,C
    CASE #x52: RESULTIS z80LDrr(pc, n_d, n_d)      // LD D,D
    CASE #x53: RESULTIS z80LDrr(pc, n_d, n_e)      // LD D,E
    CASE #x54: RESULTIS z80LDrr(pc, n_d, n_h)      // LD D,H
    CASE #x55: RESULTIS z80LDrr(pc, n_d, n_l)      // LD D,L
    CASE #x56: RESULTIS z80LDrr(pc, n_d, n_ihl)    // LD D,(HL)
    CASE #x57: RESULTIS z80LDrr(pc, n_d, n_a)      // LD D,A
    CASE #x58: RESULTIS z80LDrr(pc, n_e, n_b)      // LD E,B
    CASE #x59: RESULTIS z80LDrr(pc, n_e, n_c)      // LD E,C
    CASE #x5A: RESULTIS z80LDrr(pc, n_e, n_d)      // LD E,D
    CASE #x5B: RESULTIS z80LDrr(pc, n_e, n_e)      // LD E,E
    CASE #x5C: RESULTIS z80LDrr(pc, n_e, n_d)      // LD E,H
    CASE #x5D: RESULTIS z80LDrr(pc, n_e, n_l)      // LD E,L
    CASE #x5E: RESULTIS z80LDrr(pc, n_e, n_ihl)    // LD E,(HL)
    CASE #x5F: RESULTIS z80LDrr(pc, n_e, n_a)      // LD E,A

    CASE #x60: RESULTIS z80LDrr(pc, n_h, n_b)      // LD H,B
    CASE #x61: RESULTIS z80LDrr(pc, n_h, n_c)      // LD H,C
    CASE #x62: RESULTIS z80LDrr(pc, n_h, n_d)      // LD H,D
    CASE #x63: RESULTIS z80LDrr(pc, n_h, n_e)      // LD H,E
    CASE #x64: RESULTIS z80LDrr(pc, n_h, n_h)      // LD H,H
    CASE #x65: RESULTIS z80LDrr(pc, n_h, n_l)      // LD H,L
    CASE #x66: RESULTIS z80LDrr(pc, n_h, n_ihl)    // LD H,(HL)
    CASE #x67: RESULTIS z80LDrr(pc, n_h, n_a)      // LD H,A
    CASE #x68: RESULTIS z80LDrr(pc, n_l, n_b)      // LD L,B
    CASE #x69: RESULTIS z80LDrr(pc, n_l, n_c)      // LD L,C
    CASE #x6A: RESULTIS z80LDrr(pc, n_l, n_d)      // LD L,D
    CASE #x6B: RESULTIS z80LDrr(pc, n_l, n_e)      // LD L,E
    CASE #x6C: RESULTIS z80LDrr(pc, n_l, n_h)      // LD L,H
    CASE #x6D: RESULTIS z80LDrr(pc, n_l, n_l)      // LD L,L
    CASE #x6E: RESULTIS z80LDrr(pc, n_l, n_ihl)    // LD L,(HL)
    CASE #x6F: RESULTIS z80LDrr(pc, n_l, n_a)      // LD L,A

    CASE #x70: RESULTIS z80LDrr(pc, n_hl, n_b)     // LD (HL),B
    CASE #x71: RESULTIS z80LDrr(pc, n_hl, n_c)     // LD (HL),C
    CASE #x72: RESULTIS z80LDrr(pc, n_hl, n_d)     // LD (HL),D
    CASE #x73: RESULTIS z80LDrr(pc, n_hl, n_e)     // LD (HL),E
    CASE #x74: RESULTIS z80LDrr(pc, n_hl, n_h)     // LD (HL),H
    CASE #x75: RESULTIS z80LDrr(pc, n_hl, n_l)     // LD (HL),L
    CASE #x76: RESULTIS z80HALT(pc)                // HALT
    CASE #x77: RESULTIS z80LDrr(pc, n_ihl, n_a)    // LD (HL),A
    CASE #x78: RESULTIS z80LDrr(pc, n_a, n_b)      // LD A,B
    CASE #x79: RESULTIS z80LDrr(pc, n_a, n_c)      // LD A,C
    CASE #x7A: RESULTIS z80LDrr(pc, n_a, n_d)      // LD A,D
    CASE #x7B: RESULTIS z80LDrr(pc, n_a, n_e)      // LD A,E
    CASE #x7C: RESULTIS z80LDrr(pc, n_a, n_h)      // LD A,H
    CASE #x7D: RESULTIS z80LDrr(pc, n_a, n_l)      // LD A,L
    CASE #x7E: RESULTIS z80LDrr(pc, n_a, n_ihl)    // LD A,(HL)
    CASE #x7F: RESULTIS z80LDrr(pc, n_a, n_a)      // LD A,A

    CASE #x80: RESULTIS z80ADDr(pc, n_b)           // ADD B
    CASE #x81: RESULTIS z80ADDr(pc, n_c)           // ADD C
    CASE #x82: RESULTIS z80ADDr(pc, n_d)           // ADD D
    CASE #x83: RESULTIS z80ADDr(pc, n_e)           // ADD E
    CASE #x84: RESULTIS z80ADDr(pc, n_h)           // ADD H
    CASE #x85: RESULTIS z80ADDr(pc, n_l)           // ADD L
    CASE #x86: RESULTIS z80ADDr(pc, n_ihl)         // ADD (HL)
    CASE #x87: RESULTIS z80ADDr(pc, n_a)           // ADD A
    CASE #x88: RESULTIS z80ADCr(pc, n_b)           // ADC B
    CASE #x89: RESULTIS z80ADCr(pc, n_c)           // ADC C
    CASE #x8A: RESULTIS z80ADCr(pc, n_d)           // ADC D
    CASE #x8B: RESULTIS z80ADCr(pc, n_e)           // ADC E
    CASE #x8C: RESULTIS z80ADCr(pc, n_h)           // ADC H
    CASE #x8D: RESULTIS z80ADCr(pc, n_l)           // ADC L
    CASE #x8E: RESULTIS z80ADCr(pc, n_ihl)         // ADC (HL)
    CASE #x8F: RESULTIS z80ADCr(pc, n_a)           // ADC A

    CASE #x90: RESULTIS z80SUBr(pc, n_b)           // SUB B
    CASE #x91: RESULTIS z80SUBr(pc, n_c)           // SUB C
    CASE #x92: RESULTIS z80SUBr(pc, n_d)           // SUB D
    CASE #x93: RESULTIS z80SUBr(pc, n_e)           // SUB E
    CASE #x94: RESULTIS z80SUBr(pc, n_h)           // SUB H
    CASE #x95: RESULTIS z80SUBr(pc, n_l)           // SUB L
    CASE #x96: RESULTIS z80SUBr(pc, n_ihl)         // SUB (HL)
    CASE #x97: RESULTIS z80SUBr(pc, n_a)           // SUB A
    CASE #x98: RESULTIS z80SBCr(pc, n_b)           // SBC B
    CASE #x99: RESULTIS z80SBCr(pc, n_c)           // SBC C
    CASE #x9A: RESULTIS z80SBCr(pc, n_d)           // SBC D
    CASE #x9B: RESULTIS z80SBCr(pc, n_e)           // SBC E
    CASE #x9C: RESULTIS z80SBCr(pc, n_h)           // SBC H
    CASE #x9D: RESULTIS z80SBCr(pc, n_l)           // SBC L
    CASE #x9E: RESULTIS z80SBCr(pc, n_ihl)         // SBC (HL)
    CASE #x9F: RESULTIS z80SBCr(pc, n_a)           // SBC A

    CASE #xA0: RESULTIS z80ANDr(pc, n_b)           // AND B
    CASE #xA1: RESULTIS z80ANDr(pc, n_c)           // AND C
    CASE #xA2: RESULTIS z80ANDr(pc, n_d)           // AND D
    CASE #xA3: RESULTIS z80ANDr(pc, n_e)           // AND E
    CASE #xA4: RESULTIS z80ANDr(pc, n_h)           // AND H
    CASE #xA5: RESULTIS z80ANDr(pc, n_l)           // AND L
    CASE #xA6: RESULTIS z80ANDr(pc, n_ihl)         // AND (HL)
    CASE #xA7: RESULTIS z80ANDr(pc, n_a)           // AND A
    CASE #xA8: RESULTIS z80XORr(pc, n_b)           // XOR B
    CASE #xA9: RESULTIS z80XORr(pc, n_c)           // XOR C
    CASE #xAA: RESULTIS z80XORr(pc, n_d)           // XOR D
    CASE #xAB: RESULTIS z80XORr(pc, n_e)           // XOR E
    CASE #xAC: RESULTIS z80XORr(pc, n_h)           // XOR H
    CASE #xAD: RESULTIS z80XORr(pc, n_l)           // XOR L
    CASE #xAE: RESULTIS z80XORr(pc, n_ihl)         // XOR (HL)
    CASE #xAF: RESULTIS z80XORr(pc, n_a)           // XOR A

    CASE #xB0: RESULTIS z80ORr(pc, n_b)            // OR B
    CASE #xB1: RESULTIS z80ORr(pc, n_c)            // OR C
    CASE #xB2: RESULTIS z80ORr(pc, n_d)            // OR D
    CASE #xB3: RESULTIS z80ORr(pc, n_e)            // OR E
    CASE #xB4: RESULTIS z80ORr(pc, n_h)            // OR H
    CASE #xB5: RESULTIS z80ORr(pc, n_l)            // OR L
    CASE #xB6: RESULTIS z80ORr(pc, n_ihl)          // OR (HL)
    CASE #xB7: RESULTIS z80ORr(pc, n_a)            // OR A
    CASE #xB8: RESULTIS z80CPr(pc, n_b)            // CP B
    CASE #xB9: RESULTIS z80CPr(pc, n_c)            // CP C
    CASE #xBA: RESULTIS z80CPr(pc, n_d)            // CP D
    CASE #xBB: RESULTIS z80CPr(pc, n_e)            // CP E
    CASE #xBC: RESULTIS z80CPr(pc, n_h)            // CP H
    CASE #xBD: RESULTIS z80CPr(pc, n_l)            // CP L
    CASE #xBE: RESULTIS z80CPr(pc, n_ihl)          // CP (HL)
    CASE #xBF: RESULTIS z80CPr(pc, n_a)            // CP A

    CASE #xC0: RESULTIS prbadop(pc)
    CASE #xC1: RESULTIS z80POP16r(pc, n_bc)        // POP BC
    CASE #xC2: RESULTIS prbadop(pc)
    CASE #xC3: RESULTIS prbadop(pc)
    CASE #xC4: RESULTIS prbadop(pc)
    CASE #xC5: RESULTIS z80PUSH16r(pc, n_bc)       // PUSH BC
    CASE #xC6: RESULTIS z80ADDrn(pc, n_a)          // ADD A,n
    CASE #xC7: RESULTIS z80RSTn(pc, #x00)          // RST #x00
    CASE #xC8: RESULTIS prbadop(pc)
    CASE #xC9: RESULTIS z80RET(pc)                 // RET
    CASE #xCA: RESULTIS prbadop(pc)
    CASE #xCB: RESULTIS doz80instrCB(pc)           // CB instructions
    CASE #xCC: RESULTIS prbadop(pc)
    CASE #xCD: RESULTIS prbadop(pc)
    CASE #xCE: RESULTIS z80ADCrn(pc, n_a)          // ADC A,n
    CASE #xCF: RESULTIS z80RSTn(pc, #x08)          // RST #x08

    CASE #xD0: RESULTIS prbadop(pc)
    CASE #xD1: RESULTIS z80POP16r(pc, n_de)        // POP DE
    CASE #xD2: RESULTIS prbadop(pc)
    CASE #xD3: RESULTIS prbadop(pc)
    CASE #xD4: RESULTIS prbadop(pc)
    CASE #xD5: RESULTIS z80PUSH16r(pc, n_de)       // PUSH DE
    CASE #xD6: RESULTIS z80SUBn(pc)                // SUB n
    CASE #xD7: RESULTIS z80RSTn(pc, #x10)          // RST #x10
    CASE #xD8: RESULTIS prbadop(pc)
    CASE #xD9: RESULTIS z80EXX(pc)                 // EXX
    CASE #xDA: RESULTIS prbadop(pc)
    CASE #xDB: RESULTIS prbadop(pc)
    CASE #xDC: RESULTIS prbadop(pc)
    CASE #xDD: RESULTIS doz80instrXD(pc, n_ix)     // DD and FD instructions
    CASE #xDE: RESULTIS z80SBCrn(pc, n_a)          // SBC A,n
    CASE #xDF: RESULTIS z80RSTn(pc, #x18)          // RST #x18

    CASE #xE0: RESULTIS prbadop(pc)
    CASE #xE1: RESULTIS z80POP16r(pc, n_hl)        // POP HL
    CASE #xE2: RESULTIS prbadop(pc)
    CASE #xE3: RESULTIS z80EXrr(pc, n_isp, n_hl)   // EX (SP),HL
    CASE #xE4: RESULTIS prbadop(pc)
    CASE #xE5: RESULTIS z80PUSH16r(pc, n_hl)       // PUSH HL
    CASE #xE6: RESULTIS z80ANDn(pc)                // AND n
    CASE #xE7: RESULTIS z80RSTn(pc, #x20)          // RST #x20
    CASE #xE8: RESULTIS prbadop(pc)
    CASE #xE9: RESULTIS z80JPr(pc, n_ihl)          // JP (HL)
    CASE #xEA: RESULTIS prbadop(pc)
    CASE #xEB: RESULTIS z80EXrr(pc, n_de, n_hl)    // EX DE,HL
    CASE #xEC: RESULTIS prbadop(pc)
    CASE #xED: RESULTIS doz80instrED(pc, op)
    CASE #xEE: RESULTIS z80XORn(pc)                // XOR n
    CASE #xEF: RESULTIS z80RSTn(pc, #x28)          // RST #x28   The BCPL BRK instruction

    CASE #xF0: RESULTIS prbadop(pc)
    CASE #xF1: RESULTIS z80POP16r(pc, n_af)        // POP AF
    CASE #xF2: RESULTIS prbadop(pc)
    CASE #xF3: RESULTIS z80DI(pc)                  // DI  Disable INT
    CASE #xF4: RESULTIS prbadop(pc)
    CASE #xF5: RESULTIS z80PUSH16r(pc, n_af)       // PUSH AF
    CASE #xF6: RESULTIS z80ORn(pc)                 // OR n
    CASE #xF7: RESULTIS z80RSTn(pc, #x30)          // RST #x30
    CASE #xF8: RESULTIS prbadop(pc)
    CASE #xF9: RESULTIS z80LD16rr(pc, n_sp, n_hl)  // LD SP,HL 
    CASE #xFA: RESULTIS prbadop(pc)
    CASE #xFB: RESULTIS z80EI(pc)                  // EI  Enable INT
    CASE #xFC: RESULTIS prbadop(pc)
    CASE #xFD: RESULTIS doz80instrXD(pc, n_iy)     // DD and FD instructions
    CASE #xFE: RESULTIS z80CPn(pc)                 // CP n
    CASE #xFF: RESULTIS z80RSTn(pc, #x38)          // RST #X38
  }
}

AND z80RSTn(pc, addr) = VALOF
{ IF addr=#x28 DO
  { // This lists or execute the RST #x28 instruction.
    // This is the BRK instruction used in the implementation
    // of breakpoints.
    // When executed it sets the faultcode to 2 and returns
    // without advancing r_pc.
    TEST pflag
    THEN { writef("BRK")
           RESULTIS 1
	 }
    ELSE {
           writef("z80RSTn: addy=%x2*n")
           RESULTIS 2
         }
  }

  IF addr=#x38 DO
  { // This lists or executes the RST instruction that causes a
    // subroutine jump to location addr in page 0. If pflag=TRUE
    // it lists the instruction returning its size in byte.
    // If pflag=FALSE it executes the instruction pointed toby r_pc
    // and return a fault code. If the instruction did not generate
    // fault the result is zero.
    // If addr is #x38 this subroutine looks at the bytes following
    // the RST #x38 instruction to decide what to do. The inline
    // arguments given to this subroutine are in the subsequent bytes.
    // Each inline argument is either an 8 bit value encoded by a
    // single byte, a 16 bit value encoded by a pair of bytes or an
    // unsigned 16 bit even numbers encoded by one or two bytes as
    // follows:

    // 0bbbbbbb          =>  00000000 0bbbbbbb
    // 1bbbbbbb cccccccc =>  0ccccccc cbbbbbbb

    // This encoding was done in bcplcgz80.b as follows:
  
    //  LET low7   =  (value)       &  #B0000000001111111
    //  LET high8  =  (value << 1)  &  #B1111111100000000

    //  TEST  high8 = 0  THEN  code_n_1( low7 )
    //  ELSE
    //  $(
    //      code_n_1( low7 + #B10000000 )
    //      code_n_1( high8 >> 8 )
    //  $)

    // Other arguments are placed in the stack and results may also be
    // placed there.
    // The first inline argument is a byte specifying which operation to
    // be perform. These operations are needed by the BCPL compiled code.
    // Some examples are:      
    //    0: m_apply
    //    3: m_fixup
    //    6: m_loadix
    //    9: m_loadiy

    LET mtype = getbb(pc+1)

    SWITCHON mtype INTO
    { DEFAULT: zerror("Bad RST #x38 op=%n*n", mtype)
               RESULTIS 2
	     
      CASE  m_apply:
      { LET size = 3
        LET arg = getbb(pc+2)
        IF arg > #x7F DO size, arg := 4, arg & #x7F | getbb(pc+3)<<7 
        TEST pflag
        THEN { writef("APPLY %n", arg)
               RESULTIS size
	     }
        ELSE { pptr := pptr + arg // The new P pointer
               putww(pptr, r_ix)
	       putww(pptr+1, r_pc+size)
	       r_ix := (pptr<<1)+128
	       r_pc := getr(n_hl)
	       RESULTIS 0
             }
      }

      CASE  m_fixup:
        writef("FIXUP")
        RESULTIS 2
      
      CASE  m_loadix:
        writef("LOADIX")
        RESULTIS 2
    
      CASE  m_loadiy:
      { // Push G!arg
        LET size = 3
        LET arg = getbb(pc+2)
        //writef("*nloadlvix: size=%n arg=%n*n", size, arg)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
      
        TEST pflag
        THEN { writef("LOADIY %n", arg+128)
               RESULTIS size
	     }
        ELSE { r_sp := r_sp-2
               putwb(r_sp, getww(gptr+arg+128))
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE m_storeix:
        zerror("STOREIX")
        RESULTIS 2
      
      CASE m_storeiy:
        writef("STOREIY")
        RESULTIS 2
      
      CASE m_setlink0:
        TEST pflag
        THEN { writef("SETLINK0")
               RESULTIS 2
	     }
        ELSE { x_exx()
               r_pc := r_pc+2
               RESULTIS 0
	     }

      CASE 21:
        TEST pflag
        THEN { writef("SETLINK1")
               RESULTIS 2
	     }
        ELSE { x_exx()
               putww(pptr+2, getr(n_hl))
               r_pc := r_pc+2
               RESULTIS 0
	     }
        RESULTIS 2
      
      CASE m_setlink2:
        TEST pflag
        THEN { writef("SETLINK2")
               RESULTIS 2
	     }
        ELSE { x_exx()
               putww(pptr+2, getr(n_hl))
               putww(pptr+3, getr(n_de))
               r_pc := r_pc+2
               RESULTIS 0
	     }
        RESULTIS 2
      
      CASE m_setlink3:
        TEST pflag
        THEN { writef("SETLINK3")
               RESULTIS 2
	     }
        ELSE { x_exx()
               putww(pptr+2, getr(n_hl))
               putww(pptr+3, getr(n_de))
               putww(pptr+4, getr(n_bc))
               r_pc := r_pc+2
               RESULTIS 0
	     }
        RESULTIS 2

      CASE m_return:
        TEST pflag
        THEN { writef("RETURN")
               RESULTIS 2
	     }
        ELSE { r_ix := getww(pptr)
	       r_pc := getww(pptr+1)
	       pptr := (r_ix-128)>>1
	       IF r_pc=0 DO
	       { //writef("Returning from start with result %n*n", getr(n_hl))
	         longjump(finp, finl)
	       }
	       //writef("doRSTn returning zero*n")
               RESULTIS 0
	     }

      CASE m_finish:
        TEST pflag
        THEN { writef("FINISH")
               RESULTIS 2
             }
        ELSE longjump(finp, finl) // Exit from z80emu
      
      CASE m_loadlvix:
      { // Push ((IX-128)>>1) + arg
        LET size = 3
        LET arg = getbb(pc+2)
        //writef("*nloadlvix: size=%n arg=%n*n", size, arg)
        IF arg > #x7F DO size, arg := 4, (arg & #x7F) + (getbb(pc+3)<<7) 
        //writef("loadlvix: size=%n IX-128=%n arg=%n*n", size, r_ix-128, arg)
        //writef("loadlvix: result %n*n", ((r_ix-128)>>1)+arg)
      
        TEST pflag
        THEN { writef("LOADLVIX %n", arg)
               RESULTIS size
	     }
        ELSE { r_sp := r_sp - 2
               putwb(r_sp, pptr+arg)
	       r_pc := r_pc+size
	       RESULTIS 0
             }
      }
      
      CASE m_loadlviy:
        writef("LOADLVIY")
        RESULTIS 2
      
      CASE m_multiply:
        writef("MULTIPLY")
        RESULTIS 2
      
      CASE m_divide:
        writef("DIVIDE");
        RESULTIS 2
      
      CASE m_rem:
        writef("REM")
        RESULTIS 2
    
      CASE m_lshift:
        writef("LSHIFT")
        RESULTIS 2
      
      CASE m_rshift:
        writef("RSHIFT")
        RESULTIS 2
      
      CASE m_eq:
        writef("EQ")
        RESULTIS 2
      
      CASE m_ne:
        writef("NE")
        RESULTIS 2
      
      CASE m_ls:
        writef("LS")
        RESULTIS 2
      
      CASE m_gr:
        writef("GR")
        RESULTIS 2
      
      CASE m_le:
        writef("LE")
        RESULTIS 2
      
      CASE m_ge:
        writef("GE")
        RESULTIS 2
      
      CASE m_rdivide:
        writef("RDIVIDE")
        RESULTIS 2
      
      CASE m_rrem:
        writef("RREM")
        RESULTIS 2
      
      CASE m_rlshift:
        writef("RLSHIFT")
        RESULTIS 2
      
      CASE m_rrshift:
        writef("RRSHIFT")
        RESULTIS 2
      
      CASE m_abs:
        writef("ABS")
        RESULTIS 2

      CASE m_linsearch:
      //       n K1 L1 ... Kn Ln Ldefault
      //       n is popped from the stack
      //       The other args are pointed to by HL
      //       The value to switch on is in DE
      //       n and the K values are always 2 bytes long
      //       The L values are even addresses encoded in 1 or 2 bytes.
        writef("LINSEARCH"); RESULTIS 2

      CASE m_sys:  // This is only used in the body of the function sys
                   // which is:
	  	   //    RST #x38  m_setlink3
		   //    RST #x38  m_sys
		   //    RST #x38  m_return
		   // It calls z80sys() with pptr, gptr setup and the
		   // first three arguments, if present, are accessible
		   // by getr(n_hl), getr(n_de) and getr(n_bc).
        TEST pflag
        THEN { writef("SYS")
               RESULTIS 2
	     }
        ELSE { z80sys()
               RESULTIS 0
	     }

    }
  }

  writef("RST #x%x2 is not implemented*n",addr)
  TEST pflag THEN RESULTIS 1
             ELSE { r_pc:=r_pc+1; RESULTIS 0 }
  
}

AND z80sys() BE SWITCHON getr(n_hl) INTO
{ DEFAULT:
    writef("*nERROR in z80sys: args %n %n %n %n*n",
         getr(n_hl), getr(n_de), getr(n_bc), getww(pptr+5))
    r_pc := r_pc+2
    RETURN

  CASE 11: // sawrch
    wrch(getr(n_de))
    r_pc := r_pc+2
    RETURN
}

AND doz80instrCB(pc) = VALOF
{ LET op = getbb(pc+1)
  LET bbb = op>>3 & 7   // bbb = 0 - 7
  LET rrr = op & 7      // rrr = B C D E H L (HL) A
  // 
  // op = 00 000 rrr    RLC rrr
  // op = 00 001 rrr    RRC rrr
  // op = 00 010 rrr    RL  rrr
  // op = 00 011 rrr    RR  rrr
  // op = 00 100 rrr    SLA rrr    
  // op = 00 101 rrr    SRA rrr
  // op = 00 110 rrr              Error
  // op = 00 111 rrr    SRL rrr

  // op = 01 bbb rrr    BIT bbb,rrr
  // op = 10 bbb rrr    RES bbb,rrr
  // op = 11 bbb rrr    SET bbb,rrr

  SWITCHON op>>6 INTO
  { DEFAULT:
    CASE #b00: 
      SWITCHON bbb INTO
      { DEFAULT:     RESULTIS zerror(pc, 2)

        CASE #b000:  RESULTIS z80RLCr(pc, rrr)  // RLC rrr
        CASE #b001:  RESULTIS z80RRCr(pc, rrr)  // RCC rrr
        CASE #b010:  RESULTIS z80RLr (pc, rrr)  // RL  rrr
        CASE #b011:  RESULTIS z80RRr (pc, rrr)  // RR  rrr
        CASE #b100:  RESULTIS z80SLAr(pc, rrr)  // SLA rrr
        CASE #b101:  RESULTIS z80SRAr(pc, rrr)  // SRA rrr
        CASE #b110:  RESULTIS zerror (pc, 2)
        CASE #b111:  RESULTIS z80SRLr(pc, rrr)  // SRL rrr
      } 
    CASE #b01: RESULTIS z80BITbr(pc, bbb, rrr)  // BIT bbb,rrr
    CASE #b10: RESULTIS z80RESbr(pc, bbb, rrr)  // RES bbb,rrr
    CASE #b11: RESULTIS z80SETbr(pc, bbb, rrr)  // SET bbb,rrr
  }
}

AND doz80instrXD(pc, r) = VALOF // For op DD or FD
{ // Deal wilh DD and FD instructions,
  // If op=DD r = n_ix
  // If op=FD r = n_iy
  LET d = getsbb(pc+2)
  SWITCHON getbb(pc+1) INTO
  { DEFAULT:   RESULTIS prbadop(pc)
    CASE #x09: RESULTIS z80ADD16rr(pc, r, n_bc)       // ADD r,BC
    CASE #x19: RESULTIS z80ADD16rr(pc, r, n_de)       // ADD r,DE
    CASE #x29: RESULTIS z80ADDrr(pc, r, r)            // ADD r,r
    CASE #x36: RESULTIS z80LDirdn(pc, r, d)           // LD (r+d),n
    CASE #x39: RESULTIS z80ADDrr(pc, r, n_sp)         // ADD r,SP
    CASE #x46: RESULTIS z80LDrird(pc, n_b, r, d)      // LD B,(r+d)
    CASE #x4E: RESULTIS z80LDrird(pc, n_b, r, d)      // LD C,(r+d)
    CASE #x56: RESULTIS z80LDrird(pc, n_d, r, d)      // LD D,(r+d)
    CASE #x5E: RESULTIS z80LDrird(pc, n_e, r, d)      // LD E,(r+d)
    CASE #x66: RESULTIS z80LDrird(pc, n_h, r, d)      // LD H,(r+d)
    CASE #x6E: RESULTIS z80LDrird(pc, n_l, r, d)      // LD L,(r+d)
    CASE #x70: RESULTIS z80LDirdr(pc, r, d, n_b)      // LD (r+d),B
    CASE #x71: RESULTIS z80LDirdr(pc, r, d, n_c)      // LD (r+d),C
    CASE #x72: RESULTIS z80LDirdr(pc, r, d, n_d)      // LD (r+d),D
    CASE #x73: RESULTIS z80LDirdr(pc, r, d, n_e)      // LD (r+d),E
    CASE #x74: RESULTIS z80LDirdr(pc, r, d, n_h)      // LD (r+d),H
    CASE #x75: RESULTIS z80LDirdr(pc, r, d, n_l)      // LD (r+d),L
    CASE #x77: RESULTIS z80LDirdr(pc, r, d, n_a)      // LD (r+d),A
    CASE #x7E: RESULTIS z80LDrird(pc, n_a, r, d)      // LD A,(r+d)
    CASE #x86: RESULTIS z80ADDrird(pc, n_a, r, d)     // ADD A,(r+d)
    CASE #x8E: RESULTIS z80ADCrird(pc, n_a, r, d)     // ADC A,(r+d)
    CASE #xA6: RESULTIS z80ANDird(pc, r, d)           // AND (r+d)
    CASE #xE1: RESULTIS z80POP16r(pc, r)              // POP r
    CASE #xE3: RESULTIS z80EXrr(n_isp, r)             // EX (SP),r
    CASE #xE5: RESULTIS z80PUSH16r(r)                 // PUSH r
  }
  RESULTIS 0
}

AND doz80instrXDCB(pc, r) = VALOF
{ // Deal with instructions:
  // DD CB d op=00-4F Shifts, r = n_ix
  // FD CB d op=00-4F Shifts, r = n_iy

  LET d  = getsbb(pc+2)
  LET op = getbb(pc+3)

  //         00-07              RLC (r+d)        r = IX or IY
  //         08-0F              RRC (r+d)        r = IX or IY
  //         10-17              RL  (r+d)        r = IX or IY
  //         18-1F              RR  (r+d)        r = IX or IY
  //         20-27              SLA (r+d)        r = IX or IY
  //         28-2F              SRA (r+d)        r = IX or IY
  //         30-37              S?? 
  //         38-3F              SRL (r+d)        r = IX or IY
  // XF CB d op=40-7F =40+8*b+r BIT b,(r+d)      r = IX or IY
  // XF CB d op=80-BF =80+8*b+r RES b,(r+d)      r = IX or IY
  // XF CB d op=C0-FF =C0+8*b+r SET b,(r+d)      r = IX or IY
  LET b = op>>6 & 7   // = 0 to 7
  SWITCHON op>>6 INTO
  { DEFAULT:
    CASE 0: 
      SWITCHON b INTO
      { DEFAULT: RESULTIS zerror(pc, 2)
        CASE 0:  RESULTIS z80RLCird(pc, r, d)  // RLC (r+d)
        CASE 1:  RESULTIS z80RCCird(pc, r, d)  // RCC (r+d)
        CASE 2:  RESULTIS z80RLird(pc, r, d)   // RL  (r+d)
        CASE 3:  RESULTIS z80RRird(pc, r, d)   // RR  (r+d)
        CASE 4:  RESULTIS z80SLAird(pc, r, d)  // SLA (r+d)
        CASE 5:  RESULTIS z80SRAird(pc, r, d)  // SRA (r+d)
        CASE 6:  RESULTIS zerror(pc, 4)       // S??
        CASE 7:  RESULTIS z80SRLird(pc, r, d)  // SRL (r+d)
      } 
    CASE 1: RESULTIS z80BITbird(pc, b, r, d)   // BIT b,(r+d)
    CASE 2: RESULTIS z80RESbird(pc, b, r, d)   // RES b,(r+d)
    CASE 3: RESULTIS z80SETbird(pc, b, r, d)   // SET b,(r+d)
  }
}

AND doz80instrED(pc, op) = VALOF
{ SWITCHON getbb(pc+1) INTO
  { DEFAULT:   RESULTIS prbadop(pc)
    CASE #x40: RESULTIS z80INrr(pc, n_b, n_ic)       // IN B,(C)
    CASE #x41: RESULTIS z80OUTrr(pc, n_ic, n_b)      // OUT (C),B
    CASE #x42: RESULTIS z80SBCrr(pc, n_hl, n_bc)     // SBC HL,BC
    CASE #x43: RESULTIS z80LD16innr(pc, getwb(pc+2), n_bc) // LD (nn),BC
    CASE #x44: RESULTIS z80NEG(pc)                   // NEG
    CASE #x45: RESULTIS z80RETN(pc)                  // RETN
    CASE #x46: RESULTIS z80IMk(pc, 0)                // IM 0  Set INT Mode 0
    CASE #x47: RESULTIS z80LDrr(pc, n_i, n_a)        // LD I,A
    CASE #x48: RESULTIS z80INrr(pc, n_c, n_ic)       // IN C,(C)
    CASE #x49: RESULTIS z80OUTrr(pc, n_ic, n_b)      // OUT (C),C
    CASE #x4A: RESULTIS z80ADC16rr(pc, n_hl, n_bc)   // ADC HL,BC
    CASE #x4B: RESULTIS z80LD16rinn(pc, n_bc, getwb(pc+2)) // LD BC,(nn)
    CASE #x4D: RESULTIS z80RETI(pc)                  // RETI
    CASE #x4F: RESULTIS z80LDrr(pc, n_r,n_a)         // LD R,A
    CASE #x50: RESULTIS z80INrr(pc, n_d, n_ic)       // IN D,(C)
    CASE #x51: RESULTIS z80OUTrr(pc, n_ic, n_d)      // OUT (C),D
    CASE #x52: RESULTIS z80SBC16rr(pc, n_hl, n_de)   // SBC HL,DE
    CASE #x53: RESULTIS z80LD16innr(pc, n_de)        // LD (nn),DE
    CASE #x56: RESULTIS z80IMk(pc, 1)                // IM 1  Set IN Mode 1
    CASE #x57: RESULTIS z80LDrr(pc, n_a, n_i)        // LD A,I
    CASE #x58: RESULTIS z80INrr(pc, n_e, n_ic)       // IN E,(C)
    CASE #x59: RESULTIS z80OUTrr(pc, n_ic, n_e)      // OUT (C),E
    CASE #x5A: RESULTIS z80ADC16rr(pc, n_hl, n_de)   // ADC HL,DE
    CASE #x5B: RESULTIS z80LD16rinn(pc, n_de)        // LD DE,(nn)
    CASE #x5E: RESULTIS z80IMk(pc, 2)                // IM 2  Set INT Mode 2
    CASE #x5F: RESULTIS z80LDrr(pc, n_a, n_r)        // LD A,R
    CASE #x60: RESULTIS z80INrr(pc, n_h, n_ic)       // IN H,(C)
    CASE #x61: RESULTIS z80OUTrr(pc, n_ic, n_h)      // OUT (C),H
    CASE #x62: RESULTIS z80SBC16rr(pc, n_hl, n_hl)   // SBC HL,HL
    CASE #x67: RESULTIS z80RRD(pc)                   // RRD
    CASE #x68: RESULTIS z80INrr(pc, n_l, n_ic)       // IN L,(C)
    CASE #x69: RESULTIS z80OUTrr(pc, n_ic, n_l)      // OUT (C),L
    CASE #x6A: RESULTIS z80ADC16rr(pc, n_hl, n_hl)   // ADC HL,HL
    CASE #x6F: RESULTIS z80RLD(pc)                   // RLD
    CASE #x70: RESULTIS z80INrr(pc, n_f, n_ic)       // IN F,(C)
    CASE #x72: RESULTIS z80SBC16rr(pc, n_hl, n_sp)   // SBC HL,SP
    CASE #x73: RESULTIS z80LD16innr(pc, n_sp)        // LD (nn),SP
    CASE #x79: RESULTIS z80OUTrr(pc, n_ic, n_a)      // OUT (C),A
    CASE #x7A: RESULTIS z80ADC16rr(pc, n_hl, n_sp)   // ADC HL,SP
    CASE #x7B: RESULTIS z80LD16rinn(pc, n_sp)        // LD SP,(nn)
    CASE #xA0: RESULTIS z80LDI(pc)                   // LDI
    CASE #xA1: RESULTIS z80CPI(pc)                   // CPI
    CASE #xA2: RESULTIS z80INI(pc)                   // INI
    CASE #xA3: RESULTIS z80OUTI(pc)                  // OUTI
    CASE #xA8: RESULTIS z80LDD(pc)                   // LDD
    CASE #xA9: RESULTIS z80CPD(pc)                   // CPD
    CASE #xAA: RESULTIS z80IND(pc)                   // IND
    CASE #xAB: RESULTIS z80OUTD(pc)                  // OUTD    CASE #xB0: RESULTIS z80LDIR(pc)                  // LDIR
    CASE #xB1: RESULTIS z80CPIR(pc)                  // CPIR
    CASE #xB2: RESULTIS z80INIR(pc)                  // INIR
    CASE #xB3: RESULTIS z80OTIR(pc)                  // OTIR
    CASE #xB8: RESULTIS z80LDDR(pc)                  // LDDR
    CASE #xB9: RESULTIS z80CPDR(pc)                  // CPDR
    CASE #xBA: RESULTIS z80INDR(pc)                  // INDR
    CASE #xBB: RESULTIS z80OTDR(pc)                  // OTDR
  //CASE #xC1: RESULTIS z80MULUBrr(pc, n_a, n_b)     // MULUB A,B
                                                     // note C1+8*r
  //CASE #xC3: RESULTIS z80MULUWrr(pc, n_hl, n_bc)   // MULUW HL,BC
  //CASE #xF3: RESULTIS z80MULUWrr(pc, n_hl, n_sp)   // MULUW HL,SP
  }
  RESULTIS 0
}

AND prbadop(pc) = VALOF
{ writef("*nInstruction at pc=%n not yet implemented: %x2 %x2 %x2 %x2*n",
           pc, getbb(pc), getbb(pc+1), getbb(pc+2), getbb(pc+3))
  val :=val+1
  RESULTIS 1
}

AND r2str(r) = VALOF SWITCHON r INTO
{ DEFAULT:    writef("r2str: Bad r=%n*n", r)
              RESULTIS "??"

  CASE n_a:   RESULTIS "A"
  CASE n_f:   RESULTIS "F"
  CASE n_b:   RESULTIS "B"
  CASE n_c:   RESULTIS "C"
  CASE n_d:   RESULTIS "D"
  CASE n_e:   RESULTIS "E"
  CASE n_h:   RESULTIS "H"
  CASE n_l:   RESULTIS "L"
  CASE n_af:  RESULTIS "AF"
  CASE n_bc:  RESULTIS "BC"
  CASE n_de:  RESULTIS "DE"
  CASE n_hl:  RESULTIS "HL"
  CASE n_sp:  RESULTIS "SP"
  CASE n_pc:  RESULTIS "PC"
  CASE n_af1: RESULTIS "AF'"
  CASE n_bc1: RESULTIS "BC'"
  CASE n_de1: RESULTIS "DE'"
  CASE n_hl1: RESULTIS "HL'"
  CASE n_ic:  RESULTIS "(C)"
  CASE n_ibc: RESULTIS "(BC)"
  CASE n_ide: RESULTIS "(DE)"
  CASE n_ihl: RESULTIS "(HL)"
  CASE n_isp: RESULTIS "(SP)"
  CASE n_ix:  RESULTIS "IX"
  CASE n_iy:  RESULTIS "IY"
  CASE n_m:   RESULTIS "M"
  CASE n_nz:  RESULTIS "NZ"
  CASE n_nc:  RESULTIS "NC"
  CASE n_p:   RESULTIS "P"
  CASE n_pe:  RESULTIS "PE"
  CASE n_po:  RESULTIS "PO"
  CASE n_z:   RESULTIS "z"
}

AND getr(rn) = VALOF SWITCHON rn INTO
{ DEFAULT:    writef("*ngetr: Bad register number %n, %s*n", rn, r2str(rn))
              RESULTIS 0

  CASE n_a:   RESULTIS r_a & #xFF
  CASE n_b:   RESULTIS r_b & #xFF
  CASE n_c:   RESULTIS r_c & #xFF
  CASE n_d:   RESULTIS r_d & #xFF
  CASE n_e:   RESULTIS r_e & #xFF
  CASE n_h:   RESULTIS r_h & #xFF
  CASE n_l:   RESULTIS r_l & #xFF
  CASE n_f:   RESULTIS r_f & #xFF
  
  CASE n_bc:  RESULTIS r_b<<8 | r_c
  CASE n_de:  RESULTIS r_d<<8 | r_e
  CASE n_hl:  RESULTIS r_h<<8 | r_l
  CASE n_af:  RESULTIS r_a<<8 | r_f
  
  CASE n_bc1: RESULTIS r_b1<<8 | r_c1
  CASE n_de1: RESULTIS r_d1<<8 | r_e1
  CASE n_hl1: RESULTIS r_h1<<8 | r_l1
  CASE n_af1: RESULTIS r_a1<<8 | r_f1
    
  CASE n_ibc: RESULTIS getbb(r_b<<8 | r_c)
  CASE n_ide: RESULTIS getbb(r_d<<8 | r_e)
  CASE n_ihl: RESULTIS getbb(r_h<<8 | r_l)
  
  CASE n_ix:  RESULTIS r_ix
  CASE n_iy:  RESULTIS r_iy
  CASE n_sp:  RESULTIS r_sp
  CASE n_pc:  RESULTIS r_pc
  CASE n_isp: RESULTIS getbb(r_sp)
}

AND putr(rn, val) BE SWITCHON rn INTO
{ DEFAULT:    writef("*nBad register number %n, %s*n", rn, r2str(rn))
              RETURN

  CASE n_a:   r_a := val; RETURN
  CASE n_b:   r_b := val; RETURN
  CASE n_c:   r_c := val; RETURN
  CASE n_d:   r_d := val; RETURN
  CASE n_e:   r_e := val; RETURN
  CASE n_h:   r_h := val; RETURN
  CASE n_l:   r_l := val; RETURN
  CASE n_f:   r_f := val; RETURN
  
  CASE n_bc:  r_b, r_c := val>>8, val & #xFF; RETURN
  CASE n_de:  r_d, r_e := val>>8, val & #xFF; RETURN
  CASE n_hl:  r_h, r_l := val>>8, val & #xFF; RETURN
  
  
  CASE n_ibc: putwb(r_b<<8 | r_c & #xFF, val); RETURN
  CASE n_ide: putwb(r_d<<8 | r_d & #xFF, val); RETURN
  CASE n_ihl: putwb(r_h<<8 | r_l & #xFF, val); RETURN
  
  CASE n_ix:  r_ix := val;      RETURN
  CASE n_iy:  r_iy := val;      RETURN
  CASE n_sp:  r_sp := val;      RETURN
  CASE n_isp: putwb(r_sp, val); RETURN
}

AND zerror(pc, n) = VALOF
{ writef("*nUnimplemented instruction at %n: ", pc)
  //pflag := TRUE
  //n := prz80instr(pc)
  newline()
  RESULTIS n
}


AND z80ADCrr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, x_adc(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 1
     }

AND z80ADC16rr(pc,r,s) = VALOF TEST pflag
THEN { writef("ADC %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, x_adc(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 1
     }

AND z80ADDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, x_add(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ADD16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("ADD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, x_add16(getr(r), getr(s)))
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80ANDrr(pc,r,s) = VALOF TEST pflag
THEN { writef("AND %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, x_and(getr(r), getr(s)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 1
     }

AND z80XORr(pc,r) = VALOF TEST pflag
THEN { writef("XOR %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(n_a, x_xor(r_a, getr(r)))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80CCP(pc) = VALOF TEST pflag
THEN { writef("CCP")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPL(pc) = VALOF TEST pflag
THEN { writef("CPL")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPr(pc, r) = VALOF TEST pflag
THEN { writef("CP %s", r2str(r))
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80DAA(pc) = VALOF TEST pflag
THEN { writef("DAA")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80DAAr(pc, r) = VALOF TEST pflag
THEN { writef("DAA %s", r2str(r))
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80DECr(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(r, getr(r)-1)
       RESULTIS 0
     }

AND z80DEC16r(pc, r) = VALOF TEST pflag
THEN { writef("DEC %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(r, getr(r)-1)
       RESULTIS 0
     }

AND z80DI(pc) = VALOF TEST pflag
THEN { writef("DI")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80EI(pc) = VALOF TEST pflag
THEN { writef("EI")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80EXX(pc) = VALOF TEST pflag
THEN { writef("EXX")
       RESULTIS 1
     }
ELSE { x_exx(pc)
       r_pc := r_pc+1
       RESULTIS 0
     }

AND z80EXrr(pc, r, s) = VALOF TEST pflag
THEN { writef("EX %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { LET t = ?
       // This implements one of
       // EB:      EX DE,HL
       // 08:      EX AF,AF'
       // E3:      EX (SP),HL
       // DD E3:   EX (SP),IX
       // FD E3:   EX (SP),IY     
       t := getr(s); putr(s, getr(r)); putr(r, t)
       TEST s=n_ix | s=n_iy THEN r_pc := r_pc+2
                            ELSE r_pc := r_pc+1
       RESULTIS 0
     }
     
AND z80HALT(pc) = VALOF TEST pflag
THEN { writef("HALT")
       RESULTIS 1
     }
ELSE { writef("*nHALT instructions encountered*n")
       abort(999)
       r_pc := pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80INCr(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80INC16r(pc, r) = VALOF TEST pflag
THEN { writef("INC %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(r, getr(r) + 1)
       r_pc :=r_pc + 1
       RESULTIS 0
     }

AND z80LDinnr(pc, r) = VALOF
{ LET nn = getwb(pc+1)
  TEST pflag THEN writef("LD (%n),%s", nn, r2str(r))
             ELSE putwb(nn, getr(r))
  RESULTIS 3
}

AND z80LD16innr(pc, r) = VALOF
{ LET nn = getwb(pc+1)
  TEST pflag THEN writef("LD (%n),%s", nn, r2str(r))
             ELSE putwb(nn, getr(r))
  RESULTIS 3
}

AND z80LDirdn(pc, r, d) = VALOF
{ // Instruction: LD (IX+d),n  or  LD (IY+d),n
  // Code:        DD 36  d  n  or  FD 36  d  n
  // r=n_ix or n_iy
  LET n = getbb(pc+3)
  TEST pflag
  THEN { TEST d<0
         THEN writef("LD (%s-%n),%n", r2str(r), -d, n)
         ELSE writef("LD (%s+%n),%n", r2str(r),  d, n)
         RESULTIS 4
       }
  ELSE { LET a = getr(r)+d
         putbb(a, n)
	 r_pc := pc+4
	 RESULTIS 0
       }
}

AND z80LDrr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, getr(s))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80LD16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("LD %s,%s", r2str(r), r2str(s))
       RESULTIS 1
     }
ELSE { putr(r, getr(s))
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80NOP(pc) = VALOF TEST pflag
THEN { writef("NOP")
       RESULTIS 1
     }
ELSE { r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80POP16r(pc, r) = VALOF TEST pflag
THEN { writef("POP %s", r2str(r))
       RESULTIS 1
     }
ELSE { putr(r, getwb(r_sp))
       r_sp := r_sp+2
       TEST r=n_ix | r=n_iy THEN r_pc := pc+2
                            ELSE r_pc := pc+1
       RESULTIS 0
     }

AND z80RRCA(pc) = VALOF TEST pflag
THEN { writef("RRCA")
       RESULTIS 1
     }
ELSE { r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80RET(pc) = VALOF TEST pflag
THEN { writef("RET")
       RESULTIS 1
     }
ELSE { r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80RETN(pc) = VALOF TEST pflag
THEN { writef("RETN")
       RESULTIS 1
     }
ELSE { r_pc := getwb(r_sp)
       r_sp := r_sp+2
       RESULTIS 0
     }

AND z80ADDrn(pc,r) = VALOF
{ IF pflag DO
  { writef("ADD %s,%n", r2str(r), getbb(pc+1))
    RESULTIS 2
  }
  //abort(6534)
  putr(r, getr(r) + getbb(pc+1))
  //abort(6535)
  
  r_pc :=r_pc+2 & #xFFFF
  RESULTIS 0
}

AND z80SBC16rr(pc, r, s) = VALOF TEST pflag
THEN { writef("SBC %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { LET res = getr(r)+getr(s)
       IF (r_f & fb_c)>0 DO res := res-1
       putr(r, res)
       r_pc := pc+2
       RESULTIS 0
     }

AND z80ANDn(pc) = VALOF TEST pflag
THEN { writef("AND %n", getbb(pc+1))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80ANDr(pc,r) = VALOF TEST pflag
THEN { writef("AND %s", r2str(r))
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80BITbr(pc, b, r) = VALOF TEST pflag
THEN { writef("BIT %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80CPD(pc) = VALOF TEST pflag
THEN { writef("CPD")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPDR(pc) = VALOF TEST pflag
THEN { writef("CPDR")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPI(pc) = VALOF TEST pflag
THEN { writef("CPI")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPIR(pc) = VALOF TEST pflag
THEN { writef("CPIR")
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80CPn(pc) = VALOF TEST pflag
THEN { writef("CP %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80DJNZo(pc) = VALOF TEST pflag
THEN { writef("DJNZo %n", getsbb(pc+1))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80IMk(pc, k) = VALOF TEST pflag
THEN { writef("IM %n", k)
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80IND(pc) = VALOF TEST pflag
THEN { writef("IND")
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80INDR(pc) = VALOF TEST pflag
THEN { writef("INDR")
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80INI(pc) = VALOF TEST pflag
THEN { writef("INI")
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80INIR(pc) = VALOF TEST pflag
THEN { writef("INIR")
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80INCird(pc, r) = VALOF TEST pflag
THEN { writef("INC "); prird(r, getsbb(pc+2))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80INrr(pc, r, s) = VALOF TEST pflag
THEN { writef("IN %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80LDrn(pc, r) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), getbb(pc+1))
       RESULTIS 2
     }
ELSE { putr(r, getbb(pc+1))
       r_pc := r_pc+2 & #xFFFF
       RESULTIS 2
     }

AND z80LD16rnn(pc, r) = VALOF TEST pflag
THEN { writef("LD %s,%n", r2str(r), getwb(pc+1))
       RESULTIS 3
     }
ELSE { putr(r, getwb(pc+1))
       r_pc := r_pc+3 & #xFFFF
       RESULTIS 0
     }

AND z80ORr(pc, r) = VALOF TEST pflag
THEN { writef("OR %s", r2str(r))
       RESULTIS 1
     }
ELSE { r_a := r_a | getr(r)
       r_f := r_f & fb_np
       r_pc := r_pc+1 & #xFFFF
       RESULTIS 0
     }

AND z80OUTrr(pc, r, s) = VALOF TEST pflag
THEN { writef("OUT %s,%s", r2str(r), r2str(s))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80PUSHr(pc, r) = VALOF TEST pflag
THEN { writef("PUSH %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RCCr(pc, r) = VALOF TEST pflag
THEN { writef("RCC %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RESbr(pc, b, r) = VALOF TEST pflag
THEN { writef("RES %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RLCr(pc, r) = VALOF TEST pflag
THEN { writef("RLC %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RRCr(pc, r) = VALOF TEST pflag
THEN { writef("RLC %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RLr(pc, r) = VALOF TEST pflag
THEN { writef("RL %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80RRr(pc, r) = VALOF TEST pflag
THEN { writef("RR %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET w = getr(r)
       LET c = r_fc
       r_fc := (w&1)>0 -> fb_c, 0
       w := w>>1
       IF c DO w := w + #x80
       putr(r, w)
       r_fs := w<0 -> fb_s, 0
       r_fz := w=0 -> fb_z, 0
       r_fh := 0
       r_fv := parity(w)
       r_fn := 0
       r_pc := r_pc+2
       RESULTIS 0
     }

AND z80SLAr(pc, r) = VALOF TEST pflag
THEN { writef("SLA %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80SRAr(pc, r) = VALOF TEST pflag
THEN { writef("SRA %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }

AND z80SRLr(pc, r) = VALOF TEST pflag
THEN { writef("SRL %s", r2str(r))
       RESULTIS 2
     }
ELSE { LET w = getr(r)
       r_fc := 0
       IF (w&1)>0 DO r_fc := fb_c
       w := (w>>1)&#x7F
       r_fs := 0
       IF w=0 DO r_fz := fb_z
       r_fh := 0
       r_fv := parity(w)
       r_fn := 0
       putr(r, w)
       r_pc := r_pc+2
       RESULTIS 0
     }


AND z80SETbr(pc, b, r) = VALOF TEST pflag
THEN { writef("SET %n, %s", r2str(r))
       RESULTIS 2
     }
ELSE { RESULTIS zerror(pc, 2)
     }


AND z80ADCrird(pc, r, x) = VALOF TEST pflag
THEN { writef("ADC %s,", r2str(r)); prird(x, getsbb(pc+2))
       RESULTIS 3
     }
ELSE { LET a = getr(x) + getsbb(pc+2)
       putr(r, getr(r) + getbb(a) + (fb_c & r_f))
       r_pc := r_pc+3 & #xFFFF
       RESULTIS 3
     }



AND z80ADDrixd(pc,r,d) = VALOF TEST pflag
THEN { writef("ADC %s,(IX+%n)", r2str(r), d)
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }


AND z80ADDriyd(pc,r,d) = VALOF TEST pflag
THEN { writef("ADC %s,(IX+%n)", r2str(r), d)
       RESULTIS 1
     }
ELSE { RESULTIS zerror(pc, 1)
     }

AND z80LDirdr(pc, r, d, s) = VALOF TEST pflag
THEN { writef("LD "); prird(r, d); writef(",%s", r2str(s)); 
       RESULTIS 3
     }
ELSE { LET a = getr(r)+d
       putbb(a, getr(s))
       r_pc := r_pc+3
       RESULTIS 0
     }

AND z80LDrird(pc, r, s, d) = VALOF TEST pflag
THEN { writef("LD %s,", r2str(r)); prird(s, d)
       RESULTIS 3
     }
ELSE { LET a = getbb(getr(s)+d)
//db("Loading reg %s with (%s+%n) value %n*n", r2str(r), r2str(s), d, a)
       putr(r, a)
       r_pc := pc+3
       //db("Returning from z80LDrird: r_pc=%n*n", r_pc)
       RESULTIS 0
     }

AND z80LDrinn(pc, r) = VALOF
{ LET w = getwb(pc+1)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), w)
         RESULTIS 3
       }
  ELSE { putr(r, getwb(w))
         r_pc := pc+3 & #xFFFF
         RESULTIS 0
       }
}

AND z80LD16rinn(pc, r) = VALOF
{ LET w = getwb(pc+1)
  TEST pflag
  THEN { writef("LD %s,(#x%x4)", r2str(r), w)
         RESULTIS 3
       }
  ELSE { putr(r, getwb(w))
         r_pc := r_pc+3 & #xFFFF
         RESULTIS 3
       }
}

AND z80RLCird(pc, r, d) = VALOF TEST pflag
THEN { writef("RLC "); prird(r, d)
       RESULTIS 4
     }
ELSE { RESULTIS zerror(pc, 4)
     }

// Arithemetic and logical functions

AND setparity(w) = VALOF
{ IF w=0 DO { r_f := r_f & fb_np; RETURN } // set P=0
  w := w & (w-1)
  IF w=0 DO { r_f := r_f | fb_p; RETURN }  // Set P=1
  w := w & (w-1)  
} REPEAT

AND x_adc(a, b) = VALOF
{ LET res = a + b
  IF (r_f & fb_c)>0 DO res := res + 1
  RESULTIS res
}

AND x_adc16(a, b) = VALOF
{ LET res = a + b
  IF (r_f & fb_c)>0 DO res := res + 1
  RESULTIS res
}

AND x_add(a, b) = VALOF
{ LET res = a + b
  RESULTIS res
}

AND x_xor(a, b) = VALOF
{ LET res = (a XOR b) &#xFFFF
  RESULTIS res
}

AND x_add16(a, b) = VALOF
{ LET res = a + b
  RESULTIS res
}

AND x_exx(pc) BE
{ LET t = ?
  t := r_a1; r_a1:=r_a; r_a:=t 
  t := r_f1; r_f1:=r_f; r_f:=t 
  t := r_b1; r_b1:=r_b; r_b:=t 
  t := r_c1; r_c1:=r_c; r_c:=t 
  t := r_d1; r_d1:=r_d; r_d:=t 
  t := r_e1; r_e1:=r_e; r_e:=t 
  t := r_h1; r_h1:=r_h; r_h:=t 
  t := r_l1; r_l1:=r_l; r_l:=t
}

AND db(mes, a,b,c,d,e) BE
{ writef("*ndb: "); writef(mes, a,b,c,d,e)
  writef("*nr_pc=%n pptr=%n faultcode=%n singlestep=%n brkstep=%n mem=%n pflag=%n",
            r_pc,   pptr,   faultcode,   singlestep,   brkstep,   mem,   pflag)
  writef("*n  100: ")
  FOR a = 100 TO 118 DO writef(" %x2", getbb(a))
  FOR i = 0 TO 19 DO
  { IF i MOD 10 = 0 DO writef("*n%i5: ", pptr+i)
    writef(" %x4", getww(pptr+i))
  }
  newline()
  indentpos := 0
  prstate()
  abort(1000)
}
