/*
This program converts a raster file generated by rastsys to Postscript.

Implemented by Martin Richards (c) Sept 2016

Usage: "FROM,SCALE,TO/K,ML,MH,MG,FL,FH,FG,DPI/K,
        INCL/K,A5/S,A4/S,A3/S,A2/S,A1/S,A0/S"

FROM   specifies the input file, default RASTER
SCALE  Specifies the scaling factor as a percentage
TO     Specifies the desination file, default RASTER.ps
ML     Specifies the lowest memory address
MH     Specifies the highest memory address
MG     Specifies the memory grid line spacing
FL     Specifies the lowest instruction count
FH     Specifies the highest instruction count
FG     Specifies the instruction gridline spacing
DPI    Specifies the dots per inch, default 300
INCL   Specifies a file to include, default psincl
A5     Specifies A5 paper size
A4     Specifies A4 paper size, the default
A3     Specifies A3 paper size
A2     Specifies A2 paper size
A1     Specifies A1 paper size
a0     Specifies A0 paper size

*/

SECTION "RAST2PS"

GET "libhdr"

GLOBAL
{ stdin:ug
  stdout
  fromfilename
  fromstream
  rasterfilename
  rasterstream
  
  rastv
  scale
  ml
  mh
  mg
  FLT m2yfac
  fl
  fh
  fg
  FLT f2xfac
  inclfilename

  fcount

  ygrid
  ygrid1
  maxfcount  // The total number of instructions executed
  maxaddress // The maximum Cintcode byte address accessed
  kval       // Number of instructions per raster line
  sval       // Number of address bytes per pixel
  ch
  stdin
  stdout

  asize
  Afac
  DPI

  Tmarg
  Bmarg
  Lmarg
  Rmarg

  YINSx10
  XINSx10

  YLEN
  XLEN
}

MANIFEST
{
  // A4 paper size
  A4H = 117 // A4 Height (Inches x 10)
  A4W = 83  // A4 Width (Inches x 10)

  // A4 margins
  A4Tmarg =  5 // Top  (Inches x 10)
  A4Bmarg = 10 // Bottom
  A4Lmarg =  5 // Left
  A4Rmarg = 10 // Right

  A5fac =  71 // percent
  A4fac = 100 // percent
  A3fac = 141 // percent
  A2fac = 200 // percent
  A1fac = 282 // percent
  A0fac = 400 // percent
}
  

LET initpsraster() BE // scale is a percentage
{ LET h,t,u = scale/100, scale/10 REM 10, scale REM 10
  LET xmax = A4H*Afac*72/1000
  LET ymax = A4W*Afac*72/1000

  selectoutput(rasterstream)

  writes("%!PS-Adobe-0.0*n")
  writef("%%%%BoundingBox: %n %n %n %n*n",
           18*scale/100, 30*scale/100,
           ymax*scale/100, xmax*scale/100)

  writef("save*n")
  writef("%n.%n%n %n.%n%n scale*n", h,t,u, h,t,u)

  writef("/YMAX %n def*n", A4W*Afac*72/1000)
  writef("/XMAX %n def*n", A4H*Afac*72/1000)
  writef("/DPI %n def*n", DPI)
  writef("/FL %n 1000000 div def*n", fl)
  writef("/FH %n 1000000 div def*n", fh)
  writef("/ML %n 1000 div def*n", ml)
  writef("/MH %n 1000 div def*n", mh)
  writef("/NMAX %n def*n", XLEN) // Number of raster lines
  writef("/N 0 def*n")

  writef("/SC{dup stringwidth pop -2 div -5 rmoveto show}bind def*n")
  writef("/SL{dup stringwidth pop neg -5 rmoveto show}bind def*n")

  writef("/F1 /Helvetica  findfont 12 scalefont def*n")
  writef("/F2 /Times      findfont 12 scalefont def*n")
  writef("/F3 /Times-Bold findfont 14 scalefont def*n")

  writef("F1 setfont*n")
  writef("YMAX 0 translate*n")
  writef("90 rotate*n")
  writes("% Landscape A%n, origin bottom left, unit=pt*n", asize)

  writef("/FMAX XMAX %n sub def*n", A4Tmarg*72*Afac/1000)
  writef("/MMAX YMAX %n sub def*n", A4Lmarg*72*Afac/1000)

  writef("/F0 %n def*n", A4Bmarg*72*Afac/1000)
  writef("/M0 %n def*n", A4Rmarg*72*Afac/1000)
  writef("/FFAC FMAX F0 sub FH FL sub div def*n")
  writef("/FBAS F0 FL FFAC mul sub def*n")
  writef("/MFAC MMAX M0 sub MH ML sub div def*n")
  writef("/MBAS M0 ML MFAC mul sub def*n")
  
  writef("/FSCALE {FFAC mul FBAS add} bind def*n")
  writef("/MSCALE {MFAC mul MBAS add} bind def*n")
  writef("/SCALE {exch FSCALE exch MSCALE} def*n")
  writef("/MVT {SCALE moveto} def*n")

  writef("/PDL {save 3 index 3 index SCALE translate*n")
  writef("/A 2 index 10 mul def /B 2 index 17.3 mul def*n")
  writef("0 0 moveto A B lineto*n")
  writef("0 6 moveto 0 0 lineto 5.2 3 lineto stroke*n")
  writef("A B moveto 5 9 rmoveto 4 index SC*n")
  writef("restore pop pop pop pop*n")
  writef("} bind def*n")

  writef("/PUL {save 3 index 3 index SCALE translate*n")
  writef("/A 2 index 10 mul def /B 2 index -17.3 mul def*n")
  writef("0 0 moveto A B lineto*n")
  writef("0 -6 moveto 0 0 lineto 5.2 -3 lineto stroke*n")
  writef("A B moveto 5 -9 rmoveto 4 index SC*n")
  writef("restore pop pop pop pop*n")
  writef("} bind def*n")

  writef("/PDR {save 3 index 3 index SCALE translate*n")
  writef("/A 2 index -10 mul def /B 2 index 17.3 mul def*n")
  writef("0 0 moveto A B lineto*n")
  writef("0 6 moveto 0 0 lineto -5.2 3 lineto stroke*n")
  writef("A B moveto -5 9 rmoveto 4 index SC*n")
  writef("restore pop pop pop pop*n")
  writef("} bind def*n")

  writef("/PUR {save 3 index 3 index SCALE translate*n")
  writef("/A 2 index -10 mul def /B 2 index -17.3 mul def*n")
  writef("0 0 moveto A B lineto*n")
  writef("0 -6 moveto 0 0 lineto -5.2 -3 lineto stroke*n")
  writef("A B moveto -5 -9 rmoveto 4 index SC*n")
  writef("restore pop pop pop pop*n")
  writef("} bind def*n")

  writef("/pl { dup length 8 mul 1 true [0 1 1 0 0 N neg] 5 -1 roll*n")
  writef("      imagemask*n")
  writef("/N N 1 add def*n")
  writef("    } bind def*n")

  writef("/TITLE {*n")
  writef("FMAX F0 sub 2 div F0 add M0 40 sub moveto SC} bind def*n")

  IF inclfilename DO
  { LET psfile = findinput(inclfilename)
    IF psfile DO
    { LET oldin = input()
      LET ch = 0
      selectinput(psfile)
      { LET ch = rdch()
        IF ch=endstreamch BREAK
        wrch(ch)
      } REPEAT
      newline()
      endread()
      selectinput(oldin)
    }
  }    

  writef("F1 setfont*n")

  // Write the memory axis grid coordinates
  { LET g = (ml/mg)*mg
    WHILE g <= mh DO
    { writef("M0 %n MSCALE moveto (%nK ) SL*n", g/1000, g/1000)
      g := g + mg
    }
  }

  // Write the instruction count axis coordinates
  { LET g = (fl/fg)*fg
    WHILE g <= fh DO
    { LET m = g / 100_000 / 10
      AND d = g /  10_000 MOD 100
      writef("%n.%n FSCALE M0 12 sub moveto (%n.%2zM) SC*n", m, d, m, d)
      g := g + fg
    }
  }

  writef("F0 M0 translate*n")
  writef("72 DPI div dup scale*n")

  selectoutput(stdout)
}

AND setline(c) BE
  FOR i=0 TO YLEN+1 DO rastv!i := c


AND wrline() BE
{ LET byte, count = 0, 0

  { LET i=ygrid1
    WHILE i<YLEN-1 DO { rastv!i := 1; rastv!(i+1) := 1; i := i+ygrid }
  }
  selectoutput(rasterstream)
  writef("<*n")

  FOR i=0 TO YLEN DO
  { byte := (byte<<1) | rastv!i
    count := count+1
    IF count REM 8 = 0 DO { writehex(byte, 2); byte := 0 }
    IF count REM 256 = 0 DO newline()
  }
  WHILE(byte ~= 0)
  { byte := (byte<<1)
    count := count+1
    IF count REM 8 = 0 DO { writehex(byte, 2); byte := 0 }
    IF count REM 256 = 0 DO newline()
  }
  writef(">pl")
  selectoutput(stdout)
}

AND start() = VALOF
{ LET argv = VEC 50
  LET form = "FROM,SCALE/N,TO/K,ML/N,MH/N,MG/N,FL/N,FH/N,FG/N,*
             *DPI/K/N,INCL/K,A5/S,A4/S,A3/S,A2/S,A1/S,A0/S"

  UNLESS rdargs(form, argv, 50) DO
  { writef("Bad args for RAST2PS*n")
    RESULTIS 20
  }

  fromfilename   := "RASTER"
  rasterfilename := "RASTER.ps"
  inclfilename := 0
  
  ml, mh, mg := -1, -1, -1 // Unsprecified values
  ml, mh, mg := -1, -1, -1

  IF argv!0 DO fromfilename := argv!0        // FROM
  scale := 80
  IF argv!1 DO scale := !(argv!1)            // SCALE/N
  IF argv!2 DO rasterfilename := argv!2      // TO/K

  IF argv!3 DO ml := !(argv!3)               // ML/N
  IF argv!4 DO mh := !(argv!4)               // MH/N
  IF argv!5 DO mg := !(argv!5)               // MG/N
  IF argv!6 DO fl := !(argv!6)               // FL/N
  IF argv!7 DO fh := !(argv!7)               // FH/N
  IF argv!8 DO fg := !(argv!8)               // FG/N

  // Choose defaults
  Afac, asize := A4fac, 4   // Choose A0, A1, A2, A3, A4 or A5
  DPI  :=   300            // Choose 300 or 600

  IF argv!9  DO DPI := !(argv!9)             // DPI/K/N
  //inclfilename := "psincl"
  IF argv!10 DO inclfilename := argv!10      // INCL/K

  IF argv!11 DO Afac, asize := A5fac, 5      // A5/S
  IF argv!12 DO Afac, asize := A4fac, 4      // A4/S
  IF argv!13 DO Afac, asize := A3fac, 3      // A3/S
  IF argv!14 DO Afac, asize := A2fac, 2      // A2/S
  IF argv!15 DO Afac, asize := A1fac, 1      // A1/S
  IF argv!16 DO Afac, asize := A0fac, 0      // A0/S

  // diagram size in inches x 10  
  YINSx10 := (A4W-A4Lmarg-A4Rmarg)*Afac/100 // allow margins on both sided
  XINSx10 := (A4H-A4Tmarg-A4Bmarg)*Afac/100 // allow margins on top and bottom

  // Margin sizes in pixels
  Tmarg := (DPI*A4Tmarg*Afac/1000)
  Bmarg := (DPI*A4Bmarg*Afac/1000)
  Lmarg := (DPI*A4Lmarg*Afac/1000)
  Rmarg := (DPI*A4Rmarg*Afac/1000)

  // diagram size in pixels
  YLEN := YINSx10*DPI/10  // pixels in the memory address direction
  XLEN := XINSx10*DPI/10  // pixels in the fcount direction

  fcount := 0
  maxaddress:= 0

  stdin := input()
  stdout := output()

  rastv, fromstream, rasterstream := 0, 0, 0

  fromstream := findinput(fromfilename)
  IF fromstream=0 DO
  { writef("Trouble with file: %s*n", fromfilename)
    RESULTIS 20
  }

  selectinput(fromstream)

  rasterstream := findoutput(rasterfilename)

  IF rasterstream=0 DO
  { writef("Trouble with file: %s*n", rasterfilename)
    endread()
    RESULTIS 20
  }

  scan()

  endstream(rasterstream)
  rasterstream := 0
  
  selectoutput(stdout)

  endstream(fromstream)
  fromstream := 0
  
  selectinput(stdin)

  freevec(rastv)

  RESULTIS 0
}

AND mark(a, b) BE
{ LET i = FIX(FLOAT(a*sval - ml) * m2yfac)
  LET j = FIX(FLOAT(b*sval - ml) * m2yfac)
  IF i>YLEN | j<0 RETURN
  IF i<0 DO i := 0
  IF j>YLEN DO j := YLEN
  IF j<i DO j := i
  FOR p = i TO j DO rastv!p := 1
  //rastv!(j+1) := 1
}

AND rdn() = VALOF
{ LET res = 0
  WHILE '0'<=ch<='9' DO { res := 10*res + ch - '0'; ch := rdch() }
  RESULTIS res
}

AND scan() = VALOF
{ LET a, b, tally = 0, 0, 0

  maxfcount, maxaddress, kval, sval := 0, 0, 0, 0
  
  // Read the parameters
  ch := rdch() REPEATWHILE ch='*s' | ch='*n'
  IF ch='F' DO
  { ch := rdch()
    maxfcount := rdn()
    maxfcount := maxfcount + maxfcount/10
  }
  WHILE ch=' ' DO ch := rdch() | ch='*n'
  IF ch='M' DO
  { ch := rdch()
    maxaddress := rdn()
    maxaddress := maxaddress + maxaddress/10
  }
  WHILE ch=' ' DO ch := rdch() | ch='*n'
  IF ch='K' DO { ch := rdch(); kval := rdn() }
  WHILE ch=' ' DO ch := rdch() | ch='*n'
  IF ch='S' DO { ch := rdch(); sval := rdn() }

  writef("maxfcount=%n maxaddress=%n kval=%n sval=%n*n",
          maxfcount, maxaddress, kval, sval)

  UNLESS maxfcount & maxaddress & kval & sval DO
  { writef("*nBad RASTER file*n")
    RESULTIS FALSE
  }

  // Set the default values, if necessary. Good for origbcpl.
  IF ml<0 DO ml := 0
  IF mh<0 DO mh := maxaddress

  IF fl<0 DO fl :=  0
  IF fh<0 DO fh :=  maxfcount

  IF fh < fl+kval DO fh := fl+kval
  
  IF mg<0 DO
  { mg := 1000
    WHILE mg <= mh-ml DO mg := 10*mg
    mg := mg/10
    IF (mh-ml)/mg <= 2 DO mg := mg/5
    IF (mh-ml)/mg <= 5 DO mg := mg/2
  }

  IF fg<0 DO
  { fg := 1000
    WHILE fg <= fh-fl DO fg := 10*fg
    fg := fg/10
    IF (fh-fl)/fg <= 2 DO fg := fg/5
    IF (fh-fl)/fg <= 5 DO fg := fg/2
  }

  writef("Converting %s to %s  size A%n at %n DPI*n",
          fromfilename, rasterfilename, asize, DPI)
  writef("Memory from %n to %n, mg=%n*n", ml, mh, mg)
  writef("Fcount from %n to %n, fg=%n*n", fl, fh, fg)

  m2yfac := FLOAT(YLEN) / FLOAT(mh-ml)
  // m2yfac is used to convert memory byte addresses
  // to pixel positions in the y direction.
  
  ygrid   := FIX(FLOAT(mg) * m2yfac)
  // ygrid is the number of pixels separating grid lines
  // on the byte address axis.
  ygrid1  := FIX(FLOAT(mg-ml MOD mg) * m2yfac)
  ygrid1  := ygrid1 MOD ygrid
  // ygrid1 is the pixel position of the lowest grid line
  // on the byte address axis.

  f2xfac := FLOAT(XLEN) / FLOAT(fh-fl)
  // f2xfac is used to convert instruction count coordinated
  // to pixel positions in the x direction.

  // Write the initial text of the .ps file.
  initpsraster()

  rastv := getvec(YLEN+1)

  IF rastv=0 DO
  { writef("Insufficient memory to allocate rastv*n")
    RETURN
  }

  selectoutput(rasterstream)

  setline(1)
  FOR i=1 TO 2 DO wrline()
  setline(0)

  fcount := 0

  // Skip to beginning of window
  UNTIL fcount>=fl | ch=endstreamch DO
  { UNTIL ch='N' | ch=endstreamch DO ch := rdch()
    IF ch='N' DO { ch := rdch(); fcount := fcount+kval }
  }

  { // Start of raster line loop
    SWITCHON capitalch(ch) INTO
    { DEFAULT:  writef("Bad ch '%c'*n", ch)
                abort(1111)
      CASE ' ':
      CASE '*c':
      CASE '*n':ch := rdch()
                LOOP

      CASE 'W': ch := rdch(); a := a + rdn(); LOOP
      CASE 'B': ch := rdch()
                b := a + rdn()
                mark(a,b)
                a := b
                LOOP

      CASE 'N': a := 0
                ch := rdch()
                // Fall through
      CASE 'Z':
      CASE endstreamch:
                // Draw a grid line if necessary
                IF fcount MOD fg < kval DO setline(1)
		// Set fcount for the next raster line
                fcount := fcount + kval
		// Output a count of progress
                IF fcount MOD 1000000 = 0 DO writef("*nfcount %i8 ", fcount)
                tally := tally - XLEN
                IF tally>=0 LOOP

                { tally := tally + (fh-fl)/kval
                  wrline()
                } REPEATUNTIL tally>0

                IF fcount>=fh BREAK
                setline(0)
                LOOP
    }
  } REPEAT
  
  selectoutput(rasterstream)
  writef("*nshowpage*n")
  selectoutput(stdout)
  newline()
}



