(*  Title: 	ZF/univ
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

The cumulative hierarchy and a small universe for recursive types
*)

(*Standard notation is V(i), but users might wish to use V for a variable*)
structure Univ =
struct
val const_decs = 
    [ (["Limit"],	"i=>o"),
      (["VVfrom"],	"[i,i]=>i"),
      (["VV"],		"i=>i"),
      (["VVrec"],	"[i, [i,i]=>i] =>i"),
      (["univ"],	"i=>i")
    ];

val thy = extend_theory Epsilon.thy "univ"
     ([], [], [], [], const_decs, None)
 [ 
  ("Limit_def",  "Limit(i) == 0:i & (ALL y:i. succ(y): i)"),

  ("VVfrom_def",  "VVfrom(A,i) == transrec(i, %x f. A Un (UN y:x. Pow(f`y)))"),
  ("VV_def",	  "VV == VVfrom(0)"),

  ("VVrec_def",   
   "VVrec(a,H) == transrec(rank(a), %x g. lam z: VV(succ(x)).          \
\                             H(z, lam w:VV(x). g`rank(w)`w)) ` a"),

  ("univ_def",	  "univ(A) == VVfrom(A,nat)")
 ];
end;

local val ax = get_axiom Univ.thy
in 
val Limit_def = ax"Limit_def";
val VVfrom_def = ax"VVfrom_def";
val VV_def = ax"VV_def";
val VVrec_def = ax"VVrec_def";
val univ_def = ax"univ_def";
end;


(*NOT SUITABLE FOR REWRITING -- RECURSIVE!*)
goal Univ.thy "VVfrom(A,i) = A Un (UN j:i. Pow(VVfrom(A,j)))";
by (rtac (VVfrom_def RS transrec_def_conv RS ssubst) 1);
by (SIMP_TAC ZF_ss 1);
val VVfrom_conv = result();

(** Monotonicity **)

val [prem] = goal Univ.thy
    "[| A<=B |] ==> ALL j. i<=j --> VVfrom(A,i) <= VVfrom(B,j)";
by (eps_ind_tac "i" 1);
by (rtac (impI RS allI) 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac (prem RS Un_mono) 1);
by (rtac UN_mono 1);
by (assume_tac 1);
by (rtac Pow_mono 1);
by (etac (bspec RS spec RS mp) 1);
by (assume_tac 1);
by (rtac subset_refl 1);
val VVfrom_mono_lemma = result();

(*  [| A<=B; i<=x |] ==> VVfrom(A,i) <= VVfrom(B,x)  *)
val VVfrom_mono = standard (VVfrom_mono_lemma RS spec RS mp);


(** A fundamental equality: VVfrom does not require ordinals! **)

goal Univ.thy "VVfrom(A,x) <= VVfrom(A,rank(x))";
by (eps_ind_tac "x" 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac (ZF_cs addSIs [rank_lt]) 1);
val VVfrom_rank_subset1 = result();

goal Univ.thy "VVfrom(A,rank(x)) <= VVfrom(A,x)";
by (eps_ind_tac "x" 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (safe_tac ZF_cs);
by (etac (rank_implies_mem RS bexE) 1);
by (swap_res_tac [UN_I]1);
by (assume_tac 1);
by (etac (subset_trans RS PowI) 1);
by (etac (subset_refl RS VVfrom_mono RS subset_trans) 1);
by (etac bspec 1);
by (assume_tac 1);
val VVfrom_rank_subset2 = result();

goal Univ.thy "VVfrom(A,rank(x)) = VVfrom(A,x)";
by (rtac equalityI 1);
by (rtac VVfrom_rank_subset2 1);
by (rtac VVfrom_rank_subset1 1);
val VVfrom_rank_eq = result();


(*** Basic closure properties ***)

val [prem] = goal Univ.thy "y:x ==> 0 : VVfrom(A,x)";
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac (ZF_cs addSIs [prem]) 1);
val zero_in_VVfrom = result();

goal Univ.thy "i <= VVfrom(A,i)";
by (eps_ind_tac "i" 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac ZF_cs 1);
val i_subset_VVfrom = result();

goal Univ.thy "A <= VVfrom(A,i)";
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac Un_upper1 1);
val A_subset_VVfrom = result();

val prems = goal Univ.thy "a <= VVfrom(A,i) ==> a: VVfrom(A,succ(i))";
by (rtac (VVfrom_conv RS ssubst) 1);
by (cfast_tac prems 1);
val subset_mem_VVfrom = result();

(** Singletons, doubletons and ordered pairs **)

val prems = goal Univ.thy "a: VVfrom(A,i) ==> {a} : VVfrom(A,succ(i))";
by (rtac subset_mem_VVfrom 1);
by (cfast_tac prems 1);
val singleton_in_VVfrom = result();

val prems = goal Univ.thy
    "[| a: VVfrom(A,i);  b: VVfrom(A,i) |] ==> {a,b} : VVfrom(A,succ(i))";
by (rtac subset_mem_VVfrom 1);
by (cfast_tac prems 1);
val doubleton_in_VVfrom = result();

val prems = goalw Univ.thy [Pair_def]
    "[| a: VVfrom(A,i);  b: VVfrom(A,i) |] ==> \
\    <a,b> : VVfrom(A,succ(succ(i)))";
by (REPEAT (resolve_tac (prems@[doubleton_in_VVfrom]) 1));
val pair_in_VVfrom = result();


(*** 0, successor and limit equations fof VVfrom ***)

goal Univ.thy "VVfrom(A,0) = A";
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac eq_cs 1);
val VVfrom_0_conv = result();

val [prem] = goal Univ.thy
    "Ord(i) ==> VVfrom(A,succ(i)) = A Un Pow(VVfrom(A,i))";
by (res_inst_tac [("i1","succ(i)")] (VVfrom_conv RS ssubst) 1);
by (resolve_tac ZF_congs 1);
by (rtac refl 1);
by (rtac equalityI 1);
by (rtac UN_least 1);
by (rtac (subset_refl RS VVfrom_mono RS Pow_mono) 1);
by (etac member_succD 1);
by (rtac prem 1);
by (rtac (succI1 RS RepFunI RS Union_upper) 1);
val VVfrom_succ_conv = result();

(*The premise distinguishes this from VVfrom(A,0);  allowing X=0 forces
  the conclusion to be VVfrom(A,Union(X)) = A Un (UN y:X. VVfrom(A,y)) *)
val [prem] = goal Univ.thy
    "y:X ==> VVfrom(A,Union(X)) = (UN y:X. VVfrom(A,y))";
by (rtac (VVfrom_conv RS ssubst) 1);
by (rtac equalityI 1);
by (rtac Un_least 1);
by (rtac (prem RS UN_I RS subsetI) 1);
by (etac (A_subset_VVfrom RS subsetD) 1);
by (safe_tac ZF_cs);
by (rtac UN_I 1);
by (assume_tac 1);
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac ZF_cs 1);
by (etac (VVfrom_conv RS equalityD1 RS subsetD RS UnE) 1);
by (assume_tac 1);
by (fast_tac ZF_cs 1);
val VVfrom_Union_conv = result();


(*** Limit ordinals ***)

val prems = goalw Univ.thy [Limit_def] "Limit(i) ==> i <= Union(i)";
by (cfast_tac prems 1);
val Limit_subset_Union = result();

val [limiti] = goalw Univ.thy [Limit_def] "Limit(i) ==> 0 : i";
by (rtac (limiti RS conjunct1) 1);
val Limit_0_I = result();

val [limiti,jprem] = goalw Univ.thy [Limit_def]
    "[| Limit(i);  j:i |] ==> succ(j) : i";
by (rtac (limiti RS conjunct2 RS bspec) 1);
by (rtac jprem 1);
val Limit_succ_I = result();

goalw Univ.thy [Limit_def] "Limit(nat)";
by (REPEAT (ares_tac [conjI, ballI, nat_0_I, nat_succ_I] 1));
val Limit_nat = result();

(*Because limit ordinals are non-empty, the conclusion is not
    			VVfrom(A,i) <= A Un (UN y:i. VVfrom(A,y)) *)
val [limiti] = goal Univ.thy
    "Limit(i) ==> VVfrom(A,i) <= (UN y:i. VVfrom(A,y))";
by (rtac (subset_refl RS VVfrom_mono RS subset_trans) 1);
by (rtac (limiti RS Limit_subset_Union) 1);
by (rtac (VVfrom_Union_conv RS equalityD1) 1);
by (rtac (limiti RS Limit_0_I) 1);
val Limit_VVfrom_subset = result();

val [ordi,limiti] = goal Univ.thy
    "[| Ord(i); Limit(i) |] ==> VVfrom(A,i) = (UN y:i. VVfrom(A,y))";
by (rtac (limiti RS Limit_VVfrom_subset RS equalityI) 1);
by (rtac subset_trans 1);
by (rtac (subset_refl RS VVfrom_mono) 2);
by (rtac (ordi RS Ord_Union_subset) 2);
by (rtac (VVfrom_Union_conv RS equalityD2) 1);
by (rtac (limiti RS Limit_0_I) 1);
val Limit_VVfrom_eq = result();

val [major,ordi,limiti] = goal Univ.thy
    "[| a: VVfrom(A,i);  Ord(i);  Limit(i) |] ==> {a} : VVfrom(A,i)";
by (rtac (limiti RS Limit_VVfrom_subset RS subsetD RS UN_E) 1);
by (rtac major 1);
by (rtac (ordi RS Limit_VVfrom_eq RS ssubst) 1);
by (rtac limiti 1);
by (rtac UN_I 1);
by (etac singleton_in_VVfrom 2);
by (etac (limiti RS Limit_succ_I) 1);
val singleton_in_VVfrom_limit = result();

(*Hard work is finding a single j:i such that {a,b}<=VVfrom(A,j)*)
val [aprem,bprem,ordi,limiti] = goal Univ.thy
    "[| a: VVfrom(A,i);  b: VVfrom(A,i);  Ord(i);  Limit(i) |] ==> \
\    {a,b} : VVfrom(A,i)";
by (rtac (aprem RS (limiti RS Limit_VVfrom_subset RS subsetD RS UN_E)) 1);
by (rtac (bprem RS (limiti RS Limit_VVfrom_subset RS subsetD RS UN_E)) 1);
by (rtac (limiti RS (ordi RS Limit_VVfrom_eq RS ssubst)) 1);
by (rtac UN_I 1);
by (rtac doubleton_in_VVfrom 2);
by (etac (Un_upper1 RS (subset_refl RS VVfrom_mono RS subsetD)) 2);
by (etac (Un_upper2 RS (subset_refl RS VVfrom_mono RS subsetD)) 2);
by (REPEAT (ares_tac [limiti RS Limit_succ_I, Ord_member_UnI, ordi] 1));
val doubleton_in_VVfrom_limit = result();

val [aprem,bprem,ordi,limiti] = goal Univ.thy
    "[| a: VVfrom(A,i);  b: VVfrom(A,i);  Ord(i);  Limit(i) |] ==> \
\    <a,b> : VVfrom(A,i)";
by (rtac (aprem RS (limiti RS Limit_VVfrom_subset RS subsetD RS UN_E)) 1);
by (rtac (bprem RS (limiti RS Limit_VVfrom_subset RS subsetD RS UN_E)) 1);
by (rtac (limiti RS (ordi RS Limit_VVfrom_eq RS ssubst)) 1);
by (rtac UN_I 1);
by (rtac pair_in_VVfrom 2);
by (etac (Un_upper1 RS (subset_refl RS VVfrom_mono RS subsetD)) 2);
by (etac (Un_upper2 RS (subset_refl RS VVfrom_mono RS subsetD)) 2);
by (REPEAT (ares_tac [limiti RS Limit_succ_I, Ord_member_UnI, ordi] 1));
val pair_in_VVfrom_limit = result();


(*** The set VV(i) ***)

goalw Univ.thy [VV_def] "VV(i) = (UN j:i. Pow(VV(j)))";
by (rtac (VVfrom_conv RS ssubst) 1);
by (fast_tac eq_cs 1);
val VV_conv = result();

goal Univ.thy "Transset(VV(a))";
by (eps_ind_tac "a" 1);
by (rtac (VV_conv RS ssubst) 1);
by (fast_tac (ZF_cs addSIs [Transset_Union,Transset_Pow]) 1);
val Transset_VV = result();

goalw Univ.thy [VV_def] "VV(rank(x)) = VV(x)";
by (rtac VVfrom_rank_eq 1);
val VV_rank_eq = result();

(** Characterisation of the elements of VV(i) **)

val [major] = goal Univ.thy "Ord(i) ==> ALL b. b : VV(i) --> rank(b) : i";
by (rtac (major RS trans_induct) 1);
by (rtac (VV_conv RS ssubst) 1);
by (safe_tac ZF_cs);
by (rtac (rank_conv RS ssubst) 1);
by (rtac sup_least2 1);
by (assume_tac 1);
by (assume_tac 1);
by (fast_tac ZF_cs 1);
val VV_rank_imp1 = result();
val VV_D = standard (VV_rank_imp1 RS spec RS mp);

val [major] = goal Univ.thy "Ord(i) ==> ALL b. rank(b) : i --> b : VV(i)";
by (rtac (major RS trans_induct) 1);
by (rtac allI 1);
by (rtac (VV_conv RS ssubst) 1);
by (fast_tac (ZF_cs addSIs [rank_lt]) 1);
val VV_rank_imp2 = result();
val VV_I = standard (VV_rank_imp2 RS spec RS mp);

val VV_rank_I = Ord_rank RS VV_I;

val [major] = goal Univ.thy "Ord(i) ==> b : VV(i) <-> rank(b) : i";
by (rtac iffI 1);
by (etac (major RS VV_D) 1);
by (etac (major RS VV_I) 1);
val VV_Ord_rank_iff = result();

goal Univ.thy "b : VV(a) <-> rank(b) : rank(a)";
by (rtac (VV_rank_eq RS subst) 1);
by (rtac (Ord_rank RS VV_Ord_rank_iff) 1);
val VV_rank_iff = result();

(** Recursion over VV levels! **)

(*NOT SUITABLE FOR REWRITING: recursive!*)
goalw Univ.thy [VVrec_def] "VVrec(a,H) = H(a, lam x:VV(rank(a)). VVrec(x,H))";
by (rtac (transrec_conv RS ssubst) 1);
by (SIMP_TAC (wf_ss addrews [Ord_rank RS Ord_succ RS VV_Ord_rank_iff]) 1);
by (SIMP_TAC (wf_ss addrews [Ord_rank RS VV_Ord_rank_iff RS iff_sym]) 1);
val VVrec_conv = result();

(*This form avoids giant explosions in proofs.  NOTE USE OF == *)
val rew::prems = goal Univ.thy
    "[| !!x. f(x)==VVrec(x,H) |] ==> \
\    f(a) = H(a, lam x: VV(rank(a)). f(x))";
by (rewtac rew);
by (rtac VVrec_conv 1);
val VVrec_def_conv = result();

val prems = goalw Univ.thy [VVrec_def]
    "[| a=a';  !!x u. H(x,u)=H'(x,u) |]  ==> VVrec(a,H)=VVrec(a',H')";
val VVrec_ss = ZF_ss addcongs ([transrec_cong] @ mk_congs Univ.thy ["rank"])
		      addrews (prems RL [sym]);
by (SIMP_TAC VVrec_ss 1);
val VVrec_cong = result();


(*** Closure properties for univ(A) ***)

goalw Univ.thy [univ_def] "0 : univ(A)";
by (rtac (nat_0_I RS zero_in_VVfrom) 1);
val zero_in_univ = result();

goalw Univ.thy [univ_def] "A <= univ(A)";
by (rtac A_subset_VVfrom 1);
val A_subset_univ = result();

(** Closure under unordered and ordered pairs **)

val prems = goalw Univ.thy [univ_def] "a: univ(A) ==> {a} : univ(A)";
by (rtac singleton_in_VVfrom_limit 1);
by (REPEAT (resolve_tac (prems@[Ord_nat,Limit_nat]) 1));
val singleton_in_univ = result();

val prems = goalw Univ.thy [univ_def] 
    "[| a: univ(A);  b: univ(A) |] ==> {a,b} : univ(A)";
by (rtac doubleton_in_VVfrom_limit 1);
by (REPEAT (resolve_tac (prems@[Ord_nat,Limit_nat]) 1));
val doubleton_in_univ = result();

val prems = goalw Univ.thy [univ_def]
    "[| a: univ(A);  b: univ(A) |] ==> <a,b> : univ(A)";
by (rtac pair_in_VVfrom_limit 1);
by (REPEAT (resolve_tac (prems@[Ord_nat,Limit_nat]) 1));
val pair_in_univ = result();

val prems = goal Univ.thy "[| B<=univ(A);  C<=univ(A) |] ==> B*C <= univ(A)";
by (rtac (product_mono RS subset_trans) 1);
by (REPEAT (ares_tac (prems@[subsetI,pair_in_univ]) 1
     ORELSE eresolve_tac [SigmaE, ssubst] 1));
val product_subset_univ = result();


(** Closure under disjoint union **)

val prems = goalw Univ.thy [Inl_def]
    "a: univ(A) ==> Inl(a) : univ(A)";
by (REPEAT (resolve_tac (prems@[zero_in_univ,pair_in_univ]) 1));
val Inl_in_univ = result();

val prems = goalw Univ.thy [Inr_def]
    "b: univ(A) ==> Inr(b) : univ(A)";
by (REPEAT (resolve_tac (prems@[zero_in_univ,singleton_in_univ,
				pair_in_univ]) 1));
val Inr_in_univ = result();

val prems = goal Univ.thy "[| B<=univ(A);  C<=univ(A) |] ==> B+C <= univ(A)";
by (rtac (sum_mono RS subset_trans) 1);
by (REPEAT (ares_tac (prems@[subsetI,Inl_in_univ,Inr_in_univ]) 1
     ORELSE eresolve_tac [sumE, ssubst] 1));
val sum_subset_univ = result();

(** Closure under binary union -- use Un_least **)
(** Closure under Collect -- use  (Collect_subset RS subset_trans)  **)
(** Closure under RepFun -- use   RepFun_subset  **)


(** The natural numbers **)


goalw Univ.thy [univ_def] "nat <= univ(A)";
by (rtac i_subset_VVfrom 1);
val nat_subset_univ = result();

(* n:nat ==> n:univ(A) *)
val nat_into_univ = standard (nat_subset_univ RS subsetD);

(** instances for 1 and 2 **)

goal Univ.thy "succ(0) : univ(A)";
by (REPEAT (ares_tac [nat_into_univ, nat_0_I, nat_succ_I] 1));
val one_in_univ = result();

goal Univ.thy "succ(succ(0)) : univ(A)";
by (REPEAT (ares_tac [nat_into_univ, nat_0_I, nat_succ_I] 1));
val two_in_univ = result();
