(*  Title: 	Printer
    Author: 	Tobias Nipkow
*)

signature PRINTER0 =
sig
  val show_types: bool ref
  val show_sorts: bool ref
end;

signature PRINTER =
sig
  include PRINTER0
  structure XGram: XGRAM
  structure Pretty:PRETTY
  type tab
  val mk_print_tab:
	string XGram.Prod list * (term list -> term)XGram.Symtab.table -> tab
  val pretty_term: tab * string list -> term -> Pretty.T
  val string_of_typ: tab * string list -> typ -> string
end;

functor PrinterFun(structure Lexicon:LEXICON and SExtension:SEXTENSION
		   and TypeExt: TYPE_EXT and XGram:XGRAM and Pretty:PRETTY)
	: PRINTER =
struct

structure XGram = XGram;
structure Pretty = Pretty;

local open XGram in

datatype symb = NTER of int | TER | S of string | BRK of int
	      | BLK of int * symb list;
datatype format = Prnt   of symb list * int * int
		| Trnslt of term list -> term
		| TorP   of (term list -> term) * (symb list * int * int)
type tab = format Symtab.table;

val show_types = ref false;
val show_sorts = ref false;


fun merge(s,S(s')::l) = S(s^s')::l
  | merge(s,l) = S(s)::l;

fun syn2pr(Terminal(s)::sy) =
	let val (symbs,sy') = syn2pr(sy) in (merge(s,symbs),sy') end
  | syn2pr(Space(s)::sy) =
	let val (symbs,sy') = syn2pr(sy) in (merge(s,symbs),sy') end
  | syn2pr(Nonterminal(s,p)::sy) =
	let val (symbs,sy') = syn2pr(sy);
	    val symb = if Lexicon.predef_term s = Lexicon.end_token
		       then NTER(p) else TER
	in (symb::symbs, sy') end
  | syn2pr(Bg(i)::sy) =
	let val (bsymbs,sy') = syn2pr(sy);
	    val (symbs,sy'') = syn2pr(sy')
	in (BLK(i,bsymbs) :: symbs, sy'') end
  | syn2pr(Brk(i)::sy) =
	let val (symbs,sy') = syn2pr(sy) in (BRK(i)::symbs,sy') end
  | syn2pr(En::sy) = ([],sy)
  | syn2pr([]) = ([],[]);


fun nargs(NTER _ :: symbs) = nargs(symbs)+1
  | nargs(TER :: symbs) = nargs(symbs)+1
  | nargs(S _ :: symbs) = nargs(symbs)
  | nargs(BRK _ :: symbs) = nargs(symbs)
  | nargs(BLK(_,bsymbs) :: symbs) = nargs(symbs) + nargs(bsymbs)
  | nargs([]) = 0;

fun mk_prnp(sy,p) = let val (pr,_) = syn2pr sy in (pr, nargs pr, p) end;

fun add_prod(Prod(_,_,"",_),prt) = prt
  | add_prod(Prod(_,sy,opn,p),prt) = (case Symtab.lookup(prt,opn) of
	None => Symtab.update((opn,Prnt(mk_prnp(sy,p))),prt)
      | Some(Prnt _) => prt
      | Some(Trnslt f) => Symtab.update((opn,TorP(f,mk_prnp(sy,p))),prt)
      | Some(TorP _) => prt);

fun add_tr(prt,(opn,f)) = Symtab.update((opn,Trnslt f),prt);

fun mk_print_tab(prodl,trtab: (term list -> term)Symtab.table) : tab =
    let val prt0 = foldl add_tr (Symtab.null,Symtab.alist_of trtab)
    in foldr add_prod (prodl,prt0) end;


fun split_comb(0, tms, tm) = (tm,tms)
  | split_comb(i, tm1::tms, tm) = split_comb(i-1, tms, tm $ tm1)
  | split_comb _ = error("split_comb");

fun pretty_term(tab,fixs) tm : Pretty.T =
let val seen = ref([]: (string * typ) list);

    fun synT([],args) = ([],args)
      | synT(NTER(p)::symbs,t::args) =
	let val (Ts,args') = synT(symbs,args) in (tmT(t,p)@Ts, args') end
      | synT(TER::symbs,t::args) =
	let val (Ts,args') = synT(symbs,args) in (tmT(t,0)@Ts, args') end
      | synT(S(s)::symbs,args) =
	let val (Ts,args') = synT(symbs,args) in (Pretty.str s::Ts, args') end
      | synT(BLK(i,bsymbs)::symbs,args) =
	let val (bTs,args') = synT(bsymbs,args);
	    val (Ts,args'') = synT(symbs,args')
	in (Pretty.blk(i,bTs)::Ts, args'') end
      | synT(BRK(i)::symbs,args) =
	let val (Ts,args') = synT(symbs,args) in (Pretty.brk i::Ts, args') end
      | synT(_::_,[]) = error("synT")

    and parT(pr,args,p,p':int) =
	if p > p' then fst(synT([BLK(1,S"(" :: pr @ [S")"])],args))
	else fst(synT(pr,args))

    and prefixT(c,a,args,p) = if null(args) then [Pretty.str(a)]
			      else tmT(SExtension.appl_tr'(c,args),p)

    and combT(tup as (c,a,args,p)) =
	let val nargs = length args
	    fun prin(pr,n,p') =
		if n = nargs then parT(pr,args,p,p') else
		if n > nargs then prefixT(tup)
		else tmT(SExtension.appl_tr'(split_comb(n,args,c)),p)
	in case Symtab.lookup(tab,a) of
	     None => prefixT(tup)
	   | Some(Prnt prnp) => prin(prnp)
	   | Some(Trnslt f) => (tmT(f args,p) handle Match => prefixT(tup))
	   | Some(TorP(f,prnp)) => (tmT(f args,p) handle Match => prin(prnp))
	end

    and applT(tm,p) = let val (f,args) = strip_comb tm
	in case f of
	     Const(a,_) => combT(f,a,args,p)
	   | _ => tmT(SExtension.appl_tr'(f,args),p)
	end

    and typeT(tm,a,U,p) =
	if not(!show_types) orelse (a,U) mem !seen then [Pretty.str a]
	else (seen := (a,U) :: !seen; show_types := false;
	      let val Ts = tmT(TypeExt.constrain(fixs,!show_sorts,tm,U), p)
	      in show_types := true; Ts end)

    and tmT(tm as Free(a,U),p) = typeT(tm,a,U,p)
      | tmT(tm as Var(v,U),p) = typeT(tm,Lexicon.string_of_vname v, U, p)
      | tmT(tm as Const(a,_),p) = combT(tm,a,[],p)
      | tmT(tm as _$_,p) = applT(tm,p)
      | tmT(tm as Abs _,p) = tmT(SExtension.abs_tr' tm,p)
      | tmT(Bound i,_) = [Pretty.str("B." ^ string_of_int i)]

in Pretty.blk(0,tmT(tm,0)) end;

fun string_of_typ(args as (tab,fixs)) typ =
let val term = TypeExt.term_of_typ(fixs,!show_sorts) typ;
    val old = !show_types;
    val unit = show_types := false;
    val s = Pretty.string_of(pretty_term args term)
    val unit = show_types := old;
in s end;

end;
end;
